!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     calculate Onsager-Machlup action
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine getaction_om_MPI
!***********************************************************************

      use common_variables, only : &
     &   pot, potential, fx, fy, fz, nbead, equation_om

      use om_variables, only: &
     &   fx_0, fy_0, fz_0, pot_0, potential_0, action_om, &
     &   fx_om, fy_om, fz_om

      implicit none

      real(8) :: dbead

!-----------------------------------------------------------------------
!     /*   constants                                                  */
!-----------------------------------------------------------------------

      dbead = dble(nbead)

!-----------------------------------------------------------------------
!     /*   calculate force at original position                       */
!-----------------------------------------------------------------------

      call getforce_MPI

!-----------------------------------------------------------------------
!     /*   save potential and force                                   */
!-----------------------------------------------------------------------

      pot_0(:) = pot(:)

      potential_0 = potential

      fx_0(:,:) = fx(:,:) * dbead
      fy_0(:,:) = fy(:,:) * dbead
      fz_0(:,:) = fz(:,:) * dbead

!-----------------------------------------------------------------------
!     /*   initialize action                                          */
!-----------------------------------------------------------------------

      action_om = 0.d0

      fx_om(:,:) = 0.d0
      fy_om(:,:) = 0.d0
      fz_om(:,:) = 0.d0

!-----------------------------------------------------------------------
!        /*   option of langevin equation                             */
!-----------------------------------------------------------------------

      if      ( equation_om(1:12) .eq. 'OVERDAMPED  ' ) then

         call getaction_om_overdamped_MPI

      else if ( equation_om(1:12) .eq. 'UNDERDAMPED ' ) then

         call getaction_om_underdamped_MPI

      end if

!-----------------------------------------------------------------------
!        /*   restore potential and force                             */
!-----------------------------------------------------------------------

      pot(:) = pot_0(:)

      potential = potential_0

      fx(:,:) = fx_0(:,:) / dbead
      fy(:,:) = fy_0(:,:) / dbead
      fz(:,:) = fz_0(:,:) / dbead

      return
      end





!***********************************************************************
      subroutine getaction_om_overdamped_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only: &
     &   fdiff, x, y, z, fx, fy, fz, physmass, iounit, natom, nbead

      use om_variables, only: &
     &   fx_om, fy_om, fz_om, fx_p, fy_p, fz_p, fx_m, fy_m, fz_m, &
     &   fx_0, fy_0, fz_0, pot_0, fdscale_om, gamma_om, &
     &   action_om, dt_om

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k

      real(8) :: pmi, dmax, c2, dbead

      real(8) :: fxij, fyij, fzij, fxik, fyik, fzik

      real(8), dimension(natom,nbead) :: dx, dy, dz

!-----------------------------------------------------------------------
!     /*   constants                                                  */
!-----------------------------------------------------------------------

      dbead = dble(nbead)

!-----------------------------------------------------------------------
!     /*   action: potential part                                     */
!-----------------------------------------------------------------------

      action_om = action_om + 0.5d0 * ( pot_0(nbead) - pot_0(1) )

      do j = 1, nbead-1

         k = j + 1

         do i = 1, natom

            pmi = physmass(i)

            c2 = 0.25d0 * dt_om / pmi / gamma_om

            fxij = fx_0(i,j)
            fyij = fy_0(i,j)
            fzij = fz_0(i,j)

            fxik = fx_0(i,k)
            fyik = fy_0(i,k)
            fzik = fz_0(i,k)

            action_om = action_om &
     &         + 0.5d0 * c2 * ( fxij*fxij + fyij*fyij + fzij*fzij ) &
     &         + 0.5d0 * c2 * ( fxik*fxik + fyik*fyik + fzik*fzik )

         end do

      end do

!-----------------------------------------------------------------------
!     /*   finite difference: find optimal increment                  */
!-----------------------------------------------------------------------

      c2 = 0.25d0 * dt_om / gamma_om

      do i = 1, natom
         pmi = physmass(i)
         dx(i,1) = 0.5d0*c2*fx_0(i,1)/pmi
         dy(i,1) = 0.5d0*c2*fy_0(i,1)/pmi
         dz(i,1) = 0.5d0*c2*fz_0(i,1)/pmi
      end do

      do j = 2, nbead-1
      do i = 1, natom
         pmi = physmass(i)
         dx(i,j) = c2*fx_0(i,j)/pmi
         dy(i,j) = c2*fy_0(i,j)/pmi
         dz(i,j) = c2*fz_0(i,j)/pmi
      end do
      end do

      do i = 1, natom
         pmi = physmass(i)
         dx(i,nbead) = 0.5d0*c2*fx_0(i,nbead)/pmi
         dy(i,nbead) = 0.5d0*c2*fy_0(i,nbead)/pmi
         dz(i,nbead) = 0.5d0*c2*fz_0(i,nbead)/pmi
      end do

!-----------------------------------------------------------------------
!     /*   finite difference: find optimal increment                  */
!-----------------------------------------------------------------------

      dmax = 0.d0

      do j = 1, nbead
      do i = 1, natom
         dmax = max( dmax, abs(dx(i,j)) )
         dmax = max( dmax, abs(dy(i,j)) )
         dmax = max( dmax, abs(dz(i,j)) )
      end do
      end do

      fdscale_om(:) = fdiff / dmax

!-----------------------------------------------------------------------
!        /*   finite difference: minus direction                      */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom
         x(i,j) = x(i,j) - fdscale_om(j)*dx(i,j)
         y(i,j) = y(i,j) - fdscale_om(j)*dy(i,j)
         z(i,j) = z(i,j) - fdscale_om(j)*dz(i,j)
      end do
      end do

      call getforce_MPI

      fx_m(:,:) = fx(:,:) * dbead
      fy_m(:,:) = fy(:,:) * dbead
      fz_m(:,:) = fz(:,:) * dbead

!-----------------------------------------------------------------------
!        /*   finite difference: plus direction                       */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom
         x(i,j) = x(i,j) + 2.d0*fdscale_om(j)*dx(i,j)
         y(i,j) = y(i,j) + 2.d0*fdscale_om(j)*dy(i,j)
         z(i,j) = z(i,j) + 2.d0*fdscale_om(j)*dz(i,j)
      end do
      end do

      call getforce_MPI

      fx_p(:,:) = fx(:,:) * dbead
      fy_p(:,:) = fy(:,:) * dbead
      fz_p(:,:) = fz(:,:) * dbead

!-----------------------------------------------------------------------
!        /*   finite difference: zero                                 */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom
         x(i,j) = x(i,j) - fdscale_om(j)*dx(i,j)
         y(i,j) = y(i,j) - fdscale_om(j)*dy(i,j)
         z(i,j) = z(i,j) - fdscale_om(j)*dz(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   potential part                                             */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom

         fx_om(i,j) = fx_om(i,j) &
     &              + ( fx_m(i,j) - fx_p(i,j) ) / ( fdscale_om(j) )

         fy_om(i,j) = fy_om(i,j) &
     &              + ( fy_m(i,j) - fy_p(i,j) ) / ( fdscale_om(j) )

         fz_om(i,j) = fz_om(i,j) &
     &              + ( fz_m(i,j) - fz_p(i,j) ) / ( fdscale_om(j) )

      end do
      end do

!-----------------------------------------------------------------------
!     /*   correction of potential part                               */
!-----------------------------------------------------------------------

      do i = 1, natom

         fx_om(i,1)     = fx_om(i,1)     - 0.5d0*fx_0(i,1)
         fy_om(i,1)     = fy_om(i,1)     - 0.5d0*fy_0(i,1)
         fz_om(i,1)     = fz_om(i,1)     - 0.5d0*fz_0(i,1)

         fx_om(i,nbead) = fx_om(i,nbead) + 0.5d0*fx_0(i,nbead)
         fy_om(i,nbead) = fy_om(i,nbead) + 0.5d0*fy_0(i,nbead)
         fz_om(i,nbead) = fz_om(i,nbead) + 0.5d0*fz_0(i,nbead)

      end do

      return
      end





!***********************************************************************
      subroutine getaction_om_underdamped_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only: &
     &   fdiff, x, y, z, fx, fy, fz, physmass, iounit, natom, nbead

      use om_variables, only: &
     &   fx_om, fy_om, fz_om, fx_p, fy_p, fz_p, fx_m, fy_m, fz_m, &
     &   fx_0, fy_0, fz_0, pot_0, fdscale_om, gamma_om, &
     &   action_om, dt_om

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m

      real(8) :: pmi, dmax, c2, c5, rx, ry, rz, dbead

      real(8) :: fxik, fyik, fzik, fxij, fyij, fzij

      real(8), dimension(natom,nbead) :: dx, dy, dz

!-----------------------------------------------------------------------
!     /*   constants                                                  */
!-----------------------------------------------------------------------

      dbead = dble(nbead)

!-----------------------------------------------------------------------
!     /*   action: potential part                                     */
!-----------------------------------------------------------------------

      action_om = action_om + 0.5d0 * ( pot_0(nbead) - pot_0(1) )

      do j = 1, nbead-1

         k = j + 1

         do i = 1, natom

            pmi = physmass(i)

            c2 = 0.25d0 * dt_om / pmi / gamma_om

            fxij = fx_0(i,j)
            fyij = fy_0(i,j)
            fzij = fz_0(i,j)

            fxik = fx_0(i,k)
            fyik = fy_0(i,k)
            fzik = fz_0(i,k)

            action_om = action_om &
     &         + 0.5d0 * c2 * ( fxij*fxij + fyij*fyij + fzij*fzij ) &
     &         + 0.5d0 * c2 * ( fxik*fxik + fyik*fyik + fzik*fzik )

         end do

      end do

!-----------------------------------------------------------------------
!     /*   action: force-acceleration cross term                      */
!-----------------------------------------------------------------------

      do j = 2, nbead-1

         l = j + 1
         m = j - 1

         do i = 1, natom

            pmi = physmass(i)

            c5 = - 0.5d0 / gamma_om / dt_om

            rx = x(i,l) + x(i,m) - 2.d0*x(i,j)
            ry = y(i,l) + y(i,m) - 2.d0*y(i,j)
            rz = z(i,l) + z(i,m) - 2.d0*z(i,j)

            fxij = fx_0(i,j)
            fyij = fy_0(i,j)
            fzij = fz_0(i,j)

            action_om = action_om + c5 * ( rx*fxij + ry*fyij + rz*fzij )

         end do

      end do

!-----------------------------------------------------------------------
!     /*   finite difference: find optimal increment                  */
!-----------------------------------------------------------------------

      c2 = 0.25d0 * dt_om / gamma_om

      do i = 1, natom
         pmi = physmass(i)
         dx(i,1) = 0.5d0*c2*fx_0(i,1)/pmi
         dy(i,1) = 0.5d0*c2*fy_0(i,1)/pmi
         dz(i,1) = 0.5d0*c2*fz_0(i,1)/pmi
      end do

      do j = 2, nbead-1
      do i = 1, natom
         pmi = physmass(i)
         dx(i,j) = c2*fx_0(i,j)/pmi
         dy(i,j) = c2*fy_0(i,j)/pmi
         dz(i,j) = c2*fz_0(i,j)/pmi
      end do
      end do

      do i = 1, natom
         pmi = physmass(i)
         dx(i,nbead) = 0.5d0*c2*fx_0(i,nbead)/pmi
         dy(i,nbead) = 0.5d0*c2*fy_0(i,nbead)/pmi
         dz(i,nbead) = 0.5d0*c2*fz_0(i,nbead)/pmi
      end do

      c5 = - 0.5d0 / gamma_om / dt_om

      do j = 2, nbead-1
      do i = 1, natom
         dx(i,j) = dx(i,j) &
     &           + 0.5d0 * c5 * ( x(i,j+1) + x(i,j-1) - 2.d0*x(i,j) )
         dy(i,j) = dy(i,j) &
     &           + 0.5d0 * c5 * ( y(i,j+1) + y(i,j-1) - 2.d0*y(i,j) )
         dz(i,j) = dz(i,j) &
     &           + 0.5d0 * c5 * ( z(i,j+1) + z(i,j-1) - 2.d0*z(i,j) )
      end do
      end do

!-----------------------------------------------------------------------
!     /*   finite difference: find optimal increment                  */
!-----------------------------------------------------------------------

      dmax = 0.d0

      do j = 1, nbead
      do i = 1, natom
         dmax = max( dmax, abs(dx(i,j)) )
         dmax = max( dmax, abs(dy(i,j)) )
         dmax = max( dmax, abs(dz(i,j)) )
      end do
      end do

      fdscale_om(:) = fdiff / dmax

!-----------------------------------------------------------------------
!        /*   finite difference: minus direction                      */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom
         x(i,j) = x(i,j) - fdscale_om(j)*dx(i,j)
         y(i,j) = y(i,j) - fdscale_om(j)*dy(i,j)
         z(i,j) = z(i,j) - fdscale_om(j)*dz(i,j)
      end do
      end do

      call getforce_MPI

      fx_m(:,:) = fx(:,:) * dbead
      fy_m(:,:) = fy(:,:) * dbead
      fz_m(:,:) = fz(:,:) * dbead

!-----------------------------------------------------------------------
!        /*   finite difference: plus direction                       */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom
         x(i,j) = x(i,j) + 2.d0*fdscale_om(j)*dx(i,j)
         y(i,j) = y(i,j) + 2.d0*fdscale_om(j)*dy(i,j)
         z(i,j) = z(i,j) + 2.d0*fdscale_om(j)*dz(i,j)
      end do
      end do

      call getforce_MPI

      fx_p(:,:) = fx(:,:) * dbead
      fy_p(:,:) = fy(:,:) * dbead
      fz_p(:,:) = fz(:,:) * dbead

!-----------------------------------------------------------------------
!        /*   finite difference: zero                                 */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom
         x(i,j) = x(i,j) - fdscale_om(j)*dx(i,j)
         y(i,j) = y(i,j) - fdscale_om(j)*dy(i,j)
         z(i,j) = z(i,j) - fdscale_om(j)*dz(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   potential part                                             */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom

         fx_om(i,j) = fx_om(i,j) &
     &              + ( fx_m(i,j) - fx_p(i,j) ) / ( fdscale_om(j) )

         fy_om(i,j) = fy_om(i,j) &
     &              + ( fy_m(i,j) - fy_p(i,j) ) / ( fdscale_om(j) )

         fz_om(i,j) = fz_om(i,j) &
     &              + ( fz_m(i,j) - fz_p(i,j) ) / ( fdscale_om(j) )

      end do
      end do

!-----------------------------------------------------------------------
!        /*   correction of potential part                            */
!-----------------------------------------------------------------------

      do i = 1, natom

         fx_om(i,1)     = fx_om(i,1)     - 0.5d0*fx_0(i,1)
         fy_om(i,1)     = fy_om(i,1)     - 0.5d0*fy_0(i,1)
         fz_om(i,1)     = fz_om(i,1)     - 0.5d0*fz_0(i,1)

         fx_om(i,nbead) = fx_om(i,nbead) + 0.5d0*fx_0(i,nbead)
         fy_om(i,nbead) = fy_om(i,nbead) + 0.5d0*fy_0(i,nbead)
         fz_om(i,nbead) = fz_om(i,nbead) + 0.5d0*fz_0(i,nbead)

      end do

!-----------------------------------------------------------------------
!     /*   force of action: correction of force-acceleration          */
!-----------------------------------------------------------------------

      c5 = - 0.5d0 / gamma_om / dt_om

      do j = 2, nbead-1

         l = j + 1
         m = j - 1

         do i = 1, natom

            fx_om(i,m) = fx_om(i,m) - c5*fx_0(i,j)
            fy_om(i,m) = fy_om(i,m) - c5*fy_0(i,j)
            fz_om(i,m) = fz_om(i,m) - c5*fz_0(i,j)

            fx_om(i,j) = fx_om(i,j) + 2.d0*c5*fx_0(i,j)
            fy_om(i,j) = fy_om(i,j) + 2.d0*c5*fy_0(i,j)
            fz_om(i,j) = fz_om(i,j) + 2.d0*c5*fz_0(i,j)

            fx_om(i,l) = fx_om(i,l) - c5*fx_0(i,j)
            fy_om(i,l) = fy_om(i,l) - c5*fy_0(i,j)
            fz_om(i,l) = fz_om(i,l) - c5*fz_0(i,j)

         end do

      end do

      return
      end
