!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     calculate collective variables in metadynamics
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine get_cv_meta_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : nbead

      use meta_variables, only : smeta, nmeta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   initialize cv                                              */
!-----------------------------------------------------------------------

!     /*   cv   */
      smeta(:,:) = 0.d0

!-----------------------------------------------------------------------
!     /*   calculate smeta                                            */
!-----------------------------------------------------------------------

!     /*   linear bonding   */
      call get_lin_meta_MPI

!     /*   angular bonding   */
      call get_angl_meta_MPI

!     /*   dihedral bonding   */
      call get_dih_meta_MPI

!     /*   bond difference   */
      call get_diff_meta_MPI

!     /*   coordination number   */
      call get_cord_meta_MPI

!     /*   difference in coordination number   */
      call get_dcord_meta_MPI

!     /*   center of mass   */
      call get_xyz_meta_MPI

!     /*   difference in center of mass   */
      call get_dxyz_meta_MPI

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   cv   */
      call my_mpi_allreduce_real_2 ( smeta, nmeta, nbead )

      return
      end





!***********************************************************************
      subroutine get_lin_meta_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pi, nbead, myrank, nprocs

      use meta_variables, only : &
     &   smeta, ipbc_meta, i_meta, j_meta, nmeta, itype_meta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: n, i, j, m

      real(8) :: xij, yij, zij, rij

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do m = 1, nbead

!        /*   only my bead   */
         if ( mod( m-1, nprocs ) .ne. myrank ) cycle

         do n = 1, nmeta

            if ( itype_meta(n) .ne. 1 )  cycle

            ipbc_meta(n) = 0

            i = i_meta(n)
            j = j_meta(n)

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            smeta(n,m) = rij

         end do

      end do

      return
      end





!***********************************************************************
      subroutine get_angl_meta_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pi, nbead, myrank, nprocs

      use meta_variables, only : &
     &   smeta, ipbc_meta, i_meta, j_meta, k_meta, nmeta, itype_meta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: n, i, j, k, m

      real(8) :: xij, yij, zij, xkj, ykj, zkj, rij2, rkj2, rijk, pijk, &
     &           qijk, bijk, aijk

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do m = 1, nbead

!        /*   only my bead   */
         if ( mod( m-1, nprocs ) .ne. myrank ) cycle

         do n = 1, nmeta

            if ( itype_meta(n) .ne. 2 )  cycle

            ipbc_meta(n) = 1

            i = i_meta(n)
            j = j_meta(n)
            k = k_meta(n)

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            xkj = x(k,m) - x(j,m)
            ykj = y(k,m) - y(j,m)
            zkj = z(k,m) - z(j,m)

            call pbc_atom_MPI ( xkj, ykj, zkj )

            rij2 = xij*xij + yij*yij + zij*zij
            rkj2 = xkj*xkj + ykj*ykj + zkj*zkj

            rijk = sqrt( rij2*rkj2 )

            pijk = xij*xkj + yij*ykj + zij*zkj

            qijk  = pijk/rijk

            qijk = max( qijk, -1.d0 )
            qijk = min( qijk,  1.d0 )

            bijk = acos( qijk )

            aijk = bijk*(180.d0/pi)

            aijk = aijk - 360.d0 * nint( aijk / 360.d0 )

            smeta(n,m) = aijk

         end do

      end do

      return
      end





!***********************************************************************
      subroutine get_dih_meta_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pi, nbead, myrank, nprocs

      use meta_variables, only : &
     &   smeta, ipbc_meta, i_meta, j_meta, k_meta, l_meta, nmeta, &
     &   itype_meta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: n, i, j, k, l, m

      real(8) :: xij, yij, zij, xkj, ykj, zkj, xlj, ylj, zlj, &
     &           xijk, yijk, zijk, xjkl, yjkl, zjkl, rijk2, rjkl2, &
     &           rijkl2, rijkl2inv, cos_phi, phi, psi, sign_phi

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do m = 1, nbead

!        /*   only my bead   */
         if ( mod( m-1, nprocs ) .ne. myrank ) cycle

         do n = 1, nmeta

            if ( itype_meta(n) .ne. 3 )  cycle

            ipbc_meta(n) = 1

            i = i_meta(n)
            j = j_meta(n)
            k = k_meta(n)
            l = l_meta(n)

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            xkj = x(k,m) - x(j,m)
            ykj = y(k,m) - y(j,m)
            zkj = z(k,m) - z(j,m)

            call pbc_atom_MPI ( xkj, ykj, zkj )

            xlj = x(l,m) - x(j,m)
            ylj = y(l,m) - y(j,m)
            zlj = z(l,m) - z(j,m)

            call pbc_atom_MPI ( xlj, ylj, zlj )

            xijk = yij*zkj - zij*ykj
            yijk = zij*xkj - xij*zkj
            zijk = xij*ykj - yij*xkj

            xjkl = ylj*zkj - zlj*ykj
            yjkl = zlj*xkj - xlj*zkj
            zjkl = xlj*ykj - ylj*xkj

            rijk2  = xijk*xijk + yijk*yijk + zijk*zijk
            rjkl2  = xjkl*xjkl + yjkl*yjkl + zjkl*zjkl

            rijkl2 = sqrt(rijk2*rjkl2)

            rijkl2inv = 1.d0 / rijkl2

            cos_phi = ( xijk*xjkl + yijk*yjkl + zijk*zjkl ) * rijkl2inv

            cos_phi = max( cos_phi, -1.d0 )
            cos_phi = min( cos_phi,  1.d0 )

            phi = acos( cos_phi )

            sign_phi = ( yijk*zjkl - zijk*yjkl ) * xkj &
     &               + ( zijk*xjkl - xijk*zjkl ) * ykj &
     &               + ( xijk*yjkl - yijk*xjkl ) * zkj

            sign_phi = sign( 1.d0, sign_phi )

            phi = phi * sign_phi

            psi = phi*(180.d0/pi)

            psi = psi - 360.d0 * nint( psi / 360.d0 )

            smeta(n,m) = psi

         end do

      end do

      return
      end





!***********************************************************************
      subroutine get_diff_meta_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pi, nbead, myrank, nprocs

      use meta_variables, only : &
     &   smeta, ipbc_meta, i_meta, j_meta, k_meta, nmeta, itype_meta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: n, i, j, k, m

      real(8) :: xij, yij, zij, rij, xkj, ykj, zkj, rkj

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do m = 1, nbead

!        /*   only my bead   */
         if ( mod( m-1, nprocs ) .ne. myrank ) cycle

         do n = 1, nmeta

            if ( itype_meta(n) .ne. 4 )  cycle

            ipbc_meta(n) = 0

            i = i_meta(n)
            j = j_meta(n)
            k = k_meta(n)

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            xkj = x(k,m) - x(j,m)
            ykj = y(k,m) - y(j,m)
            zkj = z(k,m) - z(j,m)

            call pbc_atom_MPI ( xkj, ykj, zkj )

            rkj = sqrt( xkj*xkj + ykj*ykj + zkj*zkj )

            smeta(n,m) = rij - rkj

         end do

      end do

      return
      end





!***********************************************************************
      subroutine get_cord_meta_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, nbead, natom, ikind, myrank, nprocs

      use meta_variables, only : &
     &   itype_meta, ipbc_meta, req_meta, smeta, &
     &   i_meta, j_meta, nu_meta, mu_meta, nmeta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: nu, mu, i, j, m, n

      real(8) :: req, cn, xij, yij, zij, rij

!-----------------------------------------------------------------------
!     /*   main loop start                                            */
!-----------------------------------------------------------------------

      do m = 1, nbead

!        /*   only my bead   */
         if ( mod( m-1, nprocs ) .ne. myrank ) cycle

         do n = 1, nmeta

            if ( itype_meta(n) .ne. 5 )  cycle

!-----------------------------------------------------------------------
!           /*   parameters                                           */
!-----------------------------------------------------------------------

            ipbc_meta(n) = 0

            nu  = nu_meta(n,1)

            mu  = mu_meta(n,1)

            req = req_meta(n,1)

!-----------------------------------------------------------------------
!           /*   calculate coordination number                        */
!-----------------------------------------------------------------------

            cn = 0.d0

            do i = 1, natom-1
            do j = i+1, natom

               if ( ( ( ikind(i) .eq. i_meta(n) ) .and. &
     &                ( ikind(j) .eq. j_meta(n) ) ) .or. &
     &              ( ( ikind(i) .eq. j_meta(n) ) .and. &
     &                ( ikind(j) .eq. i_meta(n) ) ) ) then

                  xij = x(i,m) - x(j,m)
                  yij = y(i,m) - y(j,m)
                  zij = z(i,m) - z(j,m)

                  call pbc_atom_MPI ( xij, yij, zij )

                  rij = sqrt( xij*xij + yij*yij + zij*zij )

                  cn = cn + ( 1.d0 - (rij/req)**nu ) &
     &                    / ( 1.d0 - (rij/req)**mu )

               end if

            end do
            end do

            smeta(n,m) = cn

!-----------------------------------------------------------------------
!     /*   main loop end                                              */
!-----------------------------------------------------------------------

         end do

      end do

      return
      end





!***********************************************************************
      subroutine get_dcord_meta_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, nbead, natom, ikind, myrank, nprocs

      use meta_variables, only : &
     &   itype_meta, ipbc_meta, req_meta, smeta, &
     &   i_meta, j_meta, k_meta, l_meta, nu_meta, mu_meta, nmeta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: nu1, nu2, mu1, mu2, i, j, m, n

      real(8) :: req1, req2, cn1, cn2, xij, yij, zij, rij

!-----------------------------------------------------------------------
!     /*   main loop start                                            */
!-----------------------------------------------------------------------

      do m = 1, nbead

!        /*   only my bead   */
         if ( mod( m-1, nprocs ) .ne. myrank ) cycle

         do n = 1, nmeta

            if ( itype_meta(n) .ne. 6 )  cycle

!-----------------------------------------------------------------------
!           /*   parameters                                           */
!-----------------------------------------------------------------------

            ipbc_meta(n) = 0

            nu1 = nu_meta(n,1)
            mu1 = mu_meta(n,1)
            nu2 = nu_meta(n,2)
            mu2 = mu_meta(n,2)

            req1 = req_meta(n,1)
            req2 = req_meta(n,2)

!-----------------------------------------------------------------------
!           /*   calculate coordination number                        */
!-----------------------------------------------------------------------

            cn1 = 0.d0
            cn2 = 0.d0

            do i = 1, natom-1
            do j = i+1, natom

               if ( ( ( ikind(i) .eq. i_meta(n) ) .and. &
     &                ( ikind(j) .eq. j_meta(n) ) ) .or. &
     &              ( ( ikind(i) .eq. j_meta(n) ) .and. &
     &                ( ikind(j) .eq. i_meta(n) ) ) ) then

                  xij = x(i,m) - x(j,m)
                  yij = y(i,m) - y(j,m)
                  zij = z(i,m) - z(j,m)

                  call pbc_atom_MPI ( xij, yij, zij )

                  rij = sqrt( xij*xij + yij*yij + zij*zij )

                  cn1 = cn1 + ( 1.d0 - (rij/req1)**nu1 ) &
     &                      / ( 1.d0 - (rij/req1)**mu1 )

               end if

               if ( ( ( ikind(i) .eq. k_meta(n) ) .and. &
     &                ( ikind(j) .eq. l_meta(n) ) ) .or. &
     &              ( ( ikind(i) .eq. l_meta(n) ) .and. &
     &                ( ikind(j) .eq. k_meta(n) ) ) ) then

                  xij = x(i,m) - x(j,m)
                  yij = y(i,m) - y(j,m)
                  zij = z(i,m) - z(j,m)

                  call pbc_atom_MPI ( xij, yij, zij )

                  rij = sqrt( xij*xij + yij*yij + zij*zij )

                  cn2 = cn2 + ( 1.d0 - (rij/req2)**nu2 ) &
     &                      / ( 1.d0 - (rij/req2)**mu2 )

               end if

            end do
            end do

            smeta(n,m) = cn1 - cn2

!-----------------------------------------------------------------------
!     /*   main loop end                                              */
!-----------------------------------------------------------------------

         end do

      end do

      return
      end





!***********************************************************************
      subroutine get_xyz_meta_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, natom, nbead, ikind, mbox, myrank, nprocs

      use meta_variables, only : &
     &   smeta, ipbc_meta, i_meta, j_meta, nmeta, itype_meta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: n, i, m, nm, m1, m2, m3

      real(8) :: xm, ym, zm, xi, yi, zi

!-----------------------------------------------------------------------
!     /*   main loop start                                            */
!-----------------------------------------------------------------------

      do m = 1, nbead

!        /*   only my bead   */
         if ( mod( m-1, nprocs ) .ne. myrank ) cycle

         do n = 1, nmeta

            if ( itype_meta(n) .ne. 7 )  cycle

!-----------------------------------------------------------------------
!           /*   parameters                                           */
!-----------------------------------------------------------------------

            ipbc_meta(n) = 0

!-----------------------------------------------------------------------
!           /*   calculate center of mass                             */
!-----------------------------------------------------------------------

            nm = 0

            xm = 0.d0
            ym = 0.d0
            zm = 0.d0

            smeta(n,m) = 0.d0

            do i = 1, natom

               if ( ikind(i) .eq. j_meta(n) ) then

                  xi = x(i,m)
                  yi = y(i,m)
                  zi = z(i,m)

                  m1 = mbox(1,i,m)
                  m2 = mbox(2,i,m)
                  m3 = mbox(3,i,m)

                  call pbc_unfold_MPI( xi, yi, zi, m1, m2, m3 )

                  nm = nm + 1

                  xm = xm + xi
                  ym = ym + yi
                  zm = zm + zi

               end if

            end do

            xm = xm / dble(nm)
            ym = ym / dble(nm)
            zm = zm / dble(nm)

            if ( i_meta(n) .eq. 1 ) smeta(n,m) = xm
            if ( i_meta(n) .eq. 2 ) smeta(n,m) = ym
            if ( i_meta(n) .eq. 3 ) smeta(n,m) = zm

!-----------------------------------------------------------------------
!     /*   main loop end                                              */
!-----------------------------------------------------------------------

         end do

      end do

      return
      end





!***********************************************************************
      subroutine get_dxyz_meta_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, natom, nbead, ikind, mbox, myrank, nprocs

      use meta_variables, only : &
     &   smeta, ipbc_meta, i_meta, j_meta, k_meta, nmeta, itype_meta

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: n, i, m, nm, nn, m1, m2, m3

      real(8) :: xm, ym, zm, xn, yn, zn, xi, yi, zi

!-----------------------------------------------------------------------
!     /*   main loop start                                            */
!-----------------------------------------------------------------------

      do m = 1, nbead

!        /*   only my bead   */
         if ( mod( m-1, nprocs ) .ne. myrank ) cycle

         do n = 1, nmeta

            if ( itype_meta(n) .ne. 8 )  cycle

!-----------------------------------------------------------------------
!           /*   parameters                                           */
!-----------------------------------------------------------------------

            ipbc_meta(n) = 0

!-----------------------------------------------------------------------
!           /*   calculate difference in center of masses             */
!-----------------------------------------------------------------------

            nm = 0
            nn = 0

            xm = 0.d0
            ym = 0.d0
            zm = 0.d0

            xn = 0.d0
            yn = 0.d0
            zn = 0.d0

            smeta(n,m) = 0.d0

            do i = 1, natom

               if ( ikind(i) .eq. j_meta(n) ) then

                  xi = x(i,m)
                  yi = y(i,m)
                  zi = z(i,m)

                  m1 = mbox(1,i,m)
                  m2 = mbox(2,i,m)
                  m3 = mbox(3,i,m)

                  call pbc_unfold_MPI( xi, yi, zi, m1, m2, m3 )

                  nm = nm + 1

                  xm = xm + xi
                  ym = ym + yi
                  zm = zm + zi

               end if

            end do

            xm = xm / dble(nm)
            ym = ym / dble(nm)
            zm = zm / dble(nm)

            do i = 1, natom

               if ( ikind(i) .eq. k_meta(n) ) then

                  xi = x(i,m)
                  yi = y(i,m)
                  zi = z(i,m)

                  m1 = mbox(1,i,m)
                  m2 = mbox(2,i,m)
                  m3 = mbox(3,i,m)

                  call pbc_unfold_MPI( xi, yi, zi, m1, m2, m3 )

                  nn = nn + 1

                  xn = xn + xi
                  yn = yn + yi
                  zn = zn + zi

               end if

            end do

            xn = xn / dble(nn)
            yn = yn / dble(nn)
            zn = zn / dble(nn)

            if ( i_meta(n) .eq. 1 ) smeta(n,m) = xm - xn
            if ( i_meta(n) .eq. 2 ) smeta(n,m) = ym - yn
            if ( i_meta(n) .eq. 3 ) smeta(n,m) = zm - zn

!-----------------------------------------------------------------------
!     /*   main loop end                                              */
!-----------------------------------------------------------------------

         end do

      end do

      return
      end
