!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jan 23, 2025 by M. Shiga
!      Description:     geometry optimization by LBFGS algorithm
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine geooptcycle_XMPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only: &
     &   x, y, z, fx, fy, fz, potential, au_energy, avogadro, natom, &
     &   nstep, iounit, istep, istep_start, istep_end, iexit, myrank

      use lbfgs_variables, only : &
     &   pos, pos0, grad, dm, ws, dmax_tol, drms_tol, fmax_tol, &
     &   frms_tol, eps, postol, frms, fmax, drms, dmax, &
     &   func, iprint, ndim, nup, iflag

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j, k

      real(8) :: har2kcal, har2kj

!-----------------------------------------------------------------------
!     /*   preparation of parallel MPI                                */
!-----------------------------------------------------------------------

      call prep_XMPI

!-----------------------------------------------------------------------
!     //   initialize step
!-----------------------------------------------------------------------

      istep = istep_start
      istep_end = istep

!-----------------------------------------------------------------------
!     //   new position:  x, y, z  ->  pos
!-----------------------------------------------------------------------

      k = 0

      do j = 1, natom
         k = k + 1
         pos(k) = x(j,1)
         k = k + 1
         pos(k) = y(j,1)
         k = k + 1
         pos(k) = z(j,1)
      end do

!-----------------------------------------------------------------------
!     //   calculate potential and gradients
!-----------------------------------------------------------------------

      call getforce_XMPI

!-----------------------------------------------------------------------
!     /*   communicate atoms for jstart_bead < j < jend_bead          */
!-----------------------------------------------------------------------

      call my_mpi_bcast_xyz_XMPI( fx, fy, fz, 1 )

!-----------------------------------------------------------------------
!     //   potential  ->  func
!-----------------------------------------------------------------------

      func = potential

!-----------------------------------------------------------------------
!     //   force  ->  grad
!-----------------------------------------------------------------------

      call update_force_geoopt_0

!-----------------------------------------------------------------------
!     //   old position: pos0
!-----------------------------------------------------------------------

      pos0(:) = pos(:)

!-----------------------------------------------------------------------
!     //   standard output
!-----------------------------------------------------------------------

      call standard_geoopt_MPI

!-----------------------------------------------------------------------
!     //   terminate if converged
!-----------------------------------------------------------------------

      if ( ( fmax .lt. fmax_tol ) .and. ( frms .lt. frms_tol ) ) then
         return
      end if

!-----------------------------------------------------------------------
!     //   start main loop
!-----------------------------------------------------------------------

      do istep = istep_start+1, nstep

!-----------------------------------------------------------------------
!        //   current step
!-----------------------------------------------------------------------

         istep_end = istep

!-----------------------------------------------------------------------
!        //   old position: pos0
!-----------------------------------------------------------------------

         pos0(:) = pos(:)

!-----------------------------------------------------------------------
!        //   call limited memory bfgs routine
!-----------------------------------------------------------------------

         call lbfgs ( ndim, nup, pos, func, grad, .false., dm, &
     &                iprint, eps, postol, ws, iflag )

!-----------------------------------------------------------------------
!        //   new position: pos  ->  x, y, z
!-----------------------------------------------------------------------

         k = 0

         do j = 1, natom
            k = k + 1
            x(j,1) = pos(k)
            k = k + 1
            y(j,1) = pos(k)
            k = k + 1
            z(j,1) = pos(k)
         end do

!-----------------------------------------------------------------------
!        //   calculate potential and gradients
!-----------------------------------------------------------------------

         call getforce_XMPI

!-----------------------------------------------------------------------
!        /*   communicate atoms for jstart_bead < j < jend_bead       */
!-----------------------------------------------------------------------

         call my_mpi_bcast_xyz_XMPI( fx, fy, fz, 1 )

!-----------------------------------------------------------------------
!        //   potential  ->  func
!-----------------------------------------------------------------------

         func = potential

!-----------------------------------------------------------------------
!        //   force  ->  grad
!-----------------------------------------------------------------------

         call update_force_geoopt_0

!-----------------------------------------------------------------------
!        //   standard output
!-----------------------------------------------------------------------

         call standard_geoopt_MPI

!-----------------------------------------------------------------------
!        //   output restart
!-----------------------------------------------------------------------

         call backup_geoopt_MPI

!-----------------------------------------------------------------------
!        //   terminate if converged
!-----------------------------------------------------------------------

         if ( ( dmax .lt. dmax_tol ) .and. ( drms .lt. drms_tol ) .and. &
     &        ( fmax .lt. fmax_tol ) .and. ( frms .lt. frms_tol ) ) then
            iflag = 0
         end if

         if ( iflag .le. 0 ) then
            iexit = 1
            exit
         end if

!-----------------------------------------------------------------------
!        /*   exit if `exit.dat' exists                               */
!-----------------------------------------------------------------------

         call softexit_MPI

         if ( iexit .eq. 1 ) exit

      end do

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 ) return

!     //   from hartree to kcal/mol
      har2kcal = au_energy / 1000.d0 * avogadro / 4.184d0

!     //   from hartree to kcal/mol
      har2kj   = au_energy / 1000.d0 * avogadro

      write( 6, '(a)' )
      write( 6, '(a)' ) &
     &   '==========================================================' &
     &    // '===================='
      write( 6, '(a)' ) &
     &   '                           potential energy values'
      write( 6, '(6x,a)' ) &
     &   '                 hartree' &
     &    // '                kcal/mol                  kJ/mol'
      write( 6, '(a)' ) &
     &   '----------------------------------------------------------' &
     &    // '--------------------'

      write( 6, '(6x,f24.8,2f24.6)' ) &
     &   potential, potential*har2kcal, potential*har2kj

      write( 6, '(a)' ) &
     &   '----------------------------------------------------------' &
     &    // '--------------------'

      return
      end
