!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     geometry optimization by LBFGS algorithm
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine geooptcycle_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use cons_variables, only : ncons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   constraints                                                */
!-----------------------------------------------------------------------

      if ( ncons .eq. 0 ) then

         call geooptcycle_0_MPI

      else

         call geooptcycle_cons_MPI

      end if

      return
      end





!***********************************************************************
      subroutine geooptcycle_0_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only: &
     &   x, y, z, potential, au_energy, avogadro, natom, &
     &   nstep, iounit, istep, istep_start, istep_end, iexit, myrank

      use lbfgs_variables, only : &
     &   pos, pos0, grad, dm, ws, dmax_tol, drms_tol, fmax_tol, &
     &   frms_tol, eps, postol, frms, fmax, drms, dmax, &
     &   func, iprint, ndim, nup, iflag

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j, k

      real(8) :: har2kcal, har2kj

!-----------------------------------------------------------------------
!     //   initialize step
!-----------------------------------------------------------------------

      istep = istep_start
      istep_end = istep

!-----------------------------------------------------------------------
!     //   new position:  x, y, z  ->  pos
!-----------------------------------------------------------------------

      k = 0

      do j = 1, natom
         k = k + 1
         pos(k) = x(j,1)
         k = k + 1
         pos(k) = y(j,1)
         k = k + 1
         pos(k) = z(j,1)
      end do

!-----------------------------------------------------------------------
!     //   calculate potential and gradients
!-----------------------------------------------------------------------

      call getforce_MPI

!-----------------------------------------------------------------------
!     //   potential  ->  func
!-----------------------------------------------------------------------

      func = potential

!-----------------------------------------------------------------------
!     //   force  ->  grad
!-----------------------------------------------------------------------

      call update_force_geoopt_0

!-----------------------------------------------------------------------
!     //   old position: pos0
!-----------------------------------------------------------------------

      pos0(:) = pos(:)

!-----------------------------------------------------------------------
!     //   standard output
!-----------------------------------------------------------------------

      call standard_geoopt_MPI

!-----------------------------------------------------------------------
!     //   do some analysis
!-----------------------------------------------------------------------

      call analysis_MPI( 1 )

!-----------------------------------------------------------------------
!     //   terminate if converged
!-----------------------------------------------------------------------

      if ( ( fmax .lt. fmax_tol ) .and. ( frms .lt. frms_tol ) ) then
         return
      end if

!-----------------------------------------------------------------------
!     //   start main loop
!-----------------------------------------------------------------------

      do istep = istep_start+1, nstep

!-----------------------------------------------------------------------
!        //   current step
!-----------------------------------------------------------------------

         istep_end = istep

!-----------------------------------------------------------------------
!        //   old position: pos0
!-----------------------------------------------------------------------

         pos0(:) = pos(:)

!-----------------------------------------------------------------------
!        //   call limited memory bfgs routine
!-----------------------------------------------------------------------

         call lbfgs ( ndim, nup, pos, func, grad, .false., dm, &
     &                iprint, eps, postol, ws, iflag )

!-----------------------------------------------------------------------
!        //   new position: pos  ->  x, y, z
!-----------------------------------------------------------------------

         k = 0

         do j = 1, natom
            k = k + 1
            x(j,1) = pos(k)
            k = k + 1
            y(j,1) = pos(k)
            k = k + 1
            z(j,1) = pos(k)
         end do

!-----------------------------------------------------------------------
!        //   calculate potential and gradients
!-----------------------------------------------------------------------

         call getforce_MPI

!-----------------------------------------------------------------------
!        //   potential  ->  func
!-----------------------------------------------------------------------

         func = potential

!-----------------------------------------------------------------------
!        //   force  ->  grad
!-----------------------------------------------------------------------

         call update_force_geoopt_0

!-----------------------------------------------------------------------
!        //   standard output
!-----------------------------------------------------------------------

         call standard_geoopt_MPI

!-----------------------------------------------------------------------
!        //   output restart
!-----------------------------------------------------------------------

         call backup_geoopt_MPI

!-----------------------------------------------------------------------
!        //   do some analysis
!-----------------------------------------------------------------------

         call analysis_MPI( 2 )

!-----------------------------------------------------------------------
!        //   terminate if converged
!-----------------------------------------------------------------------

         if ( ( dmax .lt. dmax_tol ) .and. ( drms .lt. drms_tol ) .and. &
     &        ( fmax .lt. fmax_tol ) .and. ( frms .lt. frms_tol ) ) then
            iflag = 0
         end if

         if ( iflag .le. 0 ) then
            iexit = 1
            exit
         end if

!-----------------------------------------------------------------------
!        /*   exit if `exit.dat' exists                               */
!-----------------------------------------------------------------------

         call softexit_MPI

         if ( iexit .eq. 1 ) exit

      end do

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 ) return

!     //   from hartree to kcal/mol
      har2kcal = au_energy / 1000.d0 * avogadro / 4.184d0

!     //   from hartree to kcal/mol
      har2kj   = au_energy / 1000.d0 * avogadro

      write( 6, '(a)' )
      write( 6, '(a)' ) &
     &   '==========================================================' &
     &    // '===================='
      write( 6, '(a)' ) &
     &   '                           potential energy values'
      write( 6, '(6x,a)' ) &
     &   '                 hartree' &
     &    // '                kcal/mol                  kJ/mol'
      write( 6, '(a)' ) &
     &   '----------------------------------------------------------' &
     &    // '--------------------'

      write( 6, '(6x,f24.8,2f24.6)' ) &
     &   potential, potential*har2kcal, potential*har2kj

      write( 6, '(a)' ) &
     &   '----------------------------------------------------------' &
     &    // '--------------------'

      return
      end





!***********************************************************************
      subroutine geooptcycle_cons_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only: &
     &   x, y, z, potential, &
     &   au_energy, avogadro, natom, nstep, iounit, istep, istep_start, &
     &   istep_end, iexit, myrank

      use lbfgs_variables, only : &
     &   pos, pos0, grad, dm, ws, dmax_tol, drms_tol, fmax_tol, &
     &   frms_tol, eps, postol, frms, fmax, drms, dmax, &
     &   func, iprint, ndim, nup, iflag

      use cons_variables, only : &
     &   fc_cons, gc_cons, fref_cons, rcons, scons, diter_cons, &
     &   pot_ref_cons, ncons, iter_cons, niter_cons

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k

      real(8) :: har2kcal, har2kj

!-----------------------------------------------------------------------
!     //   initialize step
!-----------------------------------------------------------------------

      istep = istep_start
      istep_end = istep

!-----------------------------------------------------------------------
!     //   new position:  x, y, z  ->  pos
!-----------------------------------------------------------------------

      k = 0

      do j = 1, natom
         k = k + 1
         pos(k) = x(j,1)
         k = k + 1
         pos(k) = y(j,1)
         k = k + 1
         pos(k) = z(j,1)
      end do

!-----------------------------------------------------------------------
!     //   calculate potential and gradients
!-----------------------------------------------------------------------

      call getforce_MPI

      call getforce_ref_cons_geoopt_MPI

!-----------------------------------------------------------------------
!     //   potential  ->  func
!-----------------------------------------------------------------------

      func = potential + pot_ref_cons(1)

!-----------------------------------------------------------------------
!     //   force  ->  grad
!-----------------------------------------------------------------------

      call update_force_geoopt_cons

!-----------------------------------------------------------------------
!     //   old position: pos0
!-----------------------------------------------------------------------

      pos0(:) = pos(:)

!-----------------------------------------------------------------------
!     //   standard output
!-----------------------------------------------------------------------

      call standard_geoopt_MPI

!-----------------------------------------------------------------------
!     //   do some analysis
!-----------------------------------------------------------------------

      call analysis_MPI( 1 )

!-----------------------------------------------------------------------
!     //   terminate if converged
!-----------------------------------------------------------------------

      if ( ( fmax .lt. fmax_tol ) .and. ( frms .lt. frms_tol ) ) then
         return
      end if

!-----------------------------------------------------------------------
!     //   start main loop
!-----------------------------------------------------------------------

      do istep = istep_start+1, nstep

!-----------------------------------------------------------------------
!        //   current step
!-----------------------------------------------------------------------

         istep_end = istep

!-----------------------------------------------------------------------
!        //   old position: pos0
!-----------------------------------------------------------------------

         pos0(:) = pos(:)

!-----------------------------------------------------------------------
!        //   call limited memory bfgs routine
!-----------------------------------------------------------------------

         call lbfgs ( ndim, nup, pos, func, grad, .false., dm, &
     &                iprint, eps, postol, ws, iflag )

!-----------------------------------------------------------------------
!        //   new position: pos  ->  x, y, z
!-----------------------------------------------------------------------

         k = 0

         do j = 1, natom
            k = k + 1
            x(j,1) = pos(k)
            k = k + 1
            y(j,1) = pos(k)
            k = k + 1
            z(j,1) = pos(k)
         end do

!-----------------------------------------------------------------------
!        //   calculate potential and gradients
!-----------------------------------------------------------------------

         call getforce_MPI

         call getforce_ref_cons_geoopt_MPI

!-----------------------------------------------------------------------
!        //   potential  ->  func
!-----------------------------------------------------------------------

         func = potential + pot_ref_cons(1)

!-----------------------------------------------------------------------
!        //   force  ->  grad
!-----------------------------------------------------------------------

         call update_force_geoopt_cons

!-----------------------------------------------------------------------
!        //   standard output
!-----------------------------------------------------------------------

         call standard_geoopt_MPI

!-----------------------------------------------------------------------
!        //   output restart
!-----------------------------------------------------------------------

         call backup_geoopt_MPI

!-----------------------------------------------------------------------
!        //   do some analysis
!-----------------------------------------------------------------------

         call analysis_MPI( 2 )

!-----------------------------------------------------------------------
!        //   terminate if converged
!-----------------------------------------------------------------------

         if ( ( dmax .lt. dmax_tol ) .and. ( drms .lt. drms_tol ) .and. &
     &        ( fmax .lt. fmax_tol ) .and. ( frms .lt. frms_tol ) ) then

            iter_cons = iter_cons + 1

            if ( iter_cons .ge. niter_cons ) then

               iflag = 0

            else

               fc_cons(:) = fc_cons(:) * diter_cons
               gc_cons(:) = gc_cons(:) + fref_cons(:,1)

               call getforce_ref_cons_geoopt_MPI

               func = potential + pot_ref_cons(1)

               call update_force_geoopt_cons

               iflag = 0

               call lbfgs ( ndim, nup, pos, func, grad, .false., dm, &
     &                      iprint, eps, postol, ws, iflag )

            end if

         end if

         if ( iflag .le. 0 ) then
            iexit = 1
            exit
         end if

!-----------------------------------------------------------------------
!        /*   exit if `exit.dat' exists                               */
!-----------------------------------------------------------------------

         call softexit_MPI

         if ( iexit .eq. 1 ) exit

      end do

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 ) return

!     //   from hartree to kcal/mol
      har2kcal = au_energy / 1000.d0 * avogadro / 4.184d0

!     //   from hartree to kcal/mol
      har2kj   = au_energy / 1000.d0 * avogadro

      write( 6, '(a)' )
      write( 6, '(a)' ) &
     &   '==========================================================' &
     &    // '===================='
      write( 6, '(a)' ) &
     &   '                           potential energy values'
      write( 6, '(6x,a)' ) &
     &   '                 hartree' &
     &    // '                kcal/mol                  kJ/mol'
      write( 6, '(a)' ) &
     &   '----------------------------------------------------------' &
     &    // '--------------------'

      write( 6, '(6x,f24.8,2f24.6)' ) &
     &   potential, potential*har2kcal, potential*har2kj

      write( 6, '(a)' ) &
     &   '----------------------------------------------------------' &
     &    // '--------------------'

      write( 6, '(a)' )
      write( 6, '(a)' ) &
     &   '==========================================================' &
     &    // '===================='
      write( 6, '(a)' ) &
     &   '                           constraints'
      write( 6, '(a)' ) &
     &   '   num                  target                  actual'
      write( 6, '(a)' ) &
     &   '----------------------------------------------------------' &
     &    // '--------------------'
      do i = 1, ncons
         write( 6, '(i6,2f24.6)' ) i, rcons(i,1), scons(i,1)
      end do
      write( 6, '(a)' ) &
     &   '----------------------------------------------------------' &
     &    // '--------------------'

      return
      end





!***********************************************************************
      subroutine update_force_geoopt_0
!***********************************************************************
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fx, fy, fz, atom_change, natom

      use lbfgs_variables, only : &
     &   grad

      implicit none

      integer :: j, k

!-----------------------------------------------------------------------

      k = 0

      do j = 1, natom

         if ( ( atom_change(j)(1:6) .eq. 'FREEZE' ) .or. &
     &        ( atom_change(j)(1:5) .eq. 'HEAVY'  ) .or. &
     &        ( atom_change(j)(1:6) .eq. 'FIXXYZ' ) ) then

            k = k + 1
            grad(k) = 0.d0
            k = k + 1
            grad(k) = 0.d0
            k = k + 1
            grad(k) = 0.d0

         else

            if      ( atom_change(j)(1:5) .eq. 'FIXX ' ) then
               k = k + 1
               grad(k) = 0.d0
            else if ( atom_change(j)(1:5) .eq. 'FIXXY' ) then
               k = k + 1
               grad(k) = 0.d0
            else if ( atom_change(j)(1:5) .eq. 'FIXXZ' ) then
               k = k + 1
               grad(k) = 0.d0
            else
               k = k + 1
               grad(k) = - fx(j,1)
            end if

            if      ( atom_change(j)(1:5) .eq. 'FIXY ' ) then
               k = k + 1
               grad(k) = 0.d0
            else if ( atom_change(j)(1:5) .eq. 'FIXXY' ) then
               k = k + 1
               grad(k) = 0.d0
            else if ( atom_change(j)(1:5) .eq. 'FIXYZ' ) then
               k = k + 1
               grad(k) = 0.d0
            else
               k = k + 1
               grad(k) = - fy(j,1)
            end if

            if      ( atom_change(j)(1:5) .eq. 'FIXZ ' ) then
               k = k + 1
               grad(k) = 0.d0
            else if ( atom_change(j)(1:5) .eq. 'FIXXZ' ) then
               k = k + 1
               grad(k) = 0.d0
            else if ( atom_change(j)(1:5) .eq. 'FIXYZ' ) then
               k = k + 1
               grad(k) = 0.d0
            else
               k = k + 1
               grad(k) = - fz(j,1)
            end if

         end if

      end do

      return
      end





!***********************************************************************
      subroutine update_force_geoopt_cons
!***********************************************************************
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fx, fy, fz, fux_ref, fuy_ref, fuz_ref, atom_change, natom

      use lbfgs_variables, only : &
     &   grad

      implicit none

      integer :: j, k

!-----------------------------------------------------------------------

      k = 0

      do j = 1, natom

         if ( ( atom_change(j)(1:6) .eq. 'FREEZE' ) .or. &
     &        ( atom_change(j)(1:5) .eq. 'HEAVY'  ) .or. &
     &        ( atom_change(j)(1:6) .eq. 'FIXXYZ' ) ) then

            k = k + 1
            grad(k) = 0.d0
            k = k + 1
            grad(k) = 0.d0
            k = k + 1
            grad(k) = 0.d0

         else

            if      ( atom_change(j)(1:5) .eq. 'FIXX ' ) then
               k = k + 1
               grad(k) = 0.d0
            else if ( atom_change(j)(1:5) .eq. 'FIXXY' ) then
               k = k + 1
               grad(k) = 0.d0
            else if ( atom_change(j)(1:5) .eq. 'FIXXZ' ) then
               k = k + 1
               grad(k) = 0.d0
            else
               k = k + 1
               grad(k) = - fx(j,1) - fux_ref(j,1)
            end if

            if      ( atom_change(j)(1:5) .eq. 'FIXY ' ) then
               k = k + 1
               grad(k) = 0.d0
            else if ( atom_change(j)(1:5) .eq. 'FIXXY' ) then
               k = k + 1
               grad(k) = 0.d0
            else if ( atom_change(j)(1:5) .eq. 'FIXYZ' ) then
               k = k + 1
               grad(k) = 0.d0
            else
               k = k + 1
               grad(k) = - fy(j,1) - fuy_ref(j,1)
            end if

            if      ( atom_change(j)(1:5) .eq. 'FIXZ ' ) then
               k = k + 1
               grad(k) = 0.d0
            else if ( atom_change(j)(1:5) .eq. 'FIXXZ' ) then
               k = k + 1
               grad(k) = 0.d0
            else if ( atom_change(j)(1:5) .eq. 'FIXYZ' ) then
               k = k + 1
               grad(k) = 0.d0
            else
               k = k + 1
               grad(k) = - fz(j,1) - fuz_ref(j,1)
            end if

         end if

      end do

      return
      end
