!///////////////////////////////////////////////////////////////////////
!
!      Author:          Y. Ota, M. Shiga
!      Last updated:    Sep 17, 2022 by M. Shiga
!      Description:     gentlest ascent dynamics
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      module gad_variables
!***********************************************************************

!     //   gamma value in gentlest ascent dynamics
      real(8) :: gamma_gad

!     //   time step
      real(8) :: dt_gad

!     //   unit vector of current step
      real(8), dimension(:), allocatable :: e_gad

!     //   direction of current step
      real(8), dimension(:), allocatable :: d_gad

!     //   unit vector of previous step
      real(8), dimension(:), allocatable :: eold_gad

!     //   direction of previous step
      real(8), dimension(:), allocatable :: dold_gad

!     //   shift vector
      real(8), dimension(:), allocatable :: he_gad

!     /*   norm of shift vector   */
      real(8) :: hehe_gad

!     /*   inner product of unit vector and shift vector   */
      real(8) :: ehe_gad

!     /*   angle between of d_gad and dold_gad   */
      real(8) :: theta_gad = 0.d0

!     /*   angle between of e_gad and eold_gad   */
      real(8) :: phi_gad = 0.d0

!     /*   convergence   */
      real(8) :: dt_conv_gad = 1.d-5

!***********************************************************************
      end module gad_variables
!***********************************************************************





!***********************************************************************
      subroutine gadcycle
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   istep, istep_start, istep_end, nstep, iexit

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     //   starting step
      istep = istep_start

!     //   current step
      istep_end = istep

!     /*   get interatomic forces   */
      call getforce_gad

!     /*   standard output   */
      call standard_gad

!     /*   do some analysis   */
      call analysis_gad ( 1 )

!     /*   backup   */
      call backup_gad

!-----------------------------------------------------------------------
!     /*   start main loop                                            */
!-----------------------------------------------------------------------

      do istep = istep_start+1, nstep

!        /*   current step   */
         istep_end = istep

!        /*   update velocity   */
         call vupdate_ascent_gad

!        /*   update guiding vector   */
         call eupdate_ascent_gad

!        /*   update position for gad   */
         call rupdate_gad

!        /*   get hessian along guiding vector   */
         call getforce_gad

!        /*   standard output   */
         call standard_gad

!        /*   do some analysis   */
         call analysis_gad ( 2 )

!        /*   adjust step size   */
         call adjust_ascent_gad
         if ( iexit .eq. 1 ) exit

!        /*   backup   */
         call backup_gad

!        /*   exit if `exit.dat' exists   */
         call softexit
         if ( iexit .eq. 1 ) exit

!-----------------------------------------------------------------------
!     /*   end of main loop                                           */
!-----------------------------------------------------------------------

      end do

!     /*   current step   */
      istep = istep_end

      return
      end





!***********************************************************************
      subroutine setup_gad
!***********************************************************************
!=======================================================================
!
!     set up gentlest ascent dynamics
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ux, uy, uz, x, y, z, fdiff, au_time, ipos_start, nstep, &
     &   iounit, natom

      use gad_variables, only : &
     &   gamma_gad, e_gad, d_gad, he_gad, dold_gad, eold_gad, &
     &   dt_conv_gad

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: itest

!-----------------------------------------------------------------------
!     /*   set up atomic positions and velocities                     */
!-----------------------------------------------------------------------

!     /*   centroid coordinates are read from file                    */

      if     ( ipos_start .eq. 0 ) then

         call init_centroid

         x(:,1) = ux(:,1)
         y(:,1) = uy(:,1)
         z(:,1) = uz(:,1)

      else if ( ipos_start .eq. 1 ) then

         call restart_cartesian( 3 )

         ux(:,1) = x(:,1)
         uy(:,1) = y(:,1)
         uz(:,1) = z(:,1)

      else

         call error_handling( 1, 'subroutine setup_gad', 20 )

      end if

!-----------------------------------------------------------------------
!     //   finite difference parameter
!-----------------------------------------------------------------------

!     //   finite difference parameter
      call read_real1 ( fdiff, '<fdiff>', 7, iounit )

!     //   read gamma value
      call read_real1 ( gamma_gad, '<gamma_gad>', 11, iounit )

!     //   read dt convergence threshold
      call read_real1 ( dt_conv_gad, '<dt_conv_gad>', 13, iounit )

!-----------------------------------------------------------------------
!     //   convert units from fs to au
!-----------------------------------------------------------------------

      dt_conv_gad = dt_conv_gad / ( au_time * 1.d+15 )

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

!     //   guiding unit vector of current step
      if ( .not. allocated( e_gad ) ) allocate( e_gad(3*natom) )

!     //   guiding unit vector of previous step
      if ( .not. allocated( eold_gad ) ) allocate( eold_gad(3*natom) )

!     //   direction of current step
      if ( .not. allocated( d_gad ) ) allocate( d_gad(3*natom) )

!     //   direction of previous step
      if ( .not. allocated( dold_gad ) ) allocate( dold_gad(3*natom) )

!     //   shift vector
      if ( .not. allocated( he_gad ) ) allocate( he_gad(3*natom) )

!-----------------------------------------------------------------------
!     //   initialize or restart
!-----------------------------------------------------------------------

!     //   check existence of restart file
      call testfile ( 'gad.ini', 7, itest )

!     //   restart file: not found
      if ( itest .eq. 1 ) then

!        //   initialize
         call init_gad

!     //   restart file: found
      else

!        //   restart
         call restart_gad( 1 )

!     //   restart file
      end if

!-----------------------------------------------------------------------
!     /*   terminate the run if nstep = 0                             */
!-----------------------------------------------------------------------

      if ( nstep .eq. 0 ) then

         stop

      end if

      return
      end





!***********************************************************************
      subroutine init_gad
!***********************************************************************
!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : natom, iounit, dt

      use gad_variables, only : e_gad, dt_gad

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ierr

!-----------------------------------------------------------------------
!     //   read guiding vector
!-----------------------------------------------------------------------

      open ( iounit, file = 'eigen.dat' )

      read ( iounit, *, iostat=ierr )

      do i = 1, 3*natom, 3
         read ( iounit, *, iostat=ierr ) e_gad(i+0:i+2)
      end do

      close( iounit )

!-----------------------------------------------------------------------
!     //   error check
!-----------------------------------------------------------------------

      call error_handling( ierr, 'subroutine init_gad', 19 )

!-----------------------------------------------------------------------
!     //   initial step size
!-----------------------------------------------------------------------

      dt_gad = dt

      return
      end





!***********************************************************************
      subroutine restart_gad( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : natom, iounit

      use gad_variables, only : e_gad, dt_gad, d_gad

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption, ierr

!-----------------------------------------------------------------------
!     //   read guiding vector, step size
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         open ( iounit, file = 'gad.ini' )

         read( iounit, *, iostat=ierr ) dt_gad

         do i = 1, 3*natom
            read( iounit, *, iostat=ierr ) e_gad(i)
         end do

         do i = 1, 3*natom
            read( iounit, *, iostat=ierr ) d_gad(i)
         end do

         close( iounit )

!        //   error check
         call error_handling( ierr, 'subroutine restart_gad', 22 )

      end if

!-----------------------------------------------------------------------
!     //   write guiding vector, step size
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         open ( iounit, file = 'gad.ini' )

         write( iounit, '(3e24.16)' ) dt_gad

         do i = 1, 3*natom
            write( iounit, '(3e24.16)' ) e_gad(i)
         end do

         do i = 1, 3*natom
            write( iounit, '(3e24.16)' ) d_gad(i)
         end do

         close( iounit )

      end if

      return
      end





!***********************************************************************
      subroutine getforce_gad
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, x, y, z, fx, fy, fz, fux, fuy, fuz, fdiff, potential, &
     &   pot, natom

      use gad_variables, only : &
     &   e_gad, he_gad

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: fdiff_gad, fd, potu, factor

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      fd = 0.d0

      do i = 1, 3*natom, 3
         j = (i+2) / 3
         factor = sqrt(physmass(j))
         fd = max( fd, abs( e_gad(i+0) / factor ) )
         fd = max( fd, abs( e_gad(i+1) / factor ) )
         fd = max( fd, abs( e_gad(i+2) / factor ) )
      end do

      fdiff_gad = fdiff / fd

!-----------------------------------------------------------------------
!     /*   force at minus position                                    */
!-----------------------------------------------------------------------

      do i = 1, 3*natom, 3
         j = (i+2) / 3
         factor = fdiff_gad / sqrt(physmass(j))
         x(j,1) = x(j,1) - e_gad(i+0) * factor
         y(j,1) = y(j,1) - e_gad(i+1) * factor
         z(j,1) = z(j,1) - e_gad(i+2) * factor
      end do

      call getforce

      fux(:,1) = fx(:,1)
      fuy(:,1) = fy(:,1)
      fuz(:,1) = fz(:,1)

      potu = pot(1)

!-----------------------------------------------------------------------
!     /*   force at plus position                                     */
!-----------------------------------------------------------------------

      do i = 1, 3*natom, 3
         j = (i+2) / 3
         factor = fdiff_gad / sqrt(physmass(j))
         x(j,1) = x(j,1) + 2.d0 * e_gad(i+0) * factor
         y(j,1) = y(j,1) + 2.d0 * e_gad(i+1) * factor
         z(j,1) = z(j,1) + 2.d0 * e_gad(i+2) * factor
      end do

      call getforce

!-----------------------------------------------------------------------
!     /*   shift in original position                                 */
!-----------------------------------------------------------------------

      do i = 1, 3*natom, 3
         j = (i+2) / 3
         factor = fdiff_gad / sqrt(physmass(j))
         x(j,1) = x(j,1) - e_gad(i+0) * factor
         y(j,1) = y(j,1) - e_gad(i+1) * factor
         z(j,1) = z(j,1) - e_gad(i+2) * factor
      end do

!-----------------------------------------------------------------------
!     /*   hessian                                                    */
!-----------------------------------------------------------------------

      do i = 1, 3*natom, 3
         j = (i+2) / 3
         factor = 1.d0 / (2.d0*fdiff_gad) / sqrt(physmass(j))
         he_gad(i+0) = - ( fx(j,1) - fux(j,1) ) * factor
         he_gad(i+1) = - ( fy(j,1) - fuy(j,1) ) * factor
         he_gad(i+2) = - ( fz(j,1) - fuz(j,1) ) * factor
      end do

!-----------------------------------------------------------------------
!     /*   average potential and average force                        */
!-----------------------------------------------------------------------

      potential = 0.5d0 * ( potu + pot(1) )

      pot(1) = potential

      do j = 1, natom
         fx(j,1) = 0.5d0 * ( fx(j,1) + fux(j,1) )
         fy(j,1) = 0.5d0 * ( fy(j,1) + fuy(j,1) )
         fz(j,1) = 0.5d0 * ( fz(j,1) + fuz(j,1) )
      end do

!-----------------------------------------------------------------------
!     /*   debugging                                                  */
!-----------------------------------------------------------------------

!cc      call debug_hessian_gad

!-----------------------------------------------------------------------
!     /*   correct hessian                                            */
!-----------------------------------------------------------------------

      call correct_hessian_gad

      return
      end





!***********************************************************************
      subroutine debug_hessian_gad
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   physmass, hessian, natom

      use gad_variables, only : &
     &   e_gad, he_gad

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l

      real(8) :: he, h0, h1, h2

!-----------------------------------------------------------------------
!     /*   compute hessian                                            */
!-----------------------------------------------------------------------

      call gethess_fdiff

!-----------------------------------------------------------------------
!     /*   hessian check                                              */
!-----------------------------------------------------------------------

      do i = 1, 3*natom

         j = (i+2) / 3

         he = 0.d0

         do k = 1, 3*natom, 3

            l = (k+2) / 3

            h0 = hessian(i,k+0,1) / sqrt(physmass(j)*physmass(l))
            h1 = hessian(i,k+1,1) / sqrt(physmass(j)*physmass(l))
            h2 = hessian(i,k+2,1) / sqrt(physmass(j)*physmass(l))

            he = he + h0*e_gad(k+0) + h1*e_gad(k+1) + h2*e_gad(k+2)

         end do

         write( 6, '(i5,2f24.16)' ) i, he, he_gad(i)

      end do

      stop
      end





!***********************************************************************
      subroutine rupdate_gad
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : x, y, z, vx, vy, vz, natom

      use gad_variables, only : dt_gad

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i

!-----------------------------------------------------------------------
!     /*   update position                                            */
!-----------------------------------------------------------------------

      do i = 1, natom
         x(i,1) = x(i,1) + vx(i,1) * dt_gad
         y(i,1) = y(i,1) + vy(i,1) * dt_gad
         z(i,1) = z(i,1) + vz(i,1) * dt_gad
      end do

      return
      end





!***********************************************************************
      subroutine vupdate_ascent_gad
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fx, fy, fz, vx, vy, vz, physmass, pi, natom, istep

      use gad_variables, only : d_gad, e_gad, dold_gad, theta_gad

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: fe, absd, sqm

!-----------------------------------------------------------------------
!     /*   save d vector                                              */
!-----------------------------------------------------------------------

      if ( istep .eq. 0 ) then

         do i = 1, 3*natom
            dold_gad(i) = 0.d0
         end do

      else

         do i = 1, 3*natom
            dold_gad(i) = d_gad(i)
         end do

      end if

!-----------------------------------------------------------------------
!     /*   gentlest ascent: force * guiding vector                    */
!-----------------------------------------------------------------------

      fe = 0.d0

      do i = 1, 3*natom, 3

         j = (i+2) / 3

         sqm = sqrt(physmass(j))

         fe = fe + fx(j,1) / sqm * e_gad(i+0)
         fe = fe + fy(j,1) / sqm * e_gad(i+1)
         fe = fe + fz(j,1) / sqm * e_gad(i+2)

      end do

!-----------------------------------------------------------------------
!     /*   gentlest ascent: ascent force vector                       */
!-----------------------------------------------------------------------

      do i = 1, 3*natom, 3

         j = (i+2) / 3

         sqm = sqrt(physmass(j))

         d_gad(i+0) = fx(j,1) / sqm - 2.d0 * fe * e_gad(i+0)
         d_gad(i+1) = fy(j,1) / sqm - 2.d0 * fe * e_gad(i+1)
         d_gad(i+2) = fz(j,1) / sqm - 2.d0 * fe * e_gad(i+2)

      end do

!-----------------------------------------------------------------------
!     /*   gentlest ascent: (normalize ascent force)                  */
!-----------------------------------------------------------------------

      absd = 0.d0

      do i = 1, 3*natom
         absd = absd + d_gad(i) * d_gad(i)
      end do

      absd = sqrt( absd )

      do i = 1, 3*natom
         d_gad(i) = d_gad(i) / absd
      end do

!-----------------------------------------------------------------------
!     /*   theta angle                                                */
!-----------------------------------------------------------------------

      if ( istep .le. 1 ) then

         theta_gad = 0.d0

      else

         absd = 0.d0

         do i = 1, 3*natom
            absd = absd + d_gad(i) * dold_gad(i)
         end do

         absd = min(max(-1.d0,absd),1.d0)

         theta_gad = acos( absd ) * 180.d0 / pi

      end if

!-----------------------------------------------------------------------
!     /*   velocity update                                            */
!-----------------------------------------------------------------------

      do i = 1, 3*natom, 3

         j = (i+2) / 3

         sqm = sqrt(physmass(j))

         vx(j,1) = d_gad(i+0) / sqm
         vy(j,1) = d_gad(i+1) / sqm
         vz(j,1) = d_gad(i+2) / sqm

      end do

!-----------------------------------------------------------------------
!     /*   correct velocity                                           */
!-----------------------------------------------------------------------

      call correct_velocity_gad

      return
      end





!***********************************************************************
      subroutine eupdate_ascent_gad
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, natom, istep

      use gad_variables, only : &
     &   he_gad, ehe_gad, hehe_gad, gamma_gad, e_gad, eold_gad, dt_gad, &
     &   phi_gad

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i

      real(8) :: const, c, absd

!-----------------------------------------------------------------------
!     /*   save d vector                                              */
!-----------------------------------------------------------------------

      if ( istep .eq. 0 ) then

         do i = 1, 3*natom
            eold_gad(i) = 0.d0
         end do

      else

         do i = 1, 3*natom
            eold_gad(i) = e_gad(i)
         end do

      end if

!-----------------------------------------------------------------------
!     /*   inner product of e and he vectors                          */
!-----------------------------------------------------------------------

      ehe_gad = 0.d0

      do i = 1, 3*natom
         ehe_gad = ehe_gad + e_gad(i) * he_gad(i)
      end do

!-----------------------------------------------------------------------
!     /*   inner product of he and he vectors                         */
!-----------------------------------------------------------------------

      hehe_gad = 0.d0

      do i = 1, 3*natom
         hehe_gad = hehe_gad + he_gad(i) * he_gad(i)
      end do

!-----------------------------------------------------------------------
!     /*   constant                                                   */
!-----------------------------------------------------------------------

      const = dt_gad / gamma_gad

!-----------------------------------------------------------------------
!     /*   update guiding vector                                      */
!-----------------------------------------------------------------------

      do i = 1, 3*natom
         e_gad(i) = e_gad(i) - const * he_gad(i)
      end do

!-----------------------------------------------------------------------
!     /*   normalize guiding vector                                   */
!-----------------------------------------------------------------------

      c = 0.d0

      do i = 1, 3*natom
         c = c + e_gad(i) * e_gad(i)
      end do

      c = sqrt( c )

      do i = 1, 3*natom
         e_gad(i) = e_gad(i) / c
      end do

!-----------------------------------------------------------------------
!     /*   phi angle                                                  */
!-----------------------------------------------------------------------

      absd = 0.d0

      do i = 1, 3*natom
         absd = absd + e_gad(i) * eold_gad(i)
      end do

      absd = min(max(-1.d0,absd),1.d0)

      phi_gad = acos( absd ) * 180.d0 / pi

      return
      end





!***********************************************************************
      subroutine standard_gad
!***********************************************************************
!=======================================================================
!
!     standard output for gentlest ascent dynamics.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ekin, potential, hamiltonian, hamiltonian_sys, au_time, &
     &   istep, iprint_std, iounit, iounit_std, char_date

      use gad_variables, only : &
     &   theta_gad, dt_gad, phi_gad

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: itest

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call read_int1 ( iprint_std, '<iprint_std>', 12, iounit )

         iset = 1

         call testfile ( 'standard.out', 12, itest )

         if ( itest .eq. 1 ) then

            open ( iounit_std, file = 'standard.out')

            write( iounit_std,'(a)' ) &
     &         '==================================================' // &
     &         '==========================='
            write( iounit_std,'(a)' ) &
     &         '    step  potential [au]  theta    phi       dt   ' // &
     &         'wall clock time           '
            write( iounit_std, '(a)' ) &
     &         '--------------------------------------------------' // &
     &         '---------------------------'

            flush( iounit_std )

            write( 6, '(a)' ) &
     &         '==================================================' // &
     &         '==========================='
            write( 6, '(a)' ) &
     &         '    step  potential [au]  theta    phi       dt   ' // &
     &         'wall clock time           '
            write( 6, '(a)' ) &
     &         '--------------------------------------------------' // &
     &         '---------------------------'

            flush( 6 )

         else

            open( iounit_std, file = 'standard.out', access = 'append' )

            write( 6, '(a)' ) &
     &         '==================================================' // &
     &         '==========================='
            write( 6, '(a)' ) &
     &         '    step  potential [au]  theta    phi       dt   ' // &
     &         'wall clock time           '
            write( 6, '(a)' ) &
     &         '--------------------------------------------------' // &
     &         '---------------------------'

            flush( 6 )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   temp  =  instantaneous temperature                         */
!-----------------------------------------------------------------------

!     /*   calculate ekin =  fictitious kinetic energy   */
      ekin = 0.d0

!-----------------------------------------------------------------------
!     /*   hamiltonian_sys  =  system Hamiltonian                     */
!-----------------------------------------------------------------------

      hamiltonian_sys = ekin + potential

!-----------------------------------------------------------------------
!     /*   hamiltonian =  total Hamiltonian                           */
!-----------------------------------------------------------------------

      hamiltonian = hamiltonian_sys

!-----------------------------------------------------------------------
!     /*   output                                                     */
!-----------------------------------------------------------------------

      if ( iprint_std .le. 0 ) return

      if ( mod(istep,iprint_std) .eq. 0 ) then

!        /*   wall clock time   */
         call getdate

!        /*   output   */
         write( iounit_std,'(i8,f16.8,2f7.2,f9.5,2x,a28)' ) &
     &      istep, potential, theta_gad, phi_gad, &
     &      dt_gad*au_time*1.d+15, char_date

         flush( iounit_std )

!        /*   output   */
         write( 6, '(i8,f16.8,2f7.2,f9.5,2x,a28)' ) &
     &      istep, potential, theta_gad, phi_gad, &
     &      dt_gad*au_time*1.d+15, char_date

         flush( 6 )

      end if

      return
      end





!***********************************************************************
      subroutine backup_gad
!***********************************************************************
!=======================================================================
!
!     finalize the calculation.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   istep_end, iprint_rest, iounit, iexit, nstep

      implicit none

!-----------------------------------------------------------------------
!     /*   conditions                                                 */
!-----------------------------------------------------------------------

      if ( mod(istep_end,iprint_rest) .eq. 0 ) then
         if ( iprint_rest .le. 0 ) then
            return
         else
            continue
         end if
      else if ( istep_end .eq. nstep ) then
         continue
      else if ( iexit .eq. 1 ) then
         continue
      else
         return
      end if

!-----------------------------------------------------------------------
!     /*   write out restart file                                     */
!-----------------------------------------------------------------------

!     /*   system position   */
      call restart_cartesian( 4 )

!-----------------------------------------------------------------------
!     /*   write out restart file for gad                             */
!-----------------------------------------------------------------------

      call restart_gad( 2 )

!-----------------------------------------------------------------------
!     /*   in `step.ini', print the step number for restart           */
!-----------------------------------------------------------------------

      open ( iounit, file = 'step.ini' )
         write ( iounit, '(i8)' ) istep_end
      close( iounit )

!-----------------------------------------------------------------------
!     /*   print final geometry                                       */
!-----------------------------------------------------------------------

      call print_final_xyz

      return
      end





!***********************************************************************
      subroutine correct_velocity_gad
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   vx, vy, vz, x, y, z, physmass, natom, iboundary

      use mech_variables, only: &
     &   nfreeze_mech

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i

      real(8) :: sump, sumx, sumy, sumz, xg, yg, zg, fm

      real(8), dimension(3,3) :: a, c

      real(8), dimension(3)   :: b, d, e, f

      real(8) :: tiny = 1.d-14

!-----------------------------------------------------------------------
!     /*   skip if frozen atoms exist                                 */
!-----------------------------------------------------------------------

      if ( natom .eq. 1 ) return
      if ( nfreeze_mech .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   remove net force                                           */
!-----------------------------------------------------------------------

      sump = 0.d0
      sumx = 0.d0
      sumy = 0.d0
      sumz = 0.d0

      do i = 1, natom
         sump = sump + physmass(i)
         sumx = sumx + physmass(i) * vx(i,1)
         sumy = sumy + physmass(i) * vy(i,1)
         sumz = sumz + physmass(i) * vz(i,1)
      end do

      sumx = sumx / sump
      sumy = sumy / sump
      sumz = sumz / sump

      do i = 1, natom
         vx(i,1) = vx(i,1) - sumx
         vy(i,1) = vy(i,1) - sumy
         vz(i,1) = vz(i,1) - sumz
      end do

!-----------------------------------------------------------------------
!     /*   skip rotational correction for periodic boundaries         */
!-----------------------------------------------------------------------

      if ( iboundary .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   center of mass                                             */
!-----------------------------------------------------------------------

      sumx = 0.d0
      sumy = 0.d0
      sumz = 0.d0
      sump = 0.d0

      do i = 1, natom
         sumx = sumx + physmass(i) * x(i,1)
         sumy = sumy + physmass(i) * y(i,1)
         sumz = sumz + physmass(i) * z(i,1)
         sump = sump + physmass(i)
      end do

      xg = sumx / sump
      yg = sumy / sump
      zg = sumz / sump

!-----------------------------------------------------------------------
!     /*   angular momentum                                           */
!-----------------------------------------------------------------------

      b(:) = 0.d0

      do i = 1, natom
         fm   = physmass(i)
         b(1) = b(1) + (y(i,1)-yg) * fm*vz(i,1) &
     &               - (z(i,1)-zg) * fm*vy(i,1)
         b(2) = b(2) + (z(i,1)-zg) * fm*vx(i,1) &
     &               - (x(i,1)-xg) * fm*vz(i,1)
         b(3) = b(3) + (x(i,1)-xg) * fm*vy(i,1) &
     &               - (y(i,1)-yg) * fm*vx(i,1)
      end do

!-----------------------------------------------------------------------
!     /*   moment of inertia                                          */
!-----------------------------------------------------------------------

      a(:,:) = 0.d0

      do i = 1, natom
         fm   = physmass(i)
         a(1,1) = a(1,1) + fm*(y(i,1)-yg)*(y(i,1)-yg) &
     &                   + fm*(z(i,1)-zg)*(z(i,1)-zg)
         a(1,2) = a(1,2) - fm*(x(i,1)-xg)*(y(i,1)-yg)
         a(1,3) = a(1,3) - fm*(x(i,1)-xg)*(z(i,1)-zg)
         a(2,1) = a(2,1) - fm*(y(i,1)-yg)*(x(i,1)-xg)
         a(2,2) = a(2,2) + fm*(z(i,1)-zg)*(z(i,1)-zg) &
     &                   + fm*(x(i,1)-xg)*(x(i,1)-xg)
         a(2,3) = a(2,3) - fm*(y(i,1)-yg)*(z(i,1)-zg)
         a(3,1) = a(3,1) - fm*(z(i,1)-zg)*(x(i,1)-xg)
         a(3,2) = a(3,2) - fm*(z(i,1)-zg)*(y(i,1)-yg)
         a(3,3) = a(3,3) + fm*(x(i,1)-xg)*(x(i,1)-xg) &
     &                   + fm*(y(i,1)-yg)*(y(i,1)-yg)
      end do

!-----------------------------------------------------------------------
!     /*   principal axis: diagonalize moment of inertia              */
!-----------------------------------------------------------------------

      call ddiag ( a, e, c, 3 )

!-----------------------------------------------------------------------
!     /*   in principal axis: angular momentum                        */
!-----------------------------------------------------------------------

      d(1) = c(1,1)*b(1) + c(2,1)*b(2) + c(3,1)*b(3)
      d(2) = c(1,2)*b(1) + c(2,2)*b(2) + c(3,2)*b(3)
      d(3) = c(1,3)*b(1) + c(2,3)*b(2) + c(3,3)*b(3)

!-----------------------------------------------------------------------
!     /*   d = angular momentum divided by moment of inertia          */
!-----------------------------------------------------------------------

      if ( natom .eq. 1 ) then
         d(1) = 0.d0
         d(2) = 0.d0
         d(3) = 0.d0
      else if ( natom .eq. 2 ) then
         d(1) = 0.d0
         if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
         if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
      else
         if ( e(1) .gt. tiny ) d(1) = d(1)/e(1)
         if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
         if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
      end if

!-----------------------------------------------------------------------
!     /*   d in laboratory frame                                      */
!-----------------------------------------------------------------------

      f(1) = c(1,1)*d(1) + c(1,2)*d(2) + c(1,3)*d(3)
      f(2) = c(2,1)*d(1) + c(2,2)*d(2) + c(2,3)*d(3)
      f(3) = c(3,1)*d(1) + c(3,2)*d(2) + c(3,3)*d(3)

!-----------------------------------------------------------------------
!     /*   remove angular momentum                                    */
!-----------------------------------------------------------------------

      do i = 1, natom
         fm   = physmass(i)
         vx(i,1) = vx(i,1) - f(2)*(z(i,1)-zg) + f(3)*(y(i,1)-yg)
         vy(i,1) = vy(i,1) - f(3)*(x(i,1)-xg) + f(1)*(z(i,1)-zg)
         vz(i,1) = vz(i,1) - f(1)*(y(i,1)-yg) + f(2)*(x(i,1)-xg)
      end do

      return
      end





!***********************************************************************
      subroutine correct_hessian_gad
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, physmass, natom, iboundary

      use gad_variables, only: &
     &   he_gad

      use mech_variables, only: &
     &   nfreeze_mech

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, k

      real(8) :: sump, sumx, sumy, sumz, xg, yg, zg, fm, sqm

      real(8), dimension(3,3) :: a, c

      real(8), dimension(3)   :: b, d, e, f

      real(8) :: tiny = 1.d-14

!-----------------------------------------------------------------------
!     /*   skip if frozen atoms exist                                 */
!-----------------------------------------------------------------------

      if ( nfreeze_mech .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   remove net force                                           */
!-----------------------------------------------------------------------

      sump = 0.d0
      sumx = 0.d0
      sumy = 0.d0
      sumz = 0.d0

      do k = 1, 3*natom, 3
         i = (k+2) / 3
         sqm  = sqrt(physmass(i))
         sump = sump + physmass(i)
         sumx = sumx + sqm * he_gad(k+0)
         sumy = sumy + sqm * he_gad(k+1)
         sumz = sumz + sqm * he_gad(k+2)
      end do

      sumx = sumx / sump
      sumy = sumy / sump
      sumz = sumz / sump

      do k = 1, 3*natom, 3
         i = (k+2) / 3
         sqm  = sqrt(physmass(i))
         he_gad(k+0) = he_gad(k+0) - sqm * sumx
         he_gad(k+1) = he_gad(k+1) - sqm * sumy
         he_gad(k+2) = he_gad(k+2) - sqm * sumz
      end do

!-----------------------------------------------------------------------
!     /*   skip rotational correction for periodic boundaries         */
!-----------------------------------------------------------------------

      if ( iboundary .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   center of mass                                             */
!-----------------------------------------------------------------------

      sump = 0.d0
      sumx = 0.d0
      sumy = 0.d0
      sumz = 0.d0

      do i = 1, natom
         sump = sump + physmass(i)
         sumx = sumx + physmass(i) * x(i,1)
         sumy = sumy + physmass(i) * y(i,1)
         sumz = sumz + physmass(i) * z(i,1)
      end do

      xg = sumx / sump
      yg = sumy / sump
      zg = sumz / sump

!-----------------------------------------------------------------------
!     /*   angular momentum                                           */
!-----------------------------------------------------------------------

      b(:) = 0.d0

      do k = 1, 3*natom, 3
         i = (k+2) / 3
         sqm  = sqrt(physmass(i))
         b(1) = b(1) + (y(i,1)-yg) * sqm * he_gad(k+2) &
     &               - (z(i,1)-zg) * sqm * he_gad(k+1)
         b(2) = b(2) + (z(i,1)-zg) * sqm * he_gad(k+0) &
     &               - (x(i,1)-xg) * sqm * he_gad(k+2)
         b(3) = b(3) + (x(i,1)-xg) * sqm * he_gad(k+1) &
     &               - (y(i,1)-yg) * sqm * he_gad(k+0)
      end do

!-----------------------------------------------------------------------
!     /*   moment of inertia                                          */
!-----------------------------------------------------------------------

      a(:,:) = 0.d0

      do i = 1, natom
         fm   = physmass(i)
         a(1,1) = a(1,1) + fm*(y(i,1)-yg)*(y(i,1)-yg) &
     &                   + fm*(z(i,1)-zg)*(z(i,1)-zg)
         a(1,2) = a(1,2) - fm*(x(i,1)-xg)*(y(i,1)-yg)
         a(1,3) = a(1,3) - fm*(x(i,1)-xg)*(z(i,1)-zg)
         a(2,1) = a(2,1) - fm*(y(i,1)-yg)*(x(i,1)-xg)
         a(2,2) = a(2,2) + fm*(z(i,1)-zg)*(z(i,1)-zg) &
     &                   + fm*(x(i,1)-xg)*(x(i,1)-xg)
         a(2,3) = a(2,3) - fm*(y(i,1)-yg)*(z(i,1)-zg)
         a(3,1) = a(3,1) - fm*(z(i,1)-zg)*(x(i,1)-xg)
         a(3,2) = a(3,2) - fm*(z(i,1)-zg)*(y(i,1)-yg)
         a(3,3) = a(3,3) + fm*(x(i,1)-xg)*(x(i,1)-xg) &
     &                   + fm*(y(i,1)-yg)*(y(i,1)-yg)
      end do

!-----------------------------------------------------------------------
!     /*   principal axis: diagonalize moment of inertia              */
!-----------------------------------------------------------------------

      call ddiag ( a, e, c, 3 )

!-----------------------------------------------------------------------
!     /*   in principal axis: angular momentum                        */
!-----------------------------------------------------------------------

      d(1) = c(1,1)*b(1) + c(2,1)*b(2) + c(3,1)*b(3)
      d(2) = c(1,2)*b(1) + c(2,2)*b(2) + c(3,2)*b(3)
      d(3) = c(1,3)*b(1) + c(2,3)*b(2) + c(3,3)*b(3)

!-----------------------------------------------------------------------
!     /*   d = angular momentum divided by moment of inertia          */
!-----------------------------------------------------------------------

      if ( natom .eq. 1 ) then
         d(1) = 0.d0
         d(2) = 0.d0
         d(3) = 0.d0
      else if ( natom .eq. 2 ) then
         d(1) = 0.d0
         if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
         if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
      else
         if ( e(1) .gt. tiny ) d(1) = d(1)/e(1)
         if ( e(2) .gt. tiny ) d(2) = d(2)/e(2)
         if ( e(3) .gt. tiny ) d(3) = d(3)/e(3)
      end if

!-----------------------------------------------------------------------
!     /*   d in laboratory frame                                      */
!-----------------------------------------------------------------------

      f(1) = c(1,1)*d(1) + c(1,2)*d(2) + c(1,3)*d(3)
      f(2) = c(2,1)*d(1) + c(2,2)*d(2) + c(2,3)*d(3)
      f(3) = c(3,1)*d(1) + c(3,2)*d(2) + c(3,3)*d(3)

!-----------------------------------------------------------------------
!     /*   remove angular momentum                                    */
!-----------------------------------------------------------------------

      do k = 1, 3*natom, 3
         i = (k+2) / 3
         sqm  = sqrt(physmass(i))
         he_gad(k+0) = he_gad(k+0) &
     &      - sqm * ( f(2)*(z(i,1)-zg) + f(3)*(y(i,1)-yg) )
         he_gad(k+1) = he_gad(k+1) &
     &      - sqm * ( f(3)*(x(i,1)-xg) + f(1)*(z(i,1)-zg) )
         he_gad(k+2) = he_gad(k+2) &
     &      - sqm * ( f(1)*(y(i,1)-yg) + f(2)*(x(i,1)-xg) )
      end do

      return
      end





!***********************************************************************
      subroutine analysis_gad ( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit

      use analysis_variables, only : &
     &   iprint_xyz

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ioption

!-----------------------------------------------------------------------
!     /*   ioption = 1:  initialize/restart                           */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

!        /*   step intervals of analysis   */
         call read_int1 ( iprint_xyz,  '<iprint_xyz>',  12, iounit )

!        /*   print trajectory   */
         call analysis_xyz_gad ( 1 )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  start analysis                               */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

!        /*   print trajectory   */
         call analysis_xyz_gad ( 2 )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

!        /*   print trajectory   */
         call analysis_xyz_gad ( 3 )

      end if

      return
      end





!***********************************************************************
      subroutine analysis_xyz_gad ( ioption )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, au_length, potential, natom, istep, species, &
     &   iounit_xyz

      use analysis_variables, only : iprint_xyz

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8), parameter :: bohr2ang  = au_length*1.d+10

      integer :: i, ioption

      real(8) :: xa, ya, za

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( iprint_xyz .le. 0 ) return

      if ( mod(istep,iprint_xyz) .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   ioption = 1:  initialize                                   */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

         continue

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 2:  calculate and print out data                 */
!-----------------------------------------------------------------------

      if ( ioption .eq. 2 ) then

         open ( iounit_xyz, file = 'trj.xyz', access = 'append' )

         write( iounit_xyz, '(i8)' ) natom

         write( iounit_xyz, '(i8,f16.8)' ) istep, potential

         do i = 1, natom

            xa = x(i,1) * bohr2ang
            ya = y(i,1) * bohr2ang
            za = z(i,1) * bohr2ang

            write( iounit_xyz, '(a4,3f16.8)' ) species(i), xa, ya, za

         end do

         close( iounit_xyz )

      end if

!-----------------------------------------------------------------------
!     /*   ioption = 3:  finalize                                     */
!-----------------------------------------------------------------------

      if ( ioption .eq. 3 ) then

         continue

      end if

      return
      end



!***********************************************************************
      subroutine adjust_ascent_gad
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : dt, iexit

      use gad_variables, only : dt_gad, theta_gad, dt_conv_gad

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     /*   step size for dt scaling   */
      real(8) :: dt_scale_gad = 0.55d0

!     //   step interval for dt scaling
      real(8) :: istep_scale_gad = 1

!     /*   maximum angle between of d_gad and dold_gad   */
      real(8) :: theta_max_gad = 10.d0

!     //   step counter
      integer, save :: istep = 0

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      if ( theta_gad .gt. theta_max_gad ) then

         dt_gad = dt_gad * dt_scale_gad

         istep = 0

      else

         istep = istep + 1

         if ( istep .eq. istep_scale_gad ) then

            dt_gad = min( dt, dt_gad/dt_scale_gad )

            istep = 0

         end if

      end if

      if ( dt_gad .le. dt_conv_gad ) iexit = 1

      return
      end
