!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    June 13, 2020 by M. Shiga
!      Description:     full optimization by LBFGS algorithm
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine fulloptcycle
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, potential, box, boxinv, fbox, tension, volume, &
     &   pressure, sigma_ref, boxinv_ref, volume_ref, boxinv, &
     &   fx, fy, fz, natom, nstep, iounit, iounit_std, istep, &
     &   istep_start, istep_end, iexit, iboundary

      use lbfgs_variables, only : &
     &   pos, pos0, grad, dm, ws, func, dmax, drms, fmax, frms, &
     &   dmax_tol, drms_tol, fmax_tol, frms_tol, postol, eps, stpmax, &
     &   ndim, nup, iflag, iprint

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: j, k, m

      real(8), dimension(3,3) :: box0
      real(8), dimension(3,3) :: boxinv0

      real(8), dimension(natom,1) :: x0
      real(8), dimension(natom,1) :: y0
      real(8), dimension(natom,1) :: z0

      real(8) :: det3

      real(8) :: stpmax_init = 1.d-3

!-----------------------------------------------------------------------
!     /*   error termination for free boundary                        */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) then

         write( 6, '(a)' ) &
     &     'Error - Fullopt is valid only for periodic boundary.'

         write( 6, '(a)' )

         call error_handling ( 1, 'subroutine fulloptcycle', 23 )

      end if

!-----------------------------------------------------------------------
!     //   initialize step
!-----------------------------------------------------------------------

      istep = istep_start
      istep_end = istep

!-----------------------------------------------------------------------
!     //   new box size:  box, x, y, z  ->  pos
!-----------------------------------------------------------------------

!     /*   inverse box matrix   */
      call inv3 ( box, boxinv )

!     /*   counter   */
      k = 0

!     /*   box   */
      do m = 1, 3
      do j = 1, 3
         k = k + 1
         pos(k) = box(j,m)
      end do
      end do

!     /*   x, y, z   */
      do j = 1, natom
         pos(k+1) = boxinv(1,1)*x(j,1) + boxinv(1,2)*y(j,1) &
     &            + boxinv(1,3)*z(j,1)
         pos(k+2) = boxinv(2,1)*x(j,1) + boxinv(2,2)*y(j,1) &
     &            + boxinv(2,3)*z(j,1)
         pos(k+3) = boxinv(3,1)*x(j,1) + boxinv(3,2)*y(j,1) &
     &            + boxinv(3,3)*z(j,1)
         k = k + 3
      end do

!-----------------------------------------------------------------------
!     //   calculate potential and gradients
!-----------------------------------------------------------------------

      call getfbox

!     /*   add pressure and tension contributions   */
      call addfbox &
     &   ( fbox, volume, pressure, boxinv, box, sigma_ref )

!-----------------------------------------------------------------------
!     //   potential  ->  func
!-----------------------------------------------------------------------

      func = potential

!     /*   add pressure and tension contribution   */
      call addebox &
     &   ( func, pressure, volume, boxinv_ref, volume_ref, box, &
     &     tension )

!-----------------------------------------------------------------------
!     //   force  ->  grad
!-----------------------------------------------------------------------

!     /*   counter   */
      k = 0

!     /*   box force   */
      do m = 1, 3
      do j = 1, 3
         k = k + 1
         grad(k) = - fbox(j,m)
      end do
      end do

!     /*   fx, fy, fz   */
      do j = 1, natom
         grad(k+1) = - box(1,1)*fx(j,1) - box(2,1)*fy(j,1) &
     &               - box(3,1)*fz(j,1)
         grad(k+2) = - box(1,2)*fx(j,1) - box(2,2)*fy(j,1) &
     &               - box(3,2)*fz(j,1)
         grad(k+3) = - box(1,3)*fx(j,1) - box(2,3)*fy(j,1) &
     &               - box(3,3)*fz(j,1)
         k = k + 3
      end do

!-----------------------------------------------------------------------
!     //   old box size: pos0
!-----------------------------------------------------------------------

!     /*   store box matrix   */
      pos0(:) = pos(:)

!     /*   store coordinates   */
      x0(:,1) = x(:,1)
      y0(:,1) = y(:,1)
      z0(:,1) = z(:,1)

!     /*   store box matrix   */
      box0(:,:) = box(:,:)

!     /*   inverse box matrix   */
      call inv3 ( box0, boxinv0 )

!-----------------------------------------------------------------------
!     //   standard output
!-----------------------------------------------------------------------

      call standard_boxopt

!-----------------------------------------------------------------------
!     //   do some analysis
!-----------------------------------------------------------------------

      call analysis( 1 )

!-----------------------------------------------------------------------
!     //   terminate if converged
!-----------------------------------------------------------------------

      if ( ( fmax .lt. fmax_tol ) .and. ( frms .lt. frms_tol ) ) then
         return
      end if

!-----------------------------------------------------------------------
!     //   main loop start
!-----------------------------------------------------------------------

      do istep = istep_start+1, nstep

!-----------------------------------------------------------------------
!        //   current step
!-----------------------------------------------------------------------

         istep_end = istep

!-----------------------------------------------------------------------
!        //   tune stpmax parameter
!-----------------------------------------------------------------------

         if ( istep .eq. istep_start+1 ) then
            stpmax = stpmax_init
         else
            stpmax = 1.d+20
         end if

!-----------------------------------------------------------------------
!        //   old position: pos0
!-----------------------------------------------------------------------

!        /*   store box matrix   */
         pos0(:) = pos(:)

!        /*   store coordinates   */
         x0(:,1) = x(:,1)
         y0(:,1) = y(:,1)
         z0(:,1) = z(:,1)

!        /*   store box matrix   */
         box0(:,:) = box(:,:)

!        /*   inverse box matrix   */
         call inv3 ( box0, boxinv0 )

!-----------------------------------------------------------------------
!        //   call limited memory bfgs routine
!-----------------------------------------------------------------------

         call lbfgs ( ndim, nup, pos, func, grad, .false., dm, &
     &                iprint, eps, postol, ws, iflag )

!-----------------------------------------------------------------------
!        //   new box: pos  ->  box
!-----------------------------------------------------------------------

!        /*   counter   */
         k = 0

!        /*   box   */
         do m = 1, 3
         do j = 1, 3
            k = k + 1
            box(j,m) = pos(k)
         end do
         end do

!        /*   x, y, z   */
         do j = 1, natom
            x(j,1) = box(1,1)*pos(k+1) + box(1,2)*pos(k+2) &
     &             + box(1,3)*pos(k+3)
            y(j,1) = box(2,1)*pos(k+1) + box(2,2)*pos(k+2) &
     &             + box(2,3)*pos(k+3)
            z(j,1) = box(3,1)*pos(k+1) + box(3,2)*pos(k+2) &
     &             + box(3,3)*pos(k+3)
            k = k + 3
         end do

!        /*   inverse matrix of box   */
         call inv3 ( box, boxinv )

!        /*   shift box volume   */
         volume = det3( box )

!-----------------------------------------------------------------------
!        //   calculate potential and gradients
!-----------------------------------------------------------------------

         call getfbox

!        /*   add pressure and tension contributions   */
         call addfbox &
     &      ( fbox, volume, pressure, boxinv, box, sigma_ref )

!-----------------------------------------------------------------------
!        //   potential  ->  func
!-----------------------------------------------------------------------

         func = potential

!        /*   add pressure and tension contribution   */
         call addebox &
     &      ( func, pressure, volume, boxinv_ref, volume_ref, box, &
     &        tension )

!-----------------------------------------------------------------------
!        //   force  ->  grad
!-----------------------------------------------------------------------

!        /*   counter   */
         k = 0

!        /*   box force   */
         do m = 1, 3
         do j = 1, 3
            k = k + 1
            grad(k) = - fbox(j,m)
         end do
         end do

!        /*   fx, fy, fz   */
         do j = 1, natom
            grad(k+1) = - box(1,1)*fx(j,1) - box(2,1)*fy(j,1) &
     &                  - box(3,1)*fz(j,1)
            grad(k+2) = - box(1,2)*fx(j,1) - box(2,2)*fy(j,1) &
     &                  - box(3,2)*fz(j,1)
            grad(k+3) = - box(1,3)*fx(j,1) - box(2,3)*fy(j,1) &
     &                  - box(3,3)*fz(j,1)
            k = k + 3
         end do

!-----------------------------------------------------------------------
!        //   standard output
!-----------------------------------------------------------------------

         call standard_boxopt

!-----------------------------------------------------------------------
!        //   output restart
!-----------------------------------------------------------------------

         call backup_boxopt

!-----------------------------------------------------------------------
!        //   do some analysis
!-----------------------------------------------------------------------

         call analysis( 2 )

!-----------------------------------------------------------------------
!        //   terminate if converged
!-----------------------------------------------------------------------

         if ( ( dmax .lt. dmax_tol ) .and. ( drms .lt. drms_tol ) .and. &
     &        ( fmax .lt. fmax_tol ) .and. ( frms .lt. frms_tol ) ) then
            iflag = 0
         end if

         if ( iflag .le. 0 ) then
            iexit = 1
            exit
         end if

!-----------------------------------------------------------------------
!        /*   exit if `exit.dat' exists                               */
!-----------------------------------------------------------------------

         call softexit

         if ( iexit .eq. 1 ) exit

      end do

      return
      end
