!///////////////////////////////////////////////////////////////////////
!
!     Calculate potential energy using xTB
!
!     Authors:         T. Murakami, T. Takayanagi, M. Shiga
!     Description:     energy and forces obtained by GFN2-xTB
!     Last updated:    Aug 28, 2023 by M. Shiga
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine force_xtb
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, au_length, natom, nbead, iounit, &
     &   mbox, species, xtb_exe_command

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   error flag
      integer :: ierr = 0

!     //   integers
      integer, save :: iset = 0

!     //   integers
      integer :: i, j, ibead

!     //   real numbers
      real(8) :: xi, yi, zi, gx, gy, gz, v, bohr2ang

!     //   characters
      character(len=3)   :: char_dir
      character(len=80)  :: char_line
      character(len=300) :: file_xtb

!-----------------------------------------------------------------------
!     /*   unit conversion                                            */
!-----------------------------------------------------------------------

!     //   bohr to angstrom
      bohr2ang = au_length * 1.d+10

!-----------------------------------------------------------------------
!     /*   read xtb execution command and make subdirectories         */
!-----------------------------------------------------------------------
!
!     from input.dat or input_default.dat
!
!     <xtb_exe_command>
!     "xtb xtb.xyz --chrg 0 --uhf 0 --grad --vparam xtb.dat > xtb.out"
!

!     //   initial visit
      if ( iset .eq. 0 ) then

!        //   read xtb execution command from keyword
         call read_char( &
     &      xtb_exe_command, 300, '<xtb_exe_command>', 17, iounit )

!        //   make subdirectories 001, ...
         do ibead = 1, nbead
            call int3_to_char( ibead, char_dir )
            call system( 'mkdir -p ' // trim(char_dir) )
            call system( 'cp xtb.* ' // trim(char_dir) )
         end do

!        //   set complete
         iset = 1

!     //   initial visit
      end if

!-----------------------------------------------------------------------
!     /*   xtb files                                                  */
!-----------------------------------------------------------------------

!     //   list of files to be deleted after each xtb execution
      file_xtb = &
     &   'charges energy xtbrestart wbo xtbtopo.mol ' // &
     &   'xtbout xtb.xyz gradient g98.out hessian ' // &
     &   'xtb.engrad vibspectrum xtbhess.xyz'

!-----------------------------------------------------------------------
!     /*   loop of beads                                              */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!        //    directory number
!-----------------------------------------------------------------------

         call int3_to_char( ibead, char_dir )

!-----------------------------------------------------------------------
!        //   print xtb.xyz
!-----------------------------------------------------------------------

         open ( iounit, file=char_dir//'/xtb.xyz', form='formatted' )

         write( iounit, '(i6)' ) natom
         write( iounit, '(a)' ) 'ANGSTROM'

         do i = 1, natom
            write( iounit, '(a2,3f15.9)' ) &
     &         trim(species(i)), x(i,ibead) * bohr2ang, &
     &                           y(i,ibead) * bohr2ang, &
     &                           z(i,ibead) * bohr2ang
         end do

         close( iounit )

!-----------------------------------------------------------------------
!        //   execution by system calls
!-----------------------------------------------------------------------

         call system( 'cd ' // char_dir // ';' // &
     &                 trim(xtb_exe_command) // '; cd ..' )

!-----------------------------------------------------------------------
!        //   read potential energy
!-----------------------------------------------------------------------

         open ( iounit, file=char_dir//'/xtb.out' )

         do
            read( iounit, '(a)', iostat=ierr ) char_line
            if ( ierr .ne. 0 ) exit
            if ( index( char_line, 'total energy' ) .ne. 0 ) exit
         end do

         close( iounit )

         if ( ierr .ne. 0 ) exit

         read ( char_line, '(30x,f24.12)', iostat=ierr ) v

         pot(ibead) = pot(ibead) + v

!-----------------------------------------------------------------------
!        //   read potential energy gradient
!-----------------------------------------------------------------------

         open( iounit, file=char_dir//'/gradient' )

         do i = 1, natom+2
            read( iounit, *, iostat=ierr )
         end do

         do i = 1, natom
            read( iounit, *, iostat=ierr ) gx, gy, gz
            fx(i,ibead) = fx(i,ibead) - gx
            fy(i,ibead) = fy(i,ibead) - gy
            fz(i,ibead) = fz(i,ibead) - gz
         end do

         close( iounit )

         if ( ierr .ne. 0 ) exit

!-----------------------------------------------------------------------
!        //   read energy gradient
!-----------------------------------------------------------------------

         call system( &
     &      'cd ' // char_dir // '; ' // &
     &      'rm -f ' // trim(file_xtb) // '; cd ..' )

!-----------------------------------------------------------------------
!     /*   loop of beads                                              */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   error handling                                             */
!-----------------------------------------------------------------------

      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) 'Error - file xtb.out or gradient.'
      end if

      call error_handling( ierr, 'subroutine force_xtbf', 21 )

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom

         xi = x(i,j)
         yi = y(i,j)
         zi = z(i,j)

         call pbc_unfold &
     &     ( xi, yi, zi, mbox(1,i,j), mbox(2,i,j), mbox(3,i,j) )

         vir(1,1) = vir(1,1) + fx(i,j)*xi
         vir(1,2) = vir(1,2) + fx(i,j)*yi
         vir(1,3) = vir(1,3) + fx(i,j)*zi
         vir(2,1) = vir(2,1) + fy(i,j)*xi
         vir(2,2) = vir(2,2) + fy(i,j)*yi
         vir(2,3) = vir(2,3) + fy(i,j)*zi
         vir(3,1) = vir(3,1) + fz(i,j)*xi
         vir(3,2) = vir(3,2) + fz(i,j)*yi
         vir(3,3) = vir(3,3) + fz(i,j)*zi

      end do
      end do

      return
      end
