!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Date:            Sep. 28, 2010
!      Description:     energy and force of water molecules
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine force_water_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   model_water, iboundary, iounit, myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   select water model                                         */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         call read_int1_MPI ( model_water, '<model_water>', 13, iounit )

         iset = 1

      end if

      if ( iboundary .ne. 0 ) then

         if ( myrank .eq. 0 ) then

            write( 6, '(a)' ) &
     &         'Error - Water potentials are available ' // &
     &         'for free boundary only.'
            write( 6, '(a)' )

         end if

         call error_handling_MPI( 1, 'subroutine force_water_MPI', 26 )

      end if

      if ( model_water .eq. 1 ) then

         call force_spcf_MPI

      else if ( model_water .eq. 2 ) then

         call force_rwk_MPI

      end if

      return
      end


!***********************************************************************
      subroutine force_spcf_MPI
!***********************************************************************
!=======================================================================
!
!     modified flexible SPC model (SPC/F2) for water:
!
!        L. Lobaugh and G. A. Voth, J. Chem. Phys. 106, 2400 (1997).
!
!     NOTE:  the atoms must be sorted as O, H, H, O, H, H, ...
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, dipx, dipy, dipz, au_energy, boltz, &
     &   au_length, pi, natom, nbead, myrank, nprocs

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer ::  i, j, k, m, i_o, i_h, i_g

      real(8) ::  rho_w, d_w, b_oh, b_hh, b_const, c_const, d_const, &
     &            sigma, epsilon, q_o, q_h, es6, es12, dvdr

      real(8) ::  rx, ry, rz, r, r2, rinv, rinv2, rinv6, rinv12

      real(8) ::  rx_oh, ry_oh, rz_oh, r2_oh, r_oh, &
     &            rx_og, ry_og, rz_og, r2_og, r_og, &
     &            rx_hh, ry_hh, rz_hh, r2_hh, r_hh

      real(8) :: qi = 0.d0, qj = 0.d0

!-----------------------------------------------------------------------
!     /*   potential parameters                                       */
!-----------------------------------------------------------------------

      rho_w   =   2.361d+10  *au_length
      d_w     =   0.708d-18  /au_energy

      b_oh    =   1.000d-10  /au_length
      b_hh    =   b_oh *sin(pi*108.d0/2.d0/180.d0) *2.d0

      b_const =   1.803d+02  /au_energy *au_length**2
      c_const = - 1.469d+02  /au_energy *au_length**2
      d_const =   0.776d+02  /au_energy *au_length**2

      sigma   =   3.165d-10 /au_length
      epsilon =   78.22d0   *boltz

      q_o     = - 0.82d0
      q_h     = + 0.41d0

!-----------------------------------------------------------------------
!     /*   intra-molecular term                                       */
!-----------------------------------------------------------------------

      do k = 1, nbead

         if ( mod( k-1, nprocs ) .ne. myrank ) cycle

         do i = 1, natom, 3

            i_o = i
            i_h = i + 1
            i_g = i + 2

            rx_oh = x(i_o,k) - x(i_h,k)
            ry_oh = y(i_o,k) - y(i_h,k)
            rz_oh = z(i_o,k) - z(i_h,k)

            rx_og = x(i_o,k) - x(i_g,k)
            ry_og = y(i_o,k) - y(i_g,k)
            rz_og = z(i_o,k) - z(i_g,k)

            rx_hh = x(i_h,k) - x(i_g,k)
            ry_hh = y(i_h,k) - y(i_g,k)
            rz_hh = z(i_h,k) - z(i_g,k)

            r2_oh = rx_oh*rx_oh + ry_oh*ry_oh + rz_oh*rz_oh
            r_oh  = sqrt(r2_oh)

            r2_og = rx_og*rx_og + ry_og*ry_og + rz_og*rz_og
            r_og  = sqrt(r2_og)

            r2_hh = rx_hh*rx_hh + ry_hh*ry_hh + rz_hh*rz_hh
            r_hh  = sqrt(r2_hh)

            pot(k) = pot(k) &
     &         + rho_w*rho_w*d_w*(r_oh-b_oh)*(r_oh-b_oh) &
     &         + rho_w*rho_w*d_w*(r_og-b_oh)*(r_og-b_oh) &
     &         + b_const*0.5d0*(r_hh-b_hh)*(r_hh-b_hh) &
     &         + c_const*(r_oh+r_og-2.d0*b_oh)*(r_hh-b_hh) &
     &         + d_const*(r_oh-b_oh)*(r_og-b_oh)

            fx(i_o,k) = fx(i_o,k) &
     &         - 2.d0*rho_w*rho_w*d_w*rx_oh/r_oh*(r_oh-b_oh) &
     &         - 2.d0*rho_w*rho_w*d_w*rx_og/r_og*(r_og-b_oh) &
     &         - c_const*rx_oh/r_oh*(r_hh-b_hh) &
     &         - c_const*rx_og/r_og*(r_hh-b_hh) &
     &         - d_const*rx_oh/r_oh*(r_og-b_oh) &
     &         - d_const*rx_og/r_og*(r_oh-b_oh)

            fy(i_o,k) = fy(i_o,k) &
     &         - 2.d0*rho_w*rho_w*d_w*ry_oh/r_oh*(r_oh-b_oh) &
     &         - 2.d0*rho_w*rho_w*d_w*ry_og/r_og*(r_og-b_oh) &
     &         - c_const*ry_oh/r_oh*(r_hh-b_hh) &
     &         - c_const*ry_og/r_og*(r_hh-b_hh) &
     &         - d_const*ry_oh/r_oh*(r_og-b_oh) &
     &         - d_const*ry_og/r_og*(r_oh-b_oh)

            fz(i_o,k) = fz(i_o,k) &
     &         - 2.d0*rho_w*rho_w*d_w*rz_oh/r_oh*(r_oh-b_oh) &
     &         - 2.d0*rho_w*rho_w*d_w*rz_og/r_og*(r_og-b_oh) &
     &         - c_const*rz_oh/r_oh*(r_hh-b_hh) &
     &         - c_const*rz_og/r_og*(r_hh-b_hh) &
     &         - d_const*rz_oh/r_oh*(r_og-b_oh) &
     &         - d_const*rz_og/r_og*(r_oh-b_oh)

            fx(i_h,k) = fx(i_h,k) &
     &         + 2.d0*rho_w*rho_w*d_w*rx_oh/r_oh*(r_oh-b_oh) &
     &         - b_const*rx_hh/r_hh*(r_hh-b_hh) &
     &         + c_const*rx_oh/r_oh*(r_hh-b_hh) &
     &         - c_const*(r_oh+r_og-2.d0*b_oh)*rx_hh/r_hh &
     &         + d_const*rx_oh/r_oh*(r_og-b_oh)

            fy(i_h,k) = fy(i_h,k) &
     &         + 2.d0*rho_w*rho_w*d_w*ry_oh/r_oh*(r_oh-b_oh) &
     &         - b_const*ry_hh/r_hh*(r_hh-b_hh) &
     &         + c_const*ry_oh/r_oh*(r_hh-b_hh) &
     &         - c_const*(r_oh+r_og-2.d0*b_oh)*ry_hh/r_hh &
     &         + d_const*ry_oh/r_oh*(r_og-b_oh)

            fz(i_h,k) = fz(i_h,k) &
     &         + 2.d0*rho_w*rho_w*d_w*rz_oh/r_oh*(r_oh-b_oh) &
     &         - b_const*rz_hh/r_hh*(r_hh-b_hh) &
     &         + c_const*rz_oh/r_oh*(r_hh-b_hh) &
     &         - c_const*(r_oh+r_og-2.d0*b_oh)*rz_hh/r_hh &
     &         + d_const*rz_oh/r_oh*(r_og-b_oh)

            fx(i_g,k) = fx(i_g,k) &
     &         + 2.d0*rho_w*rho_w*d_w*rx_og/r_og*(r_og-b_oh) &
     &         + b_const*rx_hh/r_hh*(r_hh-b_hh) &
     &         + c_const*rx_og/r_og*(r_hh-b_hh) &
     &         + c_const*(r_oh+r_og-2.d0*b_oh)*rx_hh/r_hh &
     &         + d_const*rx_og/r_og*(r_oh-b_oh)

            fy(i_g,k) = fy(i_g,k) &
     &         + 2.d0*rho_w*rho_w*d_w*ry_og/r_og*(r_og-b_oh) &
     &         + b_const*ry_hh/r_hh*(r_hh-b_hh) &
     &         + c_const*ry_og/r_og*(r_hh-b_hh) &
     &         + c_const*(r_oh+r_og-2.d0*b_oh)*ry_hh/r_hh &
     &         + d_const*ry_og/r_og*(r_oh-b_oh)

            fz(i_g,k) = fz(i_g,k) &
     &         + 2.d0*rho_w*rho_w*d_w*rz_og/r_og*(r_og-b_oh) &
     &         + b_const*rz_hh/r_hh*(r_hh-b_hh) &
     &         + c_const*rz_og/r_og*(r_hh-b_hh) &
     &         + c_const*(r_oh+r_og-2.d0*b_oh)*rz_hh/r_hh &
     &         + d_const*rz_og/r_og*(r_oh-b_oh)

         end do

      end do

!-----------------------------------------------------------------------
!     /*   inter-molecular Lennard-Jones term                         */
!-----------------------------------------------------------------------

      es6  = epsilon*sigma**6
      es12 = epsilon*sigma**12

      do k = 1, nbead

         if ( mod( k-1, nprocs ) .ne. myrank ) cycle

         do i = 1, natom, 3

            m = (i-1)/3

            do j = (m+1)*3+1, natom, 3

               rx = x(i,k) - x(j,k)
               ry = y(i,k) - y(j,k)
               rz = z(i,k) - z(j,k)

               r2     =  rx*rx + ry*ry + rz*rz
               r      =  sqrt(r2)
               rinv   =  1.d0/r
               rinv2  =  rinv*rinv
               rinv6  =  rinv2*rinv2*rinv2
               rinv12 =  rinv6*rinv6

               pot(k) = pot(k) - 4.d0*es6*rinv6 + 4.d0*es12*rinv12

               dvdr = + 24.d0*es6*rinv6*rinv - 48.d0*es12*rinv12*rinv

               fx(i,k) = fx(i,k) - dvdr*rx*rinv
               fy(i,k) = fy(i,k) - dvdr*ry*rinv
               fz(i,k) = fz(i,k) - dvdr*rz*rinv

               fx(j,k) = fx(j,k) + dvdr*rx*rinv
               fy(j,k) = fy(j,k) + dvdr*ry*rinv
               fz(j,k) = fz(j,k) + dvdr*rz*rinv

            end do

         end do

      end do

!-----------------------------------------------------------------------
!     /*   inter-molecular Coulomb term                               */
!-----------------------------------------------------------------------

      do k = 1, nbead

         if ( mod( k-1, nprocs ) .ne. myrank ) cycle

         do i = 1, natom

            m = (i-1)/3

            do j = (m+1)*3+1, natom

               rx = x(i,k) - x(j,k)
               ry = y(i,k) - y(j,k)
               rz = z(i,k) - z(j,k)

               if ( mod( i, 3 ) .eq. 1 ) qi = q_o
               if ( mod( i, 3 ) .eq. 2 ) qi = q_h
               if ( mod( i, 3 ) .eq. 0 ) qi = q_h

               if ( mod( j, 3 ) .eq. 1 ) qj = q_o
               if ( mod( j, 3 ) .eq. 2 ) qj = q_h
               if ( mod( j, 3 ) .eq. 0 ) qj = q_h

               r2    =  rx*rx + ry*ry + rz*rz
               r     =  sqrt(r2)
               rinv  =  1.d0/r
               rinv2 =  rinv*rinv

               pot(k) = pot(k) + qi*qj*rinv

               dvdr = - qi*qj*rinv2

               fx(i,k) = fx(i,k) - dvdr*rx*rinv
               fy(i,k) = fy(i,k) - dvdr*ry*rinv
               fz(i,k) = fz(i,k) - dvdr*rz*rinv

               fx(j,k) = fx(j,k) + dvdr*rx*rinv
               fy(j,k) = fy(j,k) + dvdr*ry*rinv
               fz(j,k) = fz(j,k) + dvdr*rz*rinv

            end do

         end do

      end do

!-----------------------------------------------------------------------
!     /*   dipole moment                                              */
!-----------------------------------------------------------------------

      do k = 1, nbead

         if ( mod( k-1, nprocs ) .ne. myrank ) cycle

         dipx(k) = 0.d0
         dipy(k) = 0.d0
         dipz(k) = 0.d0

         do i = 1, natom, 3

            i_o = i
            i_h = i + 1
            i_g = i + 2

            dipx(k) = dipx(k) + q_o*x(i_o,k) + q_h*(x(i_h,k)+x(i_g,k))
            dipy(k) = dipy(k) + q_o*y(i_o,k) + q_h*(y(i_h,k)+y(i_g,k))
            dipz(k) = dipz(k) + q_o*z(i_o,k) + q_h*(z(i_h,k)+z(i_g,k))

         end do

      end do

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz, natom, nbead )

!     /*   dipole   */
      call my_mpi_allreduce_real_1 ( dipx, nbead )
      call my_mpi_allreduce_real_1 ( dipy, nbead )
      call my_mpi_allreduce_real_1 ( dipz, nbead )

      return
      end





!***********************************************************************
      subroutine force_rwk_MPI
!***********************************************************************
!=======================================================================
!
!     RWK model for water:
!
!     J. R. Reimers, R. O. Watts, and M. L. Klein,
!     Chem. Phys. 64, 95 (1982).
!
!     NOTE:  the atoms must be sorted as O, H, H, O, H, H, ...
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, x, y, z, fx, fy, fz, pot, natom, nbead, myrank, nprocs

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer  i, j, k, m

      real(8) :: r_eq, a_eq, hess_r, hess_a
      real(8) :: dxij, dyij, dzij, drij, ajik, sin_ajik
      real(8) :: dxik, dyik, dzik, drik, cosf
      real(8) :: x3a, x3b, y3a, y3b, z3a, z3b
      real(8) :: dg06r, dg08r, dg10r, vcn
      real(8) :: r11, r12, r13, r21, r22, r23, r31, r32, r33
      real(8) :: roo, ro1, r1o, ro2, r2o, fr, dr, g06, g08, g10
      real(8) :: ah, ao, alfh, alfo, ex, a, alf, rm, c6, c8, c10
      real(8) :: v11, v12, v21, v22, voo, ga, gb
      real(8) :: dv11, dv12, dv13, dv21, dv22, dv23, dv31, dv32, dv33
      real(8) :: dvoo, dvo1, dv1o, dvo2, dv2o, dg06, dg08, dg10, dfr
      real(8) :: dx11, dx12, dx13, dx21, dx22, dx23, dx31, dx32, dx33
      real(8) :: dxoo, dxo1, dxo2, dx1o, dx2o
      real(8) :: dy11, dy12, dy13, dy21, dy22, dy23, dy31, dy32, dy33
      real(8) :: dyoo, dyo1, dyo2, dy1o, dy2o
      real(8) :: dz11, dz12, dz13, dz21, dz22, dz23, dz31, dz32, dz33
      real(8) :: dzoo, dzo1, dzo2, dz1o, dz2o
      real(8) :: coef, acof

      data    ah, alfh        /     1.0077d0,   1.736d0 /
      data    ao, alfo        /     5110.7d0,  2.6301d0 /
      data     a,  alf,   rm  /   0.003066d0, 3.89556d0, 3.0951d0 /
      data    c6,   c8,  c10  /     62.437d0,  1342.7d0, 50050.d0 /
      data  acof,   ga,   gb  / 0.22183756d0,   1.111d0,   0.15d0 /

!-----------------------------------------------------------------------
!     /*   constants                                                  */
!-----------------------------------------------------------------------

      r_eq   = 1.82056d0
      a_eq   = 105.35d0*pi/180.d0

      hess_r = 0.5569d0
      hess_a = 0.157d0

!-----------------------------------------------------------------------
!     /*   intramolecular forces                                      */
!-----------------------------------------------------------------------

      do m=1, nbead

         if ( mod( m-1, nprocs ) .ne. myrank ) cycle

      do i=1, natom, 3

          j = i+1
          k = i+2

          dxij = x(j,m) - x(i,m)
          dyij = y(j,m) - y(i,m)
          dzij = z(j,m) - z(i,m)

          drij = sqrt( dxij*dxij + dyij*dyij + dzij*dzij )

          pot(m) = pot(m) + 0.5d0*hess_r*(drij-r_eq)*(drij-r_eq)

          fx(i,m) = fx(i,m) + hess_r*(drij-r_eq)*dxij/drij
          fy(i,m) = fy(i,m) + hess_r*(drij-r_eq)*dyij/drij
          fz(i,m) = fz(i,m) + hess_r*(drij-r_eq)*dzij/drij
          fx(j,m) = fx(j,m) - hess_r*(drij-r_eq)*dxij/drij
          fy(j,m) = fy(j,m) - hess_r*(drij-r_eq)*dyij/drij
          fz(j,m) = fz(j,m) - hess_r*(drij-r_eq)*dzij/drij

          dxik = x(k,m) - x(i,m)
          dyik = y(k,m) - y(i,m)
          dzik = z(k,m) - z(i,m)

          drik = sqrt( dxik*dxik + dyik*dyik + dzik*dzik )

          pot(m) = pot(m) + 0.5d0*hess_r*(drik-r_eq)*(drik-r_eq)

          fx(i,m) = fx(i,m) + hess_r*(drik-r_eq)*dxik/drik
          fy(i,m) = fy(i,m) + hess_r*(drik-r_eq)*dyik/drik
          fz(i,m) = fz(i,m) + hess_r*(drik-r_eq)*dzik/drik
          fx(k,m) = fx(k,m) - hess_r*(drik-r_eq)*dxik/drik
          fy(k,m) = fy(k,m) - hess_r*(drik-r_eq)*dyik/drik
          fz(k,m) = fz(k,m) - hess_r*(drik-r_eq)*dzik/drik

          cosf = ( dxij*dxik + dyij*dyik + dzij*dzik ) /drij/drik

          cosf = max( cosf, -1.d0 )
          cosf = min( cosf,  1.d0 )

          ajik = acos(cosf)

          pot(m) = pot(m) + 0.5d0*hess_a*(ajik-a_eq)*(ajik-a_eq)

          sin_ajik = sin(ajik)

          fx(i,m) = fx(i,m) &
     &       - hess_a*(ajik-a_eq)/(-sin_ajik) &
     &       * ( + dxij/drij**3/drik*(dxij*dxik+dyij*dyik+dzij*dzik) &
     &           + dxik/drik**3/drij*(dxij*dxik+dyij*dyik+dzij*dzik) &
     &           + 1.d0/drik/drij*(-dxij-dxik) )
          fy(i,m) = fy(i,m) &
     &       - hess_a*(ajik-a_eq)/(-sin_ajik) &
     &       * ( + dyij/drij**3/drik*(dxij*dxik+dyij*dyik+dzij*dzik) &
     &           + dyik/drik**3/drij*(dxij*dxik+dyij*dyik+dzij*dzik) &
     &           + 1.d0/drik/drij*(-dyij-dyik) )
          fz(i,m) = fz(i,m) &
     &       - hess_a*(ajik-a_eq)/(-sin_ajik) &
     &       * ( + dzij/drij**3/drik*(dxij*dxik+dyij*dyik+dzij*dzik) &
     &           + dzik/drik**3/drij*(dxij*dxik+dyij*dyik+dzij*dzik) &
     &           + 1.d0/drik/drij*(-dzij-dzik) )

          fx(j,m) = fx(j,m) &
     &       - hess_a*(ajik-a_eq)/(-sin_ajik) &
     &       * ( - dxij/drij**3/drik*(dxij*dxik+dyij*dyik+dzij*dzik) &
     &           - 1.d0/drik/drij*(-dxik) )
          fy(j,m) = fy(j,m) &
     &       - hess_a*(ajik-a_eq)/(-sin_ajik) &
     &       * ( - dyij/drij**3/drik*(dxij*dxik+dyij*dyik+dzij*dzik) &
     &           - 1.d0/drik/drij*(-dyik) )
          fz(j,m) = fz(j,m) &
     &       - hess_a*(ajik-a_eq)/(-sin_ajik) &
     &       * ( - dzij/drij**3/drik*(dxij*dxik+dyij*dyik+dzij*dzik) &
     &           - 1.d0/drik/drij*(-dzik) )

          fx(k,m) = fx(k,m) &
     &       - hess_a*(ajik-a_eq)/(-sin_ajik) &
     &       * ( - dxik/drik**3/drij*(dxij*dxik+dyij*dyik+dzij*dzik) &
     &           - 1.d0/drik/drij*(-dxij) )
          fy(k,m) = fy(k,m) &
     &       - hess_a*(ajik-a_eq)/(-sin_ajik) &
     &       * ( - dyik/drik**3/drij*(dxij*dxik+dyij*dyik+dzij*dzik) &
     &           - 1.d0/drik/drij*(-dyij) )
          fz(k,m) = fz(k,m) &
     &       - hess_a*(ajik-a_eq)/(-sin_ajik) &
     &       * ( - dzik/drik**3/drij*(dxij*dxik+dyij*dyik+dzij*dzik) &
     &           - 1.d0/drik/drij*(-dzij) )

      end do
      end do

      coef = 1.d0 - 2.d0*acof

!-----------------------------------------------------------------------
!     /*   intermolecular forces                                      */
!-----------------------------------------------------------------------

      do m =   1, nbead

         if ( mod( m-1, nprocs ) .ne. myrank ) cycle

      do i =   1, natom, 3
      do j = i+3, natom, 3

         x3a = x(i,m) + acof*( x(i+1,m) + x(i+2,m) - 2.d0*x(i,m) )
         y3a = y(i,m) + acof*( y(i+1,m) + y(i+2,m) - 2.d0*y(i,m) )
         z3a = z(i,m) + acof*( z(i+1,m) + z(i+2,m) - 2.d0*z(i,m) )

         x3b = x(j,m) + acof*( x(j+1,m) + x(j+2,m) - 2.d0*x(j,m) )
         y3b = y(j,m) + acof*( y(j+1,m) + y(j+2,m) - 2.d0*y(j,m) )
         z3b = z(j,m) + acof*( z(j+1,m) + z(j+2,m) - 2.d0*z(j,m) )

         dx11 = x(i+1,m) - x(j+1,m)
         dx12 = x(i+1,m) - x(j+2,m)
         dx13 = x(i+1,m) - x3b
         dx21 = x(i+2,m) - x(j+1,m)
         dx22 = x(i+2,m) - x(j+2,m)
         dx23 = x(i+2,m) - x3b
         dx31 = x3a      - x(j+1,m)
         dx32 = x3a      - x(j+2,m)
         dx33 = x3a      - x3b
         dxoo = x(i,m)   - x(j,m)
         dxo1 = x(i,m)   - x(j+1,m)
         dxo2 = x(i,m)   - x(j+2,m)
         dx1o = x(i+1,m) - x(j,m)
         dx2o = x(i+2,m) - x(j,m)

         dy11 = y(i+1,m) - y(j+1,m)
         dy12 = y(i+1,m) - y(j+2,m)
         dy13 = y(i+1,m) - y3b
         dy21 = y(i+2,m) - y(j+1,m)
         dy22 = y(i+2,m) - y(j+2,m)
         dy23 = y(i+2,m) - y3b
         dy31 = y3a      - y(j+1,m)
         dy32 = y3a      - y(j+2,m)
         dy33 = y3a      - y3b
         dyoo = y(i,m)   - y(j,m)
         dyo1 = y(i,m)   - y(j+1,m)
         dyo2 = y(i,m)   - y(j+2,m)
         dy1o = y(i+1,m) - y(j,m)
         dy2o = y(i+2,m) - y(j,m)

         dz11 = z(i+1,m) - z(j+1,m)
         dz12 = z(i+1,m) - z(j+2,m)
         dz13 = z(i+1,m) - z3b
         dz21 = z(i+2,m) - z(j+1,m)
         dz22 = z(i+2,m) - z(j+2,m)
         dz23 = z(i+2,m) - z3b
         dz31 = z3a      - z(j+1,m)
         dz32 = z3a      - z(j+2,m)
         dz33 = z3a      - z3b
         dzoo = z(i,m)   - z(j,m)
         dzo1 = z(i,m)   - z(j+1,m)
         dzo2 = z(i,m)   - z(j+2,m)
         dz1o = z(i+1,m) - z(j,m)
         dz2o = z(i+2,m) - z(j,m)

         r11 = sqrt( dx11**2 + dy11**2 + dz11**2 )
         r12 = sqrt( dx12**2 + dy12**2 + dz12**2 )
         r13 = sqrt( dx13**2 + dy13**2 + dz13**2 )
         r21 = sqrt( dx21**2 + dy21**2 + dz21**2 )
         r22 = sqrt( dx22**2 + dy22**2 + dz22**2 )
         r23 = sqrt( dx23**2 + dy23**2 + dz23**2 )
         r31 = sqrt( dx31**2 + dy31**2 + dz31**2 )
         r32 = sqrt( dx32**2 + dy32**2 + dz32**2 )
         r33 = sqrt( dx33**2 + dy33**2 + dz33**2 )
         roo = sqrt( dxoo**2 + dyoo**2 + dzoo**2 )
         ro1 = sqrt( dxo1**2 + dyo1**2 + dzo1**2 )
         ro2 = sqrt( dxo2**2 + dyo2**2 + dzo2**2 )
         r1o = sqrt( dx1o**2 + dy1o**2 + dz1o**2 )
         r2o = sqrt( dx2o**2 + dy2o**2 + dz2o**2 )

         pot(m) = pot(m) &
     &   + 0.6d0*0.6d0*( 1.d0/r11 + 1.d0/r12 + 1.d0/r21 + 1.d0/r22 )
         pot(m) = pot(m) &
     &   + 1.2d0*1.2d0/r33
         pot(m) = pot(m) &
     &   - 0.6d0*1.2d0*( 1.d0/r13 + 1.d0/r23 + 1.d0/r31 + 1.d0/r32 )

         v11 = ah * exp(-alfh*r11)
         v12 = ah * exp(-alfh*r12)
         v21 = ah * exp(-alfh*r21)
         v22 = ah * exp(-alfh*r22)
         voo = ao * exp(-alfo*roo)

         pot(m) = pot(m) + v11
         pot(m) = pot(m) + v12
         pot(m) = pot(m) + v21
         pot(m) = pot(m) + v22
         pot(m) = pot(m) + voo

         dvoo = - alfo*voo
         dv11 = - alfh*v11 - 0.6d0*0.6d0/r11**2
         dv12 = - alfh*v12 - 0.6d0*0.6d0/r12**2
         dv21 = - alfh*v21 - 0.6d0*0.6d0/r21**2
         dv22 = - alfh*v22 - 0.6d0*0.6d0/r22**2

         dv33 = - 1.2d0*1.2d0/r33**2
         dv13 =   0.6d0*1.2d0/r13**2
         dv23 =   0.6d0*1.2d0/r23**2
         dv31 =   0.6d0*1.2d0/r31**2
         dv32 =   0.6d0*1.2d0/r32**2

         ex     = exp(-alf*(ro1-rm))
         pot(m) = pot(m) + a*ex*(ex-2.d0)
         dvo1   = 2.d0*alf*a*ex*(1.d0-ex)

         ex     = exp(-alf*(ro2-rm))
         pot(m) = pot(m) + a*ex*(ex-2.d0)
         dvo2   = 2.d0*alf*a*ex*(1.d0-ex)

         ex     = exp(-alf*(r1o-rm))
         pot(m) = pot(m) + a*ex*(ex-2.d0)
         dv1o   = 2.d0*alf*a*ex*(1.d0-ex)

         ex     = exp(-alf*(r2o-rm))
         pot(m) = pot(m) + a*ex*(ex-2.d0)
         dv2o   = 2.d0*alf*a*ex*(1.d0-ex)

         g06 = 1.d0 - exp( - ga*roo/6.d0  - gb*roo**2/sqrt(6.d0)  )
         g08 = 1.d0 - exp( - ga*roo/8.d0  - gb*roo**2/sqrt(8.d0)  )
         g10 = 1.d0 - exp( - ga*roo/10.d0 - gb*roo**2/sqrt(10.d0) )

         dg06 = ( ga/6.d0  + 2.d0*gb*roo/sqrt(6.d0)  )*( 1.d0 - g06 )
         dg08 = ( ga/8.d0  + 2.d0*gb*roo/sqrt(8.d0)  )*( 1.d0 - g08 )
         dg10 = ( ga/10.d0 + 2.d0*gb*roo/sqrt(10.d0) )*( 1.d0 - g10 )

         fr  = 1.d0 - (0.94835d0*roo)**2.326d0*exp(-0.94835d0*roo)

         dfr = 0.94835d0*(1.d0-fr)*(1.d0-2.326d0/(0.94835d0*roo))

         vcn = c6*(g06/roo)**6 + c8*(g08/roo)**8 + c10*(g10/roo)**10

         dr  = - fr*vcn

         pot(m) = pot(m) + dr

         dg06r =  6.d0 * (dg06*roo-g06)/roo**2 * (g06/roo)**5
         dg08r =  8.d0 * (dg08*roo-g08)/roo**2 * (g08/roo)**7
         dg10r = 10.d0 * (dg10*roo-g10)/roo**2 * (g10/roo)**9

         dvoo = dvoo - dfr*vcn - fr*(c6*dg06r + c8*dg08r + c10*dg10r)

         fx(i+1,m) = fx(i+1,m) &
     &   - dx11*dv11/r11 - dx12*dv12/r12 - dx13*dv13/r13 - dx1o*dv1o/r1o &
     &   - acof*dx31*dv31/r31 - acof*dx32*dv32/r32 - acof*dx33*dv33/r33
         fx(i+2,m) = fx(i+2,m) &
     &   - dx21*dv21/r21 - dx22*dv22/r22 - dx23*dv23/r23 - dx2o*dv2o/r2o &
     &   - acof*dx31*dv31/r31 - acof*dx32*dv32/r32 - acof*dx33*dv33/r33

         fx(j+1,m) = fx(j+1,m) &
     &   + dx11*dv11/r11 + dx21*dv21/r21 + dx31*dv31/r31 + dxo1*dvo1/ro1 &
     &   + acof*dx13*dv13/r13 + acof*dx23*dv23/r23 + acof*dx33*dv33/r33
         fx(j+2,m) = fx(j+2,m) &
     &   + dx12*dv12/r12 + dx22*dv22/r22 + dx32*dv32/r32 + dxo2*dvo2/ro2 &
     &   + acof*dx13*dv13/r13 + acof*dx23*dv23/r23 + acof*dx33*dv33/r33

         fx(i,m) = fx(i,m) &
     &   - dxoo*dvoo/roo - dxo1*dvo1/ro1 - dxo2*dvo2/ro2 &
     &   - coef*dx31*dv31/r31 - coef*dx32*dv32/r32 - coef*dx33*dv33/r33
         fx(j,m) = fx(j,m) &
     &   + dxoo*dvoo/roo + dx1o*dv1o/r1o + dx2o*dv2o/r2o &
     &   + coef*dx13*dv13/r13 + coef*dx23*dv23/r23 + coef*dx33*dv33/r33

         fy(i+1,m) = fy(i+1,m) &
     &   - dy11*dv11/r11 - dy12*dv12/r12 - dy13*dv13/r13 - dy1o*dv1o/r1o &
     &   - acof*dy31*dv31/r31 - acof*dy32*dv32/r32 - acof*dy33*dv33/r33
         fy(i+2,m) = fy(i+2,m) &
     &   - dy21*dv21/r21 - dy22*dv22/r22 - dy23*dv23/r23 - dy2o*dv2o/r2o &
     &   - acof*dy31*dv31/r31 - acof*dy32*dv32/r32 - acof*dy33*dv33/r33

         fy(j+1,m) = fy(j+1,m) &
     &   + dy11*dv11/r11 + dy21*dv21/r21 + dy31*dv31/r31 + dyo1*dvo1/ro1 &
     &   + acof*dy13*dv13/r13 + acof*dy23*dv23/r23 + acof*dy33*dv33/r33
         fy(j+2,m) = fy(j+2,m) &
     &   + dy12*dv12/r12 + dy22*dv22/r22 + dy32*dv32/r32 + dyo2*dvo2/ro2 &
     &   + acof*dy13*dv13/r13 + acof*dy23*dv23/r23 + acof*dy33*dv33/r33

         fy(i,m) = fy(i,m) &
     &   - dyoo*dvoo/roo - dyo1*dvo1/ro1 - dyo2*dvo2/ro2 &
     &   - coef*dy31*dv31/r31 - coef*dy32*dv32/r32 - coef*dy33*dv33/r33
         fy(j,m) = fy(j,m) &
     &   + dyoo*dvoo/roo + dy1o*dv1o/r1o + dy2o*dv2o/r2o &
     &   + coef*dy13*dv13/r13 + coef*dy23*dv23/r23 + coef*dy33*dv33/r33

         fz(i+1,m) = fz(i+1,m) &
     &   - dz11*dv11/r11 - dz12*dv12/r12 - dz13*dv13/r13 - dz1o*dv1o/r1o &
     &   - acof*dz31*dv31/r31 - acof*dz32*dv32/r32 - acof*dz33*dv33/r33
         fz(i+2,m) = fz(i+2,m) &
     &   - dz21*dv21/r21 - dz22*dv22/r22 - dz23*dv23/r23 - dz2o*dv2o/r2o &
     &   - acof*dz31*dv31/r31 - acof*dz32*dv32/r32 - acof*dz33*dv33/r33

         fz(j+1,m) = fz(j+1,m) &
     &   + dz11*dv11/r11 + dz21*dv21/r21 + dz31*dv31/r31 + dzo1*dvo1/ro1 &
     &   + acof*dz13*dv13/r13 + acof*dz23*dv23/r23 + acof*dz33*dv33/r33
         fz(j+2,m) = fz(j+2,m) &
     &   + dz12*dv12/r12 + dz22*dv22/r22 + dz32*dv32/r32 + dzo2*dvo2/ro2 &
     &   + acof*dz13*dv13/r13 + acof*dz23*dv23/r23 + acof*dz33*dv33/r33

         fz(i,m) = fz(i,m) &
     &   - dzoo*dvoo/roo - dzo1*dvo1/ro1 - dzo2*dvo2/ro2 &
     &   - coef*dz31*dv31/r31 - coef*dz32*dv32/r32 - coef*dz33*dv33/r33
         fz(j,m) = fz(j,m) &
     &   + dzoo*dvoo/roo + dz1o*dv1o/r1o + dz2o*dv2o/r2o &
     &   + coef*dz13*dv13/r13 + coef*dz23*dv23/r23 + coef*dz33*dv33/r33

      end do
      end do

      end do

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz, natom, nbead )

      return
      end

