!///////////////////////////////////////////////////////////////////////
!
!      Author:          B. Thomsen, M. Shiga
!      Last updated:    Apr 17, 2023 by B. Thomsen
!      Description:     energy and force from VASP6 calculation
!
!///////////////////////////////////////////////////////////////////////

!     // vasp6 defined
#ifdef vasp6

!***********************************************************************
      subroutine force_vasp6_MPI( )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, au_energy, volume, &
     &   volume_bead, au_length, box, vir, vir_bead, volume_bead, &
     &   method, natom, nbead, method, boxinv, myrank_main, &
     &   nprocs_main, myrank, mpi_comm_sub, np_beads, au_charge, &
     &   iounit, au_length, myrank_sub

      use vasp_variables, only : &
     &   stress_tensor

!-----------------------------------------------------------------------
!     /*   variables and methods from VASP6                           */
!-----------------------------------------------------------------------

      use VASP_LIB, only : VASP_INIT, VASP_RUNCALC, TIFOR, TOTEN, &
     &   TSIF, DYN, LATT_CUR, VASP_RUNCALCSEC, E, NORDER, &
     &   VASP_REINIT, FINALINIT, INFO

      use LATTICE, only : LATTIC

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer, save :: iset = 0
      logical, save :: free_en = .TRUE.
      
      character(len=3) :: char_num
      character(len=12):: tmp_get_enstat
      
      integer :: ibead, i, j
      real(8) :: stress_tensor_kb_sum(3,3)

!-----------------------------------------------------------------------
!     /*   initialize stress tensor                                   */
!-----------------------------------------------------------------------

      stress_tensor(:,:) = 0.d0
      stress_tensor_kb_sum(:,:) = 0.d0

!-----------------------------------------------------------------------
!     /*   initialize the directories and VASP6                       */
!-----------------------------------------------------------------------

      if ( iset == 0) then
      
      call read_char_MPI (tmp_get_enstat , 12, '<vasp_energy>', &
    &                     13, iounit )

      if( tmp_get_enstat(1:11) .eq. 'FREE_ENERGY' ) then
         free_en = .TRUE.
      else if( tmp_get_enstat(1:12) .eq. 'EXTRAPOLATED' ) then
         free_en = .FALSE.
      else
         if( myrank .eq. 0 ) then
            write(6, '(a)') 'Error termination.'
            write(6, '(a)') 'Wrong setting for <vasp_energy>'
         end if
         
         call my_mpi_barrier
         call error_handling_MPI &
     &         ( 1, 'subroutine force_vasp6_MPI', 26 )
      end if

      if ( myrank .eq. 0 ) then
         do ibead = 1, np_beads
            call int3_to_char( ibead, char_num )
            call system ('rm -f -r ./' // char_num )
            call system ('mkdir -p ./' // char_num )
            call system ("cp INCAR ./" // char_num // "/INCAR")
            call system ("cp KPOINTS ./"  &
     &                      // char_num // "/KPOINTS")
            call system ("cp POSCAR ./" // char_num // "/POSCAR")
            call system ("cp POTCAR ./" // char_num // "/POTCAR")
         end do
      end if

!     /*   call MPI_barrier   */
      call my_mpi_barrier


            do ibead = 1, np_beads
                  /*   allocated bead only   */
                  if ( mod(ibead-1,nprocs_main) .ne. myrank_main ) cycle
                  call int3_to_char( ibead, char_num )
                  call CHDIR('./' // char_num)
                  call VASP_INIT(mpi_comm_sub)
                  call VASP_REINIT
                  call FINALINIT
                  call VASP_RUNCALC
                  call CHDIR('./..')
            end do
            iset = 1
      end if

!     /*   call MPI_barrier   */
      call my_mpi_barrier

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!     /*   skip if `ibead is not my job'                              */
!-----------------------------------------------------------------------

      if ( mod(ibead-1,nprocs_main) .ne. myrank_main ) cycle

!-----------------------------------------------------------------------
!     /*   Pass coordinates and box to VASP                           */
!-----------------------------------------------------------------------
      
      do i = 1, natom
     
           DYN%POSION(1, i) = x(i, ibead) * boxinv(1, 1) +  &
     & y(i, ibead) * boxinv(1, 2) + z(i, ibead) * boxinv(1, 3)
           DYN%POSION(2, i) = x(i, ibead) * boxinv(2, 1) +  &
     & y(i, ibead) * boxinv(2, 2) + z(i, ibead) * boxinv(2, 3)
           DYN%POSION(3, i) = x(i, ibead) * boxinv(3, 1) +  &
     & y(i, ibead) * boxinv(3, 2) + z(i, ibead) * boxinv(3, 3)
     
      end do 

      LATT_CUR%A = box(:,:) * au_length *1.d+10
      call LATTIC(LATT_CUR)


!-----------------------------------------------------------------------
!     /*   execute VASP subroutine                                    */
!-----------------------------------------------------------------------
      call VASP_RUNCALCSEC
      call VASP_RUNCALC
   
!-----------------------------------------------------------------------
!     /* if VASP Failed to converge, try to restart with random WF    */
!-----------------------------------------------------------------------
      if (INFO%LABORT) then
         call VASP_RUNCALCSEC
         call VASP_REINIT
         call VASP_RUNCALC
         if (INFO%LABORT) then
           if ( myrank_sub .eq. 0 ) then
               write( 6, '(a)' ) 'Error - VASP Failed to Converge.'
               write( 6, * ) 'In bead number : ', ibead
               write( 6, '(a)' ) 'Try increasing scf steps (NELM) in INCAR'
           endif
           call error_handling_MPI( 1, 'subroutine force_vasp6_MPI', 26 )
        end if
      endif
      
!-----------------------------------------------------------------------
!     /*   Get Energy, Forces and Stress Tensor from VASP             */
!-----------------------------------------------------------------------

!     /*   get total energy   */
      if(free_en) then
         pot(ibead) = TOTEN
      else
         pot(ibead) = TOTEN-E%EENTROPY/(2+NORDER)  
      end if

      pot(ibead) = pot(ibead) * au_charge / au_energy

!     /*   get force   */
      do i = 1, natom
         fx(i,ibead) = TIFOR(1, i) * au_charge / au_energy  &
     &                                * au_length * 1d10
         fy(i,ibead) = TIFOR(2, i) * au_charge / au_energy  &
     &                                * au_length * 1d10
         fz(i,ibead) = TIFOR(3, i) * au_charge / au_energy  &
     &                                * au_length * 1d10
      end do

!     /*   get stress   */
      stress_tensor_kb_sum(:,:) = stress_tensor_kb_sum(:,:) &
     &                          + TSIF(:,:) / LATT_CUR%OMEGA

!     /*   virial of each bead   */
      if ( method(1:6) .eq. 'REHMC ' ) then
         vir_bead(:,:,ibead) = vir_bead(:,:,ibead) &
     &                       + TSIF(:,:) * volume_bead(ibead) &
     &                       * 1.e+8 / au_energy * au_length**3 &
     &                       * 1.60217733D-19*1D22 / LATT_CUR%OMEGA
      end if

!     /* Not implemented yet */
!     /*   get dipole moment   */
!      dipx(ibead) = dipole_x
!      dipy(ibead) = dipole_y
!      dipz(ibead) = dipole_z

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   change units from kilobar to hartree/bohr**3               */
!-----------------------------------------------------------------------

!     /*   get stress   */
      stress_tensor(:,:) = stress_tensor_kb_sum(:,:) * 1.e+8 / au_energy &
     &                            * au_length**3 &
     &                           *1.60217733D-19*1D22

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1_main ( pot, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2_main ( fx, natom, nbead )
      call my_mpi_allreduce_real_2_main ( fy, natom, nbead )
      call my_mpi_allreduce_real_2_main ( fz, natom, nbead )

!     /* Not implemented yet */
!     /*   dipole moment   */
!      call my_mpi_allreduce_real_1_main ( dipx, nbead )
!      call my_mpi_allreduce_real_1_main ( dipy, nbead )
!      call my_mpi_allreduce_real_1_main ( dipz, nbead )

!     /*   stress   */
      call my_mpi_allreduce_real_2_main ( stress_tensor, 3, 3 )

!     /*   virial of each bead   */
      if ( method(1:6) .eq. 'REHMC ' ) then
         call my_mpi_allreduce_real_3_main ( vir_bead, 3, 3, nbead )
      end if

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

      if ( method(1:6) .eq. 'REHMC ' ) then

         do j = 1, nbead

            vir(1,1) = vir(1,1) + vir_bead(1,1,j) / nbead
            vir(1,2) = vir(1,2) + vir_bead(1,2,j) / nbead
            vir(1,3) = vir(1,3) + vir_bead(1,3,j) / nbead
            vir(2,1) = vir(2,1) + vir_bead(2,1,j) / nbead
            vir(2,2) = vir(2,2) + vir_bead(2,2,j) / nbead
            vir(2,3) = vir(2,3) + vir_bead(2,3,j) / nbead
            vir(3,1) = vir(3,1) + vir_bead(3,1,j) / nbead
            vir(3,2) = vir(3,2) + vir_bead(3,2,j) / nbead
            vir(3,3) = vir(3,3) + vir_bead(3,3,j) / nbead

         end do

      else

         do j = 1, nbead

            vir(1,1) = vir(1,1) + stress_tensor(1,1) * volume / nbead
            vir(1,2) = vir(1,2) + stress_tensor(1,2) * volume / nbead
            vir(1,3) = vir(1,3) + stress_tensor(1,3) * volume / nbead
            vir(2,1) = vir(2,1) + stress_tensor(2,1) * volume / nbead
            vir(2,2) = vir(2,2) + stress_tensor(2,2) * volume / nbead
            vir(2,3) = vir(2,3) + stress_tensor(2,3) * volume / nbead
            vir(3,1) = vir(3,1) + stress_tensor(3,1) * volume / nbead
            vir(3,2) = vir(3,2) + stress_tensor(3,2) * volume / nbead
            vir(3,3) = vir(3,3) + stress_tensor(3,3) * volume / nbead

         end do

      end if

      return
      end

!***********************************************************************
      subroutine finalize_vasp6_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   VASP6 function                                            */
!-----------------------------------------------------------------------
      use VASP_LIB, only : FINALIZE_VASP

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none
!-----------------------------------------------------------------------
!     /*   finalize VASP                                              */
!-----------------------------------------------------------------------

!     /*   finalize VASP subroutine   */
      call FINALIZE_VASP

      return
      end



!     // vasp6 not defined
#else



!***********************************************************************
      subroutine force_vasp6_MPI( )
!***********************************************************************

      use common_variables, only : myrank

      implicit none

      if( myrank .eq. 0 ) then

         write(6, '(a)') 'Error termination.'
         write(6, '(a)') ''
         write(6, '(a)') 'ipotential=VASP6 is not available because'
         write(6, '(a)') 'VASP6 subroutines were not linked.'
         write(6, '(a)') 'Compile the pimd.mpi.x with the following.'
         write(6, '(a)') ''
         write(6, '(a)') '  FLAGMP = -Dvasp6'
         write(6, '(a)') ''
      end if

      call my_mpi_barrier
      call my_mpi_abort

      return
      end

!***********************************************************************
      subroutine finalize_vasp6_MPI( ierr )
!***********************************************************************

      use common_variables, only : myrank

      implicit none

      integer, intent(out) :: ierr

      ierr = 1

      if( myrank .eq. 0 ) then

         write(6, '(a)') 'Error termination.'
         write(6, '(a)') ''
         write(6, '(a)') &
     &      'VASP is not available. To link VASP6 routines, ' &
     &      // ' recompile pimd.mpi.x with the following options'
         write(6, '(a)') ''
         write(6, '(a)') '  FLAGMP = -Dvasp6'
         write(6, '(a)') ''

      end if

      call my_mpi_barrier
      call my_mpi_abort

      return
      end

!     // vasp6
#endif

