!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jan 26, 2022 by M. Shiga
!      Description:     energy and force from VASP calculation
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      module vasp5_variables
!***********************************************************************

!     /*    stress tensor (au)   */
      real(8) :: stress_tensor(3,3)

!     /*    stress tensor (kB)   */
      real(8) :: stress_tensor_kb(3,3)

!***********************************************************************
      end module vasp5_variables
!***********************************************************************





!***********************************************************************
      subroutine force_vasp5
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   vir, vir_bead, au_energy, au_length, &
     &   volume_bead, volume, nbead, method

      use vasp5_variables, only : &
     &   stress_tensor, stress_tensor_kb

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ibead, j, ierr

      character(len=3) :: char_num

      integer, save :: iset = 0

      real(8) :: stress_tensor_kb_sum(3,3)

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         do ibead = 1, nbead

!           /*   character   */
            call int3_to_char( ibead, char_num )

!           /*   remove old directory and create new directory   */
            call system ('mkdir -p ./' // char_num )

!           /*   copy data files   */
            call system ('cp -f input.dat          ./' // char_num )
            call system ('cp -f input_default.dat  ./' // char_num )
            call system ('cp -f poscar.dat         ./' // char_num )
            call system ('cp -f structure.dat      ./' // char_num )
            call system ('cp -f INCAR              ./' // char_num )
            call system ('cp -f KPOINTS            ./' // char_num )
            call system ('cp -f POTCAR             ./' // char_num )

         end do

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   initialize stress tensor                                   */
!-----------------------------------------------------------------------

      stress_tensor(:,:) = 0.d0
      stress_tensor_kb_sum(:,:) = 0.d0

!-----------------------------------------------------------------------
!        /*   make input                                              */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!        /*   character   */
         call int3_to_char( ibead, char_num )

!        /*   make input   */
         call iovasp ( 1, ibead, char_num )

!        /*   run vasp   */
         call vasp5_run ( char_num, 3, ierr )

!        /*   read output   */
         call iovasp ( 2, ibead, char_num )

!        /*   get stress   */
         stress_tensor_kb_sum(:,:) = stress_tensor_kb_sum(:,:) &
     &                             + stress_tensor_kb(:,:)

!        /*   virial of each bead   */
         if ( method(1:6) .eq. 'REHMC ' ) then
            vir_bead(:,:,ibead) = vir_bead(:,:,ibead) &
     &                          + stress_tensor_kb(:,:) &
     &                          * 1.e+8 / au_energy * au_length**3 &
     &                          * volume_bead(ibead)
         end if

      end do

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

!     /*   get stress   */
      stress_tensor(:,:) = stress_tensor_kb_sum(:,:) &
     &                   * 1.e+8 / au_energy * au_length**3

      if ( method(1:6) .eq. 'REHMC ' ) then

         do j = 1, nbead

            vir(1,1) = vir(1,1) + vir_bead(1,1,j) / nbead
            vir(1,2) = vir(1,2) + vir_bead(1,2,j) / nbead
            vir(1,3) = vir(1,3) + vir_bead(1,3,j) / nbead
            vir(2,1) = vir(2,1) + vir_bead(2,1,j) / nbead
            vir(2,2) = vir(2,2) + vir_bead(2,2,j) / nbead
            vir(2,3) = vir(2,3) + vir_bead(2,3,j) / nbead
            vir(3,1) = vir(3,1) + vir_bead(3,1,j) / nbead
            vir(3,2) = vir(3,2) + vir_bead(3,2,j) / nbead
            vir(3,3) = vir(3,3) + vir_bead(3,3,j) / nbead

         end do

      else

         do j = 1, nbead

            vir(1,1) = vir(1,1) + stress_tensor(1,1) * volume / nbead
            vir(1,2) = vir(1,2) + stress_tensor(1,2) * volume / nbead
            vir(1,3) = vir(1,3) + stress_tensor(1,3) * volume / nbead
            vir(2,1) = vir(2,1) + stress_tensor(2,1) * volume / nbead
            vir(2,2) = vir(2,2) + stress_tensor(2,2) * volume / nbead
            vir(2,3) = vir(2,3) + stress_tensor(2,3) * volume / nbead
            vir(3,1) = vir(3,1) + stress_tensor(3,1) * volume / nbead
            vir(3,2) = vir(3,2) + stress_tensor(3,2) * volume / nbead
            vir(3,3) = vir(3,3) + stress_tensor(3,3) * volume / nbead

         end do

      end if

      return
      end





!***********************************************************************
      subroutine iovasp ( ioption, ibead, char_num )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, box, natom, iounit

      use vasp5_variables, only : &
     &   stress_tensor_kb

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer          :: i, ibead, ioption, ierr

      character(len=3) :: char_num

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      ierr = 0

      if ( ioption .eq. 1 ) then

         open ( iounit, file = char_num // '/geometry.ini' )

         do i = 1, natom
            write( iounit, '(i8,6e24.16,3i4)' ) &
     &         1, x(i,ibead), y(i,ibead), z(i,ibead), &
     &         0.d0, 0.d0, 0.d0, 0, 0, 0
         end do

         close ( iounit )

         open ( iounit, file = char_num // '/box.ini' )

         write( iounit, '(i8,3e24.16)' ) 1, box(1,1), box(1,2), box(1,3)
         write( iounit, '(i8,3e24.16)' ) 1, box(2,1), box(2,2), box(2,3)
         write( iounit, '(i8,3e24.16)' ) 1, box(3,1), box(3,2), box(3,3)

         close ( iounit )

      else if ( ioption .eq. 2 ) then

         open ( iounit, file = char_num // '/results.vasp' )

!        /*   potential   */

         read( iounit, *, iostat=ierr ) pot(ibead)

         /*   error termination  */
         call error_handling ( ierr, 'subroutine iovasp', 17 )

!        /*   gradient   */

         do i = 1, natom
            read( iounit, *, iostat=ierr ) &
     &         fx(i,ibead), fy(i,ibead), fz(i,ibead)
         end do

         /*   error termination  */
         call error_handling ( ierr, 'subroutine iovasp', 17 )

!        /*   gradient -> force  */
         do i = 1, natom
            fx(i,ibead) = - fx(i,ibead)
            fy(i,ibead) = - fy(i,ibead)
            fz(i,ibead) = - fz(i,ibead)
         end do

!        /*   stress in kb  */
         do i = 1, 3
            read( iounit, *, iostat=ierr ) stress_tensor_kb(i,1:3)
         end do

         /*   error termination  */
         call error_handling ( ierr, 'subroutine iovasp', 17 )

         close ( iounit )

      end if

      return
      end





!***********************************************************************
      subroutine vasp5_run( vasp_dir, len_vasp_dir, ierr )
!***********************************************************************

!     /*   number of atoms   */
      integer :: natom

!     /*   character length   */
      integer :: len_vasp_dir

!     /*   geometry   */
      real(8), dimension(:,:), allocatable:: x, y, z

!     /*   potential   */
      real(8), dimension(:),   allocatable:: pot

!     /*   gradient   */
      real(8), dimension(:,:), allocatable:: gx, gy, gz

!     /*   stress   */
      real(8), dimension(3,3) :: stress_kb

!     /*   unit cell matrix   */
      real(8), dimension(3,3) :: box, boxinv

!     /*   vasp execution command   */
      character(len=80) :: vasp_command

!     /*   key words to find energy values   */
      character(len=80) :: vasp_keyword_e
      integer           :: vasp_line_e, vasp_column_e

!     /*   key words to find gradient values   */
      character(len=80) :: vasp_keyword_g
      integer           :: vasp_line_g, vasp_column_g

!     /*   key words to find stress values   */
      character(len=80) :: vasp_keyword_s
      integer           :: vasp_line_s, vasp_column_s

!     /*   directory   */
      character(len=len_vasp_dir) :: vasp_dir

!     /*   file numbers   */
      integer :: iounit         = 10

!     /*   vasp file   */
      integer :: iounit_vasp    = 47

!     /*   characters   */
      character(len=80) :: char_line, char_line_2, char_dummy(10)

!     /*   unit conversion factors   */
      real(8), parameter :: au_length = 0.529177249d-10
      real(8), parameter :: au_energy = 4.3597482d-18
      real(8), parameter :: au_charge = 1.60217646d-19
      real(8), parameter :: bohr2ang  = au_length*1.d+10

!     /*   real numbers   */
      real(8) :: xi, yi, zi, ai, bi, ci

!     /*   integers   */
      integer :: i, j, idummy, ierr, l

!-----------------------------------------------------------------------
!     /*   read number of atoms                                       */
!-----------------------------------------------------------------------

      open ( iounit, file = trim(vasp_dir) // '/input.dat' )
         do
            read ( iounit, *, iostat=ierr )  char_line
            if ( char_line(1:7) .eq. '<natom>' ) then
               read( iounit, *, iostat=ierr ) natom
               exit
            end if
            if ( ierr .ne. 0 ) exit
         end do
      close( iounit )

      if ( ierr .ne. 0 )  then
         open ( iounit, file = trim(vasp_dir) // '/structure.dat' )
            read( iounit, *, iostat=ierr ) natom
         close( iounit )
      end if

      if ( natom .eq. 0 ) ierr = 1

      call error_handling_vasp5( ierr, 'reading natom', 13 )
      if ( ierr .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      allocate ( x(natom,1) )
      allocate ( y(natom,1) )
      allocate ( z(natom,1) )

      allocate ( pot(1) )

      allocate ( gx(natom,1) )
      allocate ( gy(natom,1) )
      allocate ( gz(natom,1) )

!-----------------------------------------------------------------------
!     /*   zero clear                                                 */
!-----------------------------------------------------------------------

      pot(:)   =  0.d0

      gx(:,:)  =  0.d0
      gy(:,:)  =  0.d0
      gz(:,:)  =  0.d0

!-----------------------------------------------------------------------
!     /*   read geometry                                              */
!-----------------------------------------------------------------------

      open ( iounit, file = trim(vasp_dir) // '/geometry.ini' )

      do i = 1, natom
         read( iounit, *, iostat=ierr ) &
     &      idummy, x(i,1), y(i,1), z(i,1)
      end do

      close( iounit )

      call error_handling_vasp5( ierr, 'reading geometry.ini', 20 )
      if ( ierr .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   set vasp execution command                                 */
!-----------------------------------------------------------------------

      open ( iounit, file = trim(vasp_dir) // '/input.dat' )
         do
            read ( iounit, *, iostat=ierr )  char_line
            if ( char_line(1:14) .eq. '<vasp_command>' ) then
                  read ( iounit, *, iostat=ierr ) vasp_command
               exit
            end if
            if ( ierr .ne. 0 ) exit
         end do
      close( iounit )

      if ( ierr .ne. 0 )  then
         open ( iounit, file = trim(vasp_dir) // '/input_default.dat' )
            do
               read ( iounit, *, iostat=ierr )  char_line
               if ( char_line(1:14) .eq. '<vasp_command>' ) then
                     read ( iounit, *, iostat=ierr ) vasp_command
                  exit
               end if
            end do
         close( iounit )
         call error_handling_vasp5( ierr, 'reading vasp_command', 20 )
         if ( ierr .ne. 0 ) return
      end if

!-----------------------------------------------------------------------
!     /*   set vasp keyword: energy                                   */
!-----------------------------------------------------------------------

      open ( iounit, file = trim(vasp_dir) // '/input.dat' )
         do
            read ( iounit, *, iostat=ierr )  char_line
            if ( char_line(1:21) .eq. '<vasp_keyword_energy>' ) then
               read ( iounit, *, iostat=ierr ) &
     &            vasp_keyword_e, vasp_line_e, vasp_column_e
               exit
            end if
            if ( ierr .ne. 0 ) exit
         end do
      close( iounit )

      if ( ierr .ne. 0 )  then
         open ( iounit, file = trim(vasp_dir) // '/input_default.dat' )
            do
               read ( iounit, *, iostat=ierr )  char_line
               if( char_line(1:21) .eq.'<vasp_keyword_energy>' ) then
                  read ( iounit, *, iostat=ierr ) &
     &               vasp_keyword_e, vasp_line_e, vasp_column_e
                  exit
               end if
               if ( ierr .ne. 0 ) exit
            end do
         close( iounit )
         call error_handling_vasp5( ierr, 'reading vasp_keyword_e', 22 )
         if ( ierr .ne. 0 ) return
      end if

!-----------------------------------------------------------------------
!     /*   set vasp keyword: gradient                                 */
!-----------------------------------------------------------------------

      open ( iounit, file = trim(vasp_dir) // '/input.dat' )
         do
            read ( iounit, *, iostat=ierr )  char_line
            if( char_line(1:23) .eq. '<vasp_keyword_gradient>' ) then
               read ( iounit, *, iostat=ierr ) &
     &            vasp_keyword_g, vasp_line_g, vasp_column_g
               exit
            end if
            if ( ierr .ne. 0 ) exit
         end do
      close( iounit )

      if ( ierr .ne. 0 )  then
         open ( iounit, file = trim(vasp_dir) // '/input_default.dat' )
            do
               read ( iounit, *, iostat=ierr )  char_line
               if( char_line(1:23).eq.'<vasp_keyword_gradient>') then
                  read ( iounit, *, iostat=ierr ) &
     &               vasp_keyword_g, vasp_line_g, vasp_column_g
                  exit
               end if
               if ( ierr .ne. 0 ) exit
            end do
         close( iounit )
         call error_handling_vasp5( ierr, 'reading vasp_keyword_g', 22 )
         if ( ierr .ne. 0 ) return
      end if

!-----------------------------------------------------------------------
!     /*   set vasp keyword: stress                                   */
!-----------------------------------------------------------------------

      open ( iounit, file = trim(vasp_dir) // '/input.dat' )
         do
            read ( iounit, *, iostat=ierr )  char_line
            if( char_line(1:21) .eq. '<vasp_keyword_stress>' ) then
               read ( iounit, *, iostat=ierr ) &
     &            vasp_keyword_s, vasp_line_s, vasp_column_s
               exit
            end if
            if ( ierr .ne. 0 ) exit
         end do
      close( iounit )

      if ( ierr .ne. 0 )  then
         open ( iounit, file = trim(vasp_dir) // '/input_default.dat' )
            do
               read ( iounit, *, iostat=ierr )  char_line
               if( char_line(1:21) .eq. '<vasp_keyword_stress>' ) then
                  read ( iounit, *, iostat=ierr ) &
     &               vasp_keyword_s, vasp_line_s, vasp_column_s
                  exit
               end if
               if ( ierr .ne. 0 ) exit
            end do
         close( iounit )
         call error_handling_vasp5( ierr, 'reading vasp_keyword_s', 22 )
         if ( ierr .ne. 0 ) return
      end if

!-----------------------------------------------------------------------
!     /*   read unit cell:  box.ini (not poscar.dat)                  */
!-----------------------------------------------------------------------

!     /*   open the vasp prototype file   */
      open( iounit, file = trim(vasp_dir) // '/box.ini' )

!        /*   3rd line: x of lattice vectors abc   */
         read ( iounit, * ) i, box(1,1), box(1,2), box(1,3)

!        /*   4th line: y of lattice vectors abc   */
         read ( iounit, * ) i, box(2,1), box(2,2), box(2,3)

!        /*   5th line: z of lattice vectors abc   */
         read ( iounit, * ) i, box(3,1), box(3,2), box(3,3)

!     /*   close files   */
      close(iounit)

!     /*   error handling   */
      call error_handling_vasp5( ierr, 'reading poscar', 14 )
      if ( ierr .ne. 0 ) return

!     /*   inverse of unit cell matrix   */
      call inv3( box, boxinv )

!-----------------------------------------------------------------------
!     /*   make vasp input:  POSCAR                                   */
!-----------------------------------------------------------------------

!     /*   open the vasp prototype file   */
      open( iounit, file = trim(vasp_dir) // '/poscar.dat' )

!     /*   open the vasp input file   */
      open( iounit_vasp, file = trim(vasp_dir) // '/POSCAR' )

!        /*   1st line: comment   */
         read ( iounit, '(a)', iostat=ierr ) char_line
         write( iounit_vasp, '(a)' )         char_line

!        /*   2nd line: scaling factor (lattice constant)   */
         read ( iounit, '(a)', iostat=ierr ) char_line
         write( iounit_vasp, '(a)' )         '1.d0'

!        /*   3rd line: lattice vector a   */
         read ( iounit, '(a)', iostat=ierr ) char_line
         write( iounit_vasp, '(3e24.16)' ) &
     &      box(1,1)*bohr2ang, box(2,1)*bohr2ang, box(3,1)*bohr2ang

!        /*   4th line: lattice vector b   */
         read ( iounit, '(a)', iostat=ierr ) char_line
         write( iounit_vasp, '(3e24.16)' ) &
     &      box(1,2)*bohr2ang, box(2,2)*bohr2ang, box(3,2)*bohr2ang

!        /*   5th line: lattice vector c   */
         read ( iounit, '(a)', iostat=ierr ) char_line
         write( iounit_vasp, '(3e24.16)' ) &
     &      box(1,3)*bohr2ang, box(2,3)*bohr2ang, box(3,3)*bohr2ang

!        /*   6th line: number of atomic species (not read)   */
         read ( iounit, '(a)', iostat=ierr ) char_line
         write( iounit_vasp, '(a)' )         char_line

!        /*   7th line: number of atomic species (not read)   */
         read ( iounit, '(a)', iostat=ierr ) char_line
         write( iounit_vasp, '(a)' )         char_line

!        /*   8th line: 'Cartesian'   */
         read ( iounit, '(a)', iostat=ierr ) char_line
         write( iounit_vasp, '(a)' )         char_line

!        /*   loop of atoms   */
         do i = 1, natom

!           /*   atomic positions   */
            xi = x(i,1)
            yi = y(i,1)
            zi = z(i,1)

!           /*   periodic boundary condition   */
            call pbc_atom_mod( xi, yi, zi, box, boxinv, 1 )

!           /*   internal coordinates   */
            ai = boxinv(1,1)*xi + boxinv(1,2)*yi + boxinv(1,3)*zi
            bi = boxinv(2,1)*xi + boxinv(2,2)*yi + boxinv(2,3)*zi
            ci = boxinv(3,1)*xi + boxinv(3,2)*yi + boxinv(3,3)*zi

!           /*   write internal coordinates   */
            write ( iounit_vasp, '(3f18.10)' ) &
     &           xi*bohr2ang, yi*bohr2ang, zi*bohr2ang

!        /*   loop of atoms   */
         end do

!        /*   add a blank line at the end   */
         write ( iounit_vasp, '(a)' )

!     /*   close files   */
      close(iounit)
      close(iounit_vasp)

!     /*   error handling   */
      call error_handling_vasp5( ierr, 'reading poscar', 14 )
      if ( ierr .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   run vasp
!-----------------------------------------------------------------------

!     /*   run vasp   */
      call system ( 'cd ' // trim(vasp_dir) // '; ' // vasp_command )

!-----------------------------------------------------------------------
!     /*   read vasp output:  potential                               */
!     /*      5th data in the second line after the vasp keyword      */
!-----------------------------------------------------------------------

!     /*   length of the keyword   */
      l = len_trim(vasp_keyword_e)

!     /*   open the vasp output file   */
      open( iounit_vasp, file = trim(vasp_dir) // '/OUTCAR' )

      do

!        /*   read a line   */
         read ( iounit_vasp, '(a)', iostat=ierr ) char_line

!        /*   error handling   */
         call error_handling_vasp5( ierr, 'reading energy', 14 )
         if ( ierr .ne. 0 ) return

!        /*   adjusted   */
         char_line_2 = adjustl(char_line)

!        /*   look for energy   */
         if ( char_line_2(1:l) .eq. vasp_keyword_e(1:l) ) then

!           /*   go back one line   */
            backspace( iounit_vasp )

            do i = 1, vasp_line_e
               read( iounit_vasp, *, iostat=ierr )
            end do

!           /*   read potential   */
            read( iounit_vasp, *, iostat=ierr ) &
     &         ( char_dummy(i), i=1, vasp_column_e-1 ), pot(1)

!           /*   error handling   */
            call error_handling_vasp5( ierr, 'reading energy', 14 )
            if ( ierr .ne. 0 ) return

!           /*   exit cycle   */
            exit

         end if

      end do

!     /*   close file   */
      close(iounit_vasp)

!     /*   electron volt --> hartree   */
      pot(1) = pot(1) * au_charge / au_energy

!-----------------------------------------------------------------------
!     /*   read vasp output:  gradient                                */
!     /*      4th, 5th, 6th data starting from the second line after  */
!     /*      the vasp keyword                                        */
!-----------------------------------------------------------------------

!     /*   open the vasp output file   */
      open( iounit_vasp, file = trim(vasp_dir) // '/OUTCAR' )

      do

!        /*   read a line   */
         read ( iounit_vasp, '(a)', iostat=ierr ) char_line

!        /*   error handling   */
         call error_handling_vasp5( ierr, 'reading force', 13 )
         if ( ierr .ne. 0 ) return

!        /*   length of the keyword   */
         l = len_trim(vasp_keyword_g)

!        /*   adjusted   */
         char_line_2 = adjustl(char_line)

!        /*   look for gradient   */
         if ( char_line_2(1:l) .eq. vasp_keyword_g(1:l) ) then

!           /*   go back one line   */
            backspace( iounit_vasp )

!           /*   read skip   */
            do i = 1, vasp_line_g
               read( iounit_vasp, *, iostat=ierr )
            end do

!           /*   read force from vasp output     */
            do i = 1, natom
               read( iounit_vasp, *, iostat=ierr ) &
     &            ( char_dummy(j), j=1, vasp_column_g-1 ), &
     &            gx(i,1), gy(i,1), gz(i,1)
            end do

!           /*   error handling   */
            call error_handling_vasp5( ierr, 'reading force', 13 )
            if ( ierr .ne. 0 ) return

!           /*   exit cycle   */
            exit

         end if

      end do

!     /*   close file   */
      close(iounit_vasp)

!-----------------------------------------------------------------------
!     /*   read vasp output:  stress                                  */
!     /*      from 2nd to 7th data on the 14th line after the vasp    */
!     /*      keyword                                                 */
!-----------------------------------------------------------------------

!     /*   open the vasp output file   */
      open( iounit_vasp, file = trim(vasp_dir) // '/OUTCAR' )

      do

!        /*   read a line   */
         read ( iounit_vasp, '(a)', iostat=ierr ) char_line

!        /*   error handling   */
         call error_handling_vasp5( ierr, 'reading stress', 14 )
         if ( ierr .ne. 0 ) return

!        /*   length of the keyword   */
         l = len_trim(vasp_keyword_s)

!        /*   adjusted   */
         char_line_2 = adjustl(char_line)

!        /*   look for gradient   */
         if ( char_line_2(1:l) .eq. vasp_keyword_s(1:l) ) then

!           /*   go back one line   */
            backspace( iounit_vasp )

!           /*   read skip   */
            do i = 1, vasp_line_s
               read( iounit_vasp, *, iostat=ierr )
            end do

!           /*   read force from vasp output     */
            read( iounit_vasp, *, iostat=ierr ) &
     &         ( char_dummy(i), i=1, vasp_column_s-1 ), &
     &         stress_kb(1,1), stress_kb(2,2), stress_kb(3,3), &
     &         stress_kb(1,2), stress_kb(2,3), stress_kb(3,1)

!           /*   error handling   */
            call error_handling_vasp5( ierr, 'reading stress', 14 )
            if ( ierr .ne. 0 ) return

            stress_kb(2,1) = stress_kb(1,2)
            stress_kb(3,2) = stress_kb(2,3)
            stress_kb(1,3) = stress_kb(3,1)

!           /*   exit cycle   */
            exit

         end if

      end do

!     /*   close file   */
      close(iounit_vasp)

!-----------------------------------------------------------------------
!     /*   force -> gradient                                          */
!-----------------------------------------------------------------------

      do i = 1, natom
         gx(i,1) = - gx(i,1)
         gy(i,1) = - gy(i,1)
         gz(i,1) = - gz(i,1)
      end do

!-----------------------------------------------------------------------
!     /*   electron volt / angstrom --> hartree / bohr                */
!-----------------------------------------------------------------------

      do i = 1, natom
         gx(i,1) = gx(i,1) *au_charge/au_energy*au_length/1.d-10
         gy(i,1) = gy(i,1) *au_charge/au_energy*au_length/1.d-10
         gz(i,1) = gz(i,1) *au_charge/au_energy*au_length/1.d-10
      end do

!-----------------------------------------------------------------------
!     /*   change sign                                                */
!-----------------------------------------------------------------------

!cc      stress_kb(:,:) = - stress_kb(:,:)

!-----------------------------------------------------------------------
!     /*   write potential and gradient                               */
!-----------------------------------------------------------------------

!     /*   open the vasp output file   */
      open( iounit, file = trim(vasp_dir) // '/results.vasp' )

!     /*   potential   */
      write( iounit, '(e24.16)' ) pot(1)

!     /*   gradient   */
      do i = 1, natom
         write( iounit, '(3e24.16)', iostat=ierr ) &
     &      gx(i,1), gy(i,1), gz(i,1)
      end do

!     /*   stress in kb   */
      do i = 1, 3
         write( iounit, '(3e24.16)', iostat=ierr ) stress_kb(i,1:3)
      end do

!     /*   close file  */
      close( iounit )

      return
      end





!***********************************************************************
      subroutine pbc_atom_mod ( xi, yi, zi, box, boxinv, iboundary )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8) :: ai, bi, ci, xi, yi, zi

      real(8), dimension(3,3) :: box, boxinv

      integer :: iboundary

!-----------------------------------------------------------------------
!     /*   apply boundary condition                                   */
!-----------------------------------------------------------------------

      if      ( iboundary .eq. 1 ) then

         ai = boxinv(1,1)*xi + boxinv(1,2)*yi + boxinv(1,3)*zi
         bi = boxinv(2,1)*xi + boxinv(2,2)*yi + boxinv(2,3)*zi
         ci = boxinv(3,1)*xi + boxinv(3,2)*yi + boxinv(3,3)*zi

!        /*   shift to the range -box/2 < x < +box/2   */

         ai = ai - nint(ai)
         bi = bi - nint(bi)
         ci = ci - nint(ci)

         xi = box(1,1)*ai + box(1,2)*bi + box(1,3)*ci
         yi = box(2,1)*ai + box(2,2)*bi + box(2,3)*ci
         zi = box(3,1)*ai + box(3,2)*bi + box(3,3)*ci

      else if ( iboundary .eq. 2 ) then

         ai = boxinv(1,1)*xi + boxinv(1,2)*yi + boxinv(1,3)*zi
         bi = boxinv(2,1)*xi + boxinv(2,2)*yi + boxinv(2,3)*zi
         ci = boxinv(3,1)*xi + boxinv(3,2)*yi + boxinv(3,3)*zi

!        /*   shift to the range -box/2 < x < +box/2   */

         ai = ai - nint(ai)
         bi = bi - nint(bi)
         ci = ci - nint(ci)

         xi = box(1,1)*ai + box(1,2)*bi + box(1,3)*ci
         yi = box(2,1)*ai + box(2,2)*bi + box(2,3)*ci
         zi = box(3,1)*ai + box(3,2)*bi + box(3,3)*ci

      end if

      return
      end





!***********************************************************************
      subroutine error_handling_vasp5( ierr, char_tag, length_tag )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer:: ierr, length_tag

      character(len=length_tag) :: char_tag

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      if ( ierr .ne. 0 ) then

         write(6,'(a)') 'Error found at: ' // char_tag // '.'

         write(6,'(a)')

      end if

      return
      end
