!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     energy and force defined by the user
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine force_user_MPI
!***********************************************************************
!
!     - Set execution code by <user_command> keyword in input.dat.
!     - The execution code reads position.user.
!     - The execution code writes potential.user, forces.user,
!       and optionally virial.user.
!
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, natom, nbead, iounit, &
     &   user_command, user_output_file, user_input_file, myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, ierr

      real(8) :: v1, v2, v3

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   read user execution command                                */
!-----------------------------------------------------------------------

!     /*   first visit only   */
      if ( iset .eq. 0 ) then

!        /*   read command within eighty letters  */
         call read_char_MPI &
     &      ( user_command, 80, '<user_command>', 14, iounit )

!        /*   input file name within eighty letters  */
         call read_char_MPI &
     &      ( user_input_file, 80, '<user_input_file>', 17, iounit )

!        /*   input file name within eighty letters  */
         call read_char_MPI &
     &      ( user_output_file, 80, '<user_output_file>', 18, iounit )

!        /*   set complete   */
         iset = 1

!     /*   first visit only   */
      end if

!-----------------------------------------------------------------------
!     /*   write atomic position in atomic units                      */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

      open( iounit, file = user_input_file )

      do j = 1, nbead

         do i = 1, natom
            write( iounit, '(3f16.8)' ) x(i,j), y(i,j), z(i,j)
         end do

      end do

      close( iounit )

      end if

!-----------------------------------------------------------------------
!     /*   user execution command                                     */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

         call system( user_command )

      end if

!-----------------------------------------------------------------------
!     /*   read potential in atomic units                             */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

      open ( iounit, file = user_output_file )

      do j = 1, nbead
         read ( iounit, *, iostat=ierr ) pot(j)
      end do

!-----------------------------------------------------------------------
!     /*   read forces in atomic units                                */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom
         read ( iounit, *, iostat=ierr ) fx(i,j), fy(i,j), fz(i,j)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   read virial in atomic units                                */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, 3

         read ( iounit, *, iostat=ierr ) v1, v2, v3

         if ( ierr .ne. 0 ) exit

         vir(i,1) = vir(i,1) + v1
         vir(i,2) = vir(i,2) + v2
         vir(i,3) = vir(i,3) + v3

      end do
      end do

      close( iounit )

      end if

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz, natom, nbead )

!     /*   virial   */
      call my_mpi_allreduce_real_2 ( vir, 3, 3 )

      return
      end





!***********************************************************************
      subroutine force_eckart_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, fx, pot, vir, nbead, natom, pi, myrank_main, nprocs_main, &
     &   myrank_sub, nprocs_sub

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, m
      real(8) :: b, c
      real(8) :: c1, c2, c3, c6, dc1, dc2, dc3, dc6

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      b = + 13.5d0 / pi
      c = +  8.0d0 / sqrt( 3.d0 * pi )

      do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         do i = 1, natom

            if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

            c1 = exp( x(i,m) / c )
            c2 = 1.d0 / c1
            c3 = 0.5d0 * ( c1 + c2 )
            c6 = 1.d0 / (c3*c3)

            dc1 = + c1 / c
            dc2 = - c2 * c2 * dc1
            dc3 = 0.5d0 * ( dc1 + dc2 )
            dc6 = - 2.d0 / (c3*c3*c3) * dc3

            pot(m) = pot(m) + b*c6

            fx(i,m) = fx(i,m) - b*dc6

         end do

      end do

      vir(:,:) = 0.d0

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

      call my_mpi_allreduce_md

      return
      end
