!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     energy and force from TURBOMOLE calculation
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      module turbo_variables
!***********************************************************************

      integer:: iset_turbo     =  0
      integer:: ioption_turbo  =  0

      character(len=20) :: turbo_guess

      integer:: iset_scf       =  0
      integer:: nsaos_scf      =  0

      real(8), dimension(:,:,:), allocatable:: coeff_scf
      real(8), dimension(6):: weight_scf

!***********************************************************************
      end module turbo_variables
!***********************************************************************





!***********************************************************************
      subroutine force_turbo_MPI
!***********************************************************************

!=======================================================================
!
!     NOTE:  before running, all the necessary files must be prepared
!            in the current directory by the turbomole command "define".
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   xs, ys, zs, fx, fy, fz, pot, charge_mol, dipx, dipy, dipz, &
     &   x, y, z, vir, natom, nbead, turbo_command, &
     &   turbo_control, iounit, iounit_turbo, mbox, myrank, nprocs

      use turbo_variables, only : &
     &   iset_turbo, turbo_guess, ioption_turbo

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr, ibead, i, j, match, itest

      integer :: iop = 0

      integer :: idummy

      real(8) :: pot_all, pot_scf, pot_mp2, pot_cc2, xi, yi, zi

      real(8) :: dummy(2)

      character(len=80):: char_line, char_an, char_file

      character(len=3)::  char_num

      character(len=80):: char_dummy(5)

!      character(len=1000):: char_sh

!-----------------------------------------------------------------------
!     /*   initial settings                                           */
!-----------------------------------------------------------------------

      ierr = 0

      if ( iset_turbo .eq. 0 ) then

!-----------------------------------------------------------------------
!        /*   read turbo:  method must be either hf, dft, or mp2.     */
!        /*                hf, dft, mp2, ridft, rimp2.                */
!-----------------------------------------------------------------------

         if ( myrank .eq. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'input.dat' )

!           /*   search for tag    */
            call search_tag ( '<turbo_command>', 15, iounit, ierr )

!           /*   read first command    */
            read ( iounit, *, iostat=ierr ) &
     &         turbo_command(1), turbo_control(1)

!           /*   read second command    */
            read ( iounit, *, iostat=ierr ) &
     &         turbo_command(2), turbo_control(2)

!           /*   file close   */
            close( iounit )

!           /*   if there is an error, read values from turbo.dat   */
            if ( ierr .ne. 0 ) then

!              /*   check if file called `turbo.dat' exists   */
               call testfile ( 'turbo.dat', 9, itest )

!              /*   if the file exists,   */
               if ( itest .eq. 0 ) then

!                 /*   file open   */
                  open ( iounit, file = 'turbo.dat' )

!                 /*   search for tag    */
                  call search_tag( '<turbo_command>', 15, iounit, ierr )

!                 /*   read first command    */
                  read ( iounit, *, iostat=ierr ) &
     &               turbo_command(1), turbo_control(1)

!                 /*   read second command    */
                  read ( iounit, *, iostat=ierr ) &
     &                  turbo_command(2), turbo_control(2)

!                 /*   file close   */
                  close( iounit )

               end if

            end if

!           /*   if there is an error, read default values   */
            if ( ierr .ne. 0 ) then

!              /*   file open   */
               open ( iounit, file = 'input_default.dat' )

!              /*   search for tag    */
               call search_tag ( '<turbo_command>', 15, iounit, ierr )

!              /*   read first command    */
               read ( iounit, *, iostat=ierr ) &
     &            turbo_command(1), turbo_control(1)

!              /*   read second command    */
               read ( iounit, *, iostat=ierr ) &
     &            turbo_command(2), turbo_control(2)

!              /*   file close   */
               close( iounit )

            end if

!           /*   error termination   */
            if ( ierr .ne. 0 ) call my_mpi_abort

!-----------------------------------------------------------------------
!        /*   confirm turbomole command                               */
!-----------------------------------------------------------------------

            call system &
     &         ("echo '0' > test.out")
            call system &
     &         ("sleep 0.1")
            call system &
     &         ("which " // turbo_command(1) // &
     &          " > /dev/null 2>&1 && echo '1' > test.out")

            open ( iounit, file = 'test.out' )

            read ( iounit, * ) itest

            close( iounit )

            if ( itest .eq. 0 ) then

               ierr = 1

               write( 6, '(a)' ) 'Error - Turbomole command not found: ' &
     &                            // trim(turbo_command(1))
               write( 6, '(a)' )

            else

               ierr = 0

               write( 6, '(a)' ) 'Turbomole command found: ' // &
     &                            trim(turbo_command(1))

            end if

            call system &
     &         ("echo '0' > test.out")
            call system &
     &         ("sleep 0.1")
            call system &
     &         ("which " // turbo_command(2) // &
     &          " > /dev/null 2>&1 && echo '1' > test.out")

            open ( iounit, file = 'test.out' )

            read ( iounit, * ) itest

            close( iounit )

            if ( itest .eq. 0 ) then

               ierr = 1 + ierr

               write( 6, '(a)' ) 'Error - Turbomole command not found: ' &
     &                            // trim(turbo_command(2))
               write( 6, '(a)' )

            else

               ierr = 0 + ierr

               write( 6, '(a)' ) 'Turbomole command found: ' // &
     &                            trim(turbo_command(2))
               write( 6, '(a)' )

            end if

            call system('rm -f test.out')

         end if

         call my_mpi_bcast_int_0( ierr )

         call error_handling_MPI &
     &       ( ierr, 'subroutine force_turbo_MPI', 26 )

!-----------------------------------------------------------------------
!        /*   confirm turbomole command                               */
!-----------------------------------------------------------------------

!        /*   communication   */
         call my_mpi_bcast_char_1 &
     &      ( turbo_command, len(turbo_command), 2 )

         call my_mpi_bcast_char_1 &
     &      ( turbo_control, len(turbo_control), 2 )

         if       ( (turbo_command(1)(1:4) .eq. 'dscf'  ) .and. &
     &              (turbo_command(2)(1:4) .eq. 'grad'  ) ) then
            ioption_turbo = 1
         else if  ( (turbo_command(1)(1:4) .eq. 'dscf'  ) .and. &
     &              (turbo_command(2)(1:6) .eq. 'mpgrad') ) then
            ioption_turbo = 2
         else if  ( (turbo_command(1)(1:5) .eq. 'ridft' ) .and. &
     &              (turbo_command(2)(1:6) .eq. 'rdgrad') ) then
            ioption_turbo = 3
         else if  ( (turbo_command(1)(1:4) .eq. 'dscf' ) .and. &
     &              (turbo_command(2)(1:5) .eq. 'rimp2') ) then
            ioption_turbo = 4
         else if  ( (turbo_command(1)(1:4) .eq. 'dscf' ) .and. &
     &              (turbo_command(2)(1:5) .eq. 'ricc2') ) then
            ioption_turbo = 5
         else
            call error_handling_MPI &
     &         ( 1, 'subroutine force_turbo_MPI', 26 )
         end if

!        /*   wave function optimization    */
         call read_char_MPI &
     &      ( turbo_guess, 20, '<turbo_guess>', 13, iounit )

!-----------------------------------------------------------------------
!        /*   initialize directory                                    */
!-----------------------------------------------------------------------

         do ibead = 1, nbead

!           /*   allocated bead only   */
            if ( mod( ibead-1, nprocs ) .ne. myrank ) cycle
            call int3_to_char( ibead, char_num )

!           /*   remove old directory and create new directory   */
            call system ('rm -f -r ./' // char_num )
            call system ('mkdir    ./' // char_num )

!           /*   back up coord file   */
            call system ('cp coord coord.turbo')

!           /*   just in case auxbasis does not exist   */
            call system ('touch ./auxbasis')

!           /*   copy all the necessary files   */
            call system('cp -f ./mos*      ' // './' // char_num // '/')
            call system('cp -f ./alpha*    ' // './' // char_num // '/')
            call system('cp -f ./beta*     ' // './' // char_num // '/')
            call system('cp -f ./basis*    ' // './' // char_num // '/')
            call system('cp -f ./auxbasis* ' // './' // char_num // '/')

!           /*   call MPI_barrier   */
            call my_mpi_barrier

         end do

         iset_turbo = 1

      end if

!-----------------------------------------------------------------------
!     /*   get shifted geometry:  xs, ys, zs                          */
!-----------------------------------------------------------------------

      call shifted_geometry_MPI

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!     /*   skip if `ibead is not my job'                              */
!-----------------------------------------------------------------------

      if ( mod( ibead-1, nprocs ) .ne. myrank ) cycle

!-----------------------------------------------------------------------
!     /*   make char_num according to myrank                          */
!-----------------------------------------------------------------------

      call int3_to_char( ibead, char_num )

!-----------------------------------------------------------------------
!     /*   make turbo input:  coord                                   */
!-----------------------------------------------------------------------

!     /*   open the turbo prototype file for geometry   */
      open ( iounit, file = 'coord.turbo' )

!     /*   open the turbo input file   */
      char_file = ('./' // char_num // '/coord')
      open ( iounit_turbo, file = char_file )

      do

!        /*   read a line   */
         read ( iounit, *, iostat=ierr ) char_line

!        /*   exit at the end of the line   */
         if ( ierr .ne. 0 ) exit

!        /*   if matched   */
         if ( char_line(1:6) .eq. '$coord' ) then

!           /*   write a copy of the line   */
            write( iounit_turbo, '(a)' ) char_line

            do i = 1, natom

!              /*   read atomic number   */
               read ( iounit, * ) xi, yi, zi, char_an

!              /*   write atomic number and Cartesian coordinates    */
               write(iounit_turbo,'(3e24.16,5x,a3)') &
     &            xs(i,ibead), ys(i,ibead), zs(i,ibead), char_an

            end do

!        /*   if not matched   */
         else

!           /*   write a copy of the line   */
            write( iounit_turbo, '(a)' ) char_line

         end if

      end do

!     /*   add a blank line at the end   */
      write(iounit_turbo,'(a)')

!     /*   close files   */
      close(iounit)
      close(iounit_turbo)

!-----------------------------------------------------------------------
!     /*   run turbo                                                  */
!-----------------------------------------------------------------------

      if      ( ioption_turbo .eq. 1 ) then
         call system ('cp ' // turbo_control(1) // &
     &                ' ./' // char_num // '/control')
         call system ('cd ./' // char_num // &
     &                ' ; nohup dscf    > dscf.turbo ; cd ..' )
         call system ('cp ' // turbo_control(2) // &
     &                ' ./' // char_num // '/control')
         call system ('cd ./' // char_num // &
     &                ' ; nohup grad    > grad.turbo ; cd ..' )
      else if ( ioption_turbo .eq. 2 ) then
         call system ('cp ' // turbo_control(1) // &
     &                ' ./' // char_num // '/control')
         call system ('cd ./' // char_num // &
     &                ' ; nohup dscf    > dscf.turbo ; cd ..' )
         call system ('cp ' // turbo_control(2) // &
     &                ' ./' // char_num // '/control')
         call system ('cd ./' // char_num // &
     &                ' ; nohup mpgrad  > grad.turbo ; cd ..' )
      else if ( ioption_turbo .eq. 3 ) then
         call system ('cp ' // turbo_control(1) // &
     &                ' ./' // char_num // '/control')
         call system ('cd ./' // char_num // &
     &                ' ; nohup ridft   > dscf.turbo ; cd ..' )
         call system ('cp ' // turbo_control(2) // &
     &                ' ./' // char_num // '/control')
         call system ('cd ./' // char_num // &
     &                ' ; nohup rdgrad  > grad.turbo ; cd ..' )
      else if ( ioption_turbo .eq. 4 ) then
         call system ('cp ' // turbo_control(1) // &
     &                ' ./' // char_num // '/control')
         call system ('cd ./' // char_num // &
     &                ' ; nohup dscf    > dscf.turbo ; cd ..' )
         call system ('cp ' // turbo_control(2) // &
     &                ' ./' // char_num // '/control')
         call system ('cd ./' // char_num // &
     &                ' ; nohup rimp2   > grad.turbo ; cd ..' )
      else if ( ioption_turbo .eq. 5 ) then
         call system ('cp ' // turbo_control(1) // &
     &                ' ./' // char_num // '/control')
         call system ('cd ./' // char_num // &
     &                ' ; nohup dscf    > dscf.turbo ; cd ..' )
         call system ('cp ' // turbo_control(2) // &
     &                ' ./' // char_num // '/control')
         call system ('cd ./' // char_num // &
     &                ' ; nohup ricc2   > ricc2.turbo ; cd ..' )
      end if

!-----------------------------------------------------------------------
!     /*   read turbo output:  potential                              */
!-----------------------------------------------------------------------

!     /*   open the turbo output file   */
      char_file = ('./' // char_num // '/energy')
      open ( iounit_turbo, file = char_file )

      do

!        /*   read a line   */
         read ( iounit_turbo, '(a80)', iostat=ierr ) char_line

!        /*   see if the line matches   */
         match = index( char_line(1:7), '$energy' )

!        /*   if matched   */
         if ( match .ge. 1 ) then

!           /*   read the potential data for hf and dft   */
            if      ( ioption_turbo .eq. 1 ) then
               read (iounit_turbo,*,iostat=ierr) &
     &            idummy, pot_all
!           /*   read the potential data for mp2   */
            else if ( ioption_turbo .eq. 2 ) then
               read (iounit_turbo,*,iostat=ierr) &
     &            idummy, pot_scf, dummy(1), dummy(2), pot_mp2
               pot_all = pot_scf + pot_mp2
!           /*   read the potential data for ridft   */
            else if ( ioption_turbo .eq. 3 ) then
               read (iounit_turbo,*,iostat=ierr) &
     &           idummy, pot_all
!           /*   read the potential data for rimp2   */
            else if ( ioption_turbo .eq. 4 ) then
               read (iounit_turbo,*,iostat=ierr) &
     &           idummy, pot_scf, dummy(1), dummy(2), pot_mp2
               pot_all = pot_scf + pot_mp2
!           /*   read the potential data for rimp2   */
            else if ( ioption_turbo .eq. 5 ) then
               read (iounit_turbo,*,iostat=ierr) &
     &           idummy, pot_scf, dummy(1), dummy(2), pot_cc2
               pot_all = pot_scf + pot_cc2
            end if

!           /*   error termination   */
            if ( ierr .ne. 0 ) call my_mpi_abort

            pot(ibead) = pot_all

!           /*   exit from the do loop   */
            exit

         end if

      end do

!     /*   close file   */
      close(iounit_turbo)

!-----------------------------------------------------------------------
!     /*   read turbo output:  potential gradient                     */
!-----------------------------------------------------------------------

!     /*   open the turbo output file   */
      char_file = ('./' // char_num // '/gradient')
      open ( iounit_turbo, file = char_file )

      do

!        /*   read a line   */
         read ( iounit_turbo, '(a80)', iostat=ierr ) char_line

!        /*   see if the line matches   */
         match = index( char_line(1:5), '$grad' )

!        /*   if matched   */
         if ( match .ge. 1 ) then

!           /*   skip lines   */
            do i = 1, natom+1
               read ( iounit_turbo, '(a80)', iostat=ierr ) char_line
            end do

!           /*   read potential gradient data   */
            do i = 1, natom
               read ( iounit_turbo, *, iostat=ierr ) &
     &            fx(i,ibead), fy(i,ibead), fz(i,ibead)
            end do

!           /*   error termination   */
            if ( ierr .ne. 0 ) call my_mpi_abort

!              /*   change sign:  gradient -> force   */
            do i = 1, natom
               fx(i,ibead) = - fx(i,ibead)
               fy(i,ibead) = - fy(i,ibead)
               fz(i,ibead) = - fz(i,ibead)
             end do

!           /*   exit from the do loop   */
            exit

         end if

      end do

!     /*   close file   */
      close( iounit_turbo )

!-----------------------------------------------------------------------
!     /*   read turbo output:  molecular charge                       */
!-----------------------------------------------------------------------

      if ( ioption_turbo .eq. 1 ) iop = 1
      if ( ioption_turbo .eq. 2 ) iop = 1
      if ( ioption_turbo .eq. 3 ) iop = 1
      if ( ioption_turbo .eq. 4 ) iop = 1
      if ( ioption_turbo .eq. 5 ) iop = 1

!     /*   open the turbo output file   */
      if ( iop.eq. 1 ) char_file = ('./' // char_num // '/dscf.turbo')

      open ( iounit_turbo, file = char_file )

      do

!        /*   read a line   */
         read ( iounit_turbo, *, iostat=ierr ) char_line

!        /*   see if the line matches   */
         match = index( char_line(1:6), 'charge' )

!        /*   if matched   */
         if ( match .ge. 1 ) then

!           /*   go back one line   */
            backspace( iounit_turbo )

!           /*   read molecular charge data   */
            read ( iounit_turbo, *, iostat=ierr ) &
     &      char_dummy(1), char_dummy(2), char_dummy(3), charge_mol

!           /*   error termination   */
            if ( ierr .ne. 0 ) call my_mpi_abort

!           /*   exit from the do loop   */
            exit

         end if

      end do

!     /*   close file   */
      close( iounit_turbo )

!c     /*   make shell script   */
!      char_file = ('./' // char_num // '/dscf.turbo')
!      char_sh =
!     &  " charge=`grep -A2 '  charge' " // char_file //
!     &  " | tail -1 | cut -c35-49`;" //
!     &  " echo $charge > ./" // char_num // "/charge.turbo"
!
!c     /*   run shell   */
!      call system( char_sh )
!
!c     /*   open the turbo output file   */
!      char_file = ('./' // char_num // '/charge.turbo')
!
!c     /*   read charge   */
!      open ( iounit_turbo, file = char_file )
!         read ( iounit_turbo, * ) charge_mol
!      close( iounit_turbo )

!-----------------------------------------------------------------------
!     /*   read turbo output:  dipole moment                          */
!-----------------------------------------------------------------------

!     /*   open the turbo output file   */

      if ( ioption_turbo .eq. 1 ) iop = 1
      if ( ioption_turbo .eq. 2 ) iop = 2
      if ( ioption_turbo .eq. 3 ) iop = 1
      if ( ioption_turbo .eq. 4 ) iop = 2
      if ( ioption_turbo .eq. 5 ) iop = 3

      if ( iop .eq. 1 ) char_file = ('./' // char_num // '/dscf.turbo')
      if ( iop .eq. 2 ) char_file = ('./' // char_num // '/grad.turbo')
      if ( iop .eq. 3 ) char_file = ('./' // char_num // '/ricc2.turbo')

      open ( iounit_turbo, file = char_file )

      do

!        /*   read a line   */
         read ( iounit_turbo, *, iostat=ierr ) char_line

!        /*   see if the line matches   */
         match = index( char_line(1:6), 'dipole' )

!        /*   if matched   */
         if ( match .ge. 1 ) then

!           /*   skip a line   */
            read ( iounit_turbo, '(a80)', iostat=ierr ) char_line

!           /*   read dipole moment data   */

            if ( iop .eq. 1 ) then
               read ( iounit_turbo, *, iostat=ierr ) &
     &         char_dummy(1), char_dummy(2), char_dummy(3), dipx(ibead)
               read ( iounit_turbo, *, iostat=ierr ) &
     &         char_dummy(1), char_dummy(2), char_dummy(3), dipy(ibead)
               read ( iounit_turbo, *, iostat=ierr ) &
     &         char_dummy(1), char_dummy(2), char_dummy(3), dipz(ibead)
            end if
            if ( iop .eq. 2 ) then
               read ( iounit_turbo, *, iostat=ierr ) &
     &         char_dummy(1), char_dummy(2), &
     &         char_dummy(3), char_dummy(4), char_dummy(5), dipx(ibead)
               read ( iounit_turbo, *, iostat=ierr ) &
     &         char_dummy(1), char_dummy(2), &
     &         char_dummy(3), char_dummy(4), char_dummy(5), dipy(ibead)
               read ( iounit_turbo, *, iostat=ierr ) &
     &         char_dummy(1), char_dummy(2), &
     &         char_dummy(3), char_dummy(4), char_dummy(5), dipz(ibead)
            end if
            if ( iop .eq. 3 ) then
               read ( iounit_turbo, '(a80)', iostat=ierr ) char_line
               read ( iounit_turbo, *, iostat=ierr ) &
     &         char_dummy(1), dipx(ibead)
               read ( iounit_turbo, *, iostat=ierr ) &
     &         char_dummy(1), dipy(ibead)
               read ( iounit_turbo, *, iostat=ierr ) &
     &         char_dummy(1), dipz(ibead)
            end if

!           /*   error termination   */
            if ( ierr .ne. 0 ) call my_mpi_abort

!           /*   exit from the do loop   */
            exit

         end if

      end do

!     /*   close file   */
      close( iounit_turbo )

!c     /*   make shell script   */
!
!      if      ( ioption_turbo .eq. 1 ) then
!         char_file = ('./' // char_num // '/dscf.turbo')
!         char_sh =
!     &  " dipx=`grep -A4 'dipole moment' " // char_file //
!     &  " | grep 'x' | cut -c35-49`;" //
!     &  " dipy=`grep -A4 'dipole moment' " // char_file //
!     &  " | grep 'y' | cut -c35-49`;" //
!     &  " dipz=`grep -A4 'dipole moment' " // char_file //
!     &  " | grep 'z' | cut -c35-49`;" //
!     &  " echo $dipx, $dipy, $dipz > ./" // char_num // "/dipole.turbo"
!      else if ( ioption_turbo .eq. 2 ) then
!         char_file = ('./' // char_num // '/grad.turbo')
!         char_sh =
!     &  " dipx=`grep -A4 'dipole moment' " // char_file //
!     &  " | grep 'x' | cut -c65-79`;" //
!     &  " dipy=`grep -A4 'dipole moment' " // char_file //
!     &  " | grep 'y' | cut -c65-79`;" //
!     &  " dipz=`grep -A4 'dipole moment' " // char_file //
!     &  " | grep 'z' | cut -c65-79`;" //
!     &  " echo $dipx, $dipy, $dipz > ./" // char_num // "/dipole.turbo"
!      else if ( ioption_turbo .eq. 3 ) then
!         char_file = ('./' // char_num // '/dscf.turbo')
!         char_sh =
!     &  " dipx=`grep -A4 'dipole moment' " // char_file //
!     &  " | grep 'x' | cut -c35-49`;" //
!     &  " dipy=`grep -A4 'dipole moment' " // char_file //
!     &  " | grep 'y' | cut -c35-49`;" //
!     &  " dipz=`grep -A4 'dipole moment' " // char_file //
!     &  " | grep 'z' | cut -c35-49`;" //
!     &  " echo $dipx, $dipy, $dipz > ./" // char_num // "/dipole.turbo"
!      else if ( ioption_turbo .eq. 4 ) then
!         char_file = ('./' // char_num // '/grad.turbo')
!         char_sh =
!     &  " dipx=`grep -A4 'dipole moment' " // char_file //
!     &  " | grep 'x' | cut -c65-79`;" //
!     &  " dipy=`grep -A4 'dipole moment' " // char_file //
!     &  " | grep 'y' | cut -c65-79`;" //
!     &  " dipz=`grep -A4 'dipole moment' " // char_file //
!     &  " | grep 'z' | cut -c65-79`;" //
!     &  " echo $dipx, $dipy, $dipz > ./" // char_num // "/dipole.turbo"
!      end if
!
!c     /*   run shell   */
!      call system( char_sh )
!
!c     /*   open the turbo output file   */
!      char_file = ('./' // char_num // '/dipole.turbo')
!
!c     /*   read dipole moment   */
!      open ( iounit_turbo, file = char_file )
!         read ( iounit_turbo, * ) dipx(ibead), dipy(ibead), dipz(ibead)
!      close( iounit_turbo )

!-----------------------------------------------------------------------
!     /*   delete unnecessary files                                   */
!-----------------------------------------------------------------------

      char_file = ('./' // char_num // '/energy')
      call system( 'rm -f ' // char_file )
      char_file = ('./' // char_num // '/gradient')
      call system( 'rm -f ' // char_file )
      char_file = ('./' // char_num // '/statistics')
      call system( 'rm -f ' // char_file )

!-----------------------------------------------------------------------
!     /*   extrapolate mos                                            */
!-----------------------------------------------------------------------

      if ( turbo_guess(1:9) .ne. 'PREVIOUS ' ) then

         char_file = ('./' // char_num // '/test.out')

         call system ( 'test -e mos ; echo $? > ' // char_file )
         open ( iounit, file=char_file )
            read ( iounit, * ) i
         close( iounit )
         if ( i .eq. 0 ) call turbo_mos_MPI( ibead )

         call system ( 'test -e alpha ; echo $? > ' // char_file )
         open ( iounit, file=char_file )
            read ( iounit, * ) i
         close( iounit )
         if ( i .eq. 0 ) call turbo_alpha_MPI( ibead )

         call system ( 'test -e beta ; echo $? > ' // char_file )
         open ( iounit, file=char_file )
            read ( iounit, * ) i
         close( iounit )
         if ( i .eq. 0 ) call turbo_beta_MPI( ibead )

         call system ( ' rm -f ' // char_file )

      end if

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz, natom, nbead )

!     /*   dipole   */
      call my_mpi_allreduce_real_1 ( dipx, nbead )
      call my_mpi_allreduce_real_1 ( dipy, nbead )
      call my_mpi_allreduce_real_1 ( dipz, nbead )

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

      do j = 1, nbead

         if ( mod( j-1, nprocs ) .ne. myrank ) cycle

         do i = 1, natom

            xi = x(i,j)
            yi = y(i,j)
            zi = z(i,j)

            call pbc_unfold_MPI &
     &        ( xi, yi, zi, mbox(1,i,j), mbox(2,i,j), mbox(3,i,j) )

            vir(1,1) = vir(1,1) + fx(i,j)*xi
            vir(1,2) = vir(1,2) + fx(i,j)*yi
            vir(1,3) = vir(1,3) + fx(i,j)*zi
            vir(2,1) = vir(2,1) + fy(i,j)*xi
            vir(2,2) = vir(2,2) + fy(i,j)*yi
            vir(2,3) = vir(2,3) + fy(i,j)*zi
            vir(3,1) = vir(3,1) + fz(i,j)*xi
            vir(3,2) = vir(3,2) + fz(i,j)*yi
            vir(3,3) = vir(3,3) + fz(i,j)*zi

         end do

      end do

      call my_mpi_allreduce_real_2 ( vir, 3, 3 )

      return
      end





!***********************************************************************
      subroutine turbo_mos_MPI ( ibead )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, iounit_turbo, myrank, nprocs, nbead, turbo_control

      use turbo_variables, only : &
     &   weight_scf, coeff_scf, iset_scf, nsaos_scf

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, ibead, jbead, ierr

      character(len=80):: char_line, char_file
      character(len=3) :: char_num

!-----------------------------------------------------------------------
!     /*   initial settings                                           */
!-----------------------------------------------------------------------

      if ( iset_scf .eq. 0 ) then

!-----------------------------------------------------------------------
!        /*   copy file mos                                           */
!-----------------------------------------------------------------------

!        /*   copy mos to mos.i   */

         do jbead = 1, nbead

            if ( mod( jbead-1, nprocs ) .ne. myrank ) cycle

            call int3_to_char( jbead, char_num )

            char_file = ('./' // char_num // '/mos.6')
            call system ('cp -f ./' // char_num // '/mos ' // char_file)
            char_file = ('./' // char_num // '/mos.5')
            call system ('cp -f ./' // char_num // '/mos ' // char_file)
            char_file = ('./' // char_num // '/mos.4')
            call system ('cp -f ./' // char_num // '/mos ' // char_file)
            char_file = ('./' // char_num // '/mos.3')
            call system ('cp -f ./' // char_num // '/mos ' // char_file)
            char_file = ('./' // char_num // '/mos.2')
            call system ('cp -f ./' // char_num // '/mos ' // char_file)
            char_file = ('./' // char_num // '/mos.1')
            call system ('cp -f ./' // char_num // '/mos ' // char_file)

         end do

!-----------------------------------------------------------------------
!        /*   read nsaos_scf = number of mos                          */
!-----------------------------------------------------------------------

         open ( iounit_turbo, file = turbo_control(1) )

            do
               read ( iounit_turbo, *, iostat=ierr ) char_line
               if ( char_line(1:8) .eq. 'nbf(AO)=' ) exit
            end do

         close( iounit_turbo )

         open ( iounit_turbo, file = 'scratch.turbo' )
            write( iounit_turbo, '(a)' ) char_line(9:16)
         close( iounit_turbo )

         open ( iounit_turbo, file = 'scratch.turbo' )
            read ( iounit_turbo, * ) nsaos_scf
         close( iounit_turbo )

!        /*   call MPI_barrier   */
         call my_mpi_barrier

!        /*   remove unnecessary file   */
         call system ( 'rm -f scratch.turbo' )

!        /*   memory allocation   */
         if ( .not. allocated( coeff_scf ) ) &
     &      allocate ( coeff_scf(0:6,nsaos_scf,nsaos_scf) )

!        /*   end of initial settings   */
         iset_scf = 1

      else

!-----------------------------------------------------------------------
!        /*   copy mos.i to mos.i+1                                   */
!-----------------------------------------------------------------------

         call int3_to_char( ibead, char_num )

         char_file = ('./' // char_num // '/mos.6')
         call system ('cp -f ./' // char_num // '/mos.5 ' // char_file)
         char_file = ('./' // char_num // '/mos.5')
         call system ('cp -f ./' // char_num // '/mos.4 ' // char_file)
         char_file = ('./' // char_num // '/mos.4')
         call system ('cp -f ./' // char_num // '/mos.3 ' // char_file)
         char_file = ('./' // char_num // '/mos.3')
         call system ('cp -f ./' // char_num // '/mos.2 ' // char_file)
         char_file = ('./' // char_num // '/mos.2')
         call system ('cp -f ./' // char_num // '/mos.1 ' // char_file)
         char_file = ('./' // char_num // '/mos.1')
         call system ('cp -f ./' // char_num // '/mos   ' // char_file)

      end if

!-----------------------------------------------------------------------
!     /*   read mo coefficients                                       */
!-----------------------------------------------------------------------

      call turbo_readmos_MPI ( 1, ibead )
      call turbo_readmos_MPI ( 2, ibead )
      call turbo_readmos_MPI ( 3, ibead )
      call turbo_readmos_MPI ( 4, ibead )
      call turbo_readmos_MPI ( 5, ibead )
      call turbo_readmos_MPI ( 6, ibead )

!-----------------------------------------------------------------------
!     /*   weighted average for mo coefficients                       */
!-----------------------------------------------------------------------

!     /*   weight parameters   */
      call turbo_weight

!     /*   extrapolated coefficients   */

      do j = 1, nsaos_scf
      do i = 1, nsaos_scf
         coeff_scf(0,i,j) = coeff_scf(1,i,j)*weight_scf(1) &
     &                    + coeff_scf(2,i,j)*weight_scf(2) &
     &                    + coeff_scf(3,i,j)*weight_scf(3) &
     &                    + coeff_scf(4,i,j)*weight_scf(4) &
     &                    + coeff_scf(5,i,j)*weight_scf(5) &
     &                    + coeff_scf(6,i,j)*weight_scf(6)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   write mo coefficients                                      */
!-----------------------------------------------------------------------

      call turbo_writemos_MPI ( ibead )

      return
      end





!***********************************************************************
      subroutine turbo_weight
!***********************************************************************

      use turbo_variables, only : turbo_guess, weight_scf

!     /*   guess option: pulay and forgasi   */
      if ( turbo_guess(1:6) .eq. 'PULAY ' ) then

         weight_scf(1) = +  2.4d0
         weight_scf(2) = -  1.2d0
         weight_scf(3) = -  0.8d0
         weight_scf(4) = +  0.6d0
         weight_scf(5) = +  0.0d0
         weight_scf(6) = +  0.0d0

!     /*   kolafa   */
      else if ( turbo_guess(1:7) .eq. 'KOLAFA ' ) then

         weight_scf(1) = + 22.d0/ 7.d0
         weight_scf(2) = - 55.d0/14.d0
         weight_scf(3) = + 55.d0/21.d0
         weight_scf(4) = - 22.d0/21.d0
         weight_scf(5) = +  5.d0/21.d0
         weight_scf(6) = -  1.d0/42.d0

!     /*   previous step   */
      else if ( turbo_guess(1:9) .eq. 'PREVIOUS ' ) then

         weight_scf(1) = +  1.d0
         weight_scf(2) = +  0.d0
         weight_scf(3) = +  0.d0
         weight_scf(4) = +  0.d0
         weight_scf(5) = +  0.d0
         weight_scf(6) = +  0.d0

!     /*   previous step   */
      else

         weight_scf(1) = +  1.d0
         weight_scf(2) = +  0.d0
         weight_scf(3) = +  0.d0
         weight_scf(4) = +  0.d0
         weight_scf(5) = +  0.d0
         weight_scf(6) = +  0.d0

!     /*   guess option   */
      end if

      return
      end





!***********************************************************************
      subroutine turbo_readmos_MPI ( ioption, ibead )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iounit, iounit_turbo

      use turbo_variables, only : coeff_scf, nsaos_scf

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption, j, k, ibead

      character(len=80):: char_line, char_file
      character(len=3) :: char_num

!-----------------------------------------------------------------------
!     /*   read molecular orbitals                                    */
!-----------------------------------------------------------------------

      call int3_to_char( ibead, char_num )

      if ( ioption .eq. 0 )  char_file = ('./' // char_num // '/mos')
      if ( ioption .eq. 1 )  char_file = ('./' // char_num // '/mos.1')
      if ( ioption .eq. 2 )  char_file = ('./' // char_num // '/mos.2')
      if ( ioption .eq. 3 )  char_file = ('./' // char_num // '/mos.3')
      if ( ioption .eq. 4 )  char_file = ('./' // char_num // '/mos.4')
      if ( ioption .eq. 5 )  char_file = ('./' // char_num // '/mos.5')
      if ( ioption .eq. 6 )  char_file = ('./' // char_num // '/mos.6')

      open ( iounit_turbo, file = char_file )

      k = ioption

!     /*   head part   */

      do
         read ( iounit_turbo, * ) char_line
         if ( char_line(1:1) .eq. '$') cycle
         if ( char_line(1:1) .eq. '#') cycle
         backspace ( iounit_turbo )
         exit
      end do

!     /*   body part   */

      do j = 1, nsaos_scf

         read ( iounit_turbo, * ) char_line

         do i = 1, 4*int(nsaos_scf/4), 4
            read (iounit_turbo, '(4d20.14)' ) &
     &      coeff_scf(k,i,j),   coeff_scf(k,i+1,j), &
     &      coeff_scf(k,i+2,j), coeff_scf(k,i+3,j)
         end do

         if      ( mod(nsaos_scf,4) .eq. 0 ) then
            continue
         else if ( mod(nsaos_scf,4) .eq. 1 ) then
            i = 4*int(nsaos_scf/4) + 1
            read (iounit_turbo, '(d20.14)' ) &
     &      coeff_scf(k,i,j)
         else if ( mod(nsaos_scf,4) .eq. 2 ) then
            i = 4*int(nsaos_scf/4) + 1
            read (iounit_turbo, '(2d20.14)' ) &
     &      coeff_scf(k,i,j),   coeff_scf(k,i+1,j)
         else if ( mod(nsaos_scf,4) .eq. 3 ) then
            i = 4*int(nsaos_scf/4) + 1
            read (iounit_turbo, '(3d20.14)' ) &
     &      coeff_scf(k,i,j),   coeff_scf(k,i+1,j), &
     &      coeff_scf(k,i+2,j)
         end if

      end do

      close( iounit_turbo )

      return
      end





!***********************************************************************
      subroutine turbo_writemos_MPI ( ibead )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iounit, iounit_turbo

      use turbo_variables, only : coeff_scf, nsaos_scf

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, ierr, ibead
      real(8) :: dummy(4)

      character(len=80):: char_line, char_file
      character(len=3) :: char_num

!-----------------------------------------------------------------------
!     /*   read molecular orbitals                                    */
!-----------------------------------------------------------------------

      call int3_to_char( ibead, char_num )

      char_file = ('./' // char_num // '/mos')
      open ( iounit,       file = char_file )
      char_file = ('./' // char_num // '/mos.1')
      open ( iounit_turbo, file = char_file )

!     /*   head part   */

      do
         read ( iounit_turbo, '(a80)' ) char_line
         if (char_line(1:1) .eq. '$') then
            write( iounit, '(a80)' ) char_line
            cycle
         end if
         if (char_line(1:1) .eq. '#') then
            write( iounit, '(a80)' ) char_line
            cycle
         end if
         backspace ( iounit_turbo )
         exit
      end do

!     /*   body part   */

      do j = 1, nsaos_scf

         read ( iounit_turbo, '(a80)' ) char_line
         write( iounit, '(a80)' ) char_line

         do i = 1, 4*int(nsaos_scf/4), 4
            read (iounit_turbo, '(4d20.14)' ) &
     &      dummy(1),        dummy(2), &
     &      dummy(3),        dummy(4)
            write(iounit, '(4d20.14)' ) &
     &      coeff_scf(0,i,j),   coeff_scf(0,i+1,j), &
     &      coeff_scf(0,i+2,j), coeff_scf(0,i+3,j)
         end do

         if      ( mod(nsaos_scf,4) .eq. 0 ) then
            continue
         else if ( mod(nsaos_scf,4) .eq. 1 ) then
            i = 4*int(nsaos_scf/4) + 1
            read (iounit_turbo, '(d20.14)' ) &
     &      dummy(1)
            write(iounit, '(d20.14)' ) &
     &      coeff_scf(0,i,j)
         else if ( mod(nsaos_scf,4) .eq. 2 ) then
            i = 4*int(nsaos_scf/4) + 1
            read (iounit_turbo, '(2d20.14)' ) &
     &      dummy(1), dummy(2)
            write(iounit, '(2d20.14)' ) &
     &      coeff_scf(0,i,j),   coeff_scf(0,i+1,j)
         else if ( mod(nsaos_scf,4) .eq. 3 ) then
            i = 4*int(nsaos_scf/4) + 1
            read (iounit_turbo, '(3d20.14)' ) &
     &      dummy(1), dummy(2), dummy(3)
            write(iounit, '(3d20.14)' ) &
     &      coeff_scf(0,i,j),   coeff_scf(0,i+1,j), &
     &      coeff_scf(0,i+2,j)
         end if

      end do

      do
         read ( iounit_turbo, '(a80)', iostat=ierr ) char_line
         if ( ierr .ne. 0 ) exit
         write( iounit, '(a80)' ) char_line
      end do

      close( iounit_turbo )
      close( iounit )

      return
      end





!***********************************************************************
      subroutine shifted_geometry_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, xs, ys, zs, fictmass, mbox, natom, nbead, method

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ibead
      real(8) :: xi, yi, zi, fi

!-----------------------------------------------------------------------

      if ( ( method(1:7) .eq. 'STRING ' ) .or. &
     &     ( method(1:7) .eq. 'GEOOPT ' ) .or. &
     &     ( method(1:7) .eq. 'OMOPT  ' ) .or. &
     &     ( method(1:7) .eq. 'OM     ' ) ) then
         xs(:,:) = x(:,:)
         ys(:,:) = y(:,:)
         zs(:,:) = z(:,:)
         return
      end if

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!     /*   shift for periodic boundary    */

         do i = 1, natom

            xi = x(i,ibead)
            yi = y(i,ibead)
            zi = z(i,ibead)

            call pbc_unfold_MPI &
     &         ( xi, yi, zi, mbox(1,i,ibead), mbox(2,i,ibead), &
     &           mbox(3,i,ibead) )

            xs(i,ibead) = xi
            ys(i,ibead) = yi
            zs(i,ibead) = zi

         end do

!        /*   shift center of mass to the origin    */

         xi = 0.d0
         yi = 0.d0
         zi = 0.d0
         fi = 0.d0

         do i = 1, natom
            xi = xi + fictmass(i,1)*xs(i,ibead)
            yi = yi + fictmass(i,1)*ys(i,ibead)
            zi = zi + fictmass(i,1)*zs(i,ibead)
            fi = fi + fictmass(i,1)
         end do

         xi = xi / fi
         yi = yi / fi
         zi = zi / fi

         do i = 1, natom
            xs(i,ibead) = xs(i,ibead) - xi
            ys(i,ibead) = ys(i,ibead) - yi
            zs(i,ibead) = zs(i,ibead) - zi
         end do

      end do

      return
      end





!***********************************************************************
      subroutine turbo_alpha_MPI ( ibead )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, iounit_turbo, myrank, nprocs, nbead, turbo_control

      use turbo_variables, only : &
     &   weight_scf, coeff_scf, nsaos_scf, iset_scf

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, ibead, jbead, ierr

      character(len=80):: char_line, char_file
      character(len=3) :: char_num

!-----------------------------------------------------------------------
!     /*   initial settings                                           */
!-----------------------------------------------------------------------

      if ( iset_scf .eq. 0 ) then

!-----------------------------------------------------------------------
!        /*   copy file alpha                                         */
!-----------------------------------------------------------------------

!        /*   copy alpha to alpha.i   */

         do jbead = 1, nbead

            if ( mod( jbead-1, nprocs ) .ne. myrank ) cycle

            call int3_to_char( jbead, char_num )

            char_file = ('./' // char_num // '/alpha.6')
            call system &
     &         ('cp -f ./' // char_num // '/alpha ' // char_file)
            char_file = ('./' // char_num // '/alpha.5')
            call system &
     &         ('cp -f ./' // char_num // '/alpha ' // char_file)
            char_file = ('./' // char_num // '/alpha.4')
            call system &
     &         ('cp -f ./' // char_num // '/alpha ' // char_file)
            char_file = ('./' // char_num // '/alpha.3')
            call system &
     &         ('cp -f ./' // char_num // '/alpha ' // char_file)
            char_file = ('./' // char_num // '/alpha.2')
            call system &
     &         ('cp -f ./' // char_num // '/alpha ' // char_file)
            char_file = ('./' // char_num // '/alpha.1')
            call system &
     &         ('cp -f ./' // char_num // '/alpha ' // char_file)

         end do

!-----------------------------------------------------------------------
!        /*   read nsaos_scf = number of alpha                        */
!-----------------------------------------------------------------------

         open ( iounit_turbo, file = turbo_control(1) )

            do
               read ( iounit_turbo, *, iostat=ierr ) char_line
               if ( char_line(1:8) .eq. 'nbf(AO)=' ) exit
            end do

         close( iounit_turbo )

         open ( iounit_turbo, file = 'scratch.turbo' )
            write( iounit_turbo, '(a)' ) char_line(9:16)
         close( iounit_turbo )

         open ( iounit_turbo, file = 'scratch.turbo' )
            read ( iounit_turbo, * ) nsaos_scf
         close( iounit_turbo )

!        /*   call MPI_barrier   */
         call my_mpi_barrier

!        /*   remove unnecessary file   */
         call system ( 'rm -f scratch.turbo' )

!        /*   memory allocation   */
         if ( .not. allocated( coeff_scf ) ) &
     &      allocate ( coeff_scf(0:6,nsaos_scf,nsaos_scf) )

!        /*   end of initial settings   */
         iset_scf = 1

      else

!-----------------------------------------------------------------------
!        /*   copy alpha.i to alpha.i+1                               */
!-----------------------------------------------------------------------

         call int3_to_char( ibead, char_num )

         char_file = ('./' // char_num // '/alpha.6')
         call system &
     &      ('cp -f ./' // char_num // '/alpha.5 ' // char_file)
         char_file = ('./' // char_num // '/alpha.5')
         call system &
     &      ('cp -f ./' // char_num // '/alpha.4 ' // char_file)
         char_file = ('./' // char_num // '/alpha.4')
         call system &
     &      ('cp -f ./' // char_num // '/alpha.3 ' // char_file)
         char_file = ('./' // char_num // '/alpha.3')
         call system &
     &      ('cp -f ./' // char_num // '/alpha.2 ' // char_file)
         char_file = ('./' // char_num // '/alpha.2')
         call system &
     &      ('cp -f ./' // char_num // '/alpha.1 ' // char_file)
         char_file = ('./' // char_num // '/alpha.1')
         call system &
     &      ('cp -f ./' // char_num // '/alpha   ' // char_file)

      end if

!-----------------------------------------------------------------------
!     /*   read mo coefficients                                       */
!-----------------------------------------------------------------------

      call turbo_readalpha_MPI ( 1, ibead )
      call turbo_readalpha_MPI ( 2, ibead )
      call turbo_readalpha_MPI ( 3, ibead )
      call turbo_readalpha_MPI ( 4, ibead )
      call turbo_readalpha_MPI ( 5, ibead )
      call turbo_readalpha_MPI ( 6, ibead )

!-----------------------------------------------------------------------
!     /*   weighted average for mo coefficients                       */
!-----------------------------------------------------------------------

!     /*   weight parameters   */
      call turbo_weight

      do j = 1, nsaos_scf
      do i = 1, nsaos_scf
         coeff_scf(0,i,j) = coeff_scf(1,i,j)*weight_scf(1) &
     &                    + coeff_scf(2,i,j)*weight_scf(2) &
     &                    + coeff_scf(3,i,j)*weight_scf(3) &
     &                    + coeff_scf(4,i,j)*weight_scf(4) &
     &                    + coeff_scf(5,i,j)*weight_scf(5) &
     &                    + coeff_scf(6,i,j)*weight_scf(6)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   write mo coefficients                                      */
!-----------------------------------------------------------------------

      call turbo_writealpha_MPI ( ibead )

      return
      end





!***********************************************************************
      subroutine turbo_readalpha_MPI ( ioption, ibead )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iounit, iounit_turbo

      use turbo_variables, only : coeff_scf, nsaos_scf

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption, j, k, ibead

      character(len=80):: char_line, char_file
      character(len=3) :: char_num

!-----------------------------------------------------------------------
!     /*   read molecular orbitals                                    */
!-----------------------------------------------------------------------

      call int3_to_char( ibead, char_num )

      if ( ioption .eq. 0 ) &
     &   char_file = ('./' // char_num // '/alpha')
      if ( ioption .eq. 1 ) &
     &   char_file = ('./' // char_num // '/alpha.1')
      if ( ioption .eq. 2 ) &
     &   char_file = ('./' // char_num // '/alpha.2')
      if ( ioption .eq. 3 ) &
     &   char_file = ('./' // char_num // '/alpha.3')
      if ( ioption .eq. 4 ) &
     &  char_file = ('./' // char_num // '/alpha.4')
      if ( ioption .eq. 5 ) &
     &  char_file = ('./' // char_num // '/alpha.5')
      if ( ioption .eq. 6 ) &
     &  char_file = ('./' // char_num // '/alpha.6')

      open ( iounit_turbo, file = char_file )

      k = ioption

!     /*   head part   */

      do
         read ( iounit_turbo, * ) char_line
         if ( char_line(1:1) .eq. '$') cycle
         if ( char_line(1:1) .eq. '#') cycle
         backspace ( iounit_turbo )
         exit
      end do

!     /*   body part   */

      do j = 1, nsaos_scf

         read ( iounit_turbo, * ) char_line

         do i = 1, 4*int(nsaos_scf/4), 4
            read (iounit_turbo, '(4d20.14)' ) &
     &      coeff_scf(k,i,j),   coeff_scf(k,i+1,j), &
     &      coeff_scf(k,i+2,j), coeff_scf(k,i+3,j)
         end do

         if      ( mod(nsaos_scf,4) .eq. 0 ) then
            continue
         else if ( mod(nsaos_scf,4) .eq. 1 ) then
            i = 4*int(nsaos_scf/4) + 1
            read (iounit_turbo, '(d20.14)' ) &
     &      coeff_scf(k,i,j)
         else if ( mod(nsaos_scf,4) .eq. 2 ) then
            i = 4*int(nsaos_scf/4) + 1
            read (iounit_turbo, '(2d20.14)' ) &
     &      coeff_scf(k,i,j),   coeff_scf(k,i+1,j)
         else if ( mod(nsaos_scf,4) .eq. 3 ) then
            i = 4*int(nsaos_scf/4) + 1
            read (iounit_turbo, '(3d20.14)' ) &
     &      coeff_scf(k,i,j),   coeff_scf(k,i+1,j), &
     &      coeff_scf(k,i+2,j)
         end if

      end do

      close( iounit_turbo )

      return
      end





!***********************************************************************
      subroutine turbo_writealpha_MPI ( ibead )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iounit, iounit_turbo

      use turbo_variables, only : coeff_scf, nsaos_scf

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, ierr, ibead
      real(8) :: dummy(4)

      character(len=80):: char_line, char_file
      character(len=3) :: char_num

!-----------------------------------------------------------------------
!     /*   read molecular orbitals                                    */
!-----------------------------------------------------------------------

      call int3_to_char( ibead, char_num )

      char_file = ('./' // char_num // '/alpha')
      open ( iounit,       file = char_file )
      char_file = ('./' // char_num // '/alpha.1')
      open ( iounit_turbo, file = char_file )

!     /*   head part   */

      do
         read ( iounit_turbo, '(a80)' ) char_line
         if (char_line(1:1) .eq. '$') then
            write( iounit, '(a80)' ) char_line
            cycle
         end if
         if (char_line(1:1) .eq. '#') then
            write( iounit, '(a80)' ) char_line
            cycle
         end if
         backspace ( iounit_turbo )
         exit
      end do

!     /*   body part   */

      do j = 1, nsaos_scf

         read ( iounit_turbo, '(a80)' ) char_line
         write( iounit, '(a80)' ) char_line

         do i = 1, 4*int(nsaos_scf/4), 4
            read (iounit_turbo, '(4d20.14)' ) &
     &      dummy(1),        dummy(2), &
     &      dummy(3),        dummy(4)
            write(iounit, '(4d20.14)' ) &
     &      coeff_scf(0,i,j),   coeff_scf(0,i+1,j), &
     &      coeff_scf(0,i+2,j), coeff_scf(0,i+3,j)
         end do

         if      ( mod(nsaos_scf,4) .eq. 0 ) then
            continue
         else if ( mod(nsaos_scf,4) .eq. 1 ) then
            i = 4*int(nsaos_scf/4) + 1
            read (iounit_turbo, '(d20.14)' ) &
     &      dummy(1)
            write(iounit, '(d20.14)' ) &
     &      coeff_scf(0,i,j)
         else if ( mod(nsaos_scf,4) .eq. 2 ) then
            i = 4*int(nsaos_scf/4) + 1
            read (iounit_turbo, '(2d20.14)' ) &
     &      dummy(1), dummy(2)
            write(iounit, '(2d20.14)' ) &
     &      coeff_scf(0,i,j),   coeff_scf(0,i+1,j)
         else if ( mod(nsaos_scf,4) .eq. 3 ) then
            i = 4*int(nsaos_scf/4) + 1
            read (iounit_turbo, '(3d20.14)' ) &
     &      dummy(1), dummy(2), dummy(3)
            write(iounit, '(3d20.14)' ) &
     &      coeff_scf(0,i,j),   coeff_scf(0,i+1,j), &
     &      coeff_scf(0,i+2,j)
         end if

      end do

      do
         read ( iounit_turbo, '(a80)', iostat=ierr ) char_line
         if ( ierr .ne. 0 ) exit
         write( iounit, '(a80)' ) char_line
      end do

      close( iounit_turbo )
      close( iounit )

      return
      end






!***********************************************************************
      subroutine turbo_beta_MPI ( ibead )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, iounit_turbo, myrank, nprocs, nbead

      use turbo_variables, only : &
     &   weight_scf, coeff_scf, nsaos_scf, iset_scf

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, ibead, jbead

      character(len=80):: char_file
      character(len=3) :: char_num

!-----------------------------------------------------------------------
!     /*   initial settings                                           */
!-----------------------------------------------------------------------

      if ( iset_scf .eq. 0 ) then

!-----------------------------------------------------------------------
!        /*   copy file beta                                          */
!-----------------------------------------------------------------------

!        /*   copy beta to beta.i   */

         do jbead = 1, nbead

            if ( mod( jbead-1, nprocs ) .ne. myrank ) cycle

            call int3_to_char( jbead, char_num )

            char_file = ('./' // char_num // '/beta.6')
            call system &
     &         ('cp -f ./' // char_num // '/beta ' // char_file)
            char_file = ('./' // char_num // '/beta.5')
            call system &
     &         ('cp -f ./' // char_num // '/beta ' // char_file)
            char_file = ('./' // char_num // '/beta.4')
            call system &
     &         ('cp -f ./' // char_num // '/beta ' // char_file)
            char_file = ('./' // char_num // '/beta.3')
            call system &
     &         ('cp -f ./' // char_num // '/beta ' // char_file)
            char_file = ('./' // char_num // '/beta.2')
            call system &
     &         ('cp -f ./' // char_num // '/beta ' // char_file)
            char_file = ('./' // char_num // '/beta.1')
            call system &
     &         ('cp -f ./' // char_num // '/beta ' // char_file)

         end do

!-----------------------------------------------------------------------
!        /*   read nsaos_scf = number of beta                         */
!-----------------------------------------------------------------------

!         open ( iounit_turbo, file = turbo_control(1) )
!
!            do
!               read ( iounit_turbo, *, iostat=ierr ) char_line
!               if ( char_line(1:8) .eq. 'nbf(AO)=' ) exit
!            end do
!
!         close( iounit_turbo )
!
!         open ( iounit_turbo, file = 'scratch.turbo' )
!            write( iounit_turbo, '(a)' ) char_line(9:16)
!         close( iounit_turbo )
!
!         open ( iounit_turbo, file = 'scratch.turbo' )
!            read ( iounit_turbo, * ) nsaos_scf
!         close( iounit_turbo )

!        /*   call MPI_barrier   */
         call my_mpi_barrier

!c        /*   remove unnecessary file   */
!         call system ( 'rm -f scratch.turbo' )
!
!c        /*   memory allocation   */
!         if ( .not. allocated( coeff_scf ) ) &
!    &       allocate ( coeff_scf(0:6,nsaos_scf,nsaos_scf) )
!
!c        /*   end of initial settings   */
         iset_scf = 1

      else

!-----------------------------------------------------------------------
!        /*   copy beta.i to beta.i+1                                 */
!-----------------------------------------------------------------------

         call int3_to_char( ibead, char_num )

         char_file = ('./' // char_num // '/beta.6')
         call system &
     &      ('cp -f ./' // char_num // '/beta.5 ' // char_file)
         char_file = ('./' // char_num // '/beta.5')
         call system &
     &      ('cp -f ./' // char_num // '/beta.4 ' // char_file)
         char_file = ('./' // char_num // '/beta.4')
         call system &
     &      ('cp -f ./' // char_num // '/beta.3 ' // char_file)
         char_file = ('./' // char_num // '/beta.3')
         call system &
     &      ('cp -f ./' // char_num // '/beta.2 ' // char_file)
         char_file = ('./' // char_num // '/beta.2')
         call system &
     &      ('cp -f ./' // char_num // '/beta.1 ' // char_file)
         char_file = ('./' // char_num // '/beta.1')
         call system &
     &      ('cp -f ./' // char_num // '/beta   ' // char_file)

      end if

!-----------------------------------------------------------------------
!     /*   read mo coefficients                                       */
!-----------------------------------------------------------------------

      call turbo_readbeta_MPI ( 1, ibead )
      call turbo_readbeta_MPI ( 2, ibead )
      call turbo_readbeta_MPI ( 3, ibead )
      call turbo_readbeta_MPI ( 4, ibead )
      call turbo_readbeta_MPI ( 5, ibead )
      call turbo_readbeta_MPI ( 6, ibead )

!-----------------------------------------------------------------------
!     /*   weighted average for mo coefficients                       */
!-----------------------------------------------------------------------

!     /*   weight parameters   */
      call turbo_weight

      do j = 1, nsaos_scf
      do i = 1, nsaos_scf
         coeff_scf(0,i,j) = coeff_scf(1,i,j)*weight_scf(1) &
     &                    + coeff_scf(2,i,j)*weight_scf(2) &
     &                    + coeff_scf(3,i,j)*weight_scf(3) &
     &                    + coeff_scf(4,i,j)*weight_scf(4) &
     &                    + coeff_scf(5,i,j)*weight_scf(5) &
     &                    + coeff_scf(6,i,j)*weight_scf(6)
      end do
      end do

!-----------------------------------------------------------------------
!     /*   write mo coefficients                                      */
!-----------------------------------------------------------------------

      call turbo_writebeta_MPI ( ibead )

      return
      end





!***********************************************************************
      subroutine turbo_readbeta_MPI ( ioption, ibead )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only: iounit, iounit_turbo

      use turbo_variables, only : coeff_scf, nsaos_scf

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ioption, j, k, ibead

      character(len=80):: char_line, char_file
      character(len=3) :: char_num

!-----------------------------------------------------------------------
!     /*   read molecular orbitals                                    */
!-----------------------------------------------------------------------

      call int3_to_char( ibead, char_num )

      if ( ioption .eq. 0 ) &
     &   char_file = ('./' // char_num // '/beta')
      if ( ioption .eq. 1 ) &
     &   char_file = ('./' // char_num // '/beta.1')
      if ( ioption .eq. 2 ) &
     &   char_file = ('./' // char_num // '/beta.2')
      if ( ioption .eq. 3 ) &
     &   char_file = ('./' // char_num // '/beta.3')
      if ( ioption .eq. 4 ) &
     &  char_file = ('./' // char_num // '/beta.4')
      if ( ioption .eq. 5 ) &
     &  char_file = ('./' // char_num // '/beta.5')
      if ( ioption .eq. 6 ) &
     &  char_file = ('./' // char_num // '/beta.6')

      open ( iounit_turbo, file = char_file )

      k = ioption

!     /*   head part   */

      do
         read ( iounit_turbo, * ) char_line
         if ( char_line(1:1) .eq. '$') cycle
         if ( char_line(1:1) .eq. '#') cycle
         backspace ( iounit_turbo )
         exit
      end do

!     /*   body part   */

      do j = 1, nsaos_scf

         read ( iounit_turbo, * ) char_line

         do i = 1, 4*int(nsaos_scf/4), 4
            read (iounit_turbo, '(4d20.14)' ) &
     &      coeff_scf(k,i,j),   coeff_scf(k,i+1,j), &
     &      coeff_scf(k,i+2,j), coeff_scf(k,i+3,j)
         end do

         if      ( mod(nsaos_scf,4) .eq. 0 ) then
            continue
         else if ( mod(nsaos_scf,4) .eq. 1 ) then
            i = 4*int(nsaos_scf/4) + 1
            read (iounit_turbo, '(d20.14)' ) &
     &      coeff_scf(k,i,j)
         else if ( mod(nsaos_scf,4) .eq. 2 ) then
            i = 4*int(nsaos_scf/4) + 1
            read (iounit_turbo, '(2d20.14)' ) &
     &      coeff_scf(k,i,j),   coeff_scf(k,i+1,j)
         else if ( mod(nsaos_scf,4) .eq. 3 ) then
            i = 4*int(nsaos_scf/4) + 1
            read (iounit_turbo, '(3d20.14)' ) &
     &      coeff_scf(k,i,j),   coeff_scf(k,i+1,j), &
     &      coeff_scf(k,i+2,j)
         end if

      end do

      close( iounit_turbo )

      return
      end





!***********************************************************************
      subroutine turbo_writebeta_MPI ( ibead )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iounit, iounit_turbo

      use turbo_variables, only : coeff_scf, nsaos_scf

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, ierr, ibead
      real(8) :: dummy(4)

      character(len=80):: char_line, char_file
      character(len=3) :: char_num

!-----------------------------------------------------------------------
!     /*   read molecular orbitals                                    */
!-----------------------------------------------------------------------

      call int3_to_char( ibead, char_num )

      char_file = ('./' // char_num // '/beta')
      open ( iounit,       file = char_file )
      char_file = ('./' // char_num // '/beta.1')
      open ( iounit_turbo, file = char_file )

!     /*   head part   */

      do
         read ( iounit_turbo, '(a80)' ) char_line
         if (char_line(1:1) .eq. '$') then
            write( iounit, '(a80)' ) char_line
            cycle
         end if
         if (char_line(1:1) .eq. '#') then
            write( iounit, '(a80)' ) char_line
            cycle
         end if
         backspace ( iounit_turbo )
         exit
      end do

!     /*   body part   */

      do j = 1, nsaos_scf

         read ( iounit_turbo, '(a80)' ) char_line
         write( iounit, '(a80)' ) char_line

         do i = 1, 4*int(nsaos_scf/4), 4
            read (iounit_turbo, '(4d20.14)' ) &
     &      dummy(1),        dummy(2), &
     &      dummy(3),        dummy(4)
            write(iounit, '(4d20.14)' ) &
     &      coeff_scf(0,i,j),   coeff_scf(0,i+1,j), &
     &      coeff_scf(0,i+2,j), coeff_scf(0,i+3,j)
         end do

         if      ( mod(nsaos_scf,4) .eq. 0 ) then
            continue
         else if ( mod(nsaos_scf,4) .eq. 1 ) then
            i = 4*int(nsaos_scf/4) + 1
            read (iounit_turbo, '(d20.14)' ) &
     &      dummy(1)
            write(iounit, '(d20.14)' ) &
     &      coeff_scf(0,i,j)
         else if ( mod(nsaos_scf,4) .eq. 2 ) then
            i = 4*int(nsaos_scf/4) + 1
            read (iounit_turbo, '(2d20.14)' ) &
     &      dummy(1), dummy(2)
            write(iounit, '(2d20.14)' ) &
     &      coeff_scf(0,i,j),   coeff_scf(0,i+1,j)
         else if ( mod(nsaos_scf,4) .eq. 3 ) then
            i = 4*int(nsaos_scf/4) + 1
            read (iounit_turbo, '(3d20.14)' ) &
     &      dummy(1), dummy(2), dummy(3)
            write(iounit, '(3d20.14)' ) &
     &      coeff_scf(0,i,j),   coeff_scf(0,i+1,j), &
     &      coeff_scf(0,i+2,j)
         end if

      end do

      do
         read ( iounit_turbo, '(a80)', iostat=ierr ) char_line
         if ( ierr .ne. 0 ) exit
         write( iounit, '(a80)' ) char_line
      end do

      close( iounit_turbo )
      close( iounit )

      return
      end

