!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     energy and force of Tully's two-state model
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine force_tully_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   common variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only:  x, y, z, natom, nbead

      use multistate_variables, only:  vstate, gxstate, gystate, &
     &    gzstate, dxstate, dystate, dzstate, nstate

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, n

      real(8) :: h

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

!     /*   unitary matrix   */
      real(8) :: u(nstate,nstate)

!     /*   potential matrix   */
      real(8) :: v(nstate,nstate)

!     /*   eigenvalues   */
      real(8) :: e(nstate)

!     /*   temporary dimension   */
      real(8) :: gx(nstate,nstate), gy(nstate,nstate), gz(nstate,nstate)

!     /*   temporary dimension   */
      real(8) :: u_xp(nstate,nstate,nbead), u_xm(nstate,nstate,nbead), &
     &           u_yp(nstate,nstate,nbead), u_ym(nstate,nstate,nbead), &
     &           u_zp(nstate,nstate,nbead), u_zm(nstate,nstate,nbead)

!     /*   temporary dimension   */
      real(8) :: dxu(nstate,nstate,nbead), dyu(nstate,nstate,nbead), &
     &           dzu(nstate,nstate,nbead)

!     /*   finite difference parameter   */
      h = 1.d-6

      do k = 1, natom

         x(k,:) = x(k,:) + h

         call force_tully_diabatic_MPI

         do n = 1, nbead
            do j = 1, nstate
            do i = 1, nstate
               v(i,j) = vstate(i,j,n)
            end do
            end do
            call ddiag_MPI ( v, e, u, nstate )
            do j = 1, nstate
               if ( u(1,j) .lt. 0.d0 ) then
                  do i = 1, nstate
                     u(i,j) = - u(i,j)
                  end do
               end if
            end do
            do j = 1, nstate
            do i = 1, nstate
               u_xp(i,j,n) = u(i,j)
            end do
            end do
         end do

         x(k,:) = x(k,:) - 2.d0*h

         call force_tully_diabatic_MPI

         do n = 1, nbead
            do j = 1, nstate
            do i = 1, nstate
               v(i,j) = vstate(i,j,n)
            end do
            end do
            call ddiag_MPI ( v, e, u, nstate )
            do j = 1, nstate
               if ( u(1,j) .lt. 0.d0 ) then
                  do i = 1, nstate
                     u(i,j) = - u(i,j)
                  end do
               end if
            end do
            do j = 1, nstate
            do i = 1, nstate
               u_xm(i,j,n) = u(i,j)
            end do
            end do
         end do

         x(k,:) = x(k,:) + h

         call force_tully_diabatic_MPI

         do n = 1, nbead
            do j = 1, nstate
            do i = 1, nstate
               v(i,j) = vstate(i,j,n)
            end do
            end do
            call ddiag_MPI ( v, e, u, nstate )
            do j = 1, nstate
               if ( u(1,j) .lt. 0.d0 ) then
                  do i = 1, nstate
                     u(i,j) = - u(i,j)
                  end do
               end if
            end do
            do j = 1, nstate
            do i = 1, nstate
               dxu(i,j,n) = (u_xp(i,j,n) - u_xm(i,j,n))/(2.d0*h)
            end do
            end do
         end do

         y(k,:) = y(k,:) + h

         call force_tully_diabatic_MPI

         do n = 1, nbead
            do j = 1, nstate
            do i = 1, nstate
               v(i,j) = vstate(i,j,n)
            end do
            end do
            call ddiag_MPI ( v, e, u, nstate )
            do j = 1, nstate
               if ( u(1,j) .lt. 0.d0 ) then
                  do i = 1, nstate
                     u(i,j) = - u(i,j)
                  end do
               end if
            end do
            do j = 1, nstate
            do i = 1, nstate
               u_yp(i,j,n) = u(i,j)
            end do
            end do
         end do

         y(k,:) = y(k,:) - 2.d0*h

         call force_tully_diabatic_MPI

         do n = 1, nbead
            do j = 1, nstate
            do i = 1, nstate
               v(i,j) = vstate(i,j,n)
            end do
            end do
            call ddiag_MPI ( v, e, u, nstate )
            do j = 1, nstate
               if ( u(1,j) .lt. 0.d0 ) then
                  do i = 1, nstate
                     u(i,j) = - u(i,j)
                  end do
               end if
            end do
            do j = 1, nstate
            do i = 1, nstate
               u_ym(i,j,n) = u(i,j)
            end do
            end do
         end do

         y(k,:) = y(k,:) + h

         call force_tully_diabatic_MPI

         do n = 1, nbead
            do j = 1, nstate
            do i = 1, nstate
               v(i,j) = vstate(i,j,n)
            end do
            end do
            call ddiag_MPI ( v, e, u, nstate )
            do j = 1, nstate
               if ( u(1,j) .lt. 0.d0 ) then
                  do i = 1, nstate
                     u(i,j) = - u(i,j)
                  end do
               end if
            end do
            do j = 1, nstate
            do i = 1, nstate
               dyu(i,j,n) = (u_yp(i,j,n) - u_ym(i,j,n))/(2.d0*h)
            end do
            end do
         end do

         z(k,:) = z(k,:) + h

         call force_tully_diabatic_MPI

         do n = 1, nbead
            do j = 1, nstate
            do i = 1, nstate
               v(i,j) = vstate(i,j,n)
            end do
            end do
            call ddiag_MPI ( v, e, u, nstate )
            do j = 1, nstate
               if ( u(1,j) .lt. 0.d0 ) then
                  do i = 1, nstate
                     u(i,j) = - u(i,j)
                  end do
               end if
            end do
            do j = 1, nstate
            do i = 1, nstate
               u_zp(i,j,n) = u(i,j)
            end do
            end do
         end do

         z(k,:) = z(k,:) - 2.d0*h

         call force_tully_diabatic_MPI

         do n = 1, nbead
            do j = 1, nstate
            do i = 1, nstate
               v(i,j) = vstate(i,j,n)
            end do
            end do
            call ddiag_MPI ( v, e, u, nstate )
            do j = 1, nstate
               if ( u(1,j) .lt. 0.d0 ) then
                  do i = 1, nstate
                     u(i,j) = - u(i,j)
                  end do
               end if
            end do
            do j = 1, nstate
            do i = 1, nstate
               u_zm(i,j,n) = u(i,j)
            end do
            end do
         end do

         z(k,:) = z(k,:) + h

         call force_tully_diabatic_MPI

         do n = 1, nbead
            do j = 1, nstate
            do i = 1, nstate
               v(i,j) = vstate(i,j,n)
            end do
            end do
            call ddiag_MPI ( v, e, u, nstate )
            do j = 1, nstate
               if ( u(1,j) .lt. 0.d0 ) then
                  do i = 1, nstate
                     u(i,j) = - u(i,j)
                  end do
               end if
            end do
            do j = 1, nstate
            do i = 1, nstate
               dzu(i,j,n) = (u_zp(i,j,n) - u_zm(i,j,n))/(2.d0*h)
            end do
            end do
         end do

!-----------------------------------------------------------------------

         call force_tully_diabatic_MPI

         do n = 1, nbead

            do j = 1, nstate
            do i = 1, nstate
               v(i,j) = vstate(i,j,n)
            end do
            end do

            call ddiag_MPI ( v, e, u, nstate )

            do j = 1, nstate
               if ( u(1,j) .lt. 0.d0 ) then
                  do i = 1, nstate
                     u(i,j) = - u(i,j)
                  end do
               end if
            end do

            do j = 1, nstate
            do i = 1, nstate
               if ( i .eq. j ) then
                  dxstate(i,i,k,n) = 0.d0
                  dystate(i,i,k,n) = 0.d0
                  dzstate(i,i,k,n) = 0.d0
               else
                  dxstate(i,j,k,n) = 0.d0
                  dystate(i,j,k,n) = 0.d0
                  dzstate(i,j,k,n) = 0.d0
                  do l = 1, nstate
                  do m = 1, nstate
                     dxstate(i,j,k,n) = dxstate(i,j,k,n) &
     &                              - u(m,i)*gxstate(m,l,k,n)*u(l,j)
                     dystate(i,j,k,n) = dystate(i,j,k,n) &
     &                              - u(m,i)*gystate(m,l,k,n)*u(l,j)
                     dzstate(i,j,k,n) = dzstate(i,j,k,n) &
     &                              - u(m,i)*gzstate(m,l,k,n)*u(l,j)
                  end do
                  end do
                  dxstate(i,j,k,n) = dxstate(i,j,k,n)/(e(i)-e(j))
                  dystate(i,j,k,n) = dystate(i,j,k,n)/(e(i)-e(j))
                  dzstate(i,j,k,n) = dzstate(i,j,k,n)/(e(i)-e(j))
               end if

               gx(i,j) = 0.d0
               gy(i,j) = 0.d0
               gz(i,j) = 0.d0

               do m = 1, nstate
               do l = 1, nstate
                  gx(i,j) = gx(i,j) + dxu(m,i,n)*vstate(m,l,n)*u(l,j)
                  gy(i,j) = gy(i,j) + dyu(m,i,n)*vstate(m,l,n)*u(l,j)
                  gz(i,j) = gz(i,j) + dzu(m,i,n)*vstate(m,l,n)*u(l,j)
                  gx(i,j) = gx(i,j) + u(m,i)*vstate(m,l,n)*dxu(l,j,n)
                  gy(i,j) = gy(i,j) + u(m,i)*vstate(m,l,n)*dyu(l,j,n)
                  gz(i,j) = gz(i,j) + u(m,i)*vstate(m,l,n)*dzu(l,j,n)
                  gx(i,j) = gx(i,j) + u(m,i)*gxstate(m,l,k,n)*u(l,j)
                  gy(i,j) = gy(i,j) + u(m,i)*gystate(m,l,k,n)*u(l,j)
                  gz(i,j) = gz(i,j) + u(m,i)*gzstate(m,l,k,n)*u(l,j)
               end do
               end do
            end do
            end do

            do j = 1, nstate
            do i = 1, nstate
               gxstate(i,j,k,n) = gx(i,j)
               gystate(i,j,k,n) = gy(i,j)
               gzstate(i,j,k,n) = gz(i,j)
            end do
            end do

            do j = 1, nstate
            do i = 1, nstate
               if ( i .eq. j ) then
                  vstate(i,i,n) = e(i)
               else
                  vstate(i,j,n) = 0.d0
               end if
            end do
            end do

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_tully_diabatic_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   common variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only:  x, natom, nbead, myrank, nprocs

      use multistate_variables, only:  vstate, gxstate, gystate, &
     &    gzstate, dxstate, dystate, dzstate, &
     &    dipxstate, dipystate, dipzstate, nstate

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: k, n

      real(8) :: a0, b0, c0, d0

!     /*   potential parameters   */
      parameter ( a0 = 0.01d0, b0 = 1.6d0, c0 = 0.005d0, d0 = 1.d0 )

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      vstate(:,:,:) = 0.d0

      gxstate(:,:,:,:) = 0.d0
      gystate(:,:,:,:) = 0.d0
      gzstate(:,:,:,:) = 0.d0

      dxstate(:,:,:,:) = 0.d0
      dystate(:,:,:,:) = 0.d0
      dzstate(:,:,:,:) = 0.d0

      dipxstate(:,:)   = 0.d0
      dipystate(:,:)   = 0.d0
      dipzstate(:,:)   = 0.d0

      do n = 1, nbead

         if ( mod(n,nprocs) .ne. myrank ) cycle

         do k = 1, natom

            if ( x(k,n) .ge. 0.d0 ) then

               vstate(1,1,n) = vstate(1,1,n) &
     &             + a0*( 1.d0 - exp(-b0*x(k,n)) )
               vstate(1,2,n) = vstate(1,2,n) &
     &             + c0*exp(-d0*x(k,n)*x(k,n))
               vstate(2,1,n) = vstate(2,1,n) &
     &             + c0*exp(-d0*x(k,n)*x(k,n))
               vstate(2,2,n) = vstate(2,2,n) &
     &             - a0*( 1.d0 - exp(-b0*x(k,n)) )
               gxstate(1,1,k,n) = gxstate(1,1,k,n) &
     &             + a0*b0*exp(-b0*x(k,n))
               gxstate(1,2,k,n) = gxstate(1,2,k,n) &
     &             - 2.d0*c0*d0*x(k,n)*exp(-d0*x(k,n)*x(k,n))
               gxstate(2,1,k,n) = gxstate(2,1,k,n) &
     &             - 2.d0*c0*d0*x(k,n)*exp(-d0*x(k,n)*x(k,n))
               gxstate(2,2,k,n) = gxstate(2,2,k,n) &
     &             - a0*b0*exp(-b0*x(k,n))

            else if ( x(k,n) .lt. 0.d0 ) then

               vstate(1,1,n) = vstate(1,1,n) &
     &             - a0*( 1.d0 - exp(+b0*x(k,n)) )
               vstate(1,2,n) = vstate(1,2,n) &
     &             + c0*exp(-d0*x(k,n)*x(k,n))
               vstate(2,1,n) = vstate(2,1,n) &
     &             + c0*exp(-d0*x(k,n)*x(k,n))
               vstate(2,2,n) = vstate(2,2,n) &
     &             + a0*( 1.d0 - exp(+b0*x(k,n)) )
               gxstate(1,1,k,n) = gxstate(1,1,k,n) &
     &             + a0*b0*exp(+b0*x(k,n))
               gxstate(1,2,k,n) = gxstate(1,2,k,n) &
     &             - 2.d0*c0*d0*x(k,n)*exp(-d0*x(k,n)*x(k,n))
               gxstate(2,1,k,n) = gxstate(2,1,k,n) &
     &             - 2.d0*c0*d0*x(k,n)*exp(-d0*x(k,n)*x(k,n))
               gxstate(2,2,k,n) = gxstate(2,2,k,n) &
     &             - a0*b0*exp(+b0*x(k,n))

            end if

         end do

      end do

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_3 ( vstate, nstate, nstate, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_4 &
     &   ( gxstate, nstate, nstate, natom, nbead )
      call my_mpi_allreduce_real_4 &
     &   ( gystate, nstate, nstate, natom, nbead )
      call my_mpi_allreduce_real_4 &
     &   ( gzstate, nstate, nstate, natom, nbead )

!     /*   dipole moment   */
      call my_mpi_allreduce_real_2 ( dipxstate, nstate, nbead )
      call my_mpi_allreduce_real_2 ( dipystate, nstate, nbead )
      call my_mpi_allreduce_real_2 ( dipzstate, nstate, nbead )

!     /*   nonadiabatic coupling matrix elements   */
      call my_mpi_allreduce_real_4 &
     &     ( dxstate, nstate, nstate, natom, nbead )
      call my_mpi_allreduce_real_4 &
     &     ( dystate, nstate, nstate, natom, nbead )
      call my_mpi_allreduce_real_4 &
     &     ( dzstate, nstate, nstate, natom, nbead )

      return
      end

