!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Dec 18, 2020 by M. Shiga
!      Description:     tersoff force field
!
!///////////////////////////////////////////////////////////////////////
!-----------------------------------------------------------------------
!      tersoff parameters
!-----------------------------------------------------------------------
!
!      name        C-C       Si-C        Si-Si     ! unit
!
!      A       1393.60     1681.731    1839.80     ! eV
!      B       346.740     432.154     471.180     ! eV
!      lambda  3.48790     2.97260     2.47990     ! angstrom**-1
!      mu      2.21190     2.01930     1.73220     ! angstrom**-1
!      gamma   1.57240e-7  0.00000     1.10000e-7  ! none
!      n       0.72751     0.00000     0.78734     ! none
!      c       38049.0     0.00000     100390.0    ! none
!      d       4.34840     0.00000     16.2170     ! none
!      h      -0.57058     0.00000    -0.59825     ! none
!      S       2.10000     2.20454     2.70000     ! angstrom
!      R       1.95000     2.50998     3.00000     ! angstrom
!      chi     1.00000     1.00000     1.00000     ! none
!      omega   1.00000     1.00000     1.00000     ! none
!      sigma   0.00000     0.00000     0.00000     ! angstrom**-3
!
!***********************************************************************
      module tersoff_variables
!***********************************************************************

!     //   number of atomic kinds
      integer :: ntrsf

!     //   A
      real(8), dimension(:,:), allocatable :: a_trsf

!     //   B
      real(8), dimension(:,:), allocatable :: b_trsf

!     //   lambda (lambda_1)
      real(8), dimension(:,:), allocatable :: lambda_trsf

!     //   mu (lambda_2)
      real(8), dimension(:,:), allocatable :: mu_trsf

!     //   gamma
      real(8), dimension(:,:), allocatable :: gamma_trsf

!     //   n
      real(8), dimension(:,:), allocatable :: n_trsf

!     //   c
      real(8), dimension(:,:), allocatable :: c_trsf

!     //   d
      real(8), dimension(:,:), allocatable :: d_trsf

!     //   h
      real(8), dimension(:,:), allocatable :: h_trsf

!     //   S
      real(8), dimension(:,:), allocatable :: s_trsf

!     //   R
      real(8), dimension(:,:), allocatable :: r_trsf

!     //   chi
      real(8), dimension(:,:), allocatable :: chi_trsf

!     //   omega
      real(8), dimension(:,:), allocatable :: omega_trsf

!     //   sigma
      real(8), dimension(:,:), allocatable :: sigma_trsf

!     //   maximum cut off distance
      real(8) :: rcut_trsf

!     //   number of replicated boxes
      integer, dimension(:), allocatable :: nbox_trsf

!     //   bigbox
      real(8), dimension(:,:), allocatable :: bigbox

!     //   inverse of bigbox
      real(8), dimension(:,:), allocatable :: bigboxinv

!     //   cut off skin of neighbor list
      real(8) :: skin_trsf

!***********************************************************************
      end module tersoff_variables
!***********************************************************************





!***********************************************************************
      subroutine force_tersoff_setup
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   au_charge, au_energy, au_length, nkind, iounit

      use tersoff_variables, only : &
     &   a_trsf, b_trsf, lambda_trsf, mu_trsf, gamma_trsf, n_trsf, &
     &   c_trsf, d_trsf, h_trsf, s_trsf, r_trsf, chi_trsf, omega_trsf, &
     &   sigma_trsf, rcut_trsf, bigbox, bigboxinv, ntrsf, nbox_trsf

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize variables
      implicit none

!     //   integers
      integer :: ierr, i, j, k

!     //   real numbers
      real(8) :: factor_1, factor_2, factor_3

!     //   characters
      character(len=8) :: char

!-----------------------------------------------------------------------
!     /*   read parameters                                            */
!-----------------------------------------------------------------------

!     /*   open file   */
      open ( iounit, file = 'tersoff.dat' )

!     /*   tag   */
      call search_tag ( '<tersoff>', 9, iounit, ierr )

!     /*   number of species   */
      read( iounit, *, iostat=ierr ) ntrsf

!     /*   close file   */
      close( iounit )

!     /*   check if parameters of all atoms are set   */
      if ( ntrsf .lt. nkind ) ierr = 1

!     /*   error termination   */
      call error_handling &
     &    ( ierr, 'subroutine force_tersoff_setup', 30 )

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( .not. allocated(a_trsf) ) allocate( a_trsf(ntrsf,ntrsf) )
      if ( .not. allocated(b_trsf) ) allocate( b_trsf(ntrsf,ntrsf) )
      if ( .not. allocated(lambda_trsf) ) &
     &                          allocate( lambda_trsf(ntrsf,ntrsf) )
      if ( .not. allocated(mu_trsf) ) &
     &                              allocate( mu_trsf(ntrsf,ntrsf) )
      if ( .not. allocated(gamma_trsf) ) &
     &                           allocate( gamma_trsf(ntrsf,ntrsf) )
      if ( .not. allocated(n_trsf) ) allocate( n_trsf(ntrsf,ntrsf) )
      if ( .not. allocated(c_trsf) ) allocate( c_trsf(ntrsf,ntrsf) )
      if ( .not. allocated(d_trsf) ) allocate( d_trsf(ntrsf,ntrsf) )
      if ( .not. allocated(h_trsf) ) allocate( h_trsf(ntrsf,ntrsf) )
      if ( .not. allocated(s_trsf) ) allocate( s_trsf(ntrsf,ntrsf) )
      if ( .not. allocated(r_trsf) ) allocate( r_trsf(ntrsf,ntrsf) )
      if ( .not. allocated(chi_trsf) ) &
     &                             allocate( chi_trsf(ntrsf,ntrsf) )
      if ( .not. allocated(omega_trsf) ) &
     &                           allocate( omega_trsf(ntrsf,ntrsf) )
      if ( .not. allocated(sigma_trsf) ) &
     &                           allocate( sigma_trsf(ntrsf,ntrsf) )

      if ( .not. allocated(nbox_trsf) )  allocate( nbox_trsf(3) )
      if ( .not. allocated(bigbox) )     allocate( bigbox(3,3) )
      if ( .not. allocated(bigboxinv) ) allocate( bigboxinv(3,3) )

!-----------------------------------------------------------------------
!     /*   read parameters                                            */
!-----------------------------------------------------------------------

!     /*   open file   */
      open ( iounit, file = 'tersoff.dat' )

!     /*   tag   */
      call search_tag ( '<tersoff>', 9, iounit, ierr )

!     /*   number of species   */
      read( iounit, *, iostat=ierr )

!     /*   loop of parameters    */
      do k = 1, 14

!        /*   line   */
         read( iounit, *, iostat=ierr ) char

!        /*   go back one line   */
         backspace( iounit )

!        /*   read parameters   */
         if      ( char(1:7) .eq. 'A      ' ) then
            read( iounit, *, iostat=ierr ) char, &
     &         ( (a_trsf(i,j), j = i, ntrsf), i = 1, ntrsf )
         else if ( char(1:7) .eq. 'B      ' ) then
            read( iounit, *, iostat=ierr ) char, &
     &         ( (b_trsf(i,j), j = i, ntrsf), i = 1, ntrsf )
         else if ( char(1:7) .eq. 'lambda ' ) then
            read( iounit, *, iostat=ierr ) char, &
     &         ( (lambda_trsf(i,j), j = i, ntrsf), i = 1, ntrsf )
         else if ( char(1:7) .eq. 'mu     ' ) then
            read( iounit, *, iostat=ierr ) char, &
     &         ( (mu_trsf(i,j), j = i, ntrsf), i = 1, ntrsf )
         else if ( char(1:7) .eq. 'gamma  ' ) then
            read( iounit, *, iostat=ierr ) char, &
     &         ( (gamma_trsf(i,j), j = i, ntrsf), i = 1, ntrsf )
         else if ( char(1:7) .eq. 'n      ' ) then
            read( iounit, *, iostat=ierr ) char, &
     &         ( (n_trsf(i,j), j = i, ntrsf), i = 1, ntrsf )
         else if ( char(1:7) .eq. 'c      ' ) then
            read( iounit, *, iostat=ierr ) char, &
     &         ( (c_trsf(i,j), j = i, ntrsf), i = 1, ntrsf )
         else if ( char(1:7) .eq. 'd      ' ) then
            read( iounit, *, iostat=ierr ) char, &
     &         ( (d_trsf(i,j), j = i, ntrsf), i = 1, ntrsf )
         else if ( char(1:7) .eq. 'h      ' ) then
            read( iounit, *, iostat=ierr ) char, &
     &         ( (h_trsf(i,j), j = i, ntrsf), i = 1, ntrsf )
         else if ( char(1:7) .eq. 'S      ' ) then
            read( iounit, *, iostat=ierr ) char, &
     &         ( (s_trsf(i,j), j = i, ntrsf), i = 1, ntrsf )
         else if ( char(1:7) .eq. 'R      ' ) then
            read( iounit, *, iostat=ierr ) char, &
     &         ( (r_trsf(i,j), j = i, ntrsf), i = 1, ntrsf )
         else if ( char(1:7) .eq. 'chi    ' ) then
            read( iounit, *, iostat=ierr ) char, &
     &         ( (chi_trsf(i,j), j = i, ntrsf), i = 1, ntrsf )
         else if ( char(1:7) .eq. 'omega  ' ) then
            read( iounit, *, iostat=ierr ) char, &
     &         ( (omega_trsf(i,j), j = i, ntrsf), i = 1, ntrsf )
         else if ( char(1:7) .eq. 'sigma  ' ) then
            read( iounit, *, iostat=ierr ) char, &
     &         ( (sigma_trsf(i,j), j = i, ntrsf), i = 1, ntrsf )
         end if

!     /*   loop of parameters    */
      end do

!     /*   close file   */
      close( iounit )

!     /*   error termination   */
      call error_handling &
     &    ( ierr, 'subroutine force_tersoff_setup', 30 )

!-----------------------------------------------------------------------
!     /*   unit conversion                                            */
!-----------------------------------------------------------------------

!     /*   electron volt -> hartree   */
      factor_1 = au_charge / au_energy

!     /*   angstrom**-1 -> bohr**-1   */
      factor_2 = 1.d+10 * au_length

!     /*   angstrom -> bohr   */
      factor_3 = 1.d-10 / au_length

!     /*   loop of parameters    */
      do i = 1, ntrsf
      do j = i, ntrsf

!        //   conversion to atomic units
         a_trsf(i,j)      = a_trsf(i,j)      * factor_1
         b_trsf(i,j)      = b_trsf(i,j)      * factor_1
         lambda_trsf(i,j) = lambda_trsf(i,j) * factor_2
         mu_trsf(i,j)     = mu_trsf(i,j)     * factor_2
         s_trsf(i,j)      = s_trsf(i,j)      * factor_3
         r_trsf(i,j)      = r_trsf(i,j)      * factor_3
         sigma_trsf(i,j)  = sigma_trsf(i,j)  * factor_2**3

!        //   symmetrize parameters
         a_trsf(j,i)      = a_trsf(i,j)
         b_trsf(j,i)      = b_trsf(i,j)
         lambda_trsf(j,i) = lambda_trsf(i,j)
         mu_trsf(j,i)     = mu_trsf(i,j)
         gamma_trsf(j,i)  = gamma_trsf(i,j)
         n_trsf(j,i)      = n_trsf(i,j)
         c_trsf(j,i)      = c_trsf(i,j)
         d_trsf(j,i)      = d_trsf(i,j)
         h_trsf(j,i)      = h_trsf(i,j)
         s_trsf(j,i)      = s_trsf(i,j)
         r_trsf(j,i)      = r_trsf(i,j)
         chi_trsf(j,i)    = chi_trsf(i,j)
         omega_trsf(j,i)  = omega_trsf(i,j)
         sigma_trsf(j,i)  = sigma_trsf(i,j)

!     /*   loop of parameters    */
      end do
      end do

!-----------------------------------------------------------------------
!     /*   maximum cut off distance                                   */
!-----------------------------------------------------------------------

      rcut_trsf = maxval( s_trsf(:,:) )

      return
      end





!***********************************************************************
      subroutine force_tersoff
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, x, y, z, fx, fy, fz, pot, vir, box, volume, iboundary, &
     &   natom, nbead, ikind

      use mm_variables, only : &
     &   j_list, n_list

      use tersoff_variables, only : &
     &   a_trsf, b_trsf, lambda_trsf, mu_trsf, gamma_trsf, n_trsf, &
     &   c_trsf, d_trsf, h_trsf, s_trsf, r_trsf, chi_trsf, omega_trsf, &
     &   sigma_trsf, rcut_trsf, bigbox, bigboxinv, nbox_trsf

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize variables
      implicit none

!     //   integers
      integer :: i, j, k, li, lj, lk, m, jx, jy, jz, j2, kx, ky, kz, &
     &           k2, jk2, n1, n2

!     //   real numbers
      real(8) :: xij, yij, zij, rij, rij2, xik, yik, zik, rik, rik2, &
     &           a_ij, b_ij, la_ij, mu_ij, ga_ij, n_ij, c_ij, d_ij, &
     &           h_ij, s_ij, r_ij, ch_ij, om_ik, fc_ij, dfc_ij, fc_ik, &
     &           dfc_ik, cos_kij, g_kij, dg_kij, zt_ij, bo_ij, ea_ij, &
     &           eb_ij, r_ik, s_ik, rij_inv, fxij, fyij, fzij, sg_ik, &
     &           fxik, fyik, fzik, f1, f2, f3, f4, f5, f6, f7, rik_inv, &
     &           q_ij, es_kij, qo, des_kij, absa, absb, absc, ax, ay, &
     &           az, bx, by, bz, cx, cy, cz, aij, bij, cij, aik, bik, &
     &           cik

!     //   flag
      integer, save :: iset = 0

!     //   real numbers
      real(8) :: tiny = 1.d-8

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     //   initial access
      if ( iset .eq. 0 ) then

!        //   setup tersoff parameters
         call force_tersoff_setup

!        //   setup complete
         iset = 1

!     //   initial access
      end if

!-----------------------------------------------------------------------
!     /*   extention of simulation box in real space sum              */
!-----------------------------------------------------------------------

!     /*   free boundary   */
      if ( iboundary .eq. 0 ) then

!        /*   number of replicated boxes   */
         nbox_trsf(1) = 1
         nbox_trsf(2) = 1
         nbox_trsf(3) = 1

!     /*   periodic boundary   */
      else

!        /*   vector product of lattice vectors b, c   */
         ax = box(2,2)*box(3,3) - box(2,3)*box(3,2)
         ay = box(3,2)*box(1,3) - box(3,3)*box(1,2)
         az = box(1,2)*box(2,3) - box(1,3)*box(2,2)

!        /*   vector product of lattice vectors c, a   */
         bx = box(2,3)*box(3,1) - box(2,1)*box(3,3)
         by = box(3,3)*box(1,1) - box(3,1)*box(1,3)
         bz = box(1,3)*box(2,1) - box(1,1)*box(2,3)

!        /*   vector product of lattice vectors a, b   */
         cx = box(2,1)*box(3,2) - box(2,2)*box(3,1)
         cy = box(3,1)*box(1,2) - box(3,2)*box(1,1)
         cz = box(1,1)*box(2,2) - box(1,2)*box(2,1)

!        /*   distance between parallel planes   */
         absa = volume / sqrt( ax*ax + ay*ay + az*az )
         absb = volume / sqrt( bx*bx + by*by + bz*bz )
         absc = volume / sqrt( cx*cx + cy*cy + cz*cz )

!        /*   number of replicated boxes   */
         nbox_trsf(1) = int(2.d0*rcut_trsf/absa) + 1
         nbox_trsf(2) = int(2.d0*rcut_trsf/absb) + 1
         nbox_trsf(3) = int(2.d0*rcut_trsf/absc) + 1

!     /*   boundary condition   */
      end if

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

!     /*   free boundary or minimum image convention   */
      if ( nbox_trsf(1)*nbox_trsf(2)*nbox_trsf(3) .eq. 1 ) then

!        /*   make neighbor list   */
         call force_tersoff_makelist

!        //   loop of beads
         do m = 1, nbead

!           //   loop of atom pairs
            do i = 1, natom
            do n1 = 1, n_list(i,m)

!              /*   atom in neighbor list   */
               j = j_list(n1,i,m)

!              //   i and j are not the same atom
               if ( i .eq. j ) cycle

!              //   atomic kinds
               li = ikind(i)
               lj = ikind(j)

!              //   interatomic separation
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              //   apply boundary condition
               call pbc_atom ( xij, yij, zij )

!              //   distance squared
               rij2 = xij*xij + yij*yij + zij*zij

!              //   tersoff parameters between i and j
               s_ij   = s_trsf(li,lj)

!              //   cut off
               if ( rij2 .gt. s_ij*s_ij ) cycle

!              //   tersoff parameters between i and j
               a_ij   = a_trsf(li,lj)
               b_ij   = b_trsf(li,lj)
               la_ij  = lambda_trsf(li,lj)
               mu_ij  = mu_trsf(li,lj)
               ga_ij  = gamma_trsf(li,lj)
               n_ij   = n_trsf(li,lj)
               c_ij   = c_trsf(li,lj)
               d_ij   = d_trsf(li,lj)
               h_ij   = h_trsf(li,lj)
               r_ij   = r_trsf(li,lj)
               ch_ij  = chi_trsf(li,lj)

!              //   distance
               rij = sqrt(rij2)

!              //   inverse of distance
               rij_inv = 1.d0 / rij

!              //   function fc and its gradient
               call fc_trsf( rij, s_ij, r_ij, pi, fc_ij, dfc_ij )

!              //   zeta function
               zt_ij = 0.d0

!              //   case n_ij = 0
               if ( abs(n_ij) .lt. tiny ) then

!                 //  function b
                  bo_ij = ch_ij

!              //   case n_ij > 0
               else

!                 //   loop of third atom
                  do n2 = 1, n_list(i,m)

!                    /*   atom in neighbor list   */
                     k = j_list(n2,i,m)

!                    //   k atom not must be i or j atom
                     if ( ( k .eq. i ) .or. ( k .eq. j ) ) cycle

!                    //   atomic kind
                     lk = ikind(k)

!                    //   interatomic separation
                     xik = x(i,m) - x(k,m)
                     yik = y(i,m) - y(k,m)
                     zik = z(i,m) - z(k,m)

!                    //   apply boundary condition
                     call pbc_atom ( xik, yik, zik )

!                    //   distance squared
                     rik2 = xik*xik + yik*yik + zik*zik

!                    //   tersoff parameters between i and j
                     s_ik = s_trsf(li,lk)

!                    //   cut off
                     if ( rik2 .gt. s_ik*s_ik ) cycle

!                    //   distance
                     rik = sqrt(rik2)

!                    //   ersoff parameters between i and j
                     r_ik = r_trsf(li,lk)

!                    //   function fc and its gradient
                     call fc_trsf &
     &                  ( rik, s_ik, r_ik, pi, fc_ik, dfc_ik )

!                    //   cosine of k-i-j angle
                     cos_kij = ( xij*xik + yij*yik + zij*zik ) &
     &                       / ( rij*rik )

!                    //   g function
                     call g_trsf &
     &                  ( cos_kij, c_ij, d_ij, h_ij, g_kij, dg_kij )

!                    //   omega constant
                     om_ik = omega_trsf(li,lk)

!                    //   sigma constant
                     sg_ik = sigma_trsf(li,lk)

!                    //   exponential factor
                     call e_trsf( sg_ik, rij, rik, es_kij, des_kij )

!                    //   zeta function
                     zt_ij = zt_ij + fc_ik * om_ik * g_kij * es_kij

!                 //   loop of third atom
                  end do

!                 //   function b
                  bo_ij = ch_ij &
     &               * ( 1.d0 + (ga_ij*zt_ij)**n_ij )**(-0.5d0/n_ij)

!              //   case n_ij
               end if

!              //   functions
               ea_ij = a_ij * exp( - la_ij * rij )
               eb_ij = b_ij * exp( - mu_ij * rij )

!              //   potential
               pot(m) = pot(m) + 0.5d0*fc_ij * ( ea_ij - bo_ij * eb_ij )

!              //   force
               f1 = + dfc_ij * ( ea_ij - bo_ij * eb_ij ) &
     &              - fc_ij * ( la_ij * ea_ij - mu_ij * bo_ij * eb_ij )

!              //   force
               fxij = - f1 * xij * rij_inv
               fyij = - f1 * yij * rij_inv
               fzij = - f1 * zij * rij_inv

!              //   force
               fx(i,m) = fx(i,m) + 0.5d0*fxij
               fy(i,m) = fy(i,m) + 0.5d0*fyij
               fz(i,m) = fz(i,m) + 0.5d0*fzij
               fx(j,m) = fx(j,m) - 0.5d0*fxij
               fy(j,m) = fy(j,m) - 0.5d0*fyij
               fz(j,m) = fz(j,m) - 0.5d0*fzij

!              //   virial
               vir(1,1) = vir(1,1) + 0.5d0*fxij*xij
               vir(1,2) = vir(1,2) + 0.5d0*fxij*yij
               vir(1,3) = vir(1,3) + 0.5d0*fxij*zij
               vir(2,1) = vir(2,1) + 0.5d0*fyij*xij
               vir(2,2) = vir(2,2) + 0.5d0*fyij*yij
               vir(2,3) = vir(2,3) + 0.5d0*fyij*zij
               vir(3,1) = vir(3,1) + 0.5d0*fzij*xij
               vir(3,2) = vir(3,2) + 0.5d0*fzij*yij
               vir(3,3) = vir(3,3) + 0.5d0*fzij*zij

!              //   case n_ij = 0
               if ( abs(n_ij) .lt. tiny ) then

!                 //   no three-body term
                  cycle

!              //   case n_ij > 0
               else

!                 //   q function
                  call q_trsf &
     &               ( fc_ij, bo_ij, eb_ij, ga_ij, zt_ij, n_ij, q_ij )

!                 //   loop of third atom
                  do n2 = 1, n_list(i,m)

!                    /*   atom in neighbor list   */
                     k = j_list(n2,i,m)

!                    //   k atom not must be i or j atom
                     if ( ( k .eq. i ) .or. ( k .eq. j ) ) cycle

!                    //   atomic kind
                     lk = ikind(k)

!                    //   interatomic separation
                     xik = x(i,m) - x(k,m)
                     yik = y(i,m) - y(k,m)
                     zik = z(i,m) - z(k,m)

!                    //   apply boundary condition
                     call pbc_atom ( xik, yik, zik )

!                    //   distance squared
                     rik2 = xik*xik + yik*yik + zik*zik

!                    //   tersoff parameters between i and j
                     s_ik = s_trsf(li,lk)

!                    //   cut off
                     if ( rik2 .gt. s_ik*s_ik ) cycle

!                    //   distance
                     rik = sqrt(rik2)

!                    //   inverse of distance
                     rik_inv = 1.d0 / rik

!                    //   tersoff parameters between i and j
                     r_ik = r_trsf(li,lk)

!                    //   function fc and its gradient
                     call fc_trsf( rik, s_ik, r_ik, pi, fc_ik, dfc_ik )

!                    //   cosine of k-i-j angle
                     cos_kij = ( xij*xik + yij*yik + zij*zik ) &
     &                       / ( rij*rik )

!                    //   g function
                     call g_trsf &
     &                  ( cos_kij, c_ij, d_ij, h_ij, g_kij, dg_kij )

!                    //   omega constant
                     om_ik = omega_trsf(li,lk)

!                    //   sigma constant
                     sg_ik = sigma_trsf(li,lk)

!                    //   e function
                     call e_trsf( sg_ik, rij, rik, es_kij, des_kij )

!                    //   constant
                     qo = q_ij * om_ik

!                    //   force
                     f2 = qo * dfc_ik * g_kij * es_kij
                     f3 = qo * fc_ik * dg_kij * es_kij * rik_inv
                     f4 = qo * fc_ik * dg_kij * es_kij * rij_inv
                     f5 = qo * fc_ik * dg_kij * es_kij * cos_kij*rik_inv
                     f6 = qo * fc_ik * dg_kij * es_kij * cos_kij*rij_inv
                     f7 = qo * fc_ik * g_kij * des_kij

!                    //   force
                     fxik = + (-f2+f5+f7) * xik * rik_inv &
     &                      - f3 * xij * rij_inv
                     fyik = + (-f2+f5+f7) * yik * rik_inv &
     &                      - f3 * yij * rij_inv
                     fzik = + (-f2+f5+f7) * zik * rik_inv &
     &                      - f3 * zij * rij_inv

!                    //   force
                     fxij = - f4 * xik * rik_inv &
     &                      + (f6-f7) * xij * rij_inv
                     fyij = - f4 * yik * rik_inv &
     &                      + (f6-f7) * yij * rij_inv
                     fzij = - f4 * zik * rik_inv &
     &                      + (f6-f7) * zij * rij_inv

!                    //   force
                     fx(i,m) = fx(i,m) + 0.5d0*(fxik + fxij)
                     fy(i,m) = fy(i,m) + 0.5d0*(fyik + fyij)
                     fz(i,m) = fz(i,m) + 0.5d0*(fzik + fzij)
                     fx(j,m) = fx(j,m) - 0.5d0*fxij
                     fy(j,m) = fy(j,m) - 0.5d0*fyij
                     fz(j,m) = fz(j,m) - 0.5d0*fzij
                     fx(k,m) = fx(k,m) - 0.5d0*fxik
                     fy(k,m) = fy(k,m) - 0.5d0*fyik
                     fz(k,m) = fz(k,m) - 0.5d0*fzik

!                    //   virial
                     vir(1,1) = vir(1,1) + 0.5d0*(fxik*xik + fxij*xij)
                     vir(1,2) = vir(1,2) + 0.5d0*(fxik*yik + fxij*yij)
                     vir(1,3) = vir(1,3) + 0.5d0*(fxik*zik + fxij*zij)
                     vir(2,1) = vir(2,1) + 0.5d0*(fyik*xik + fyij*xij)
                     vir(2,2) = vir(2,2) + 0.5d0*(fyik*yik + fyij*yij)
                     vir(2,3) = vir(2,3) + 0.5d0*(fyik*zik + fyij*zij)
                     vir(3,1) = vir(3,1) + 0.5d0*(fzik*xik + fzij*xij)
                     vir(3,2) = vir(3,2) + 0.5d0*(fzik*yik + fzij*yij)
                     vir(3,3) = vir(3,3) + 0.5d0*(fzik*zik + fzij*zij)

!                 //   loop of third atom
                  end do

!              //   case n_ij
               end if

!           //   loop of atom pairs
            end do
            end do

!        //   loop of beads
         end do

!     /*   periodic boundary   */
      else

!        /*   replicated boxes   */
         bigbox(:,1) = dble(nbox_trsf(1))*box(:,1)
         bigbox(:,2) = dble(nbox_trsf(2))*box(:,2)
         bigbox(:,3) = dble(nbox_trsf(3))*box(:,3)

!        /*   inverse of box matrix   */
         call inv3 ( bigbox, bigboxinv )

!        //   loop of beads
         do m = 1, nbead

!           /*   loop of atom pairs   */
            do i = 1, natom
            do j = 1, natom

!           //   atomic kinds
            li = ikind(i)
            lj = ikind(j)

!           /*   loop of replicated boxes   */
            do jx = 0, nbox_trsf(1)-1
            do jy = 0, nbox_trsf(2)-1
            do jz = 0, nbox_trsf(3)-1

!              /*   square of box index   */
               j2 = jx*jx + jy*jy + jz*jz

!              /*   skip same atom   */
               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

!              /*   interatomic distance of i and j in same box   */
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              /*   distance of i and j in different box  */
               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

!              /*   vector in big box   */
               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

!              /*   apply periodic boundary in big box   */
               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

!              /*   distance of nearest i and j   */
               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

!              /*    interatomic distance squared   */
               rij2 = xij*xij + yij*yij + zij*zij

!              //   tersoff parameters between i and j
               s_ij   = s_trsf(li,lj)

!              //   cut off
               if ( rij2 .gt. s_ij*s_ij ) cycle

!              //   tersoff parameters between i and j
               a_ij   = a_trsf(li,lj)
               b_ij   = b_trsf(li,lj)
               la_ij  = lambda_trsf(li,lj)
               mu_ij  = mu_trsf(li,lj)
               ga_ij  = gamma_trsf(li,lj)
               n_ij   = n_trsf(li,lj)
               c_ij   = c_trsf(li,lj)
               d_ij   = d_trsf(li,lj)
               h_ij   = h_trsf(li,lj)
               r_ij   = r_trsf(li,lj)
               ch_ij  = chi_trsf(li,lj)

!              //   distance
               rij = sqrt(rij2)

!              //   inverse of distance
               rij_inv = 1.d0 / rij

!              //   function fc and its gradient
               call fc_trsf( rij, s_ij, r_ij, pi, fc_ij, dfc_ij )

!              //   zeta function
               zt_ij = 0.d0

!              //   case n_ij = 0
               if ( abs(n_ij) .lt. tiny ) then

!                 //  function b
                  bo_ij = ch_ij

!              //   case n_ij > 0
               else

!                 //   loop of third atom
                  do k = 1, natom

!                 //   atomic kind
                  lk = ikind(k)

!                 /*   loop of replicated boxes   */
                  do kx = 0, nbox_trsf(1)-1
                  do ky = 0, nbox_trsf(2)-1
                  do kz = 0, nbox_trsf(3)-1

!                    /*   square of box index   */
                     k2 = kx*kx + ky*ky + kz*kz

!                    /*   skip same atom   */
                     if ( ( k2 .eq. 0 ) .and. ( k .eq. i ) ) cycle

!                    /*   square of box index   */
                     jk2 = (jx-kx)*(jx-kx) + (jy-ky)*(jy-ky) &
     &                   + (jz-kz)*(jz-kz)

!                    /*   skip same atom   */
                     if ( ( jk2 .eq. 0 ) .and. ( k .eq. j ) ) cycle

!                    /*   interatomic distance in same box   */
                     xik = x(i,m) - x(k,m)
                     yik = y(i,m) - y(k,m)
                     zik = z(i,m) - z(k,m)

!                    /*   distance in different box  */
                     xik = xik - box(1,1)*kx - box(1,2)*ky - box(1,3)*kz
                     yik = yik - box(2,1)*kx - box(2,2)*ky - box(2,3)*kz
                     zik = zik - box(3,1)*kx - box(3,2)*ky - box(3,3)*kz

!                    /*   vector in big box   */
                     aik = bigboxinv(1,1)*xik + bigboxinv(1,2)*yik &
     &                   + bigboxinv(1,3)*zik
                     bik = bigboxinv(2,1)*xik + bigboxinv(2,2)*yik &
     &                   + bigboxinv(2,3)*zik
                     cik = bigboxinv(3,1)*xik + bigboxinv(3,2)*yik &
     &                   + bigboxinv(3,3)*zik

!                    /*   apply periodic boundary in big box   */
                     aik = aik - nint(aik)
                     bik = bik - nint(bik)
                     cik = cik - nint(cik)

!                    /*   distance of nearest i and k   */
                     xik = bigbox(1,1)*aik + bigbox(1,2)*bik &
     &                   + bigbox(1,3)*cik
                     yik = bigbox(2,1)*aik + bigbox(2,2)*bik &
     &                   + bigbox(2,3)*cik
                     zik = bigbox(3,1)*aik + bigbox(3,2)*bik &
     &                   + bigbox(3,3)*cik

!                    /*    interatomic distance squared   */
                     rik2 = xik*xik + yik*yik + zik*zik

!                    //   tersoff parameters between i and j
                     s_ik = s_trsf(li,lk)

!                    //   cut off
                     if ( rik2 .gt. s_ik*s_ik ) cycle

!                    //   distance
                     rik = sqrt(rik2)

!                    //   ersoff parameters between i and j
                     r_ik = r_trsf(li,lk)

!                    //   function fc and its gradient
                     call fc_trsf &
     &                  ( rik, s_ik, r_ik, pi, fc_ik, dfc_ik )

!                    //   cosine of k-i-j angle
                     cos_kij = ( xij*xik + yij*yik + zij*zik ) &
     &                       / ( rij*rik )

!                    //   g function
                     call g_trsf &
     &                  ( cos_kij, c_ij, d_ij, h_ij, g_kij, dg_kij )

!                    //   omega constant
                     om_ik = omega_trsf(li,lk)

!                    //   sigma constant
                     sg_ik = sigma_trsf(li,lk)

!                    //   exponential factor
                     call e_trsf( sg_ik, rij, rik, es_kij, des_kij )

!                    //   zeta function
                     zt_ij = zt_ij + fc_ik * om_ik * g_kij * es_kij

!                 /*   loop of replicated boxes   */
                  end do
                  end do
                  end do

!                 //   loop of third atom
                  end do

!                 //   function b
                  bo_ij = ch_ij &
     &               * ( 1.d0 + (ga_ij*zt_ij)**n_ij )**(-0.5d0/n_ij)

!              //   case n_ij
               end if

!              //   functions
               ea_ij = a_ij * exp( - la_ij * rij )
               eb_ij = b_ij * exp( - mu_ij * rij )

!              //   potential
               pot(m) = pot(m) + 0.5d0*fc_ij * ( ea_ij - bo_ij * eb_ij )

!              //   force
               f1 = + dfc_ij * ( ea_ij - bo_ij * eb_ij ) &
     &              - fc_ij * ( la_ij * ea_ij - mu_ij * bo_ij * eb_ij )

!              //   force
               fxij = - f1 * xij * rij_inv
               fyij = - f1 * yij * rij_inv
               fzij = - f1 * zij * rij_inv

!              //   force
               fx(i,m) = fx(i,m) + 0.5d0*fxij
               fy(i,m) = fy(i,m) + 0.5d0*fyij
               fz(i,m) = fz(i,m) + 0.5d0*fzij
               fx(j,m) = fx(j,m) - 0.5d0*fxij
               fy(j,m) = fy(j,m) - 0.5d0*fyij
               fz(j,m) = fz(j,m) - 0.5d0*fzij

!              //   virial
               vir(1,1) = vir(1,1) + 0.5d0*fxij*xij
               vir(1,2) = vir(1,2) + 0.5d0*fxij*yij
               vir(1,3) = vir(1,3) + 0.5d0*fxij*zij
               vir(2,1) = vir(2,1) + 0.5d0*fyij*xij
               vir(2,2) = vir(2,2) + 0.5d0*fyij*yij
               vir(2,3) = vir(2,3) + 0.5d0*fyij*zij
               vir(3,1) = vir(3,1) + 0.5d0*fzij*xij
               vir(3,2) = vir(3,2) + 0.5d0*fzij*yij
               vir(3,3) = vir(3,3) + 0.5d0*fzij*zij

!              //   case n_ij = 0
               if ( abs(n_ij) .lt. tiny ) then

!                 //   no three-body term
                  cycle

!              //   case n_ij > 0
               else

!                 //   q function
                  call q_trsf &
     &               ( fc_ij, bo_ij, eb_ij, ga_ij, zt_ij, n_ij, q_ij )

!                 //   loop of third atom
                  do k = 1, natom

!                 //   atomic kind
                  lk = ikind(k)

!                 /*   loop of replicated boxes   */
                  do kx = 0, nbox_trsf(1)-1
                  do ky = 0, nbox_trsf(2)-1
                  do kz = 0, nbox_trsf(3)-1

!                    /*   square of box index   */
                     k2 = kx*kx + ky*ky + kz*kz

!                    /*   skip same atom   */
                     if ( ( k2 .eq. 0 ) .and. ( k .eq. i ) ) cycle

!                    /*   square of box index   */
                     jk2 = (jx-kx)*(jx-kx) + (jy-ky)*(jy-ky) &
     &                   + (jz-kz)*(jz-kz)

!                    /*   skip same atom   */
                     if ( ( jk2 .eq. 0 ) .and. ( k .eq. j ) ) cycle

!                    /*   interatomic distance in same box   */
                     xik = x(i,m) - x(k,m)
                     yik = y(i,m) - y(k,m)
                     zik = z(i,m) - z(k,m)

!                    /*   distance in different box  */
                     xik = xik - box(1,1)*kx - box(1,2)*ky - box(1,3)*kz
                     yik = yik - box(2,1)*kx - box(2,2)*ky - box(2,3)*kz
                     zik = zik - box(3,1)*kx - box(3,2)*ky - box(3,3)*kz

!                    /*   vector in big box   */
                     aik = bigboxinv(1,1)*xik + bigboxinv(1,2)*yik &
     &                   + bigboxinv(1,3)*zik
                     bik = bigboxinv(2,1)*xik + bigboxinv(2,2)*yik &
     &                   + bigboxinv(2,3)*zik
                     cik = bigboxinv(3,1)*xik + bigboxinv(3,2)*yik &
     &                   + bigboxinv(3,3)*zik

!                    /*   apply periodic boundary in big box   */
                     aik = aik - nint(aik)
                     bik = bik - nint(bik)
                     cik = cik - nint(cik)

!                    /*   distance of nearest i and k   */
                     xik = bigbox(1,1)*aik + bigbox(1,2)*bik &
     &                   + bigbox(1,3)*cik
                     yik = bigbox(2,1)*aik + bigbox(2,2)*bik &
     &                   + bigbox(2,3)*cik
                     zik = bigbox(3,1)*aik + bigbox(3,2)*bik &
     &                   + bigbox(3,3)*cik

!                    /*    interatomic distance squared   */
                     rik2 = xik*xik + yik*yik + zik*zik

!                    //   tersoff parameters between i and j
                     s_ik = s_trsf(li,lk)

!                    //   cut off
                     if ( rik2 .gt. s_ik*s_ik ) cycle

!                    //   distance
                     rik = sqrt(rik2)

!                    //   inverse of distance
                     rik_inv = 1.d0 / rik

!                    //   tersoff parameters between i and j
                     r_ik = r_trsf(li,lk)

!                    //   function fc and its gradient
                     call fc_trsf( rik, s_ik, r_ik, pi, fc_ik, dfc_ik )

!                    //   cosine of k-i-j angle
                     cos_kij = ( xij*xik + yij*yik + zij*zik ) &
     &                       / ( rij*rik )

!                    //   g function
                     call g_trsf &
     &                  ( cos_kij, c_ij, d_ij, h_ij, g_kij, dg_kij )

!                    //   omega constant
                     om_ik = omega_trsf(li,lk)

!                    //   sigma constant
                     sg_ik = sigma_trsf(li,lk)

!                    //   e function
                     call e_trsf( sg_ik, rij, rik, es_kij, des_kij )

!                    //   constant
                     qo = q_ij * om_ik

!                    //   force
                     f2 = qo * dfc_ik * g_kij * es_kij
                     f3 = qo * fc_ik * dg_kij * es_kij * rik_inv
                     f4 = qo * fc_ik * dg_kij * es_kij * rij_inv
                     f5 = qo * fc_ik * dg_kij * es_kij * cos_kij*rik_inv
                     f6 = qo * fc_ik * dg_kij * es_kij * cos_kij*rij_inv
                     f7 = qo * fc_ik * g_kij * des_kij

!                    //   force
                     fxik = + (-f2+f5+f7) * xik * rik_inv &
     &                      - f3 * xij * rij_inv
                     fyik = + (-f2+f5+f7) * yik * rik_inv &
     &                      - f3 * yij * rij_inv
                     fzik = + (-f2+f5+f7) * zik * rik_inv &
     &                      - f3 * zij * rij_inv

!                    //   force
                     fxij = - f4 * xik * rik_inv &
     &                      + (f6-f7) * xij * rij_inv
                     fyij = - f4 * yik * rik_inv &
     &                      + (f6-f7) * yij * rij_inv
                     fzij = - f4 * zik * rik_inv &
     &                      + (f6-f7) * zij * rij_inv

!                    //   force
                     fx(i,m) = fx(i,m) + 0.5d0*(fxik + fxij)
                     fy(i,m) = fy(i,m) + 0.5d0*(fyik + fyij)
                     fz(i,m) = fz(i,m) + 0.5d0*(fzik + fzij)
                     fx(j,m) = fx(j,m) - 0.5d0*fxij
                     fy(j,m) = fy(j,m) - 0.5d0*fyij
                     fz(j,m) = fz(j,m) - 0.5d0*fzij
                     fx(k,m) = fx(k,m) - 0.5d0*fxik
                     fy(k,m) = fy(k,m) - 0.5d0*fyik
                     fz(k,m) = fz(k,m) - 0.5d0*fzik

!                    //   virial
                     vir(1,1) = vir(1,1) + 0.5d0*(fxik*xik + fxij*xij)
                     vir(1,2) = vir(1,2) + 0.5d0*(fxik*yik + fxij*yij)
                     vir(1,3) = vir(1,3) + 0.5d0*(fxik*zik + fxij*zij)
                     vir(2,1) = vir(2,1) + 0.5d0*(fyik*xik + fyij*xij)
                     vir(2,2) = vir(2,2) + 0.5d0*(fyik*yik + fyij*yij)
                     vir(2,3) = vir(2,3) + 0.5d0*(fyik*zik + fyij*zij)
                     vir(3,1) = vir(3,1) + 0.5d0*(fzik*xik + fzij*xij)
                     vir(3,2) = vir(3,2) + 0.5d0*(fzik*yik + fzij*yij)
                     vir(3,3) = vir(3,3) + 0.5d0*(fzik*zik + fzij*zij)

!                 /*   loop of replicated boxes   */
                  end do
                  end do
                  end do

!                 //   loop of third atom
                  end do

!              //   case n_ij
               end if

!           /*   loop of replicated boxes   */
            end do
            end do
            end do

!           //   loop of atom pairs
            end do
            end do

!        //   loop of beads
         end do

!     /*   boundary condition   */
      end if

      return
      end





!***********************************************************************
      subroutine fc_trsf( rij, s_ij, r_ij, pi, fc_ij, dfc_ij )
!***********************************************************************

!     //   reset variables
      implicit none

!     //   real numbers
      real(8) :: fc_ij, dfc_ij, rij, s_ij, r_ij, pi

!     //   range: long
      if      ( rij .gt. s_ij ) then

!        //   function
         fc_ij  = 0.d0

!        //   gradient
         dfc_ij = 0.d0

!     //   range: middle
      else if ( rij .gt. r_ij ) then

!        //   function
         fc_ij  = 0.5d0 + 0.5d0 &
     &          * cos( pi * ( rij - r_ij ) / ( s_ij - r_ij ) )

!        //   gradient
         dfc_ij = - 0.5d0 * pi  / ( s_ij - r_ij ) &
     &            * sin( pi * ( rij - r_ij ) / ( s_ij - r_ij ) )

!     //   range: short
      else

!        //   function
         fc_ij  = 1.d0

!        //   gradient
         dfc_ij = 0.d0

!     //   range
      end if

      return
      end





!***********************************************************************
      subroutine g_trsf( cos_kij, c_ij, d_ij, h_ij, g_kij, dg_kij )
!***********************************************************************

!     //   reset variables
      implicit none

!     //   real numbers
      real(8) :: cos_kij, c_ij, d_ij, h_ij, g_kij, dg_kij, a

!     //   constant
      a = 1.d0 / ( d_ij*d_ij + (h_ij-cos_kij)*(h_ij-cos_kij) )

!     //   function
      g_kij = 1.d0 + c_ij*c_ij/(d_ij*d_ij) - a*c_ij*c_ij

!     //   gradient
      dg_kij = 2.d0 * c_ij*c_ij * a*a * (cos_kij-h_ij)

      return
      end





!***********************************************************************
      subroutine q_trsf( fc_ij, bo_ij, eb_ij, ga_ij, zt_ij, n_ij, q_ij )
!***********************************************************************

!     //   reset variables
      implicit none

!     //   real numbers
      real(8) :: fc_ij, bo_ij, eb_ij, ga_ij, zt_ij, n_ij, q_ij, a

!     //   real numbers
      real(8) :: tiny = 1.d-8

!     //   constant
      a = ( ga_ij * zt_ij )**n_ij

!     //   q function
      if ( abs(zt_ij) .gt. tiny ) then
         q_ij = 0.5d0 * fc_ij * eb_ij * bo_ij / ( 1.d0 + a ) * a / zt_ij
      else
         q_ij = 0.d0
      end if

      return
      end





!***********************************************************************
      subroutine e_trsf( sg_ik, rij, rik, es_kij, des_kij )
!***********************************************************************

!     //   reset variables
      implicit none

!     //   real numbers
      real(8) :: sg_ik, rij, rik, es_kij, des_kij

!     //   e function
      if ( sg_ik .eq. 0.d0 ) then
         es_kij = 1.d0
         des_kij = 0.d0
      else
         es_kij = exp( - sg_ik*(rij-rik)*(rij-rik)*(rij-rik) )
         des_kij = - 3.d0*sg_ik*(rij-rik)*(rij-rik) * es_kij
      end if

      return
      end





!***********************************************************************
      subroutine force_tersoff_makelist
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, au_length, natom, nbead, iounit

      use mm_variables, only : &
     &   x_list, y_list, z_list, rcut_list, rcut2_list, &
     &   dmax_list, n_list, j_list, nmax_list

      use tersoff_variables, only : &
     &   rcut_trsf, skin_trsf

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize variables
      implicit none

!     //   visit flag
      integer, save :: iset = 0

!     //   integers
      integer :: i, j, k, l, ierr

!     //   real variables
      real(8) :: dx, dy, dz, d2, dmax, d2max, rx, ry, rz, r2

!     //   real variables
      real(8) :: bohr2ang = au_length * 1.d+10

!     //   real variables
      real(8) :: skin_trsf_default = 2.d0

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

!     //   only first visit of this routine
      if ( iset .eq. 0 ) then

!        //   open file
         open ( iounit, file = 'tersoff.dat' )

!        /*   tag   */
         call search_tag ( '<skin_tersoff>', 14, iounit, ierr )

!        /*   cut off distance   */
         read( iounit, *, iostat=ierr ) skin_trsf

!        //   close file
         close( iounit )

!        //   default value
         if ( ierr .ne. 0 ) skin_trsf = skin_trsf_default

!        //   angstrom to bohr
         skin_trsf = skin_trsf / bohr2ang

!        //   list cutoff radius
         rcut_list = rcut_trsf + skin_trsf

!        //   list cutoff radius squared
         rcut2_list = rcut_list * rcut_list

!        //   maximum deviation allowed without updating neighbor list
         dmax_list = 0.5d0 * skin_trsf

!        //   memory allocation
         if ( .not. allocated(x_list) ) allocate( x_list(natom,nbead) )
         if ( .not. allocated(y_list) ) allocate( y_list(natom,nbead) )
         if ( .not. allocated(z_list) ) allocate( z_list(natom,nbead) )

!        //   setup end
         iset = 1

!-----------------------------------------------------------------------
!     /*   deviation from last update of neighbor list                */
!-----------------------------------------------------------------------

!     //   from second visit to this routine
      else

!        //   initialize maximum deviation squared
         d2max = 0.d0

!        //   loop of beads and atoms
         do k = 1, nbead
         do i = 1, natom

!           //   deviation
            dx = x(i,k) - x_list(i,k)
            dy = y(i,k) - y_list(i,k)
            dz = z(i,k) - z_list(i,k)

!           //   apply boundary condition
            call pbc_atom( dx, dy, dz )

!           //   deviation squared
            d2 = dx*dx + dy*dy + dz*dz

!           //   maximum deviation squared
            d2max = max( d2, d2max )

!        //   loop of beads and atoms
         end do
         end do

!        //   maximum deviation
         dmax = sqrt( d2max )

!        //   if maximum deviation is small, skip neighbor list update
         if ( dmax .lt. dmax_list ) return

!     //   end of if statement
      end if

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( allocated( n_list ) ) deallocate( n_list )
      if ( .not. allocated(n_list) ) allocate( n_list(natom,nbead) )

!-----------------------------------------------------------------------
!     /*   number of atoms in neighbor list                           */
!-----------------------------------------------------------------------

!     //   clear
      n_list(:,:) = 0

!     //   loop of beads
      do k = 1, nbead

!        //   loop of atom i
         do i = 1, natom

!           //   counter
            l = 0

!           //  loop of atom j
            do j = 1, natom

!              //   same atom
               if ( i .eq. j ) cycle

!              //   separation of atoms i and j
               rx = x(j,k) - x(i,k)
               ry = y(j,k) - y(i,k)
               rz = z(j,k) - z(i,k)

!              //   apply boundary condition
               call pbc_atom( rx, ry, rz )

!              //   distance of atoms i and j squared
               r2 = rx*rx + ry*ry + rz*rz

!              //   if distance is smaller than list cutoff radius
               if ( r2 .lt. rcut2_list ) then

!                 //   update counter
                  l = l + 1

!              //   end of if statement
               end if

!           //   loop of atom j
            end do

!           //   number of j atoms in neighbor list for atom i, bead k
            n_list(i,k) = l

!        //   loop of atom i
         end do

!     //   loop of beads
      end do

!-----------------------------------------------------------------------
!     /*   maximum number of atoms                                    */
!-----------------------------------------------------------------------

!     //   counter
      nmax_list = 0

!     //   loop of beads
      do k = 1, nbead

!        //   loop of atom i
         do i = 1, natom

!           //   maximum number of j atoms in neighbor list
            nmax_list = max( n_list(i,k), nmax_list )

!        //   loop of atom i
         end do

!     //   loop of beads
      end do

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( allocated( j_list ) ) deallocate( j_list )
      allocate( j_list(nmax_list,natom,nbead) )

!-----------------------------------------------------------------------
!     /*   atoms in neighbor list                                     */
!-----------------------------------------------------------------------

!     //   clear
      j_list(:,:,:) = 0

!     //   loop of beads
      do k = 1, nbead

!        //   loop of atom i
         do i = 1, natom

!           //   counter
            l = 0

!           //  loop of atom j
            do j = 1, natom

!              //   same atom
               if ( i .eq. j ) cycle

!              //   separation of atoms i and j
               rx = x(j,k) - x(i,k)
               ry = y(j,k) - y(i,k)
               rz = z(j,k) - z(i,k)

!              //   apply boundary condition
               call pbc_atom( rx, ry, rz )

!              //   distance of atoms i and j squared
               r2 = rx*rx + ry*ry + rz*rz

!              //   if distance is smaller than list cutoff radius
               if ( r2 .lt. rcut2_list ) then

!                 //   update counter
                  l = l + 1

!                 //   list of j atoms for atom i, bead k
                  j_list(l,i,k) = j

               end if

!           //  loop of atom j
            end do

!        //   loop of atom i
         end do

!     //   loop of beads
      end do

!-----------------------------------------------------------------------
!     /*   update neighbor list                                       */
!-----------------------------------------------------------------------

      x_list(:,:) = x(:,:)
      y_list(:,:) = y(:,:)
      z_list(:,:) = z(:,:)

      return
      end

