!///////////////////////////////////////////////////////////////////////
!
!      Authors:         S. Ruiz-Barragan, K. Ishimura, M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     PIMD-SMASH interface
!
!///////////////////////////////////////////////////////////////////////

!#######################################################################
#ifdef smash
!#######################################################################





!***********************************************************************
      subroutine calcmaxgrad( egradmax, egradrms, egrad, natom3 )
!***********************************************************************
!=======================================================================
!
!     This routine computes maximum and root mean square gradient
!     values and save the gradient for PIMD
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use smash_variables, only : grad_s

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   reset   */
      implicit none

!     /*   number of dimensions   */
      integer, intent(in) :: natom3

!     /*   gradients   */
      real(8), intent(in) :: egrad(3,natom3/3)

!     /*   max and rms of gradient   */
      real(8), intent(out) :: egradmax, egradrms

!     /*   root mean square   */
      real(8) :: ddot

!     /*   old gradient   */
      real(8) :: egrad_old(natom3)

!     /*   integers   */
      integer :: ipos, idamax, i, j

!-----------------------------------------------------------------------
!     /*   start calculation                                          */
!-----------------------------------------------------------------------

!     /*   counter   */
      ipos = 0

!     /*   loop of atoms   */
      do i = 1, natom3/3

!     /*   loop of xyz   */
      do j = 1, 3

!       /*   counter   */
        ipos = ipos + 1

!       /*   save old gradient   */
        egrad_old(ipos) = egrad(j,i)

!       /*   gradient   */
        grad_s(j,i) = egrad(j,i)

!     /*   loop of xyz   */
      end do

!     /*   loop of atoms   */
      end do

!     /*   find maximum   */
      ipos = idamax(natom3,egrad_old,1)

!     /*   max gradient   */
      egradmax = abs(egrad_old(ipos))

!     /*   rms gradient   */
      egradrms = ddot(natom3,egrad_old,1,egrad_old,1)
      egradrms = sqrt(egradrms/natom3)

      return

!***********************************************************************
      end subroutine calcmaxgrad
!***********************************************************************





!***********************************************************************
      subroutine savemo( cmo, cmob, nproc, myrank, mpi_comm )
!***********************************************************************
!=======================================================================
!
!     This subroutine saves the orbitals of the last step for PIMD
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

!     /*   from pimd   */
      use common_variables, only : istep

!     /*   from smash   */
      use modbasis, only : nao
      use modjob, only : scftype

!     /*   from pimd-smash interface   */
      use smash_variables, only : cmo_s, cmob_s, nbead_s

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   reset   */
      implicit none

!     /*   integers, but these values are not used   */
      integer, intent(in) :: nproc, myrank, mpi_comm

!     /*   alpha spin orbitals   */
      real(8), intent(out) :: cmo(nao,nao)

!     /*   beta spin orbitals   */
      real(8), intent(out) :: cmob(nao,nao)

!     /*   integers   */
      integer, save :: ibead_s = 1, istep_s = -1

!----------------------------------------------------------------------
!     /*   cycle number                                              */
!----------------------------------------------------------------------

      if ( istep_s .eq. istep ) then
         ibead_s = ibead_s + 1
         if ( ibead_s .gt. nbead_s ) ibead_s = 1
      else
         ibead_s = 1
         istep_s = istep
      end if

!-----------------------------------------------------------------------
!     /*   copy alpha orbitals                                        */
!-----------------------------------------------------------------------

      cmo(:,:) = cmo_s(:,:,0,ibead_s)

!-----------------------------------------------------------------------
!     /*   copy beta orbitals                                         */
!-----------------------------------------------------------------------

      if ( scftype == 'UHF' ) then
         cmob(:,:) = cmob_s(:,:,0,ibead_s)
      end if

      call readwritecharges( 1 )

!***********************************************************************
      end subroutine savemo
!***********************************************************************


!***********************************************************************
      subroutine readwritecharges( ioption )
!***********************************************************************

#ifdef pbc

      use modpbc, only : znuc2
      use smash_variables, only : natom_smash, znuc2_s

      implicit none
      integer :: i, ioption

      if ( ioption .eq. 0 ) then
         do i = 1, natom_smash
            znuc2_s(i) = znuc2(i)
         end do
      end if

      if ( ioption .eq. 1 ) then
         do i = 1, natom_smash
            znuc2(i) = znuc2_s(i)
         end do
      end if

#else

      implicit none
      integer :: i, ioption

#endif

      return
      end


!***********************************************************************
      subroutine writeeigenvector( cmo, eigen )
!***********************************************************************
!=======================================================================
!
!     This subroutine saves the LCAO coefficients (cmo).
!
!     While original SMASH prints cmo on the screen, this is not needed
!     in PIMD. Rather cmo is saved in the memory as initial guess for
!     the next step.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

!     /*   from pimd   */
      use common_variables, only : istep

!     /*   from smash   */
      use modbasis, only : nao
      use modmolecule, only : nmo
      use modjob, only : scftype

!     /*   from pimd-smash interface   */
      use smash_variables, only : cmo_s, cmob_s, ncmo_s, nbead_s

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   reset   */
      implicit none

!     /*   LCAO coefficients    */
      real(8), intent(in) :: cmo(nao,nao)

!     /*   orbital energies, but not used    */
      real(8), intent(in) :: eigen(nmo)

!     /*   option   */
      integer, save :: ibeta = 0

!     /*   integers   */
      integer, save :: ibead_s = 1, istep_s = -1

!----------------------------------------------------------------------
!     /*   cycle number                                              */
!----------------------------------------------------------------------

      if ( istep_s .eq. istep ) then
         ibead_s = ibead_s + 1
         if ( ibead_s .gt. nbead_s ) ibead_s = 1
      else
         ibead_s = 1
         istep_s = istep
      end if

!-----------------------------------------------------------------------
!     /*   save beta orbitals                                         */
!-----------------------------------------------------------------------

      if ( scftype == 'UHF' ) then

         if ( ibeta == 0 ) then

            if ( ncmo_s .eq. 0 ) then
               cmo_s(:,:,0,ibead_s) = cmo(:,:)
            else
               call smash_extrapolate( cmo, ibeta )
            end if

            ibeta = 1

         else if ( ibeta == 1 ) then

            if ( ncmo_s .eq. 0 ) then
               cmob_s(:,:,0,ibead_s) = cmo(:,:)
            else
               call smash_extrapolate( cmo, ibeta )
            end if

            ibeta = 0

         end if

!-----------------------------------------------------------------------
!     /*   save alpha orbitals                                        */
!-----------------------------------------------------------------------

      else

         if ( ncmo_s .eq. 0 ) then
            cmo_s(:,:,0,ibead_s) = cmo(:,:)
         else
            call smash_extrapolate( cmo, ibeta )
         end if

      end if

      call readwritecharges( 0 )

!***********************************************************************
      end subroutine writeeigenvector
!***********************************************************************





!**********************************************************************
      subroutine smash_extrapolate( cmo, ibeta )
!**********************************************************************

!----------------------------------------------------------------------
!     /*   shared variables                                          */
!----------------------------------------------------------------------

!     /*   from pimd   */
      use common_variables, only : istep, istep_start

!     /*   from pimd-smash interface   */
      use smash_variables, only : &
     &   cmo_s, cmob_s, coeff_s, ncmo_s, nbead_s

!     /*   from smash   */
      use modbasis, only : nao

!----------------------------------------------------------------------
!     /*   local variables                                           */
!----------------------------------------------------------------------

!     /*   reset   */
      implicit none

!     /*   LCAO coefficients    */
      real(8) :: cmo(nao,nao)

!     /*   alpha or beta   */
      integer :: ibeta

!     /*   integers   */
      integer :: i

!     /*   integers   */
      integer, save :: ibead_s = 1, istep_s = -1

!----------------------------------------------------------------------
!     /*   cycle number                                              */
!----------------------------------------------------------------------

      if ( istep_s .eq. istep ) then
         ibead_s = ibead_s + 1
         if ( ibead_s .gt. nbead_s ) ibead_s = 1
      else
         ibead_s = 1
         istep_s = istep
      end if

!----------------------------------------------------------------------
!     /*   initial settings                                          */
!----------------------------------------------------------------------

!     /*   on first visit to this routine   */
      if ( istep .eq. istep_start ) then

!        /*   alpha orbitals   */
         if ( ibeta .eq. 0 ) then

!           /*   save lcao coefficients   */
            do i = 0, ncmo_s
               cmo_s(:,:,i,ibead_s) = cmo(:,:)
            end do

!        /*   alpha orbitals   */
         end if

!        /*   beta orbitals   */
         if ( ibeta .eq. 1 ) then

!           /*   save lcao coefficients   */
            do i = 0, ncmo_s
               cmob_s(:,:,i,ibead_s) = cmo(:,:)
            end do

!        /*   beta orbitals   */
         end if

!     /*   on first visit to this routine   */
      end if

!----------------------------------------------------------------------
!     /*   alpha orbitals                                            */
!----------------------------------------------------------------------

      if ( ibeta .eq. 0 ) then

!-----------------------------------------------------------------------
!        /*   check phase of lcao coefficients                        */
!-----------------------------------------------------------------------

         call smash_checkphase &
     &      ( cmo, cmo_s, nao, ncmo_s, ibead_s, nbead_s )

!-----------------------------------------------------------------------
!        /*   copy old lcao coefficients                              */
!-----------------------------------------------------------------------

         do i = ncmo_s, 2, -1
            cmo_s(:,:,i,ibead_s) = cmo_s(:,:,i-1,ibead_s)
         end do

         cmo_s(:,:,1,ibead_s) = cmo(:,:)

!-----------------------------------------------------------------------
!        /*   guess for next step                                     */
!-----------------------------------------------------------------------

         cmo_s(:,:,0,ibead_s) = coeff_s(1) * cmo_s(:,:,1,ibead_s)

         do i = 2, ncmo_s
            cmo_s(:,:,0,ibead_s) = cmo_s(:,:,0,ibead_s) &
     &         + coeff_s(i) * cmo_s(:,:,i,ibead_s)
         end do

!----------------------------------------------------------------------
!     /*   alpha orbitals                                            */
!----------------------------------------------------------------------

      end if

!----------------------------------------------------------------------
!     /*   beta orbitals                                             */
!----------------------------------------------------------------------

      if ( ibeta .eq. 1 ) then

!-----------------------------------------------------------------------
!        /*   check phase of lcao coefficients                        */
!-----------------------------------------------------------------------

         call smash_checkphase &
     &      ( cmo, cmob_s, nao, ncmo_s, ibead_s, nbead_s )

!-----------------------------------------------------------------------
!        /*   guess for next step                                     */
!-----------------------------------------------------------------------

         do i = ncmo_s, 2, -1
            cmob_s(:,:,i,ibead_s) = cmob_s(:,:,i-1,ibead_s)
         end do

         cmob_s(:,:,1,ibead_s) = cmo(:,:)

         do i = ncmo_s, 1, -1
            cmob_s(:,:,0,ibead_s) = cmob_s(:,:,0,ibead_s) &
     &         + coeff_s(i) * cmob_s(:,:,i,ibead_s)
         end do

!----------------------------------------------------------------------
!     /*   beta orbitals                                             */
!----------------------------------------------------------------------

      end if

      return
      end





!***********************************************************************
      subroutine smash_checkphase &
     &   ( cmo, cmo_s, nao, ncmo_s, ibead_s, nbead_s )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   number of atomic orbitals   */
      integer :: nao

!     /*   order of gear method   */
      integer :: ncmo_s

!     /*   new lcao coefficients   */
      real(8) :: cmo(nao,nao)

!     /*   old lcao coefficients   */
      real(8) :: cmo_s(nao,nao,0:ncmo_s,nbead_s)

!     /*   phase factor   */
      integer, dimension(nao) :: iphase

!     /*   cycle number   */
      integer :: ibead_s, nbead_s

!     /*   integers   */
      integer :: i, j, k, ierr

!     /*   real numbers   */
      real(8) :: sum_1, sum_2, sum_3, sum_4

!-----------------------------------------------------------------------
!     /*    calculate phase change                                    */
!-----------------------------------------------------------------------

      i = ibead_s

      iphase(1) = + 1

      do k = 1, nao

!         sum_1 = 0.d0
!         sum_2 = 0.d0
!         sum_3 = 0.d0
!
!         do j = 1, nao
!
!            sum_1 = sum_1 + cmo(j,k)       * cmo(j,k)
!            sum_2 = sum_2 + cmo_s(j,k,0,i) * cmo_s(j,k,0,i)
!            sum_3 = sum_3 + cmo(j,k)       * cmo_s(j,k,0,i)
!
!         end do

         sum_1 = dot_product( cmo(:,k),       cmo(:,k)       )
         sum_2 = dot_product( cmo_s(:,k,0,i), cmo_s(:,k,0,i) )
         sum_3 = dot_product( cmo(:,k),       cmo_s(:,k,0,i) )

         sum_4 = sum_1 * sum_2

         if ( sum_4 .eq. 0.d0 ) then
            iphase(k) = + 1
            cycle
         end if

         if ( sum_3 .ge. 0.d0 ) then
            iphase(k) = + 1
         else
            iphase(k) = - 1
         end if

      end do

!-----------------------------------------------------------------------
!     /*   correct phase change                                       */
!-----------------------------------------------------------------------

      do k = 1, nao
         j = iphase(k)
         cmo(:,k) = cmo(:,k) * j
         cmo(:,k) = cmo(:,k) * j
         cmo(:,k) = cmo(:,k) * j
      end do

      return

!-----------------------------------------------------------------------
!     /*   check                                                      */
!-----------------------------------------------------------------------

      do k = 1, nao

!         sum_1 = 0.d0
!         sum_2 = 0.d0
!         sum_3 = 0.d0
!
!         do j = 1, nao
!
!            sum_1 = sum_1 + cmo(j,k)       * cmo(j,k)
!            sum_2 = sum_2 + cmo_s(j,k,0,i) * cmo_s(j,k,0,i)
!            sum_3 = sum_3 + cmo(j,k)       * cmo_s(j,k,0,i)
!
!         end do

         sum_1 = dot_product( cmo(:,k),       cmo(:,k)       )
         sum_2 = dot_product( cmo_s(:,k,0,i), cmo_s(:,k,0,i) )
         sum_3 = dot_product( cmo(:,k),       cmo_s(:,k,0,i) )

         sum_4 = sum_1 * sum_2

         if ( sum_4 .eq. 0.d0 ) cycle

         if ( sum_3 .ge. 0.d0 ) ierr = 0
         if ( sum_3 .lt. 0.d0 ) ierr = 1

      end do

      return
      end





!***********************************************************************
      subroutine readatom
!***********************************************************************

!=======================================================================
!
!     This subroutine passes information from PIMD to SMASH.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use modmolecule, only : numatomic, natom, znuc

      use common_variables, only : int_spec, ipotential, qmmm_embedding

      use smash_variables, only : natom_smash

#ifdef pbc
      use modpbc, only : natom_qm
#endif

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i

!-----------------------------------------------------------------------
!     /*    atomic numbers and nuclear charges                        */
!-----------------------------------------------------------------------

#ifdef pbc

      if      ( ( ipotential(1:5) .eq. 'QMMM ' ) .and. &
     &          ( qmmm_embedding(1:3) .eq. 'EE ' ) ) then

!        /*   number of atoms in smash: all atoms   */
         natom = natom_smash

!        /*    atomic numbers and nuclear charges   */
         call readatom_ee

      else if ( ( ipotential(1:5) .eq. 'QMMM ' ) .and. &
     &          ( qmmm_embedding(1:3) .eq. 'ME ' ) ) then

!        /*   number of atoms in smash: atoms in primary region   */
         natom = natom_smash

!        /*   number of atoms in smash: atoms in primary region   */
         natom_qm = natom_smash

!        /*    atomic numbers and nuclear charges   */
         call readatom_me

      else

!        /*   number of atoms in smash: all atoms   */
         natom = natom_smash

!        /*   number of atoms in smash: all atoms   */
         natom_qm = natom_smash

!        /*    atomic numbers   */

         do i = 1, natom
            numatomic(i) = int_spec(i)
         end do

!        /*    nuclear charges   */

         do i = 1, natom
            znuc(i) = dble(int_spec(i))
         end do

      end if

#else

      if      ( ( ipotential(1:5) .eq. 'QMMM ' ) .and. &
     &          ( qmmm_embedding(1:3) .eq. 'EE ' ) ) then

!        /*   number of atoms in smash: all atoms   */
         natom = natom_smash

!        /*    atomic numbers and nuclear charges   */
         call readatom_ee

      else if ( ( ipotential(1:5) .eq. 'QMMM ' ) .and. &
     &          ( qmmm_embedding(1:3) .eq. 'ME ' ) ) then

!        /*   number of atoms in smash: atoms in primary region   */
         natom = natom_smash

!        /*    atomic numbers and nuclear charges   */
         call readatom_me

      else

!        /*   number of atoms in smash: all atoms   */
         natom = natom_smash

!        /*    atomic numbers   */

         do i = 1, natom
            numatomic(i) = int_spec(i)
         end do

!        /*    nuclear charges   */

         do i = 1, natom
            znuc(i) = dble(int_spec(i))
         end do

      end if

#endif

      return
      end





!***********************************************************************
      subroutine readatom_ee
!***********************************************************************

!=======================================================================
!
!     This subroutine passes information from PIMD to SMASH.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &    int_spec, iboundary, iounit, volume, box, boxinv, &
     &    au_length

      use qmmm_variables, only : &
     &    layer, natom_p, natom_s, natom_l, int_spec_link, &
     &    ivar_qmmm

      use mm_variables, only : q, nbcp, i_bcp, j_bcp, factor_bcp, &
     &    lmax_ewald, nbox_ewald, alpha_ewald, rcut_ewald

      use modmolecule, only : numatomic, znuc

#ifdef pbc
      use modpbc, only : volume_smash, dopbc, natom_qm, natom_mm, &
     &    boxl_smash, boxinv_smash, znuc2, alpha_pbc_smash, &
     &    alpha2_pbc_smash, kmax_smash, nbox_smash, &
     &    nchrgpr_neg, ichrgpr_neg, jchrgpr_neg, fact_chrgpr_neg, &
     &    bigboxl_smash, bigboxinv_smash, ax_smash, ay_smash, &
     &    az_smash, bx_smash, by_smash, bz_smash, cx_smash, &
     &    cy_smash, cz_smash, iqm_index, iatom_qm, &
     &    jatom_qm, pi_smash, rcut_smash, dopbc_mulliken
#endif

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, l

!-----------------------------------------------------------------------
!     /*    number of atoms                                           */
!-----------------------------------------------------------------------

#ifdef pbc

!     /*    number of atoms in primary region   */
      natom_qm = natom_p + natom_l

!     /*    number of atoms in secondary region   */
      natom_mm = natom_s

#endif

!-----------------------------------------------------------------------
!     /*   bonded charge pairs in secondary region                    */
!-----------------------------------------------------------------------

#ifdef pbc

!     /*   counter   */
      l = 0

!     /*   loop of bonded charge pairs   */
      do i = 1, nbcp

!        /*   both atoms in secondary region   */
         if ( ( layer(i_bcp(i))(1:1) .eq. 'B' ) .and.  &
     &        ( layer(j_bcp(i))(1:1) .eq. 'B' ) ) then

!           /*   counter   */
            l = l + 1

!           /*   list of atom i   */
            ichrgpr_neg(l) = i_bcp(i)

!           /*   list of atom j   */
            jchrgpr_neg(l) = j_bcp(i)

!           /*   list of scaling factor   */
            fact_chrgpr_neg(l) = factor_bcp(i)

!        /*   both atoms in secondary region   */
         end if

!     /*   loop of bonded charge pairs   */
      end do

!     /*   number of bonded charge pairs in secondary region   */
      nchrgpr_neg = l

#endif

!-----------------------------------------------------------------------
!     /*   boundary condition                                         */
!-----------------------------------------------------------------------

#ifdef pbc

!     /*   boundary condition: free   */
      if ( iboundary .eq. 0 ) then

!        /*   pbc flag in smash   */
         dopbc = .false.

!     /*   boundary condition: periodic   */
      else

!        /*   pbc flag in smash   */
         dopbc = .true.

!     /*   boundary condition   */
      end if

#endif

!-----------------------------------------------------------------------
!     /*   ewald parameters                                           */
!-----------------------------------------------------------------------

#ifdef pbc

!     /*   periodic boundary condition   */
      if ( dopbc ) then

!        /*   volume in smash   */
         volume_smash = volume

!        /*   parameter alpha in smash   */
         alpha_pbc_smash = alpha_ewald

!        /*   square of alpha in smash   */
         alpha2_pbc_smash = alpha_pbc_smash * alpha_pbc_smash

!        /*   cut off parameter in smash   */
         rcut_smash = rcut_ewald

!        /*   box matrix   */

         do i = 1, 3
         do j = 1, 3
            boxl_smash(j,i) = box(j,i)
         end do
         end do

!        /*   inverse of box matrix   */

         do i = 1, 3
         do j = 1, 3
            boxinv_smash(j,i) = boxinv(j,i)
         end do
         end do

!        /*   maximum wavenumber   */

         do i = 1, 3
            kmax_smash(i) = lmax_ewald(i)
         end do

!        /*   number of periodic images   */

         do i = 1, 3
            nbox_smash(i) = nbox_ewald(i)
         end do

!        /*   big box with real and periodic images   */

         bigboxl_smash(:,1) = dble(nbox_smash(1)) * boxl_smash(:,1)
         bigboxl_smash(:,2) = dble(nbox_smash(2)) * boxl_smash(:,2)
         bigboxl_smash(:,3) = dble(nbox_smash(3)) * boxl_smash(:,3)

!        /*   inverse of big box matrix   */

         call inv3(bigboxl_smash,bigboxinv_smash)

!        /*   lattice vectors   */

         ax_smash = 2.d0 * pi_smash * boxinv_smash(1,1)
         ay_smash = 2.d0 * pi_smash * boxinv_smash(1,2)
         az_smash = 2.d0 * pi_smash * boxinv_smash(1,3)
         bx_smash = 2.d0 * pi_smash * boxinv_smash(2,1)
         by_smash = 2.d0 * pi_smash * boxinv_smash(2,2)
         bz_smash = 2.d0 * pi_smash * boxinv_smash(2,3)
         cx_smash = 2.d0 * pi_smash * boxinv_smash(3,1)
         cy_smash = 2.d0 * pi_smash * boxinv_smash(3,2)
         cz_smash = 2.d0 * pi_smash * boxinv_smash(3,3)

!        /*   charge of qm images: flexible   */
         if ( ivar_qmmm .eq. 1 ) then

!           /*   mulliken charge   */
            dopbc_mulliken = .true.

!        /*   charge of qm images: fixed   */
         else

!           /*   fixed charge   */
            dopbc_mulliken = .false.

!        /*   charge of qm images   */
         end if

!     /*   periodic boundary condition   */
      end if

#endif

!-----------------------------------------------------------------------
!     /*   atom indices                                               */
!-----------------------------------------------------------------------

#ifdef pbc

!     /*   counter   */
      j = 0

!     /*   loop of real atoms   */
      do i = 1, natom_p + natom_s

!        /*   primary layer   */
         if ( layer(i)(1:1) .eq. 'A' ) then

!           /*   counter   */
            j = j + 1

!           /*   qm/mm index in smash   */
            iqm_index(i) = 1

!           /*   qm atom index in smash   */
            jatom_qm(i) = j

!           /*   atomic number in smash   */
            numatomic(i) = int_spec(i)

!           /*   atom index in smash   */
            iatom_qm(j) = i

!        /*   secondary layer   */
         else

!           /*   qm/mm index in smash   */
            iqm_index(i) = 0

!           /*   qm atom index in smash   */
            jatom_qm(i) = 0

!           /*   atomic number in smash   */
            numatomic(i) = 0

!        /*   layer   */
         end if

!     /*   loop of real atoms   */
      end do

!     /*   loop of link atoms   */
      do l = 1, natom_l

!        /*   counter   */
         j = j + 1

!        /*   qm/mm index in smash   */
         iqm_index(natom_p+natom_s+l) = 1

!        /*   qm atom index in smash   */
         jatom_qm(j) = natom_p + natom_s + l

!        /*   atomic number in smash   */
         numatomic(natom_p+natom_s+l) = int_spec_link(l)

!        /*   qm atom index in smash   */
         iatom_qm(natom_p+natom_s+l) = j

!     /*   loop of link atoms   */
      end do

#else

!     /*   counter   */
      j = 0

!     /*   loop of real atoms   */
      do i = 1, natom_p + natom_s

!        /*   primary layer   */
         if ( layer(i)(1:1) .eq. 'A' ) then

!           /*   counter   */
            j = j + 1

!           /*   atomic number in smash   */
            numatomic(i) = int_spec(i)

!        /*   secondary layer   */
         else

!           /*   atomic number in smash   */
            numatomic(i) = 0

!        /*   layer   */
         end if

!     /*   loop of real atoms   */
      end do

!     /*   loop of link atoms   */
      do l = 1, natom_l

!        /*   counter   */
         j = j + 1

!        /*   atomic number in smash   */
         numatomic(natom_p+natom_s+l) = int_spec_link(l)

!     /*   loop of link atoms   */
      end do

#endif

!-----------------------------------------------------------------------
!     /*    nuclear charges                                           */
!-----------------------------------------------------------------------

#ifdef pbc

!     /*   loop of real atoms   */
      do i = 1, natom_p + natom_s

!        /*   primary layer   */
         if ( layer(i)(1:1) .eq. 'A' ) then

!           /*   nuclear charge   */
            znuc(i) = dble(int_spec(i))

!           /*   point charge for periodic boundary    */
            if ( dopbc ) znuc2(i) = q(i)

!        /*   secondary layer   */
         else

!           /*   point charge   */
            znuc(i) = q(i)

!           /*   point charge for periodic boundary    */
            if ( dopbc ) znuc2(i) = q(i)

!        /*   layer   */
         end if

!     /*   loop of real atoms   */
      end do

!     /*   loop of link atoms   */
      do l = 1, natom_l

!        /*   nuclear charge   */
         znuc(natom_p+natom_s+l) = dble(int_spec_link(l))

!        /*   point charge for periodic boundary    */
         if ( dopbc ) znuc2(i) = 0.d0

!     /*   loop of link atoms   */
      end do

#else

!     /*   loop of real atoms   */
      do i = 1, natom_p + natom_s

!        /*   primary layer   */
         if ( layer(i)(1:1) .eq. 'A' ) then

!           /*   nuclear charge   */
            znuc(i) = dble(int_spec(i))

!        /*   secondary layer   */
         else

!           /*   point charge   */
            znuc(i) = q(i)

!        /*   layer   */
         end if

!     /*   loop of real atoms   */
      end do

!     /*   loop of link atoms   */
      do l = 1, natom_l

!        /*   nuclear charge   */
         znuc(natom_p+natom_s+l) = dble(int_spec_link(l))

!     /*   loop of link atoms   */
      end do

#endif

      return
      end





!***********************************************************************
      subroutine readatom_me
!***********************************************************************

!=======================================================================
!
!     This subroutine passes information from PIMD to SMASH.
!
!=======================================================================

      use modmolecule, only : numatomic, znuc

      use common_variables, only : int_spec

      use qmmm_variables, only : &
     &   layer, natom_p, natom_s, natom_l, int_spec_link

#ifdef pbc
      use modpbc, only : iqm_index, iatom_qm, jatom_qm
#endif

      integer :: i, j

!-----------------------------------------------------------------------
!     /*   atom indices                                               */
!-----------------------------------------------------------------------

#ifdef pbc

      j = 0

      do i = 1, natom_p + natom_s

         if ( layer(i)(1:1) .eq. 'A' ) then

            j = j + 1

!           /*   qm/mm index in smash   */
            iqm_index(j) = 1

!           /*   qm atom index in smash   */
            jatom_qm(j) = j

!           /*   atomic number in smash   */
            numatomic(j) = int_spec(i)

!           /*   atom index in smash   */
            iatom_qm(j) = j

         end if

      end do

      do l = 1, natom_l

         j = j + 1

!        /*   qm/mm index in smash   */
         iqm_index(j) = 1

!        /*   qm atom index in smash   */
         jatom_qm(j) = j

!        /*   atomic number in smash   */
         numatomic(j) = int_spec_link(l)

!        /*   atom index in smash   */
         iatom_qm(j) = j

      end do

#else

      j = 0

      do i = 1, natom_p + natom_s

         if ( layer(i)(1:1) .eq. 'A' ) then

            j = j + 1

!           /*   atomic number in smash   */
            numatomic(j) = int_spec(i)

         end if

      end do

      do l = 1, natom_l

         j = j + 1

!        /*   atomic number in smash   */
         numatomic(j) = int_spec_link(l)

      end do

#endif

!-----------------------------------------------------------------------
!     /*    nuclear charges                                           */
!-----------------------------------------------------------------------

      j = 0

      do i = 1, natom_p + natom_s

         if ( layer(i)(1:1) .eq. 'A' ) then

            j = j + 1

            znuc(j) = dble(int_spec(i))

         end if

      end do

      do l = 1, natom_l

         j = j + 1

         znuc(j) = dble(int_spec_link(l))

      end do

      return
      end





!***********************************************************************
      subroutine calcrdipole &
     &   (dipmat,work,dmtrx,nproc,myrank,mpi_comm)
!***********************************************************************
!
! Driver of dipole moment calculation for closed-shell
!
!      use modparallel, only : master
      use modbasis, only : nao
      use modunit, only : todebye
      use modmolecule, only : natom, coord, znuc
!
! PIMD variables
!
      use smash_variables, only : dipx_s, dipy_s, dipz_s
!
      implicit none
      integer,intent(in) :: nproc, myrank, mpi_comm
      integer :: iatom
      real(8),parameter :: zero=0.0D+00
      real(8),intent(in) :: dmtrx((nao*(nao+1))/2)
      real(8),intent(out) :: dipmat((nao*(nao+1))/2,3), &
     &                       work((nao*(nao+1))/2,3)
      real(8) :: dipcenter(3), xdip, ydip, zdip, totaldip, tridot
      real(8) :: xdipplus, ydipplus, zdipplus, xdipminus, &
     &           ydipminus, zdipminus
!
! Nuclear part
!
      xdipplus= zero
      ydipplus= zero
      zdipplus= zero
!
      do iatom= 1,natom
        xdipplus= xdipplus+coord(1,iatom)*znuc(iatom)
        ydipplus= ydipplus+coord(2,iatom)*znuc(iatom)
        zdipplus= zdipplus+coord(3,iatom)*znuc(iatom)
      enddo
!
! Electron part
!
      dipcenter(:)= zero
!
      call calcmatdipole(dipmat,work,dipcenter,nproc,myrank,mpi_comm)
!
      xdipminus=-tridot(dmtrx,dipmat(1,1),nao)
      ydipminus=-tridot(dmtrx,dipmat(1,2),nao)
      zdipminus=-tridot(dmtrx,dipmat(1,3),nao)
!
! Sum Nuclear and Electron parts
!
      xdip=(xdipplus+xdipminus)*todebye
      ydip=(ydipplus+ydipminus)*todebye
      zdip=(zdipplus+zdipminus)*todebye
      totaldip= sqrt(xdip*xdip+ydip*ydip+zdip*zdip)
!
! PIMD variables (atomic unit)
!
      dipx_s = xdipplus+xdipminus
      dipy_s = ydipplus+ydipminus
      dipz_s = zdipplus+zdipminus
!
      return
      end





!***********************************************************************
      subroutine calcudipole &
     &   (dipmat,work,dmtrxa,dmtrxb,nproc,myrank,mpi_comm)
!***********************************************************************
!
! Driver of dipole moment calculation for open-shell
!
      use modbasis, only : nao
      use modunit, only : todebye
      use modmolecule, only : natom, coord, znuc
!
! PIMD variables
!
      use smash_variables, only : dipx_s, dipy_s, dipz_s
!
      implicit none
      integer,intent(in) :: nproc, myrank, mpi_comm
      integer :: iatom
      real(8),parameter :: zero=0.0D+00
      real(8),intent(in) :: dmtrxa((nao*(nao+1))/2), &
     &                      dmtrxb((nao*(nao+1))/2)
      real(8),intent(out) :: dipmat((nao*(nao+1))/2,3), &
     &                       work((nao*(nao+1))/2,3)
      real(8) :: dipcenter(3), xdip, ydip, zdip, totaldip, tridot
      real(8) :: xdipplus, ydipplus, zdipplus, xdipminus, &
     &           ydipminus, zdipminus
!
! Nuclear part
!
      xdipplus= zero
      ydipplus= zero
      zdipplus= zero
!
      do iatom= 1,natom
        xdipplus= xdipplus+coord(1,iatom)*znuc(iatom)
        ydipplus= ydipplus+coord(2,iatom)*znuc(iatom)
        zdipplus= zdipplus+coord(3,iatom)*znuc(iatom)
      enddo
!
! Electron part
!
      dipcenter(:)= zero
!
      call calcmatdipole(dipmat,work,dipcenter,nproc,myrank,mpi_comm)
!
      xdipminus=-tridot(dmtrxa,dipmat(1,1),nao) &
     &          -tridot(dmtrxb,dipmat(1,1),nao)
      ydipminus=-tridot(dmtrxa,dipmat(1,2),nao) &
     &          -tridot(dmtrxb,dipmat(1,2),nao)
      zdipminus=-tridot(dmtrxa,dipmat(1,3),nao) &
     &          -tridot(dmtrxb,dipmat(1,3),nao)
!
! Sum Nuclear and Electron parts
!
      xdip=(xdipplus+xdipminus)*todebye
      ydip=(ydipplus+ydipminus)*todebye
      zdip=(zdipplus+zdipminus)*todebye
      totaldip= sqrt(xdip*xdip+ydip*ydip+zdip*zdip)
!
! PIMD variables (atomic unit)
!
      dipx_s = xdipplus+xdipminus
      dipy_s = ydipplus+ydipminus
      dipz_s = zdipplus+zdipminus
!
      return
      end



!#######################################################################
#endif
!#######################################################################

