!///////////////////////////////////////////////////////////////////////
!
!      Authors:         S. Ruiz-Barragan, K. Ishimura, M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     SMASH force routine for PIMD
!
!///////////////////////////////////////////////////////////////////////

!#######################################################################
#ifdef smash
!#######################################################################



!***********************************************************************
      subroutine force_smash
!***********************************************************************
!=======================================================================
!
!     this subroutine runs SMASH via internal calls
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables from PIMD                                 */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, dipx, dipy, dipz, pot, vir, &
     &   natom, nbead, istep, mbox

!-----------------------------------------------------------------------
!     /*   shared variables from SMASH                                */
!-----------------------------------------------------------------------

      use modparallel, only : &
     &   master, nproc1, nproc2, myrank1, myrank2, mpi_comm1, mpi_comm2
      use modwarn, only : nwarn
      use modmemory, only : memusedmax
      use modjob, only : runtype, scftype
!cc      use modecp, only : flagecp
      use modiofile, only : input, icheck, version
      use modmolecule, only : coord
      use modenergy, only : escf, emp2
      use modguess, only : guess

!-----------------------------------------------------------------------
!     /*   shared variables from PIMD-SMASH interface                 */
!-----------------------------------------------------------------------

      use smash_variables, only : grad_s, dipx_s, dipy_s, dipz_s, &
     &   znuc2_s, natom_smash

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   reset   */
      implicit none

!     /*   integers   */
      integer :: ibead, i, j, m1, m2, m3

!     /*   real numbers   */
      real(8) :: xi, yi, zi

!     /*   flag for initial setting   */
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   visited: first time   */
      if ( iset .eq. 0 ) then

!        /*   number of atoms in SMASH   */
         natom_smash = natom

!        /*   memory allocatation of gradient   */
         if ( .not. allocated( grad_s ) ) &
     &      allocate( grad_s(3,natom_smash) )

!        /*   point charges of qm images   */
         if ( .not. allocated( znuc2_s ) ) &
     &      allocate( znuc2_s(natom_smash) )

!        /*   reset all SMASH energies to zero   */
         escf = 0.d0
         emp2 = 0.d0

!        /*   initialize mpi environment   */
         call setparallel

!        /*   open SMASH output file   */
         if ( master ) open( 99, file = 'smash.out' )

!c        /*   open SMASH input file   */
!         if ( master ) open( 98, file = 'smash.dat' )

!        /*   version of SMASH   */
         version = '2.2.0'

!        /*   print title of SMASH   */

         if ( master ) then
            write( 99, &
     &      '(" *******************************************",/, &
     &        "            SMASH Version ",a10/, &
     &        "           written by K. ISHIMURA",/, &
     &        " *******************************************",/)') &
     &      version
         end if

!        /*   SMASH timer routine   */
         call tstamp(0)

!        /*   initialize parallel calculation of SMASH   */
         call parallelinfo

!        /*   read input file and set details   */
         call setdetails(mpi_comm1)

!        /*   set indispensible data for PIMD-SMASH   */
         call force_smash_start

!c        /*   open SMASH output file   */
!        if ( master ) close( 98 )

!        /*   setup finished  */
         iset = 1

!     /*   visit: not first time   */
      else

!        /*   initial guess from previous step   */
         guess = 'PIMD'

!        /*   open SMASH output file   */
         if ( master ) open( 99, file = 'smash.out' )

!        /*   print title of SMASH   */

         if ( master ) then
            write( 99, &
     &      '(" *******************************************",/, &
     &        "            SMASH Version ",a10/, &
     &        "           written by K. ISHIMURA",/, &
     &        " *******************************************",/)') &
     &      version
         end if

!        /*   SMASH timer routine   */
         call tstamp(0)

!     /*   visit   */
      end if

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!     /*   coordinates according to periodic boundary condition       */
!-----------------------------------------------------------------------

      do i = 1, natom

         xi = x(i,ibead)
         yi = y(i,ibead)
         zi = z(i,ibead)

         m1 = mbox(1,i,ibead)
         m2 = mbox(2,i,ibead)
         m3 = mbox(3,i,ibead)

         call pbc_unfold ( xi, yi, zi, m1, m2, m3 )

         coord(1,i) = xi
         coord(2,i) = yi
         coord(3,i) = zi

      end do

!-----------------------------------------------------------------------
!     /*   print to SMASH output                                      */
!-----------------------------------------------------------------------

      if ( master ) then
         write( 99, '(a)' ) &
     &   "#####################################################"
         write( 99, '(a,i8,a,i8)' ) &
     &   "## STEP: ", istep, "  ## BEAD", ibead
         write( 99, '(a)' ) &
     &   "#####################################################"
      end if

!-----------------------------------------------------------------------
!     /*   run SMASH                                                  */
!-----------------------------------------------------------------------

      if ( scftype == 'RHF' ) then

         if ( runtype == 'ENERGY' ) then

            call calcrenergy &
     &         (nproc1, nproc2, myrank1, myrank2, mpi_comm1, mpi_comm2)

         else if (runtype == 'GRADIENT' ) then

            call calcrgradient &
     &         (nproc1, nproc2, myrank1, myrank2, mpi_comm1, mpi_comm2)

         else

           if (master) then
              write ( 99, &
     &           '(" Error! This program does not support runtype= ", &
     &           a16,".")') runtype
              call iabort
           end if

         end if

      else if ( scftype == 'UHF' ) then

         if ( runtype == 'ENERGY' ) then

            call calcuenergy &
     &         (nproc1, nproc2, myrank1, myrank2, mpi_comm1, mpi_comm2)

         else if ( runtype == 'GRADIENT' ) then

            call calcugradient &
     &         (nproc1, nproc2, myrank1, myrank2, mpi_comm1, mpi_comm2)

         end if

      else

         if (master) then
            write( 99, '(" Error! SCFtype=",a16," is not supported.")') &
     &      scftype
         end if

         call iabort

      end if

!-----------------------------------------------------------------------
!     /*   close input.dat and checkpoint files                       */
!-----------------------------------------------------------------------

!     /*   close and remove input.dat.number   */
!cc      if ( master ) close ( unit=input, status='DELETE' )

!     /*   close checkpoint files if is there   */
!cc      if ( master .and. (check /= '') ) close(unit=icheck)

!     /*   SMASH memory check   */
      call memcheck

!     /*   SMASH timer routine   */
      call tstamp(2)

      if (master) then
         write( 99, &
     &     '(" Used memory :",1x,i6," MB")') memusedmax/125000
         write( 99, &
     &     '(" Your calculation finished with",i3," warning(s).") &
     &     ') nwarn
      end if

!-----------------------------------------------------------------------
!     /*   SMASH potential                                            */
!-----------------------------------------------------------------------

      pot(ibead) = pot(ibead) + escf + emp2

!-----------------------------------------------------------------------
!     /*   SMASH potential gradient                                   */
!-----------------------------------------------------------------------

!     /*   change sign:  gradient -> force   */

      do i = 1, natom
         fx(i,ibead) = fx(i,ibead) - grad_s(1,i)
         fy(i,ibead) = fy(i,ibead) - grad_s(2,i)
         fz(i,ibead) = fz(i,ibead) - grad_s(3,i)
      end do

!-----------------------------------------------------------------------
!     /*   SMASH dipole moment - SCF level                            */
!-----------------------------------------------------------------------

      dipx(ibead) = dipx(ibead) + dipx_s
      dipy(ibead) = dipy(ibead) + dipy_s
      dipz(ibead) = dipz(ibead) + dipz_s

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom

         xi = x(i,j)
         yi = y(i,j)
         zi = z(i,j)

         call pbc_unfold &
     &     ( xi, yi, zi, mbox(1,i,j), mbox(2,i,j), mbox(3,i,j) )

         vir(1,1) = vir(1,1) + fx(i,j)*xi
         vir(1,2) = vir(1,2) + fx(i,j)*yi
         vir(1,3) = vir(1,3) + fx(i,j)*zi
         vir(2,1) = vir(2,1) + fy(i,j)*xi
         vir(2,2) = vir(2,2) + fy(i,j)*yi
         vir(2,3) = vir(2,3) + fy(i,j)*zi
         vir(3,1) = vir(3,1) + fz(i,j)*xi
         vir(3,2) = vir(3,2) + fz(i,j)*yi
         vir(3,3) = vir(3,3) + fz(i,j)*zi

      end do
      end do

!-----------------------------------------------------------------------
!     /*   close file                                                 */
!-----------------------------------------------------------------------

      if ( master ) close( 99 )

      return
      end





!***********************************************************************
      subroutine force_smash_start
!***********************************************************************
!=======================================================================
!
!     This subroutine sets SMASH parameters for PIMD
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

!     /*   from pimd   */
      use common_variables, only : &
     &   nprocs, np_cycle, iounit, nbead, smash_guess

!     /*   from smash    */
      use modprint, only : iprint
      use modunit, only : bohr
      use modjob, only : runtype
      use modbasis, only : nao
      use modjob, only : scftype

!     /*   from pimd-smash interface   */
      use smash_variables, only : &
     &   cmo_s, cmob_s, coeff_s, ncmo_s, nbead_s

!     /*   reset   */
      implicit none

!-----------------------------------------------------------------------
!     /*   memory allocation of LCAO coefficients                     */
!-----------------------------------------------------------------------

!     /*   method   */
      call read_char ( smash_guess, 20, '<smash_guess>', 13, iounit )

!-----------------------------------------------------------------------
!     /*   extrapolation coefficients                                 */
!-----------------------------------------------------------------------

!     /*   coefficients: pulay-4   */
      if      ( ( smash_guess(1:6) .eq. 'PULAY '   ) .or. &
     &          ( smash_guess(1:8) .eq. 'PULAY-4 ' ) ) then

!        /*   points used for extrapolation   */
         ncmo_s = 4

!        /*   memory allocation   */
         if ( .not. allocated( coeff_s ) ) &
     &      allocate( coeff_s(ncmo_s) )

         coeff_s(1) =  2.4d0
         coeff_s(2) = -1.2d0
         coeff_s(3) = -0.8d0
         coeff_s(4) =  0.6d0

!     /*   coefficients: KOLAFA-2   */
      else if ( smash_guess(1:9) .eq. 'KOLAFA-2 ' ) then

!        /*   points used for extrapolation   */
         ncmo_s = 2

!        /*   memory allocation   */
         if ( .not. allocated( coeff_s ) ) &
     &      allocate( coeff_s(ncmo_s) )

         coeff_s(1) =  2.0d0
         coeff_s(2) = -1.0d0

!     /*   coefficients: KOLAFA-4   */
      else if ( smash_guess(1:9) .eq. 'KOLAFA-4 ' ) then

!        /*   points used for extrapolation   */
         ncmo_s = 4

!        /*   memory allocation   */
         if ( .not. allocated( coeff_s ) ) &
     &      allocate( coeff_s(ncmo_s) )

!        /*   KOLAFA-4   */
         coeff_s(1) =  2.8d0
         coeff_s(2) = -2.8d0
         coeff_s(3) =  1.2d0
         coeff_s(4) = -0.2d0

!     /*   coefficients: KOLAFA-6   */
      else if ( ( smash_guess(1:7) .eq. 'KOLAFA '   ) .or. &
     &          ( smash_guess(1:9) .eq. 'KOLAFA-6 ' ) ) then

!        /*   points used for extrapolation   */
         ncmo_s = 6

!        /*   memory allocation   */
         if ( .not. allocated( coeff_s ) ) &
     &      allocate( coeff_s(ncmo_s) )

         coeff_s(1) = + 22.d0/ 7.d0
         coeff_s(2) = - 55.d0/14.d0
         coeff_s(3) = + 55.d0/21.d0
         coeff_s(4) = - 22.d0/21.d0
         coeff_s(5) = +  5.d0/21.d0
         coeff_s(6) = -  1.d0/42.d0

!     /*   coefficients: niklasson-6   */
      else if ( smash_guess(1:10) .eq. 'NIKLASSON ' ) then

!        /*   points used for extrapolation   */
         ncmo_s = 6

!        /*   memory allocation   */
         if ( .not. allocated( coeff_s ) ) &
     &      allocate( coeff_s(ncmo_s) )

         coeff_s(1) = + 30.d0/13.d0
         coeff_s(2) = -  3.d0/13.d0
         coeff_s(3) = - 28.d0/13.d0
         coeff_s(4) = -  3.d0/13.d0
         coeff_s(5) = + 30.d0/13.d0
         coeff_s(6) = - 13.d0/13.d0

!     /*   coefficients: previous step only   */
      else if ( smash_guess(1:9) .eq. 'PREVIOUS ' ) then

!        /*   points used for extrapolation   */
         ncmo_s = 0

!     /*   coefficients: otherwise   */
      else

!        /*   points used for extrapolation   */
         ncmo_s = 0

!     /*   coefficients   */
      end if

!-----------------------------------------------------------------------
!     /*   number of beads per process                                */
!-----------------------------------------------------------------------

      if ( nprocs .eq. 1 ) then
         nbead_s = nbead
      else
         nbead_s = np_cycle
      end if

!-----------------------------------------------------------------------
!     /*   memory allocation of lcao coefficients                     */
!-----------------------------------------------------------------------

!     /*   alpha orbitals   */
      if ( .not. allocated( cmo_s ) ) &
     &   allocate( cmo_s(nao,nao,0:ncmo_s,nbead_s) )

!     /*   beta orbitals   */
      if ( scftype == 'UHF' ) then
         if ( .not. allocated( cmob_s ) ) &
     &      allocate( cmob_s(nao,nao,0:ncmo_s,nbead_s) )
      end if

!-----------------------------------------------------------------------
!     /*   smash setup                                                */
!-----------------------------------------------------------------------

!     /*   print level   */
      iprint  = 2

!     /*   bohr or angstrom   */
      bohr    = .true.

!     /*   runtype   */
      runtype = 'GRADIENT'

!***********************************************************************
      end subroutine force_smash_start
!***********************************************************************



!#######################################################################
#else
!#######################################################################



!***********************************************************************
      subroutine force_smash
!***********************************************************************
!=======================================================================
!
!     this subroutine runs SMASH via system calls
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, fx, fy, fz, dipx, dipy, dipz, vir, &
     &   x, y, z, iounit, nbead, istep, mbox, &
     &   istep_start, natom, i_threads_smash, iexe_grad_smash, &
     &   iexe_dip_smash, smash_exe_command, iounit_smash

      use smash_variables, only : &
     &   natom_smash

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   reset   */
      implicit none

!     /*   integers   */
      integer :: ierr, jerr, ibead, i, j, match, m1, m2, m3

!     /*   real numbers   */
      real(8) :: xi, yi, zi, poti, fxi, fyi, fzi

!     /*   characters   */
      character(len=120) :: char_line, char_control

!     /*   atomic symbol or atomic number   */
      character(len=3) :: char_atom

!     /*   integer for SMASH   */
      integer, dimension(3) :: ipos

!     /*   integer for SMASH   */
      integer :: ichar_control

!     /*   control option for SMASH   */
      logical :: lcontrol

!     /*   mp2 option of SMASH   */
      logical :: lmp2

!     /*   flag for initialization   */
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   initial settings   */
      if ( iset .eq. 0 ) then

!        /*   number of atoms   */
         natom_smash = natom

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag ( '<smash_command>', 15, iounit, ierr )

!        /*   read a line   */
         read ( iounit, *, iostat=ierr ) smash_exe_command

!        /*   file close   */
         close ( iounit )

!        /*   look into default input   */
         if ( ierr .ne. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'input_default.dat' )

!           /*   search for tag    */
            call search_tag ( '<smash_command>', 15, iounit, ierr )

!           /*   read a line   */
            read ( iounit, *, iostat=ierr ) smash_exe_command

!           /*   file close   */
            close ( iounit )

!        /*   look into default input   */
         end if

!        /*   check error: confirm input   */
         if ( ierr .ne. 0 ) then
            write( 6, '(a)' ) 'Error in <smash_command>.'
         end if

!        /*   error termination   */
         call error_handling ( ierr, 'subroutine force_smash', 22 )

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag ( '<smash_grad>', 12, iounit, ierr )

!        /*   read a line   */
         read ( iounit, *, iostat=ierr ) iexe_grad_smash

!        /*   file close   */
         close ( iounit )

!        /*   look into default input   */
         if ( ierr .ne. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'input_default.dat' )

!           /*   search for tag    */
            call search_tag ( '<smash_grad>', 12, iounit, ierr )

!           /*   read a line   */
            read ( iounit, *, iostat=ierr ) iexe_grad_smash

!           /*   file close   */
            close ( iounit )

!        /*   look into default input   */
         end if

!        /*   check error: confirm input   */
         if ( ierr .ne. 0 ) then
            write( 6, '(a)' ) 'Error in <smash_grad>.'
         end if

!        /*   error termination   */
         call error_handling ( ierr, 'subroutine force_smash', 22 )

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag ( '<smash_dip>', 11, iounit, ierr )

!        /*   read a line   */
         read ( iounit, *, iostat=ierr ) iexe_dip_smash

!        /*   file close   */
         close ( iounit )

!        /*   look into default input   */
         if ( ierr .ne. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'input_default.dat' )

!           /*   search for tag    */
            call search_tag ( '<smash_dip>', 11, iounit, ierr )

!           /*   read a line   */
            read ( iounit, *, iostat=ierr ) iexe_dip_smash

!           /*   file close   */
            close ( iounit )

!        /*   look into default input   */
         end if

!        /*   check error: confirm input   */
         if ( ierr .ne. 0 ) then
            write( 6, '(a)' ) 'Error in <smash_dip>.'
         end if

!        /*   error termination   */
         call error_handling ( ierr, 'subroutine force_smash', 22 )

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag ( '<smash_threads>', 15, iounit, ierr )

!        /*   read a line   */
         read ( iounit, *, iostat=ierr ) i_threads_smash

!        /*   file close   */
         close ( iounit )

!        /*   look into default input   */
         if ( ierr .ne. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'input_default.dat' )

!           /*   search for tag    */
            call search_tag ( '<smash_threads>', 15, iounit, ierr )

!           /*   read a line   */
            read ( iounit, *, iostat=ierr ) i_threads_smash

!           /*   file close   */
            close ( iounit )

!        /*   look into default input   */
         end if

!        /*   check error: confirm input   */
         if ( ierr .ne. 0 ) then
            write( 6, '(a)' ) 'Error in <smash_threads>.'
         end if

!        /*   error termination   */
         call error_handling ( ierr, 'subroutine force_smash', 22 )

!        /*   set completed   */
         iset = 1

!     /*   initial settings   */
      end if

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!     /*   make SMASH input:  input.smash                             */
!-----------------------------------------------------------------------

!     /*   open SMASH prototype file   */
      open ( iounit, file = 'smash.dat'   )

!     /*   open SMASH input file   */
      open ( iounit_smash, file = 'input.smash' )

!     /*   control option for smash   */
      lcontrol = .true.

!     /*   do loop start   */
      do

!        /*   read a line   */
         read ( iounit, '(a80)', iostat=ierr ) char_line

!        /*   exit at the end of the line   */
         if ( ierr .ne. 0 ) exit

!        /*   new line: when "geom" is found, start printing   */
         if ( index(char_line(1:4),'geom') .ge. 1 ) then

!           /*   if there is no control option   */
            if (lcontrol) then

!              /*   guess: in the first step   */
               if ( istep .eq. istep_start ) then

!                 /*   guess is not used   */
                  char_control = &
     &               ' bohr=.true. check=check.smash'

!              /*   guess: in other steps   */
               else

!                 /*   use checkpoint file   */
                  char_control = &
     &               ' bohr=.true. check=check.smash guess=check'

!              /*   guess   */
               end if

!              /*   print line   */
               write( iounit_smash, '(a)' ) char_control

!           /*   if there is no control option   */
            end if

!           /*   print "geom" at the beginning of geometry   */
            write( iounit_smash, '(a)' ) 'geom'

!           /*   loop of atoms   */
            do i = 1, natom

!              /*   read atomic number   */
               read ( iounit, * ) char_atom

!              /*   apply the periodic boundary condition   */

               xi = x(i,ibead)
               yi = y(i,ibead)
               zi = z(i,ibead)

               m1 = mbox(1,i,ibead)
               m2 = mbox(2,i,ibead)
               m3 = mbox(3,i,ibead)

               call pbc_unfold ( xi, yi, zi, m1, m2, m3 )

!              /*   print atomic number and cartesian coordinates   */
               write( iounit_smash, '(a3,1x,3d24.16)' ) &
     &            char_atom, xi, yi, zi

!           /*   loop of atoms   */
            end do

!        /*   new line: when "job" is found restart printing   */
         else if ( index(char_line(1:3),'job') .ge. 1 ) then

!           /*   variable   */
            char_control = 'job'

!           /*   variable   */
            ichar_control = 3

!           /*   column position   */
            ipos(1) = index(char_line,' ')

!           /*   line   */
            lcontrol = ( ipos(1) .ne. 0 )

!           /*   define lmp2  */
            lmp2 = .false.

!           /*   read job line   */
            do while(lcontrol)

!              /*   set column position   */
               ipos(2) = &
     &            index(char_line(ipos(1)+1:120),' ') + ipos(1)

!              /*   set column position   */
               ipos(3) = &
     &            index(char_line(ipos(1)+1:ipos(2)),'=') + ipos(1) - 1

!              /*   if "runtype" is not found   */
               if ( char_line(ipos(1):ipos(3)) .ne. " runtype" ) then

!                 /*   variable   */
                  char_control = &
     &               char_control(1:ichar_control) // ' ' // &
     &               char_line(ipos(1)+1:ipos(2))

!                 /*   variable   */
                  ichar_control = ichar_control + ipos(2) - ipos(1)

!                 /*   no gradient calculation for mp2   */
                  if ( char_line(ipos(1):ipos(3)) .ne. " method" ) then
                  if ( char_line(ipos(3)+1:ipos(2)) .eq. "mp2" .or. &
     &                 char_line(ipos(3)+2:ipos(2)) .eq. "MP2" ) then
                     lmp2 = .true.
                  end if
                  end if

!              /*   if "runtype" is not found   */
               end if

!              /*   reset column   */
               ipos(1) = ipos(2)

!              /*   continue until "=" is not found   */
               lcontrol = ( index(char_line(ipos(3):120),'=') .ne. 0 )

!           /*   read job line   */
            end do

!           /*   no gradient calculation for mp2  */
            if (lmp2) iexe_grad_smash = 0

!           /*   runtype: with gradient   */
            if ( iexe_grad_smash .eq. 1 ) then

               char_control = char_control(1:ichar_control) &
     &            // ' ' // ' runtype=gradient'

!           /*   runtype: without gradient   */
            else

               char_control = char_control(1:ichar_control) &
     &            // ' ' // ' runtype=energy'

!           /*   runtype   */
            end if

!           /*   print options   */
            write( iounit_smash, '(a)') char_control(1:ichar_control+20)

!        /*   new line: with more control options   */
         else if ( index(char_line(1:8),'control') .ge. 1 ) then

!           /*   variable   */
            char_control = 'control'

!           /*   variable   */
            ichar_control = 7

!           /*   column   */
            ipos(1) = index(char_line,' ')

!           /*   variable   */
            lcontrol = ( ipos(1) .ne. 0 )

!           /*   read control line   */
            do while(lcontrol)

!              /*   column   */
               ipos(2) = &
     &            index(char_line(ipos(1)+1:120),' ') + ipos(1)

!              /*   column   */
               ipos(3) = &
     &            index(char_line(ipos(1)+1:ipos(2)),'=') + ipos(1) - 1

!              /*   add the control line   */
               if ( ( char_line(ipos(1):ipos(3)) .ne. " guess" ) .and. &
     &              ( char_line(ipos(1):ipos(3)) .ne. " check" ) .and. &
     &              ( char_line(ipos(1):ipos(3)) .ne. " bohr"  ) ) then

!                 /*   character   */
                  char_control = &
     &               char_control(1:ichar_control) // ' ' // &
     &               char_line(ipos(1)+1:ipos(2))

!                 /*   column   */
                  ichar_control = ichar_control + ipos(2) - ipos(1)

!              /*   add the control line   */
               end if

!              /*   stating point for the next cycle   */
               ipos(1) = ipos(2)

!              /*   continue until the end  */
               lcontrol = ( index(char_line(ipos(3):120),'=') .ne. 0 )

!           /*   read control line   */
            end do

!           /*   control options: first step   */
            if ( istep .eq. istep_start ) then

               char_control = char_control(1:ichar_control) // ' ' // &
     &         ' bohr=.true. check=check.smash'

!           /*   control options: other steps   */
            else

               char_control = char_control(1:ichar_control) // ' ' // &
     &         ' bohr=.true. check=check.smash guess=check'

!           /*   control options   */
            end if

!           /*   print line   */
            write( iounit_smash, '(a)' ) &
     &         char_control(1:ichar_control+55)

!        /*   new line: if not matched   */
         else

!           /*   write a copy of the line   */
            write( iounit_smash, '(a)' ) char_line

!        /*   new line   */
         end if

!     /*   do loop end   */
      end do

!     /*   close files   */
      close( iounit )
      close( iounit_smash )

!-----------------------------------------------------------------------
!     /*   make bash script                                           */
!-----------------------------------------------------------------------

!     /*   create a jobfile for SMASH calculation   */
      open ( iounit_smash, file = 'input.job.smash' )

!     /*   set the length of i_threads_smash  */
      write( char_line,'(i1)' ) int(log10(real(i_threads_smash))) + 1

!     /*   the number of threads   */
      char_control = "(i" // char_line(1:1) // ")"

!     /*   read it as characters   */
      write( char_line, char_control ) i_threads_smash

!     /*   write bash script   */
      write( iounit_smash, '(a)' ) '#!/bin/bash'
      write( iounit_smash, '(a)' ) 'ulimit -s unlimited'
      write( iounit_smash, '(a)' ) 'export OMP_STACKSIZE=1G'
      write( iounit_smash, '(a)' ) "export OMP_NUM_THREADS="//char_line

!     /*   current working directory   */
      call getcwd( char_line )

!     /*   change directory   */
      write( iounit_smash, '(a)' ) 'cd ' // char_line

!     /*   write SMASH command   */
      write( iounit_smash, '(a)' ) &
     &   trim(smash_exe_command) // ' < input.smash > output.smash'

!     /*   close files   */
      close( iounit_smash )

!-----------------------------------------------------------------------
!     /*   run SMASH                                                  */
!-----------------------------------------------------------------------

!      call system ( ' bash input.job.smash ' )

      call system ( &
     &   trim(smash_exe_command) // ' < input.smash > output.smash' )

!-----------------------------------------------------------------------
!     /*   read SMASH output:  potential                              */
!-----------------------------------------------------------------------

!     /*   open SMASH output file   */
      open ( iounit_smash, file = 'output.smash' )

!     /*   do loop start   */
      do

!        /*   read a line   */
         read ( iounit_smash, '(a80)', iostat=ierr ) char_line

!        /*   error handling   */
         if ( ierr .ne. 0 ) exit

!        /*   see if the line matches   */
         match = index( char_line, 'HF + MP2 Energy        = ' )
         match = match + index( char_line, ' Energy = ' )

!        /*   if matched   */
         if ( match .ge. 1 ) then

!           /*   find the position of "="   */
            ipos(1) = index(char_line,'=') + 2

!           /*   read the potential energy   */
            read ( char_line(ipos(1):90), *, iostat=jerr ) poti

!           /*   potential energy   */
            pot(ibead) = pot(ibead) + poti

         end if

!     /*   do loop end   */
      end do

!     /*   close file   */
      close( iounit_smash )

!     /*   check error: confirm output   */
      if ( jerr .ne. 0 ) then
         write( 6, '(a)' ) 'Error - while reading smash potential.'
      end if

!     /*   error termination   */
      call error_handling ( jerr, 'subroutine force_smash', 22 )

!-----------------------------------------------------------------------
!     /*   read SMASH output:  potential gradient                     */
!-----------------------------------------------------------------------

      if ( iexe_grad_smash .eq. 0 ) then

         do i = 1, natom
            fx(i,ibead) = fx(i,ibead) + 0.d0
            fy(i,ibead) = fy(i,ibead) + 0.d0
            fz(i,ibead) = fz(i,ibead) + 0.d0
         end do

      else

!        /*   open SMASH output file   */
         open ( iounit_smash, file = 'output.smash' )

!        /*   do loop start   */
         do

!        /*   read a line   */
         read ( iounit_smash, '(a80)', iostat=ierr ) char_line

!        /*   error handling   */
         if ( ierr .ne. 0 ) exit

!        /*   see if the line matches   */
         match = index( char_line, 'Gradient (Hartree/Bohr)' )

!        /*   if matched   */
         if ( match .ge. 1 ) then

!           /*   read two lines   */
            read ( iounit_smash, *, iostat=ierr )
            read ( iounit_smash, *, iostat=ierr )

!           /*   loop of atoms   */
            do i = 1, natom

!              /*   read gradient   */
               read ( iounit_smash, * , iostat=jerr ) &
     &            char_line, fxi, fyi, fzi

!              /*   change sign:  gradient -> force   */
               fx(i,ibead) = fx(i,ibead) - fxi
               fy(i,ibead) = fy(i,ibead) - fyi
               fz(i,ibead) = fz(i,ibead) - fzi

!           /*   loop of atoms   */
            end do

!           /*   exit from the do loop   */
            exit

         end if

!        /*  do loop end   */
         end do

!        /*   close file   */
         close( iounit_smash )

      end if

!     /*   check error: confirm output   */
      if ( jerr .ne. 0 ) then
         write( 6, '(a)' ) 'Error - while reading smash gradients.'
      end if

!     /*   error termination   */
      call error_handling ( jerr, 'subroutine force_smash', 22 )

!-----------------------------------------------------------------------
!     /*   read smash output:  dipole moment is NOT calculated.       */
!-----------------------------------------------------------------------

      dipx(ibead) = dipx(ibead) + 0.d0
      dipy(ibead) = dipy(ibead) + 0.d0
      dipz(ibead) = dipz(ibead) + 0.d0

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom

         xi = x(i,j)
         yi = y(i,j)
         zi = z(i,j)

         call pbc_unfold &
     &     ( xi, yi, zi, mbox(1,i,j), mbox(2,i,j), mbox(3,i,j) )

         vir(1,1) = vir(1,1) + fx(i,j)*xi
         vir(1,2) = vir(1,2) + fx(i,j)*yi
         vir(1,3) = vir(1,3) + fx(i,j)*zi
         vir(2,1) = vir(2,1) + fy(i,j)*xi
         vir(2,2) = vir(2,2) + fy(i,j)*yi
         vir(2,3) = vir(2,3) + fy(i,j)*zi
         vir(3,1) = vir(3,1) + fz(i,j)*xi
         vir(3,2) = vir(3,2) + fz(i,j)*yi
         vir(3,3) = vir(3,3) + fz(i,j)*zi

      end do
      end do

      return
      end

!#######################################################################
#endif
!#######################################################################
