!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Jan 23, 2025 by M. Shiga
!      Description:     quartic potential
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine force_quartic_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, pot, vir, dipx, natom, nbead, iounit, mbox, &
     &   nprocs, myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

      real(8) :: r1, r2, r3, r4, xi, yi, zi

      real(8), save :: const(5)

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   read user execution command                                */
!-----------------------------------------------------------------------

!     /*   first visit only   */
      if ( iset .eq. 0 ) then

!        /*   read command within eighty letters  */
         call read_realn_MPI( const, 5, '<quartic>', 9, iounit )

!        /*   set complete   */
         iset = 1

!     /*   first visit only   */
      end if

!-----------------------------------------------------------------------
!     /*   potential and force                                        */
!-----------------------------------------------------------------------

      do j = 1, nbead

         if ( mod( j-1, nprocs ) .ne. myrank ) cycle

         do i = 1, natom

            r1 = x(i,j)

            r2 = r1 * r1
            r3 = r2 * r1
            r4 = r3 * r1

            pot(j) = pot(j) + const(1)
            pot(j) = pot(j) + const(2) * r1
            pot(j) = pot(j) + const(3) * r2
            pot(j) = pot(j) + const(4) * r3
            pot(j) = pot(j) + const(5) * r4

            fx(i,j) = fx(i,j) - 1.d0 * const(2)
            fx(i,j) = fx(i,j) - 2.d0 * const(3) * r1
            fx(i,j) = fx(i,j) - 3.d0 * const(4) * r2
            fx(i,j) = fx(i,j) - 4.d0 * const(5) * r3

         end do

      end do

!-----------------------------------------------------------------------
!     /*   dipole                                                     */
!-----------------------------------------------------------------------

      do j = 1, nbead

         if ( mod( j-1, nprocs ) .ne. myrank ) cycle

         do i = 1, natom
            dipx(j) = dipx(j) + x(i,j)
         end do

      end do

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

      do j = 1, nbead

         if ( mod( j-1, nprocs ) .ne. myrank ) cycle

         do i = 1, natom

            xi = x(i,j)
            yi = y(i,j)
            zi = z(i,j)

            call pbc_unfold_MPI &
     &        ( xi, yi, zi, mbox(1,i,j), mbox(2,i,j), mbox(3,i,j) )

            vir(1,1) = vir(1,1) + fx(i,j)*xi

         end do

      end do

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx, natom, nbead )

!     /*   dipole   */
      call my_mpi_allreduce_real_1 ( dipx, nbead )

!     /*   virial   */
      call my_mpi_allreduce_real_2 ( vir, 3, 3 )

      return
      end

