!///////////////////////////////////////////////////////////////////////
!
!      Author:          B. Thomsen, M. Shiga
!      Last updated:    May 2, 2025 by B. Thomsen
!      Description:     energy and force from QE calculation
!
!///////////////////////////////////////////////////////////////////////

#ifdef qegit

subroutine force_qegit_MPI
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, &
     &   box, iounit, nbead, natom, iboundary, &
     &   myrank, myrank_main, nprocs_main, method, &
     &   iounit_null, np_beads, istep, ensemble, iounit_qe,mpi_comm_sub

!     /*   Modules and variables from Quantum Espresso                */
      use io_files, only : prefix

      USE cell_base,            ONLY : alat,at,bg,omega
      USE ions_base,            ONLY : nat, tau
      USE force_mod,            ONLY : force,sigma
      USE ener,                 ONLY : etot
      USE check_stop,           ONLY : check_stop_init, check_stop_now
      USE extrapolation,        ONLY : update_file, update_pot
      USE command_line_options, ONLY : nimage_, input_file_, npool_, nband_, ntg_, nyfft_
      USE mp_global,            ONLY : mp_startup
      USE read_input,           ONLY : read_input_file
      USE environment,          ONLY : environment_start
      USE cellmd,               ONLY : omega_old, at_old
      use io_global,            only : stdout
      USE input_parameters,     ONLY : outdir


!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      character(len=3) :: char_num

      real(8) :: stress_tensor(3,3)

      integer :: ibead, i

      integer, save :: iset = 0

      logical, save :: lmovecell = .true.

      include 'mpif.h'

!-----------------------------------------------------------------------
!     /*   initialize Quantum Espresso                                */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

        if ( iboundary .eq. 0 ) then
            write( 6, '(a)' ) &
     &   'Error - QE should only be used with PBC'
            write( 6, '(a)' ) &
     &   'Please give a boundary box or consider using another program'
            write( 6, '(a)' )
            call error_handling_MPI ( 1, 'subroutine force_qegit', 24 )
        end if

        stdout = iounit_qe
 
        call int3_to_char( myrank_main, char_num )
        open(iounit_qe, file = './' // char_num // '_qe.out')
        input_file_ = "qe.dat"
        call mp_startup(mpi_comm_sub, start_images=.TRUE.)
        call environment_start( 'PWSCF' )
        call read_input_file('PW', input_file_ )
        outdir = './' // char_num // '/'
        call iosys()
        call check_stop_init()
        call setup()
        call init_run()

        if (( method(1:11) .eq. 'TESTVIRIAL' )) then
            lmovecell = .true.
        else if (( ensemble(1:3) .eq. 'NVT' ) .or. ( ensemble(1:3) .eq. 'NVE' )) then
            lmovecell = .false.
        endif


        iset = 1
    endif


    stress_tensor(:,:) = 0

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!     /*   allocated bead only   */
      if ( mod(ibead-1,nprocs_main) .ne. myrank_main ) cycle

      at_old    = at
      omega_old = omega

!-----------------------------------------------------------------------
!     /*   Transfer geometry and box                                  */
!-----------------------------------------------------------------------
      at(:,:)  = box(:,:)/alat
      tau(1,:) = x(:,ibead)/alat
      tau(2,:) = y(:,ibead)/alat
      tau(3,:) = z(:,ibead)/alat

      if( lmovecell ) then
            call recips(at(1,1), at(1,2), at(1,3), bg(1,1), bg(1,2), bg(1,3))
            call volume(alat, at(1,1), at(1,2), at(1,3), omega)
            call scale_h()
            call reset_gvectors()
            call update_pot()
            call hinit1()
      else
            call update_pot()
            call hinit1()
      endif

!-----------------------------------------------------------------------
!     /*   Run calculation                                            */
!-----------------------------------------------------------------------

      call electrons()
      call forces()
      call stress(sigma)


!-----------------------------------------------------------------------
!     /*   Collect results                                            */
!-----------------------------------------------------------------------

!     /*   unit [Ry] -> [Hatree]   */
      pot(ibead) = etot * 0.5d0

!     /*   unit [Ry/a.u.] -> [Hatree/a.u.]   */
      do i = 1, nat
        fx(i,ibead) = force(1,i) * 0.5d0
        fy(i,ibead) = force(2,i) * 0.5d0
        fz(i,ibead) = force(3,i) * 0.5d0
      end do

!     /*   unit [Ry/a.u.] -> [Hatree/a.u.]   */
      stress_tensor(:,:) = stress_tensor(:,:) + sigma(:,:) * 0.5d0


!      /* Loop over beads */
       end do  

!     /*   potential   */
      call my_mpi_allreduce_real_1_main ( pot, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2_main ( fx, natom, nbead )
      call my_mpi_allreduce_real_2_main ( fy, natom, nbead )
      call my_mpi_allreduce_real_2_main ( fz, natom, nbead )

!     /*   stress   */
      call my_mpi_allreduce_real_2_main ( stress_tensor, 3, 3 )
      call force_qegit_updatevir_MPI(stress_tensor)

!      if (myrank .eq. 0) write(*,*) fx

!       call my_mpi_barrier()
!      call EXIT(1)

      end subroutine force_qegit_MPI


!***********************************************************************
      subroutine force_qegit_updatevir_MPI(stress_tensor)
!***********************************************************************

      use common_variables, only : vir, volume, nbead

      implicit none

      real(8) :: stress_tensor(3,3)
      integer :: j

      do j = 1, nbead

         vir(1,1) = vir(1,1) + stress_tensor(1,1) * volume / nbead
         vir(1,2) = vir(1,2) + stress_tensor(1,2) * volume / nbead
         vir(1,3) = vir(1,3) + stress_tensor(1,3) * volume / nbead
         vir(2,1) = vir(2,1) + stress_tensor(2,1) * volume / nbead
         vir(2,2) = vir(2,2) + stress_tensor(2,2) * volume / nbead
         vir(2,3) = vir(2,3) + stress_tensor(2,3) * volume / nbead
         vir(3,1) = vir(3,1) + stress_tensor(3,1) * volume / nbead
         vir(3,2) = vir(3,2) + stress_tensor(3,2) * volume / nbead
         vir(3,3) = vir(3,3) + stress_tensor(3,3) * volume / nbead

      end do
      

      end subroutine force_qegit_updatevir_MPI

!***********************************************************************
      subroutine finalize_qegit_MPI
!***********************************************************************

      implicit none


      end subroutine finalize_qegit_MPI

#else

!***********************************************************************
      subroutine force_qegit_MPI
!***********************************************************************

      implicit none

      write( 6, '(a)' ) &
     &   'Error - qegit is not compiled.'
      write( 6, '(a)' ) &
     &   'Try compiling with the option -DQEGIT=ON.'
      write( 6, '(a)' )

      call error_handling_MPI ( 1, 'subroutine force_qegit', 24 )

      return
      end





!***********************************************************************
      subroutine finalize_qegit_MPI
!***********************************************************************

      implicit none

      write( 6, '(a)' ) &
     &   'Error - qegit is not compiled.'
      write( 6, '(a)' ) &
     &   'Try compiling with the option -DQEGIT=ON.'
      write( 6, '(a)' )

      call error_handling_MPI ( 1, 'subroutine finalize_qegit', 27 )

      return
      end

#endif
