!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     energy and force from QE calculation
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine force_qe_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, ry_energy, au_energy, au_length, &
     &   box, volume, &
     &   vir, istep, natom, nbead, myrank_main, nprocs_main, &
     &   myrank_sub, nprocs_sub

      use qe_variables, only : &
     &   coord_x, coord_y, coord_z, force_x, force_y, force_z, &
     &   total_energy, stress_tensor, stress_tensor_ry, &
     &   pimd_istep, pimd_ibead, cell_param

!      use qe_variables, only :
!     &   vwave, vwaves

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr, ibead, ierr_bead, i, j
      real(8) :: stress_tensor_ry_sum(3,3)

      integer, save :: ierr1 = 0
      integer, save :: ierr_bead1 = 0

!-----------------------------------------------------------------------
!     /*   initialize stress tensor                                   */
!-----------------------------------------------------------------------
!
      stress_tensor(:,:) = 0.d0
      stress_tensor_ry_sum(:,:) = 0.d0

!     /*   set box   */
      cell_param(:,:) = box(:,:)

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!     /*   skip if `ibead is not my job'                              */
!-----------------------------------------------------------------------

         if ( mod(ibead-1,nprocs_main) .ne. myrank_main ) cycle

!-----------------------------------------------------------------------
!        /*   execute QE subroutine                                   */
!-----------------------------------------------------------------------

!        /*   set istep   */
         pimd_istep = istep

!        /*   set ibaed   */
         pimd_ibead = ibead

!        /*   set coordinate   */
         do i = 1, natom
            coord_x(i) = x(i,ibead)
            coord_y(i) = y(i,ibead)
            coord_z(i) = z(i,ibead)
         end do

!c        /*   set wavefunctions   */
!         vwave => vwaves(ibead)

!        /*   error flag   */
         ierr = 0
         ierr_bead = 0

!        /*   execute QE subroutine   */
         call qe_force( ierr, ierr_bead )

!        /*   error flag   */       
         if ( ierr .ne. 0 ) then
            ierr1 = ierr
            ierr_bead1 = ierr_bead
            exit
         end if

!        /*   total energy   */
         pot(ibead) = total_energy

!        /*   force   */
         do i = 1, natom
            fx(i,ibead) = force_x(i)
            fy(i,ibead) = force_y(i)
            fz(i,ibead) = force_z(i)
         end do

!        /*   stress   */
         stress_tensor_ry_sum(:,:) = stress_tensor_ry_sum(:,:) &
     &                             + stress_tensor_ry(:,:)

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do


!    /*   error termination   */

      if ( ierr1 .ne. 0 ) then
          if(myrank_sub .eq. 0) then
          write( 6, '(a)' ) &
     &           'Error - unable to execute QE. See log file.'
          write( 6, '(a)' )
          write( 6, '(a)' )
          write( 6, '(a)' )  &
     &   '############################################################'
          write( 6, '(a)' )  &
     &   '############################################################'
          write( 6, '(a)' )
          write( 6, '(a,i5)' )  &
     &   ' ##ERROR## convergence of electronic state NOT achieved'// &
     &   ' in bead ', ierr_bead
          write( 6, '(a)' )
          write( 6, '(a)' )  &
     &   '############################################################'
               write( 6, '(a)' )  &
     &   '############################################################'
          write( 6, '(a)' ) 
         call error_handling_MPI &
     &           ( 1, 'subroutine force_qe_MPI', 23 )
         end if
      end if

!-----------------------------------------------------------------------
!     /*   change units from Ry/bohr**3 to Hartree/bohr**3            */
!-----------------------------------------------------------------------

!     /*   stress   */
      stress_tensor(:,:) = stress_tensor_ry_sum(:,:) * 0.5d0

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   loop of process of a same bead  */
      do i = 1, nprocs_sub

!        /*   same bead  */
         if ( myrank_sub .eq. i-1 ) then

!           /*   potential   */
            call my_mpi_allreduce_real_1_main ( pot, nbead )

!           /*   force   */
            call my_mpi_allreduce_real_2_main ( fx, natom, nbead )
            call my_mpi_allreduce_real_2_main ( fy, natom, nbead )
            call my_mpi_allreduce_real_2_main ( fz, natom, nbead )

!           /*   stress   */
            call my_mpi_allreduce_real_2_main ( stress_tensor, 3, 3 )

!        /*   same bead  */
         end if

!     /*   loop of process of a same bead  */
      end do

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

      do j = 1, nbead

         vir(1,1) = vir(1,1) + stress_tensor(1,1) * volume / nbead
         vir(1,2) = vir(1,2) + stress_tensor(1,2) * volume / nbead
         vir(1,3) = vir(1,3) + stress_tensor(1,3) * volume / nbead
         vir(2,1) = vir(2,1) + stress_tensor(2,1) * volume / nbead
         vir(2,2) = vir(2,2) + stress_tensor(2,2) * volume / nbead
         vir(2,3) = vir(2,3) + stress_tensor(2,3) * volume / nbead
         vir(3,1) = vir(3,1) + stress_tensor(3,1) * volume / nbead
         vir(3,2) = vir(3,2) + stress_tensor(3,2) * volume / nbead
         vir(3,3) = vir(3,3) + stress_tensor(3,3) * volume / nbead

      end do

      return
      end





!***********************************************************************
      subroutine init_qe_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   myrank, iounit, natom, nstep, nbead, method

      use qe_variables, only : &
     &   qe_output_all_proc, qe_output_every_nstep, qe_input_file_name

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr

!-----------------------------------------------------------------------
!     /*   set QE parameters                                        */
!-----------------------------------------------------------------------

!     /*   read QE input file   */
      call read_char_MPI ( qe_input_file_name, 80, &
     &    '<qe_input_file_name>', 20, iounit )

!     /*   parent process only   */
      if ( myrank .eq. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag ( '<qe_output>', 11, iounit, ierr )

!        /*   read a line   */
         read ( iounit, *, iostat=ierr ) &
     &      qe_output_all_proc, qe_output_every_nstep

!        /*   file close   */
         close ( iounit )

!        /*   otherwise, read from default input   */
         if ( ierr .ne. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'input_default.dat' )

!           /*   search for tag    */
            call search_tag( '<qe_output>', 13, iounit, ierr )

!           /*   read a line   */
            read ( iounit, *, iostat=ierr ) &
     &         qe_output_all_proc, qe_output_every_nstep

!           /*   file close   */
            close ( iounit )

!        /*   otherwise, read from default input   */
         end if

!     /*   parent process only   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   error message   */
      if ( ierr .ne. 0 ) then
         if ( myrank .eq. 0 ) then
            write ( 6, '(a)' ) 'Error - <qe_output> read incorrectly.'
            write ( 6, '(a)' )
         end if
      end if

!     /*   error termination   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine init_qe_MPI', 22 )

!     /*   communicate   */
      call my_mpi_bcast_int_0( qe_output_all_proc )

!     /*   communicate   */
      call my_mpi_bcast_int_0( qe_output_every_nstep )

!-----------------------------------------------------------------------
!     /*   reset nstep (this is necessary somehow)                    */
!-----------------------------------------------------------------------

      if      ( method(1:10) .eq. 'TESTFORCE '  ) then

         nstep = (6*natom+1) * nbead

      else if ( method(1:11) .eq. 'TESTVIRIAL ' ) then

         nstep = (6*9+1) * nbead

      end if

!-----------------------------------------------------------------------
!     /*   print input parameters                                     */
!-----------------------------------------------------------------------

!c     /*   parent process only   */
!      if ( myrank .eq. 0 ) then
!
!         write( 6, '(a)' )
!     &      'QE input files: '//trim(qe_input_file_name)
!
!         if      ( qe_output_all_proc .eq. -1 ) then
!            write( 6, '(a,i0,a)' )
!     &         'QE log files are not printed. '
!         else if      ( qe_output_all_proc .eq. 0 ) then
!            if(qe_output_every_nstep > 0 ) then
!              write( 6, '(a,i0,a)' )
!     &         'QE log files: ' //
!     &         'printed in master process every ',
!     &         qe_output_every_nstep, ' steps.'
!            else
!              write( 6, '(a,i0,a)' )
!     &         'QE log files: ' //
!     &         'printed in master process. '//
!     &         'The convergence processes of electronic states'//
!     &         'are not printed. '
!            end if
!         else if ( qe_output_all_proc .eq. 1 ) then
!            if(qe_output_every_nstep > 0 ) then
!              write( 6, '(a,i0,a)' )
!     &         'QE log files:  ' //
!     &         'printed in all processes every ',
!     &         qe_output_every_nstep, ' steps.'
!            else
!              write( 6, '(a,i0,a)' )
!     &         'QE log files: ' //
!     &         'printed in all processes. '//
!     &         'The convergence processes of electronic states'//
!     &         'are not printed. '
!            end if
!         else
!            ierr = 1
!         end if
!
!         write( 6, '(a)' )
!
!c     /*   parent process only   */
!      end if

!-----------------------------------------------------------------------
!     /*   error termination                                          */
!-----------------------------------------------------------------------
!
!      if ( ierr .ne. 0 ) then
!
!         if ( myrank .eq. 0 ) then
!
!            write( 6, '(a)' )
!     &        'Error - QE settings incorrect.'
!            write( 6, '(a)' )
!
!         end if
!
!         call error_handling_MPI
!     &        ( 1, 'subroutine init_qe_MPI', 22 )
!
!      end if
!
!-----------------------------------------------------------------------
!     /*   initialize QE                                              */
!-----------------------------------------------------------------------

      call qe_init( ierr )

!-----------------------------------------------------------------------
!     /*   error termination                                          */
!-----------------------------------------------------------------------

      if( ierr .ne. 0 ) then

         if ( myrank .eq. 0 ) then

            write( 6, '(a)' ) &
     &        'Error - unable to initialize QE. See log file.'
            write( 6, '(a)' )

         end if

         call error_handling_MPI &
     &        ( 1, 'subroutine init_qe_MPI', 22 )

      end if

      return
      end





!***********************************************************************
      subroutine finalize_qe_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : myrank

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr

!-----------------------------------------------------------------------
!     /*   finalize QE                                               */
!-----------------------------------------------------------------------

!     /*   finalize QE subroutine   */

      ierr = 0

      call qe_finalize( ierr )

      if ( ierr .ne. 0 ) then

         if ( myrank .eq. 0 ) then
            write( 6, '(a)' ) &
     &        'Error - unable to finalize QE. See log file.'
            write( 6, '(a)' )
         end if

         call error_handling_MPI &
     &        ( 1, 'subroutine finalize_qe_MPI', 26 )

      end if

      return
      end





!=======================================================================
!
!     The following dummy subroutines are applied
!     when libqe.a is not available
!
!=======================================================================

#ifndef qe

!***********************************************************************
      subroutine qe_init( ierr )
!***********************************************************************

      use common_variables, only : myrank

      implicit none

      integer, intent(out) :: ierr

      ierr = 1

      if ( myrank .eq. 0 ) then

         write( 6, '(a)' ) &
     &      'Error termination - QE is not linked to PIMD.'
         write( 6, '(a)' )
         write( 6, '(a)' ) &
     &      'Recompile pimd.mpi.x with ../lib/libqe.a ' // &
     &      'with the options'
         write( 6, '(a)' )
         write( 6, '(a)' ) &
     &       '  QE = -Dqe'
         write( 6, '(a)' ) &
     &       '  LIBQE = -L../lib -lqe -ldmy'
         write( 6, '(a)' )

      end if

      call my_mpi_barrier
      call my_mpi_abort

      return
      end

#endif





#ifndef qe

!***********************************************************************
      subroutine qe_force( ierr, ierr_bead )
!***********************************************************************

      use common_variables, only : myrank

      implicit none

      integer, intent(out) :: ierr
      integer, intent(out) :: ierr_bead

      ierr = 1
      ierr_bead = 1

      if( myrank .eq. 0 ) then

         write(6, '(a)') 'Error termination.'
         write(6, '(a)') ''
         write(6, '(a)') 'ipotential=QE is not available because'
         write(6, '(a)') 'QE subroutines were not linked.'
         write(6, '(a)') 'Compile the pimd.mpi.x with the followings.'
         write(6, '(a)') '(You need libqe.a in ../lib)'
         write(6, '(a)') ''
         write(6, '(a)') '  FLAGMP = -Dqe'
         write(6, '(a)') '  LINKMP = -L../lib -lqe -ldmy'
         write(6, '(a)') ''
      end if

      call my_mpi_barrier
      call my_mpi_abort

      return
      end

#endif





#ifndef qe

!***********************************************************************
      subroutine qe_finalize( ierr )
!***********************************************************************

      use common_variables, only : myrank

      implicit none

      integer, intent(out) :: ierr

      ierr = 1

      if( myrank .eq. 0 ) then

         write(6, '(a)') 'Error termination.'
         write(6, '(a)') ''
         write(6, '(a)') &
     &      'QE is not available. To link QE routines, copy' &
     &     // ' libqe.a to'
         write(6, '(a)') &
     &      ' ../lib directory, then recompile pimd.mpi.x with the' &
     &      // ' following options'
         write(6, '(a)') &
     &      'included in makefile.'
         write(6, '(a)') ''
         write(6, '(a)') '  FLAGMP = -Dqe'
         write(6, '(a)') '  LINKMP = -L../lib -lqe -ldmy'
         write(6, '(a)') ''

      end if

      call my_mpi_barrier
      call my_mpi_abort

      return
      end

#endif

