!///////////////////////////////////////////////////////////////////////
!
!      Author:          Y. Nagai, M. Shiga
!      Last updated:    Jan. 18, 2022 by Y. Nagai
!      Description:     energy and force from QE with system calls
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      module qe7_variables
!***********************************************************************
!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fx, fy, fz, x, y, z, vir, nbead, natom, mbox, iounit, pot, &
     &   box, volume, iounit_qe

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   QE execution command
      character(len=80) :: qe7_command

!     //   QE input file
      character(len=80) :: qe7_input_file_name

!     //   stress tensors
      real(8) :: stress_tensor(3,3), stress_tensor_Ry(3,3)

!     //   stress tensors
      real(8) :: stress_tensor_Ry_sum(3,3)

!***********************************************************************
      end module qe7_variables
!***********************************************************************





!***********************************************************************
      subroutine force_qe
!***********************************************************************

      implicit none

      write( 6, '(a)' ) &
     &   'Error - QE (serial) is not supported.'
      write( 6, '(a)' ) &
     &   'Try QE (parallel).'
      write( 6, '(a)' )

      call error_handling ( 1, 'subroutine force_qe', 19 )

      return
      end





!***********************************************************************
      subroutine force_qe7
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     /*   shared variables   */
      use qe7_variables

!     /*   local variables   */
      implicit none

!-----------------------------------------------------------------------
!     /*   run QE with system calls                                   */
!-----------------------------------------------------------------------

!     //   read keywords
      call initialize_qe7()

!     //   write QE input file: qeinput.inp
      call write_inputfile_qe7()

!     //   execute QE
      call run_qe7()

!     //   read QE output file: qeoutput.out
      call read_outputfile_qe7()

      return
      end





!***********************************************************************
      subroutine initialize_qe7()
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     //   shared variables
      use common_variables, only: iounit, nbead

!     //   shared variables
      use qe7_variables, only: qe7_command, qe7_input_file_name

!     //   local variables
      implicit none

!     //   integers
      integer :: ibead

!     //   integers
      integer :: ierr = 0

!     //   characters
      character(len=3) :: char_num

!     //   flag
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     //   read QE execution command
!-----------------------------------------------------------------------

!     //   initial visit
      if ( iset .eq. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag ( '<qe7_command>', 13, iounit, ierr )

!        /*   read a line   */
         read ( iounit, '(a)', iostat=ierr ) qe7_command
!
         /*   file close   */
         close ( iounit )

!        //   read default
         if ( ierr .ne. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'input_default.dat' )
                        
!           /*   search for tag    */
            call search_tag ( '<qe7_command>', 13, iounit, ierr )

!           /*   read a line   */
            read ( iounit, '(a)', iostat=ierr ) qe7_command
                        
!           /*   file close   */
            close( iounit )

!        //   read default
         end if

!        //   error handling
         call error_handling ( ierr, 'subroutine initialize_qe7', 25 )

!-----------------------------------------------------------------------
!        //   read QE input file
!-----------------------------------------------------------------------

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag ( '<qe_input_file_name>', 20, iounit, ierr )

!        /*   read a line   */
         read ( iounit, *, iostat=ierr ) qe7_input_file_name

!        /*   file close   */
         close ( iounit )

!        //   read default
         if ( ierr .ne. 0 ) then

            /*   file open   */
            open ( iounit, file = 'input_default.dat' )
                        
            /*   search for tag    */
            call search_tag ( '<qe_input_file_name>', 20, iounit, ierr )

!           /*   read a line   */
            read ( iounit, *, iostat=ierr ) qe7_input_file_name
                        
!           /*   file close   */
            close( iounit )

!        //   read default
         end if

!        //   error handling
         call error_handling ( ierr, 'subroutine initialize_qe7', 25 )

!-----------------------------------------------------------------------
!        //   create directory for each beads
!-----------------------------------------------------------------------

!        //   loop of beads
         do ibead = 1, nbead

!           /*   character   */
            call int3_to_char( ibead, char_num )

!           /*   remove old directory and create new directory   */
            call system ('mkdir -p ./' // char_num )

!        //   loop of beads
         end do

!        //   initial setting ended
         iset = 1

!     //   initial visit
      end if

      return
      end





!***********************************************************************
      subroutine write_inputfile_qe7()
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     //   shared variables
      use common_variables, only: &
     &   x, y, z, box, natom, nbead, iounit, iounit_qe

!     //   shared variables
      use qe7_variables, only: qe7_input_file_name

!     //   local variables
      implicit none

!     //   integers
      integer :: ibead, match, i, iflag

!     //   real numbers
      real(8) :: xi, yi, zi

!     //   chracters
      character(len=80) :: char_line, char_dummy

!     //   chracters
      character(len=3) :: char_num

!     //   integers
      integer :: ierr = 0

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

!     //   loop of beads
      do ibead = 1, nbead

!-----------------------------------------------------------------------
!     /*   make qe input: ATOMIC_POSITIONS                            */
!-----------------------------------------------------------------------

!        /*   character   */
         call int3_to_char( ibead, char_num )

!        //   open files
         open ( iounit, file = qe7_input_file_name )
         open ( iounit_qe, file = char_num // '/' // 'qeinput.inp' )

!        //   flag
         iflag = 0

!        //   loop of lines
         do

!            /*   read a line   */
             read ( iounit, '(a80)', iostat=ierr ) char_line

!            /*   end of line   */
             if ( ierr .ne. 0 ) exit

!            /*   see if the line matches   */
             match = index( char_line(1:16), 'ATOMIC_POSITIONS' )

!            /*   if matched   */
             if ( match .ge. 1 ) then

!               //   write a line
                write( iounit_qe, '(a)' ) 'ATOMIC_POSITIONS {bohr}'

!               //   loop of atoms
                do i = 1, natom

!                  /*   read atomic name   */
                   read ( iounit, * ) char_dummy

!                  //   coordinates
                   xi = x(i,ibead)
                   yi = y(i,ibead)
                   zi = z(i,ibead)

                   write( iounit_qe, '(a,3e20.12)' ) &
     &                char_dummy(1:3), xi, yi, zi

!               //   loop of atoms
                end do

!               //   flag
                iflag = iflag + 1

!-----------------------------------------------------------------------
!            /*   make qe input: CELL_PARAMETERS                      */
!-----------------------------------------------------------------------

!            /*   if not matched   */
             else

!               /*   see if the line matches   */
                match = index( char_line(1:15), 'CELL_PARAMETERS' )

!               /*   if matched   */
                if ( match .ge. 1 ) then

                   write( iounit_qe, '(a)' ) 'CELL_PARAMETERS {bohr}'

                   write( iounit_qe, '(3e20.12)' ) &
     &                box(1,1), box(2,1), box(3,1)
                   write( iounit_qe, '(3e20.12)' ) &
     &                box(1,2), box(2,2), box(3,2)
                   write( iounit_qe, '(3e20.12)' ) &
     &                box(1,3), box(2,3), box(3,3)

                   read ( iounit, * ) char_dummy
                   read ( iounit, * ) char_dummy
                   read ( iounit, * ) char_dummy

!                  //   flag
                   iflag = iflag + 2

!               /*   if not matched   */
                else

!                  /*   write a copy of the line   */
                   write( iounit_qe, '(a)' ) char_line

!               /*   end of if statement   */
                end if

!            /*   end of if statement   */
             end if

!        //   loop of lines
         end do

!        //   close files
         close( iounit )
         close( iounit_qe )

!        //   error handling
         if ( iflag .ne. 3 ) then
            write( 6, '(a,i1)' ) 'Error: Code ', iflag
            write( 6, '(a)' )
            call error_handling &
     &         ( 1, 'subroutine write_inputfile_qe7', 30 )
         end if

!     //   loop of beads
      end do

      return
      end





!***********************************************************************
      subroutine run_qe7()
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

!     //   shared variables
      use common_variables, only: nbead

!     //   shared variables
      use qe7_variables, only: qe7_command

!     /*   local variables   */
      implicit none

!     //   integers
      integer :: ibead

!     //   characters
      character(len=3) :: char_num

!-----------------------------------------------------------------------
!     /*   run QE                                                     */
!-----------------------------------------------------------------------

!     //   loop of beads
      do ibead = 1, nbead

!        /*   character   */
         call int3_to_char( ibead, char_num )

!        //   run QE by system calls
         call system ( qe7_command // ' < '// char_num // &
     &                 '/qeinput.inp > ' // char_num // &
     &                 '/qeoutput.out' )

!     //   loop of beads
      end do

      return
      end





!***********************************************************************
      subroutine read_outputfile_qe7()
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

      use common_variables, only: pot, fx, fy, fz, vir, volume, volume, &
     &   natom, nbead, iounit_qe

!     //   shared variables
      use qe7_variables, only: stress_tensor_Ry, stress_tensor_Ry_sum, &
     &   stress_tensor

!     /*   local variables   */
      implicit none

!     //   integers
      integer :: i, ibead, match, j, iflag

!     //   integers
      integer :: ierr = 0

!     //   characters
      character(len=3)  :: char_num
      character(len=80) :: char_line
      character(len=80), dimension(6) :: char_dummy

!-----------------------------------------------------------------------
!     /*   read qe7 output:  potential                                */
!-----------------------------------------------------------------------

!     //   initialize stress
      stress_tensor_Ry_sum(:,:) = 0.d0

!     //   loop of beads
      do ibead = 1, nbead

!        /*   character   */
         call int3_to_char( ibead, char_num )

         /*   open the QE output file   */
         open ( iounit_qe, file = char_num // '/' // 'qeoutput.out' )

!        //   flag
         iflag = 0

!        //   loop of lines
         do

!           /*   read a line   */
            read ( iounit_qe, '(a80)', iostat=ierr ) char_line

!           /*   error handling   */
            if ( ierr .ne. 0 ) exit

!           /*   see if the line matches   */
            match = index( char_line(1:17), '!    total energy' )

!           /*   if matched   */
            if ( match .ge. 1 ) then

!              /*   go back one line   */
               backspace( iounit_qe )

!              /*   read the potential in [Ry]   */
               read ( iounit_qe, *, iostat=ierr ) &
     &            char_dummy(1), char_dummy(2), &
     &            char_dummy(3), char_dummy(4), &
     &            pot(ibead), char_dummy(5)

!              /*   convert [Ry] to [Hartree]   */
               pot(ibead) = pot(ibead) * 0.5d0 

!              //   flag
               if ( ierr .eq. 0 ) iflag = iflag + 1

!              //   exit from loop
               exit

!           /*   if matched   */
            end if

!        //   loop of lines
         end do

!        /*   close file   */
         close( iounit_qe )

!-----------------------------------------------------------------------
!     /*   read qe7 output:  forces                                   */
!-----------------------------------------------------------------------

!        /*   character   */
         call int3_to_char( ibead, char_num )

!        /*   open the QE output file   */
         open ( iounit_qe, file = char_num // '/' // 'qeoutput.out' )

!        //   loop of lines
         do

!           /*   read a line   */
            read ( iounit_qe, '(a80)', iostat=ierr ) char_line

!           /*   error handling   */
            if ( ierr .ne. 0 ) exit

!           /*   see if the line matches   */
            match = index( char_line(1:27),  &
     &         '     Forces acting on atoms' )

!           /*   if matched   */
            if ( match .ge. 1 ) then

!              /*   read a line   */
               read ( iounit_qe, '(a80)', iostat=ierr ) char_line

!              /*   loop of atoms   */
               do i = 1, natom

!                 /*   read forces in [Ry/bohr]  */
                  read (  iounit_qe, *, iostat=ierr ) &
     &               char_dummy(1), char_dummy(2),  &
     &               char_dummy(3), char_dummy(4), &
     &               char_dummy(5), char_dummy(6), &
     &               fx(i,ibead), fy(i,ibead), fz(i,ibead)

!                 /*   convert to [Hartree/bohr]  */
                  fx(i,ibead) = fx(i,ibead) * 0.5d0
                  fy(i,ibead) = fy(i,ibead) * 0.5d0
                  fz(i,ibead) = fz(i,ibead) * 0.5d0

!              /*   loop of atoms   */
               end do

!              //   flag
               if ( ierr .eq. 0 ) iflag = iflag + 2

!              //   exit from loop
               exit

!           /*   if matched   */
            end if

!        //   loop of lines
         end do

!        /*   close file   */
         close( iounit_qe )

!-----------------------------------------------------------------------
!     /*   read qe7 output:  stress tensor                            */
!-----------------------------------------------------------------------

!        /*   character   */
         call int3_to_char( ibead, char_num )

!        /*   open the QE output file   */
         open ( iounit_qe, file = char_num // '/' // 'qeoutput.out' )

!        //   loop of lines
         do

!           /*   read a line   */
            read ( iounit_qe, '(a80)', iostat=ierr ) char_line

!           /*   error handling   */
            if ( ierr .ne. 0 ) exit

!           /*   see if the line matches   */
            match = index( char_line(1:21), '     Computing stress' )

!           /*   if matched   */
            if ( match .ge. 1 ) then

!              /*   read a line   */
               read ( iounit_qe, '(a80)', iostat=ierr ) char_line
               read ( iounit_qe, '(a80)', iostat=ierr ) char_line

!              /*   read stress in [Ry/bohr**3]   */
               read ( iounit_qe, *, iostat=ierr ) &
     &            stress_tensor_Ry(1,1), stress_tensor_Ry(1,2), &
     &            stress_tensor_Ry(1,3)
               read ( iounit_qe, *, iostat=ierr ) &
     &            stress_tensor_Ry(2,1), stress_tensor_Ry(2,2), &
     &            stress_tensor_Ry(2,3)
     
               read ( iounit_qe, *, iostat=ierr ) &
     &            stress_tensor_Ry(3,1), stress_tensor_Ry(3,2), &
     &            stress_tensor_Ry(3,3)

!              //   sum over beads
               stress_tensor_Ry_sum(:,:) = stress_tensor_Ry_sum(:,:) &
     &            + stress_tensor_Ry(:,:)

!              //   flag
               if ( ierr .eq. 0 ) iflag = iflag + 4

!              //   exit from loop
               exit

!           /*   if matched   */
            end if

!        //   loop of lines
         end do

!        /*   close file   */
         close( iounit_qe )

!        //   error handling
         if ( iflag .ne. 7 ) then
            write( 6, '(a,i1)' ) 'Error: Code ', iflag
            write( 6, '(a)' )
            call error_handling &
     &         ( 1, 'subroutine read_outputfile_qe7', 30 )
         end if

!     //   loop of beads
      end do

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

!     /*   stress in [Hartree/bohr**3]   */
      stress_tensor(:,:) = stress_tensor_Ry_sum(:,:) * 0.5d0

      do j = 1, nbead

         vir(1,1) = vir(1,1) + stress_tensor(1,1) * volume / nbead
         vir(1,2) = vir(1,2) + stress_tensor(1,2) * volume / nbead
         vir(1,3) = vir(1,3) + stress_tensor(1,3) * volume / nbead
         vir(2,1) = vir(2,1) + stress_tensor(2,1) * volume / nbead
         vir(2,2) = vir(2,2) + stress_tensor(2,2) * volume / nbead
         vir(2,3) = vir(2,3) + stress_tensor(2,3) * volume / nbead
         vir(3,1) = vir(3,1) + stress_tensor(3,1) * volume / nbead
         vir(3,2) = vir(3,2) + stress_tensor(3,2) * volume / nbead
         vir(3,3) = vir(3,3) + stress_tensor(3,3) * volume / nbead

      end do

      return
      end
