!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    May 1, 2020 by M. Shiga
!      Description:     energy and force from polarizable mm
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine force_pol
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   setup                                                      */
!-----------------------------------------------------------------------

      call force_pol_setup

!-----------------------------------------------------------------------
!     /*   main routine                                               */
!-----------------------------------------------------------------------

      call force_pol_main

      return
      end





!***********************************************************************
      subroutine force_pol_setup
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iboundary

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   only for the first access                                  */
!-----------------------------------------------------------------------

      if ( iset .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   linear bonds                                               */
!-----------------------------------------------------------------------

      call force_mm_lin_setup

!-----------------------------------------------------------------------
!     /*   general linear bonds                                       */
!-----------------------------------------------------------------------

      call force_mm_genlin_setup

!-----------------------------------------------------------------------
!     /*   angular bonds                                              */
!-----------------------------------------------------------------------

      call force_mm_angl_setup

!-----------------------------------------------------------------------
!     /*   dihedral bonds                                             */
!-----------------------------------------------------------------------

      call force_mm_dih_setup

!-----------------------------------------------------------------------
!     /*   improper bonds                                             */
!-----------------------------------------------------------------------

      call force_mm_improper_setup

!-----------------------------------------------------------------------
!     /*   cmap of two dihedral bonds                                 */
!-----------------------------------------------------------------------

      call force_mm_cmap_setup

!-----------------------------------------------------------------------
!     /*   lennard-jones                                              */
!-----------------------------------------------------------------------

      call force_mm_lj_setup

!-----------------------------------------------------------------------
!     /*   lennard-jones pair                                         */
!-----------------------------------------------------------------------

      call force_mm_ljpair_setup

!-----------------------------------------------------------------------
!     /*   buckingham                                                 */
!-----------------------------------------------------------------------

      call force_mm_buck_setup

!-----------------------------------------------------------------------
!     /*   morse potential                                            */
!-----------------------------------------------------------------------

      call force_mm_morse_setup

!-----------------------------------------------------------------------
!     /*   polarizable model                                          */
!-----------------------------------------------------------------------

!     /*   free boundary   */
      if ( iboundary .eq. 0 ) then

!        /*   direct sum   */
         call force_pol_coulomb_setup

      else if ( iboundary .eq. 1 ) then

!        /*   Ewald sum   */
         call force_pol_ewald_setup

      else if ( iboundary .eq. 2 ) then

!        /*   Ewald sum   */
         call force_pol_ewald_setup

!     /*   end boundary condition   */
      end if

!-----------------------------------------------------------------------
!     /*   setup done                                                 */
!-----------------------------------------------------------------------

      iset = 1

      return
      end





!***********************************************************************
      subroutine force_pol_main
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iboundary

      implicit none

!-----------------------------------------------------------------------
!     /*   linear bonds                                               */
!-----------------------------------------------------------------------

      call force_mm_lin

!-----------------------------------------------------------------------
!     /*   generalized linear bonds                                   */
!-----------------------------------------------------------------------

      call force_mm_genlin

!-----------------------------------------------------------------------
!     /*   angular bonds                                              */
!-----------------------------------------------------------------------

      call force_mm_angl

!-----------------------------------------------------------------------
!     /*   dihedral bonds                                             */
!-----------------------------------------------------------------------

      call force_mm_dih

!-----------------------------------------------------------------------
!     /*   improper bonds                                             */
!-----------------------------------------------------------------------

      call force_mm_improper

!-----------------------------------------------------------------------
!     /*   cmap of two dihedral bonds                                 */
!-----------------------------------------------------------------------

      call force_mm_cmap

!-----------------------------------------------------------------------
!     /*   lennard-jones                                              */
!-----------------------------------------------------------------------

      call force_mm_lj

!-----------------------------------------------------------------------
!     /*   lennard-jones pair                                         */
!-----------------------------------------------------------------------

      call force_mm_ljpair

!-----------------------------------------------------------------------
!     /*   buckingham                                                 */
!-----------------------------------------------------------------------

      call force_mm_buck

!-----------------------------------------------------------------------
!     /*   morse potential                                            */
!-----------------------------------------------------------------------

      call force_mm_morse

!-----------------------------------------------------------------------
!     /*   polarizable model                                          */
!-----------------------------------------------------------------------

!     /*   free boundary   */
      if ( iboundary .eq. 0 ) then

!        /*   direct sum   */
         call force_pol_coulomb

      else if ( iboundary .eq. 1 ) then

!        /*   Ewald sum   */
         call force_pol_ewald

      else if ( iboundary .eq. 2 ) then

!        /*   Ewald sum   */
         call force_pol_ewald

!     /*   end boundary condition   */
      end if

      return
      end





!***********************************************************************
      subroutine force_pol_coulomb
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   electric field                                             */
!-----------------------------------------------------------------------

      call field_pol_coulomb

!-----------------------------------------------------------------------
!     /*   induced dipole term                                        */
!-----------------------------------------------------------------------

      call force_pol_induced

!-----------------------------------------------------------------------
!     /*   pair interactions                                          */
!-----------------------------------------------------------------------

      call force_pol_coulomb_pair

      return
      end





!***********************************************************************
      subroutine force_pol_ewald
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use mm_variables, only : ncharge

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   return if no charges                                       */
!-----------------------------------------------------------------------

      if ( ncharge .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   electric field                                             */
!-----------------------------------------------------------------------

      call field_pol_ewald

!-----------------------------------------------------------------------
!     /*   induced dipole term                                        */
!-----------------------------------------------------------------------

      call force_pol_induced

!-----------------------------------------------------------------------
!     /*   real space contribution of Ewald sum                       */
!-----------------------------------------------------------------------

      call force_pol_ewald_rs

!-----------------------------------------------------------------------
!     /*   Fourier space contribution of Ewald sum                    */
!-----------------------------------------------------------------------

      call force_pol_ewald_fs

!-----------------------------------------------------------------------
!     /*   self contribution of Ewald sum                             */
!-----------------------------------------------------------------------

      call force_pol_ewald_self

!-----------------------------------------------------------------------
!     /*   charged system contribution of Ewald sum                   */
!-----------------------------------------------------------------------

      call force_pol_charge

!-----------------------------------------------------------------------
!     /*   dipole moment and surface dipole of Ewald sum              */
!-----------------------------------------------------------------------

      call force_pol_dipole

      return
      end





!***********************************************************************
      subroutine force_pol_coulomb_setup
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, natom, nbead

      use mm_variables, only : &
     &   q, pol, px, py, pz, factor_bcp, ncharge, npol, list_pol, nbcp, &
     &   ndampint, dampform_qq, dampform_qp, dampform_pp, damppar_qq, &
     &   damppar_qp, damppar_pp, kind_q, nkind_q, i_q, i_bcp, j_bcp

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer i, j, k, l, ierr

      real(8) :: qi, poli

      character(len=8) :: dampint, dampform

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initial settings                                           */
!-----------------------------------------------------------------------

      if ( iset .eq. 1 ) return

      iset = 1

!-----------------------------------------------------------------------
!     /*   number of charges                                          */
!-----------------------------------------------------------------------

!     //   open file
      open ( iounit, file = 'mm.dat' )

!     /*   tag   */
      call search_tag ( '<charges>', 9, iounit, ierr )

!     /*   number of charges   */
      if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr ) ncharge

!     //   close file
      close( iounit )

!     //   on error no charges
      if ( ierr .ne. 0 ) ncharge = 0

!     //   charges
      if ( .not. allocated( q ) ) allocate( q(natom) )

!     //   initially all charges are zero
      q(:) = 0.d0

!     //   return if no charges
      if ( ncharge .eq. 0 ) return

!     //   charged atoms
      if ( .not. allocated( i_q ) ) allocate( i_q(ncharge) )

!     //   open file
      open ( iounit, file = 'mm.dat' )

!     /*   tag   */
      call search_tag ( '<charges>', 9, iounit, ierr )

!     /*   number of charges   */
      if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr )

!     /*   loop of charges   */
      do k = 1, ncharge

!        //   read line
         read( iounit, *, iostat=ierr ) i

!        //   check error
         if ( ierr .ne. 0 ) exit

!        //   check error
         if ( ( i .lt. 1 ) .or. ( i .gt. natom ) ) ierr = 1

!        //   check error
         if ( ierr .ne. 0 ) exit

!        //   go back a line
         backspace( iounit )

!        //   read line
         read( iounit, *, iostat=ierr ) j, q(i)

!        //   charged atoms
         i_q(k) = i

!     /*   loop of charges   */
      end do

!     //   close file
      close( iounit )

!-----------------------------------------------------------------------
!     /*   number of polarizable atoms                                */
!-----------------------------------------------------------------------

!     //   open file
      open ( iounit, file = 'mm.dat' )

!     /*   tag   */
      call search_tag ( '<charges>', 9, iounit, ierr )

!     /*   number of charges   */
      if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr )

!     /*   counter   */
      l = 0

!     //   loop of charges
      do k = 1, ncharge

!        //   atom
         read( iounit, *, iostat=ierr ) i

!        //   check error
         if ( ierr .ne. 0 ) exit

!        //   one line back
         backspace( iounit )

!        //   check for error
         if ( ( i .lt. 1 ) .or. ( i .gt. natom ) ) then
            ierr = 1
            exit
         end if

!        /*   atom, charge, polarizability   */
         read( iounit, *, iostat=ierr ) i, qi, poli, j

!        //   if read with error polarizability is zero
         if ( ierr .ne. 0 ) poli = 0.d0
         if ( ierr .ne. 0 ) j = 1

!        //   if the atom is polarizable update counter
         if ( poli .ne. 0.d0 ) l = l + 1

!     //   loop of charges
      end do

!     //   close file
      close( iounit )

!     //   error handling
      call error_handling &
     &   ( ierr, 'subroutine force_pol_coulomb_setup', 34 )

!     //   number of polarizable atoms
      npol = l

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!ccc     //   return if no polarizable atoms
!cc      if ( npol .eq. 0 ) return

!     //   polarizable charge id
      if ( .not. allocated( list_pol ) ) allocate( list_pol(ncharge) )

!     //   charge kind
      if ( .not. allocated( kind_q ) ) allocate( kind_q(ncharge) )

!     //   polarizablities
      if ( .not. allocated( pol ) ) allocate( pol(ncharge) )

!     //   induced dipoles
      if ( .not. allocated( px ) ) allocate( px(ncharge,nbead) )
      if ( .not. allocated( py ) ) allocate( py(ncharge,nbead) )
      if ( .not. allocated( pz ) ) allocate( pz(ncharge,nbead) )

!     //   reset variables
      list_pol(:) = 0
      kind_q(:)   = 0
      pol(:)   = 0.d0
      px(:,:)  = 0.d0
      py(:,:)  = 0.d0
      pz(:,:)  = 0.d0

!-----------------------------------------------------------------------
!     /*   set up polarizable atoms                                   */
!-----------------------------------------------------------------------

!     //   open file
      open ( iounit, file = 'mm.dat' )

!     /*   tag   */
      call search_tag ( '<charges>', 9, iounit, ierr )

!     /*   number of charges   */
      if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr )

!     /*   counter   */
      l = 0

!     //   loop of charges
      do k = 1, ncharge

!        /*   atom, charge, polarizability   */
         read( iounit, *, iostat=ierr ) i, qi, poli, j

!        //   if read with error polarizability is zero
         if ( ierr .ne. 0 ) poli = 0.d0
         if ( ierr .ne. 0 ) j = 1

!        //   save polarizability
         pol(k) = poli

!        //   save kind
         kind_q(k) = j

!        //   if the atom is polarizable
         if ( poli .ne. 0.d0 ) then

!           //   update counter
            l = l + 1

!           //   polarizable charge id
            list_pol(k) = l

!        //   end of if statement
         end if

!     //   loop of charges
      end do

!     //   close file
      close( iounit )

!     //   error handling
      call error_handling &
     &   ( ierr, 'subroutine force_pol_coulomb_setup', 34 )

!     //   number of kinds
      nkind_q = maxval(kind_q(:))

!-----------------------------------------------------------------------
!     /*   set up polarizable atoms                                   */
!-----------------------------------------------------------------------

      if ( .not. allocated( dampform_qq ) ) &
     &   allocate( dampform_qq(nkind_q,nkind_q) )

      if ( .not. allocated( dampform_qp ) ) &
     &   allocate( dampform_qp(nkind_q,nkind_q) )

      if ( .not. allocated( dampform_pp ) ) &
     &   allocate( dampform_pp(nkind_q,nkind_q) )

      if ( .not. allocated( damppar_qq ) ) &
     &   allocate( damppar_qq(2,nkind_q,nkind_q) )

      if ( .not. allocated( damppar_qp ) ) &
     &   allocate( damppar_qp(2,nkind_q,nkind_q) )

      if ( .not. allocated( damppar_pp ) ) &
     &   allocate( damppar_pp(2,nkind_q,nkind_q) )

      dampform_qq(:,:) = 'NONE '
      dampform_qp(:,:) = 'NONE '
      dampform_pp(:,:) = 'NONE '

      damppar_qq(:,:,:) = 0.d0
      damppar_qp(:,:,:) = 0.d0
      damppar_pp(:,:,:) = 0.d0

!     //   open file
      open ( iounit, file = 'mm.dat' )

!     /*   tag   */
      call search_tag ( '<damping>', 9, iounit, ierr )

!     //   if read correctly
      if ( ierr .eq. 0 ) then

!        //   number of damping interactions
         read( iounit, *, iostat=ierr ) ndampint

!        //   check error
         if ( ierr .ne. 0 ) ndampint = 0

!        //   loop of damping interactions
         do i = 1, ndampint

!           //   read interaction, atom pair, function
            read( iounit, *, iostat=ierr ) dampint, k, l, dampform

!           //   check error
            if ( ierr .ne. 0 ) exit

!           //   check error
            if ( ( k .lt. 0 ) .or. ( k .gt. nkind_q ) .or. &
     &           ( l .lt. 0 ) .or. ( l .gt. nkind_q ) ) then
               ierr = 1
               exit
            end if

!           //   charge-charge interactions
            if      ( dampint(1:3) .eq. 'CC ' ) then

               dampform_qq(k,l) = dampform
               dampform_qq(l,k) = dampform

!              //   oss form
               if ( dampform(1:4) .eq. 'OSS ' ) then

!                 //   go back one line
                  backspace( iounit )

!                 //   read interaction, atom pair, function
                  read( iounit, *, iostat=ierr ) &
     &               dampint, k, l, dampform, damppar_qq(1:2,k,l)

!                 //   check error
                  if ( ierr .ne. 0 ) exit

!                 //   substitution
                  damppar_qq(1:2,l,k) = damppar_qq(1:2,k,l)

!              //   linear, exponential thole and gaussian forms
               else if ( ( dampform(1:4) .eq. 'LIN ' ) .or. &
     &                   ( dampform(1:4) .eq. 'EXP ' ) .or. &
     &                   ( dampform(1:4) .eq. 'GAU ' ) ) then

!                 //   go back one line
                  backspace( iounit )

!                 //   read interaction, atom pair, function
                  read( iounit, *, iostat=ierr ) &
     &               dampint, k, l, dampform, damppar_qq(1:1,k,l)

!                 //   check error
                  if ( ierr .ne. 0 ) exit

!                 //   substitution
                  damppar_qq(1:1,l,k) = damppar_qq(1:1,k,l)

!              //   damping form
               end if

!           //   charge-dipole interactions
            else if ( dampint(1:3) .eq. 'CD ' ) then

               dampform_qp(k,l) = dampform
!cc               dampform_qp(l,k) = dampform

!              //   oss form
               if ( dampform(1:4) .eq. 'OSS ' ) then

!                 //   go back one line
                  backspace( iounit )

!                 //   read interaction, atom pair, function
                  read( iounit, *, iostat=ierr ) &
     &               dampint, k, l, dampform, damppar_qp(1:2,k,l)

!                 //   check error
                  if ( ierr .ne. 0 ) exit

!ccc                 //   substitution
!cc                  damppar_qp(1:2,l,k) = damppar_qp(1:2,k,l)

!              //   linear, exponential thole and gaussian forms
               else if ( ( dampform(1:4) .eq. 'LIN ' ) .or. &
     &                   ( dampform(1:4) .eq. 'EXP ' ) .or. &
     &                   ( dampform(1:4) .eq. 'GAU ' ) ) then

!                 //   go back one line
                  backspace( iounit )

!                 //   read interaction, atom pair, function
                  read( iounit, *, iostat=ierr ) &
     &               dampint, k, l, dampform, damppar_qp(1:1,k,l)

!                 //   check error
                  if ( ierr .ne. 0 ) exit

!ccc                 //   substitution
!cc                  damppar_qp(1:1,l,k) = damppar_qp(1:1,k,l)

!              //   damping form
               end if

!           //   dipole-dipole interactions
            else if ( dampint(1:3) .eq. 'DD ' ) then

               dampform_pp(k,l) = dampform
               dampform_pp(l,k) = dampform

!              //   oss form
               if ( dampform(1:4) .eq. 'OSS ' ) then

!                 //   go back one line
                  backspace( iounit )

!                 //   read interaction, atom pair, function
                  read( iounit, *, iostat=ierr ) &
     &               dampint, k, l, dampform, damppar_pp(1:2,k,l)

!                 //   check error
                  if ( ierr .ne. 0 ) exit

!                 //   substitution
                  damppar_pp(1:2,l,k) = damppar_pp(1:2,k,l)

!              //   linear, exponential thole and gaussian forms
               else if ( ( dampform(1:4) .eq. 'LIN ' ) .or. &
     &                   ( dampform(1:4) .eq. 'EXP ' ) .or. &
     &                   ( dampform(1:4) .eq. 'GAU ' ) ) then

!                 //   go back one line
                  backspace( iounit )

!                 //   read interaction, atom pair, function
                  read( iounit, *, iostat=ierr ) &
     &               dampint, k, l, dampform, damppar_pp(1:1,k,l)

!                 //   check error
                  if ( ierr .ne. 0 ) exit

!                 //   substitution
                  damppar_pp(1:1,l,k) = damppar_pp(1:1,k,l)

!              //   damping form
               end if

!           //   interactions
            end if

!        //   loop of damping interactions
         end do

!     //   if read correctly
      end if

!     //   close file
      close( iounit )

!     //   error handling
      call error_handling &
     &   ( ierr, 'subroutine force_pol_coulomb_setup', 34 )

!-----------------------------------------------------------------------
!     /*   set up bonded charge pairs                                 */
!-----------------------------------------------------------------------

!     //   open file
      open ( iounit, file = 'mm.dat' )

!     /*   tag   */
      call search_tag ( '<nbcp>', 6, iounit, ierr )

!     /*   number of bonded charge pairs   */
      if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr ) nbcp

!     //   close file
      close( iounit )

!     //   if bonded charge pairs are absent
      if ( ierr .ne. 0 ) nbcp = 0

!     //   if bonded charge pairs are present
      if ( nbcp .ge. 1 ) then

!        //   memory allocation
         if ( .not. allocated(  i_bcp ) ) &
     &      allocate(  i_bcp(nbcp))
         if ( .not. allocated(  j_bcp ) ) &
     &      allocate(  j_bcp(nbcp))
         if ( .not. allocated(  factor_bcp ) ) &
     &      allocate(  factor_bcp(nbcp))

!        //   open file
         open ( iounit, file = 'mm.dat' )

!        /*   tag   */
         call search_tag ( '<nbcp>', 6, iounit, ierr )

!        /*   number of bonded charge pairs   */
         if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr )

!        //   read atomic pair and scaling factor
         do k = 1, nbcp
            read( iounit, *, iostat=ierr ) &
     &         i_bcp(k), j_bcp(k), factor_bcp(k)
         end do

!        //   close file
        close( iounit )

!        //   error handling
         call error_handling &
     &      ( ierr, 'subroutine force_pol_coulomb_setup', 34 )

!     //   if bonded charge pairs are present
      end if

      return
      end





!***********************************************************************
      subroutine force_pol_ewald_setup
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, box, boxinv, volume, iounit, natom, nbead

      use mm_variables, only : &
     &   q, pol, px, py, pz, s_ewald, eps_ewald, ratio_ewald, &
     &   alpha_ewald, rcut_ewald, s_ewpol, eps_ewpol, ratio_ewpol, &
     &   alpha_ewpol, rcut_ewpol, factor_bcp, lmax_ewald, nbox_ewald, &
     &   nbox_ewpol, ncharge, npol, list_pol, kind_q, nkind_q, i_q, &
     &   ndampint, dampform_qq, dampform_qp, dampform_pp, damppar_qq, &
     &   damppar_qp, damppar_pp, ioption_ewald, lmax_ewpol, &
     &   nbcp, i_bcp, j_bcp

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, ierr

      real(8) :: snew, sold, sdif, absx, absy, absz, absa, absb, absc, &
     &           qi, poli

      character(len=80) :: char_line

      character(len=8) :: dampint, dampform

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initial settings                                           */
!-----------------------------------------------------------------------

      if ( iset .eq. 1 ) return

      iset = 1

!-----------------------------------------------------------------------
!     /*   number of charges                                          */
!-----------------------------------------------------------------------

!     //   open file
      open ( iounit, file = 'mm.dat' )

!     /*   tag   */
      call search_tag ( '<charges>', 9, iounit, ierr )

!     /*   number of charges   */
      if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr ) ncharge

!     //   close file
      close( iounit )

!     //   on error no charges
      if ( ierr .ne. 0 ) ncharge = 0

!     //   charges
      if ( .not. allocated( q ) ) allocate( q(natom) )

!     //   initially all charges are zero
      q(:) = 0.d0

!     //   return if no charges
      if ( ncharge .eq. 0 ) return

!     //   charged atoms
      if ( .not. allocated( i_q ) ) allocate( i_q(ncharge) )

!     //   open file
      open ( iounit, file = 'mm.dat' )

!     /*   tag   */
      call search_tag ( '<charges>', 9, iounit, ierr )

!     /*   number of charges   */
      if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr )

!     /*   loop of charges   */
      do k = 1, ncharge

!        //   read line
         read( iounit, *, iostat=ierr ) i

!        //   check error
         if ( ierr .ne. 0 ) exit

!        //   check error
         if ( ( i .lt. 1 ) .or. ( i .gt. natom ) ) ierr = 1

!        //   check error
         if ( ierr .ne. 0 ) exit

!        //   go back a line
         backspace( iounit )

!        //   read line
         read( iounit, *, iostat=ierr ) j, q(i)

!        //   charged atoms
         i_q(k) = i

!     /*   loop of charges   */
      end do

!     //   close file
      close( iounit )

!-----------------------------------------------------------------------
!     /*   number of polarizable atoms                                */
!-----------------------------------------------------------------------

!     //   default value
      npol = 0

!     //   return if no charges
      if ( ncharge .eq. 0 ) return

!     //   open file
      open ( iounit, file = 'mm.dat' )

!     /*   tag   */
      call search_tag ( '<charges>', 9, iounit, ierr )

!     /*   number of charges   */
      if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr )

!     /*   counter   */
      l = 0

!     //   loop of charges
      do k = 1, ncharge

!        //   atom
         read( iounit, *, iostat=ierr ) i

!        //   check error
         if ( ierr .ne. 0 ) exit

!        //   one line back
         backspace( iounit )

!        //   check for error
         if ( ( i .lt. 1 ) .or. ( i .gt. natom ) ) then
            ierr = 1
            exit
         end if

!        /*   atom, charge, polarizability   */
         read( iounit, *, iostat=ierr ) i, qi, poli, j

!        //   if read with error polarizability is zero
         if ( ierr .ne. 0 ) poli = 0.d0
         if ( ierr .ne. 0 ) j = 1

!        //   if the atom is polarizable update counter
         if ( poli .ne. 0.d0 ) l = l + 1

!     //   loop of charges
      end do

!     //   close file
      close( iounit )

!     //   error handling
      call error_handling &
     &   ( ierr, 'subroutine force_pol_ewald_setup', 32 )

!     //   number of polarizable atoms
      npol = l

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!ccc     //   return if no polarizable atoms
!cc      if ( npol .eq. 0 ) return

!     //   polarizable charge id
      if ( .not. allocated( list_pol ) ) allocate( list_pol(ncharge) )

!     //   charge kind
      if ( .not. allocated( kind_q ) ) allocate( kind_q(ncharge) )

!     //   polarizablities
      if ( .not. allocated( pol ) ) allocate( pol(ncharge) )

!     //   induced dipoles
      if ( .not. allocated( px ) ) allocate( px(ncharge,nbead) )
      if ( .not. allocated( py ) ) allocate( py(ncharge,nbead) )
      if ( .not. allocated( pz ) ) allocate( pz(ncharge,nbead) )

!     //   reset variables
      list_pol(:) = 0
      kind_q(:)   = 0
      pol(:)   = 0.d0
      px(:,:)  = 0.d0
      py(:,:)  = 0.d0
      pz(:,:)  = 0.d0

!-----------------------------------------------------------------------
!     /*   set up polarizable atoms                                   */
!-----------------------------------------------------------------------

!     //   open file
      open ( iounit, file = 'mm.dat' )

!     /*   tag   */
      call search_tag ( '<charges>', 9, iounit, ierr )

!     /*   number of charges   */
      if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr )

!     /*   counter   */
      l = 0

!     //   loop of charges
      do k = 1, ncharge

!        /*   atom, charge, polarizability   */
         read( iounit, *, iostat=ierr ) i, qi, poli, j

!        //   if read with error polarizability is zero
         if ( ierr .ne. 0 ) poli = 0.d0
         if ( ierr .ne. 0 ) j = 1

!        //   save polarizability
         pol(k) = poli

!        //   save kind
         kind_q(k) = j

!        //   if the atom is polarizable
         if ( poli .ne. 0.d0 ) then

!           //   update counter
            l = l + 1

!           //   polarizable charge id
            list_pol(k) = l

!        //   end of if statement
         end if

!     //   loop of charges
      end do

!     //   close file
      close( iounit )

!     //   error handling
      call error_handling &
     &   ( ierr, 'subroutine force_pol_ewald_setup', 32 )

!     //   number of kinds
      nkind_q = maxval(kind_q(:))

!-----------------------------------------------------------------------
!     /*   set up polarizable atoms                                   */
!-----------------------------------------------------------------------

      if ( .not. allocated( dampform_qq ) ) &
     &   allocate( dampform_qq(nkind_q,nkind_q) )

      if ( .not. allocated( dampform_qp ) ) &
     &   allocate( dampform_qp(nkind_q,nkind_q) )

      if ( .not. allocated( dampform_pp ) ) &
     &   allocate( dampform_pp(nkind_q,nkind_q) )

      if ( .not. allocated( damppar_qq ) ) &
     &   allocate( damppar_qq(2,nkind_q,nkind_q) )

      if ( .not. allocated( damppar_qp ) ) &
     &   allocate( damppar_qp(2,nkind_q,nkind_q) )

      if ( .not. allocated( damppar_pp ) ) &
     &   allocate( damppar_pp(2,nkind_q,nkind_q) )

      dampform_qq(:,:) = 'NONE '
      dampform_qp(:,:) = 'NONE '
      dampform_pp(:,:) = 'NONE '

      damppar_qq(:,:,:) = 0.d0
      damppar_qp(:,:,:) = 0.d0
      damppar_pp(:,:,:) = 0.d0

!     //   open file
      open ( iounit, file = 'mm.dat' )

!     /*   tag   */
      call search_tag ( '<damping>', 9, iounit, ierr )

!     //   if read correctly
      if ( ierr .eq. 0 ) then

!        //   number of damping interactions
         read( iounit, *, iostat=ierr ) ndampint

!        //   check error
         if ( ierr .ne. 0 ) ndampint = 0

!        //   loop of damping interactions
         do i = 1, ndampint

!           //   read interaction, atom pair, function
            read( iounit, *, iostat=ierr ) dampint, k, l, dampform

!           //   check error
            if ( ierr .ne. 0 ) exit

!           //   check error
            if ( ( k .lt. 0 ) .or. ( k .gt. nkind_q ) .or. &
     &           ( l .lt. 0 ) .or. ( l .gt. nkind_q ) ) then
               ierr = 1
               exit
            end if

!           //   charge-charge interactions
            if      ( dampint(1:3) .eq. 'CC ' ) then

               dampform_qq(k,l) = dampform
               dampform_qq(l,k) = dampform

!              //   oss form
               if ( dampform(1:4) .eq. 'OSS ' ) then

!                 //   go back one line
                  backspace( iounit )

!                 //   read interaction, atom pair, function
                  read( iounit, *, iostat=ierr ) &
     &               dampint, k, l, dampform, damppar_qq(1:2,k,l)

!                 //   check error
                  if ( ierr .ne. 0 ) exit

!                 //   substitution
                  damppar_qq(1:2,l,k) = damppar_qq(1:2,k,l)

!              //   linear, exponential thole and gaussian forms
               else if ( ( dampform(1:4) .eq. 'LIN ' ) .or. &
     &                   ( dampform(1:4) .eq. 'EXP ' ) .or. &
     &                   ( dampform(1:4) .eq. 'GAU ' ) ) then

!                 //   go back one line
                  backspace( iounit )

!                 //   read interaction, atom pair, function
                  read( iounit, *, iostat=ierr ) &
     &               dampint, k, l, dampform, damppar_qq(1:1,k,l)

!                 //   check error
                  if ( ierr .ne. 0 ) exit

!                 //   substitution
                  damppar_qq(1:1,l,k) = damppar_qq(1:1,k,l)

!              //   damping form
               end if

!           //   charge-dipole interactions
            else if ( dampint(1:3) .eq. 'CD ' ) then

               dampform_qp(k,l) = dampform
!cc               dampform_qp(l,k) = dampform

!              //   oss form
               if ( dampform(1:4) .eq. 'OSS ' ) then

!                 //   go back one line
                  backspace( iounit )

!                 //   read interaction, atom pair, function
                  read( iounit, *, iostat=ierr ) &
     &               dampint, k, l, dampform, damppar_qp(1:2,k,l)

!                 //   check error
                  if ( ierr .ne. 0 ) exit

!ccc                 //   substitution
!cc                  damppar_qp(1:2,l,k) = damppar_qp(1:2,k,l)

!              //   linear, exponential thole and gaussian forms
               else if ( ( dampform(1:4) .eq. 'LIN ' ) .or. &
     &                   ( dampform(1:4) .eq. 'EXP ' ) .or. &
     &                   ( dampform(1:4) .eq. 'GAU ' ) ) then

!                 //   go back one line
                  backspace( iounit )

!                 //   read interaction, atom pair, function
                  read( iounit, *, iostat=ierr ) &
     &               dampint, k, l, dampform, damppar_qp(1:1,k,l)

!                 //   check error
                  if ( ierr .ne. 0 ) exit

!ccc                 //   substitution
!cc                  damppar_qp(1:1,l,k) = damppar_qp(1:1,k,l)

!              //   damping form
               end if

!           //   dipole-dipole interactions
            else if ( dampint(1:3) .eq. 'DD ' ) then

               dampform_pp(k,l) = dampform
               dampform_pp(l,k) = dampform

!              //   oss form
               if ( dampform(1:4) .eq. 'OSS ' ) then

!                 //   go back one line
                  backspace( iounit )

!                 //   read interaction, atom pair, function
                  read( iounit, *, iostat=ierr ) &
     &               dampint, k, l, dampform, damppar_pp(1:2,k,l)

!                 //   check error
                  if ( ierr .ne. 0 ) exit

!                 //   substitution
                  damppar_pp(1:2,l,k) = damppar_pp(1:2,k,l)

!              //   linear, exponential thole and gaussian forms
               else if ( ( dampform(1:4) .eq. 'LIN ' ) .or. &
     &                   ( dampform(1:4) .eq. 'EXP ' ) .or. &
     &                   ( dampform(1:4) .eq. 'GAU ' ) ) then

!                 //   go back one line
                  backspace( iounit )

!                 //   read interaction, atom pair, function
                  read( iounit, *, iostat=ierr ) &
     &               dampint, k, l, dampform, damppar_pp(1:1,k,l)

!                 //   check error
                  if ( ierr .ne. 0 ) exit

!                 //   substitution
                  damppar_pp(1:1,l,k) = damppar_pp(1:1,k,l)

!              //   damping form
               end if

!           //   interactions
            end if

!        //   loop of damping interactions
         end do

!     //   if read correctly
      end if

!     //   close file
      close( iounit )

!     //   error handling
      call error_handling &
     &   ( ierr, 'subroutine force_pol_ewald_setup', 32 )

!-----------------------------------------------------------------------
!     /*                                                              */
!     /*   Ewald parameters for charge-charge interactions            */
!     /*                                                              */
!     /*      eps_ewald    =  required accuracy                       */
!     /*      ratio_ewald  =  ratio of calculation time               */
!     /*                      real space / Fourier space              */
!     /*                                                              */
!     /*   ratio_ewald depends on the error function routine          */
!     /*                                                              */
!-----------------------------------------------------------------------

      open (iounit, file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<ewald>', 7, iounit, ierr )

         if ( ierr .eq. 0 ) then
            read( iounit, '(a)' ) char_line
            read( char_line, *, iostat=ierr ) &
     &         eps_ewald, ratio_ewald, ioption_ewald
            if ( ierr .ne. 0 ) then
               read( char_line, *, iostat=ierr ) &
     &            eps_ewald, ratio_ewald
               ioption_ewald = 0
               if ( ierr .ne. 0 ) then
                  eps_ewald    =  1.d-08
                  ratio_ewald  =  4.d+00
                  ioption_ewald =  0
               end if
            end if
         else
            eps_ewald    =  1.d-08
            ratio_ewald  =  4.d+00
            ioption_ewald =  0
         end if

      close(iounit)

!-----------------------------------------------------------------------
!     /*   parameter s:  solve exp(-s*s)/(s*s) = eps_ewald            */
!-----------------------------------------------------------------------

      snew = 0.d0

      do i = 1, 1000
         sold = snew
         snew = exp(-snew)*(snew+1.d0)/(eps_ewald+exp(-snew))
         sdif = abs(sold/snew - 1.d0)
         if ( sdif .lt. 1.d-15 ) exit
      end do

      s_ewald = sqrt(snew)

!-----------------------------------------------------------------------
!     /*   alpha:  exponent of fictitious Gaussian charge             */
!-----------------------------------------------------------------------

      alpha_ewald = (ratio_ewald*natom*pi**3/volume**2)**(1.d0/6.d0)

!-----------------------------------------------------------------------
!     /*   rcut:  cut off distance of real space sum                  */
!-----------------------------------------------------------------------

      rcut_ewald = s_ewald/alpha_ewald

!-----------------------------------------------------------------------
!     /*   lmax:  cut off in Fourier space sum                        */
!     /*          kmax = 2*pi/boxl*lmax                               */
!-----------------------------------------------------------------------

      absx = sqrt ( box(1,1)*box(1,1) &
     &            + box(2,1)*box(2,1) &
     &            + box(3,1)*box(3,1) )
      absy = sqrt ( box(1,2)*box(1,2) &
     &            + box(2,2)*box(2,2) &
     &            + box(3,2)*box(3,2) )
      absz = sqrt ( box(1,3)*box(1,3) &
     &            + box(2,3)*box(2,3) &
     &            + box(3,3)*box(3,3) )

      lmax_ewald(1) = int(s_ewald*absx*alpha_ewald/pi) + 1
      lmax_ewald(2) = int(s_ewald*absy*alpha_ewald/pi) + 1
      lmax_ewald(3) = int(s_ewald*absz*alpha_ewald/pi) + 1

!-----------------------------------------------------------------------
!     /*   extention of simulation box in real space sum              */
!-----------------------------------------------------------------------

      absa = sqrt ( boxinv(1,1)*boxinv(1,1) &
     &            + boxinv(1,2)*boxinv(1,2) &
     &            + boxinv(1,3)*boxinv(1,3) )
      absb = sqrt ( boxinv(2,1)*boxinv(2,1) &
     &            + boxinv(2,2)*boxinv(2,2) &
     &            + boxinv(2,3)*boxinv(2,3) )
      absc = sqrt ( boxinv(3,1)*boxinv(3,1) &
     &            + boxinv(3,2)*boxinv(3,2) &
     &            + boxinv(3,3)*boxinv(3,3) )

      nbox_ewald(1) = int(2.d0*rcut_ewald*absa) + 1
      nbox_ewald(2) = int(2.d0*rcut_ewald*absb) + 1
      nbox_ewald(3) = int(2.d0*rcut_ewald*absc) + 1

!-----------------------------------------------------------------------
!     /*                                                              */
!     /*   Ewald parameters for charge-dipoles and dipole-dipoles     */
!     /*                                                              */
!     /*      eps_ewpol    =  required accuracy                       */
!     /*      ratio_ewpol  =  ratio of calculation time               */
!     /*                      real space / Fourier space              */
!     /*                                                              */
!     /*   ratio_ewpol depends on the error function routine          */
!     /*                                                              */
!-----------------------------------------------------------------------

!     //   open file
      open ( iounit, file = 'mm.dat' )

!     /*   tag   */
      call search_tag ( '<ewpol>', 7, iounit, ierr )

!     //   if tag found
      if ( ierr .eq. 0 ) then

!        //   read data
         read( iounit, *, iostat=ierr ) eps_ewpol, ratio_ewpol

!        //   on error set to default values
         if ( ierr .ne. 0 ) then
            eps_ewpol    =  1.d-08
            ratio_ewpol  =  4.d+00
         end if

!     //   if tag not found
      else

!        //   set to default values
         eps_ewpol    =  1.d-08
         ratio_ewpol  =  4.d+00

!     //   end of if statement
      end if

!     //   close file
      close( iounit )

!-----------------------------------------------------------------------
!     /*   parameter s:  solve exp(-s*s)/(s*s) = eps_ewpol            */
!-----------------------------------------------------------------------

      snew = 0.d0

      do i = 1, 1000
         sold = snew
         snew = exp(-snew) * (snew + 1.d0) / (eps_ewpol + exp(-snew))
         sdif = abs( sold / snew - 1.d0 )
         if ( sdif .lt. 1.d-15 ) exit
      end do

      s_ewpol = sqrt(snew)

!-----------------------------------------------------------------------
!     /*   alpha:  exponent of fictitious Gaussian charge             */
!-----------------------------------------------------------------------

      alpha_ewpol = (ratio_ewpol*natom*pi**3/volume**2)**(1.d0/6.d0)

!-----------------------------------------------------------------------
!     /*   rcut:  cut off distance of real space sum                  */
!-----------------------------------------------------------------------

      rcut_ewpol = s_ewpol / alpha_ewpol

!-----------------------------------------------------------------------
!     /*   lmax:  cut off in Fourier space sum                        */
!     /*          kmax = 2*pi/boxl*lmax                               */
!-----------------------------------------------------------------------

      absx = sqrt ( box(1,1)*box(1,1) &
     &            + box(2,1)*box(2,1) &
     &            + box(3,1)*box(3,1) )
      absy = sqrt ( box(1,2)*box(1,2) &
     &            + box(2,2)*box(2,2) &
     &            + box(3,2)*box(3,2) )
      absz = sqrt ( box(1,3)*box(1,3) &
     &            + box(2,3)*box(2,3) &
     &            + box(3,3)*box(3,3) )

      lmax_ewpol(1) = int( s_ewpol * absx * alpha_ewpol / pi ) + 1
      lmax_ewpol(2) = int( s_ewpol * absy * alpha_ewpol / pi ) + 1
      lmax_ewpol(3) = int( s_ewpol * absz * alpha_ewpol / pi ) + 1

!-----------------------------------------------------------------------
!     /*   extention of simulation box in real space sum              */
!-----------------------------------------------------------------------

      absa = sqrt ( boxinv(1,1) * boxinv(1,1) &
     &            + boxinv(1,2) * boxinv(1,2) &
     &            + boxinv(1,3) * boxinv(1,3) )
      absb = sqrt ( boxinv(2,1) * boxinv(2,1) &
     &            + boxinv(2,2) * boxinv(2,2) &
     &            + boxinv(2,3) * boxinv(2,3) )
      absc = sqrt ( boxinv(3,1) * boxinv(3,1) &
     &            + boxinv(3,2) * boxinv(3,2) &
     &            + boxinv(3,3) * boxinv(3,3) )

      nbox_ewpol(1) = int( 2.d0 * rcut_ewpol * absa ) + 1
      nbox_ewpol(2) = int( 2.d0 * rcut_ewpol * absb ) + 1
      nbox_ewpol(3) = int( 2.d0 * rcut_ewpol * absc ) + 1

!-----------------------------------------------------------------------
!     /*   set up bonded charge pairs                                 */
!-----------------------------------------------------------------------

!     //   open file
      open ( iounit, file = 'mm.dat' )

!     /*   tag   */
      call search_tag ( '<nbcp>', 6, iounit, ierr )

!     /*   number of bonded charge pairs   */
      if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr ) nbcp

!     //   close file
      close( iounit )

!     //   if bonded charge pairs are absent
      if ( ierr .ne. 0 ) nbcp = 0

!     //   if bonded charge pairs are present
      if ( nbcp .ge. 1 ) then

!        //   memory allocation
         if ( .not. allocated(  i_bcp ) ) &
     &      allocate(  i_bcp(nbcp))
         if ( .not. allocated(  j_bcp ) ) &
     &      allocate(  j_bcp(nbcp))
         if ( .not. allocated(  factor_bcp ) ) &
     &      allocate(  factor_bcp(nbcp))

!        //   open file
         open ( iounit, file = 'mm.dat' )

!        /*   tag   */
         call search_tag ( '<nbcp>', 6, iounit, ierr )

!        /*   number of bonded charge pairs   */
         if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr )

!        //   read atomic pair and scaling factor
         do k = 1, nbcp
            read( iounit, *, iostat=ierr ) &
     &         i_bcp(k), j_bcp(k), factor_bcp(k)
         end do

!        //   close file
        close( iounit )

!        //   error handling
         call error_handling &
     &      ( ierr, 'subroutine force_pol_coulomb_setup', 34 )

!     //   if bonded charge pairs are present
      end if

      return
      end





!***********************************************************************
      subroutine force_pol_induced
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, nbead

      use mm_variables, only : &
     &   px, py, pz, pol, ncharge, npol, i_q

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, k, m

      real(8) :: poli, factor

      real(8) :: tiny = 1.d-12

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( npol .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   induced dipoles                                            */
!-----------------------------------------------------------------------

      do m = 1, nbead

         do k = 1, ncharge

            poli = pol(k)

            if ( abs(poli) .lt. tiny ) cycle

            factor = 0.5d0 / poli

            i = i_q(k)

            pot(m) = pot(m) + factor * px(i,m) * px(i,m)
            pot(m) = pot(m) + factor * py(i,m) * py(i,m)
            pot(m) = pot(m) + factor * pz(i,m) * pz(i,m)

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_pol_coulomb_pair
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, nbead

      use mm_variables, only : &
     &   q, px, py, pz, pol, factor_bcp, ncharge, npol, i_q, i_bcp, &
     &   j_bcp, nbcp

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m

      real(8) :: qi, qj, xij, yij, zij, r, rinv, uij, fxi, fyi, fzi, &
     &           rinv3, rinv5, rinv7, pxi, pyi, pzi, pxj, pyj, &
     &           pzj, rinv2, rinv4, rinv6, a1qq, a3qq, a3qp, a5qp, a3pp, &
     &           a5pp, s0qq, s1qq, s0qp, s1qp, s0pp, s1pp, &
     &           txqq, tyqq, tzqq, txqp, tyqp, tzqp, txxqp, txyqp, &
     &           txzqp, tyxqp, tyyqp, tyzqp, tzxqp, tzyqp, tzzqp, &
     &           txxpp, txypp, txzpp, tyxpp, tyypp, tyzpp, tzxpp, &
     &           tzypp, tzzpp, txxxpp, txxypp, txxzpp, txyxpp, &
     &           txyypp, txyzpp, txzxpp, txzypp, txzzpp, tyxxpp, &
     &           tyxypp, tyxzpp, tyyxpp, tyyypp, tyyzpp, tyzxpp, &
     &           tyzypp, tyzzpp, tzxxpp, tzxypp, tzxzpp, tzyxpp, &
     &           tzyypp, tzyzpp, tzzxpp, tzzypp, tzzzpp, b5pp, b7pp, &
     &           poli, polj, txpq, typq, tzpq, txxpq, &
     &           txypq, txzpq, tyxpq, tyypq, tyzpq, tzxpq, tzypq, tzzpq, &
     &           s0pq, s1pq, a3pq, a5pq, t0pp, t1pp, duij, factor, rij

      real(8) :: tiny = 1.d-12

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( ( ncharge .eq. 0 ) .and. ( npol .eq. 0 ) ) return

!-----------------------------------------------------------------------
!     /*   main loop:  direct sum between all charges                 */
!-----------------------------------------------------------------------

      do m = 1, nbead

         do k = 1, ncharge

         do l = k+1, ncharge

            i = i_q(k)
            j = i_q(l)

            qi = q(i)
            qj = q(j)

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom ( xij, yij, zij )

            r = sqrt( xij*xij + yij*yij + zij*zij )

            rinv = 1.d0 / r
            rinv2 = rinv * rinv
            rinv3 = rinv * rinv2

            call dampfunc_qq( r, k, l, s0qq, s1qq )

            a1qq = s0qq * rinv
            a3qq = a1qq * rinv2 - s1qq * rinv2

            txqq = - xij * a3qq
            tyqq = - yij * a3qq
            tzqq = - zij * a3qq

            uij = + a1qq * qi * qj

            fxi = - qi * qj * txqq
            fyi = - qi * qj * tyqq
            fzi = - qi * qj * tzqq

            poli = pol(k)
            polj = pol(l)

            if ( abs(poli)+abs(polj) .gt. 2.d0*tiny ) then

               call dampfunc_qp( r, k, l, s0qp, s1qp )
               call dampfunc_qp( r, l, k, s0pq, s1pq )
               call dampfunc_pp( r, k, l, s0pp, s1pp, t0pp, t1pp )

               pxi = px(i,m)
               pyi = py(i,m)
               pzi = pz(i,m)
               pxj = px(j,m)
               pyj = py(j,m)
               pzj = pz(j,m)

               rinv4 = rinv2 * rinv2
               rinv5 = rinv2 * rinv3
               rinv6 = rinv2 * rinv4
               rinv7 = rinv2 * rinv5

               a3qp = s0qp * rinv3
               a5qp = 3.d0 * a3qp * rinv2 - s1qp * rinv4

               a3pq = s0pq * rinv3
               a5pq = 3.d0 * a3pq * rinv2 - s1pq * rinv4

               a3pp = s0pp * rinv3
               a5pp = 3.d0 * s0pp * rinv5 - s1pp * rinv4

               b5pp = 3.d0 * t0pp * rinv5
               b7pp = 5.d0 * b5pp * rinv2 - 3.d0 * t1pp * rinv6

               txqp = - xij * a3qp
               tyqp = - yij * a3qp
               tzqp = - zij * a3qp

               txpq = - xij * a3pq
               typq = - yij * a3pq
               tzpq = - zij * a3pq

               txxqp = xij * xij * a5qp - a3qp
               txyqp = xij * yij * a5qp
               txzqp = xij * zij * a5qp
               tyxqp = txyqp
               tyyqp = yij * yij * a5qp - a3qp
               tyzqp = yij * zij * a5qp
               tzxqp = txzqp
               tzyqp = tyzqp
               tzzqp = zij * zij * a5qp - a3qp

               txxpq = xij * xij * a5pq - a3pq
               txypq = xij * yij * a5pq
               txzpq = xij * zij * a5pq
               tyxpq = txypq
               tyypq = yij * yij * a5pq - a3pq
               tyzpq = yij * zij * a5pq
               tzxpq = txzpq
               tzypq = tyzpq
               tzzpq = zij * zij * a5pq - a3pq

               txxpp = xij * xij * b5pp - a3pp
               txypp = xij * yij * b5pp
               txzpp = xij * zij * b5pp
               tyxpp = yij * xij * b5pp
               tyypp = yij * yij * b5pp - a3pp
               tyzpp = yij * zij * b5pp
               tzxpp = zij * xij * b5pp
               tzypp = zij * yij * b5pp
               tzzpp = zij * zij * b5pp - a3pp

               txxxpp = - b7pp*xij*xij*xij + ( 2.d0*b5pp + a5pp )*xij
               txxypp = - b7pp*xij*xij*yij + b5pp*yij
               txxzpp = - b7pp*xij*xij*zij + b5pp*zij
               txyxpp = - b7pp*xij*yij*xij + b5pp*yij
               txyypp = - b7pp*xij*yij*yij + a5pp*xij
               txyzpp = - b7pp*xij*yij*zij
               txzxpp = - b7pp*xij*zij*xij + b5pp*zij
               txzypp = - b7pp*xij*zij*yij
               txzzpp = - b7pp*xij*zij*zij + a5pp*xij

               tyxxpp = - b7pp*yij*xij*xij + a5pp*yij
               tyxypp = - b7pp*yij*xij*yij + b5pp*xij
               tyxzpp = - b7pp*yij*xij*zij
               tyyxpp = - b7pp*yij*yij*xij + b5pp*xij
               tyyypp = - b7pp*yij*yij*yij + ( 2.d0*b5pp + a5pp )*yij
               tyyzpp = - b7pp*yij*yij*zij + b5pp*zij
               tyzxpp = - b7pp*yij*zij*xij
               tyzypp = - b7pp*yij*zij*yij + b5pp*zij
               tyzzpp = - b7pp*yij*zij*zij + a5pp*yij

               tzxxpp = - b7pp*zij*xij*xij + a5pp*zij
               tzxypp = - b7pp*zij*xij*yij
               tzxzpp = - b7pp*zij*xij*zij + b5pp*xij
               tzyxpp = - b7pp*zij*yij*xij
               tzyypp = - b7pp*zij*yij*yij + a5pp*zij
               tzyzpp = - b7pp*zij*yij*zij + b5pp*yij
               tzzxpp = - b7pp*zij*zij*xij + b5pp*xij
               tzzypp = - b7pp*zij*zij*yij + b5pp*yij
               tzzzpp = - b7pp*zij*zij*zij + ( 2.d0*b5pp + a5pp )*zij

               uij = uij - qi * ( txqp*pxj + tyqp*pyj + tzqp*pzj ) &
     &                   + qj * ( txpq*pxi + typq*pyi + tzpq*pzi )

               uij = uij - pxi * ( txxpp*pxj + txypp*pyj + txzpp*pzj ) &
     &                   - pyi * ( tyxpp*pxj + tyypp*pyj + tyzpp*pzj ) &
     &                   - pzi * ( tzxpp*pxj + tzypp*pyj + tzzpp*pzj )

               fxi = fxi + qi * ( txxqp*pxj + txyqp*pyj + txzqp*pzj ) &
     &                   - qj * ( txxpq*pxi + txypq*pyi + txzpq*pzi )
               fyi = fyi + qi * ( txyqp*pxj + tyyqp*pyj + tyzqp*pzj ) &
     &                   - qj * ( txypq*pxi + tyypq*pyi + tyzpq*pzi )
               fzi = fzi + qi * ( txzqp*pxj + tyzqp*pyj + tzzqp*pzj ) &
     &                   - qj * ( txzpq*pxi + tyzpq*pyi + tzzpq*pzi ) 

               fxi = fxi &
     &             + pxi * ( txxxpp*pxj + txxypp*pyj + txxzpp*pzj ) &
     &             + pyi * ( txyxpp*pxj + txyypp*pyj + txyzpp*pzj ) &
     &             + pzi * ( txzxpp*pxj + txzypp*pyj + txzzpp*pzj )
               fyi = fyi &
     &             + pxi * ( tyxxpp*pxj + tyxypp*pyj + tyxzpp*pzj ) &
     &             + pyi * ( tyyxpp*pxj + tyyypp*pyj + tyyzpp*pzj ) &
     &             + pzi * ( tyzxpp*pxj + tyzypp*pyj + tyzzpp*pzj )
               fzi = fzi &
     &             + pxi * ( tzxxpp*pxj + tzxypp*pyj + tzxzpp*pzj ) &
     &             + pyi * ( tzyxpp*pxj + tzyypp*pyj + tzyzpp*pzj ) &
     &             + pzi * ( tzzxpp*pxj + tzzypp*pyj + tzzzpp*pzj )

            end if

            pot(m) = pot(m) + uij

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi * xij
            vir(1,2) = vir(1,2) + fxi * yij
            vir(1,3) = vir(1,3) + fxi * zij
            vir(2,1) = vir(2,1) + fyi * xij
            vir(2,2) = vir(2,2) + fyi * yij
            vir(2,3) = vir(2,3) + fyi * zij
            vir(3,1) = vir(3,1) + fzi * xij
            vir(3,2) = vir(3,2) + fzi * yij
            vir(3,3) = vir(3,3) + fzi * zij

         end do
         end do

      end do

!-----------------------------------------------------------------------
!     /*   main loop:  subtract bonded charge pairs                   */
!-----------------------------------------------------------------------

      do m = 1, nbead

         do k = 1, nbcp

            i  = i_bcp(k)
            j  = j_bcp(k)

            factor = factor_bcp(k)

            qi = q(i)
            qj = q(j)

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom( xij, yij, zij )

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            rinv = 1.d0/rij

            uij = (factor - 1.d0) * qi*qj*rinv

            pot(m) = pot(m) + uij

            duij = - uij*rinv

            fxi = - duij*xij*rinv
            fyi = - duij*yij*rinv
            fzi = - duij*zij*rinv

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_pol_ewald_rs
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pot, fx, fy, fz, vir, box, nbead

      use mm_variables, only : &
     &   q, rcut_ewald, alpha_ewald, bigbox, bigboxinv, factor_bcp, &
     &   px, py, pz, pol, rcut_ewpol, alpha_ewpol, nbox_ewpol, &
     &   nbox_ewald, i_q, ncharge, npol, i_bcp, j_bcp, nbcp

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, jx, jy, jz, j2

      real(8) :: qi, qj, xij, yij, zij, r, r2, rinv, uij, fxi, fyi, fzi, &
     &           rinv3, rinv5, rinv7, pxi, pyi, pzi, pxj, pyj, duij, &
     &           pzj, rinv2, rinv4, rinv6, a1qq, a3qq, a3qp, a5qp, a3pp, &
     &           a5pp, s0qq, s1qq, s0qp, s1qp, s0pp, s1pp, b5pp, rij, &
     &           b1er, a1er, a3er, a5er, a7er, ar, erfc_ar, erf_0, &
     &           erf_1, alpha_ewpol2, alpha_ewpol4, poli, polj, factor, &
     &           txqq, tyqq, tzqq, txqp, tyqp, tzqp, txxqp, &
     &           txyqp, txzqp, tyxqp, tyyqp, tyzqp, tzxqp, tzyqp, tzzqp, &
     &           txxpp, txypp, txzpp, tyxpp, tyypp, tyzpp, tzxpp, b5ps, &
     &           tzypp, tzzpp, txxxpp, txxypp, txxzpp, txyxpp, txyypp, &
     &           txyzpp, txzxpp, txzypp, txzzpp, tyxxpp, tyxypp, tyxzpp, &
     &           tyyxpp, tyyypp, tyyzpp, tyzxpp, tyzypp, tyzzpp, tzxxpp, &
     &           tzxypp, tzxzpp, tzyxpp, tzyypp, tzyzpp, tzzxpp, tzzypp, &
     &           tzzzpp, rcut_ewpol2, rcut_ewald2, aij, bij, cij, b7ps, &
     &           a3qs, a5qs, a3ps, a5ps, txpq, typq, tzpq, txxpq, &
     &           txypq, txzpq, tyxpq, tyypq, tyzpq, tzxpq, tzypq, tzzpq, &
     &           s0pq, s1pq, a3pq, a5pq, a3sq, a5sq, t0pp, t1pp, b7pp

!      real(8) :: rin_damp, rout_damp, swf, dswf

      real(8) :: tiny = 1.d-12

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( ( ncharge .eq. 0 ) .and. ( npol .eq. 0 ) ) return

!-----------------------------------------------------------------------
!     /*   parameters                                                 */
!-----------------------------------------------------------------------

      rcut_ewald2 = rcut_ewald*rcut_ewald
      rcut_ewpol2 = rcut_ewpol*rcut_ewpol

      alpha_ewpol2 = alpha_ewpol * alpha_ewpol
      alpha_ewpol4 = alpha_ewpol2 * alpha_ewpol2

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         bigbox(:,1) = dble(nbox_ewald(1)) * box(:,1)
         bigbox(:,2) = dble(nbox_ewald(2)) * box(:,2)
         bigbox(:,3) = dble(nbox_ewald(3)) * box(:,3)

         call inv3 ( bigbox, bigboxinv )

         do k = 1, ncharge

            i = i_q(k)

            qi = q(i)

            if ( qi .eq. 0.d0 ) cycle

            do l = 1, ncharge

               j = i_q(l)

               qj = q(j)

               if ( qj .eq. 0.d0 ) cycle

               do jx = 0, nbox_ewald(1)-1
               do jy = 0, nbox_ewald(2)-1
               do jz = 0, nbox_ewald(3)-1

                  j2 = jx*jx + jy*jy + jz*jz

                  if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

                  xij = x(i,m) - x(j,m)
                  yij = y(i,m) - y(j,m)
                  zij = z(i,m) - z(j,m)

                  xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
                  yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
                  zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

                  aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &                + bigboxinv(1,3)*zij
                  bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &                + bigboxinv(2,3)*zij
                  cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &                + bigboxinv(3,3)*zij

                  aij = aij - nint(aij)
                  bij = bij - nint(bij)
                  cij = cij - nint(cij)

                  xij = bigbox(1,1)*aij + bigbox(1,2)*bij &
     &                + bigbox(1,3)*cij
                  yij = bigbox(2,1)*aij + bigbox(2,2)*bij &
     &                + bigbox(2,3)*cij
                  zij = bigbox(3,1)*aij + bigbox(3,2)*bij &
     &                + bigbox(3,3)*cij

                  r2 = xij*xij + yij*yij + zij*zij

                  if ( r2 .gt. rcut_ewald2 ) cycle

                  r = sqrt(r2)

                  rinv  = 1.d0/r
                  rinv2 = rinv*rinv
                  rinv3 = rinv*rinv2

                  call dampfunc_qq( r, k, l, s0qq, s1qq )

!                  call getswf( r, rin_damp, rout_damp, swf, dswf )
!
!                  s0qq = swf*s0qq - swf + 1.d0
!                  s1qq = swf*s1qq + dswf*s0qq - dswf

                  ar = alpha_ewald*r

                  erfc_ar = 1.d0 - erf_0(ar)

                  a1qq = ( erfc_ar + s0qq - 1.d0 ) * rinv
                  a3qq = ( a1qq + alpha_ewald*erf_1(ar) - s1qq ) * rinv2

                  txqq = - xij * a3qq
                  tyqq = - yij * a3qq
                  tzqq = - zij * a3qq

                  uij = + qi * qj * a1qq

                  fxi = - qi * qj * txqq
                  fyi = - qi * qj * tyqq
                  fzi = - qi * qj * tzqq

                  pot(m) = pot(m) + 0.5d0*uij

                  fx(i,m) = fx(i,m) + fxi
                  fy(i,m) = fy(i,m) + fyi
                  fz(i,m) = fz(i,m) + fzi

                  vir(1,1) = vir(1,1) + 0.5d0*fxi*xij
                  vir(1,2) = vir(1,2) + 0.5d0*fxi*yij
                  vir(1,3) = vir(1,3) + 0.5d0*fxi*zij
                  vir(2,1) = vir(2,1) + 0.5d0*fyi*xij
                  vir(2,2) = vir(2,2) + 0.5d0*fyi*yij
                  vir(2,3) = vir(2,3) + 0.5d0*fyi*zij
                  vir(3,1) = vir(3,1) + 0.5d0*fzi*xij
                  vir(3,2) = vir(3,2) + 0.5d0*fzi*yij
                  vir(3,3) = vir(3,3) + 0.5d0*fzi*zij

               end do
               end do
               end do

            end do

         end do

         bigbox(:,1) = dble(nbox_ewpol(1)) * box(:,1)
         bigbox(:,2) = dble(nbox_ewpol(2)) * box(:,2)
         bigbox(:,3) = dble(nbox_ewpol(3)) * box(:,3)

         call inv3 ( bigbox, bigboxinv )

         do k = 1, ncharge

            poli = pol(k)

            i = i_q(k)

            qi = q(i)

            do l = 1, ncharge

               polj = pol(l)

               if ( abs(poli)+abs(polj) .lt. 2.d0*tiny ) cycle

               j = i_q(l)

               qj = q(j)

               do jx = 0, nbox_ewpol(1)-1
               do jy = 0, nbox_ewpol(2)-1
               do jz = 0, nbox_ewpol(3)-1

                  j2 = jx*jx + jy*jy + jz*jz

                  if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

                  xij = x(i,m) - x(j,m)
                  yij = y(i,m) - y(j,m)
                  zij = z(i,m) - z(j,m)

                  xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
                  yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
                  zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

                  aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &                + bigboxinv(1,3)*zij
                  bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &                + bigboxinv(2,3)*zij
                  cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &                + bigboxinv(3,3)*zij

                  aij = aij - nint(aij)
                  bij = bij - nint(bij)
                  cij = cij - nint(cij)

                  xij = bigbox(1,1)*aij + bigbox(1,2)*bij &
     &                + bigbox(1,3)*cij
                  yij = bigbox(2,1)*aij + bigbox(2,2)*bij &
     &                + bigbox(2,3)*cij
                  zij = bigbox(3,1)*aij + bigbox(3,2)*bij &
     &                + bigbox(3,3)*cij

                  r2 = xij*xij + yij*yij + zij*zij

                  if ( r2 .gt. rcut_ewpol2 ) cycle

                  r = sqrt(r2)

                  rinv  = 1.d0/r
                  rinv2 = rinv*rinv
                  rinv3 = rinv*rinv2
                  rinv4 = rinv2 * rinv2
                  rinv5 = rinv2 * rinv3
                  rinv6 = rinv2 * rinv4
                  rinv7 = rinv2 * rinv5

                  call dampfunc_qp( r, k, l, s0qp, s1qp )
                  call dampfunc_qp( r, l, k, s0pq, s1pq )
                  call dampfunc_pp( r, k, l, s0pp, s1pp, t0pp, t1pp )

!                  call getswf( r, rin_damp, rout_damp, swf, dswf )
!
!                  s0qp = swf*s0qp - swf + 1.d0
!                  s0pq = swf*s0pq - swf + 1.d0
!                  s0pp = swf*s0pp - swf + 1.d0
!                  t0pp = swf*t0pp - swf + 1.d0
!
!                  s1qp = swf*s1qp + dswf*s0qp - dswf
!                  s1pq = swf*s1pq + dswf*s0pq - dswf
!                  s1pp = swf*s1pp + dswf*s0qp - dswf
!                  t1pp = swf*t1pp + dswf*t0qp - dswf

                  pxi = px(i,m)
                  pyi = py(i,m)
                  pzi = pz(i,m)
                  pxj = px(j,m)
                  pyj = py(j,m)
                  pzj = pz(j,m)

                  ar = alpha_ewpol * r

                  erfc_ar = 1.d0 - erf_0(ar)

                  b1er = alpha_ewpol * erf_1(ar)

                  a1er = erfc_ar * rinv
                  a3er = ( a1er + b1er ) * rinv2
                  a5er = ( 3.d0*a3er + 2.d0*alpha_ewpol2*b1er ) * rinv2
                  a7er = ( 5.d0*a5er + 4.d0*alpha_ewpol4*b1er ) * rinv2

                  a3qs = ( s0qp - 1.d0 ) * rinv3
                  a5qs = 3.d0 * a3qs * rinv2 - s1qp * rinv4

                  a3sq = ( s0pq - 1.d0 ) * rinv3
                  a5sq = 3.d0 * a3sq * rinv2 - s1pq * rinv4

                  a3qp = a3er + a3qs
                  a5qp = a5er + a5qs

                  a3pq = a3er + a3sq
                  a5pq = a5er + a5sq

                  a3ps = ( s0pp - 1.d0 ) * rinv3
                  a5ps = 3.d0 * ( s0pp - 1.d0 ) * rinv5 - s1pp * rinv4

                  b5ps = 3.d0 * ( t0pp - 1.d0 ) * rinv5
                  b7ps = 5.d0 * b5ps * rinv2 - 3.d0 * t1pp * rinv6

                  a3pp = a3er + a3ps
                  a5pp = a5er + a5ps

                  b5pp = a5er + b5ps
                  b7pp = a7er + b7ps

                  txqp = - xij * a3qp
                  tyqp = - yij * a3qp
                  tzqp = - zij * a3qp

                  txpq = - xij * a3pq
                  typq = - yij * a3pq
                  tzpq = - zij * a3pq

                  txxqp = xij * xij * a5qp - a3qp
                  txyqp = xij * yij * a5qp
                  txzqp = xij * zij * a5qp
                  tyxqp = txyqp
                  tyyqp = yij * yij * a5qp - a3qp
                  tyzqp = yij * zij * a5qp
                  tzxqp = txzqp
                  tzyqp = tyzqp
                  tzzqp = zij * zij * a5qp - a3qp

                  txxpq = xij * xij * a5pq - a3pq
                  txypq = xij * yij * a5pq
                  txzpq = xij * zij * a5pq
                  tyxpq = txypq
                  tyypq = yij * yij * a5pq - a3pq
                  tyzpq = yij * zij * a5pq
                  tzxpq = txzpq
                  tzypq = tyzpq
                  tzzpq = zij * zij * a5pq - a3pq

                  txxpp = xij * xij * b5pp - a3pp
                  txypp = xij * yij * b5pp
                  txzpp = xij * zij * b5pp
                  tyxpp = yij * xij * b5pp
                  tyypp = yij * yij * b5pp - a3pp
                  tyzpp = yij * zij * b5pp
                  tzxpp = zij * xij * b5pp
                  tzypp = zij * yij * b5pp
                  tzzpp = zij * zij * b5pp - a3pp

                  txxxpp = - b7pp*xij*xij*xij + ( 2.d0*b5pp + a5pp )*xij
                  txxypp = - b7pp*xij*xij*yij + b5pp*yij
                  txxzpp = - b7pp*xij*xij*zij + b5pp*zij
                  txyxpp = - b7pp*xij*yij*xij + b5pp*yij
                  txyypp = - b7pp*xij*yij*yij + a5pp*xij
                  txyzpp = - b7pp*xij*yij*zij
                  txzxpp = - b7pp*xij*zij*xij + b5pp*zij
                  txzypp = - b7pp*xij*zij*yij
                  txzzpp = - b7pp*xij*zij*zij + a5pp*xij

                  tyxxpp = - b7pp*yij*xij*xij + a5pp*yij
                  tyxypp = - b7pp*yij*xij*yij + b5pp*xij
                  tyxzpp = - b7pp*yij*xij*zij
                  tyyxpp = - b7pp*yij*yij*xij + b5pp*xij
                  tyyypp = - b7pp*yij*yij*yij + ( 2.d0*b5pp + a5pp )*yij
                  tyyzpp = - b7pp*yij*yij*zij + b5pp*zij
                  tyzxpp = - b7pp*yij*zij*xij
                  tyzypp = - b7pp*yij*zij*yij + b5pp*zij
                  tyzzpp = - b7pp*yij*zij*zij + a5pp*yij

                  tzxxpp = - b7pp*zij*xij*xij + a5pp*zij
                  tzxypp = - b7pp*zij*xij*yij
                  tzxzpp = - b7pp*zij*xij*zij + b5pp*xij
                  tzyxpp = - b7pp*zij*yij*xij
                  tzyypp = - b7pp*zij*yij*yij + a5pp*zij
                  tzyzpp = - b7pp*zij*yij*zij + b5pp*yij
                  tzzxpp = - b7pp*zij*zij*xij + b5pp*xij
                  tzzypp = - b7pp*zij*zij*yij + b5pp*yij
                  tzzzpp = - b7pp*zij*zij*zij + ( 2.d0*b5pp + a5pp )*zij

                  uij = &
     &               - qi * ( txqp*pxj + tyqp*pyj + tzqp*pzj ) &
     &               + qj * ( txpq*pxi + typq*pyi + tzpq*pzi )

                  uij = uij &
     &               - pxi * ( txxpp*pxj + txypp*pyj + txzpp*pzj ) &
     &               - pyi * ( tyxpp*pxj + tyypp*pyj + tyzpp*pzj ) &
     &               - pzi * ( tzxpp*pxj + tzypp*pyj + tzzpp*pzj )

                  fxi = &
     &               + qi * ( txxqp*pxj + txyqp*pyj + txzqp*pzj ) &
     &               - qj * ( txxpq*pxi + txypq*pyi + txzpq*pzi )
                  fyi = &
     &               + qi * ( txyqp*pxj + tyyqp*pyj + tyzqp*pzj ) &
     &               - qj * ( txypq*pxi + tyypq*pyi + tyzpq*pzi )
                  fzi = &
     &               + qi * ( txzqp*pxj + tyzqp*pyj + tzzqp*pzj ) &
     &               - qj * ( txzpq*pxi + tyzpq*pyi + tzzpq*pzi ) 

                  fxi = fxi &
     &               + pxi * ( txxxpp*pxj + txxypp*pyj + txxzpp*pzj ) &
     &               + pyi * ( txyxpp*pxj + txyypp*pyj + txyzpp*pzj ) &
     &               + pzi * ( txzxpp*pxj + txzypp*pyj + txzzpp*pzj )
                  fyi = fyi &
     &               + pxi * ( tyxxpp*pxj + tyxypp*pyj + tyxzpp*pzj ) &
     &               + pyi * ( tyyxpp*pxj + tyyypp*pyj + tyyzpp*pzj ) &
     &               + pzi * ( tyzxpp*pxj + tyzypp*pyj + tyzzpp*pzj )
                  fzi = fzi &
     &               + pxi * ( tzxxpp*pxj + tzxypp*pyj + tzxzpp*pzj ) &
     &               + pyi * ( tzyxpp*pxj + tzyypp*pyj + tzyzpp*pzj ) &
     &               + pzi * ( tzzxpp*pxj + tzzypp*pyj + tzzzpp*pzj )

                  pot(m) = pot(m) + 0.5d0*uij

                  fx(i,m) = fx(i,m) + fxi
                  fy(i,m) = fy(i,m) + fyi
                  fz(i,m) = fz(i,m) + fzi

                  vir(1,1) = vir(1,1) + 0.5d0*fxi*xij
                  vir(1,2) = vir(1,2) + 0.5d0*fxi*yij
                  vir(1,3) = vir(1,3) + 0.5d0*fxi*zij
                  vir(2,1) = vir(2,1) + 0.5d0*fyi*xij
                  vir(2,2) = vir(2,2) + 0.5d0*fyi*yij
                  vir(2,3) = vir(2,3) + 0.5d0*fyi*zij
                  vir(3,1) = vir(3,1) + 0.5d0*fzi*xij
                  vir(3,2) = vir(3,2) + 0.5d0*fzi*yij
                  vir(3,3) = vir(3,3) + 0.5d0*fzi*zij

               end do
               end do
               end do

            end do

         end do

      end do

!-----------------------------------------------------------------------
!     /*   main loop:  subtract bonded charge pairs                   */
!-----------------------------------------------------------------------

      do m = 1, nbead

         do k = 1, nbcp

            i  = i_bcp(k)
            j  = j_bcp(k)

            factor = factor_bcp(k)

            qi = q(i)
            qj = q(j)

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom( xij, yij, zij )

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            rinv = 1.d0/rij

            uij = (factor - 1.d0) * qi*qj*rinv

            pot(m) = pot(m) + uij

            duij = - uij*rinv

            fxi = - duij*xij*rinv
            fyi = - duij*yij*rinv
            fzi = - duij*zij*rinv

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_pol_ewald_fs
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pot, vir, fx, fy, fz, pi, boxinv, volume, nbead, natom

      use mm_variables, only : &
     &   eigax, eigay, eigaz, eigbx, eigby, eigbz, eigcx, eigcy, eigcz, &
     &   alpha_ewald, alpha_ewpol, q, px, py, pz, pol, i_q, lmax_ewald, &
     &   lmax_ewpol, ncharge

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: m, k, i, l, l2, la, lb, lc, m1, m2, m3

      real(8) :: ax, ay, az, bx, by, bz, cx, cy, cz, a2, b2, c2, poli, &
     &           al2, bl2, cl2, factor_1, factor_2, factor_3, factor_4, &
     &           factor_5, factor_6, factor_7, factor_8, factor_9, &
     &           gx, gy, gz, g2, g2max, qcos, qsin, psin, pcos, fxi, &
     &           fyi, fzi, cos_gxyz, sin_gxyz, qexp2, qpexp, pexp2, &
     &           gpx, gpy, gpz, pxcos, pycos, pzcos, pxsin, pysin, &
     &           pzsin, gpxyz, factor_0

      real(8) :: tiny = 1.d-12

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      m1 = max( lmax_ewald(1), lmax_ewpol(1) )
      m2 = max( lmax_ewald(2), lmax_ewpol(2) )
      m3 = max( lmax_ewald(3), lmax_ewpol(3) )

      if ( allocated( eigax ) ) deallocate( eigax )
      if ( allocated( eigay ) ) deallocate( eigay )
      if ( allocated( eigaz ) ) deallocate( eigaz )
      if ( allocated( eigbx ) ) deallocate( eigbx )
      if ( allocated( eigby ) ) deallocate( eigby )
      if ( allocated( eigbz ) ) deallocate( eigbz )
      if ( allocated( eigcx ) ) deallocate( eigcx )
      if ( allocated( eigcy ) ) deallocate( eigcy )
      if ( allocated( eigcz ) ) deallocate( eigcz )

      if ( .not. allocated( eigax ) ) allocate( eigax(natom,-m1:m1) )
      if ( .not. allocated( eigay ) ) allocate( eigay(natom,-m1:m1) )
      if ( .not. allocated( eigaz ) ) allocate( eigaz(natom,-m1:m1) )
      if ( .not. allocated( eigbx ) ) allocate( eigbx(natom,-m2:m2) )
      if ( .not. allocated( eigby ) ) allocate( eigby(natom,-m2:m2) )
      if ( .not. allocated( eigbz ) ) allocate( eigbz(natom,-m2:m2) )
      if ( .not. allocated( eigcx ) ) allocate( eigcx(natom,-m3:m3) )
      if ( .not. allocated( eigcy ) ) allocate( eigcy(natom,-m3:m3) )
      if ( .not. allocated( eigcz ) ) allocate( eigcz(natom,-m3:m3) )

!-----------------------------------------------------------------------
!     /*   loop of beads: start                                       */
!-----------------------------------------------------------------------

      do m = 1, nbead

!-----------------------------------------------------------------------
!        /*   parameters                                              */
!-----------------------------------------------------------------------

         ax = 2.d0*pi*boxinv(1,1)
         ay = 2.d0*pi*boxinv(1,2)
         az = 2.d0*pi*boxinv(1,3)
         bx = 2.d0*pi*boxinv(2,1)
         by = 2.d0*pi*boxinv(2,2)
         bz = 2.d0*pi*boxinv(2,3)
         cx = 2.d0*pi*boxinv(3,1)
         cy = 2.d0*pi*boxinv(3,2)
         cz = 2.d0*pi*boxinv(3,3)

         a2 = ax*ax + ay*ay + az*az
         b2 = bx*bx + by*by + bz*bz
         c2 = cx*cx + cy*cy + cz*cz

!-----------------------------------------------------------------------
!        /*   main loop                                               */
!-----------------------------------------------------------------------

         do k = 1, ncharge

            i  = i_q(k)

            eigax(i, 0)  = (1.d0,0.d0)
            eigay(i, 0)  = (1.d0,0.d0)
            eigaz(i, 0)  = (1.d0,0.d0)
            eigbx(i, 0)  = (1.d0,0.d0)
            eigby(i, 0)  = (1.d0,0.d0)
            eigbz(i, 0)  = (1.d0,0.d0)
            eigcx(i, 0)  = (1.d0,0.d0)
            eigcy(i, 0)  = (1.d0,0.d0)
            eigcz(i, 0)  = (1.d0,0.d0)

            if ( m1 .gt. 0 ) then
               eigax(i, 1)  = dcmplx ( cos(ax*x(i,m)), sin(ax*x(i,m)) )
               eigay(i, 1)  = dcmplx ( cos(ay*y(i,m)), sin(ay*y(i,m)) )
               eigaz(i, 1)  = dcmplx ( cos(az*z(i,m)), sin(az*z(i,m)) )
               eigax(i,-1)  = dconjg ( eigax(i,1) )
               eigay(i,-1)  = dconjg ( eigay(i,1) )
               eigaz(i,-1)  = dconjg ( eigaz(i,1) )
            end if
            if ( m2 .gt. 0 ) then
               eigbx(i, 1)  = dcmplx ( cos(bx*x(i,m)), sin(bx*x(i,m)) )
               eigby(i, 1)  = dcmplx ( cos(by*y(i,m)), sin(by*y(i,m)) )
               eigbz(i, 1)  = dcmplx ( cos(bz*z(i,m)), sin(bz*z(i,m)) )
               eigbx(i,-1)  = dconjg ( eigbx(i,1) )
               eigby(i,-1)  = dconjg ( eigby(i,1) )
               eigbz(i,-1)  = dconjg ( eigbz(i,1) )
            end if
            if ( m3 .gt. 0 ) then
               eigcx(i, 1)  = dcmplx ( cos(cx*x(i,m)), sin(cx*x(i,m)) )
               eigcy(i, 1)  = dcmplx ( cos(cy*y(i,m)), sin(cy*y(i,m)) )
               eigcz(i, 1)  = dcmplx ( cos(cz*z(i,m)), sin(cz*z(i,m)) )
               eigcx(i,-1)  = dconjg ( eigcx(i,1) )
               eigcy(i,-1)  = dconjg ( eigcy(i,1) )
               eigcz(i,-1)  = dconjg ( eigcz(i,1) )
            end if

            do l = 2, m1
               eigax(i, l)  = eigax(i,l-1)*eigax(i,1)
               eigay(i, l)  = eigay(i,l-1)*eigay(i,1)
               eigaz(i, l)  = eigaz(i,l-1)*eigaz(i,1)
               eigax(i,-l)  = dconjg ( eigax(i,l) )
               eigay(i,-l)  = dconjg ( eigay(i,l) )
               eigaz(i,-l)  = dconjg ( eigaz(i,l) )
            end do
            do l = 2, m2
               eigbx(i, l)  = eigbx(i,l-1)*eigbx(i,1)
               eigby(i, l)  = eigby(i,l-1)*eigby(i,1)
               eigbz(i, l)  = eigbz(i,l-1)*eigbz(i,1)
               eigbx(i,-l)  = dconjg ( eigbx(i,l) )
               eigby(i,-l)  = dconjg ( eigby(i,l) )
               eigbz(i,-l)  = dconjg ( eigbz(i,l) )
            end do
            do l = 2, m3
               eigcx(i, l)  = eigcx(i,l-1)*eigcx(i,1)
               eigcy(i, l)  = eigcy(i,l-1)*eigcy(i,1)
               eigcz(i, l)  = eigcz(i,l-1)*eigcz(i,1)
               eigcx(i,-l)  = dconjg ( eigcx(i,l) )
               eigcy(i,-l)  = dconjg ( eigcy(i,l) )
               eigcz(i,-l)  = dconjg ( eigcz(i,l) )
            end do

         end do

!-----------------------------------------------------------------------
!     /*   charge-charge interactions                                 */
!-----------------------------------------------------------------------

         al2 = a2*lmax_ewald(1)**2
         bl2 = b2*lmax_ewald(2)**2
         cl2 = c2*lmax_ewald(3)**2

         g2max = min( al2, bl2, cl2 )

         factor_1 = (4.d0*pi)/(2.d0*volume)

         do la =              0, lmax_ewald(1)
         do lb = -lmax_ewald(2), lmax_ewald(2)
         do lc = -lmax_ewald(3), lmax_ewald(3)

            l2 = la*la + lb*lb + lc*lc

            if ( l2 .eq. 0 ) cycle

            if ( la .eq. 0 ) then
               factor_2 = 1.d0
            else
               factor_2 = 2.d0
            end if

            gx = ax*la + bx*lb + cx*lc
            gy = ay*la + by*lb + cy*lc
            gz = az*la + bz*lb + cz*lc

            g2 = gx*gx + gy*gy + gz*gz

            if ( g2 .gt. g2max ) cycle

            factor_3 = exp(-g2/(4.d0*alpha_ewald*alpha_ewald))/g2

            qcos = 0.d0
            qsin = 0.d0

            do k = 1, ncharge

               i  = i_q(k)

               cos_gxyz = dreal(eigax(i,la)*eigbx(i,lb)*eigcx(i,lc) &
     &                         *eigay(i,la)*eigby(i,lb)*eigcy(i,lc) &
     &                         *eigaz(i,la)*eigbz(i,lb)*eigcz(i,lc))
               sin_gxyz = dimag(eigax(i,la)*eigbx(i,lb)*eigcx(i,lc) &
     &                         *eigay(i,la)*eigby(i,lb)*eigcy(i,lc) &
     &                         *eigaz(i,la)*eigbz(i,lb)*eigcz(i,lc))

               qcos = qcos + q(i)*cos_gxyz
               qsin = qsin + q(i)*sin_gxyz

            end do

            qexp2 = qcos*qcos + qsin*qsin

            factor_6 = factor_1*factor_2*factor_3*qexp2

            pot(m) = pot(m) + factor_6

            do k = 1, ncharge

               i  = i_q(k)

               cos_gxyz = dreal(eigax(i,la)*eigbx(i,lb)*eigcx(i,lc) &
     &                         *eigay(i,la)*eigby(i,lb)*eigcy(i,lc) &
     &                         *eigaz(i,la)*eigbz(i,lb)*eigcz(i,lc))
               sin_gxyz = dimag(eigax(i,la)*eigbx(i,lb)*eigcx(i,lc) &
     &                         *eigay(i,la)*eigby(i,lb)*eigcy(i,lc) &
     &                         *eigaz(i,la)*eigbz(i,lb)*eigcz(i,lc))

               factor_4 = sin_gxyz*qcos - cos_gxyz*qsin

               factor_5 = 2.d0*q(i)*factor_1*factor_2*factor_3*factor_4

               fxi = factor_5*gx
               fyi = factor_5*gy
               fzi = factor_5*gz

               fx(i,m) = fx(i,m) + fxi
               fy(i,m) = fy(i,m) + fyi
               fz(i,m) = fz(i,m) + fzi

            end do

            factor_7 = 1.d0/(4.d0*alpha_ewald*alpha_ewald)
            factor_8 = 2.d0 * ( 1.d0 + factor_7*g2 ) / g2

            vir(1,1) = vir(1,1) + factor_6 * ( 1.d0 - factor_8*gx*gx )
            vir(1,2) = vir(1,2) - factor_6 * factor_8*gx*gy
            vir(1,3) = vir(1,3) - factor_6 * factor_8*gx*gz
            vir(2,1) = vir(2,1) - factor_6 * factor_8*gy*gx
            vir(2,2) = vir(2,2) + factor_6 * ( 1.d0 - factor_8*gy*gy )
            vir(2,3) = vir(2,3) - factor_6 * factor_8*gy*gz
            vir(3,1) = vir(3,1) - factor_6 * factor_8*gz*gx
            vir(3,2) = vir(3,2) - factor_6 * factor_8*gz*gy
            vir(3,3) = vir(3,3) + factor_6 * ( 1.d0 - factor_8*gz*gz )

         end do
         end do
         end do

!-----------------------------------------------------------------------
!     /*   charge-dipole, dipole-dipole interactions                  */
!-----------------------------------------------------------------------

         al2 = a2*lmax_ewpol(1)**2
         bl2 = b2*lmax_ewpol(2)**2
         cl2 = c2*lmax_ewpol(3)**2

         g2max = min( al2, bl2, cl2 )

         factor_1 = (4.d0*pi)/(2.d0*volume)

         do la =              0, lmax_ewpol(1)
         do lb = -lmax_ewpol(2), lmax_ewpol(2)
         do lc = -lmax_ewpol(3), lmax_ewpol(3)

            l2 = la*la + lb*lb + lc*lc

            if ( l2 .eq. 0 ) cycle

            if ( la .eq. 0 ) then
               factor_2 = 1.d0
            else
               factor_2 = 2.d0
            end if

            gx = ax*la + bx*lb + cx*lc
            gy = ay*la + by*lb + cy*lc
            gz = az*la + bz*lb + cz*lc

            g2 = gx*gx + gy*gy + gz*gz

            if ( g2 .gt. g2max ) cycle

            factor_3 = exp(-g2/(4.d0*alpha_ewpol*alpha_ewpol))/g2

            qcos = 0.d0
            qsin = 0.d0

            pcos = 0.d0
            psin = 0.d0

            pxcos = 0.d0
            pycos = 0.d0
            pzcos = 0.d0
            pxsin = 0.d0
            pysin = 0.d0
            pzsin = 0.d0

            do k = 1, ncharge

               i  = i_q(k)

               cos_gxyz = dreal(eigax(i,la)*eigbx(i,lb)*eigcx(i,lc) &
     &                         *eigay(i,la)*eigby(i,lb)*eigcy(i,lc) &
     &                         *eigaz(i,la)*eigbz(i,lb)*eigcz(i,lc))
               sin_gxyz = dimag(eigax(i,la)*eigbx(i,lb)*eigcx(i,lc) &
     &                         *eigay(i,la)*eigby(i,lb)*eigcy(i,lc) &
     &                         *eigaz(i,la)*eigbz(i,lb)*eigcz(i,lc))

               qcos = qcos + q(i)*cos_gxyz
               qsin = qsin + q(i)*sin_gxyz

               poli = pol(k)

               if ( abs(poli) .gt. tiny ) then

                  gpx = gx*px(i,m)
                  gpy = gy*py(i,m)
                  gpz = gz*pz(i,m)

                  gpxyz = gpx + gpy + gpz

                  pcos = pcos + gpxyz*cos_gxyz
                  psin = psin + gpxyz*sin_gxyz

                  pxcos = pxcos + px(i,m)*cos_gxyz
                  pycos = pycos + py(i,m)*cos_gxyz
                  pzcos = pzcos + pz(i,m)*cos_gxyz

                  pxsin = pxsin + px(i,m)*sin_gxyz
                  pysin = pysin + py(i,m)*sin_gxyz
                  pzsin = pzsin + pz(i,m)*sin_gxyz

               end if

            end do

            qpexp = qsin*pcos - qcos*psin
            pexp2 = pcos*pcos + psin*psin

            factor_9 = 2.d0*qpexp + pexp2

            factor_6 = factor_1*factor_2*factor_3*factor_9

            pot(m) = pot(m) + factor_6

            do k = 1, ncharge

               i  = i_q(k)

               cos_gxyz = dreal(eigax(i,la)*eigbx(i,lb)*eigcx(i,lc) &
     &                         *eigay(i,la)*eigby(i,lb)*eigcy(i,lc) &
     &                         *eigaz(i,la)*eigbz(i,lb)*eigcz(i,lc))
               sin_gxyz = dimag(eigax(i,la)*eigbx(i,lb)*eigcx(i,lc) &
     &                         *eigay(i,la)*eigby(i,lb)*eigcy(i,lc) &
     &                         *eigaz(i,la)*eigbz(i,lb)*eigcz(i,lc))

               factor_4 = - q(i) * ( cos_gxyz*pcos + sin_gxyz*psin )

               poli = pol(k)

               if ( abs(poli) .gt. tiny ) then

                  gpx = gx*px(i,m)
                  gpy = gy*py(i,m)
                  gpz = gz*pz(i,m)

                  gpxyz = gpx + gpy + gpz

                  factor_4 = factor_4 &
     &                     + gpxyz  * ( cos_gxyz*qcos + sin_gxyz*qsin ) &
     &                     + gpxyz  * ( sin_gxyz*pcos - cos_gxyz*psin )

               end if

               factor_5 = 2.d0*factor_1*factor_2*factor_3*factor_4

               fxi = factor_5*gx
               fyi = factor_5*gy
               fzi = factor_5*gz

               fx(i,m) = fx(i,m) + fxi
               fy(i,m) = fy(i,m) + fyi
               fz(i,m) = fz(i,m) + fzi

            end do

            factor_7 = 1.d0/(4.d0*alpha_ewpol*alpha_ewpol)
            factor_8 = 2.d0 * ( 1.d0 + factor_7*g2 ) / g2

            vir(1,1) = vir(1,1) + factor_6 * ( 1.d0 - factor_8*gx*gx )
            vir(1,2) = vir(1,2) - factor_6 * factor_8*gx*gy
            vir(1,3) = vir(1,3) - factor_6 * factor_8*gx*gz
            vir(2,1) = vir(2,1) - factor_6 * factor_8*gy*gx
            vir(2,2) = vir(2,2) + factor_6 * ( 1.d0 - factor_8*gy*gy )
            vir(2,3) = vir(2,3) - factor_6 * factor_8*gy*gz
            vir(3,1) = vir(3,1) - factor_6 * factor_8*gz*gx
            vir(3,2) = vir(3,2) - factor_6 * factor_8*gz*gy
            vir(3,3) = vir(3,3) + factor_6 * ( 1.d0 - factor_8*gz*gz )

            factor_0 = 2.d0*factor_1*factor_2*factor_3

            vir(1,1) = vir(1,1) - factor_0*qcos*pxsin*gx &
     &                          + factor_0*qsin*pxcos*gx
            vir(2,1) = vir(2,1) - factor_0*qcos*pxsin*gy &
     &                          + factor_0*qsin*pxcos*gy
            vir(3,1) = vir(3,1) - factor_0*qcos*pxsin*gz &
     &                          + factor_0*qsin*pxcos*gz

            vir(1,2) = vir(1,2) - factor_0*qcos*pysin*gx &
     &                          + factor_0*qsin*pycos*gx
            vir(2,2) = vir(2,2) - factor_0*qcos*pysin*gy &
     &                          + factor_0*qsin*pycos*gy
            vir(3,2) = vir(3,2) - factor_0*qcos*pysin*gz &
     &                          + factor_0*qsin*pycos*gz

            vir(1,3) = vir(1,3) - factor_0*qcos*pzsin*gx &
     &                          + factor_0*qsin*pzcos*gx
            vir(2,3) = vir(2,3) - factor_0*qcos*pzsin*gy &
     &                          + factor_0*qsin*pzcos*gy
            vir(3,3) = vir(3,3) - factor_0*qcos*pzsin*gz &
     &                          + factor_0*qsin*pzcos*gz

            vir(1,1) = vir(1,1) + factor_0*pcos*pxcos*gx &
     &                          + factor_0*psin*pxsin*gx
            vir(2,1) = vir(2,1) + factor_0*pcos*pxcos*gy &
     &                          + factor_0*psin*pxsin*gy
            vir(3,1) = vir(3,1) + factor_0*pcos*pxcos*gz &
     &                          + factor_0*psin*pxsin*gz

            vir(1,2) = vir(1,2) + factor_0*pcos*pycos*gx &
     &                          + factor_0*psin*pysin*gx
            vir(2,2) = vir(2,2) + factor_0*pcos*pycos*gy &
     &                          + factor_0*psin*pysin*gy
            vir(3,2) = vir(3,2) + factor_0*pcos*pycos*gz &
     &                          + factor_0*psin*pysin*gz

            vir(1,3) = vir(1,3) + factor_0*pcos*pzcos*gx &
     &                          + factor_0*psin*pzsin*gx
            vir(2,3) = vir(2,3) + factor_0*pcos*pzcos*gy &
     &                          + factor_0*psin*pzsin*gy
            vir(3,3) = vir(3,3) + factor_0*pcos*pzcos*gz &
     &                          + factor_0*psin*pzsin*gz

         end do
         end do
         end do

!-----------------------------------------------------------------------
!     /*   loop of beads: end                                         */
!-----------------------------------------------------------------------

      end do

      return
      end





!***********************************************************************
      subroutine force_pol_ewald_self
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, pot, nbead

      use mm_variables, only : &
     &   alpha_ewald, alpha_ewpol, q, px, py, pz, i_q, ncharge

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, m, k

      real(8) :: const_self_qq, const_self_pp, q2sum, p2sum, sqrt_pi

!-----------------------------------------------------------------------
!     /*   parameters                                                 */
!-----------------------------------------------------------------------

      sqrt_pi = sqrt(pi)

      const_self_qq = alpha_ewald / sqrt_pi

      const_self_pp = 2.d0 * alpha_ewpol**3 / ( 3.d0*sqrt_pi )

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         q2sum = 0.d0

         p2sum = 0.d0

         do k = 1, ncharge

            i  = i_q(k)

            q2sum = q2sum + q(i)*q(i)

            p2sum = p2sum + px(i,m)*px(i,m) &
     &                    + py(i,m)*py(i,m) &
     &                    + pz(i,m)*pz(i,m)

         end do

         pot(m) = pot(m) - q2sum*const_self_qq - p2sum*const_self_pp

      end do

      return
      end





!***********************************************************************
      subroutine force_pol_charge
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, volume, pot, vir, nbead

      use mm_variables, only : &
     &   alpha_ewald, q, i_q, pol, px, py, pz, ncharge

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, m, k

      real(8) :: qsum, pxsum, pysum, pzsum, p1, p2, p3, const_net_qq, &
     &           const_net_pp

      real(8) :: tiny = 1.d-12

!-----------------------------------------------------------------------
!     /*   parameters                                                 */
!-----------------------------------------------------------------------

      const_net_qq = pi / ( 2.d0*volume*alpha_ewald*alpha_ewald )

      const_net_pp = 2.d0*pi / (3.d0*volume)

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         qsum = 0.d0

         pxsum = 0.d0
         pysum = 0.d0
         pzsum = 0.d0

         do k = 1, ncharge

            i  = i_q(k)

            qsum = qsum + q(i)

            if ( abs(pol(k)) .lt. tiny ) cycle

            pxsum = pxsum + px(i,m)
            pysum = pysum + py(i,m)
            pzsum = pzsum + pz(i,m)

         end do

         p1 = qsum*qsum*const_net_qq
         p2 = ( pxsum*pxsum + pysum*pysum + pzsum*pzsum ) * const_net_pp

         p3 = - p1 + p2

         pot(m) = pot(m) + p3

         vir(1,1) = vir(1,1) + p3
         vir(2,2) = vir(2,2) + p3
         vir(3,3) = vir(3,3) + p3

      end do

      return
      end





!***********************************************************************
      subroutine force_pol_dipole
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables from PIMD                                 */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, volume, pi, dipx, dipy, dipz, &
     &   mbox, nbead

      use mm_variables, only : &
     &   i_q, q, pol, px, py, pz, ncharge, ioption_ewald

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   reset   */
      implicit none

!     /*   integers   */
      integer :: i, j, k, m1, m2, m3

!     /*   real numbers   */
      real(8) :: dip2, xi, yi, zi, const_srf_pp

!     /*   real numbers   */
      real(8) :: tiny = 1.d-12

!     /*   vectors   */
      real(8), dimension(:), allocatable :: ex, ey, ez, dx, dy, dz

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( .not. allocated(ex) ) allocate( ex(nbead) )
      if ( .not. allocated(ey) ) allocate( ey(nbead) )
      if ( .not. allocated(ez) ) allocate( ez(nbead) )
      if ( .not. allocated(dx) ) allocate( dx(nbead) )
      if ( .not. allocated(dy) ) allocate( dy(nbead) )
      if ( .not. allocated(dz) ) allocate( dz(nbead) )

!-----------------------------------------------------------------------
!     /*   dipole moment                                              */
!-----------------------------------------------------------------------

!     /*   constant   */
      const_srf_pp = 2.d0 * pi / ( 3.d0 * volume )

!     /*   dipole moment   */
      dipx(:) = 0.d0
      dipy(:) = 0.d0
      dipz(:) = 0.d0

!     /*   initialize   */
      dx(:) = 0.d0
      dy(:) = 0.d0
      dz(:) = 0.d0
      ex(:) = 0.d0
      ey(:) = 0.d0
      ez(:) = 0.d0

!     /*   loop of beads   */
      do j = 1, nbead

!        /*   loop of charges   */
         do k = 1, ncharge

!           /*   atom   */
            i  = i_q(k)

!           /*   coordinates   */
            xi = x(i,j)
            yi = y(i,j)
            zi = z(i,j)

!           /*   box number   */
            m1 = mbox(1,i,j)
            m2 = mbox(2,i,j)
            m3 = mbox(3,i,j)

!           /*   apply periodic boundary condition   */
            call pbc_unfold ( xi, yi, zi, m1, m2, m3 )

!           /*   dipole moment   */
            ex(j) = ex(j) + q(i) * xi
            ey(j) = ey(j) + q(i) * yi
            ez(j) = ez(j) + q(i) * zi

!           /*   dipole moment   */
            if ( abs(pol(k)) .gt. tiny ) then
               dx(j) = dx(j) + px(i,j)
               dy(j) = dy(j) + py(i,j)
               dz(j) = dz(j) + pz(i,j)
            end if

!        /*   loop of atoms   */
         end do

!        /*   dipole moment   */
         dipx(j) = dx(j) + ex(j)
         dipy(j) = dy(j) + ey(j)
         dipz(j) = dz(j) + ez(j)

!     /*   loop of beads   */
      end do

!-----------------------------------------------------------------------
!     /*   surface dipole correction                                  */
!-----------------------------------------------------------------------

!     /*   only when option is turned on   */
      if ( ioption_ewald .eq. 1 ) then

!        /*   loop of beads   */
         do j = 1, nbead

!           /*   square of dipole moment   */
            dip2 = dipx(j)*dipx(j) + dipy(j)*dipy(j) + dipz(j)*dipz(j)

!           /*   potential   */
            pot(j) = pot(j) + const_srf_pp * dip2

!           /*   loop of charges   */
            do k = 1, ncharge

!              /*   atom   */
               i  = i_q(k)

!              /*   coordinates   */
               xi = x(i,j)
               yi = y(i,j)
               zi = z(i,j)

!              /*   box number   */
               m1 = mbox(1,i,j)
               m2 = mbox(2,i,j)
               m3 = mbox(3,i,j)

!              /*   apply periodic boundary condition   */
               call pbc_unfold ( xi, yi, zi, m1, m2, m3 )

!              /*   forces   */
               fx(i,j) = fx(i,j) - 2.d0 * const_srf_pp * q(i) * dipx(j)
               fy(i,j) = fy(i,j) - 2.d0 * const_srf_pp * q(i) * dipy(j)
               fz(i,j) = fz(i,j) - 2.d0 * const_srf_pp * q(i) * dipz(j)

!           /*   loop of charges   */
            end do

!           /*   virial   */
            vir(1,1) = vir(1,1) - 2.d0 * const_srf_pp * dipx(j) * ex(j) &
     &                          + const_srf_pp * dip2
            vir(1,2) = vir(1,2) - 2.d0 * const_srf_pp * dipx(j) * ey(j)
            vir(1,3) = vir(1,3) - 2.d0 * const_srf_pp * dipx(j) * ez(j)
            vir(2,1) = vir(2,1) - 2.d0 * const_srf_pp * dipy(j) * ex(j)
            vir(2,2) = vir(2,2) - 2.d0 * const_srf_pp * dipy(j) * ey(j) &
     &                          + const_srf_pp * dip2
            vir(2,3) = vir(2,3) - 2.d0 * const_srf_pp * dipy(j) * ez(j)
            vir(3,1) = vir(3,1) - 2.d0 * const_srf_pp * dipz(j) * ex(j)
            vir(3,2) = vir(3,2) - 2.d0 * const_srf_pp * dipz(j) * ey(j)
            vir(3,3) = vir(3,3) - 2.d0 * const_srf_pp * dipz(j) * ez(j) &
     &                          + const_srf_pp * dip2

!        /*   loop of beads   */
         end do

!     /*   only when option is turned on   */
      end if

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( allocated(ex) ) deallocate( ex )
      if ( allocated(ey) ) deallocate( ey )
      if ( allocated(ez) ) deallocate( ez )
      if ( allocated(dx) ) deallocate( dx )
      if ( allocated(dy) ) deallocate( dy )
      if ( allocated(dz) ) deallocate( dz )

      return
      end





!***********************************************************************
      subroutine field_pol_coulomb
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, nbead, dipx, dipy, dipz

      use mm_variables, only : &
     &   q, pol, px, py, pz, ncharge, npol, i_q, list_pol

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, ix, iy, iz, jx, jy, jz, npol3, info

      real(8) :: qi, qj, xij, yij, zij, r, r2, rinv, rinv5, poli, polj, &
     &           txqp, tyqp, tzqp, txxpp, txypp, txzpp, tyxpp, tyypp, &
     &           tyzpp, tzxpp, tzypp, tzzpp, rinv3, a3qp, a3pp, b5pp, &
     &           rinv2, s0qp, s1qp, s0pp, s1pp, txpq, typq, tzpq, s0pq, &
     &           s1pq, a3pq, t0pp, t1pp

      real(8), dimension(:,:), allocatable :: amat
      real(8), dimension(:,:), allocatable :: bvec
      integer, dimension(:), allocatable   :: ipiv
!     /*   vectors   */
      real(8), dimension(:), allocatable :: ex, ey, ez, dx, dy, dz

      real(8) :: tiny = 1.d-12

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( ncharge .eq. 0 ) return

      px(:,:) = 0.d0
      py(:,:) = 0.d0
      pz(:,:) = 0.d0

      if ( npol .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      npol3 = npol * 3

      if ( .not. allocated(amat) ) allocate( amat(npol3,npol3) )
      if ( .not. allocated(bvec) ) allocate( bvec(npol3,1) )
      if ( .not. allocated(ipiv) ) allocate( ipiv(npol3) )
      if ( .not. allocated(ex) ) allocate( ex(nbead) )
      if ( .not. allocated(ey) ) allocate( ey(nbead) )
      if ( .not. allocated(ez) ) allocate( ez(nbead) )
      if ( .not. allocated(dx) ) allocate( dx(nbead) )
      if ( .not. allocated(dy) ) allocate( dy(nbead) )
      if ( .not. allocated(dz) ) allocate( dz(nbead) )

!     /*   dipole moment   */
      dipx(:) = 0.d0
      dipy(:) = 0.d0
      dipz(:) = 0.d0

!     /*   initialize   */
      dx(:) = 0.d0
      dy(:) = 0.d0
      dz(:) = 0.d0
      ex(:) = 0.d0
      ey(:) = 0.d0
      ez(:) = 0.d0

!-----------------------------------------------------------------------
!     /*   main loop:  make amat and bvec                             */
!-----------------------------------------------------------------------

      do m = 1, nbead

         bvec(:,:) = 0.d0
         amat(:,:) = 0.d0

         ix = 0
         iy = 0
         iz = 0
         jx = 0
         jy = 0
         jz = 0

         do k = 1, ncharge

            i = i_q(k)

            qi = q(i)

            poli = pol(k)

            ix = 3 * list_pol(k) - 2
            iy = 3 * list_pol(k) - 1
            iz = 3 * list_pol(k) - 0

            if ( abs(poli) .gt. tiny ) then
               amat(ix,ix) = 1.d0
               amat(iy,iy) = 1.d0
               amat(iz,iz) = 1.d0
            end if

            do l = k+1, ncharge

               j = i_q(l)

               qj = q(j)

               polj = pol(l)

               jx = 3 * list_pol(l) - 2
               jy = 3 * list_pol(l) - 1
               jz = 3 * list_pol(l) - 0

               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

               call pbc_atom ( xij, yij, zij )

               r2 = xij*xij + yij*yij + zij*zij

               r = sqrt( r2 )

               rinv = 1.d0 / r
               rinv2 = rinv * rinv
               rinv3 = rinv * rinv2
               rinv5 = rinv2 * rinv3

               call dampfunc_qp( r, k, l, s0qp, s1qp )
               call dampfunc_qp( r, l, k, s0pq, s1pq )
               call dampfunc_pp( r, k, l, s0pp, s1pp, t0pp, t1pp )

               a3qp = s0qp * rinv3
               a3pq = s0pq * rinv3

               a3pp = s0pp * rinv3
               b5pp = 3.d0 * t0pp * rinv5

               txqp = - xij * a3qp
               tyqp = - yij * a3qp
               tzqp = - zij * a3qp

               txpq = - xij * a3pq
               typq = - yij * a3pq
               tzpq = - zij * a3pq

               if ( abs(poli) .gt. tiny ) then
                  bvec(ix,1) = bvec(ix,1) - poli * qj * txpq
                  bvec(iy,1) = bvec(iy,1) - poli * qj * typq
                  bvec(iz,1) = bvec(iz,1) - poli * qj * tzpq
               end if

               if ( abs(polj) .gt. tiny ) then
                  bvec(jx,1) = bvec(jx,1) + polj * qi * txqp
                  bvec(jy,1) = bvec(jy,1) + polj * qi * tyqp
                  bvec(jz,1) = bvec(jz,1) + polj * qi * tzqp
               end if

               txxpp = xij * xij * b5pp - a3pp
               txypp = xij * yij * b5pp
               txzpp = xij * zij * b5pp
               tyxpp = yij * xij * b5pp
               tyypp = yij * yij * b5pp - a3pp
               tyzpp = yij * zij * b5pp
               tzxpp = zij * xij * b5pp
               tzypp = zij * yij * b5pp
               tzzpp = zij * zij * b5pp - a3pp

               if ( abs(poli)*abs(polj) .gt. tiny*tiny ) then
                  amat(ix,jx) = amat(ix,jx) - poli * txxpp
                  amat(ix,jy) = amat(ix,jy) - poli * txypp
                  amat(ix,jz) = amat(ix,jz) - poli * txzpp
                  amat(iy,jx) = amat(iy,jx) - poli * tyxpp
                  amat(iy,jy) = amat(iy,jy) - poli * tyypp
                  amat(iy,jz) = amat(iy,jz) - poli * tyzpp
                  amat(iz,jx) = amat(iz,jx) - poli * tzxpp
                  amat(iz,jy) = amat(iz,jy) - poli * tzypp
                  amat(iz,jz) = amat(iz,jz) - poli * tzzpp

                  amat(jx,ix) = amat(jx,ix) - polj * txxpp
                  amat(jy,ix) = amat(jy,ix) - polj * txypp
                  amat(jz,ix) = amat(jz,ix) - polj * txzpp
                  amat(jx,iy) = amat(jx,iy) - polj * tyxpp
                  amat(jy,iy) = amat(jy,iy) - polj * tyypp
                  amat(jz,iy) = amat(jz,iy) - polj * tyzpp
                  amat(jx,iz) = amat(jx,iz) - polj * tzxpp
                  amat(jy,iz) = amat(jy,iz) - polj * tzypp
                  amat(jz,iz) = amat(jz,iz) - polj * tzzpp
               end if

            end do

         end do

!-----------------------------------------------------------------------
!        /*   solve linear equation                                   */
!-----------------------------------------------------------------------

#ifdef nolapack
         write( 6, '(a)' ) &
     &      'Error - Diagonalization routine not linked.'
         call error_handling &
     &      ( 1, 'subroutine field_pol_coulomb', 28 )
#else
         call dgesv( npol3, 1, amat, npol3, ipiv, bvec, npol3, info )
#endif

!-----------------------------------------------------------------------
!        /*   substitution                                            */
!-----------------------------------------------------------------------


         do k = 1, ncharge

            i = i_q(k)

            poli = pol(k)

!           /*   dipole moment   */
            ex(m) = ex(m) + q(i) * x(k,m)
            ey(m) = ey(m) + q(i) * y(k,m)
            ez(m) = ez(m) + q(i) * z(k,m)

            if ( abs(poli) .lt. tiny ) cycle

            ix = 3 * list_pol(k) - 2
            iy = 3 * list_pol(k) - 1
            iz = 3 * list_pol(k) - 0

            px(i,m) = bvec(ix,1)
            py(i,m) = bvec(iy,1)
            pz(i,m) = bvec(iz,1)

!           /*   dipole moment   */
            dx(m) = dx(m) + px(i,m)
            dy(m) = dy(m) + py(i,m)
            dz(m) = dz(m) + pz(i,m)

!        /*   loop of atoms   */
         end do

!        /*   dipole moment   */
         dipx(m) = dx(m) + ex(m)
         dipy(m) = dy(m) + ey(m)
         dipz(m) = dz(m) + ez(m)

      end do

!-----------------------------------------------------------------------
!     /*   memory deallocation                                        */
!-----------------------------------------------------------------------

      if ( allocated(amat) ) deallocate( amat )
      if ( allocated(bvec) ) deallocate( bvec )
      if ( allocated(ipiv) ) deallocate( ipiv )

      return
      end





!***********************************************************************
      subroutine field_pol_ewald
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pi, box, boxinv, volume, nbead, natom, mbox

      use mm_variables, only : &
     &   q, pol, px, py, pz, bigbox, bigboxinv, rcut_ewpol, alpha_ewpol, &
     &   eigax, eigay, eigaz, eigbx, eigby, eigbz, eigcx, eigcy, eigcz, &
     &   nbox_ewpol, ncharge, npol, i_q, list_pol, lmax_ewpol, &
     &   ioption_ewald

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, ix, iy, iz, jx, jy, jz, npol3, info, &
     &           kx, ky, kz, k2, l2, la, lb, lc, m1, m2, m3, n1, n2, n3

      real(8) :: ax, ay, az, bx, by, bz, cx, cy, cz, a2, b2, c2, &
     &           al2, bl2, cl2, factor_1, factor_2, factor_3, factor_4, &
     &           factor_5, gx, gy, gz, g2, g2max, qcos, qsin, psin, &
     &           pcos, gpx, gpy, gpz, gpxyz, expgh, factor_6, &
     &           qi, qj, xij, yij, zij, r, r2, rinv, rinv5, poli, polj, &
     &           txqp, tyqp, tzqp, txxpp, txypp, txzpp, tyxpp, tyypp, &
     &           tyzpp, tzxpp, tzypp, tzzpp, rinv3, ar, erf_0, erf_1, &
     &           erfc_ar, rinv2, s0qp, s1qp, s0pp, s1pp, aij, bij, cij, &
     &           b1er, a1er, a3er, a5er, a3qs, a3qp, a3ps, b5ps, a3pp, &
     &           b5pp, rcut_ewpol2, alpha_ewpol2, const_self_pp, &
     &           txpq, typq, tzpq, s0pq, s1pq, a3pq, a3sq, t0pp, t1pp, &
     &           gxx, gxy, gxz, gyy, gyz, gzz, const_net_pp, &
     &           const_srf_pp, ex, ey, ez

      real(8), dimension(:,:), allocatable :: amat
      real(8), dimension(:,:), allocatable :: bvec
      integer, dimension(:), allocatable   :: ipiv
      real(8), dimension(:), allocatable   :: cos_gxyz
      real(8), dimension(:), allocatable   :: sin_gxyz

      real(8) :: tiny = 1.d-12

!      real(8) :: rin_damp, rout_damp, swf, dswf

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( ncharge .eq. 0 ) return

      px(:,:) = 0.d0
      py(:,:) = 0.d0
      pz(:,:) = 0.d0

      if ( npol .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      npol3 = npol * 3

      if ( .not. allocated(amat) ) allocate( amat(npol3,npol3) )
      if ( .not. allocated(bvec) ) allocate( bvec(npol3,1) )
      if ( .not. allocated(ipiv) ) allocate( ipiv(npol3) )
      if ( .not. allocated(cos_gxyz) ) allocate( cos_gxyz(ncharge) )
      if ( .not. allocated(sin_gxyz) ) allocate( sin_gxyz(ncharge) )

      m1 = lmax_ewpol(1)
      m2 = lmax_ewpol(2)
      m3 = lmax_ewpol(3)

      if ( allocated( eigax ) ) deallocate( eigax )
      if ( allocated( eigay ) ) deallocate( eigay )
      if ( allocated( eigaz ) ) deallocate( eigaz )
      if ( allocated( eigbx ) ) deallocate( eigbx )
      if ( allocated( eigby ) ) deallocate( eigby )
      if ( allocated( eigbz ) ) deallocate( eigbz )
      if ( allocated( eigcx ) ) deallocate( eigcx )
      if ( allocated( eigcy ) ) deallocate( eigcy )
      if ( allocated( eigcz ) ) deallocate( eigcz )

      if ( .not. allocated( eigax ) ) allocate( eigax(natom,-m1:m1) )
      if ( .not. allocated( eigay ) ) allocate( eigay(natom,-m1:m1) )
      if ( .not. allocated( eigaz ) ) allocate( eigaz(natom,-m1:m1) )
      if ( .not. allocated( eigbx ) ) allocate( eigbx(natom,-m2:m2) )
      if ( .not. allocated( eigby ) ) allocate( eigby(natom,-m2:m2) )
      if ( .not. allocated( eigbz ) ) allocate( eigbz(natom,-m2:m2) )
      if ( .not. allocated( eigcx ) ) allocate( eigcx(natom,-m3:m3) )
      if ( .not. allocated( eigcy ) ) allocate( eigcy(natom,-m3:m3) )
      if ( .not. allocated( eigcz ) ) allocate( eigcz(natom,-m3:m3) )

!-----------------------------------------------------------------------
!     /*   parameters                                                 */
!-----------------------------------------------------------------------

      rcut_ewpol2 = rcut_ewpol * rcut_ewpol

      alpha_ewpol2 = alpha_ewpol * alpha_ewpol

      const_self_pp = 2.d0 * alpha_ewpol**3 / ( 3.d0*sqrt(pi) )

      const_net_pp = 2.d0*pi / ( 3.d0*volume )

!-----------------------------------------------------------------------
!     /*   main loop:  make amat and bvec                             */
!-----------------------------------------------------------------------

      do m = 1, nbead

         bvec(:,:) = 0.d0
         amat(:,:) = 0.d0

         ix = 0
         iy = 0
         iz = 0
         jx = 0
         jy = 0
         jz = 0

         bigbox(:,1) = dble(nbox_ewpol(1)) * box(:,1)
         bigbox(:,2) = dble(nbox_ewpol(2)) * box(:,2)
         bigbox(:,3) = dble(nbox_ewpol(3)) * box(:,3)

         call inv3 ( bigbox, bigboxinv )

         do k = 1, ncharge

            i = i_q(k)

            qi = q(i)

            poli = pol(k)

            ix = 3 * list_pol(k) - 2
            iy = 3 * list_pol(k) - 1
            iz = 3 * list_pol(k) - 0

            if ( abs(poli) .gt. tiny ) then
               amat(ix,ix) = 1.d0 - 2.d0*const_self_pp*poli
               amat(iy,iy) = 1.d0 - 2.d0*const_self_pp*poli
               amat(iz,iz) = 1.d0 - 2.d0*const_self_pp*poli
            end if

            do l = 1, ncharge

               polj = pol(l)

               if ( abs(poli)+abs(polj) .lt. 2.d0*tiny ) cycle

               j = i_q(l)

               qj = q(j)

               jx = 3 * list_pol(l) - 2
               jy = 3 * list_pol(l) - 1
               jz = 3 * list_pol(l) - 0

               if ( abs(poli)*abs(polj) .gt. tiny*tiny ) then
                  amat(ix,jx) = amat(ix,jx) + const_net_pp*poli
                  amat(iy,jy) = amat(iy,jy) + const_net_pp*poli
                  amat(iz,jz) = amat(iz,jz) + const_net_pp*poli

                  amat(jx,ix) = amat(jx,ix) + const_net_pp*polj
                  amat(jy,iy) = amat(jy,iy) + const_net_pp*polj
                  amat(jz,iz) = amat(jz,iz) + const_net_pp*polj
               end if

               do kx = 0, nbox_ewpol(1)-1
               do ky = 0, nbox_ewpol(2)-1
               do kz = 0, nbox_ewpol(3)-1

                  k2 = kx*kx + ky*ky + kz*kz

                  if ( ( k2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

                  xij = x(i,m) - x(j,m)
                  yij = y(i,m) - y(j,m)
                  zij = z(i,m) - z(j,m)

                  xij = xij - box(1,1)*kx - box(1,2)*ky - box(1,3)*kz
                  yij = yij - box(2,1)*kx - box(2,2)*ky - box(2,3)*kz
                  zij = zij - box(3,1)*kx - box(3,2)*ky - box(3,3)*kz

                  aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &                + bigboxinv(1,3)*zij
                  bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &                + bigboxinv(2,3)*zij
                  cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &                + bigboxinv(3,3)*zij

                  aij = aij - nint(aij)
                  bij = bij - nint(bij)
                  cij = cij - nint(cij)

                  xij = bigbox(1,1)*aij + bigbox(1,2)*bij &
     &                + bigbox(1,3)*cij
                  yij = bigbox(2,1)*aij + bigbox(2,2)*bij &
     &                + bigbox(2,3)*cij
                  zij = bigbox(3,1)*aij + bigbox(3,2)*bij &
     &                + bigbox(3,3)*cij

                  r2 = xij*xij + yij*yij + zij*zij

                  if ( r2 .gt. rcut_ewpol2 ) cycle

                  r = sqrt( r2 )

                  rinv = 1.d0 / r
                  rinv2 = rinv * rinv
                  rinv3 = rinv * rinv2
                  rinv5 = rinv2 * rinv3

                  call dampfunc_qp( r, k, l, s0qp, s1qp )
                  call dampfunc_qp( r, l, k, s0pq, s1pq )
                  call dampfunc_pp( r, k, l, s0pp, s1pp, t0pp, t1pp )

!                  call getswf( r, rin_damp, rout_damp, swf, dswf )
!
!                  s0qp = swf*s0qp - swf + 1.d0
!                  s0pq = swf*s0pq - swf + 1.d0
!                  s0pp = swf*s0pp - swf + 1.d0
!                  t0pp = swf*t0pp - swf + 1.d0
!
!                  s1qp = swf*s1qp + dswf*s0qp - dswf
!                  s1pq = swf*s1pq + dswf*s0pq - dswf
!                  s1pp = swf*s1pp + dswf*s0qp - dswf
!                  t1pp = swf*t1pp + dswf*t0qp - dswf

                  ar = alpha_ewpol * r

                  erfc_ar = 1.d0 - erf_0(ar)

                  b1er = alpha_ewpol * erf_1(ar)

                  a1er = erfc_ar * rinv
                  a3er = ( a1er + b1er ) * rinv2
                  a5er = ( 3.d0*a3er + 2.d0*alpha_ewpol2*b1er ) * rinv2

                  a3qs = ( s0qp - 1.d0 ) * rinv3
                  a3sq = ( s0pq - 1.d0 ) * rinv3

                  a3qp = a3er + a3qs
                  a3pq = a3er + a3sq

                  a3ps = ( s0pp - 1.d0 ) * rinv3
                  b5ps = 3.d0 * ( t0pp - 1.d0 ) * rinv5

                  a3pp = a3er + a3ps
                  b5pp = a5er + b5ps

                  txqp = - xij * a3qp
                  tyqp = - yij * a3qp
                  tzqp = - zij * a3qp

                  txpq = - xij * a3pq
                  typq = - yij * a3pq
                  tzpq = - zij * a3pq

                  if ( abs(poli) .gt. tiny ) then
                     bvec(ix,1) = bvec(ix,1) - 0.5d0 * poli * qj * txpq
                     bvec(iy,1) = bvec(iy,1) - 0.5d0 * poli * qj * typq
                     bvec(iz,1) = bvec(iz,1) - 0.5d0 * poli * qj * tzpq
                  end if

                  if ( abs(polj) .gt. tiny ) then
                     bvec(jx,1) = bvec(jx,1) + 0.5d0 * polj * qi * txqp
                     bvec(jy,1) = bvec(jy,1) + 0.5d0 * polj * qi * tyqp
                     bvec(jz,1) = bvec(jz,1) + 0.5d0 * polj * qi * tzqp
                  end if

                  txxpp = xij * xij * b5pp - a3pp
                  txypp = xij * yij * b5pp
                  txzpp = xij * zij * b5pp
                  tyxpp = yij * xij * b5pp
                  tyypp = yij * yij * b5pp - a3pp
                  tyzpp = yij * zij * b5pp
                  tzxpp = zij * xij * b5pp
                  tzypp = zij * yij * b5pp
                  tzzpp = zij * zij * b5pp - a3pp

                  if ( abs(poli)*abs(polj) .gt. tiny*tiny ) then
                     amat(ix,jx) = amat(ix,jx) - 0.5d0 * poli * txxpp
                     amat(ix,jy) = amat(ix,jy) - 0.5d0 * poli * txypp
                     amat(ix,jz) = amat(ix,jz) - 0.5d0 * poli * txzpp
                     amat(iy,jx) = amat(iy,jx) - 0.5d0 * poli * tyxpp
                     amat(iy,jy) = amat(iy,jy) - 0.5d0 * poli * tyypp
                     amat(iy,jz) = amat(iy,jz) - 0.5d0 * poli * tyzpp
                     amat(iz,jx) = amat(iz,jx) - 0.5d0 * poli * tzxpp
                     amat(iz,jy) = amat(iz,jy) - 0.5d0 * poli * tzypp
                     amat(iz,jz) = amat(iz,jz) - 0.5d0 * poli * tzzpp

                     amat(jx,ix) = amat(jx,ix) - 0.5d0 * polj * txxpp
                     amat(jy,ix) = amat(jy,ix) - 0.5d0 * polj * txypp
                     amat(jz,ix) = amat(jz,ix) - 0.5d0 * polj * txzpp
                     amat(jx,iy) = amat(jx,iy) - 0.5d0 * polj * tyxpp
                     amat(jy,iy) = amat(jy,iy) - 0.5d0 * polj * tyypp
                     amat(jz,iy) = amat(jz,iy) - 0.5d0 * polj * tyzpp
                     amat(jx,iz) = amat(jx,iz) - 0.5d0 * polj * tzxpp
                     amat(jy,iz) = amat(jy,iz) - 0.5d0 * polj * tzypp
                     amat(jz,iz) = amat(jz,iz) - 0.5d0 * polj * tzzpp
                  end if

               end do
               end do
               end do

            end do

         end do

!-----------------------------------------------------------------------
!        /*   surface term                                            */
!-----------------------------------------------------------------------

         if ( ioption_ewald .eq. 1 ) then

            const_srf_pp = 2.d0 * pi / ( 3.d0 * volume )

            ex = 0.d0
            ey = 0.d0
            ez = 0.d0

            do k = 1, ncharge

               i  = i_q(k)

               xij = x(i,m)
               yij = y(i,m)
               zij = z(i,m)

               n1 = mbox(1,i,m)
               n2 = mbox(2,i,m)
               n3 = mbox(3,i,m)

               call pbc_unfold ( xij, yij, zij, n1, n2, n3 )

               qi = q(i)

               ex = ex + qi * xij
               ey = ey + qi * yij
               ez = ez + qi * zij

            end do

            do k = 1, ncharge

               poli = pol(k)

               ix = 3 * list_pol(k) - 2
               iy = 3 * list_pol(k) - 1
               iz = 3 * list_pol(k) - 0

               if ( abs(poli) .gt. tiny ) then
                  bvec(ix,1) = bvec(ix,1) - 2.d0*const_srf_pp*ex*poli
                  bvec(iy,1) = bvec(iy,1) - 2.d0*const_srf_pp*ey*poli
                  bvec(iz,1) = bvec(iz,1) - 2.d0*const_srf_pp*ez*poli
               end if

               do l = 1, ncharge

                  polj = pol(l)

                  jx = 3 * list_pol(l) - 2
                  jy = 3 * list_pol(l) - 1
                  jz = 3 * list_pol(l) - 0

                  if ( abs(poli)*abs(polj) .gt. tiny*tiny ) then
                     amat(ix,jx) = amat(ix,jx) + const_srf_pp*poli
                     amat(iy,jy) = amat(iy,jy) + const_srf_pp*poli
                     amat(iz,jz) = amat(iz,jz) + const_srf_pp*poli

                     amat(jx,ix) = amat(jx,ix) + const_srf_pp*polj
                     amat(jy,iy) = amat(jy,iy) + const_srf_pp*polj
                     amat(jz,iz) = amat(jz,iz) + const_srf_pp*polj
                  end if

               end do

            end do

         end if

!-----------------------------------------------------------------------
!        /*   parameters                                              */
!-----------------------------------------------------------------------

         ax = 2.d0*pi*boxinv(1,1)
         ay = 2.d0*pi*boxinv(1,2)
         az = 2.d0*pi*boxinv(1,3)
         bx = 2.d0*pi*boxinv(2,1)
         by = 2.d0*pi*boxinv(2,2)
         bz = 2.d0*pi*boxinv(2,3)
         cx = 2.d0*pi*boxinv(3,1)
         cy = 2.d0*pi*boxinv(3,2)
         cz = 2.d0*pi*boxinv(3,3)

         a2 = ax*ax + ay*ay + az*az
         b2 = bx*bx + by*by + bz*bz
         c2 = cx*cx + cy*cy + cz*cz

!-----------------------------------------------------------------------
!        /*   main loop                                               */
!-----------------------------------------------------------------------

         do k = 1, ncharge

            i  = i_q(k)

            eigax(i, 0)  = (1.d0,0.d0)
            eigay(i, 0)  = (1.d0,0.d0)
            eigaz(i, 0)  = (1.d0,0.d0)
            eigbx(i, 0)  = (1.d0,0.d0)
            eigby(i, 0)  = (1.d0,0.d0)
            eigbz(i, 0)  = (1.d0,0.d0)
            eigcx(i, 0)  = (1.d0,0.d0)
            eigcy(i, 0)  = (1.d0,0.d0)
            eigcz(i, 0)  = (1.d0,0.d0)

            if ( m1 .gt. 0 ) then
               eigax(i, 1)  = dcmplx ( cos(ax*x(i,m)), sin(ax*x(i,m)) )
               eigay(i, 1)  = dcmplx ( cos(ay*y(i,m)), sin(ay*y(i,m)) )
               eigaz(i, 1)  = dcmplx ( cos(az*z(i,m)), sin(az*z(i,m)) )
               eigax(i,-1)  = dconjg ( eigax(i,1) )
               eigay(i,-1)  = dconjg ( eigay(i,1) )
               eigaz(i,-1)  = dconjg ( eigaz(i,1) )
            end if
            if ( m2 .gt. 0 ) then
               eigbx(i, 1)  = dcmplx ( cos(bx*x(i,m)), sin(bx*x(i,m)) )
               eigby(i, 1)  = dcmplx ( cos(by*y(i,m)), sin(by*y(i,m)) )
               eigbz(i, 1)  = dcmplx ( cos(bz*z(i,m)), sin(bz*z(i,m)) )
               eigbx(i,-1)  = dconjg ( eigbx(i,1) )
               eigby(i,-1)  = dconjg ( eigby(i,1) )
               eigbz(i,-1)  = dconjg ( eigbz(i,1) )
            end if
            if ( m3 .gt. 0 ) then
               eigcx(i, 1)  = dcmplx ( cos(cx*x(i,m)), sin(cx*x(i,m)) )
               eigcy(i, 1)  = dcmplx ( cos(cy*y(i,m)), sin(cy*y(i,m)) )
               eigcz(i, 1)  = dcmplx ( cos(cz*z(i,m)), sin(cz*z(i,m)) )
               eigcx(i,-1)  = dconjg ( eigcx(i,1) )
               eigcy(i,-1)  = dconjg ( eigcy(i,1) )
               eigcz(i,-1)  = dconjg ( eigcz(i,1) )
            end if

            do l = 2, m1
               eigax(i, l)  = eigax(i,l-1)*eigax(i,1)
               eigay(i, l)  = eigay(i,l-1)*eigay(i,1)
               eigaz(i, l)  = eigaz(i,l-1)*eigaz(i,1)
               eigax(i,-l)  = dconjg ( eigax(i,l) )
               eigay(i,-l)  = dconjg ( eigay(i,l) )
               eigaz(i,-l)  = dconjg ( eigaz(i,l) )
            end do
            do l = 2, m2
               eigbx(i, l)  = eigbx(i,l-1)*eigbx(i,1)
               eigby(i, l)  = eigby(i,l-1)*eigby(i,1)
               eigbz(i, l)  = eigbz(i,l-1)*eigbz(i,1)
               eigbx(i,-l)  = dconjg ( eigbx(i,l) )
               eigby(i,-l)  = dconjg ( eigby(i,l) )
               eigbz(i,-l)  = dconjg ( eigbz(i,l) )
            end do
            do l = 2, m3
               eigcx(i, l)  = eigcx(i,l-1)*eigcx(i,1)
               eigcy(i, l)  = eigcy(i,l-1)*eigcy(i,1)
               eigcz(i, l)  = eigcz(i,l-1)*eigcz(i,1)
               eigcx(i,-l)  = dconjg ( eigcx(i,l) )
               eigcy(i,-l)  = dconjg ( eigcy(i,l) )
               eigcz(i,-l)  = dconjg ( eigcz(i,l) )
            end do

         end do

!-----------------------------------------------------------------------
!     /*   charge-dipole, dipole-dipole interactions                  */
!-----------------------------------------------------------------------

         al2 = a2*lmax_ewpol(1)**2
         bl2 = b2*lmax_ewpol(2)**2
         cl2 = c2*lmax_ewpol(3)**2

         g2max = min( al2, bl2, cl2 )

         factor_1 = (4.d0*pi)/(2.d0*volume)

         do la =              0, lmax_ewpol(1)
         do lb = -lmax_ewpol(2), lmax_ewpol(2)
         do lc = -lmax_ewpol(3), lmax_ewpol(3)

            l2 = la*la + lb*lb + lc*lc

            if ( l2 .eq. 0 ) cycle

            if ( la .eq. 0 ) then
               factor_2 = 1.d0
            else
               factor_2 = 2.d0
            end if

            gx = ax*la + bx*lb + cx*lc
            gy = ay*la + by*lb + cy*lc
            gz = az*la + bz*lb + cz*lc

            g2 = gx*gx + gy*gy + gz*gz

            if ( g2 .gt. g2max ) cycle

            factor_3 = exp(-g2/(4.d0*alpha_ewpol*alpha_ewpol))/g2

            qcos = 0.d0
            qsin = 0.d0

            pcos = 0.d0
            psin = 0.d0

            do k = 1, ncharge

               i  = i_q(k)

               cos_gxyz(k) = dreal(eigax(i,la)*eigbx(i,lb)*eigcx(i,lc) &
     &                            *eigay(i,la)*eigby(i,lb)*eigcy(i,lc) &
     &                            *eigaz(i,la)*eigbz(i,lb)*eigcz(i,lc))
               sin_gxyz(k) = dimag(eigax(i,la)*eigbx(i,lb)*eigcx(i,lc) &
     &                            *eigay(i,la)*eigby(i,lb)*eigcy(i,lc) &
     &                            *eigaz(i,la)*eigbz(i,lb)*eigcz(i,lc))

               qcos = qcos + q(i)*cos_gxyz(k)
               qsin = qsin + q(i)*sin_gxyz(k)

               poli = pol(k)

               if ( abs(poli) .gt. tiny ) then

                  gpx = gx*px(i,m)
                  gpy = gy*py(i,m)
                  gpz = gz*pz(i,m)

                  gpxyz = gpx + gpy + gpz

                  pcos = pcos + gpxyz*cos_gxyz(k)
                  psin = psin + gpxyz*sin_gxyz(k)

               end if

            end do

            gxx = gx * gx
            gxy = gx * gy
            gxz = gx * gz
            gyy = gy * gy
            gyz = gy * gz
            gzz = gz * gz

            do k = 1, ncharge

               i  = i_q(k)

               poli = pol(k)

               if ( abs(poli) .lt. tiny ) cycle

!               cos_gxyz(k) = dreal(eigax(i,la)*eigbx(i,lb)*eigcx(i,lc)
!     &                            *eigay(i,la)*eigby(i,lb)*eigcy(i,lc)
!     &                            *eigaz(i,la)*eigbz(i,lb)*eigcz(i,lc))
!               sin_gxyz(k) = dimag(eigax(i,la)*eigbx(i,lb)*eigcx(i,lc)
!     &                            *eigay(i,la)*eigby(i,lb)*eigcy(i,lc)
!     &                            *eigaz(i,la)*eigbz(i,lb)*eigcz(i,lc))

               factor_4 = cos_gxyz(k)*qsin - sin_gxyz(k)*qcos

               factor_5 = 2.d0*factor_1*factor_2*factor_3*factor_4

               ix = 3 * list_pol(k) - 2
               iy = 3 * list_pol(k) - 1
               iz = 3 * list_pol(k) - 0

               bvec(ix,1) = bvec(ix,1) - poli * factor_5 * gx
               bvec(iy,1) = bvec(iy,1) - poli * factor_5 * gy
               bvec(iz,1) = bvec(iz,1) - poli * factor_5 * gz

               do l = 1, ncharge

                  j  = i_q(l)

                  polj = pol(l)

                  if ( abs(polj) .lt. tiny ) cycle

!                  cos_hxyz(l) &
!     &               = dreal(eigax(j,la)*eigbx(j,lb)*eigcx(j,lc) &
!     &                      *eigay(j,la)*eigby(j,lb)*eigcy(j,lc) &
!     &                      *eigaz(j,la)*eigbz(j,lb)*eigcz(j,lc))
!                  sin_hxyz(l) &
!     &               = dimag(eigax(j,la)*eigbx(j,lb)*eigcx(j,lc) &
!     &                      *eigay(j,la)*eigby(j,lb)*eigcy(j,lc) &
!     &                      *eigaz(j,la)*eigbz(j,lb)*eigcz(j,lc))

                  expgh = cos_gxyz(k)*cos_gxyz(l) &
     &                  + sin_gxyz(k)*sin_gxyz(l)

                  factor_6 = 2.d0*factor_1*factor_2*factor_3*expgh*poli

                  jx = 3 * list_pol(l) - 2
                  jy = 3 * list_pol(l) - 1
                  jz = 3 * list_pol(l) - 0

                  amat(ix,jx) = amat(ix,jx) + gxx*factor_6
                  amat(ix,jy) = amat(ix,jy) + gxy*factor_6
                  amat(ix,jz) = amat(ix,jz) + gxz*factor_6
                  amat(iy,jx) = amat(iy,jx) + gxy*factor_6
                  amat(iy,jy) = amat(iy,jy) + gyy*factor_6
                  amat(iy,jz) = amat(iy,jz) + gyz*factor_6
                  amat(iz,jx) = amat(iz,jx) + gxz*factor_6
                  amat(iz,jy) = amat(iz,jy) + gyz*factor_6
                  amat(iz,jz) = amat(iz,jz) + gzz*factor_6

               end do

            end do

         end do
         end do
         end do

!-----------------------------------------------------------------------
!        /*   solve linear equation                                   */
!-----------------------------------------------------------------------

#ifdef nolapack
         write( 6, '(a)' ) &
     &      'Error - Diagonalization routine not linked.'
         call error_handling &
     &      ( 1, 'subroutine field_pol_coulomb', 28 )
#else
         call dgesv( npol3, 1, amat, npol3, ipiv, bvec, npol3, info )
#endif

!-----------------------------------------------------------------------
!        /*   substitution                                            */
!-----------------------------------------------------------------------

         do k = 1, ncharge

            i = i_q(k)

            poli = pol(k)

            if ( abs(poli) .lt. tiny ) cycle

            ix = 3 * list_pol(k) - 2
            iy = 3 * list_pol(k) - 1
            iz = 3 * list_pol(k) - 0

            px(i,m) = bvec(ix,1)
            py(i,m) = bvec(iy,1)
            pz(i,m) = bvec(iz,1)

         end do

!-----------------------------------------------------------------------
!     /*   loop of beads: end                                         */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   memory deallocation                                        */
!-----------------------------------------------------------------------

      if ( allocated(amat) ) deallocate( amat )
      if ( allocated(bvec) ) deallocate( bvec )
      if ( allocated(ipiv) ) deallocate( ipiv )
      if ( allocated(cos_gxyz) ) deallocate(cos_gxyz)
      if ( allocated(sin_gxyz) ) deallocate(sin_gxyz)

      return
      end





!***********************************************************************
      subroutine dampfunc_qq( r, k, l, s0qq, s1qq )
!***********************************************************************

      use common_variables, only : pi

      use mm_variables, only : dampform_qq, damppar_qq, kind_q

      implicit none
      integer :: k, l, kq, lq
      real(8) :: r, s0qq, s1qq, a, b, c, d, r2, ainv, z1, z2, z3, z4, &
     &           erf_0, erf_z, exp_z, gammq, gam_z
      real(8), save :: gamma_twothirds
      integer, save :: iset = 0
      character(len=8) :: dampform

      kq = kind_q(k)
      lq = kind_q(l)

      dampform = dampform_qq(kq,lq)

      if      ( dampform(1:4) .eq. 'OSS ' ) then

        a = damppar_qq(1,kq,lq)
        b = damppar_qq(2,kq,lq)
        r2 = r * r
        c = a * exp( - b * r )
        d = 1.d0 / ( r2 + c )
        s0qq = r2 * d
        s1qq = 2.d0*r*d - r2*d*d * ( 2.d0*r - b*c )

      else if ( dampform(1:4) .eq. 'LIN ' ) then

         a = damppar_qq(1,kq,lq)

         ainv = 1.d0 / a
         z1 = r * ainv
         z2 = z1 * z1
         z3 = z1 * z2
         z4 = z2 * z2

         if ( z1 .lt. 1.d0 ) then
            s0qq = 2.d0*z1 - 2.d0*z3 + z4
            s1qq = ( 2.d0 - 6.d0*z2 + 4.d0*z3 ) * ainv
         else
            s0qq = 1.d0
            s1qq = 0.d0
         end if

      else if ( dampform(1:4) .eq. 'EXP ' ) then

         if ( iset .eq. 0 ) then
            gamma_twothirds = gamma(2.d0/3.d0)
            iset = 1
         end if

         a = damppar_qq(1,kq,lq)

         ainv = 1.d0 / a
         z1 = r * ainv
         z2 = z1 * z1
         z3 = z1 * z2

         exp_z = exp( - z3 )
         gam_z = gammq(2.d0/3.d0,z3)*gamma_twothirds

         s0qq = 1.d0 - exp_z + z1*gam_z
         s1qq = gam_z * ainv

      else if ( dampform(1:4) .eq. 'GAU ' ) then

         a = damppar_qq(1,kq,lq)

         ainv = 1.d0 / a
         z1 = r * ainv
         z2 = z1 * z1

         c = 2.d0 / sqrt(pi)
         erf_z = erf_0(z1)
         exp_z = c * exp( - z2 )

         s0qq = erf_z
         s1qq = ( exp_z ) * ainv

      else if ( dampform(1:5) .eq. 'ZERO ' ) then

         s0qq = 0.d0
         s1qq = 0.d0

      else if ( dampform(1:5) .eq. 'NONE ' ) then

         s0qq = 1.d0
         s1qq = 0.d0

      else

         s0qq = 1.d0
         s1qq = 0.d0

      end if

      return
      end





!***********************************************************************
      subroutine dampfunc_qp( r, k, l, s0qp, s1qp )
!***********************************************************************

      use common_variables, only : pi

      use mm_variables, only : dampform_qp, damppar_qp, kind_q

      implicit none
      integer :: k, l, kq, lq
      real(8) :: r, s0qp, s1qp, a, b, c, d, r2, ainv, z1, z2, z3, z4, &
     &           erf_0, erf_z, exp_z
      character(len=8) :: dampform

      kq = kind_q(k)
      lq = kind_q(l)

      dampform = dampform_qp(kq,lq)

      if ( dampform(1:4) .eq. 'OSS ' ) then

        a = damppar_qp(1,kq,lq)
        b = damppar_qp(2,kq,lq)
        r2 = r * r
        c = a * exp( - b * r )
        d = 1.d0 / ( r2 + c )
        s0qp = r2 * d
        s1qp = 2.d0*r*d - r2*d*d * ( 2.d0*r - b*c )

      else if ( dampform(1:4) .eq. 'LIN ' ) then

         a = damppar_qp(1,kq,lq)

         ainv = 1.d0 / a
         z1 = r * ainv
         z2 = z1 * z1
         z3 = z1 * z2
         z4 = z2 * z2

         if ( z1 .lt. 1.d0 ) then
            s0qp = 4.d0*z3 - 3.d0*z4
            s1qp = ( 12.d0*z2 - 12.d0*z3 ) * ainv
         else
            s0qp = 1.d0
            s1qp = 0.d0
         end if

      else if ( dampform(1:4) .eq. 'EXP ' ) then

         a = damppar_qp(1,kq,lq)

         ainv = 1.d0 / a
         z1 = r * ainv
         z2 = z1 * z1
         z3 = z1 * z2

         exp_z = exp( - z3 )

         s0qp = 1.d0 - exp_z
         s1qp = ( 3.d0*z2*exp_z ) * ainv

      else if ( dampform(1:4) .eq. 'GAU ' ) then

         a = damppar_qp(1,kq,lq)

         ainv = 1.d0 / a
         z1 = r * ainv
         z2 = z1 * z1

         c = 2.d0 / sqrt(pi)
         erf_z = erf_0(z1)
         exp_z = c*exp( - z2 )

         s0qp = erf_z - z1*exp_z
         s1qp = ( 2.d0*z2*exp_z ) * ainv

      else if ( dampform(1:5) .eq. 'ZERO ' ) then

         s0qp = 0.d0
         s1qp = 0.d0

      else if ( dampform(1:5) .eq. 'NONE ' ) then

         s0qp = 1.d0
         s1qp = 0.d0

      else

         s0qp = 1.d0
         s1qp = 0.d0

      end if

      return
      end





!***********************************************************************
      subroutine dampfunc_pp( r, k, l, s0pp, s1pp, t0pp, t1pp )
!***********************************************************************

      use common_variables, only : pi

      use mm_variables, only : dampform_pp, damppar_pp, kind_q

      implicit none
      integer :: k, l, kq, lq
      real(8) :: r, s0pp, s1pp, t0pp, t1pp, a, b, c, d, r2, ainv, &
     &           z1, z2, z3, z4, z5, erf_0, erf_z, exp_z
      character(len=8) :: dampform

      kq = kind_q(k)
      lq = kind_q(l)

      dampform = dampform_pp(kq,lq)

      if ( dampform(1:4) .eq. 'OSS ' ) then

         a = damppar_pp(1,kq,lq)
         b = damppar_pp(2,kq,lq)
         r2 = r * r
         c = a * exp( - b * r )
         d = 1.d0 / ( r2 + c )
         s0pp = r2 * d
         s1pp = 2.d0*r*d - r2*d*d * ( 2.d0*r - b*c )
         t0pp = s0pp
         t1pp = s1pp

      else if ( dampform(1:4) .eq. 'LIN ' ) then

         a = damppar_pp(1,kq,lq)

         ainv = 1.d0 / a
         z1 = r * ainv
         z2 = z1 * z1
         z3 = z1 * z2
         z4 = z2 * z2

         if ( z1 .lt. 1.d0 ) then
            s0pp = 4.d0*z3 - 3.d0*z4
            t0pp = z4
            s1pp = ( 12.d0*z2 - 12.d0*z3 ) * ainv
            t1pp = ( 4.d0*z3 ) * ainv
         else
            s0pp = 1.d0
            t0pp = 1.d0
            s1pp = 0.d0
            t1pp = 0.d0
         end if

      else if ( dampform(1:4) .eq. 'EXP ' ) then

         a = damppar_pp(1,kq,lq)

         ainv = 1.d0 / a
         z1 = r * ainv
         z2 = z1 * z1
         z3 = z1 * z2
         z5 = z2 * z3

         exp_z = exp( - z3 )

         s0pp = 1.d0 - exp_z
         t0pp = 1.d0 - exp_z - z3*exp_z
         s1pp = ( 3.d0*z2*exp_z ) * ainv
         t1pp = ( 3.d0*z2*exp_z + 3.d0*z5*exp_z ) * ainv

      else if ( dampform(1:4) .eq. 'GAU ' ) then

         a = damppar_pp(1,kq,lq)

         ainv = 1.d0 / a
         z1 = r * ainv
         z2 = z1 * z1
         z3 = z1 * z2
         z4 = z2 * z2

         b = 2.d0 / 3.d0
         c = 2.d0 / sqrt(pi)
         erf_z = erf_0(z1)
         exp_z = c * exp( - z2 )

         s0pp = erf_z - z1*exp_z
         t0pp = erf_z - z1*exp_z - b*z3*exp_z
         s1pp = ( 2.d0*z2*exp_z ) * ainv
         t1pp = ( 2.d0*z2*exp_z + 2.d0*b*z4*exp_z ) * ainv

      else if ( dampform(1:5) .eq. 'ZERO ' ) then

         s0pp = 0.d0
         s1pp = 0.d0
         t0pp = s0pp
         t1pp = s1pp

      else if ( dampform(1:5) .eq. 'NONE ' ) then

         s0pp = 1.d0
         s1pp = 0.d0
         t0pp = s0pp
         t1pp = s1pp

      else

         s0pp = 1.d0
         s1pp = 0.d0
         t0pp = s0pp
         t1pp = s1pp

      end if

      return
      end





!***********************************************************************
      real(8) function gammq( a, x )
!***********************************************************************

      implicit none

      real(8) :: a, x, gammcf, gamser, gln

      if ( x .lt. a + 1.d0 ) then
         call gser( gamser, a, x, gln )
         gammq = 1.d0 - gamser
      else
         call gcf( gammcf, a, x, gln )
         gammq = gammcf
      end if

      return
      end





!***********************************************************************
      subroutine gcf( gammcf, a, x, gln )
!***********************************************************************

      implicit none

      integer :: i
      integer, parameter :: itmax = 10000000
      real(8) :: a, gammcf, gln, x
      real(8) :: an, b, c, d, del, h
      real(8), parameter :: eps = 3.d-16, fpmin = 1.d-30
      real(8), external :: gammln

      gln = gammln(a)
      b = x + 1.d0 - a
      c = 1.d0 / fpmin
      d = 1.d0 / b
      h = d
      do i = 1, itmax
         an = - i * ( i - a )
         b = b + 2.d0
         d = an * d + b
         if ( abs(d) .lt. fpmin ) d = fpmin
            c = b + an / c
            if ( abs(c) .lt. fpmin ) c = fpmin
            d = 1.d0 / d
            del = d * c
            h = h * del
            if ( abs(del-1.d0) .lt. eps ) exit
      end do
      gammcf = exp( - x + a*log(x) - gln ) * h

      return
      end





!***********************************************************************
      subroutine gser( gamser, a, x, gln )
!***********************************************************************

      implicit none

      integer :: n
      integer, parameter :: itmax = 100
      real(8), parameter :: eps = 3.d-16
      real(8) :: a, gamser, gln, x, ap, del, sum 
      real(8), external :: gammln

      gln = gammln(a)
      if ( x .le. 0.d0 ) then
         gamser = 0.d0
         return
      end if
      ap = a
      sum = 1.d0 / a
      del = sum
      do n = 1, itmax
         ap = ap + 1.d0
         del = del * x / ap
         sum = sum + del
         if ( abs(del) .lt. abs(sum)*eps) exit
      end do
      gamser = sum * exp( -x + a*log(x) - gln )

      return
      end





!***********************************************************************
      real(8) function gammln( xx )
!***********************************************************************

      implicit none

      real(8) :: xx, ser, stp, tmp, x, y, cof(6)
      integer :: j 
      save cof, stp
      data cof, stp / 76.18009172947146d0,   -86.50532032941677d0, &
     &                24.01409824083091d0,   -1.231739572450155d0, &
     &                0.1208650973866179d-2, -0.5395239384953d-5, &
     &                2.5066282746310005d0 /

      x = xx
      y = x
      tmp = x + 5.5d0
      tmp = ( x + 0.5d0 ) * log(tmp) - tmp
      ser = 1.000000000190015d0
      do j = 1, 6
         y = y + 1.d0
         ser = ser + cof(j) / y
      end do
      gammln = tmp + log( stp * ser / x )

      return
      end
