!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, J. Koga
!      Last updated:    January, 2022 by J. Koga
!      Description:     energy and force from PHASE0 calculation
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine force_phase0_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, ry_energy, au_energy, au_length, &
     &   box, volume, &
     &   vir, istep, natom, nbead, myrank_main, nprocs_main, &
     &   myrank_sub, nprocs_sub

      use phase0_variables, only : &
     &   coord_x, coord_y, coord_z, force_x, force_y, force_z, &
     &   total_energy, stress_tensor, pimd_istep, pimd_ibead, cell_param


!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr, ibead, ierr_bead, i, j
      real(8) :: stress_tensor_sum(3,3)

      integer, save :: ierr1 = 0
      integer, save :: ierr_bead1 = 0

!-----------------------------------------------------------------------
!     /*   initialize stress tensor                                   */
!-----------------------------------------------------------------------
!
      stress_tensor(:,:) = 0.d0

!     /*   set box   */
      cell_param(:,:) = box(:,:)

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!     /*   skip if `ibead is not my job'                              */
!-----------------------------------------------------------------------

         if ( mod(ibead-1,nprocs_main) .ne. myrank_main ) cycle

!-----------------------------------------------------------------------
!        /*   execute QE subroutine                                   */
!-----------------------------------------------------------------------

!        /*   set istep   */
         pimd_istep = istep

!        /*   set ibaed   */
         pimd_ibead = ibead

!        /*   set coordinate   */
         do i = 1, natom
            coord_x(i) = x(i,ibead)
            coord_y(i) = y(i,ibead)
            coord_z(i) = z(i,ibead)
         end do

!c        /*   set wavefunctions   */
!         vwave => vwaves(ibead)

!        /*   error flag   */
         ierr = 0
         ierr_bead = 0

!        /*   execute PHASE/0 subroutine   */
#ifdef phase0
         call phase0_set_coordinate_xyz(coord_x, coord_y, coord_z) 
         call phase0_scf(total_energy, force_x, force_y, force_z,  &
     &        stress_tensor)
#endif

!        /*   error flag   */       
         if ( ierr .ne. 0 ) then
            ierr1 = ierr
            ierr_bead1 = ierr_bead
            exit
         end if

!        /*   total energy   */
         pot(ibead) = total_energy

!        /*   force   */
         do i = 1, natom
            fx(i,ibead) = force_x(i)
            fy(i,ibead) = force_y(i)
            fz(i,ibead) = force_z(i)
         end do

!        /*   stress   */
         stress_tensor_sum(:,:) = stress_tensor_sum(:,:) &
     &                             + stress_tensor(:,:)

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

!    /*   error termination   */

      if ( ierr1 .ne. 0 ) then
          if(myrank_sub .eq. 0) then
          write( 6, '(a)' ) &
     &           'Error - unable to execute PHASE0. See log file.'
          write( 6, '(a)' )
          write( 6, '(a)' )
          write( 6, '(a)' )  &
     &   '############################################################'
          write( 6, '(a)' )  &
     &   '############################################################'
          write( 6, '(a)' )
          write( 6, '(a,i5)' )  &
     &   ' ##ERROR## convergence of electronic state NOT achieved'// &
     &   ' in bead ', ierr_bead
          write( 6, '(a)' )
          write( 6, '(a)' )  &
     &   '############################################################'
               write( 6, '(a)' )  &
     &   '############################################################'
          write( 6, '(a)' ) 
         call error_handling_MPI &
     &           ( 1, 'subroutine force_phase0_MPI', 27 )
         end if
      end if

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   loop of process of a same bead  */
      do i = 1, nprocs_sub

!        /*   same bead  */
         if ( myrank_sub .eq. i-1 ) then

!           /*   potential   */
            call my_mpi_allreduce_real_1_main ( pot, nbead )

!           /*   force   */
            call my_mpi_allreduce_real_2_main ( fx, natom, nbead )
            call my_mpi_allreduce_real_2_main ( fy, natom, nbead )
            call my_mpi_allreduce_real_2_main ( fz, natom, nbead )

!           /*   stress   */
            call my_mpi_allreduce_real_2_main ( stress_tensor, 3, 3 )

!        /*   same bead  */
         end if

!     /*   loop of process of a same bead  */
      end do

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

      do j = 1, nbead

         vir(1,1) = vir(1,1) + stress_tensor(1,1) * volume / nbead
         vir(1,2) = vir(1,2) + stress_tensor(1,2) * volume / nbead
         vir(1,3) = vir(1,3) + stress_tensor(1,3) * volume / nbead
         vir(2,1) = vir(2,1) + stress_tensor(2,1) * volume / nbead
         vir(2,2) = vir(2,2) + stress_tensor(2,2) * volume / nbead
         vir(2,3) = vir(2,3) + stress_tensor(2,3) * volume / nbead
         vir(3,1) = vir(3,1) + stress_tensor(3,1) * volume / nbead
         vir(3,2) = vir(3,2) + stress_tensor(3,2) * volume / nbead
         vir(3,3) = vir(3,3) + stress_tensor(3,3) * volume / nbead

      end do

      return
      end





!***********************************************************************
      subroutine init_phase0_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

!      use common_variables, only :
!     &   myrank, iounit, natom, nstep, nbead, method, mpi_comm_sub,
!     &   mpi_comm_main, mpi_comm_pimd, myrank_main, nprocs_sub

      use common_variables, only : &
     &   myrank, iounit, natom, nstep, nbead, method, nprocs_sub

#ifdef phase0
      use common_variables, only : mpi_comm_sub, myrank_main
#endif

      use phase0_variables, only : coord_x, coord_y, coord_z, &
     &                             force_x, force_y, force_z, &
     &                             ne, nk, ng

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr
      integer :: iset = 0

!-----------------------------------------------------------------------
!     /*   set PHASE0 parameters                                      */
!-----------------------------------------------------------------------

!     /*   parent process only   */
      if ( myrank .eq. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag ( '<phase0_proc>', 13, iounit, ierr )

!        /*   read a line   */
         read ( iounit, *, iostat=ierr ) &
     &      ne, nk, ng

!        /*   file close   */
         close ( iounit )

!        /*   otherwise, read from default input   */
         if ( ierr .ne. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'input_default.dat' )

!           /*   search for tag    */
            call search_tag( '<phase0_proc>', 13, iounit, ierr )

!           /*   read a line   */
            read ( iounit, *, iostat=ierr ) &
     &         ne, nk, ng

!           /*   file close   */
            close ( iounit )

!        /*   otherwise, read from default input   */
         end if

!     /*   parent process only   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   error message   */
      if ( ierr .ne. 0 ) then
         if ( myrank .eq. 0 ) then
            write ( 6, '(a)' ) 'Error - <phase0_proc> read incorrectly.'
            write ( 6, '(a)' )
         end if
      end if

!     /*   error termination   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine init_phase0_MPI', 26 )

!     /*   communicate   */
      call my_mpi_bcast_int_0( ne )

!     /*   communicate   */
      call my_mpi_bcast_int_0( nk )

!     /*   communicate   */
      call my_mpi_bcast_int_0( ng )

!-----------------------------------------------------------------------
!     /*   check values                                               */
!-----------------------------------------------------------------------

      if ( ne .le. 0 ) then

         if ( nk .le. 0 ) then

            if ( ng .le. 0 ) then

               ne = nint(dble(nprocs_sub)**(1.d0/3.d0))
               nk = nint(dble(nprocs_sub)**(1.d0/3.d0))
               ng = nint(dble(nprocs_sub)**(1.d0/3.d0))
               iset = 1

            else

               ne = nint(sqrt(dble(nprocs_sub)/dble(ng)))
               nk = nint(sqrt(dble(nprocs_sub)/dble(ng)))
               iset = 1

            end if

         else

            if ( ng .le. 0 ) then

               ne = nint(sqrt(dble(nprocs_sub)/dble(nk)))
               ng = nint(sqrt(dble(nprocs_sub)/dble(nk)))
               iset = 1

            else

               ne = nint(dble(nprocs_sub)/dble(nk*ng))
               iset = 1

            end if

         end if

      else

         if ( nk .le. 0 ) then

            if ( ng .le. 0 ) then

               nk = nint(sqrt(dble(nprocs_sub)/dble(ne)))
               ng = nint(sqrt(dble(nprocs_sub)/dble(ne)))
               iset = 1

            else

               nk = nint(dble(nprocs_sub)/dble(ng*ne))
               iset = 1

            end if

         else

            if ( ng .le. 0 ) then

               ng = nint(dble(nprocs_sub)/dble(ne*nk))
               iset = 1

            end if

         end if

      end if

      if ( (ne*nk*ng) .ne. nprocs_sub ) then

         ne = nprocs_sub
         nk = 1
         ng = 1
         iset = 1

      end if

      if ( iset .eq. 1 ) then

         if ( myrank .eq. 0 ) then
            write ( 6, '(a,3i8,a)' ) &
     &         'Reset values of ne, nk, ng =', ne, nk, ng, '.'
            write ( 6, '(a)' )
         end if

      end if

!-----------------------------------------------------------------------
!     /*   reset nstep (this is necessary somehow)                    */
!-----------------------------------------------------------------------

      if      ( method(1:10) .eq. 'TESTFORCE '  ) then

         nstep = (6*natom+1) * nbead

      else if ( method(1:11) .eq. 'TESTVIRIAL ' ) then

         nstep = (6*9+1) * nbead

      end if

!-----------------------------------------------------------------------
!     /*   initialize PHASE0                                          */
!-----------------------------------------------------------------------

      if(.not.allocated(coord_x)) allocate(coord_x(natom))
      if(.not.allocated(coord_y)) allocate(coord_y(natom))
      if(.not.allocated(coord_z)) allocate(coord_z(natom))
      if(.not.allocated(force_x)) allocate(force_x(natom))
      if(.not.allocated(force_y)) allocate(force_y(natom))
      if(.not.allocated(force_z)) allocate(force_z(natom))

#ifdef phase0
      call phase0_initialize(mpi_comm_sub, myrank_main, ne, nk, ng, 7)
#endif

!-----------------------------------------------------------------------
!     /*   error termination                                          */
!-----------------------------------------------------------------------

      if( ierr .ne. 0 ) then

         if ( myrank .eq. 0 ) then

            write( 6, '(a)' ) &
     &        'Error - unable to initialize PHASE0. See log file.'
            write( 6, '(a)' )

         end if

         call error_handling_MPI &
     &        ( 1, 'subroutine init_phase0_MPI', 26 )

      end if

      return
      end





!***********************************************************************
      subroutine finalize_phase0_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : myrank
      use phase0_variables, only : coord_x, coord_y, coord_z, &
     &                             force_x, force_y, force_z

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr

!-----------------------------------------------------------------------
!     /*   finalize PHASE0                                            */
!-----------------------------------------------------------------------

!     /*   finalize QE subroutine   */

      ierr = 0

      call phase0_finalize( ierr )

      deallocate(coord_x)
      deallocate(coord_y)
      deallocate(coord_z)
      deallocate(force_x)
      deallocate(force_y)
      deallocate(force_z)

      if ( ierr .ne. 0 ) then

         if ( myrank .eq. 0 ) then
            write( 6, '(a)' ) &
     &        'Error - unable to finalize PHASE0. See log file.'
            write( 6, '(a)' )
         end if

         call error_handling_MPI &
     &        ( 1, 'subroutine finalize_phase0_MPI', 30 )

      end if

      return
      end





!=======================================================================
!
!     The following dummy subroutines are applied
!     when libphase.a is not available
!
!=======================================================================

#ifndef phase0

!***********************************************************************
      subroutine phase0_init( ierr )
!***********************************************************************

      use common_variables, only : myrank

      implicit none

      integer, intent(out) :: ierr

      ierr = 1

      if ( myrank .eq. 0 ) then

         write( 6, '(a)' ) &
     &      'Error termination - PHASE0 is not linked to PIMD.'
         write( 6, '(a)' )
         write( 6, '(a)' ) &
     &      'Recompile pimd.mpi.x with libphase.a and libesm.a ' // &
     &      'with the options'
         write( 6, '(a)' )
         write( 6, '(a)' ) &
     &       '  PHASE0 = -Dphase0'
         write( 6, '(a)' ) &
     &       '  LIBQE = -L../lib/phase -lphase -lesm'
         write( 6, '(a)' )

      end if

      call my_mpi_barrier
      call my_mpi_abort

      return
      end

#endif





#ifndef phase0

!***********************************************************************
      subroutine phase0_force( ierr, ierr_bead )
!***********************************************************************

      use common_variables, only : myrank

      implicit none

      integer, intent(out) :: ierr
      integer, intent(out) :: ierr_bead

      ierr = 1
      ierr_bead = 1

      if( myrank .eq. 0 ) then

         write(6, '(a)') 'Error termination.'
         write(6, '(a)') ''
         write(6, '(a)') 'ipotential=PHASE0 is not available because'
         write(6, '(a)') 'PHASE0 subroutines were not linked.'
         write(6, '(a)') 'Compile the pimd.mpi.x with the followings.'
         write(6, '(a)') '(You need libphase.a and libesm.a in '
         write(6, '(a)') '../lib/phase)'
         write(6, '(a)') ''
         write(6, '(a)') '  FLAGMP = -Dphase0'
         write(6, '(a)') '  LINKMP = -L../lib/phase -lphase -lesm'
         write(6, '(a)') ''
      end if

      call my_mpi_barrier
      call my_mpi_abort

      return
      end

#endif





#ifndef phase0

!***********************************************************************
      subroutine phase0_finalize( ierr )
!***********************************************************************

      use common_variables, only : myrank

      implicit none

      integer, intent(out) :: ierr

      ierr = 1

      if( myrank .eq. 0 ) then

         write(6, '(a)') 'Error termination.'
         write(6, '(a)') ''
         write(6, '(a)') &
     &      'PHASE0 is not available. To link PHASE0 routines, copy' &
     &     // ' libphase.a and libesm.a to'
         write(6, '(a)') &
     &      ' ../lib/phase directory, then recompile pimd.mpi.x with ' &
     &    //' the following options'
         write(6, '(a)') &
     &      'included in makefile.'
         write(6, '(a)') ''
         write(6, '(a)') '  FLAGMP = -Dphase0'
         write(6, '(a)') '  LINKMP = -L../lib/phase -lphase0 -lesm'
         write(6, '(a)') ''

      end if

      call my_mpi_barrier
      call my_mpi_abort

      return
      end

#endif

