!///////////////////////////////////////////////////////////////////////
!
!      Author:          Y. Nagai, M. Shiga, T. Hasegawa
!      Last updated:    Dec. 25, 2025 by T. Hasegawa
!      Description:     energy and force from Matlantis PFP
!
!///////////////////////////////////////////////////////////////////////

#ifdef pfp

!***********************************************************************
      module pfp_variables
!***********************************************************************
!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

!     //   c language
      use, intrinsic :: iso_c_binding

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   reset
      implicit none

!     //   position array
      real(8), allocatable :: x_i(:), y_i(:), z_i(:)

!     //   force array
      real(8), allocatable :: fx_i(:), fy_i(:), fz_i(:)

!     // atomic number for i-th atom type
      integer, allocatable :: atomic_num(:)

!     // atom types of each atom
      integer, allocatable :: atom_type(:)

!     //   object of the pfp interface
      type(c_ptr) :: obj;

!     //   version and model
      character(len=:), allocatable :: version
      character(len=:), allocatable :: model

!***********************************************************************
      end module pfp_variables
!***********************************************************************





!***********************************************************************
      subroutine force_pfp_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   run Matlantis pfp                                          */
!-----------------------------------------------------------------------

      call run_pfp_MPI()

      return
      end





!***********************************************************************
      subroutine initialize_pfp_MPI()
!***********************************************************************
!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

!     //   shared variables
      use common_variables, only: &
     &   nbead, natom, int_spec, mspec, iounit_pfp, myrank

!     //   shared variables
      use pfp_variables, only: &
     &   x_i, y_i, z_i, fx_i, fy_i, fz_i, obj, &
     &   atomic_num, atom_type, version, model

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   reset
      implicit none

!     //   integers
      integer :: ibead

!     //   integers
      integer :: ierr = 0

!     //   flag
      integer, save :: iset = 0

      character(len=80) :: line1
      character(len=80) :: line2

      integer :: i, j, cindex
      logical :: found
      logical :: iexist

!-----------------------------------------------------------------------
!     //   interface
!-----------------------------------------------------------------------

      interface

         function initpfp( version, mode, version_len, mode_len ) &
     &                     bind(C, name='initpfp_')

         use, intrinsic :: iso_c_binding
         character(kind=c_char), intent(in) :: version(*)
         character(kind=c_char), intent(in) :: mode(*)
         integer(c_int), value :: version_len, mode_len
         type(c_ptr) :: initpfp

         end function initpfp

      end interface

!-----------------------------------------------------------------------
!     //   initialize
!-----------------------------------------------------------------------

!     //   initial visit
      if ( iset .eq. 0 ) then

!-----------------------------------------------------------------------
!        //   read pfp model file
!-----------------------------------------------------------------------

         inquire( file = 'pfp_model.dat', exist = iexist )

         if ( iexist ) then

            line1 = ''
            line2 = ''

            if ( myrank .eq. 0 ) then

            open ( unit=iounit_pfp, file='pfp_model.dat', status='old' )
            read ( iounit_pfp, '(a80)', iostat=ierr ) line1
            read ( iounit_pfp, '(a80)', iostat=ierr ) line2
            close( iounit_pfp )

            end if

            call my_mpi_bcast_char_0( line1, len(line1) )
            call my_mpi_bcast_char_0( line2, len(line2) )

            if ( allocated(version) ) deallocate(version)
            if ( allocated(model) )   deallocate(model)

            if ( ( ierr .ne. 0 ) .or. &
    &            ( len_trim(line1)+len_trim(line2) .le. 0 ) ) then

               allocate( character(len=20) :: version )
               allocate( character(len=20) :: model )

               version = '                    '
               model   = '                    '

            else

               allocate( character(len=len_trim(line1)) :: version )
               allocate( character(len=len_trim(line2)) :: model )

               version = trim(line1)
               model = trim(line2)

            end if

!-----------------------------------------------------------------------
!        //   set pfp version and model
!-----------------------------------------------------------------------

         else

!           //   set default values

            if ( allocated(version) ) deallocate(version)
            if ( allocated(model) )   deallocate(model)

            allocate( character(len=20) :: version )
            allocate( character(len=20) :: model )

            version = '                    '
            model   = '                    '

         end if

!-----------------------------------------------------------------------
!        //   initialize pfp
!-----------------------------------------------------------------------

         obj = initpfp( version, model, len(version), len(model) )

!-----------------------------------------------------------------------
!        //   memory allocation
!-----------------------------------------------------------------------

         if ( allocated(x_i) ) deallocate(x_i)
         if ( allocated(y_i) ) deallocate(y_i)
         if ( allocated(z_i) ) deallocate(z_i)

         if ( allocated(fx_i) ) deallocate(fx_i)
         if ( allocated(fy_i) ) deallocate(fy_i)
         if ( allocated(fz_i) ) deallocate(fz_i)

         if ( allocated(atomic_num) ) deallocate(atomic_num)
         if ( allocated(atom_type) ) deallocate(atom_type)

         allocate( x_i(natom) )
         allocate( y_i(natom) )
         allocate( z_i(natom) )

         allocate( fx_i(natom) )
         allocate( fy_i(natom) )
         allocate( fz_i(natom) )

         allocate( atomic_num(mspec) )
         allocate( atom_type(natom) )

!-----------------------------------------------------------------------
!        //   make atomic_num table
!-----------------------------------------------------------------------

         atomic_num(1) = int_spec(1)
         cindex = 1

         do i = 2, natom
            found = .false.
            do j = 1, cindex
               if ( int_spec(i) == atomic_num(j)) then
                  found = .true.
                  exit
               end if
            end do
            if ( .not. found ) then
               cindex = cindex + 1
               atomic_num(cindex) = int_spec(i)
            end if
         end do

         do i = 1, natom
            do j = 1, mspec
               if ( int_spec(i) == atomic_num(j) ) then
                  atom_type(i) = j - 1
               end if
            end do
         end do

!        //   initial setting ended
         iset = 1

!     //   initial visit
      end if

      return
      end





!***********************************************************************
      subroutine finalize_pfp_MPI()
!***********************************************************************
!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use pfp_variables, only: &
     &   x_i, y_i, z_i, fx_i, fy_i, fz_i, obj, atomic_num, &
     &   atom_type, version, model

      use iso_c_binding, only: c_ptr

      implicit none

!-----------------------------------------------------------------------
!     //   interface
!-----------------------------------------------------------------------

      interface

         subroutine finpfp(obj) bind(C, name='finpfp_')

         use, intrinsic :: iso_c_binding
         type(c_ptr), value :: obj

         end subroutine finpfp

      end interface

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

      if ( allocated(x_i) ) deallocate( x_i )
      if ( allocated(y_i) ) deallocate( y_i )
      if ( allocated(z_i) ) deallocate( z_i )

      if ( allocated(fx_i) ) deallocate( fx_i )
      if ( allocated(fy_i) ) deallocate( fy_i )
      if ( allocated(fz_i) ) deallocate( fz_i )

      if ( allocated(atomic_num) ) deallocate( atomic_num )
      if ( allocated(atom_type) ) deallocate( atom_type )

      if ( allocated(version) ) deallocate( version )
      if ( allocated(model) ) deallocate( model )

!-----------------------------------------------------------------------
!     //   finalize pfp
!-----------------------------------------------------------------------

      call finpfp( obj )

      return
      end





!***********************************************************************
      subroutine titles_pfp_MPI()
!***********************************************************************
!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use pfp_variables, only: myrank
      use pfp_variables, only: version, model

      implicit none

!-----------------------------------------------------------------------
!     /*   variables                                                  */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then
         write( 6, '(2a)' ) '    Version :  ' , version 
         write( 6, '(2a)' ) '    Model   :  ' , model 
         write( 6, '(a)'  ) '               '
      end if

      return
      end





!***********************************************************************
      subroutine run_pfp_MPI()
!***********************************************************************
!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

!     //   shared variables
      use common_variables, only: &
     &   x, y, z, box, pot, fx, fy, fz, vir, mbox, nprocs, myrank, &
     &   natom, nbead, mspec, au_length, au_charge, au_energy

      use pfp_variables, only: &
     &   x_i, y_i, z_i, fx_i, fy_i, fz_i, obj, atomic_num, atom_type

      use iso_c_binding, only: c_loc, c_null_char

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     /*   reset   */
      implicit none

      real(8) :: virial_tensor_i(3,3)
      real(8) :: virial_tensor_sum(3,3)
      real(8) :: box_i(3,3)
      real(8) :: energy
      real(8) :: bohr2ang
      real(8) :: ev2Eh
      integer :: i, j, ibead
      real(8) :: xi, yi, zi

!-----------------------------------------------------------------------
!     //   interface
!-----------------------------------------------------------------------

      interface

         subroutine pfp_interface( &
     &      obj, x, y, z, box, fx, fy, fz, energy, virial, atomic_num, &
     &      nelement, atom_type, natom) bind(C, name='pfp_interface_' )

         use, intrinsic :: iso_c_binding
         type(c_ptr), value :: obj

         real(8), intent(in) :: x(*)
         real(8), intent(in) :: y(*)
         real(8), intent(in) :: z(*)
         real(8), intent(in) :: box(3,3)

         real(8), intent(out) :: fx(*)
         real(8), intent(out) :: fy(*)
         real(8), intent(out) :: fz(*)
         real(8), intent(out) :: energy
         real(8), intent(out) :: virial(3,3)

         integer, intent(in) :: atomic_num(*)
         integer, intent(in), value :: nelement
         integer, intent(in) :: atom_type(*)
         integer, intent(in), value :: natom

         end subroutine pfp_interface

      end interface

!-----------------------------------------------------------------------
!     /*   run pfp                                                    */
!-----------------------------------------------------------------------

      bohr2ang = au_length * 1.d+10

      eV2Eh = au_charge/au_energy

      virial_tensor_sum(:,:) = 0.d0

!     //   loop of beads
      do ibead = 1, nbead

         if ( mod( ibead-1, nprocs ) .ne. myrank ) cycle

         x_i(:) = x(:,ibead) * bohr2ang
         y_i(:) = y(:,ibead) * bohr2ang
         z_i(:) = z(:,ibead) * bohr2ang

         box_i(:,:) = box(:,:) * bohr2ang

         virial_tensor_i(:,:) = 0.d0

         call pfp_interface ( &
     &      obj, x_i, y_i, z_i, box_i, fx_i, fy_i, fz_i, energy, &
     &      virial_tensor_i, atomic_num, mspec, atom_type, natom )

         pot(ibead) = energy * eV2Eh

         fx(:,ibead) = fx_i(:) * eV2Eh * bohr2ang
         fy(:,ibead) = fy_i(:) * eV2Eh * bohr2ang
         fz(:,ibead) = fz_i(:) * eV2Eh * bohr2ang

         virial_tensor_sum(:,:) = virial_tensor_sum(:,:) &
     &                          + virial_tensor_i(:,:) * eV2Eh

      end do

      vir(1:3,1:3) = vir(1:3,1:3) + virial_tensor_sum(1:3,1:3)

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz, natom, nbead )

!     /*   virial   */
      call my_mpi_allreduce_real_2 ( vir, 3, 3 )

      return
      end

#else

!-----------------------------------------------------------------------
!     /*   dummy subroutines                                          */
!-----------------------------------------------------------------------

!***********************************************************************
      subroutine force_pfp_MPI
!***********************************************************************
      use common_variables, only: myrank
      implicit none

      write( 6, '(a)' ) &
     &   'Error - This binary does not support Matlantis PFP.'

      if ( myrank .eq. 0 ) then
         call error_handling_MPI ( 1, 'subroutine force_pfp_MPI', 24 )
      end if

      return
      end subroutine force_pfp_MPI

!***********************************************************************
      subroutine initialize_pfp_MPI()
!***********************************************************************
      use common_variables, only: myrank
      implicit none

      if ( myrank .eq. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - This binary does not support Matlantis PFP.'
      end if

      call error_handling_MPI ( 1, 'subroutine initialize_pfp_MPI', 29 )

      return
      end subroutine initialize_pfp_MPI

!***********************************************************************
      subroutine run_pfp_MPI()
!***********************************************************************
      use common_variables, only: myrank
      implicit none

      if ( myrank .eq. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - This binary does not support Matlantis PFP.'
      end if
      call error_handling_MPI ( 1, 'subroutine run_pfp_MPI', 22 )

      return
      end subroutine run_pfp_MPI

!***********************************************************************
      subroutine titles_pfp()
!***********************************************************************
      implicit none

      write( 6, '(a)' ) &
     &   'Error - This binary does not support Matlantis PFP.'

      return
      end subroutine titles_pfp

!***********************************************************************
      subroutine finalize_pfp_MPI()
!***********************************************************************
      use common_variables, only: myrank
      implicit none

      if ( myrank .eq. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - This binary does not support Matlantis PFP.'
         call error_handling_MPI( 1, 'subroutine finalize_pfp_MPI', 27 )
      end if

      return
      end subroutine finalize_pfp_MPI

#endif
 



