!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    May 21, 2020 by M. Shiga
!      Description:     energy and force from pair potential
!
!///////////////////////////////////////////////////////////////////////
!**********************************************************************
      subroutine force_pair_MPI
!**********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, vir, pot, au_length, au_charge, volume, &
     &   box, au_energy, natom, nbead, iounit, iboundary, &
     &   myrank_main, nprocs_main, myrank_sub, nprocs_sub

      use mm_variables, only : &
     &   rcut_eam, bigbox, bigboxinv, rcut_eam2, &
     &   neam, nbox_eam, ikind_eam, n_list, j_list

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: m, i, j, k, l, n, jx, jy, jz, j2

      real(8) :: xij, yij, zij, rij, rinv, phir_eam, ax, ay, az, &
     &           phir_grad_eam, bx, by, bz, cx, cy, cz, dphirdr, &
     &           absa, absb, absc, aij, bij, cij, rij2, fxi, fyi, fzi

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   for initial access   */
      if ( iset .eq. 0 ) then

!        /*   read eam files   */
         call force_pair_setup_MPI

!        /*   set complete   */
         iset = 1

!     /*   for initial access   */
      end if

!     /*   return if no eam   */
      if ( neam .eq. 0 ) return

!     /*   cut off distance squared   */
      rcut_eam2 = rcut_eam*rcut_eam

!-----------------------------------------------------------------------
!     /*   extention of simulation box in real space sum              */
!-----------------------------------------------------------------------

!     /*   free boundary   */
      if ( iboundary .eq. 0 ) then

!        /*   number of replicated boxes   */
         nbox_eam(1) = 1
         nbox_eam(2) = 1
         nbox_eam(3) = 1

!     /*   periodic boundary   */
      else

!        /*   vector product of lattice vectors b, c   */
         ax = box(2,2)*box(3,3) - box(2,3)*box(3,2)
         ay = box(3,2)*box(1,3) - box(3,3)*box(1,2)
         az = box(1,2)*box(2,3) - box(1,3)*box(2,2)

!        /*   vector product of lattice vectors c, a   */
         bx = box(2,3)*box(3,1) - box(2,1)*box(3,3)
         by = box(3,3)*box(1,1) - box(3,1)*box(1,3)
         bz = box(1,3)*box(2,1) - box(1,1)*box(2,3)

!        /*   vector product of lattice vectors a, b   */
         cx = box(2,1)*box(3,2) - box(2,2)*box(3,1)
         cy = box(3,1)*box(1,2) - box(3,2)*box(1,1)
         cz = box(1,1)*box(2,2) - box(1,2)*box(2,1)

!        /*   distance between parallel planes   */
         absa = volume / sqrt( ax*ax + ay*ay + az*az )
         absb = volume / sqrt( bx*bx + by*by + bz*bz )
         absc = volume / sqrt( cx*cx + cy*cy + cz*cz )

!        /*   number of replicated boxes   */
         nbox_eam(1) = int(2.d0*rcut_eam/absa) + 1
         nbox_eam(2) = int(2.d0*rcut_eam/absb) + 1
         nbox_eam(3) = int(2.d0*rcut_eam/absc) + 1

!     /*   boundary condition   */
      end if

!-----------------------------------------------------------------------
!     /*   first loop                                                 */
!-----------------------------------------------------------------------

!     /*   free boundary or minimum image convention   */
      if ( nbox_eam(1)*nbox_eam(2)*nbox_eam(3) .eq. 1 ) then

!        /*   make neighbor list   */
         call force_eam_makelist_MPI

!        /*   loop of beads   */
         do m = 1, nbead

!           /*   bead parallel   */
            if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

!           /*   loop of atom pairs   */
            do i = 1, natom

!           /*   force parallel   */
            if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!           /*   loop of atom pairs   */
            do n = 1, n_list(i,m)

!              /*   atom in neighbor list   */
               j = j_list(n,i,m)

!              /*   interatomic distance   */
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              /*   apply free or periodic boundary   */
               call pbc_atom_MPI ( xij, yij, zij )

!              /*   interatomic distance squared   */
               rij2 = xij*xij + yij*yij + zij*zij

!              /*   neglect beyond cutoff distance   */
               if ( rij2 .gt. rcut_eam2 ) cycle

!              /*   interatomic distance   */
               rij = sqrt( rij2 )

!              /*   inverse of interatomic distance   */
               rinv = 1.d0/rij

!              /*   species number   */
               k = ikind_eam(i)
               l = ikind_eam(j)

!              /*   gradient of pair potential   */
               dphirdr  =  phir_grad_eam(rij,k,l)

!-----------------------------------------------------------------------
!              /*   pair potential   */
!-----------------------------------------------------------------------

               pot(m)   =  pot(m) + phir_eam(rij,k,l)

!-----------------------------------------------------------------------
!              /*   forces   */
!-----------------------------------------------------------------------

               fxi = - dphirdr * xij * rinv
               fyi = - dphirdr * yij * rinv
               fzi = - dphirdr * zij * rinv

!-----------------------------------------------------------------------
!              /*   total force and virial   */
!-----------------------------------------------------------------------

               fx(i,m) = fx(i,m) + fxi
               fy(i,m) = fy(i,m) + fyi
               fz(i,m) = fz(i,m) + fzi

               fx(j,m) = fx(j,m) - fxi
               fy(j,m) = fy(j,m) - fyi
               fz(j,m) = fz(j,m) - fzi

               vir(1,1) = vir(1,1) + fxi*xij
               vir(1,2) = vir(1,2) + fxi*yij
               vir(1,3) = vir(1,3) + fxi*zij
               vir(2,1) = vir(2,1) + fyi*xij
               vir(2,2) = vir(2,2) + fyi*yij
               vir(2,3) = vir(2,3) + fyi*zij
               vir(3,1) = vir(3,1) + fzi*xij
               vir(3,2) = vir(3,2) + fzi*yij
               vir(3,3) = vir(3,3) + fzi*zij

!           /*   loop of atom pairs   */
            end do

!           /*   loop of atom pairs   */
            end do

!        /*   loop of beads   */
         end do

!-----------------------------------------------------------------------
!     /*   first loop                                                 */
!-----------------------------------------------------------------------

!     /*   periodic boundary   */
      else

!        /*   replicated boxes   */
         bigbox(:,1) = dble(nbox_eam(1))*box(:,1)
         bigbox(:,2) = dble(nbox_eam(2))*box(:,2)
         bigbox(:,3) = dble(nbox_eam(3))*box(:,3)

!        /*   inverse of box matrix   */
         call inv3 ( bigbox, bigboxinv )

!        /*   loop of beads   */
         do m = 1, nbead

!           /*   bead parallel   */
            if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

!           /*   loop of atom pairs   */
            do i = 1, natom

!           /*   force parallel   */
            if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!           /*   loop of atom pairs   */
            do j = 1, natom

!           /*   loop of replicated boxes   */
            do jx = 0, nbox_eam(1)-1
            do jy = 0, nbox_eam(2)-1
            do jz = 0, nbox_eam(3)-1

!              /*   square of box index   */
               j2 = jx*jx + jy*jy + jz*jz

!              /*   skip same atom   */
               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

!              /*   interatomic distance of i and j in same box   */
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              /*   distance of i and j in different box  */
               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

!              /*   vector in big box   */
               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

!              /*   apply periodic boundary in big box   */
               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

!              /*   distance of nearest i and j   */
               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

!              /*   interatomic distance squared   */
               rij2 = xij*xij + yij*yij + zij*zij

!              /*   interatomic distance   */
               if ( rij2 .gt. rcut_eam2 ) cycle

!              /*   interatomic distance   */
               rij = sqrt( rij2 )

!              /*   interatomic distance   */
               rinv = 1.d0/rij

!              /*   species number   */
               k = ikind_eam(i)
               l = ikind_eam(j)

!              /*   gradient of pair potential   */
               dphirdr  =  phir_grad_eam(rij,k,l)

!-----------------------------------------------------------------------
!              /*   pair potential   */
!-----------------------------------------------------------------------

               pot(m)   =  pot(m) + 0.5d0*phir_eam(rij,k,l)

!-----------------------------------------------------------------------
!              /*   forces   */
!-----------------------------------------------------------------------

               fxi = - dphirdr * xij * rinv
               fyi = - dphirdr * yij * rinv
               fzi = - dphirdr * zij * rinv

!-----------------------------------------------------------------------
!              /*   total force and virial   */
!-----------------------------------------------------------------------

               fx(i,m) = fx(i,m) + fxi
               fy(i,m) = fy(i,m) + fyi
               fz(i,m) = fz(i,m) + fzi

               vir(1,1) = vir(1,1) + 0.5d0*fxi*xij
               vir(1,2) = vir(1,2) + 0.5d0*fxi*yij
               vir(1,3) = vir(1,3) + 0.5d0*fxi*zij
               vir(2,1) = vir(2,1) + 0.5d0*fyi*xij
               vir(2,2) = vir(2,2) + 0.5d0*fyi*yij
               vir(2,3) = vir(2,3) + 0.5d0*fyi*zij
               vir(3,1) = vir(3,1) + 0.5d0*fzi*xij
               vir(3,2) = vir(3,2) + 0.5d0*fzi*yij
               vir(3,3) = vir(3,3) + 0.5d0*fzi*zij

!           /*   loop of replicated boxes   */
            end do
            end do
            end do

!           /*   loop of atom pairs   */
            end do

!           /*   loop of atom pairs   */
            end do

!        /*   loop of beads   */
         end do

      end if

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz, natom, nbead )

!     /*   virial   */
      call my_mpi_allreduce_real_2 ( vir, 3, 3 )

      return
      end





!***********************************************************************
      subroutine force_pair_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   au_length, au_charge, au_energy, ikind, nkind, natom, iounit, &
     &   myrank

      use mm_variables, only : &
     &   xref_eam, yref_eam, y2ref_eam,rcut_eam, &
     &   neam, nref_eam, ikind_eam, iphir_eam

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, ierr, nphir_eam

!-----------------------------------------------------------------------
!     /*   number of tables                                           */
!-----------------------------------------------------------------------

!     /*   eam   */
      nphir_eam = nkind*(nkind+1)/2

!     /*   neam = total number of eam reference tables   */
      neam = nphir_eam

!-----------------------------------------------------------------------
!     /*   read eam data                                              */
!-----------------------------------------------------------------------

!     /*   master rank   */
      if ( myrank .eq. 0 ) then

!        /*   open file   */
         open (iounit,file = 'eam.dat')

!        /*   tag   */
         call search_tag ( '<nref_eam>', 5, iounit, ierr )

!        /*   number of reference data points in the table   */
         if ( neam .ne. 0 ) read( iounit, *, iostat=ierr ) nref_eam

!        /*   tag   */
         call search_tag ( '<rcut_eam>', 10, iounit, ierr )

!        /*   cut off distance   */
         if ( neam .ne. 0 ) read( iounit, *, iostat=ierr ) rcut_eam

!        /*   angstrom --> bohr   */
         rcut_eam = rcut_eam / au_length * 1.d-10

!        /*   close file   */
         close(iounit)

!        /*   on error, no eam   */
         if ( ierr .ne. 0 ) neam = 0

!     /*   master rank   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   error handling   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_adp_setup_MPI', 30 )

!     /*   communicate   */
      call my_mpi_bcast_int_0( neam )
      call my_mpi_bcast_int_0( nref_eam )
      call my_mpi_bcast_real_0( rcut_eam )

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   species number   */
      if ( .not. allocated( ikind_eam ) ) &
     &   allocate( ikind_eam(natom) )

!     /*   table number of rho-r, f-rho, phi-r   */
      if ( .not. allocated( iphir_eam ) ) &
     &   allocate( iphir_eam(nkind,nkind) )

!     /*   table rho-r, f-rho, phi-r   */
      if ( .not. allocated( xref_eam ) ) &
     &   allocate( xref_eam(nref_eam,neam) )
      if ( .not. allocated( yref_eam ) ) &
     &   allocate( yref_eam(nref_eam,neam) )

!     /*   spline dimensions for rho-r, f-rho, phi-r   */
      if ( .not. allocated( y2ref_eam ) ) &
     &   allocate( y2ref_eam(nref_eam,neam) )

!-----------------------------------------------------------------------
!     /*   make species number                                        */
!-----------------------------------------------------------------------

      ikind_eam(:) = ikind(:)

!-----------------------------------------------------------------------
!     /*   read eam table:  r[angstrom] - phi[electron volt]          */
!-----------------------------------------------------------------------

!     /*   master rank   */
      if ( myrank .eq. 0 ) then

!        /*   open file   */
         open (iounit,file = 'eam.dat')

!        /*   tag   */
         call search_tag ( '<phir_eam>', 10, iounit, ierr )

!        /*   loop of table   */
         do i = 1, nphir_eam

!           /*   read species for the table   */
            read( iounit, *, iostat=ierr ) k, l

!           /*   read table for species k-l   */
            do j = 1, nref_eam
               read( iounit, *, iostat=ierr ) &
     &            xref_eam(j,i), yref_eam(j,i)
            end do

!           /*   check error   */
            if ( ierr .ne. 0 ) exit

!           /*   angstrom --> bohr,  electron volt --> hartree   */
            do j = 1, nref_eam
               xref_eam(j,i) = xref_eam(j,i) / au_length * 1.d-10
               yref_eam(j,i) = yref_eam(j,i) * au_charge / au_energy
            end do

!           /*   i-th table is for species pair k-l   */
            iphir_eam(k,l) = i
            iphir_eam(l,k) = i

!        /*   loop of table   */
         end do

!        /*   close file   */
         close(iounit)

!     /*   master rank   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   error handling   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_eam_setup_MPI', 30 )

!-----------------------------------------------------------------------
!     /*   communication                                              */
!-----------------------------------------------------------------------

      call my_mpi_bcast_real_2( xref_eam, nref_eam, neam )
      call my_mpi_bcast_real_2( yref_eam, nref_eam, neam )
      call my_mpi_bcast_real_2( y2ref_eam, nref_eam, neam )

      call my_mpi_bcast_int_2( iphir_eam, nkind, nkind )

!-----------------------------------------------------------------------
!     /*   initialization of spline                                   */
!-----------------------------------------------------------------------

      do i = 1, neam
         call spline_init_eam &
     &      ( xref_eam(:,i), yref_eam(:,i), y2ref_eam(:,i), nref_eam )
      end do

!-----------------------------------------------------------------------

      return
      end
