!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, B. Thomsen
!      Last updated:    May 1, 2020 by M. Shiga
!      Description:     Ojamae-Shavitt-Singer (OSS2) potential
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      module oss_variables
!***********************************************************************

!     //   charges
      real(8), dimension(2)  :: q_oss

!     //   damping parameters for charge/dipole-dipole interactions
      real(8), dimension(2)  :: a_oss
      real(8), dimension(2)  :: b_oss
      real(8), dimension(2)  :: c_oss
      real(8)                :: alpha_oss

!     //   two body terms
      real(8), dimension(5)  :: h_oss
      real(8), dimension(7)  :: o_oss

!     //   three body terms
      real(8)                :: r_oss
      real(8)                :: theta_oss
      real(8), dimension(16) :: k_oss
      real(8), dimension(3)  :: m_oss

!     //   cross term
      real(8), dimension(12) :: p_oss
      real(8)                :: mu_oss
      real(8)                :: d_oss

!     //   induced dipole term
      real(8), dimension(:,:), allocatable :: amat_oss
      real(8), dimension(:),   allocatable :: pvec_oss
      real(8), dimension(:),   allocatable :: bvec_oss
      real(8), dimension(:,:), allocatable :: px_oss
      real(8), dimension(:,:), allocatable :: py_oss
      real(8), dimension(:,:), allocatable :: pz_oss
      integer                              :: noxygen_oss
      integer, dimension(:),   allocatable :: ioxygen_oss

!     //   cut off parameters
      real(8) :: rin_oo_oss
      real(8) :: rin_oh_oss
      real(8) :: rout_oo_oss
      real(8) :: rout_oh_oss
      real(8) :: rcut_3b_oss

!     //   augment parameters
      integer :: ioption_oss
      real(8) :: raug_oo_oss
      real(8) :: aug1_oss
      real(8) :: aug2_oss

!     //   ewald parameters
      real(8) :: eps_ewald_oss
      real(8) :: ratio_ewald_oss
      real(8) :: eps_ewpol_oss
      real(8) :: ratio_ewpol_oss

!***********************************************************************
      end module oss_variables
!***********************************************************************





!***********************************************************************
      subroutine force_oss_setup
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   au_length, pi, iounit, natom, species

      use oss_variables, only : &
     &   q_oss, a_oss, b_oss, c_oss, h_oss, o_oss, k_oss, m_oss, &
     &   r_oss, theta_oss, alpha_oss, p_oss, mu_oss, d_oss

      use oss_variables, only : &
     &   rin_oo_oss, rin_oh_oss, rout_oo_oss, rout_oh_oss, raug_oo_oss, &
     &   rcut_3b_oss, aug1_oss, aug2_oss, eps_ewald_oss, &
     &   ratio_ewald_oss,  eps_ewpol_oss, ratio_ewpol_oss, ioption_oss

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8), parameter :: bohr2ang = au_length * 1.d+10

      integer :: i, ierr

!-----------------------------------------------------------------------
!     /*   oss parameters                                            */
!-----------------------------------------------------------------------

      q_oss(1)     =  -2.00000000d0
      q_oss(2)     =   1.00000000d0

      r_oss     =   0.96140000d0 / bohr2ang

      theta_oss =   1.81733805d0

      a_oss(1)  =   1.73089134d0 / bohr2ang**2
      a_oss(2)  =   0.29575999d0 * bohr2ang

      b_oss(1)  = 332.28529224d0 / bohr2ang**2
      b_oss(2)  =   2.84683483d0 * bohr2ang

      c_oss(1)  =   1.09568640d0 / bohr2ang**2
      c_oss(2)  =   0.00000204d0 * bohr2ang

      h_oss(1)  =   0.00575558d0
      h_oss(2)  =   2.64253740d0 / bohr2ang
      h_oss(3)  =   1.12743850d0 * bohr2ang
      h_oss(4)  =   3.32556550d0 * bohr2ang
      h_oss(5)  =   0.07847734d0

      o_oss(1)  =  40.48587340d0
      o_oss(2)  =   1.32909550d0 * bohr2ang
      o_oss(3)  = -41.72606080d0
      o_oss(4)  =   1.35094910d0 * bohr2ang
      o_oss(5)  =   0.06293960d0
      o_oss(6)  =   6.77909903d0 * bohr2ang**2
      o_oss(7)  =   1.81178360d0 / bohr2ang

      k_oss(1)  =  -0.04200730d0
      k_oss(2)  =   0.16488265d0 * bohr2ang
      k_oss(3)  =  -0.02509795d0
      k_oss(4)  =  -0.37814525d0 * bohr2ang**2
      k_oss(5)  =  -0.31667595d0 * bohr2ang**2
      k_oss(6)  =  -0.01146720d0
      k_oss(7)  =   0.06061075d0 * bohr2ang
      k_oss(8)  =   0.52828100d0 * bohr2ang**3
      k_oss(9)  =   1.16176270d0 * bohr2ang**3
      k_oss(10) =   0.07652320d0
      k_oss(11) =  -0.21208835d0 * bohr2ang**2
      k_oss(12) =  -0.10253850d0 * bohr2ang**2
      k_oss(13) =  -0.07622160d0 * bohr2ang
      k_oss(14) =  -0.22869400d0 * bohr2ang**4
      k_oss(15) =   0.00000000d0 * bohr2ang**4
      k_oss(16) =  -0.02909200d0

      m_oss(1)  =   6.25000000d0 * bohr2ang**2
      m_oss(2)  =   0.00377233d0
      m_oss(3)  =   6.25000000d0 * bohr2ang**2

      p_oss(1)  =   0.00000000d0
      p_oss(2)  =   0.00000000d0
      p_oss(3)  =   0.00000000d0 * bohr2ang**2
      p_oss(4)  =   0.00000000d0
      p_oss(5)  =   0.00000000d0 * bohr2ang
      p_oss(6)  =   0.00000000d0 * bohr2ang**3
      p_oss(7)  =   0.00000000d0 * bohr2ang**2
      p_oss(8)  =   0.00000000d0 * bohr2ang**2
      p_oss(9)  =   0.00000000d0 * bohr2ang
      p_oss(10) =   0.00000000d0
      p_oss(11) =   0.00000000d0 * bohr2ang**4
      p_oss(12) =   0.00000000d0

      mu_oss    =   0.00000000d0 / bohr2ang

      d_oss     =   0.00000000d0 * bohr2ang**2

      alpha_oss =   1.44400000d0 / bohr2ang**3

!-----------------------------------------------------------------------
!     /*   cut off parameters                                         */
!-----------------------------------------------------------------------

      open ( iounit, file = 'oss.dat' )

      call search_tag ( '<oh_cutoff_oss>', 15, iounit, ierr )

      read( iounit, *, iostat=ierr ) rin_oh_oss, rout_oh_oss

      if ( ierr .ne. 0 ) then

         rin_oh_oss = 50.d0
         rout_oh_oss = rin_oh_oss + 3.d0

      end if

      close( iounit )

      open ( iounit, file = 'oss.dat' )

      call search_tag ( '<oo_cutoff_oss>', 15, iounit, ierr )

      read( iounit, *, iostat=ierr ) rin_oo_oss, rout_oo_oss

      if ( ierr .ne. 0 ) then

         rin_oo_oss = 50.d0
         rout_oo_oss = rin_oo_oss + 3.d0

      end if

      close( iounit )

      open ( iounit, file = 'oss.dat' )

      call search_tag ( '<3b_cutoff_oss>', 15, iounit, ierr )

      read( iounit, *, iostat=ierr ) rcut_3b_oss

      if ( ierr .ne. 0 ) then

         rcut_3b_oss = 5.4d0

      end if

      close( iounit )

!-----------------------------------------------------------------------
!     /*   augment parameters                                         */
!-----------------------------------------------------------------------

      open ( iounit, file = 'oss.dat' )

      call search_tag ( '<ioption_oss>', 13, iounit, ierr )

      read( iounit, *, iostat=ierr ) ioption_oss

      if ( ierr .ne. 0 ) ioption_oss = 1

      close( iounit )

      raug_oo_oss = 2.d0 / bohr2ang

      aug1_oss = 0.4169516202506216d+01
      aug2_oss = 0.1040919676844401d-01

!-----------------------------------------------------------------------
!     /*   ewald parameters                                           */
!-----------------------------------------------------------------------

      open ( iounit, file = 'oss.dat' )

      call search_tag ( '<ewald_oss>', 11, iounit, ierr )

      read( iounit, *, iostat=ierr ) eps_ewald_oss, ratio_ewald_oss

      if ( ierr .ne. 0 ) then
         eps_ewald_oss   = 1.d-8
         ratio_ewald_oss = 4.d0
      end if

      close( iounit )

      open ( iounit, file = 'oss.dat' )

      call search_tag ( '<ewpol_oss>', 11, iounit, ierr )

      read( iounit, *, iostat=ierr ) eps_ewpol_oss, ratio_ewpol_oss

      if ( ierr .ne. 0 ) then
         eps_ewpol_oss   = 1.d-10
         ratio_ewpol_oss = 0.1d0
      end if

      close( iounit )

!-----------------------------------------------------------------------
!     /*   charges                                                    */
!-----------------------------------------------------------------------

      open ( iounit, file = 'mm.dat' )

      write( iounit, '(a)' ) '<charges>'
      write( iounit, '(i8)' ) natom

      do i = 1, natom

         if      ( species(i)(1:3) .eq. 'O  ' ) then
            write( iounit, '(i8,e10.1,e24.16,i2)' ) &
     &         i, -2.d0, alpha_oss, 1
         else if ( species(i)(1:3) .eq. 'H  ' ) then
            write( iounit, '(i8,e10.1,e24.16,i2)' ) &
     &         i, +1.d0,      0.d0, 2
         else if ( species(i)(1:3) .eq. 'D  ' ) then
            write( iounit, '(i8,e10.1,e24.16,i2)' ) &
     &         i, +1.d0,      0.d0, 2
         else if ( species(i)(1:3) .eq. 'T  ' ) then
            write( iounit, '(i8,e10.1,e24.16,i2)' ) &
     &         i, +1.d0,      0.d0, 2
         else if ( species(i)(1:3) .eq. 'Mu ' ) then
            write( iounit, '(i8,e10.1,e24.16,i2)' ) &
     &         i, +1.d0,      0.d0, 2
         else
            write( iounit, '(i8,e10.1,e24.16,i2)' ) &
     &         i,  0.d0,      0.d0, 3
         end if

      end do

      write( iounit, '(a)' ) 

      write( iounit, '(a)' ) '<damping>'
      write( iounit, '(a)' ) '3'
      write( iounit, '(a,2e24.16)' ) &
     &     'CD  2  1  OSS ', a_oss(1), a_oss(2)
      write( iounit, '(a,2e24.16)' ) &
     &     'CD  1  1  OSS ', b_oss(1), b_oss(2)
      write( iounit, '(a,2e24.16)' ) &
     &     'DD  1  1  OSS ', c_oss(1), c_oss(2)

      write( iounit, '(a)' ) 

      write( iounit, '(a)' ) '<ewald>'
      write( iounit, '(2d24.16)' ) eps_ewald_oss, ratio_ewald_oss

      write( iounit, '(a)' ) 

      write( iounit, '(a)' ) '<ewpol>'
      write( iounit, '(2d24.16)' ) eps_ewpol_oss, ratio_ewpol_oss

      close( iounit )

      return
      end





!***********************************************************************
      subroutine force_oss
!***********************************************************************

!-----------------------------------------------------------------------
!     //   variables
!-----------------------------------------------------------------------

      implicit none

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     //   set oss parameters
!-----------------------------------------------------------------------

!     //   initial settings
      if ( iset .eq. 0 ) then

!        //   setup oss parameters
         call force_oss_setup

!        //   flag
         iset = 1

!     //   initial settings
      end if

!-----------------------------------------------------------------------
!     //   two body interactions
!-----------------------------------------------------------------------

      call force_oss_twobody

!-----------------------------------------------------------------------
!     //   three body interactions
!-----------------------------------------------------------------------

      call force_oss_threebody

!-----------------------------------------------------------------------
!     //   electrostatic interactions: polarizable mm
!-----------------------------------------------------------------------

      call force_pol

      return
      end





!***********************************************************************
      subroutine force_oss_twobody
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pot, fx, fy, fz, vir, au_length, natom, nbead, species

      use oss_variables, only : &
     &   h_oss, o_oss

      use oss_variables, only : &
     &   rin_oo_oss, rin_oh_oss, rout_oo_oss, rout_oh_oss, raug_oo_oss, &
     &   aug1_oss, aug2_oss, ioption_oss

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, m

      real(8) :: rx, ry, rz, r1, rinv, r2, fxi, fyi, fzi, r2in_oh_oss, &
     &           p1, p2, p3, p4, p5, p6, p7, p8, p9, p0, r2in_oo_oss, &
     &           r2out_oo_oss, r2out_oh_oss, r2aug_oo_oss, r2inv, &
     &           swf, r3inv, r5inv, dswf

!-----------------------------------------------------------------------
!     //   two body interactions
!-----------------------------------------------------------------------

      r2in_oo_oss  = rin_oo_oss  * rin_oo_oss
      r2in_oh_oss  = rin_oh_oss  * rin_oh_oss
      r2out_oo_oss = rout_oo_oss * rout_oo_oss
      r2out_oh_oss = rout_oh_oss * rout_oh_oss
      r2aug_oo_oss = raug_oo_oss * raug_oo_oss

      do m = 1, nbead

!-----------------------------------------------------------------------
!        //   O-O interactions
!-----------------------------------------------------------------------

         do i = 1, natom

            if ( species(i)(1:3) .ne. 'O  ' ) cycle

            do j = i+1, natom

               if ( species(j)(1:3) .ne. 'O  ' ) cycle

               rx = x(j,m) - x(i,m)
               ry = y(j,m) - y(i,m)
               rz = z(j,m) - z(i,m)

               call pbc_atom( rx, ry, rz )

               r2 = rx*rx + ry*ry + rz*rz

               if ( r2 .gt. r2out_oo_oss ) cycle

               if ( ( ioption_oss .eq. 0 ) .or. &
     &              ( r2 .gt. r2aug_oo_oss ) ) then

                  r1 = sqrt( r2 )
                  rinv = 1.d0 / r1

                  p1 = o_oss(1) * exp( - o_oss(2) * r1 )
                  p2 = o_oss(3) * exp( - o_oss(4) * r1 )
                  p3 = ( r1 - o_oss(7) ) * ( r1 - o_oss(7) )
                  p4 = o_oss(5) * exp( - o_oss(6) * p3 )
                  p5 = ( r1 - o_oss(7) ) * p4
                  p6 = p1 + p2 + p4

                  call getswf( r1, rin_oo_oss, rout_oo_oss, swf, dswf )

                  pot(m) = pot(m) + p6 * swf

                  p7 = - o_oss(2) * p1 &
     &                 - o_oss(4) * p2 &
     &                 - 2.d0 * o_oss(6) * p5

                  p8 = ( p7 * swf + p6 * dswf ) * rinv

                  fxi = p8 * rx
                  fyi = p8 * ry
                  fzi = p8 * rz

               else

                  r1 = sqrt( r2 )
                  rinv = 1.d0 / r1
                  r2inv = rinv * rinv
                  r3inv = rinv * r2inv
                  r5inv = r2inv * r3inv

                  p1 = aug1_oss*r3inv + aug2_oss

                  pot(m) = pot(m) + p1

                  p2 = - 3.d0 * aug1_oss * r5inv

                  fxi = p2 * rx
                  fyi = p2 * ry
                  fzi = p2 * rz

               end if

               fx(i,m) = fx(i,m) + fxi
               fy(i,m) = fy(i,m) + fyi
               fz(i,m) = fz(i,m) + fzi

               fx(j,m) = fx(j,m) - fxi
               fy(j,m) = fy(j,m) - fyi
               fz(j,m) = fz(j,m) - fzi

               vir(1,1) = vir(1,1) - fxi * rx
               vir(1,2) = vir(1,2) - fxi * ry
               vir(1,3) = vir(1,3) - fxi * rz
               vir(2,1) = vir(2,1) - fyi * rx
               vir(2,2) = vir(2,2) - fyi * ry
               vir(2,3) = vir(2,3) - fyi * rz
               vir(3,1) = vir(3,1) - fzi * rx
               vir(3,2) = vir(3,2) - fzi * ry
               vir(3,3) = vir(3,3) - fzi * rz

            end do

!-----------------------------------------------------------------------
!           //   O-H interactions
!-----------------------------------------------------------------------

            do j = 1, natom

               if ( ( species(j)(1:3) .ne. 'H  ' ) .and. &
     &              ( species(j)(1:3) .ne. 'D  ' ) .and. &
     &              ( species(j)(1:3) .ne. 'T  ' ) .and. &
     &              ( species(j)(1:3) .ne. 'Mu ' ) ) cycle

               rx = x(j,m) - x(i,m)
               ry = y(j,m) - y(i,m)
               rz = z(j,m) - z(i,m)

               call pbc_atom( rx, ry, rz )

               r2 = rx*rx + ry*ry + rz*rz

               if ( r2 .gt. r2out_oh_oss ) cycle

               r1 = sqrt( r2 )
               rinv = 1.d0 / r1

               p1 = ( 1.d0 - h_oss(5) ) * ( 1.d0 - h_oss(5) )
               p2 = exp( - h_oss(3) * ( r1 - h_oss(2) ) )
               p3 = h_oss(5) * h_oss(5)
               p4 = 1.d0 / ( p1 + p3 )
               p5 = exp( - h_oss(4) * ( r1 - h_oss(2) ) )
               p6 = 1.d0 - p1 * p2 * p4 - p3 * p5 * p4
               p7 = 2.d0 * h_oss(1) * p6 * h_oss(3) * p1 * p2 * p4
               p8 = 2.d0 * h_oss(1) * p6 * h_oss(4) * p3 * p5 * p4
               p9 = h_oss(1) * ( p6 * p6 - 1.d0 )

               call getswf( r1, rin_oh_oss, rout_oh_oss, swf, dswf )

               pot(m) = pot(m) + p9 * swf

               p0 = ( (p7+p8)*swf + p9*dswf ) * rinv

               fxi = p0 * rx
               fyi = p0 * ry
               fzi = p0 * rz

               fx(i,m) = fx(i,m) + fxi
               fy(i,m) = fy(i,m) + fyi
               fz(i,m) = fz(i,m) + fzi

               fx(j,m) = fx(j,m) - fxi
               fy(j,m) = fy(j,m) - fyi
               fz(j,m) = fz(j,m) - fzi

               vir(1,1) = vir(1,1) - fxi * rx
               vir(1,2) = vir(1,2) - fxi * ry
               vir(1,3) = vir(1,3) - fxi * rz
               vir(2,1) = vir(2,1) - fyi * rx
               vir(2,2) = vir(2,2) - fyi * ry
               vir(2,3) = vir(2,3) - fyi * rz
               vir(3,1) = vir(3,1) - fzi * rx
               vir(3,2) = vir(3,2) - fzi * ry
               vir(3,3) = vir(3,3) - fzi * rz

            end do

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_oss_threebody
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pot, fx, fy, fz, vir, pi, natom, nbead, species

      use oss_variables, only : &
     &   k_oss, m_oss, r_oss, theta_oss

      use oss_variables, only : &
     &   rcut_3b_oss

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, m, il, jl, kl

      real(8) :: xij, yij, zij, xkj, ykj, zkj, rij, rkj, dij, dkj, &
     &           rij2, rkj2, pijk, qijk, theta, dtheta, dtheta2, dens, &
     &           dtheta3, dtheta4, dij2, dij3, dij4, dkj2, dkj3, dkj4, &
     &           fxi, fyi, fzi, fxj, fyj, fzj, fxk, fyk, fzk, rijk, &
     &           q1, q2, dtdxi, dtdyi, dtdzi, dtdxk, dtdyk, dtdzk, &
     &           rij2_inv, rkj2_inv, rij_inv, rkj_inv, dijdxi, dijdyi, &
     &           dijdzi, dkjdxk, dkjdyk, dkjdzk, r2, r2cut_3b_oss, &
     &           r3cut_3b_oss, p1, p2, p3, p4, p5, p6, p7, p8, p9, &
     &           p10, p11, p12, p13, p14, p15, p16, p17, p18, p19, p20

!     //   maximum number of neighbors
      integer :: nnmax

!     //   hydrogen list
      integer, dimension(:,:), allocatable :: i_list
      integer, dimension(:), allocatable :: ni_list

!     //   number of oxygen atoms
      integer, save :: nj_list = 0

!     //   number of oxygen atoms
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     //   cut off distance
!-----------------------------------------------------------------------

      r2cut_3b_oss = rcut_3b_oss * rcut_3b_oss
      r3cut_3b_oss = r2cut_3b_oss * rcut_3b_oss

      dens = 0.05d0
      nnmax = int( dens * 4.d0/3.d0*pi*r3cut_3b_oss )

!-----------------------------------------------------------------------
!     //   number of oxygen
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         jl = 0

         do j = 1, natom

            if ( species(j)(1:3) .ne. 'O  ' ) cycle

            jl = jl + 1

         end do

         nj_list = jl

      end if

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

      allocate( i_list(nnmax,nj_list) )
      allocate( ni_list(nj_list) )

!-----------------------------------------------------------------------
!     //   three body interactions
!-----------------------------------------------------------------------

      do m = 1, nbead

         jl = 0

         do j = 1, natom

            if ( species(j)(1:3) .ne. 'O  ' ) cycle

            jl = jl + 1

            il = 0

!-----------------------------------------------------------------------
!           //   make hydrogen list
!-----------------------------------------------------------------------

            do i = 1, natom

               if ( ( species(i)(1:3) .ne. 'H  ' ) .and. &
     &              ( species(i)(1:3) .ne. 'D  ' ) .and. &
     &              ( species(i)(1:3) .ne. 'T  ' ) .and. &
     &              ( species(i)(1:3) .ne. 'Mu ' ) ) cycle

               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

               call pbc_atom ( xij, yij, zij )

               r2 = xij*xij + yij*yij + zij*zij

               if ( r2 .gt. r2cut_3b_oss ) cycle

               il = il + 1

               i_list(il,jl) = i

            end do

            ni_list(jl) = il

!-----------------------------------------------------------------------
!           //   calculate forces
!-----------------------------------------------------------------------

            do il = 1, ni_list(jl)

               i = i_list(il,jl)

               do kl = il+1, ni_list(jl)

                  k = i_list(kl,jl)

                  xij = x(i,m) - x(j,m)
                  yij = y(i,m) - y(j,m)
                  zij = z(i,m) - z(j,m)

                  call pbc_atom ( xij, yij, zij )

                  rij = sqrt( xij*xij + yij*yij + zij*zij )

                  dij = rij - r_oss

                  xkj = x(k,m) - x(j,m)
                  ykj = y(k,m) - y(j,m)
                  zkj = z(k,m) - z(j,m)

                  call pbc_atom ( xkj, ykj, zkj )

                  rkj = sqrt( xkj*xkj + ykj*ykj + zkj*zkj )

                  dkj = rkj - r_oss

                  rij2 = xij*xij + yij*yij + zij*zij
                  rkj2 = xkj*xkj + ykj*ykj + zkj*zkj

                  rijk = sqrt( rij2*rkj2 )

                  pijk = xij*xkj + yij*ykj + zij*zkj

                  qijk  = pijk/rijk

                  qijk = max( qijk, -1.d0 )
                  qijk = min( qijk,  1.d0 )

                  theta = acos( qijk )

                  dtheta = theta - theta_oss

                  dij2 = dij * dij
                  dij3 = dij * dij2
                  dij4 = dij * dij3

                  dkj2 = dkj * dkj
                  dkj3 = dkj * dkj2
                  dkj4 = dkj * dkj3

                  dtheta2 = dtheta * dtheta
                  dtheta3 = dtheta * dtheta2
                  dtheta4 = dtheta * dtheta3

                  rij_inv = 1.d0 / rij
                  rkj_inv = 1.d0 / rkj

                  rij2_inv = rij_inv * rij_inv
                  rkj2_inv = rkj_inv * rkj_inv

                  p1  = k_oss(1) + k_oss(2) * ( dij + dkj )
                  p2  = k_oss(3)  * dtheta
                  p3  = k_oss(4)  * ( dij2 + dkj2 )
                  p4  = k_oss(5)  * dij * dkj
                  p5  = k_oss(6)  * dtheta2
                  p6  = k_oss(7)  * ( dij + dkj ) * dtheta
                  p7  = k_oss(8)  * ( dij3 + dkj3 )
                  p8  = k_oss(9)  * ( dij2*dkj + dij*dkj2 )
                  p9  = k_oss(10) * dtheta3
                  p10 = k_oss(11) * ( dij2 + dkj2 ) * dtheta
                  p11 = k_oss(12) * dij * dkj * dtheta
                  p12 = k_oss(13) * ( dij + dkj ) * dtheta2
                  p13 = k_oss(14) * ( dij4 + dkj4 )
                  p14 = k_oss(15) * dij2 * dkj2
                  p15 = k_oss(16) * dtheta4

                  p16 = m_oss(1) * ( dij2 + dkj2 )
                  p17 = m_oss(2) * dtheta2
                  p18 = m_oss(3) * ( dij2 + dkj2 ) * dtheta2
                  p19 = exp( - ( p16 + p17 + p18) )

                  p20 = +  p1 +  p2 +  p3 +  p4 +  p5 &
     &                  +  p6 +  p7 +  p8 +  p9 + p10 &
     &                  + p11 + p12 + p13 + p14 + p15

                  pot(m) = pot(m) + p20 * p19

                  q1 = - p19 / sin(theta) / rijk

                  dtdxi = q1 * ( xkj - pijk * rij2_inv * xij )
                  dtdyi = q1 * ( ykj - pijk * rij2_inv * yij )
                  dtdzi = q1 * ( zkj - pijk * rij2_inv * zij )

                  dtdxk = q1 * ( xij - pijk * rkj2_inv * xkj )
                  dtdyk = q1 * ( yij - pijk * rkj2_inv * ykj )
                  dtdzk = q1 * ( zij - pijk * rkj2_inv * zkj )

                  q2 = + p19

                  dijdxi = q2 * xij * rij_inv
                  dijdyi = q2 * yij * rij_inv
                  dijdzi = q2 * zij * rij_inv

                  dkjdxk = q2 * xkj * rkj_inv
                  dkjdyk = q2 * ykj * rkj_inv
                  dkjdzk = q2 * zkj * rkj_inv

                  fxi = - k_oss(2) * dijdxi &
     &                  - k_oss(3) * dtdxi &
     &                  - k_oss(4) * 2.d0 * dij * dijdxi &
     &                  - k_oss(5) * dkj * dijdxi &
     &                  - k_oss(6) * 2.d0 * dtheta * dtdxi &
     &                  - k_oss(7) * ( dijdxi*dtheta + (dij+dkj)*dtdxi ) &
     &                  - k_oss(8) * 3.d0 * dij2 * dijdxi &
     &                  - k_oss(9) * ( 2.d0*dij*dkj + dkj2 ) * dijdxi &
     &                  - k_oss(10) * 3.d0 * dtheta2 * dtdxi &
     &                  - k_oss(11) * ( 2.d0*dij*dijdxi*dtheta &
     &                                + (dij2+dkj2)*dtdxi ) &
     &                  - k_oss(12) * dkj *( dijdxi*dtheta + dij*dtdxi ) &
     &                  - k_oss(13) * ( dijdxi*dtheta2 &
     &                                + (dij+dkj)*2.d0*dtheta*dtdxi ) &
     &                  - k_oss(14) * 4.d0 * dij3 * dijdxi &
     &                  - k_oss(15) * 2.d0 * dij * dkj2 * dijdxi &
     &                  - k_oss(16) * 4.d0 * dtheta3 * dtdxi &
     &                  + p20 * m_oss(1) * 2.d0 * dij * dijdxi &
     &                  + p20 * m_oss(2) * 2.d0 * dtheta * dtdxi &
     &                  + p20 * m_oss(3) * 2.d0 * ( dij*dijdxi*dtheta2 &
     &                                      + (dij2+dkj2)*dtheta*dtdxi )

                  fyi = - k_oss(2) * dijdyi &
     &                  - k_oss(3) * dtdyi &
     &                  - k_oss(4) * 2.d0 * dij * dijdyi &
     &                  - k_oss(5) * dkj * dijdyi &
     &                  - k_oss(6)  * 2.d0 * dtheta * dtdyi &
     &                  - k_oss(7) * ( dijdyi*dtheta + (dij+dkj)*dtdyi ) &
     &                  - k_oss(8) * 3.d0 * dij2 * dijdyi &
     &                  - k_oss(9) * ( 2.d0*dij*dkj + dkj2 ) * dijdyi &
     &                  - k_oss(10) * 3.d0 * dtheta2 * dtdyi &
     &                  - k_oss(11) * ( 2.d0*dij*dijdyi*dtheta &
     &                                + (dij2+dkj2)*dtdyi ) &
     &                  - k_oss(12) * dkj *( dijdyi*dtheta + dij*dtdyi ) &
     &                  - k_oss(13) * ( dijdyi*dtheta2 &
     &                                + (dij+dkj)*2.d0*dtheta*dtdyi ) &
     &                  - k_oss(14) * 4.d0 * dij3 * dijdyi &
     &                  - k_oss(15) * 2.d0 * dij * dkj2 * dijdyi &
     &                  - k_oss(16) * 4.d0 * dtheta3 * dtdyi &
     &                  + p20 * m_oss(1) * 2.d0 * dij * dijdyi &
     &                  + p20 * m_oss(2) * 2.d0 * dtheta * dtdyi &
     &                  + p20 * m_oss(3) * 2.d0 * ( dij*dijdyi*dtheta2 &
     &                                      + (dij2+dkj2)*dtheta*dtdyi )

                  fzi = - k_oss(2) * dijdzi &
     &                  - k_oss(3)  * dtdzi &
     &                  - k_oss(4) * 2.d0 * dij * dijdzi &
     &                  - k_oss(5) * dkj * dijdzi &
     &                  - k_oss(6)  * 2.d0 * dtheta * dtdzi &
     &                  - k_oss(7) * ( dijdzi*dtheta + (dij+dkj)*dtdzi ) &
     &                  - k_oss(8) * 3.d0 * dij2 * dijdzi &
     &                  - k_oss(9) * ( 2.d0*dij*dkj + dkj2 ) * dijdzi &
     &                  - k_oss(10) * 3.d0 * dtheta2 * dtdzi &
     &                  - k_oss(11) * ( 2.d0*dij*dijdzi*dtheta &
     &                                + (dij2+dkj2)*dtdzi ) &
     &                  - k_oss(12) * dkj *( dijdzi*dtheta + dij*dtdzi ) &
     &                  - k_oss(13) * ( dijdzi*dtheta2  &
     &                                + (dij+dkj)*2.d0*dtheta*dtdzi ) &
     &                  - k_oss(14) * 4.d0 * dij3 * dijdzi &
     &                  - k_oss(15) * 2.d0 * dij * dkj2 * dijdzi &
     &                  - k_oss(16) * 4.d0 * dtheta3 * dtdzi &
     &                  + p20 * m_oss(1) * 2.d0 * dij * dijdzi &
     &                  + p20 * m_oss(2) * 2.d0 * dtheta * dtdzi &
     &                  + p20 * m_oss(3) * 2.d0 * ( dij*dijdzi*dtheta2 &
     &                                      + (dij2+dkj2)*dtheta*dtdzi )

                  fxk = - k_oss(2) * dkjdxk &
     &                  - k_oss(3) * dtdxk &
     &                  - k_oss(4) * 2.d0 * dkj * dkjdxk &
     &                  - k_oss(5) * dij * dkjdxk &
     &                  - k_oss(6) * 2.d0 * dtheta * dtdxk &
     &                  - k_oss(7) * ( dkjdxk*dtheta + (dij+dkj)*dtdxk ) &
     &                  - k_oss(8) * 3.d0 * dkj2 * dkjdxk &
     &                  - k_oss(9) * ( 2.d0*dkj*dij + dij2 ) * dkjdxk &
     &                  - k_oss(10) * 3.d0 * dtheta2 * dtdxk &
     &                  - k_oss(11) * ( 2.d0*dkj*dkjdxk*dtheta &
     &                                + (dij2+dkj2)*dtdxk ) &
     &                  - k_oss(12) * dij *( dkjdxk*dtheta + dkj*dtdxk ) &
     &                  - k_oss(13) * ( dkjdxk*dtheta2  &
     &                               + (dij+dkj)*2.d0*dtheta*dtdxk ) &
     &                  - k_oss(14) * 4.d0 * dkj3 * dkjdxk &
     &                  - k_oss(15) * 2.d0 * dkj * dij2 * dkjdxk &
     &                  - k_oss(16) * 4.d0 * dtheta3 * dtdxk &
     &                  + p20 * m_oss(1) * 2.d0 * dkj * dkjdxk &
     &                  + p20 * m_oss(2) * 2.d0 * dtheta * dtdxk &
     &                  + p20 * m_oss(3) * 2.d0 * ( dkj*dkjdxk*dtheta2 &
     &                                      + (dij2+dkj2)*dtheta*dtdxk )

                  fyk = - k_oss(2) * dkjdyk &
     &                  - k_oss(3) * dtdyk &
     &                  - k_oss(4) * 2.d0 * dkj * dkjdyk &
     &                  - k_oss(5) * dij * dkjdyk &
     &                  - k_oss(6) * 2.d0 * dtheta * dtdyk &
     &                  - k_oss(7) * ( dkjdyk*dtheta + (dij+dkj)*dtdyk ) &
     &                  - k_oss(8) * 3.d0 * dkj2 * dkjdyk &
     &                  - k_oss(9) * ( 2.d0*dkj*dij + dij2 ) * dkjdyk &
     &                  - k_oss(10) * 3.d0 * dtheta2 * dtdyk &
     &                  - k_oss(11) * ( 2.d0*dkj*dkjdyk*dtheta &
     &                                + (dij2+dkj2)*dtdyk ) &
     &                  - k_oss(12) * dij *( dkjdyk*dtheta + dkj*dtdyk ) &
     &                  - k_oss(13) * ( dkjdyk*dtheta2  &
     &                                + (dij+dkj)*2.d0*dtheta*dtdyk ) &
     &                  - k_oss(14) * 4.d0 * dkj3 * dkjdyk &
     &                  - k_oss(15) * 2.d0 * dkj * dij2 * dkjdyk &
     &                  - k_oss(16) * 4.d0 * dtheta3 * dtdyk &
     &                  + p20 * m_oss(1) * 2.d0 * dkj * dkjdyk &
     &                  + p20 * m_oss(2) * 2.d0 * dtheta * dtdyk &
     &                  + p20 * m_oss(3) * 2.d0 * ( dkj*dkjdyk*dtheta2 &
     &                                      + (dij2+dkj2)*dtheta*dtdyk )

                  fzk = - k_oss(2) * dkjdzk &
     &                  - k_oss(3) * dtdzk &
     &                  - k_oss(4) * 2.d0 * dkj * dkjdzk &
     &                  - k_oss(5) * dij * dkjdzk &
     &                  - k_oss(6) * 2.d0 * dtheta * dtdzk &
     &                  - k_oss(7) * ( dkjdzk*dtheta + (dij+dkj)*dtdzk ) &
     &                  - k_oss(8) * 3.d0 * dkj2 * dkjdzk &
     &                  - k_oss(9) * ( 2.d0*dkj*dij + dij2 ) * dkjdzk &
     &                  - k_oss(10) * 3.d0 * dtheta2 * dtdzk &
     &                  - k_oss(11) * ( 2.d0*dkj*dkjdzk*dtheta &
     &                                + (dij2+dkj2)*dtdzk ) &
     &                  - k_oss(12) * dij *( dkjdzk*dtheta + dkj*dtdzk ) &
     &                  - k_oss(13) * ( dkjdzk*dtheta2  &
     &                                + (dij+dkj)*2.d0*dtheta*dtdzk ) &
     &                  - k_oss(14) * 4.d0 * dkj3 * dkjdzk &
     &                  - k_oss(15) * 2.d0 * dkj * dij2 * dkjdzk &
     &                  - k_oss(16) * 4.d0 * dtheta3 * dtdzk &
     &                  + p20 * m_oss(1) * 2.d0 * dkj * dkjdzk &
     &                  + p20 * m_oss(2) * 2.d0 * dtheta * dtdzk &
     &                  + p20 * m_oss(3) * 2.d0 * ( dkj*dkjdzk*dtheta2 &
     &                                      + (dij2+dkj2)*dtheta*dtdzk )

                  fxj = - fxi - fxk
                  fyj = - fyi - fyk
                  fzj = - fzi - fzk

                  fx(i,m) = fx(i,m) + fxi
                  fy(i,m) = fy(i,m) + fyi
                  fz(i,m) = fz(i,m) + fzi

                  fx(j,m) = fx(j,m) + fxj
                  fy(j,m) = fy(j,m) + fyj
                  fz(j,m) = fz(j,m) + fzj

                  fx(k,m) = fx(k,m) + fxk
                  fy(k,m) = fy(k,m) + fyk
                  fz(k,m) = fz(k,m) + fzk

                  vir(1,1) = vir(1,1) + fxi*xij + fxk*xkj
                  vir(1,2) = vir(1,2) + fxi*yij + fxk*ykj
                  vir(1,3) = vir(1,3) + fxi*zij + fxk*zkj
                  vir(2,1) = vir(2,1) + fyi*xij + fyk*xkj
                  vir(2,2) = vir(2,2) + fyi*yij + fyk*ykj
                  vir(2,3) = vir(2,3) + fyi*zij + fyk*zkj
                  vir(3,1) = vir(3,1) + fzi*xij + fzk*xkj
                  vir(3,2) = vir(3,2) + fzi*yij + fzk*ykj
                  vir(3,3) = vir(3,3) + fzi*zij + fzk*zkj

               end do

            end do

         end do

      end do

!-----------------------------------------------------------------------
!     //   memory deallocation
!-----------------------------------------------------------------------

      deallocate( i_list )
      deallocate( ni_list )

      return
      end
