!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     energy and force from ONIOM
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine force_oniom
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables from PIMD                                 */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, fx, fy, fz, vir, dipx, dipy, dipz, iounit, nbead, istep

      use qmmm_variables, only : &
     &   pot_a, fx_a, fy_a, fz_a, vir_a, dipx_a, dipy_a, dipz_a, &
     &   pot_b, fx_b, fy_b, fz_b, vir_b, dipx_b, dipy_b, dipz_b, &
     &   natom_p, natom_s, natom_l, iprint_oniom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      integer :: i

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then
         call setup_oniom
         call setup_best
         iset = 1
      end if

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      pot_a(:)   = 0.d0
      pot_b(:)   = 0.d0

      fx_a(:,:)  = 0.d0
      fy_a(:,:)  = 0.d0
      fz_a(:,:)  = 0.d0
      fx_b(:,:)  = 0.d0
      fy_b(:,:)  = 0.d0
      fz_b(:,:)  = 0.d0

      vir_a(:,:) = 0.d0
      vir_b(:,:) = 0.d0

      dipx_a(:)  = 0.d0
      dipy_a(:)  = 0.d0
      dipz_a(:)  = 0.d0
      dipx_b(:)  = 0.d0
      dipy_b(:)  = 0.d0
      dipz_b(:)  = 0.d0

!-----------------------------------------------------------------------
!     /*   part a: high level calculation of primary subsystem        */
!-----------------------------------------------------------------------

      if ( natom_p+natom_l .gt. 0 ) call force_oniom_part_hi_pl

!-----------------------------------------------------------------------
!     /*   part b: low level calculation of primary subsystem         */
!-----------------------------------------------------------------------

      if ( natom_p+natom_l .gt. 0 ) call force_oniom_part_lo_pl

!-----------------------------------------------------------------------
!     /*   part b: low level calculation of whole system              */
!-----------------------------------------------------------------------

      if ( natom_s .gt. 0 ) call force_oniom_part_lo_ps

!-----------------------------------------------------------------------
!     /*   sum                                                        */
!-----------------------------------------------------------------------

      pot(:)  = pot_a(:) + pot_b(:)

      fx(:,:) = fx_a(:,:) + fx_b(:,:)
      fy(:,:) = fy_a(:,:) + fy_b(:,:)
      fz(:,:) = fz_a(:,:) + fz_b(:,:)

      dipx(:) = dipx_a(:) + dipx_b(:)
      dipy(:) = dipy_a(:) + dipy_b(:)
      dipz(:) = dipz_a(:) + dipz_b(:)

      vir(:,:) = vir_a(:,:) + vir_b(:,:)

!-----------------------------------------------------------------------
!     /*   print to file                                              */
!-----------------------------------------------------------------------

      if ( iprint_oniom .gt. 0 ) then
      if ( mod(istep,iprint_oniom) .eq. 0 ) then

      open ( iounit, file = 'oniom.out', access = 'append' )

      do i = 1, nbead
         write( iounit, '(i8,2f16.8)' ) istep, pot_a(i), pot_b(i)
      end do

      close( iounit )

      end if
      end if

      return
      end





!***********************************************************************
      subroutine force_oniom_part_hi_pl
!***********************************************************************
!=======================================================================
!
!     this subroutine runs external calls
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables from PIMD                                 */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, nbead, mbox, oniom_hi_potential, box, iboundary, &
     &   species, iounit, pimd_command

      use qmmm_variables, only : &
     &   fx_a, fy_a, fz_a, dipx_a, dipy_a, dipz_a, pot_a, vir_a, &
     &   r_link, i_link, j_link, natom_p, natom_l, species_link, &
     &   oniom_hi_pl_dat_dir, oniom_hi_pl_scr_dir, layer, natom_s

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   reset   */
      implicit none

!     /*   integers   */
      integer :: ibead, i, j, l, m1, m2, m3

!     /*   real numbers   */
      real(8) :: xi, yi, zi, xj, yj, zj

!     /*   characters   */
      character(len=3)  :: char_num

!     /*   flag for initial setting   */
      integer, save :: iset = 0

!     /*   coordinates in qm calculation   */
      real(8), dimension(3,natom_p+natom_l) :: coord

!     /*   energy in qm calculation   */
      real(8) :: escf

!     /*   forces in qm calculation   */
      real(8), dimension(3,natom_p+natom_l) :: grad_s

!     /*   dipole moment in qm calculation   */
      real(8) :: dipx_s, dipy_s, dipz_s

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   visit first time   */
      if ( iset .eq. 0 ) then

!        /*   loop of beads   */
         do ibead = 1, nbead

!           /*   bead number   */
            call int3_to_char( ibead, char_num )

!           /*   make subdirectories for pimd execution   */
            call system( 'mkdir -p ' // &
     &                   trim(oniom_hi_pl_scr_dir) // '/' // &
     &                   char_num )

!           /*   copy files for pimd execution   */
            call system( 'cp -f ' // trim(oniom_hi_pl_dat_dir) // &
     &                   '/* ' // &
     &                   trim(oniom_hi_pl_scr_dir) // '/' // char_num )

!           /*   copy files for pimd execution   */
            call system( 'cp -f input_default.dat ' // &
     &                   trim(oniom_hi_pl_scr_dir) // '/' // char_num )

!           /*   edit input.dat   */
            open ( iounit, file = trim(oniom_hi_pl_scr_dir) // '/' &
     &                    // char_num // '/input.dat' )

            write( iounit, '(a)'  ) '<input_style>'
            write( iounit, '(a)'  ) 'NEW'
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<method>'
            write( iounit, '(a)'  ) 'STATIC'
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<natom>'
            write( iounit, '(i8)' ) natom_p + natom_l
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<ipotential>'
            write( iounit, '(a)'  ) oniom_hi_potential
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<iboundary>'
            write( iounit, '(i1)' ) iboundary
            if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then
               write( iounit, '(3e24.16)' ) box(1,1:3)
               write( iounit, '(3e24.16)' ) box(2,1:3)
               write( iounit, '(3e24.16)' ) box(3,1:3)
            end if
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<iprint_dip>'
            write( iounit, '(i8)' ) 1
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<pimd_command>'
            write( iounit, '(a)'  ) pimd_command
            write( iounit, '(a)'  )

!           /*   close input file   */
            close( iounit )

         end do

         iset = 1

!     /*   visit first time  */
      end if

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!     /*   coordinates in the frame of the first step (nstep=0)       */
!-----------------------------------------------------------------------

!     //   real atoms

      j = 0

      do i = 1, natom_p + natom_s

         if ( layer(i)(1:1) .eq. 'A' ) then

            j = j + 1

            xi = x(i,ibead)
            yi = y(i,ibead)
            zi = z(i,ibead)

            m1 = mbox(1,i,ibead)
            m2 = mbox(2,i,ibead)
            m3 = mbox(3,i,ibead)

            call pbc_unfold ( xi, yi, zi, m1, m2, m3 )

            coord(1,j) = xi
            coord(2,j) = yi
            coord(3,j) = zi

         end if

      end do

!     //   link atoms

      do l = 1, natom_l

         j = j + 1

         xi = x(i_link(l),ibead)
         yi = y(i_link(l),ibead)
         zi = z(i_link(l),ibead)

         m1 = mbox(1,i_link(l),ibead)
         m2 = mbox(2,i_link(l),ibead)
         m3 = mbox(3,i_link(l),ibead)

         call pbc_unfold ( xi, yi, zi, m1, m2, m3 )

         xj = x(j_link(l),ibead)
         yj = y(j_link(l),ibead)
         zj = z(j_link(l),ibead)

         m1 = mbox(1,j_link(l),ibead)
         m2 = mbox(2,j_link(l),ibead)
         m3 = mbox(3,j_link(l),ibead)

         call pbc_unfold ( xj, yj, zj, m1, m2, m3 )

         coord(1,j) = xi + r_link(l) * ( xj - xi )
         coord(2,j) = yi + r_link(l) * ( yj - yi )
         coord(3,j) = zi + r_link(l) * ( zj - zi )

      end do

!-----------------------------------------------------------------------
!     /*   print structure.dat                                        */
!-----------------------------------------------------------------------

      call int3_to_char( ibead, char_num )

      open ( iounit, file = trim(oniom_hi_pl_scr_dir) // '/' // &
     &                      char_num // '/structure.dat' )

      write( iounit, '(i8)' ) natom_p + natom_l
      write( iounit, '(a)' ) 'BOHR'

      j = 0

      do i = 1, natom_p + natom_s

         if ( layer(i)(1:1) .eq. 'A' ) then

            j = j + 1

            write( iounit, '(a,3e24.16,i2)' ) &
     &         species(i), coord(1:3,j), 1

         end if

      end do

      do i = 1, natom_l

         j = j + 1

         write( iounit, '(a,3e24.16,i2)' ) &
     &      species_link(i), coord(1:3,j), 1

      end do

!-----------------------------------------------------------------------
!     /*   print geometry.ini                                         */
!-----------------------------------------------------------------------

      open ( iounit, file = trim(oniom_hi_pl_scr_dir) // '/' // &
     &                      char_num // '/geometry.ini' )

      do j = 1, natom_p + natom_l

         write( iounit, '(i8,6e24.16,3i4)' ) &
     &      0, coord(1:3,j), 0.d0, 0.d0, 0.d0, 0, 0, 0

      end do

      close( iounit )

!-----------------------------------------------------------------------
!     /*   run PIMD                                                   */
!-----------------------------------------------------------------------

      call system( 'cd ' // trim(oniom_hi_pl_scr_dir) // &
     &             '/' // char_num // &
     &             '; ' // pimd_command // '> monitor.out; cd ../..' )

!-----------------------------------------------------------------------
!     /*   read output file                                           */
!-----------------------------------------------------------------------

      open ( iounit, file = trim(oniom_hi_pl_scr_dir) // '/' // &
     &                      char_num // '/forces.out' )

      read ( iounit, * ) escf

      do j = 1, natom_p + natom_l

         read ( iounit, * ) grad_s(1:3,j)

      end do

      close( iounit )

      open ( iounit, file = trim(oniom_hi_pl_scr_dir) // '/' // &
     &                      char_num // '/dipole.out' )

      read ( iounit, * ) dipx_s, dipy_s, dipz_s

      close( iounit )

!-----------------------------------------------------------------------
!     /*   potential                                                  */
!-----------------------------------------------------------------------

      pot_a(ibead) = pot_a(ibead) + escf

!-----------------------------------------------------------------------
!     /*   force                                                      */
!-----------------------------------------------------------------------

!     /*   real atoms   */

      j = 0

      do i = 1, natom_p + natom_s

         if ( layer(i)(1:1) .eq. 'A' ) then

            j = j + 1

            fx_a(i,ibead) = fx_a(i,ibead) + grad_s(1,j)
            fy_a(i,ibead) = fy_a(i,ibead) + grad_s(2,j)
            fz_a(i,ibead) = fz_a(i,ibead) + grad_s(3,j)

         end if

      end do

!     /*   link atoms   */

      do l = 1, natom_l

         j = j + 1

         fx_a(i_link(l),ibead) = fx_a(i_link(l),ibead) &
     &      + ( 1.d0 - r_link(l) ) * grad_s(1,j)
         fy_a(i_link(l),ibead) = fy_a(i_link(l),ibead) &
     &      + ( 1.d0 - r_link(l) ) * grad_s(2,j)
         fz_a(i_link(l),ibead) = fz_a(i_link(l),ibead) &
     &      + ( 1.d0 - r_link(l) ) * grad_s(3,j)

         fx_a(j_link(l),ibead) = fx_a(j_link(l),ibead) &
     &      + r_link(l) * grad_s(1,j)
         fy_a(j_link(l),ibead) = fy_a(j_link(l),ibead) &
     &      + r_link(l) * grad_s(2,j)
         fz_a(j_link(l),ibead) = fz_a(j_link(l),ibead) &
     &      + r_link(l) * grad_s(3,j)

      end do

!-----------------------------------------------------------------------
!     /*   dipole moment                                              */
!-----------------------------------------------------------------------

      dipx_a(ibead) = dipx_a(ibead) + dipx_s
      dipy_a(ibead) = dipy_a(ibead) + dipy_s
      dipz_a(ibead) = dipz_a(ibead) + dipz_s

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

!     /*   real atoms   */

      j = 0

      do i = 1, natom_p + natom_s

         if ( layer(i)(1:1) .eq. 'A' ) then

            j = j + 1

            xi = coord(1,j)
            yi = coord(2,j)
            zi = coord(3,j)

            vir_a(1,1) = vir_a(1,1) + grad_s(1,j)*xi
            vir_a(1,2) = vir_a(1,2) + grad_s(1,j)*yi
            vir_a(1,3) = vir_a(1,3) + grad_s(1,j)*zi
            vir_a(2,1) = vir_a(2,1) + grad_s(2,j)*xi
            vir_a(2,2) = vir_a(2,2) + grad_s(2,j)*yi
            vir_a(2,3) = vir_a(2,3) + grad_s(2,j)*zi
            vir_a(3,1) = vir_a(3,1) + grad_s(3,j)*xi
            vir_a(3,2) = vir_a(3,2) + grad_s(3,j)*yi
            vir_a(3,3) = vir_a(3,3) + grad_s(3,j)*zi

         end if

      end do

!     /*   link atoms   */

      do l = 1, natom_l

         j = j + 1

         xi = coord(1,j)
         yi = coord(2,j)
         zi = coord(3,j)

         vir_a(1,1) = vir_a(1,1) + grad_s(1,j)*xi
         vir_a(1,2) = vir_a(1,2) + grad_s(1,j)*yi
         vir_a(1,3) = vir_a(1,3) + grad_s(1,j)*zi
         vir_a(2,1) = vir_a(2,1) + grad_s(2,j)*xi
         vir_a(2,2) = vir_a(2,2) + grad_s(2,j)*yi
         vir_a(2,3) = vir_a(2,3) + grad_s(2,j)*zi
         vir_a(3,1) = vir_a(3,1) + grad_s(3,j)*xi
         vir_a(3,2) = vir_a(3,2) + grad_s(3,j)*yi
         vir_a(3,3) = vir_a(3,3) + grad_s(3,j)*zi

      end do

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

      return
      end





!***********************************************************************
      subroutine force_oniom_part_lo_pl
!***********************************************************************
!=======================================================================
!
!     this subroutine runs external calls
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables from PIMD                                 */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, nbead, mbox, oniom_lo_potential, box, iboundary, &
     &   species, iounit, pimd_command

      use qmmm_variables, only : &
     &   fx_b, fy_b, fz_b, dipx_b, dipy_b, dipz_b, pot_b, vir_b, &
     &   r_link, i_link, j_link, natom_p, natom_l, species_link, &
     &   oniom_lo_pl_dat_dir, oniom_lo_pl_scr_dir, natom_s, layer

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   reset   */
      implicit none

!     /*   integers   */
      integer :: ibead, i, j, l, m1, m2, m3

!     /*   real numbers   */
      real(8) :: xi, yi, zi, xj, yj, zj

!     /*   characters   */
      character(len=3)  :: char_num

!     /*   flag for initial setting   */
      integer, save :: iset = 0

!     /*   coordinates in qm calculation   */
      real(8), dimension(3,natom_p+natom_l) :: coord

!     /*   energy in qm calculation   */
      real(8) :: escf

!     /*   forces in qm calculation   */
      real(8), dimension(3,natom_p+natom_l) :: grad_s

!     /*   dipole moment in qm calculation   */
      real(8) :: dipx_s, dipy_s, dipz_s

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   visit first time   */
      if ( iset .eq. 0 ) then

!        /*   loop of beads   */
         do ibead = 1, nbead

!           /*   bead number   */
            call int3_to_char( ibead, char_num )

!           /*   make subdirectories for pimd execution   */
            call system( 'mkdir -p ' // trim(oniom_lo_pl_scr_dir) // &
     &                   '/' // char_num )

!           /*   copy files for pimd execution   */
            call system( 'cp -f ' // trim(oniom_lo_pl_dat_dir) // &
     &                   '/* ' // &
     &                   trim(oniom_lo_pl_scr_dir) // '/' // char_num )

!           /*   copy files for pimd execution   */
            call system( 'cp -f input_default.dat ' // &
     &                   trim(oniom_lo_pl_scr_dir) // '/' // char_num )

!           /*   edit input.dat   */
            open ( iounit, file = trim(oniom_lo_pl_scr_dir) // '/' &
     &                    // char_num // '/input.dat' )

            write( iounit, '(a)'  ) '<input_style>'
            write( iounit, '(a)'  ) 'NEW'
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<method>'
            write( iounit, '(a)'  ) 'STATIC'
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<natom>'
            write( iounit, '(i8)' ) natom_p + natom_l
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<ipotential>'
            write( iounit, '(a)'  ) oniom_lo_potential
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<iboundary>'
            write( iounit, '(i1)' ) iboundary
            if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then
               write( iounit, '(3e24.16)' ) box(1,1:3)
               write( iounit, '(3e24.16)' ) box(2,1:3)
               write( iounit, '(3e24.16)' ) box(3,1:3)
            end if
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<iprint_dip>'
            write( iounit, '(i8)' ) 1
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<pimd_command>'
            write( iounit, '(a)'  ) pimd_command
            write( iounit, '(a)'  )

!           /*   close input file   */
            close( iounit )

         end do

         iset = 1

!     /*   visit first time  */
      end if

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!     /*   coordinates in the frame of the first step (nstep=0)       */
!-----------------------------------------------------------------------

!     //   real atoms

      j = 0

      do i = 1, natom_p + natom_s

         if ( layer(i)(1:1) .eq. 'A' ) then

            j = j + 1

            xi = x(i,ibead)
            yi = y(i,ibead)
            zi = z(i,ibead)

            m1 = mbox(1,i,ibead)
            m2 = mbox(2,i,ibead)
            m3 = mbox(3,i,ibead)

            call pbc_unfold ( xi, yi, zi, m1, m2, m3 )

            coord(1,j) = xi
            coord(2,j) = yi
            coord(3,j) = zi

         end if

      end do

!     //   link atoms

      do l = 1, natom_l

         j = j + 1

         xi = x(i_link(l),ibead)
         yi = y(i_link(l),ibead)
         zi = z(i_link(l),ibead)

         m1 = mbox(1,i_link(l),ibead)
         m2 = mbox(2,i_link(l),ibead)
         m3 = mbox(3,i_link(l),ibead)

         call pbc_unfold ( xi, yi, zi, m1, m2, m3 )

         xj = x(j_link(l),ibead)
         yj = y(j_link(l),ibead)
         zj = z(j_link(l),ibead)

         m1 = mbox(1,j_link(l),ibead)
         m2 = mbox(2,j_link(l),ibead)
         m3 = mbox(3,j_link(l),ibead)

         call pbc_unfold ( xj, yj, zj, m1, m2, m3 )

         coord(1,j) = xi + r_link(l) * ( xj - xi )
         coord(2,j) = yi + r_link(l) * ( yj - yi )
         coord(3,j) = zi + r_link(l) * ( zj - zi )

      end do

!-----------------------------------------------------------------------
!     /*   print structure.dat                                        */
!-----------------------------------------------------------------------

      call int3_to_char( ibead, char_num )

      open ( iounit, file = trim(oniom_lo_pl_scr_dir) // '/' // &
     &                      char_num // '/structure.dat' )

      write( iounit, '(i8)' ) natom_p + natom_l
      write( iounit, '(a)' ) 'BOHR'

      j = 0

      do i = 1, natom_p + natom_s

         if ( layer(i)(1:1) .eq. 'A' ) then

            j = j + 1

            write( iounit, '(a,3e24.16,i2)' ) &
     &         species(i), coord(1:3,j), 1

         end if

      end do

      do i = 1, natom_l

         j = j + 1

         write( iounit, '(a,3e24.16,i2)' ) &
     &      species_link(i), coord(1:3,j), 1

      end do

!-----------------------------------------------------------------------
!     /*   print geometry.ini                                         */
!-----------------------------------------------------------------------

      open ( iounit, file = trim(oniom_lo_pl_scr_dir) // '/' // &
     &                      char_num // '/geometry.ini' )

      do j = 1, natom_p + natom_l

         write( iounit, '(i8,6e24.16,3i4)' ) &
     &      0, coord(1:3,j), 0.d0, 0.d0, 0.d0, 0, 0, 0

      end do

      close( iounit )

!-----------------------------------------------------------------------
!     /*   run PIMD                                                   */
!-----------------------------------------------------------------------

      call system( 'cd ' // trim(oniom_lo_pl_scr_dir) // &
     &             '/' // char_num // &
     &             '; ' // pimd_command // '> monitor.out; cd ../..' )

!-----------------------------------------------------------------------
!     /*   read output file                                           */
!-----------------------------------------------------------------------

      open ( iounit, file = trim(oniom_lo_pl_scr_dir) // '/' // &
     &                      char_num // '/forces.out' )

      read ( iounit, * ) escf

      do j = 1, natom_p + natom_l

         read ( iounit, * ) grad_s(1:3,j)

      end do

      close( iounit )

      open ( iounit, file = trim(oniom_lo_pl_scr_dir) // '/' // &
     &                      char_num // '/dipole.out' )

      read ( iounit, * ) dipx_s, dipy_s, dipz_s

      close( iounit )

!-----------------------------------------------------------------------
!     /*   potential                                                  */
!-----------------------------------------------------------------------

      pot_b(ibead) = pot_b(ibead) - escf

!-----------------------------------------------------------------------
!     /*   force                                                      */
!-----------------------------------------------------------------------

!     /*   real atoms   */

      j = 0

      do i = 1, natom_p + natom_s

         if ( layer(i)(1:1) .eq. 'A' ) then

            j = j + 1

            fx_b(i,ibead) = fx_b(i,ibead) - grad_s(1,j)
            fy_b(i,ibead) = fy_b(i,ibead) - grad_s(2,j)
            fz_b(i,ibead) = fz_b(i,ibead) - grad_s(3,j)

         end if

      end do

!     /*   link atoms   */

      do l = 1, natom_l

         j = j + 1

         fx_b(i_link(l),ibead) = fx_b(i_link(l),ibead) &
     &      - ( 1.d0 - r_link(l) ) * grad_s(1,j)
         fy_b(i_link(l),ibead) = fy_b(i_link(l),ibead) &
     &      - ( 1.d0 - r_link(l) ) * grad_s(2,j)
         fz_b(i_link(l),ibead) = fz_b(i_link(l),ibead) &
     &      - ( 1.d0 - r_link(l) ) * grad_s(3,j)

         fx_b(j_link(l),ibead) = fx_b(j_link(l),ibead) &
     &      - r_link(l) * grad_s(1,j)
         fy_b(j_link(l),ibead) = fy_b(j_link(l),ibead) &
     &      - r_link(l) * grad_s(2,j)
         fz_b(j_link(l),ibead) = fz_b(j_link(l),ibead) &
     &      - r_link(l) * grad_s(3,j)

      end do

!-----------------------------------------------------------------------
!     /*   dipole moment                                              */
!-----------------------------------------------------------------------

      dipx_b(ibead) = dipx_b(ibead) - dipx_s
      dipy_b(ibead) = dipy_b(ibead) - dipy_s
      dipz_b(ibead) = dipz_b(ibead) - dipz_s

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

!     /*   real atoms   */

      j = 0

      do i = 1, natom_p + natom_s

         if ( layer(i)(1:1) .eq. 'A' ) then

            j = j + 1

            xi = coord(1,j)
            yi = coord(2,j)
            zi = coord(3,j)

            vir_b(1,1) = vir_b(1,1) - grad_s(1,j)*xi
            vir_b(1,2) = vir_b(1,2) - grad_s(1,j)*yi
            vir_b(1,3) = vir_b(1,3) - grad_s(1,j)*zi
            vir_b(2,1) = vir_b(2,1) - grad_s(2,j)*xi
            vir_b(2,2) = vir_b(2,2) - grad_s(2,j)*yi
            vir_b(2,3) = vir_b(2,3) - grad_s(2,j)*zi
            vir_b(3,1) = vir_b(3,1) - grad_s(3,j)*xi
            vir_b(3,2) = vir_b(3,2) - grad_s(3,j)*yi
            vir_b(3,3) = vir_b(3,3) - grad_s(3,j)*zi

         end if

      end do

!     /*   link atoms   */

      do l = 1, natom_l

         j = j + 1

         xi = coord(1,j)
         yi = coord(2,j)
         zi = coord(3,j)

         vir_b(1,1) = vir_b(1,1) - grad_s(1,j)*xi
         vir_b(1,2) = vir_b(1,2) - grad_s(1,j)*yi
         vir_b(1,3) = vir_b(1,3) - grad_s(1,j)*zi
         vir_b(2,1) = vir_b(2,1) - grad_s(2,j)*xi
         vir_b(2,2) = vir_b(2,2) - grad_s(2,j)*yi
         vir_b(2,3) = vir_b(2,3) - grad_s(2,j)*zi
         vir_b(3,1) = vir_b(3,1) - grad_s(3,j)*xi
         vir_b(3,2) = vir_b(3,2) - grad_s(3,j)*yi
         vir_b(3,3) = vir_b(3,3) - grad_s(3,j)*zi

      end do

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

      return
      end





!***********************************************************************
      subroutine force_oniom_part_lo_ps
!***********************************************************************
!=======================================================================
!
!     this subroutine runs external calls
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables from PIMD                                 */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, nbead, mbox, oniom_lo_potential, box, iboundary, &
     &   species, iounit, pimd_command

      use qmmm_variables, only : &
     &   fx_b, fy_b, fz_b, dipx_b, dipy_b, dipz_b, pot_b, vir_b, &
     &   natom_p, natom_s, oniom_lo_ps_dat_dir, oniom_lo_ps_scr_dir

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   reset   */
      implicit none

!     /*   integers   */
      integer :: ibead, i, m1, m2, m3

!     /*   real numbers   */
      real(8) :: xi, yi, zi

!     /*   characters   */
      character(len=3)  :: char_num

!     /*   flag for initial setting   */
      integer, save :: iset = 0

!     /*   coordinates in qm calculation   */
      real(8), dimension(3,natom_p+natom_s) :: coord

!     /*   energy in qm calculation   */
      real(8) :: escf

!     /*   forces in qm calculation   */
      real(8), dimension(3,natom_p+natom_s) :: grad_s

!     /*   dipole moment in qm calculation   */
      real(8) :: dipx_s, dipy_s, dipz_s

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   visit first time   */
      if ( iset .eq. 0 ) then

!        /*   loop of beads   */
         do ibead = 1, nbead

!           /*   bead number   */
            call int3_to_char( ibead, char_num )

!           /*   make subdirectories for pimd execution   */
            call system( 'mkdir -p ' // trim(oniom_lo_ps_scr_dir) // &
     &                   '/' // char_num )

!           /*   copy files for pimd execution   */
            call system( 'cp -f ' // &
     &                   trim(oniom_lo_ps_dat_dir) // '/* ' // &
     &                   trim(oniom_lo_ps_scr_dir) // '/' // char_num )

!           /*   copy files for pimd execution   */
            call system( 'cp -f input_default.dat ' // &
     &                   trim(oniom_lo_ps_scr_dir) // '/' // char_num )

!           /*   edit input.dat   */
            open ( iounit, file = trim(oniom_lo_ps_scr_dir) // '/' &
     &                    // char_num // '/input.dat' )

            write( iounit, '(a)'  ) '<input_style>'
            write( iounit, '(a)'  ) 'NEW'
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<method>'
            write( iounit, '(a)'  ) 'STATIC'
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<natom>'
            write( iounit, '(i8)' ) natom_p + natom_s
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<ipotential>'
            write( iounit, '(a)'  ) oniom_lo_potential
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<iboundary>'
            write( iounit, '(i1)' ) iboundary
            if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then
               write( iounit, '(3e24.16)' ) box(1,1:3)
               write( iounit, '(3e24.16)' ) box(2,1:3)
               write( iounit, '(3e24.16)' ) box(3,1:3)
            end if
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<iprint_dip>'
            write( iounit, '(i8)' ) 1
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<pimd_command>'
            write( iounit, '(a)'  ) pimd_command
            write( iounit, '(a)'  )

!           /*   close input file   */
            close( iounit )

         end do

         iset = 1

!     /*   visit first time  */
      end if

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!     /*   coordinates in the frame of the first step (nstep=0)       */
!-----------------------------------------------------------------------

!     //   real atoms

      do i = 1, natom_p + natom_s

         xi = x(i,ibead)
         yi = y(i,ibead)
         zi = z(i,ibead)

         m1 = mbox(1,i,ibead)
         m2 = mbox(2,i,ibead)
         m3 = mbox(3,i,ibead)

         call pbc_unfold ( xi, yi, zi, m1, m2, m3 )

         coord(1,i) = xi
         coord(2,i) = yi
         coord(3,i) = zi

      end do

!-----------------------------------------------------------------------
!     /*   print structure.dat                                        */
!-----------------------------------------------------------------------

      call int3_to_char( ibead, char_num )

      open ( iounit, file = trim(oniom_lo_ps_scr_dir) // &
     &                      '/' // char_num // '/structure.dat' )

      write( iounit, '(i8)' ) natom_p + natom_s
      write( iounit, '(a)' ) 'BOHR'

      do i = 1, natom_p + natom_s
         write( iounit, '(a,3e24.16,i2)' ) &
     &      species(i), coord(1:3,i), 1
      end do

!-----------------------------------------------------------------------
!     /*   print geometry.ini                                         */
!-----------------------------------------------------------------------

      open ( iounit, file = trim(oniom_lo_ps_scr_dir) // &
     &                      '/' // char_num // '/geometry.ini' )

      do i = 1, natom_p + natom_s
         write( iounit, '(i8,6e24.16,3i4)' ) &
     &      0, coord(1:3,i), 0.d0, 0.d0, 0.d0, 0, 0, 0
      end do

      close( iounit )

!-----------------------------------------------------------------------
!     /*   run PIMD                                                   */
!-----------------------------------------------------------------------

      call system( 'cd ' // trim(oniom_lo_ps_scr_dir) // &
     &             '/' // char_num // &
     &             '; ' // pimd_command // '> monitor.out; cd ../..' )

!-----------------------------------------------------------------------
!     /*   read output file                                           */
!-----------------------------------------------------------------------

      open ( iounit, file = trim(oniom_lo_ps_scr_dir) // '/' // &
     &                      char_num // '/forces.out' )

      read ( iounit, * ) escf

      do i = 1, natom_p + natom_s

         read ( iounit, * ) grad_s(1:3,i)

      end do

      close( iounit )

      open ( iounit, file = trim(oniom_lo_ps_scr_dir) // '/' // &
     &                      char_num // '/dipole.out' )

      read ( iounit, * ) dipx_s, dipy_s, dipz_s

      close( iounit )

!-----------------------------------------------------------------------
!     /*   potential                                                  */
!-----------------------------------------------------------------------

      pot_b(ibead) = pot_b(ibead) + escf

!-----------------------------------------------------------------------
!     /*   force                                                      */
!-----------------------------------------------------------------------

!     /*   real atoms   */

      do i = 1, natom_p + natom_s
         fx_b(i,ibead) = fx_b(i,ibead) + grad_s(1,i)
         fy_b(i,ibead) = fy_b(i,ibead) + grad_s(2,i)
         fz_b(i,ibead) = fz_b(i,ibead) + grad_s(3,i)
      end do

!-----------------------------------------------------------------------
!     /*   dipole moment                                              */
!-----------------------------------------------------------------------

      dipx_b(ibead) = dipx_b(ibead) + dipx_s
      dipy_b(ibead) = dipy_b(ibead) + dipy_s
      dipz_b(ibead) = dipz_b(ibead) + dipz_s

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

!     /*   real atoms   */

      do i = 1, natom_p + natom_s

         xi = coord(1,i)
         yi = coord(2,i)
         zi = coord(3,i)

         vir_b(1,1) = vir_b(1,1) + grad_s(1,i)*xi
         vir_b(1,2) = vir_b(1,2) + grad_s(1,i)*yi
         vir_b(1,3) = vir_b(1,3) + grad_s(1,i)*zi
         vir_b(2,1) = vir_b(2,1) + grad_s(2,i)*xi
         vir_b(2,2) = vir_b(2,2) + grad_s(2,i)*yi
         vir_b(2,3) = vir_b(2,3) + grad_s(2,i)*zi
         vir_b(3,1) = vir_b(3,1) + grad_s(3,i)*xi
         vir_b(3,2) = vir_b(3,2) + grad_s(3,i)*yi
         vir_b(3,3) = vir_b(3,3) + grad_s(3,i)*zi

      end do

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

      return
      end
