!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by Y. Kawashima, M. Shiga
!      Description:     energy and force from NTCHEM calculation
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      module ntchem_variables
!***********************************************************************

!     /*   number of cores per bead   */
      integer :: ncore_replica_ntchem

!     /*   bead rank   */
      integer :: mycolor_ntchem

!     /*   ntchem rank    */
      integer :: mykey_ntchem

!     /*   communicator within a bead   */
      integer :: mpi_comm_replica_ntchem

!***********************************************************************
      end module ntchem_variables
!***********************************************************************



#ifdef ntchem

!***********************************************************************
      subroutine force_ntchem_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, dipx, dipy, dipz, au_length, &
     &   au2debye, species, nbead, natom, mbox, iounit, iounit_ntchem, &
     &   myrank, nprocs, mpi_comm_pimd

      use ntchem_variables, only : &
     &   ncore_replica_ntchem, mycolor_ntchem, mykey_ntchem, &
     &   mpi_comm_replica_ntchem

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize variables
      implicit none

!     //   integers
      integer :: i, j, k, ierr

!     //   real numbers
      real(8) :: xi, yi, zi, ai, bi, ci, di, ei

!     //   number character
      character(len=3) :: char_num

!     //   atomic symbol character
      character(len=4) :: char_symbol

!     //   file character
      character(len=80) :: char_file

!     //   flag
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   create i/o directories for NTCHEM                          */
!-----------------------------------------------------------------------

!     //   initial visit only
      if ( iset .eq. 0 ) then

!        //   master rank only
         if ( myrank .eq. 0 ) then

!           //   loop of beads
            do j = 1, nbead

!              //   bead number in characters
               call int3_to_char( j, char_num )

!              //   set directory name
               if      ( j .lt. 10  ) then
                  char_file = ( './ntchem_' // char_num(3:3) )
               else if ( j .lt. 100 ) then
                  char_file = ( './ntchem_' // char_num(2:3) )
               else
                  char_file = ( './ntchem_' // char_num(1:3) )
               end if

!              //   clear directories
               call system ('rm -f -r ' // trim(char_file) )

!              //   create directories
               call system ('mkdir -p ' // trim(char_file) )

!           //   loop of beads
            end do

!        //   master rank only
         end if

!     //   initial visit only
      end if

!-----------------------------------------------------------------------
!     /*   write geometries in input files                            */
!-----------------------------------------------------------------------

!     //   master rank only
      if ( myrank .eq. 0 ) then

!     //   loop of beads
      do j = 1, nbead

!        //   bead number in characters 000-999
         call int3_to_char( j, char_num )

!        //   set file name
         if      ( j .lt. 10  ) then
            char_file = ( './INPUT.Geom.' // char_num(3:3) )
         else if ( j .lt. 100 ) then
            char_file = ( './INPUT.Geom.' // char_num(2:3) )
         else
            char_file = ( './INPUT.Geom.' // char_num(1:3) )
         end if

!        //   open input file
         open ( iounit, file = trim(char_file) )

!        //   write comment
         write( iounit, '(a)' ) 'Geom'

!        //   loop of atoms
         do i = 1, natom

!           //   cartesian coordinates [bohr]
            xi = x(i,j)
            yi = y(i,j)
            zi = z(i,j)

!           //   unfold if periodic boundary condition is applied
            call pbc_unfold_MPI &
     &         ( xi, yi, zi, mbox(1,i,j), mbox(2,i,j), mbox(3,i,j) )

!           /*   write ntchem input file   */
            write( iounit, '(a4,3f16.8)' ) species(i), xi, yi, zi

!        //   loop of atoms
         end do

!        //   write comment
         write( iounit, '(a)' ) 'End'

!        //   close input file
         close( iounit )

!     //   loop of beads
      end do

!     //   master rank only
      end if

!-----------------------------------------------------------------------
!     /*   write geometries in input files                            */
!-----------------------------------------------------------------------

!     //   master rank only
      if ( myrank .eq. 0 ) then

!        //   loop of beads
         do j = 1, nbead

!           //   bead number in characters 000-999
            call int3_to_char( j, char_num )

!           //   set file name
            if      ( j .lt. 10  ) then
               char_file = ( './ntchem_' // char_num(3:3) // &
     &                        '/ntchem_' // char_num(3:3) // &
     &                        '.Geom' )
            else if ( j .lt. 100 ) then
               char_file = ( './ntchem_' // char_num(2:3) // &
     &                        '/ntchem_' // char_num(2:3) // &
     &                        '.Geom' )
            else
               char_file = ( './ntchem_' // char_num(1:3) // &
     &                        '/ntchem_' // char_num(1:3) // &
     &                        '.Geom' )
            end if

!           //   open input file
            open ( iounit, file = trim(char_file) )

!           //   open format file
            open ( iounit_ntchem, file = 'INPUT.xyz' )

!           //   read a line
            read ( iounit_ntchem, *, iostat=ierr )

!           //   write number of atoms
            write( iounit, '(i8)' ) natom

!           //   loop of atoms
            do i = 1, natom

!              //   cartesian coordinates [bohr]
               xi = x(i,j)
               yi = y(i,j)
               zi = z(i,j)

!              //   unfold if periodic boundary condition is applied
               call pbc_unfold_MPI &
     &            ( xi, yi, zi, mbox(1,i,j), mbox(2,i,j), mbox(3,i,j) )

!              /*   read ntchem format file   */
               read ( iounit_ntchem, *, iostat=ierr ) &
     &            char_symbol, ai, bi, ci, di, ei

!              /*   write ntchem input file   */
               write( iounit, '(a4,5f16.8)' ) &
     &            char_symbol, xi, yi, zi, di, ei

!           //   loop of atoms
            end do

!           //   close format file
            close( iounit_ntchem )

!           //   close input file
            close( iounit )

!        //   loop of beads
         end do

!     //   master rank only
      end if

!     //   communicate flag
      call my_mpi_bcast_int_0( ierr )

!     //   stop on error
      call error_handling_MPI( ierr, 'subroutine force_ntchem_MPI', 27 )

!-----------------------------------------------------------------------
!     /*   mpi environment for NTCHEM                                 */
!-----------------------------------------------------------------------

!     //   initial visit only
      if ( iset .eq. 0 ) then

!        /*   read ncore_replica_ntchem: number of cores per bead   */
         call mntchem_read_input &
     &      ( nprocs, myrank, ncore_replica_ntchem )

!        //   check if ncore_replica_ntchem is correct
         ierr = abs( mod( nprocs, nbead ) )
         ierr = ierr + abs( nprocs / nbead - ncore_replica_ntchem )

!        //   error message
         if ( ierr .ne. 0 ) then
            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) 'Error - Number of cores ' // &
     &            'is inconsistent with number of beads.'
            end if
         end if

!        //   stop on error
         call error_handling_MPI &
     &      ( ierr, 'subroutine force_ntchem_MPI', 27 )

!        /*   bead rank   */
         mycolor_ntchem = myrank / ncore_replica_ntchem

!        /*   ntchem rank   */
         mykey_ntchem = mod( myrank, ncore_replica_ntchem )

!        /*   create communicator: mpi_comm_replica_ntchem   */
         call MPI_COMM_SPLIT &
     &      ( mpi_comm_pimd, mycolor_ntchem, mykey_ntchem, &
     &        mpi_comm_replica_ntchem, ierr )

!        //   stop on error
         call error_handling_MPI &
     &      ( ierr, 'subroutine force_ntchem_MPI', 27 )

!        /*   preparation of NTCHEM   */
         call mntchem_prep &
     &      ( mycolor_ntchem, mpi_comm_replica_ntchem )

!     //   initial visit only
      end if

!-----------------------------------------------------------------------
!     /*   set done                                                   */
!-----------------------------------------------------------------------

!     //   change flag
      iset = 1

!-----------------------------------------------------------------------
!     /*   execute NTCHEM                                             */
!-----------------------------------------------------------------------

!     //   wait for all processes
      call my_mpi_barrier

#ifdef omp
!     //   run NTCHEM with OPENMP
      call mntchem_driv_mpiomp
#else
!     //   run NTCHEM without OPENMP
      call mntchem_driv_mpi
#endif

!     //   wait for all processes
      call my_mpi_barrier

!-----------------------------------------------------------------------
!     /*   read potential energies                                    */
!-----------------------------------------------------------------------

!     //   master process only
      if ( myrank .eq. 0 ) then

!        //   loop of beads
         do j = 1, nbead

!           //   bead number in characters 000-999
            call int3_to_char( j, char_num )

            if      ( j .lt. 10  ) then
               char_file = ( './ntchem_' // char_num(3:3) // &
     &                        '/ntchem_' // char_num(3:3) // &
     &                        '.TotEne' )
            else if ( j .lt. 100 ) then
               char_file = ( './ntchem_' // char_num(2:3) // &
     &                        '/ntchem_' // char_num(2:3) // &
     &                        '.TotEne' )
            else
               char_file = ( './ntchem_' // char_num(1:3) // &
     &                        '/ntchem_' // char_num(1:3) // &
     &                        '.TotEne' )
            end if

!           //   open output file
            open ( iounit, file = trim(char_file) )

!           //   read potential energy
            read( iounit, *, iostat=ierr ) pot(j)

!           //   close output file
            close( iounit )

!        //   loop of beads
         end do

!     //   master process only
      end if

!     //   communicate flag
      call my_mpi_bcast_int_0( ierr )

!     //   error termination
      call error_handling_MPI( ierr, 'subroutine force_ntchem_MPI', 27 )

!-----------------------------------------------------------------------
!     /*   read forces                                                */
!-----------------------------------------------------------------------

!     //   master process only
      if ( myrank .eq. 0 ) then

!        //   loop of beads
         do j = 1, nbead

!           //   bead number in characters 000-999
            call int3_to_char( j, char_num )

!           //   set file name
            if      ( j .lt. 10  ) then
               char_file = ( './ntchem_' // char_num(3:3) // &
     &                        '/ntchem_' // char_num(3:3) // &
     &                        '.Grad' )
            else if ( j .lt. 100 ) then
               char_file = ( './ntchem_' // char_num(2:3) // &
     &                        '/ntchem_' // char_num(2:3) // &
     &                        '.Grad' )
            else
               char_file = ( './ntchem_' // char_num(1:3) // &
     &                        '/ntchem_' // char_num(1:3) // &
     &                        '.Grad' )
            end if

!           //   open output file
            open ( iounit, file = trim(char_file) )

!           //   read forces
            do i = 1, natom
               read( iounit, *, iostat=ierr ) fx(i,j), fy(i,j), fz(i,j)
            end do

!           //   change sign: gradient to force
            fx(:,j) = - fx(:,j)
            fy(:,j) = - fy(:,j)
            fz(:,j) = - fz(:,j)

!           //   close output file
            close( iounit )

!        //   loop of beads
         end do

!     //   master process only
      end if

!     //   communicate flag
      call my_mpi_bcast_int_0( ierr )

!     //   error termination
      call error_handling_MPI( ierr, 'subroutine force_ntchem_MPI', 27 )

!-----------------------------------------------------------------------
!     /*   read dipole moments                                        */
!-----------------------------------------------------------------------

!     //   master process only
      if ( myrank .eq. 0 ) then

!        //   loop of beads
         do j = 1, nbead

!           //   bead number in characters 000-999
            call int3_to_char( j, char_num )

!           //   set file name
            if      ( j .lt. 10  ) then
               char_file = ( './ntchem_' // char_num(3:3) // &
     &                        '/ntchem_' // char_num(3:3) // &
     &                        '.Dipole.Txt' )
            else if ( j .lt. 100 ) then
               char_file = ( './ntchem_' // char_num(2:3) // &
     &                        '/ntchem_' // char_num(2:3) // &
     &                        '.Dipole.Txt' )
            else
               char_file = ( './ntchem_' // char_num(1:3) // &
     &                        '/ntchem_' // char_num(1:3) // &
     &                        '.Dipole.Txt' )
            end if

!           //   open output file
            open ( iounit, file = trim(char_file) )

!           //   skip a line
            read( iounit, *, iostat=ierr )

!           //   read dipole moment
            read( iounit, *, iostat=ierr ) k, dipx(j)
            read( iounit, *, iostat=ierr ) k, dipy(j)
            read( iounit, *, iostat=ierr ) k, dipz(j)

!           /*   change unit:  debye -> au   */
            dipx(j) = dipx(j) / au2debye
            dipy(j) = dipy(j) / au2debye
            dipz(j) = dipz(j) / au2debye

!           //   close output file
            close( iounit )

!        //   loop of beads
         end do

!     //   master process only
      end if

!     //   communicate flag
      call my_mpi_bcast_int_0( ierr )

!     //   error termination
!cc   call error_handling_MPI( ierr, 'subroutine force_ntchem_MPI', 27 )

!-----------------------------------------------------------------------
!     //   on error set dipole moment equal to zero
!-----------------------------------------------------------------------

      if ( ierr .ne. 0 ) then
         dipx(:) = 0.d0
         dipy(:) = 0.d0
         dipz(:) = 0.d0
      end if

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz, natom, nbead )

!     /*   dipole   */
      call my_mpi_allreduce_real_1 ( dipx, nbead )
      call my_mpi_allreduce_real_1 ( dipy, nbead )
      call my_mpi_allreduce_real_1 ( dipz, nbead )

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

!     //   loops of atoms and beads
      do j = 1, nbead
      do i = 1, natom

!        //   atomic positions
         xi = x(i,j)
         yi = y(i,j)
         zi = z(i,j)

!        //   unfold if periodic boundary condition is applied
         call pbc_unfold_MPI &
     &     ( xi, yi, zi, mbox(1,i,j), mbox(2,i,j), mbox(3,i,j) )

!        //   calculate virial
         vir(1,1) = vir(1,1) + fx(i,j)*xi
         vir(1,2) = vir(1,2) + fx(i,j)*yi
         vir(1,3) = vir(1,3) + fx(i,j)*zi
         vir(2,1) = vir(2,1) + fy(i,j)*xi
         vir(2,2) = vir(2,2) + fy(i,j)*yi
         vir(2,3) = vir(2,3) + fy(i,j)*zi
         vir(3,1) = vir(3,1) + fz(i,j)*xi
         vir(3,2) = vir(3,2) + fz(i,j)*yi
         vir(3,3) = vir(3,3) + fz(i,j)*zi

!     //   loops of atoms and beads
      end do
      end do

      return
      end





!***********************************************************************
      subroutine finalize_ntchem_MPI
!***********************************************************************

      use ntchem_variables, only : mpi_comm_replica_ntchem

      implicit none

      integer :: ierr

      call mntchem_comm_free

      call MPI_COMM_FREE( mpi_comm_replica_ntchem, ierr )

      return
      end


#else



!***********************************************************************
      subroutine force_ntchem_MPI
!***********************************************************************

      use common_variables, only : myrank

      implicit none

      if ( myrank .eq. 0 ) then

         write( 6, '(a)' ) &
     &      'Error - NTCHEM is not linked.'
         write( 6, '(a)' ) &
     &      'Try to recompile with -Dntchem.'
         write( 6, '(a)' )

      end if

      call error_handling_MPI ( 1, 'subroutine force_ntchem_MPI', 27 )

      return
      end





!***********************************************************************
      subroutine finalize_ntchem_MPI
!***********************************************************************

      return
      end

#endif
