!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, Y. Nagai
!      Last updated:    Jan 23, 2025 by M. Shiga
!      Description:     energy and force by n2p2
!
!///////////////////////////////////////////////////////////////////////



#ifdef n2p2



!***********************************************************************
      subroutine force_n2p2_XMPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, myrank, natom, char_spec, nbead, natom, &
     &   x, y, z, box, au_charge, au_energy, au_length, boxinv

      use n2p2_variables, only : &
     &   istep_train_n2p2, iprint_xsf_n2p2, minxsf_train_n2p2, &
     &   ioption_xsf_n2p2, lstep_train_n2p2, n2p2_box_inv_data, &
     &   istep_save_n2p2, dir_save_n2p2, n2p2_box, n2p2_box_inv, &
     &   cut_n2p2, rcut2_n2p2, n2p2_lenscale, n2p2_enscale, type_j, &
     &   len_scale_n2p2, en_scale_n2p2, n2p2_x_data, n2p2_x, ielement, &
     &   n2p2_y_data, n2p2_y, n2p2_z_data, n2p2_z, n2p2_box_data

      use n2p2_general_interface, only : &
     &   n2p2_interface_setup, n2p2_max_cutoff, n2p2interface_ptr, &
     &   n2p2_get_atom_types

      use iso_c_binding

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   flag for initial visit   */
      integer, save :: iset = 0

!     /*   local variables   */
      integer :: i = 0
      integer :: ierr

!     /*   string for reading n2p2 units   */
      character(len=20) :: line

!     /*   real for holding special units   */
      real(8) :: value

!-----------------------------------------------------------------------
!     /*   read keywords                                              */
!-----------------------------------------------------------------------

!     /*   first visit   */
      if ( iset .eq. 0 ) then

!        /*   step interval of training ann   */
         call read_int1_MPI &
     &      ( istep_train_n2p2, '<istep_train_n2p2>', 18, iounit )

!        /*   step interval for saving the trained networks   */
         call read_int1_MPI &
     &      ( istep_save_n2p2, '<istep_save_n2p2>', 17, iounit )

!        /*   last step of training ann   */
         call read_int1_MPI &
     &      ( lstep_train_n2p2, '<lstep_train_n2p2>', 18, iounit )

!        /*   print interval of xsf files  */
         call read_int1_MPI &
     &      ( iprint_xsf_n2p2, '<iprint_xsf_n2p2>', 17, iounit )

!        /*   print interval of xsf files  */
         call read_int1_MPI &
     &      ( minxsf_train_n2p2, '<minxsf_train_n2p2>', 19, iounit )

!        /*   directory for periodically saving trained networks    */
         call read_char_MPI &
     &      ( dir_save_n2p2, 80, '<dir_save_n2p2>', 15, iounit )

!        /*   option of xsf files  */
         call read_int1_MPI &
     &      ( ioption_xsf_n2p2, '<ioption_xsf_n2p2>', 18, iounit )

!        /*   Read the n2p2 energy unit used   */
         call read_char_MPI &
     &      (line, 20, '<n2p2_en_unit>', 14, iounit )

!        /*   Read the n2p2 energy unit used   */
         read(line, *, iostat=ierr) value

!        /*   Set the n2p2 energy unit used   */
         if( ierr .eq. 0 ) then
            en_scale_n2p2 = .true.
            if ( myrank .eq. 0 ) then
               write( 6, '(a,f16.8,a)' ) &
     &            "N2P2 energy unit is ", value, "."
            end if
            n2p2_enscale = value
         else
            if( line(1:8) .eq. 'HARTREE ' ) then
               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               "N2P2 energy unit is hartree."
               end if
            else if ( line(1:3) .eq. 'EV ' ) then
               en_scale_n2p2 = .true.
               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) "N2P2 energy unit is eV."
               end if
               n2p2_enscale = au_charge / au_energy
            else
               if ( myrank .eq. 0 ) then
                  write( 6, '(a,a)' ) &
     &               "Error - Unknown N2P2 energy unit: ", line
               end if
               call error_handling_MPI ( 1, 'force_n2p2_MPI', 14 )
            end if
         end if

!        /*   Read the n2p2 length unit used   */
         call read_char_MPI &
     &      (line, 20, '<n2p2_len_unit>', 15, iounit )

!        /*   Read the n2p2 length unit used   */
         read(line, *, iostat=ierr) value

!        /*   Set the n2p2 length unit used   */
         if( ierr .eq. 0 ) then
            len_scale_n2p2 = .true.
            if ( myrank .eq. 0 ) then
               write( 6, '(a,f16.8,a)' ) &
     &            "N2P2 length unit is ", value, "."
            end if
            n2p2_lenscale = value
         else
            if( line(1:5) .eq. 'BOHR ' ) then
               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               "N2P2 length unit is bohr."
               end if
            else if ( line(1:3) .eq. 'AA ' ) then
               len_scale_n2p2 = .true.
               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               "N2P2 length unit is Aangstroms."
               end if
               n2p2_lenscale = ( au_length * 1.d+10 )
            else
               if ( myrank .eq. 0 ) then
                  write( 6, '(a,a)' ) &
     &               "Error - Unknown N2P2 length unit: ", line
               end if
               call error_handling_MPI ( 1, 'force_n2p2_MPI', 14 )
            end if
         end if

!        /*   Blank line   */
         if ( myrank .eq. 0 ) write( 6, '(a)' )

!        /*   Read the n2p2 input and setup   */
         call n2p2_interface_setup(myrank, n2p2interface_ptr)

!        /*   Get the cutoff from N2P2   */
         call n2p2_max_cutoff(n2p2interface_ptr, cut_n2p2)

!        /*   Cutoff used on input to N2P2 is N2P2 internal units   */
         rcut2_n2p2 = cut_n2p2*cut_n2p2

!        /*   For length scaling additional arrays and setup   */
         if( len_scale_n2p2 ) then

            if ( .not. allocated(n2p2_x_data) ) &
     &          allocate( n2p2_x_data(natom, nbead) )
            if ( .not. allocated(n2p2_y_data) ) &
     &          allocate( n2p2_y_data(natom, nbead) )
            if ( .not. allocated(n2p2_z_data) ) &
     &          allocate( n2p2_z_data(natom, nbead) )

            if ( .not. allocated(n2p2_box_data) ) &
     &          allocate( n2p2_box_data(3, 3) )
            if ( .not. allocated(n2p2_box_inv_data) ) &
     &          allocate( n2p2_box_inv_data(3, 3) )

            n2p2_x => n2p2_x_data
            n2p2_y => n2p2_y_data
            n2p2_z => n2p2_z_data

            n2p2_box => n2p2_box_data
            n2p2_box_inv => n2p2_box_inv_data

!            write(*,*) rcut2_n2p2

!         /*   Otherwise the coordinate arrays and units from N2P2   */
          else

            n2p2_x => x
            n2p2_y => y
            n2p2_z => z

            n2p2_box => box
            n2p2_box_inv => boxinv

          end if

!        /*   Get atom types from N2P2   */
         if ( allocated(type_j) ) deallocate( type_j )
         allocate( type_j(natom) )

         if ( allocated(ielement) ) deallocate( ielement )
         allocate( ielement(natom) )

         do i = 1, natom
            type_j(i) = trim(char_spec(i))//c_null_char
         end do

         call n2p2_get_atom_types(n2p2interface_ptr, &
     &      c_loc(type_j(1)), c_loc(ielement(1)), natom)

!         write(*,*) ielement

!        /*   second visit   */
         iset = 1

!     /*   first visit   */
      end if

!-----------------------------------------------------------------------
!     /*   start predict                                              */
!-----------------------------------------------------------------------

      call predict_n2p2_XMPI

      return

!***********************************************************************
      end subroutine force_n2p2_XMPI
!***********************************************************************



#else



!***********************************************************************
      subroutine force_n2p2_XMPI
!***********************************************************************

      use common_variables, only : myrank

      implicit none

      if ( myrank .eq. 0 ) then

         write( 6, '(a)' ) &
     &      'Error - n2p2 (parallel) is not linked.'
         write( 6, '(a)' ) &
     &      'Try to compile with -Dn2p2 option.'
         write( 6, '(a)' )

      end if

      call error_handling_MPI ( 1, 'subroutine force_n2p2_XMPI', 26 )

      return

!***********************************************************************
      end subroutine force_n2p2_XMPI
!***********************************************************************



#endif
