!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, Y. Nagai
!      Last updated:    Jan 23, 2025 by M. Shiga
!      Description:     energy and force by n2p2
!
!///////////////////////////////////////////////////////////////////////



#ifdef n2p2



!***********************************************************************
      module n2p2_train_interface
!***********************************************************************

      use iso_c_binding
      implicit none

      interface
         function n2p2_atomic_number(symbol) &
     &      bind(c, name="n2p2_atomic_number")
            import c_size_t
            import c_char
            character(1, kind=c_char), intent(in) :: symbol(*)
            integer(c_size_t) :: n2p2_atomic_number
         end function

         subroutine generate_sub_n2p2(mpi_comm) &
     &      bind(c, name="generate_sub_n2p2")
            import c_int
            integer(c_int), intent(in) :: mpi_comm
         end subroutine generate_sub_n2p2

         subroutine train_sub_n2p2(mpi_comm) &
     &      bind(c, name="train_sub_n2p2")
            import c_int
            integer(c_int), intent(in) :: mpi_comm
         end subroutine train_sub_n2p2
      end interface

!***********************************************************************
      end module n2p2_train_interface
!***********************************************************************



!***********************************************************************
      module n2p2_general_interface
!***********************************************************************

      use iso_c_binding
      implicit none

!     /* pointer to the N2P2 structure */
      type(c_ptr) :: n2p2interface_ptr = c_null_ptr

      interface

         subroutine n2p2_max_cutoff(obj_ptr, max_cutoff) &
     &      bind(c, name='n2p2_pimd_max_cutoff')
            import c_double
            import c_ptr
            type(c_ptr) , intent(in) :: obj_ptr
            real(c_double), intent(out) :: max_cutoff
         end subroutine n2p2_max_cutoff

         subroutine n2p2_interface_setup( &
      &     rank , obj_ptr) &
      &     bind(c, name='n2p2_pimd_interfacesetup')
            import c_int
            import c_ptr
            integer(c_int), intent(in) :: rank
            type(c_ptr), intent(out) :: obj_ptr
         end subroutine n2p2_interface_setup

         subroutine n2p2_pimd_closeinterface(&
      &     obj_ptr) &
      &     bind(c, name='n2p2_pimd_closeinterface')
            import c_ptr
            type(c_ptr), intent(inout) :: obj_ptr
         end subroutine n2p2_pimd_closeinterface

         subroutine n2p2_get_atom_types( &
     &      obj_ptr, types, ielement, n_at) &
     &      bind(c, name='n2p2_get_atom_types')
            import c_ptr
            import c_int
            type(c_ptr)   , intent(in)  :: obj_ptr
            type(c_ptr)   , value       :: types
            type(c_ptr)   , value       :: ielement
            integer(c_int), intent(in)  :: n_at
         end subroutine n2p2_get_atom_types

      end interface

!***********************************************************************
      end module n2p2_general_interface
!***********************************************************************



!***********************************************************************
      module n2p2_variables
!***********************************************************************

!     /*   use c language   */
      use iso_c_binding

!     /*   number of types   */
      integer :: ntype_n2p2

!     /*   types   */
      character(len=4), dimension(:), allocatable :: type_n2p2

!     /*   networks   */
      character(len=80), dimension(:), allocatable :: network_n2p2

!     /*   interval of training   */
      integer :: istep_train_n2p2

!     /*   interval for saving the trained networks    */
      integer :: istep_save_n2p2

!     /*   last step of training   */
      integer :: lstep_train_n2p2

!     /*   print interval of xsf trajectory   */
      integer :: iprint_xsf_n2p2

!     /*   flag for the training of n2p2   */
      integer :: jflag_n2p2 = 0

!     /*   number of xsf files for training   */
      integer :: nxsf_train_n2p2

!     /*   minimum number of xsf files required for training   */
      integer :: minxsf_train_n2p2

!     /*   option of xsf files - 0: accepted, 1: accepted/rejected   */
      integer :: ioption_xsf_n2p2 = 0

!     /*   option of scaling coordinates for N2P2  */
      logical :: len_scale_n2p2 = .false. 
      logical :: en_scale_n2p2 = .false. 

!     /*   directory name for saving trained networks periodically   */
      character(len=80) :: dir_save_n2p2

!     /*   atomic coordinates   */
      real(c_double), dimension(3) :: coo_i
      real(c_double), dimension(:,:), allocatable, target :: coo_j
      real(c_double), dimension(:), allocatable, target :: r2_j

!     /*   atomic indicies   */
      integer(c_int) :: idx_i
      integer(c_int), allocatable, target :: idx_j(:)

!     /*   atomic kinds   */
      character(4, kind=c_char) :: type_i
      character(4, kind=c_char), allocatable, target :: type_j(:)
      integer(c_int), allocatable, target :: ielement(:)
      integer(c_int), allocatable, target :: neighbor_e(:)

!     /* transfer vector sizes, should be max(number of neighbours) */

      integer :: transfer_size = 0 

!     /*   print interval of heat flux trajectory   */
      integer :: iprint_hfx_n2p2

!     /*   energy   */
      real(c_double) :: e_i

!     /*   force   */
      real(c_double), dimension(:), allocatable, target :: fx_i
      real(c_double), dimension(:), allocatable, target :: fy_i
      real(c_double), dimension(:), allocatable, target :: fz_i

!     /*   virial   */
      real(c_double), dimension(:,:), allocatable, target :: vir_i

!     /*   Storage and points for coordinates for length scaling  */
      real(8), dimension(:,:), allocatable, target :: n2p2_x_data
      real(8), dimension(:,:), allocatable, target :: n2p2_y_data
      real(8), dimension(:,:), allocatable, target :: n2p2_z_data
      real(8), dimension(:,:), allocatable, target :: n2p2_box_data
      real(8), dimension(:,:), allocatable, target :: n2p2_box_inv_data

      real(8), dimension(:,:), pointer :: n2p2_x
      real(8), dimension(:,:), pointer :: n2p2_y
      real(8), dimension(:,:), pointer :: n2p2_z
      real(8), dimension(:,:), pointer :: n2p2_box
      real(8), dimension(:,:), pointer :: n2p2_box_inv

!     /*   heat flux   */
      real(8), dimension(:), allocatable :: hfx, hfy, hfz

!     /*   initialize flag   */
      integer,save::iset_predict = 0

!     /*   variables for the cutoff of neighbour list */
      real(c_double) :: cut_n2p2
      real(8) :: cut_skin_n2p2, rcut2_n2p2, skin_n2p2

!     /*   scaling variables for length and energy scaling   */
      real(8) :: n2p2_lenscale = 1.0 
      real(8) :: n2p2_enscale = 1.0

!***********************************************************************
      end module n2p2_variables
!***********************************************************************





!***********************************************************************
      subroutine force_n2p2_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, myrank, natom, char_spec, nbead, natom, &
     &   x, y, z, box, au_charge, au_energy, au_length, boxinv

      use n2p2_variables, only : &
     &   istep_train_n2p2, iprint_xsf_n2p2, &
     &   minxsf_train_n2p2, &
     &   ioption_xsf_n2p2, lstep_train_n2p2, &
     &   istep_save_n2p2, dir_save_n2p2, &
     &   cut_n2p2, cut_skin_n2p2, rcut2_n2p2, skin_n2p2, &
     &   n2p2_lenscale, n2p2_enscale, type_j, ielement, &
     &   len_scale_n2p2, en_scale_n2p2, n2p2_x_data, n2p2_x, &
     &   n2p2_y_data, n2p2_y, n2p2_z_data, n2p2_z, n2p2_box_data, &
     &   n2p2_box, n2p2_box_inv, n2p2_box_inv_data

      use n2p2_general_interface, only : &
     &   n2p2_interface_setup, n2p2_max_cutoff, n2p2interface_ptr, &
     &   n2p2_get_atom_types

      use iso_c_binding

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   flag for initial visit   */
      integer, save :: iset = 0

!     /* local variables */
      integer :: i = 0
      integer :: iostat

!     /* string for reading n2p2 units */
      character(len=20) :: line

!     /* real for holding special units */
      real(8) :: value 
      
!-----------------------------------------------------------------------
!     /*   read keywords                                              */
!-----------------------------------------------------------------------

!     /*   first visit   */
      if ( iset .eq. 0 ) then

!        /*   step interval of training ann   */
         call read_int1_MPI &
     &      ( istep_train_n2p2, '<istep_train_n2p2>', 18, iounit )

!        /*   step interval for saving the trained networks   */
         call read_int1_MPI &
     &      ( istep_save_n2p2, '<istep_save_n2p2>', 17, iounit )

!        /*   last step of training ann   */
         call read_int1_MPI &
     &      ( lstep_train_n2p2, '<lstep_train_n2p2>', 18, iounit )

!        /*   print interval of xsf files  */
         call read_int1_MPI &
     &      ( iprint_xsf_n2p2, '<iprint_xsf_n2p2>', 17, iounit )

!        /*   print interval of xsf files  */
         call read_int1_MPI &
     &      ( minxsf_train_n2p2, '<minxsf_train_n2p2>', 19, iounit )

!        /*   directory for periodically saving trained networks    */
         call read_char_MPI &
     &      ( dir_save_n2p2, 80, '<dir_save_n2p2>', 15, iounit )

!        /*   option of xsf files  */
         call read_int1_MPI &
     &      ( ioption_xsf_n2p2, '<ioption_xsf_n2p2>', 18, iounit )

!        /*   skin value   */
         call read_real1_MPI &
     &      ( skin_n2p2 , '<neighbor_list_skin>', 20, iounit )

!        /*   Read the n2p2 energy unit used   */
         call read_char_MPI &
     &      (line, 20, '<n2p2_en_unit>', 14, iounit )

         read(line, *, IOSTAT=iostat) value
         if( iostat .eq. 0 ) then
            en_scale_n2p2 = .true.
            if ( myrank .eq. 0 ) then
               write( 6, '(a,f16.8,a)' ) &
     &            "N2P2 energy unit is ", value, "."
            end if
            n2p2_enscale = value
         else
            if( line(1:8) .eq. 'HARTREE ' ) then
               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               "N2P2 energy unit is hartree." 
               end if
            else if ( line(1:3) .eq. 'EV ' ) then
               en_scale_n2p2 = .true.
               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) "N2P2 energy unit is eV." 
               end if
               n2p2_enscale = au_charge / au_energy
            else
               if ( myrank .eq. 0 ) then
                  write( 6, '(a,a)' ) &
     &               "Error - Unknown N2P2 energy unit: ", line
               end if 
               call error_handling_MPI ( 1, 'force_n2p2_MPI', 14 )
            end if
         end if

!        /*   Read the n2p2 length unit used   */
         call read_char_MPI &
     &      (line, 20, '<n2p2_len_unit>', 15, iounit )

         read(line, *, IOSTAT=iostat) value
         if( iostat .eq. 0 ) then
            len_scale_n2p2 = .true.
            if ( myrank .eq. 0 ) then
               write( 6, '(a,f16.8,a)' ) &
     &            "N2P2 length unit is ", value, "."
            end if
            n2p2_lenscale = value
         else
            if( line(1:5) .eq. 'BOHR ' ) then
               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               "N2P2 length unit is bohr."
               end if
            else if ( line(1:3) .eq. 'AA ' ) then
               len_scale_n2p2 = .true.
               if ( myrank .eq. 0 ) then
                  write( 6, '(a)' ) &
     &               "N2P2 length unit is Aangstroms." 
               end if
               n2p2_lenscale = ( au_length * 1.d+10 )
            else
               if ( myrank .eq. 0 ) then
                  write( 6, '(a,a)' ) &
     &               "Error - Unknown N2P2 length unit: ", line
               end if 
               call error_handling_MPI ( 1, 'force_n2p2_MPI', 14 )
            end if
         end if

!        /*   Blank line   */
         if ( myrank .eq. 0 ) write( 6, '(a)' )

!        /*   Read the n2p2 input and setup   */
         call n2p2_interface_setup(myrank, n2p2interface_ptr)

!        /*   Get the cutoff from N2P2   */
         call n2p2_max_cutoff(n2p2interface_ptr, cut_n2p2)

!        /*   Cutoff used on input to N2P2 is N2P2 internal units   */
         rcut2_n2p2 = cut_n2p2*cut_n2p2

!        /*   Cutoff used for the neighbour list is in au always   */
         cut_skin_n2p2 = cut_n2p2/n2p2_lenscale + skin_n2p2 

!        /*   For length scaling additional arrays and setup   */
         if( len_scale_n2p2 ) then

            if ( .not. allocated(n2p2_x_data) ) &
    &          allocate( n2p2_x_data(natom, nbead) )
            if ( .not. allocated(n2p2_y_data) ) &
    &          allocate( n2p2_y_data(natom, nbead) )
            if ( .not. allocated(n2p2_z_data) ) &
    &          allocate( n2p2_z_data(natom, nbead) )

            if ( .not. allocated(n2p2_box_data) ) &
    &          allocate( n2p2_box_data(3, 3) )
            if ( .not. allocated(n2p2_box_inv_data) ) &
    &          allocate( n2p2_box_inv_data(3, 3) )

            n2p2_x => n2p2_x_data
            n2p2_y => n2p2_y_data
            n2p2_z => n2p2_z_data

            n2p2_box => n2p2_box_data
            n2p2_box_inv => n2p2_box_inv_data

            write(*,*) cut_skin_n2p2
            write(*,*) rcut2_n2p2

!         /*   Otherwise the coordinate arrays and units from N2P2   */
          else

            n2p2_x => x
            n2p2_y => y
            n2p2_z => z

            n2p2_box => box
            n2p2_box_inv => boxinv

          end if

!         write(*,*) &
!            "Max cutoff from N2P2 neighbor list : ", cut_skin_n2p2

!        /*   Get atom types from N2P2   */
         if ( allocated(type_j) ) deallocate( type_j )
         allocate( type_j(natom) )

         if ( allocated(ielement) ) deallocate( ielement )
         allocate( ielement(natom) ) 

         do i = 1, natom
            type_j(i) = trim(char_spec(i))//c_null_char
         end do

         call n2p2_get_atom_types(n2p2interface_ptr, &
     &      c_loc(type_j(1)), c_loc(ielement(1)), natom)

!         write(*,*) ielement

!        /*   second visit   */
         iset = 1

!     /*   first visit   */
      end if

!     /*   start predict   */
      call predict_n2p2_MPI

      return

!***********************************************************************
      end subroutine force_n2p2_MPI
!***********************************************************************





!***********************************************************************
      subroutine analysis_n2p2_MPI( ioption )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, au_charge, au_energy, au_length, box, char_spec, &
     &   pot, fx, fy, fz, istep, natom, nbead, iounit, &
     &   iboundary, istep, istep_start, method, myrank, nprocs

      use n2p2_variables, only : &
     &   iprint_xsf_n2p2, nxsf_train_n2p2, minxsf_train_n2p2, &
     &   jflag_n2p2, istep_train_n2p2, &
     &   ioption_xsf_n2p2, lstep_train_n2p2, &
     &   istep_save_n2p2, dir_save_n2p2

      use hmc_variables, only : &
     &   x_hmc_last, y_hmc_last, z_hmc_last, pot_hmc_last, &
     &   fx_hmc_last, fy_hmc_last, fz_hmc_last, box_hmc_last

      use rehmc_variables, only : &
     &   x_rehmc_last, y_rehmc_last, z_rehmc_last, pot_rehmc_last, &
     &   fx_rehmc_last, fy_rehmc_last, fz_rehmc_last, box_rehmc_last

      use dual_variables, only : justtrained_dual

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: i, j, ierr, kstep_n2p2, itest, ioption

!     /*   integers   */
      integer :: itssize_n2p2 = 1000

!     /*   real numbers   */
      real(8) :: xa, ya, za, ax, ay, az, bx, by, bz, cx, cy, cz

!     /*   real numbers   */
      real(8) :: const_1, const_2, const_3

!     /*   high level potential and force   */
      real(8) :: pot_high_j, fxj_high, fyj_high, fzj_high

!     /*   low level potential and force   */
      real(8) :: pot_low_j, fxj_low, fyj_low, fzj_low

!     /*   bead number   */
      character(len=8) :: char_num

!     /*   directory name   */
      character(len=10) :: char_dir = 'structures'

!     /*   step number   */
      integer, save :: istep_n2p2 = 0

!     /*   flag for initial visit   */
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( istep .eq. 0 ) then
         continue
      else if ( ioption .ne. 2 ) then
         return
      end if

!-----------------------------------------------------------------------
!     /*   read keywords                                              */
!
!     jflag_aenet=1 for inappropriate number of model files in aenet.
!     If jflag_aenet=1, generate & train for all steps.
!     Warning: in n2p2 jflag_n2p2 is always set to 0.
!-----------------------------------------------------------------------

!     /*   first visit   */
      if ( iset .eq. 0 ) then

!        /*   step interval of training ann   */
         call read_int1_MPI &
     &      ( istep_train_n2p2, '<istep_train_n2p2>', 18, iounit )

!        /*   return by option   */
         if ( istep_train_n2p2 .le. 0 ) then
            iset = 1
            return
         end if

!        /*   step interval for saving the trained networks   */
         call read_int1_MPI &
     &      ( istep_save_n2p2, '<istep_save_n2p2>', 17, iounit )

!        /*   last step of training ann   */
         call read_int1_MPI &
     &      ( lstep_train_n2p2, '<lstep_train_n2p2>', 18, iounit )

!        /*   print interval of xsf files  */
         call read_int1_MPI &
     &      ( iprint_xsf_n2p2, '<iprint_xsf_n2p2>', 17, iounit )

!        /*   print interval of xsf files  */
         call read_int1_MPI &
     &      ( minxsf_train_n2p2, '<minxsf_train_n2p2>', 19, iounit )

!        /*   directory for periodically saving trained networks    */
         call read_char_MPI &
     &      ( dir_save_n2p2, 80, '<dir_save_n2p2>', 15, iounit )

!        /*   option of xsf files  */
         call read_int1_MPI &
     &      ( ioption_xsf_n2p2, '<ioption_xsf_n2p2>', 18, iounit )

!        /*   reset flag   */
         ierr = 0

!        /*   master process   */
         if ( myrank .eq. 0 ) then

!           /*   make directory   */
            call system( 'mkdir -p ' // trim(adjustl(char_dir)) )

!        /*   master process   */
         end if

!        /*   wait   */
         call my_mpi_bcast_int_0( ierr )

!        /*   error message   */
         if ( ( myrank .eq. 0 ) .and. ( ierr .ne. 0 ) ) then
            write( 6, '(a)' )
            write( 6, '(a)' ) 'Error - directory ' // &
     &                         trim(adjustl(char_dir)) // ' not found.'
            write( 6, '(a)' )
         end if

!        /*   stop on error   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine analysis_n2p2_MPI', 28 )

!        /*   reset flag   */
         itest = 0

!        /*   master rank only   */
         if ( myrank .eq. 0 ) then

!           /*   test if restart file exists */
            call testfile ( 'n2p2.ini', 8, itest, iounit )

!           /*   on error   */
            if ( itest .eq. 1 ) then
               istep_n2p2 = 0
               ierr = 0
!           /*   read restart file   */
            else
               open( iounit, file = 'n2p2.ini' )
               read( iounit, *, iostat=ierr ) istep_n2p2
               close( iounit )
            end if

!        /*   master rank only   */
         end if

!        /*   communicate   */
         call my_mpi_bcast_int_0( ierr )

!        /*   error message   */
         if ( ( myrank .eq. 0 ) .and. ( ierr .ne. 0 ) ) then
            write( 6, '(a)' )
            write( 6, '(a)' ) 'Error - n2p2.ini read incorrectly.'
            write( 6, '(a)' )
         end if

!        /*   stop on error   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine analysis_n2p2_MPI', 28 )

!        /*   communicate   */
         call my_mpi_bcast_int_0( istep_n2p2 )

!        /*   reset flag   */
!         itest = 0

!        /*   check existence of ann potentials  */
!         if ( myrank .eq. 0 ) then
!            j = 0
!            do i = 1, ntype_n2p2
!               call testfile ( network_n2p2(i), len(network_n2p2(i)),
!     &                         itest, iounit )
!               if ( itest .eq. 0 ) j = j + 1
!            end do
!         end if
!         call my_mpi_bcast_int_0( j )

!        /*   if they do not exist, generate and train   */
!         if ( j .ne. ntype_n2p2 ) jflag_n2p2 = 1

!        /*   set complete   */
         iset = 1

!     /*   initial visit   */
      end if

!-----------------------------------------------------------------------
!     /*   return by option                                           */
!-----------------------------------------------------------------------

      if ( istep_train_n2p2 .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   print xsf file                                             */
!-----------------------------------------------------------------------

!     /*   every iprint_xsf_n2p2 steps   */
      if ( ( iprint_xsf_n2p2 .gt. 0 ) .and. &
     &     ( mod(istep,iprint_xsf_n2p2) .eq. 0 ) ) then

!        /*   conversion factor   */
         const_1 = au_length * 1.d+10

!        /*   conversion factor   */
         const_2 =  au_charge / au_energy

!        /*   conversion factor   */
         const_3 =  1.d0 / const_2 / const_1

!        /*   loop of beads   */
         do j = 1, nbead

!           /*   skip if this is not my bead   */
            if ( mod( j-1, nprocs ) .ne. myrank ) cycle

!           /*   counter   */
            kstep_n2p2 = istep_n2p2 + j

!           /*   counter   */
            call int8_to_char( kstep_n2p2, char_num )

!           /*   open sample file   */
            open( iounit, &
     &         file = trim(adjustl(char_dir)) &
     &         // '/' // char_num // '.xsf' )

!           /*   for periodic boundary condition   */
            if ( (iboundary .eq. 1) .or. (iboundary .eq. 2) ) then

!              /*   potential energy in eV   */
               if ( istep .eq. istep_start ) then
                  pot_high_j = pot(j) / const_2
                  pot_low_j = pot(j) / const_2
               else if ( ioption_xsf_n2p2 .eq. 0 ) then
                  pot_high_j = pot(j) / const_2
                  pot_low_j = pot(j) / const_2
               else if ( method(1:6) .eq. 'PIHMC ' ) then
                  pot_high_j = pot_hmc_last(j) / const_2
                  pot_low_j = pot_hmc_last(j) / const_2
               else if ( method(1:6) .eq. 'REHMC ' ) then
                  pot_high_j = pot_rehmc_last(j) / const_2
                  pot_low_j = pot_rehmc_last(j) / const_2
               else
                  pot_high_j = pot(j) / const_2
                  pot_low_j = pot(j) / const_2
               end if

               write( iounit, '(a,f24.12,a)' ) &
     &            '# total energy = ', pot_high_j, ' eV'

               write( iounit, '(a)' ) 'CRYSTAL'
               write( iounit, '(a)' ) 'PRIMVEC'

!              /*   lattice vectors   */
               if ( istep .eq. istep_start ) then
                  ax = box(1,1) * const_1
                  ay = box(2,1) * const_1
                  az = box(3,1) * const_1
                  bx = box(1,2) * const_1
                  by = box(2,2) * const_1
                  bz = box(3,2) * const_1
                  cx = box(1,3) * const_1
                  cy = box(2,3) * const_1
                  cz = box(3,3) * const_1
               else if ( ioption_xsf_n2p2 .eq. 0 ) then
                  ax = box(1,1) * const_1
                  ay = box(2,1) * const_1
                  az = box(3,1) * const_1
                  bx = box(1,2) * const_1
                  by = box(2,2) * const_1
                  bz = box(3,2) * const_1
                  cx = box(1,3) * const_1
                  cy = box(2,3) * const_1
                  cz = box(3,3) * const_1
               else if ( method(1:6) .eq. 'PIHMC ' ) then
                  ax = box_hmc_last(1,1) * const_1
                  ay = box_hmc_last(2,1) * const_1
                  az = box_hmc_last(3,1) * const_1
                  bx = box_hmc_last(1,2) * const_1
                  by = box_hmc_last(2,2) * const_1
                  bz = box_hmc_last(3,2) * const_1
                  cx = box_hmc_last(1,3) * const_1
                  cy = box_hmc_last(2,3) * const_1
                  cz = box_hmc_last(3,3) * const_1
               else if ( method(1:6) .eq. 'REHMC ' ) then
                  ax = box_rehmc_last(1,1,j) * const_1
                  ay = box_rehmc_last(2,1,j) * const_1
                  az = box_rehmc_last(3,1,j) * const_1
                  bx = box_rehmc_last(1,2,j) * const_1
                  by = box_rehmc_last(2,2,j) * const_1
                  bz = box_rehmc_last(3,2,j) * const_1
                  cx = box_rehmc_last(1,3,j) * const_1
                  cy = box_rehmc_last(2,3,j) * const_1
                  cz = box_rehmc_last(3,3,j) * const_1
               else
                  ax = box(1,1) * const_1
                  ay = box(2,1) * const_1
                  az = box(3,1) * const_1
                  bx = box(1,2) * const_1
                  by = box(2,2) * const_1
                  bz = box(3,2) * const_1
                  cx = box(1,3) * const_1
                  cy = box(2,3) * const_1
                  cz = box(3,3) * const_1
               end if

!              /*   write three lines   */
               write( iounit, '(3f16.8)' ) ax, ay, az
               write( iounit, '(3f16.8)' ) bx, by, bz
               write( iounit, '(3f16.8)' ) cx, cy, cz

!              /*   write one line   */
               write( iounit, '(a)' ) 'PRIMCOORD'

!              /*   write one line   */
               write( iounit, '(i8,i2)' ) natom, 1

!           /*   for free boundary condition   */
            else if ( iboundary .eq. 0 ) then

!              /*   potential energy in eV   */
               if ( istep .eq. istep_start ) then
                  pot_high_j = pot(j) / const_2
                  pot_low_j = pot(j) / const_2
               else if ( ioption_xsf_n2p2 .eq. 0 ) then
                  pot_high_j = pot(j) / const_2
                  pot_low_j = pot(j) / const_2
               else if ( method(1:6) .eq. 'PIHMC ' ) then
                  pot_high_j = pot_hmc_last(j) / const_2
                  pot_low_j = pot_hmc_last(j) / const_2
               else if ( method(1:6) .eq. 'REHMC ' ) then
                  pot_high_j = pot_rehmc_last(j) / const_2
                  pot_low_j = pot_rehmc_last(j) / const_2
               else
                  pot_high_j = pot(j) / const_2
                  pot_low_j = pot(j) / const_2
               end if

               write( iounit, '(a,f24.12,a)' ) &
     &            '# total energy = ', pot_high_j, ' eV'

!              /*   write one line   */
               write( iounit, '(a)' ) 'ATOMS'

!           /*   for periodic boundary condition   */
            end if

!           /*   loop of atoms   */
            do i = 1, natom

!              /*   geometry in angstroms   */
               if ( istep .eq. istep_start ) then
                  xa = x(i,j) * const_1
                  ya = y(i,j) * const_1
                  za = z(i,j) * const_1
               else if ( ioption_xsf_n2p2 .eq. 0 ) then
                  xa = x(i,j) * const_1
                  ya = y(i,j) * const_1
                  za = z(i,j) * const_1
               else if ( method(1:6) .eq. 'PIHMC ' ) then
                  xa = x_hmc_last(i,j) * const_1
                  ya = y_hmc_last(i,j) * const_1
                  za = z_hmc_last(i,j) * const_1
               else if ( method(1:6) .eq. 'REHMC ' ) then
                  xa = x_rehmc_last(i,j) * const_1
                  ya = y_rehmc_last(i,j) * const_1
                  za = z_rehmc_last(i,j) * const_1
               else
                  xa = x(i,j) * const_1
                  ya = y(i,j) * const_1
                  za = z(i,j) * const_1
               end if

!              /*   geometry in eV per angstrom   */
               if ( istep .eq. istep_start ) then
                  fxj_high = fx(i,j) * const_3 * dble(nbead)
                  fyj_high = fy(i,j) * const_3 * dble(nbead)
                  fzj_high = fz(i,j) * const_3 * dble(nbead)
                  fxj_low = fx(i,j) * const_3 * dble(nbead)
                  fyj_low = fy(i,j) * const_3 * dble(nbead)
                  fzj_low = fz(i,j) * const_3 * dble(nbead)
               else if ( ioption_xsf_n2p2 .eq. 0 ) then
                  fxj_high = fx(i,j) * const_3 * dble(nbead)
                  fyj_high = fy(i,j) * const_3 * dble(nbead)
                  fzj_high = fz(i,j) * const_3 * dble(nbead)
                  fxj_low = fx(i,j) * const_3 * dble(nbead)
                  fyj_low = fy(i,j) * const_3 * dble(nbead)
                  fzj_low = fz(i,j) * const_3 * dble(nbead)
               else if ( method(1:6) .eq. 'PIHMC ' ) then
                  fxj_high = fx_hmc_last(i,j) * const_3 * dble(nbead)
                  fyj_high = fy_hmc_last(i,j) * const_3 * dble(nbead)
                  fzj_high = fz_hmc_last(i,j) * const_3 * dble(nbead)
                  fxj_low = fx_hmc_last(i,j) * const_3 * dble(nbead)
                  fyj_low = fy_hmc_last(i,j) * const_3 * dble(nbead)
                  fzj_low = fz_hmc_last(i,j) * const_3 * dble(nbead)
               else if ( method(1:6) .eq. 'REHMC ' ) then
                  fxj_high = fx_rehmc_last(i,j) * const_3 * dble(nbead)
                  fyj_high = fy_rehmc_last(i,j) * const_3 * dble(nbead)
                  fzj_high = fz_rehmc_last(i,j) * const_3 * dble(nbead)
                  fxj_low = fx_rehmc_last(i,j) * const_3 * dble(nbead)
                  fyj_low = fy_rehmc_last(i,j) * const_3 * dble(nbead)
                  fzj_low = fz_rehmc_last(i,j) * const_3 * dble(nbead)
               else
                  fxj_high = fx(i,j) * const_3 * dble(nbead)
                  fyj_high = fy(i,j) * const_3 * dble(nbead)
                  fzj_high = fz(i,j) * const_3 * dble(nbead)
                  fxj_low = fx(i,j) * const_3 * dble(nbead)
                  fyj_low = fy(i,j) * const_3 * dble(nbead)
                  fzj_low = fz(i,j) * const_3 * dble(nbead)
               end if

!              /*   write one line   */
               write( iounit, '(a4,6f16.8)' ) &
     &            char_spec(i)(1:4), xa, ya, za, &
     &            fxj_high, fyj_high, fzj_high

!           /*   loop of atoms   */
            end do

!           /*   close input file   */
            close( iounit )

!        /*   loop of beads   */
         end do

!        /*   update counter   */
         istep_n2p2 = istep_n2p2 + nbead

!        /*   wait   */
         call my_mpi_barrier

!-----------------------------------------------------------------------
!     /*   restart file                                               */
!-----------------------------------------------------------------------

!        /*   master rank only   */
         if ( myrank .eq. 0 ) then

!           /*   write step  */
            open( iounit, file = 'n2p2.ini' )
            write( iounit, '(i8)' ) istep_n2p2
            close( iounit )

!        /*   master rank only   */
         end if

!     /*   every iprint_xsf_n2p2 steps   */
      end if

!-----------------------------------------------------------------------
!     /*   count number of xsf files                                  */
!-----------------------------------------------------------------------

!     /*   conditions to generate and train   */
      if ( ( istep_train_n2p2 .gt. 0 ) .and. &
     &     ( istep .le. lstep_train_n2p2 ) .and. &
     &     ( mod(istep,istep_train_n2p2) .eq. 0 ) ) then

!        /*   wait for all processes   */
         call my_mpi_barrier

!        /*   add number of structure files   */
         if ( myrank .eq. 0 ) then
            call system &
     &         ( 'ls -U1 ' // trim(adjustl(char_dir)) // &
     &           '/*.xsf > test.1.out; ' &
     &        // 'cat test.1.out | wc -l > test.2.out' )
         end if

!        /*   wait for all processes   */
         call my_mpi_barrier

!        /*   read number of xsf files   */
         if ( myrank .eq. 0 ) then
            open( iounit, file = 'test.2.out' )
            read( iounit, *, iostat=ierr ) nxsf_train_n2p2
            close( iounit )
         end if

!        /*   communicate   */
         call my_mpi_bcast_int_0( ierr )

!        /*   error message   */
         if ( ( myrank .eq. 0 ) .and. ( ierr .ne. 0 ) ) then
            write( 6, '(a)' )
            write( 6, '(a)' ) 'Error - Xsf files could not be found.'
            write( 6, '(a)' )
         end if

!        /*   stop on error   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine analysis_n2p2_MPI', 28 )

!        /*   wait for all processes   */
         call my_mpi_barrier

!        /*   remove file   */
         if ( myrank .eq. 0 ) &
     &      call system( 'rm -f test.1.out test.2.out 2> /dev/null' )

!        /*   wait for all processes   */
         call my_mpi_barrier

!     /*   conditions to generate and train   */
      end if

!     /*   number of xsf files   */
      call my_mpi_bcast_int_0( nxsf_train_n2p2 )

!-----------------------------------------------------------------------
!     /*   generate and train                                         */
!-----------------------------------------------------------------------

!     /*   conditions to generate and train   */
      if ( ( istep_train_n2p2 .gt. 0 ) .and. &
     &     ( ( ( istep .le. lstep_train_n2p2 ) .and. &
     &         ( mod(istep,istep_train_n2p2) .eq. 0 ) &
     &      .and. (istep .ne. 0) ) .or. &
     &         ( jflag_n2p2 .eq. 1 ) ) .and. &
     &     ( nxsf_train_n2p2 .ge. minxsf_train_n2p2 ) ) then

!----------------------------------------------------------------------

!        /*   wait for all processes   */
         call my_mpi_barrier

!        /*   initial value   */
         j = 0

!        /*   generate   */
         call generate_n2p2_MPI

!        /*   add number of structure files   */
         if ( myrank .eq. 0 ) then
            call system &
     &         ( 'cat input.data | wc -l > test.2.out 2> /dev/null' )
         end if

!        /*   wait for all processes   */
         call my_mpi_barrier

!        /*   number of files that match   */
         if ( myrank .eq. 0 ) then
            open( iounit, file = 'test.2.out' )
            read( iounit, *, iostat=ierr ) j
            close( iounit )
         end if

!        /*   communicate   */
         call my_mpi_bcast_int_0( ierr )
         call my_mpi_bcast_int_0( j )

!        /*   no error   */
         if ( ( ierr .eq. 0 ) .and. ( j .ge. itssize_n2p2 ) ) then
            if ( myrank .eq. 0 ) then
               write( 6, '(a,i8,a)' ) &
     &            'Train file is found: ', j, ' lines.'
            end if
!        /*   error   */
         else
            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) &
     &            'Train file is missing.'
            end if
            ierr = 1
            call my_mpi_bcast_int_0( ierr )
         end if

!        /*   error message   */
         if ( ( myrank .eq. 0 ) .and. ( ierr .ne. 0 ) ) then
            write( 6, '(a)' )
            write( 6, '(a)' ) 'Error - Train file could not be found.'
            write( 6, '(a)' )
         end if

!        /*   stop on error   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine analysis_n2p2_MPI', 28 )

!        /*   wait for all processes   */
         call my_mpi_barrier

!        /*   remove file   */
         if ( myrank .eq. 0 ) &
     &      call system( 'rm -f test.1.out test.2.out 2> /dev/null' )

!        /*   wait for all processes   */
         call my_mpi_barrier

!----------------------------------------------------------------------

!        /*   train   */
         call train_n2p2_MPI

!        /*   change flag   */
         jflag_n2p2 = 0
         call reset_flag_predict_n2p2_MPI

!        /*   flag just trained   */
         justtrained_dual = 1

!     /*   conditions to generate and train   */
      end if

      return

!***********************************************************************
      end subroutine analysis_n2p2_MPI
!***********************************************************************



!***********************************************************************
      subroutine train_n2p2_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &    iounit, iounit_n2p2, istep, mpi_comm_pimd, myrank, nprocs

      use n2p2_variables, only : &
     &   dir_save_n2p2, istep_save_n2p2, &
     &   ntype_n2p2, type_n2p2, network_n2p2, nxsf_train_n2p2

      use n2p2_train_interface, only : &
     &   n2p2_atomic_number, train_sub_n2p2

      use iso_c_binding

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: i, k, ierr, itest, color
!     /*   tmp communicator if we need to reduce parallelization here */
      integer :: mpi_comm_tmp

!     /*   integers   */
      integer, save :: iset = 0

!     /*   characters   */
      character(len=100) :: char_line
      character(len=80) :: char_word
      character(len=8) :: char_num

!     /*   real numbers  */
      real(8) :: time_started, time_ended

!     /*   mpi   */
      include 'mpif.h'

!-----------------------------------------------------------------------
!     /*   print message                                              */
!-----------------------------------------------------------------------

!     /*   time started   */
      time_started = mpi_wtime()

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

         write( 6, '(a)' )

         write( 6, '(a)' ) &
     &      'N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2' // &
     &      'N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2'

         write( 6, '(a)' ) &
     &      'Train started.'

         write( 6, '(a)' ) &
     &      'N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2' // &
     &      'N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2'
         write( 6, '(a)' )

         write( 6, '(a)' )

!     /*   master rank only   */
      end if

!-----------------------------------------------------------------------
!     /*   read keyword from input.nn                                 */
!-----------------------------------------------------------------------

!     /*   first visit   */
      if ( iset .eq. 0 ) then

!        /*   master rank only   */
         if ( myrank .eq. 0 ) then

!           /*   file open   */
            open( iounit, file = 'input.nn' )

!           /*   read number of types  */
            do
               read( iounit, '(a)', iostat=ierr ) char_line
               if ( ierr .ne. 0) exit
               k = index(char_line(1:18), 'number_of_elements')
               if ( k .ge. 1) read(char_line(19:), *) ntype_n2p2
            end do

!           /*   memory allocation: types   */
            if ( .not. allocated(type_n2p2) ) &
     &         allocate( type_n2p2(ntype_n2p2) )

!           /*   memory allocation: networks   */
            if ( .not. allocated(network_n2p2) ) &
     &         allocate( network_n2p2(ntype_n2p2) )

!           /*   rewind input.nn   */
            rewind( iounit, iostat=ierr )

!           /*   read types   */
            do
               read ( iounit, '(a)', iostat=ierr ) char_line
               if ( ierr .ne. 0) exit
               k = index(char_line(1:8), 'elements')
               if ( k .ge. 1) read(char_line(9:), *) type_n2p2
            end do

!           /*   generate network file name   */
            do i = 1, ntype_n2p2
               write(network_n2p2(i), '(a,i3.3,a)') &
     &            'weights.', &
     &            n2p2_atomic_number( &
     &               trim(adjustl(type_n2p2(i))) // c_null_char), &
     &            '.data'
            end do

!           /*   file close   */
            close( iounit )

!        /*   master rank only   */
         end if

!     /*   first visit   */
      end if

!-----------------------------------------------------------------------
!     /*   make train directory and remove train files                */
!-----------------------------------------------------------------------

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

!        /*   make directory as needed   */
         if ( istep_save_n2p2 .gt. 0) then
            call system( 'mkdir -p ' // trim(adjustl(dir_save_n2p2)) )
         end if

!        /*   remove files   */
         call system( 'rm -f test.data train.data 2> /dev/null' )

!     /*   master rank only   */
      end if

!     /*   wait for all processes to finish   */
      call my_mpi_barrier

!-----------------------------------------------------------------------
!     /*   execute train                                              */
!-----------------------------------------------------------------------

!     /*   If we have fewer structure data than MPI processes we need */
!     /*   to run with fewer MPI processes so make temp comm for that */
      if ( nxsf_train_n2p2 .lt. nprocs ) then

!     /*   Exclude all higher rank processes from training   */
            if ( myrank .lt. nxsf_train_n2p2 ) then
                  color = 1   
            else
                  color = MPI_UNDEFINED 
            endif

            call MPI_Comm_split( MPI_COMM_WORLD, color, myrank, mpi_comm_tmp, ierr )

            if (color /= MPI_UNDEFINED) then
                  call train_sub_n2p2( mpi_comm_tmp )
!     /*   once we are done delete the temp communicator   */
                  call MPI_Comm_free( mpi_comm_tmp, ierr )
            endif
      else
!     /*   execution of train   */
            call train_sub_n2p2( mpi_comm_pimd )
      endif

!     /*   wait for all processes to finish   */
      call my_mpi_barrier

!-----------------------------------------------------------------------
!     /*   copy network file                                          */
!-----------------------------------------------------------------------

!     /*   reset flag   */
      itest = 0

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

!        /*   loop of atomic types   */
         do i = 1, ntype_n2p2

!           /*   main directory file  */
            char_line = trim(adjustl(network_n2p2(i)))

!           /*   check existence of file   */
            call testfile ( char_line, len(char_line), &
     &                      itest, iounit )

!           /*   detect error   */
            if ( itest .ne. 0 ) then
               write( 6, '(a)' )
               write( 6, '(a)' ) &
     &            'Error - trained network not found: ' &
     &            // trim(adjustl(char_line))
               write( 6, '(a)' )
               exit
            else
               write( 6, '(a)' ) &
     &            'trained network overwritten by: ' &
     &            // trim(adjustl(char_line))
            end if

!           /*   If we reach a step where we wish to    */
!           /*   save the trained network we save it.   */
            if ( ( istep_save_n2p2 .gt. 0 ) .and. &
     &           ( mod(istep,istep_save_n2p2) .eq. 0 )) then

               call int8_to_char( istep, char_num )

               char_word = trim(adjustl(dir_save_n2p2)) // '/' // &
     &                     trim(adjustl(network_n2p2(i))) // &
     &                     '_iter_' // trim(adjustl(char_num))

               call system( 'cp ' // char_line // ' ' // char_word )

            end if

!        /*   loop of atomic types   */
         end do

!        /*   save input.nn   */
         if ( ( istep_save_n2p2 .gt. 0 ) .and. &
     &      ( mod(istep,istep_save_n2p2) .eq. 0 )) then

            char_line = 'input.nn'

            call int8_to_char( istep, char_num )

            char_word = trim(adjustl(dir_save_n2p2)) // '/' // &
     &                  'input.nn' // &
     &                  '_iter_' // trim(adjustl(char_num))

            call system( 'cp ' // char_line // ' ' // char_word )

         end if

!        /*   save scaling.data   */
         if ( ( istep_save_n2p2 .gt. 0 ) .and. &
     &      ( mod(istep,istep_save_n2p2) .eq. 0 )) then

            char_line = 'scaling.data'

            call int8_to_char( istep, char_num )

            char_word = trim(adjustl(dir_save_n2p2)) // '/' // &
     &                  'scaling.data' // &
     &                  '_iter_' // trim(adjustl(char_num))

            call system( 'cp ' // char_line // ' ' // char_word )

         end if

!     /*   master rank only   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( itest )

!     /*   error termination   */
      call error_handling_MPI( itest, 'subroutine train_n2p2_MPI', 25 )

!-----------------------------------------------------------------------
!     /*   print message                                              */
!-----------------------------------------------------------------------

!     /*   time started   */
      time_ended = mpi_wtime()

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

         write( 6, '(a)' )

         write( 6, '(a)' ) &
     &      'N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2' // &
     &      'N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2'

         write( 6, '(a,f10.2,a)' ) &
     &      'Train ended in', &
     &      time_ended-time_started, &
     &      ' seconds.'

         write( 6, '(a)' ) &
     &      'N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2' // &
     &      'N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2'

         write( 6, '(a)' )

!     /*   master rank only   */
      end if

      return

!***********************************************************************
      end subroutine train_n2p2_MPI
!***********************************************************************



!***********************************************************************
      subroutine generate_n2p2_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, iounit_n2p2, myrank, mpi_comm_pimd, au_length, &
     &   au_charge, au_energy, nprocs

      use n2p2_variables, only : nxsf_train_n2p2

      use n2p2_train_interface, only : &
     &   generate_sub_n2p2

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: ierr, jerr, i, j, k, color
!     /*   tmp communicator if we need to reduce parallelization here */
      integer :: mpi_comm_tmp

!     /*   characters   */
      character(len=100) :: char_line
      character(len=80) :: char_filename
      character(len=80) :: symbol

!     /*   integers   */
      integer :: iounit_xsf = 1060

!     /*   real numbers   */
      real(8) :: xa, ya, za
      real(8) :: ax, ay, az, bx, by, bz, cx, cy, cz
      real(8) :: fxj_high, fyj_high, fzj_high
      real(8) :: energy, const_1, const_2, const_3

!     /*   real numbers  */
      real(8) :: time_started, time_ended

!     /*   directory name   */
      character(len=10) :: char_dir = 'structures'

!     /*   mpi   */
      include 'mpif.h'

!-----------------------------------------------------------------------
!     /*   print message                                              */
!-----------------------------------------------------------------------

!     /*   time started   */
      time_started = mpi_wtime()

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

         write( 6, '(a)' )

         write( 6, '(a)' ) &
     &      'N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2' // &
     &      'N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2'

         write( 6, '(a)' ) &
     &      'Generate started.'

         write( 6, '(a)' ) &
     &      'N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2' // &
     &      'N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2'

         write( 6, '(a)' )

!     /*   master rank only   */
      end if

!-----------------------------------------------------------------------
!     /*   conversion factors to AU                                   */
!-----------------------------------------------------------------------
!        /*   conversion factor   */
         const_1 = 1.0 / ( au_length * 1.d+10 )

!        /*   conversion factor   */
         const_2 =  au_charge / au_energy

!        /*   conversion factor   */
         const_3 =  const_2 / const_1

!-----------------------------------------------------------------------
!     /*   remove train files                                         */
!-----------------------------------------------------------------------

!     /*   wait for all processes to finish   */
      call my_mpi_barrier

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then
         call system( 'rm -f test.1.out test.2.out 2> /dev/null' )
         call system( 'rm -f input.data scaling.data 2> /dev/null' )
      end if

!     /*   wait for all processes to finish   */
      call my_mpi_barrier

!-----------------------------------------------------------------------
!     /*   genereate input.data from xsf files                        */
!-----------------------------------------------------------------------

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

!        /*   add number of structure files   */
         call system &
!     &      ( 'ls -U16 ' // trim(adjustl(char_dir)) // &
     &      ( 'ls -U1 ' // trim(adjustl(char_dir)) // &
     &        '/*.xsf > test.1.out; ' // &
     &        'cat test.1.out | wc -l > test.2.out' )

!        /*   read number of xsf files   */
         open( iounit, file = 'test.2.out' )
         read( iounit, *, iostat=ierr ) j
         close( iounit )

!        /*   no error   */
         if ( ( ierr .eq. 0 ) .and. ( j .gt. 0 ) ) then
            write( 6, '(a,i8,a)' ) &
     &         'Xsf files are found: ', j, ' files.'
            write( 6, '(a)' )
!        /*   error   */
         else
            ierr = 1
         end if

!        /*   error message   */
         if ( ierr .ne. 0 ) then
            write( 6, '(a)' )
            write( 6, '(a)' ) 'Error - Xsf files could not be found.'
            write( 6, '(a)' )
         end if

!        /*   stop on error   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine analysis_n2p2_MPI', 28 )

!        /*   generate input.data   */
         open( iounit_n2p2, file = 'input.data' )
         open( iounit, file = 'test.1.out' )

         do i = 1, j
            read( iounit, '(a)', iostat=ierr ) char_filename
            open( iounit_xsf, file = char_filename)

!           /*   write begin tag   */
            write( iounit_n2p2, '(a)' ) 'begin'

!           /*   write source filename as comment   */
            write( iounit_n2p2, '(a)' ) &
     &         'comment source_file_name=' // &
     &         trim(adjustl(char_filename))

!           /*   read and write lattice   */
            do
               read( iounit_xsf, '(a)', iostat=jerr) char_line
               if ( jerr .ne. 0) exit
               k = index(char_line(1:7), 'PRIMVEC')
               if ( k .ge. 1) then
                  read( iounit_xsf, *) ax, ay, az
                  read( iounit_xsf, *) bx, by, bz
                  read( iounit_xsf, *) cx, cy, cz

                  ax = ax * const_1
                  ay = ay * const_1
                  az = az * const_1
                  bx = bx * const_1
                  by = by * const_1
                  bz = bz * const_1
                  cx = cx * const_1
                  cy = cy * const_1
                  cz = cz * const_1

                  write( iounit_n2p2, '(a,3f16.8)') &
     &               'lattice', ax, ay, az
                  write( iounit_n2p2, '(a,3f16.8)') &
     &               'lattice', bx, by, bz
                  write( iounit_n2p2, '(a,3f16.8)') &
     &               'lattice', cx, cy, cz
                 exit
               end if
            end do

!           /*   rewind xsf file   */
            rewind( iounit_xsf, iostat=jerr )

!           /*   read and write atom coordinate and force   */
            do
               read( iounit_xsf, '(a)', iostat=jerr) char_line
               if ( jerr .ne. 0) exit
               k = index(char_line(1:9), 'PRIMCOORD')
               if ( k .ge. 1) then
                  read( iounit_xsf, '()')  ! skip a row
                  do
                     read( iounit_xsf, *, iostat=jerr) &
     &                  symbol, xa, ya, za, &
     &                  fxj_high, fyj_high, fzj_high
                        
                     xa = xa * const_1
                     ya = ya * const_1
                     za = za * const_1

                     fxj_high = fxj_high * const_3
                     fyj_high = fyj_high * const_3
                     fzj_high = fzj_high * const_3
                     if ( jerr .ne. 0 ) exit
                     write( iounit_n2p2, '(a,3f16.8,a8,5f16.8)') &
     &                  'atom', xa, ya, za, &
     &                  trim(adjustl(symbol)), 0.0d0, 0.0d0, &
     &                  fxj_high, fyj_high, fzj_high
                  end do
                  exit
               end if
            end do

!           /*   rewind xsf file   */
            rewind( iounit_xsf, iostat=jerr )

!           /*   read and write total energy   */
            do
               read( iounit_xsf, '(a)', iostat=jerr) char_line
               if ( jerr .ne. 0) exit
               k = index(char_line(1:16), '# total energy =')
               if ( k .ge. 1) then
                  read(char_line(17:), *) energy

                  energy = energy * const_2

                  write( iounit_n2p2, '(a,f24.12)') 'energy', energy
                  exit
               end if
            end do

!           /*   write end tag   */
            write( iounit_n2p2, '(a)' ) 'end'

            close( iounit_xsf )
         end do

         close( iounit_n2p2 )
         close( iounit )

!        /*   remove tmp files   */
         call system( 'rm -f test.1.out test.2.out 2> /dev/null' )

!     /*   master rank only   */
      end if

!     /*   wait for all processes to finish   */
      call my_mpi_barrier

!-----------------------------------------------------------------------
!     /*   generate scaling.data                                      */
!-----------------------------------------------------------------------

      if ( nxsf_train_n2p2 .lt. nprocs ) then
            write(*,*) "DOING SPECIAL THING!!!"

!     /*   Exclude all higher rank processes from training   */
            if ( myrank .lt. nxsf_train_n2p2 ) then
                  color = 1   
            else
                  color = MPI_UNDEFINED 
            endif

            call MPI_Comm_split( MPI_COMM_WORLD, color, myrank, mpi_comm_tmp, ierr )

            if (color /= MPI_UNDEFINED) then
                  call generate_sub_n2p2( mpi_comm_tmp )
!     /*   once we are done delete the temp communicator   */
                  call MPI_Comm_free( mpi_comm_tmp, ierr )
            endif
      else
!     /*   execution of train   */
            call generate_sub_n2p2( mpi_comm_pimd )
      endif

!     /*   wait for all processes to finish   */
      call my_mpi_barrier

!-----------------------------------------------------------------------
!     /*   print message                                              */
!-----------------------------------------------------------------------

!     /*   time started   */
      time_ended = mpi_wtime()

!     /*   master rank only   */
      if ( myrank .eq. 0 ) then

         write( 6, '(a)' )

         write( 6, '(a)' ) &
     &      'N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2' // &
     &      'N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2'

         write( 6, '(a,f10.2,a)' ) &
     &      'Generate ended in', &
     &      time_ended - time_started, &
     &      ' seconds.'

         write( 6, '(a)' ) &
     &      'N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2' // &
     &      'N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2P2N2'

         write( 6, '(a)' )

!     /*   master rank only   */
      end if

      return

!***********************************************************************
      end subroutine generate_n2p2_MPI
!***********************************************************************


!***********************************************************************
      subroutine finalize_n2p2_MPI
!***********************************************************************

      implicit none



!***********************************************************************
      end subroutine finalize_n2p2_MPI
!***********************************************************************

#else



!***********************************************************************
      subroutine force_n2p2_MPI
!***********************************************************************

      use common_variables, only : myrank

      implicit none

      if ( myrank .eq. 0 ) then

         write( 6, '(a)' ) &
     &      'Error - n2p2 (parallel) is not linked.'
         write( 6, '(a)' ) &
     &      'Try to compile with -Dn2p2 option.'
         write( 6, '(a)' )

      end if

      call error_handling_MPI ( 1, 'subroutine force_n2p2_MPI', 25 )

      return

!***********************************************************************
      end subroutine force_n2p2_MPI
!***********************************************************************



#endif
