!///////////////////////////////////////////////////////////////////////
!
!      Authors:        Hyukjoon Kwon (bagho27@snu.ac.kr), M. Shiga
!      Last updated:   Jan 05, 2023 by H. Kwon
!      Description:    energy, force, virial from moment tensor
!                      potential This codes was adapted from the MLIP
!                      code (https://mlip.skoltech.ru/)
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine force_mtp
!***********************************************************************
!=======================================================================
!
!   reference [1]:
!     I.S. Novikov, K. Gubaev, E.V. Podryabinkin, A.V. Shapeev,
!     "The MLIP package: Moment tensor potentials with mpi and active
!     learning."
!     Machine Learning: Science and Technology, 2(2):025002, 2020.
!     (doi: 10.1088/2632-2153/abc9fe)
!
!   reference [2]:
!     A.V. Shapeev,
!     "Moment tensor potentials: A class of systematically improvable
!     interatomic potentials,"
!     Multiscale Modeling and Simululation, volume 14, p.1153-1173,
!     2016. (doi: 10.1137/15M1054183)
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   clear
      implicit none

!     //   visit flag
      integer, save :: iset = 0

!     //   integers
      integer :: bead, atom

!-----------------------------------------------------------------------
!     /*   Read potential file (pot.mtp)                              */
!-----------------------------------------------------------------------

!     //   only first visit of this routine
      if ( iset .eq. 0 ) then

!        /*   read pot.mtp file   */
         call Read_MTP_File()

!        /*   set complete   */
         iset = 1

!     //   only first visit of this routine
      end if

!-----------------------------------------------------------------------
!     /*   Periodic Boundary Condition                                */
!-----------------------------------------------------------------------

!     //   Construct replicated boxes
      call MTP_Box()

!-----------------------------------------------------------------------
!     /*   Loop - Bead & Site EFS calculation                         */
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do bead = 1, nbead

!        /*   loop of atoms   */
         do atom = 1, natom

!           /*   Energy, Force, Stress calculation   */
            call SiteCalcEFS( bead, atom )

!        /*   loop of atoms   */
         end do

!     /*   loop of beads   */
      end do

      return
      end





!***********************************************************************
      subroutine SiteCalcEFS(bead,me)
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   natom

      use mtp_variables, only : &
     &   max_alpha_moment_mapping

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer:: i, j, me, bead

      real(8), dimension(max_alpha_moment_mapping+1):: moment_vals
      real(8), dimension(:,:,:), allocatable:: moment_jacobian

!     //   atomic neighborhood
      integer:: nbh, nbh_count
      integer, dimension(2*natom) :: nbh_index
      real(8), dimension(2*natom,3) :: nbh_vecs
      real(8), dimension(2*natom) :: nbh_dists

!-----------------------------------------------------------------------
!     /*   Search Neighborhood                                        */
!-----------------------------------------------------------------------

      call MTP_search_nbh &
     &   ( bead, me, nbh_count, nbh_index, nbh_vecs, nbh_dists )

!-----------------------------------------------------------------------
!     /*   Allocation & Initialization                                */
!-----------------------------------------------------------------------

      allocate(moment_jacobian(max_alpha_moment_mapping+1,nbh_count,3))

      do i = 1, max_alpha_moment_mapping+1
         moment_vals(i) = 0.d0
         do nbh = 1, nbh_count
         do j = 1, 3
            moment_jacobian(i,nbh,j) = 0.d0
         end do
         end do
      end do

!-----------------------------------------------------------------------
!     /*   Energy, Force, Stress from atomic neighborhood             */
!-----------------------------------------------------------------------

      do nbh = 1, nbh_count

         call Moment_Calc &
     &      ( me, nbh, moment_vals, moment_jacobian, &
     &        nbh_count, nbh_index, nbh_vecs, nbh_dists )

      end do

      call EFS_Calc &
     &   ( bead, me, moment_vals, moment_jacobian, &
     &     nbh_count, nbh_index, nbh_vecs )

      return
      end





!***********************************************************************
      subroutine EFS_Calc &
     &   ( bead, me, moment_vals, moment_jacobian, &
     &     nbh_count, nbh_index, nbh_vecs )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   natom, pot, fx, fy, fz, vir, ikind

      use mtp_variables, only : &
     &   moment_coeffs, alpha_index_times_count, &
     &   alpha_scalar_moments, alpha_index_times, alpha_moment_mapping, &
     &   alpha_index_basic_count, alpha_moments_count, &
     &   max_alpha_moment_mapping, species_coeffs

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     // atomic neighborhood
      integer :: nbh, nbh_count, you
      integer, dimension(2*natom) :: nbh_index
      real(8), dimension(2*natom,3) :: nbh_vecs

      integer :: bead, me, i, a, b
      real(8), dimension(3) :: vec
      real(8), dimension(alpha_moments_count) :: &
     &           site_energy_ders_wrt_moments
      real(8), dimension(nbh_count,3) :: buff_site_energy_ders
      real(8) :: val0, val1, val2, mypot
      real(8), dimension(alpha_scalar_moments) :: basis_vals
      real(8), dimension(max_alpha_moment_mapping+1) :: moment_vals
      real(8), dimension(max_alpha_moment_mapping+1,nbh_count,3) &
     &        :: moment_jacobian
      real(8) :: hartree_to_eV = 0.03674932218d0

!-----------------------------------------------------------------------
!     /*   Initialization                                             */
!-----------------------------------------------------------------------

      do i = 1, alpha_scalar_moments
         basis_vals(i) = 0.d0
      end do

      do nbh = 1, nbh_count
         do i = 1, 3
            buff_site_energy_ders(nbh,i) = 0.d0
         end do
      end do

      do i = 1, alpha_moments_count
         site_energy_ders_wrt_moments(i) = 0.d0
      end do

!-----------------------------------------------------------------------
!     /*   Site energy                                                */
!-----------------------------------------------------------------------

      do i = 1, alpha_index_times_count

         val2 = REAL(alpha_index_times(i,3))

         moment_vals(alpha_index_times(i,4)+1) &
     &      = moment_vals(alpha_index_times(i,4)+1) &
     &      + moment_vals(alpha_index_times(i,1)+1) &
     &      * moment_vals(alpha_index_times(i,2)+1) * val2

      end do

      mypot = 0.d0

      do i = 1, alpha_scalar_moments
         mypot = mypot &
     &      + moment_vals(alpha_moment_mapping(i)+1) * moment_coeffs(i)
      end do

      mypot = mypot + species_coeffs(ikind(me))
      mypot = mypot * hartree_to_eV

      pot(bead) = pot(bead) + mypot

!-----------------------------------------------------------------------
!     /*   Back Propagation                                           */
!-----------------------------------------------------------------------

!     Backpropagation step 1.

      do i = 1, alpha_scalar_moments
        site_energy_ders_wrt_moments(alpha_moment_mapping(i)+1) &
     &     = moment_coeffs(i)
      end do

!     Backpropagation step 2.

      do i = alpha_index_times_count, 1, -1

         val0 = moment_vals(alpha_index_times(i,1)+1)
         val1 = moment_vals(alpha_index_times(i,2)+1)
         val2 = REAL(alpha_index_times(i,3))

         site_energy_ders_wrt_moments(alpha_index_times(i,2)+1) &
     &      = site_energy_ders_wrt_moments(alpha_index_times(i,2)+1) &
     &      + site_energy_ders_wrt_moments(alpha_index_times(i,4)+1) &
     &      * val2 * val0

         site_energy_ders_wrt_moments(alpha_index_times(i,1)+1) &
     &      = site_energy_ders_wrt_moments(alpha_index_times(i,1)+1) &
     &      + site_energy_ders_wrt_moments(alpha_index_times(i,4)+1) &
     &      * val2 * val1

      end do

!     Backpropagation step 3.

      do nbh = 1, nbh_count

         do i = 1, alpha_index_basic_count
            do a = 1, 3
               buff_site_energy_ders(nbh,a) &
     &            = buff_site_energy_ders(nbh,a) &
     &            + site_energy_ders_wrt_moments(i) &
     &            * moment_jacobian(i,nbh,a)
            end do
         end do

!-----------------------------------------------------------------------
!     /*   Force & Virial                                             */
!-----------------------------------------------------------------------

         do a = 1, 3
            buff_site_energy_ders(nbh,a) &
     &         = buff_site_energy_ders(nbh,a) * hartree_to_eV
         end do

!        //   read information from neighborhood
         you = nbh_index(nbh)
         vec(1) = nbh_vecs(nbh,1)
         vec(2) = nbh_vecs(nbh,2)
         vec(3) = nbh_vecs(nbh,3)

!        //   force
         fx(me,bead) = fx(me,bead) + buff_site_energy_ders(nbh,1)
         fy(me,bead) = fy(me,bead) + buff_site_energy_ders(nbh,2)
         fz(me,bead) = fz(me,bead) + buff_site_energy_ders(nbh,3)
         fx(you,bead) = fx(you,bead) - buff_site_energy_ders(nbh,1)
         fy(you,bead) = fy(you,bead) - buff_site_energy_ders(nbh,2)
         fz(you,bead) = fz(you,bead) - buff_site_energy_ders(nbh,3)

!        //   virial
         do a = 1, 3
         do b = 1, 3
            vir(a,b) = vir(a,b) &
     &         - buff_site_energy_ders(nbh,a) * vec(b) * 0.5d0 &
     &         - buff_site_energy_ders(nbh,b) * vec(a) * 0.5d0
         end do
         end do

      end do

      return
      end





!***********************************************************************
      subroutine Moment_Calc &
     &   ( me, nbh,moment_vals, moment_jacobian, &
     &     nbh_count, nbh_index, nbh_vecs, nbh_dists )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ikind, natom

      use mtp_variables, only : &
     &   radial_funcs_count, max_alpha_index_basic, alpha_index_basic, &
     &   radial_coeffs, alpha_index_basic_count, &
     &   max_alpha_moment_mapping

!-----------------------------------------------------------------------
!     /*   Local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     // atomic neighborhood
      integer:: nbh, nbh_count
      integer, dimension(2*natom) :: nbh_index
      real(8), dimension(2*natom,3) :: nbh_vecs
      real(8), dimension(2*natom) :: nbh_dists

      integer :: me, you, i, j, k
      real(8) :: r, val, der, real_k, powk, pow0, pow1, pow2, mult0
      real(8), dimension(3) :: vec
      real(8), dimension(radial_funcs_count) :: RB_vals, RB_ders
      real(8), dimension(max_alpha_index_basic+1) :: dist_powers
      real(8), dimension(max_alpha_index_basic+1,3) :: coords_powers
      real(8), dimension(8) :: rb_vals_array, rb_ders_array
      real(8), dimension(max_alpha_moment_mapping+1) :: moment_vals
      real(8), dimension(max_alpha_moment_mapping+1,nbh_count,3) &
     &        :: moment_jacobian

!-----------------------------------------------------------------------
!     /*   Initiation                                                 */
!-----------------------------------------------------------------------

      do i = 1, radial_funcs_count
        RB_vals(i) = 0.d0
        RB_ders(i) = 0.d0
      end do

!-----------------------------------------------------------------------
!     /*  Get Information from neighborhood                           */
!-----------------------------------------------------------------------

      you = nbh_index(nbh)
      vec(1) = nbh_vecs(nbh,1)
      vec(2) = nbh_vecs(nbh,2)
      vec(3) = nbh_vecs(nbh,3)
      r = nbh_dists(nbh)

!-----------------------------------------------------------------------
!     /*  Get Radial Basis Function (RB Chebyshev)                    */
!-----------------------------------------------------------------------

      dist_powers(1) = 1.d0

      do i = 1, 3
         coords_powers(1,i) = 1.d0
      end do

      do i = 2, max_alpha_index_basic + 1
         dist_powers(i) = dist_powers(i-1) * r
         do j = 1, 3
            coords_powers(i,j) = coords_powers(i-1,j) * vec(j)
         end do
      end do

      call Cheby_RB( r, rb_vals_array, rb_ders_array )

      do i = 1, radial_funcs_count
         do j = 1, 8
            RB_vals(i) = RB_vals(i) &
     &         + rb_vals_array(j) &
     &         * radial_coeffs(ikind(me),ikind(you),i,j)
            RB_ders(i) = RB_ders(i) &
     &         + rb_ders_array(j) &
     &         * radial_coeffs(ikind(me),ikind(you),i,j)
         end do
      end do

!-----------------------------------------------------------------------
!     /*   Basic Moment Calculation                                   */
!-----------------------------------------------------------------------

      do i = 1, alpha_index_basic_count

         val = RB_vals(alpha_index_basic(i,1)+1)
         der = RB_ders(alpha_index_basic(i,1)+1)

         k = alpha_index_basic(i,2) + alpha_index_basic(i,3) &
     &     + alpha_index_basic(i,4)

         powk = 1.d0 / dist_powers(k+1)
         val = val * powk
         real_k = REAL(k)
         der = der * powk - real_k * val / r

         pow0 = coords_powers(alpha_index_basic(i,2)+1,1)
         pow1 = coords_powers(alpha_index_basic(i,3)+1,2)
         pow2 = coords_powers(alpha_index_basic(i,4)+1,3)

         mult0 = pow0 * pow1 * pow2
         moment_vals(i) = moment_vals(i) + val * mult0
         mult0 = mult0 * der /r

         do j = 1, 3
            moment_jacobian(i,nbh,j) = mult0 * vec(j)
         end do

         if ( alpha_index_basic(i,2) .ne. 0 ) then
            moment_jacobian(i,nbh,1) = moment_jacobian(i,nbh,1) &
     &         + val * alpha_index_basic(i,2) &
     &         * coords_powers(alpha_index_basic(i,2)+1-1,1) &
     &         * pow1 * pow2
         end if

         if ( alpha_index_basic(i,3) .ne. 0 ) then
            moment_jacobian(i,nbh,2) = moment_jacobian(i,nbh,2) &
     &         + val * alpha_index_basic(i,3) &
     &         * pow0 * coords_powers(alpha_index_basic(i,3)+1-1,2) &
     &         * pow2
         end if

         if ( alpha_index_basic(i,4) .ne. 0 ) then
            moment_jacobian(i,nbh,3) = moment_jacobian(i,nbh,3) &
     &         + val * alpha_index_basic(i,4) &
     &         * pow0 * pow1 &
     &         * coords_powers(alpha_index_basic(i,4)+1-1,3)
         end if

      end do

      return
      end





!***********************************************************************
      subroutine Cheby_RB( r, rb_vals, rb_ders )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use mtp_variables, only : &
     &   min_dist, max_dist, scaling

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      implicit none

      real(8) ::  r
      real(8), dimension(8) :: rb_vals, rb_ders

      real(8) :: mult, ksi
      integer :: i

!-----------------------------------------------------------------------
!     /*   Calculate values & derivatives                             */
!-----------------------------------------------------------------------

      mult = 2.d0 / ( max_dist - min_dist )
      ksi = ( 2.d0 * r - (max_dist + min_dist) ) / ( max_dist-min_dist )

      rb_vals(1) = scaling * ( 1.d0 * (r-max_dist) * (r-max_dist) )
      rb_ders(1) = scaling * ( 2.d0 * (r-max_dist) )
      rb_vals(2) = scaling * ( ksi  * (r-max_dist) * (r-max_dist) )
      rb_ders(2) = scaling * ( mult * (r-max_dist) * (r-max_dist) &
     &                       + 2.d0 * ksi * ( r-max_dist) )

      do i = 3, 8
         rb_vals(i) = 2.d0 * ksi * rb_vals(i-1) - rb_vals(i-2)
         rb_ders(i) = 2.d0 * ( mult*rb_vals(i-1) + ksi*rb_ders(i-1) ) &
     &              - rb_ders(i-2)
      end do

      return
      end





!***********************************************************************
      subroutine MTP_search_nbh &
     &   ( bead, me, nbh_count, nbh_index, nbh_vecs, nbh_dists )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   natom, x, y, z, iboundary, box

      use mtp_variables, only : &
     &   max_dist,min_dist, nbox_mtp, bigbox_mtp, bigboxinv_mtp

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: me, atom, bead, nbh_count
      integer, dimension(2*natom) :: nbh_index
      real(8), dimension(2*natom,3) :: nbh_vecs
      real(8), dimension(2*natom) :: nbh_dists
      real(8) :: r

      integer :: jx, jy ,jz, j2
      real(8) :: xij, yij, zij, aij, bij, cij

!-----------------------------------------------------------------------
!     /*   Box check                                                  */
!-----------------------------------------------------------------------

      nbh_count = 0

      do atom = 1, natom

!     /*   loop of replicated boxes   */
      do jx = 0, nbox_mtp(1)-1
      do jy = 0, nbox_mtp(2)-1
      do jz = 0, nbox_mtp(3)-1

!       /*   square of box index   */
        j2 = jx*jx + jy*jy + jz*jz

!       /*   skip the same atom   */
        if ( ( j2 .eq. 0 ) .and. ( me .eq. atom ) ) cycle

!       /*   interatomic distance in same box   */
        xij = x(atom,bead) - x(me,bead)
        yij = y(atom,bead) - y(me,bead)
        zij = z(atom,bead) - z(me,bead)

        if ( iboundary .ne. 0 ) then

!       /*   distance in different box   */
        xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
        yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
        zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

!       /*   vector in big box   */
        aij = bigboxinv_mtp(1,1)*xij + bigboxinv_mtp(1,2)*yij &
     &      + bigboxinv_mtp(1,3)*zij
        bij = bigboxinv_mtp(2,1)*xij + bigboxinv_mtp(2,2)*yij &
     &      + bigboxinv_mtp(2,3)*zij
        cij = bigboxinv_mtp(3,1)*xij + bigboxinv_mtp(3,2)*yij &
     &      + bigboxinv_mtp(3,3)*zij

!       /*   apply periodic boundary in big box   */
        aij = aij - nint(aij)
        bij = bij - nint(bij)
        cij = cij - nint(cij)

!       /*   distance of nearest i and j   */
        xij = bigbox_mtp(1,1)*aij + bigbox_mtp(1,2)*bij &
     &      + bigbox_mtp(1,3)*cij
        yij = bigbox_mtp(2,1)*aij + bigbox_mtp(2,2)*bij &
     &      + bigbox_mtp(2,3)*cij
        zij = bigbox_mtp(3,1)*aij + bigbox_mtp(3,2)*bij &
     &      + bigbox_mtp(3,3)*cij

        end if

!       /*   interatomic distance squared   */
        r = sqrt(xij*xij + yij*yij + zij*zij)

        if ( r .gt. max_dist ) cycle
        if ( r .lt. min_dist ) cycle

!       /*   save neighborhood infromation   */
        nbh_count = nbh_count + 1
        nbh_index(nbh_count) = atom
        nbh_vecs(nbh_count,1) = xij
        nbh_vecs(nbh_count,2) = yij
        nbh_vecs(nbh_count,3) = zij
        nbh_dists(nbh_count) = r

      end do
      end do
      end do

      end do

      return
      end





!***********************************************************************
      subroutine MTP_Box
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   box, volume, iboundary

      use mtp_variables, only : &
     &   nbox_mtp, bigbox_mtp, bigboxinv_mtp, max_dist

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8):: ax, ay, az, bx, by, bz, cx, cy, cz, absa, absb, absc

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   free boundary   */
      if ( iboundary .eq. 0 ) then

!        /*   number of replicated boxes   */
         nbox_mtp(1) = 1
         nbox_mtp(2) = 1
         nbox_mtp(3) = 1

!     /*   periodic boundary   */
      else

!        /*   vector product of lattice vectors b, c   */
         ax = box(2,2)*box(3,3) - box(2,3)*box(3,2)
         ay = box(3,2)*box(1,3) - box(3,3)*box(1,2)
         az = box(1,2)*box(2,3) - box(1,3)*box(2,2)

!        /*   vector product of lattice vectors c, a   */
         bx = box(2,3)*box(3,1) - box(2,1)*box(3,3)
         by = box(3,3)*box(1,1) - box(3,1)*box(1,3)
         bz = box(1,3)*box(2,1) - box(1,1)*box(2,3)

!        /*   vector product of lattice vectors a, b   */
         cx = box(2,1)*box(3,2) - box(2,2)*box(3,1)
         cy = box(3,1)*box(1,2) - box(3,2)*box(1,1)
         cz = box(1,1)*box(2,2) - box(1,2)*box(2,1)

!        /*   distance between parallel planes   */
         absa = volume / sqrt( ax*ax + ay*ay + az*az )
         absb = volume / sqrt( bx*bx + by*by + bz*bz )
         absc = volume / sqrt( cx*cx + cy*cy + cz*cz )

!        /*   number of replicated boxes   */
         nbox_mtp(1) = int(2.d0*max_dist/absa) + 1
         nbox_mtp(2) = int(2.d0*max_dist/absb) + 1
         nbox_mtp(3) = int(2.d0*max_dist/absc) + 1

      end if

      bigbox_mtp(:,1) = dble(nbox_mtp(1))*box(:,1)
      bigbox_mtp(:,2) = dble(nbox_mtp(2))*box(:,2)
      bigbox_mtp(:,3) = dble(nbox_mtp(3))*box(:,3)

      call inv3 ( bigbox_mtp, bigboxinv_mtp )

      return
      end





!***********************************************************************
      subroutine Read_MTP_File()
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, au_length

      use mtp_variables, only : &
     &   species_count, radial_funcs_count, alpha_moments_count, &
     &   alpha_index_times_count, alpha_scalar_moments, &
     &   max_alpha_moment_mapping, max_alpha_index_basic, &
     &   alpha_index_basic, alpha_index_times, alpha_moment_mapping, &
     &   radial_coeffs, species_coeffs, moment_coeffs, scaling, &
     &   min_dist,max_dist, alpha_index_basic_count

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, m
      real(8), dimension(8) :: radial_buffer
      character(40) :: temp
      real(8), parameter :: bohr2ang = au_length / 1.d-10

!-----------------------------------------------------------------------
!     /*   Read pot.mtp                                               */
!-----------------------------------------------------------------------

      open ( unit = iounit, file = 'pot.mtp' )
      read ( iounit, * )
      read ( iounit, * )
      read ( iounit, * )
      read ( iounit, * ) temp, temp, scaling
      read ( iounit, * ) temp, temp, species_count
      read ( iounit, * )
      read ( iounit, * )
      read ( iounit, * ) temp, temp, min_dist
      read ( iounit, * ) temp, temp, max_dist
      read ( iounit, * )
      read ( iounit, * ) temp, temp, radial_funcs_count

!-----------------------------------------------------------------------
!     /*  Radial Coefficients                                         */
!-----------------------------------------------------------------------

      if ( .not. allocated(radial_coeffs) ) &
     &   allocate( &
     &   radial_coeffs(species_count,species_count,radial_funcs_count,8) &
     &   )

      read ( iounit, * ) temp

      do i = 1, species_count
      do j = 1, species_count

         read ( iounit, * )

         do k =  1, radial_funcs_count
            call Find_Radial_Coeffs( iounit, radial_buffer )
            do m = 1, 8
               radial_coeffs(i,j,k, m) = radial_buffer(m)
            end do
         end do

      end do
      end do

!-----------------------------------------------------------------------
!     /*   alpha_index                                                */
!-----------------------------------------------------------------------

      read ( iounit, * ) temp, temp, alpha_moments_count
      read ( iounit, * ) temp, temp, alpha_index_basic_count

      if ( .not. allocated(alpha_index_basic) ) &
     &   allocate( alpha_index_basic(alpha_index_basic_count,4) )

      call Find_Alpha_Index( iounit, alpha_index_basic_count, &
     &                       alpha_index_basic )

      read( iounit, * ) temp, temp, alpha_index_times_count

      if ( .not. allocated(alpha_index_times) ) &
     &   allocate( alpha_index_times( alpha_index_times_count,4) )

      call Find_Alpha_Index( iounit, alpha_index_times_count, &
     &                       alpha_index_times)

      do i = 1, alpha_index_times_count
         alpha_index_times(i,1) = alpha_index_times(i,1)
         alpha_index_times(i,2) = alpha_index_times(i,2)
         alpha_index_times(i,4) = alpha_index_times(i,4)
      end do

!-----------------------------------------------------------------------
!     /*    moment index                                              */
!-----------------------------------------------------------------------

      read ( iounit, * ) temp, temp, alpha_scalar_moments

      if ( .not. allocated(alpha_moment_mapping) ) &
     &   allocate( alpha_moment_mapping(alpha_scalar_moments) )

      call Find_Moment_Index &
     &   ( iounit, alpha_scalar_moments, alpha_moment_mapping )

      if ( .not. allocated(species_coeffs) ) &
     &   allocate( species_coeffs(species_count) )

      call Find_Species_Coeffs &
     &   ( iounit, species_count, species_coeffs )

      if ( .not. allocated(moment_coeffs) ) &
     &   allocate( moment_coeffs(alpha_scalar_moments) )

      call Find_Species_Coeffs &
     &   ( iounit, alpha_scalar_moments, moment_coeffs)

!-----------------------------------------------------------------------
!     /*  max_alpha_index_basic                                       */
!-----------------------------------------------------------------------

      max_alpha_index_basic = 0

      do i = 1, alpha_index_basic_count
         max_alpha_index_basic &
     &      = max( max_alpha_index_basic, &
     &             alpha_index_basic(i,2) + alpha_index_basic(i,3) &
     &           + alpha_index_basic(i,4) )
      end do

      max_alpha_index_basic = max_alpha_index_basic + 1

!-----------------------------------------------------------------------
!     /*   max_alpha_moment_mapping                                   */
!-----------------------------------------------------------------------

      max_alpha_moment_mapping = 0

      do i = 1, alpha_scalar_moments
         max_alpha_moment_mapping &
     &      = max( max_alpha_moment_mapping, alpha_moment_mapping(i) )
      end do

!-----------------------------------------------------------------------
!     /*  Unit conversion for Angstrom -> Bohr radius                 */
!-----------------------------------------------------------------------

      min_dist = min_dist / bohr2ang
      max_dist = max_dist / bohr2ang

      do i = 1, species_count
      do j = 1, species_count
      do m = 1, radial_funcs_count
      do k = 1, 8
         radial_coeffs(i,j,m,k) = radial_coeffs(i,j,m,k) &
     &                          * bohr2ang * bohr2ang
      end do
      end do
      end do
      end do

      close( unit = iounit )

      return
      end





!***********************************************************************
      subroutine Find_Species_Coeffs &
     &   ( iounit, ntype, species_coeffs )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none
      integer :: iounit, ntype
      real(8), dimension(ntype) :: species_coeffs

      integer:: i, j, buffer_index
      integer, dimension(:), allocatable:: buffer
      character(1):: chr
      character(40):: val

!-----------------------------------------------------------------------
!     /*   READ and find positions of buffer                          */
!-----------------------------------------------------------------------

      allocate(buffer(ntype + 1))

      buffer_index = 0
      i = 0
      chr = '0'

      do
         i = i + 1
         read ( iounit, '(a1)', Advance = 'No' ) chr
         if ( (chr.eq.'{') .OR. (chr.eq.'}') .OR. (chr.eq.',') ) then
            buffer_index = buffer_index + 1
            buffer(buffer_index) = i
         end if
         if ( chr .eq. '}' ) exit
      end do

!-----------------------------------------------------------------------
!     /*   Get elements                                               */
!-----------------------------------------------------------------------

      BACKSPACE iounit

      do i = 1, buffer(1)-1
         read ( iounit, '(a1)', Advance = 'No' ) chr
      end do

      do i = 1, ntype
         read ( iounit, '(a1)', Advance = 'No' ) chr
         buffer_index = 0
         do j = 1, 40
            val(j:j) = ' '
         end do
         do j = buffer(i)+1, buffer(i+1)-1
            buffer_index = buffer_index + 1
            read ( iounit, '(a1)', Advance = 'No' ) chr
            val(buffer_index:buffer_index) = chr
         end do
         read ( val, * ) species_coeffs(i)
      end do

      read ( iounit, * )

      return
      end





!***********************************************************************
      subroutine Find_Moment_Index &
     &   ( iounit, alpha_scalar_moments, alpha_moment_mapping )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: iounit, alpha_scalar_moments
      integer, dimension(alpha_scalar_moments) :: alpha_moment_mapping

      integer :: i, j, buffer_index
      integer, dimension(:), allocatable :: buffer
      character(1) :: chr
      character(40) :: val

!-----------------------------------------------------------------------
!     /*   READ and find positions of buffer                          */
!-----------------------------------------------------------------------

      allocate( buffer(alpha_scalar_moments + 1) )

      buffer_index = 0
      i = 0
      chr = '0'

      do
         i = i + 1
         read ( iounit, '(a1)', Advance = 'No' ) chr
         if ( (chr.eq.'{') .OR. (chr.eq.'}') .OR. (chr.eq.',') ) then
            buffer_index = buffer_index + 1
            buffer(buffer_index) = i
         end if
         if (chr .eq. '}') exit
      end do

!-----------------------------------------------------------------------
!     /*   Get elements                                               */
!-----------------------------------------------------------------------

      BACKSPACE iounit

      do i = 1, buffer(1)-1
         read ( iounit, '(a1)', Advance = 'No' ) chr
      end do

      do i = 1, alpha_scalar_moments
         read ( iounit, '(a1)', Advance = 'No' ) chr
         buffer_index = 0
         do j = 1, 40
            val(j:j) = ' '
         end do
         do j = buffer(i)+1 , buffer(i+1)-1
            buffer_index = buffer_index + 1
            read ( iounit, '(a1)', Advance = 'No' ) chr
            val(buffer_index:buffer_index) = chr
         end do
         read ( val, * ) alpha_moment_mapping(i)
      end do

      read ( iounit, * )

      return
      end





!***********************************************************************
      subroutine Find_Alpha_Index &
     &   ( iounit, alpha_index_basic_count, alpha_index_basic)
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: iounit, alpha_index_basic_count
      integer, dimension(alpha_index_basic_count,4) :: alpha_index_basic

      integer :: i, j, k, m, n, buffer_index
      integer, dimension(:), allocatable:: buffer
      integer, dimension(:), allocatable:: linear_array
      character(1):: chr, old
      character(10):: val

!-----------------------------------------------------------------------
!     /*   READ and find positions of buffer                          */
!-----------------------------------------------------------------------

      allocate( buffer(6*alpha_index_basic_count+1) )
      allocate( linear_array(alpha_index_basic_count*4) )

      buffer_index = 0
      i = 0
      chr = '0'

      do

         old = chr
         i = i + 1

         read (iounit,'(a1)',Advance = 'No') chr

         if ( (chr.eq.'{') .OR. (chr.eq.'}') .OR. (chr.eq.',') ) then
            buffer_index = buffer_index + 1
            buffer(buffer_index) = i
         end if

         if ( (chr.eq.'}') .AND. (old.eq.'}') ) exit

      end do

!-----------------------------------------------------------------------
!     /*   Allocate vales to arrays                                   */
!-----------------------------------------------------------------------

      BACKSPACE iounit

      do j = 1, buffer(1)
         read ( iounit, '(a1)', Advance = 'No' ) chr
      end do

      m = 1
      n = 1

      do i = 1, alpha_index_basic_count
         read ( iounit, '(a1)', Advance = 'No' ) chr
         do j = 1, 4
            do k = 1, 10
               val(k:k) = ' '
            end do
            buffer_index = 0
            do k = buffer(n+1)+1, buffer(n+2)-1
               buffer_index = buffer_index + 1
               read ( iounit, '(a1)', Advance = 'No' ) chr
               val(buffer_index:buffer_index) = chr
            end do
            read ( iounit, '(a1)', Advance = 'No' ) chr
            read ( val, * ) linear_array(m)
            m = m + 1
            n = n + 1
         end do
         if ( i .eq. alpha_index_basic_count ) exit
         read ( iounit, '(a1)', Advance = 'No' ) chr
         read ( iounit, '(a1)', Advance = 'No' ) chr
         n = n + 2
      end do

      m = 1

      do i = 1, alpha_index_basic_count
         do j = 1, 4
            alpha_index_basic(i,j) = linear_array(m)
            m = m + 1
         end do
      end do

      read ( iounit, * )

      return
      end





!***********************************************************************
      subroutine Find_Radial_Coeffs( iounit, radial_buffer )
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none
      integer :: iounit
      real(8), dimension(8) :: radial_buffer

      integer, dimension(9) :: buffer
      character(50) :: chr, val
      integer :: i, j, buffer_index

!-----------------------------------------------------------------------
!     /*   Find buffer                                                */
!-----------------------------------------------------------------------

      buffer_index = 0
      i = 0

      do

         i = i + 1

         read ( iounit, '(a1)', Advance = 'No' ) chr

         if ( (chr.eq.'{') .OR. (chr.eq.'}') .OR. (chr.eq.',') ) then
            buffer_index = buffer_index + 1
            buffer(buffer_index) = i
         end if

         if ( chr .eq. '}' ) exit

      end do

!-----------------------------------------------------------------------
!     /*   Get radial coeffs                                          */
!-----------------------------------------------------------------------

      BACKSPACE iounit

      do j = 1, buffer(1)-1
         read ( iounit, '(a1)', Advance = 'No' ) chr
      end do

      do i = 1, 8

         read ( iounit, '(a1)', Advance = 'No' ) chr

         buffer_index = 0

         do j = 1, 40
            val(j:j) = ' '
         end do

         do j = buffer(i)+1 , buffer(i+1)-1
            buffer_index = buffer_index + 1
            read ( iounit, '(a1)', Advance = 'No' ) chr
            val(buffer_index:buffer_index) = chr
         end do

         read ( val, * ) radial_buffer(i)

      end do

      read ( iounit, * )

      return
      end
