!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     energy and force from MOPAC calculation
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      module mopac_variables
!***********************************************************************

      integer, dimension(:), allocatable:: isymbol

      character(len=4), dimension(:), allocatable:: char_symbol

!***********************************************************************
      end module mopac_variables
!***********************************************************************





!***********************************************************************
      subroutine force_mopac_MPI
!***********************************************************************
!=======================================================================
!
!     WARNING:  DIPOLE MOMENT IS NOT READ.
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, dipx, dipy, dipz, &
     &   au_length, mopac_command, mbox, iounit, iounit_mopac, &
     &   nbead, nprocs, myrank, natom

      use mopac_variables, only : char_symbol

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr, ibead, i, j, itest

      real(8) :: bohr2ang, au2kcal, xi, yi, zi, &
     &           pot_kcal, grad_kcal

      character(len=80):: char_line, char_file, char_command

      character(len=3)::  char_num

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   constants                                                  */
!-----------------------------------------------------------------------

      bohr2ang   = au_length * 1.d+10
      au2kcal    = 627.5095d0

!-----------------------------------------------------------------------
!     /*   memory allocation and initialize directory                 */
!-----------------------------------------------------------------------

      ierr = 0

      if ( iset .eq. 0 ) then

         if ( .not. allocated( char_symbol ) ) &
     &      allocate ( char_symbol(natom) )

         do ibead = 1, nbead

!           /*   allocated bead only   */
            if ( mod( ibead-1, nprocs ) .ne. myrank ) cycle
            call int3_to_char( ibead, char_num )

!           /*   remove old directory and create new directory   */
            call system ('rm -f -r ./' // char_num )
            call system ('mkdir -p ./' // char_num )

!           /*   call MPI_barrier   */
            call my_mpi_barrier

         end do

         if ( myrank .eq. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'input.dat' )

!           /*   search for tag    */
            call search_tag ( '<mopac_command>', 15, iounit, ierr )

!           /*   read a line   */
            read ( iounit, *, iostat=ierr ) mopac_command

!           /*   file close   */
            close ( iounit )

            if ( ierr .ne. 0 ) then

!              /*   file open   */
               open ( iounit, file = 'input_default.dat' )

!              /*   search for tag    */
               call search_tag ( '<mopac_command>', 15, iounit, ierr )

!              /*   read a line   */
               read ( iounit, *, iostat=ierr ) mopac_command

!              /*   file close   */
               close ( iounit )

            end if

!-----------------------------------------------------------------------
!        /*   confirm mopac command                                   */
!-----------------------------------------------------------------------

!            call system
!     &         ("echo '0' > test.out")
!            call system
!     &         ("sleep 0.1")
!            call system
!     &         ("which " // mopac_command //
!     &          " > /dev/null 2>&1 && echo '1' > test.out")
!
!            open ( iounit, file = 'test.out' )
!
!            read ( iounit, * ) itest
!
!            close( iounit )

            itest = 1

            if ( itest .eq. 0 ) then

               ierr = 1

               write( 6, '(a)' ) 'Error - Mopac command not found: ' // &
     &                            trim(mopac_command)
               write( 6, '(a)' )

            else

               ierr = 0

               write( 6, '(a)' ) 'Mopac command found: ' // &
     &                            trim(mopac_command)
               write( 6, '(a)' )

            end if

            call system('rm -f test.out')

         end if

         call error_handling_MPI &
     &       ( ierr, 'subroutine force_mopac_MPI', 26 )

         call my_mpi_bcast_char_0 ( mopac_command, len(mopac_command) )

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!     /*   restart from here when error occurred                      */
!-----------------------------------------------------------------------

  100 continue

!-----------------------------------------------------------------------
!     /*   skip if `ibead is not my job'                              */
!-----------------------------------------------------------------------

      if ( mod( ibead-1, nprocs ) .ne. myrank ) cycle

!-----------------------------------------------------------------------
!     /*   make char_num according to myrank                          */
!-----------------------------------------------------------------------

      call int3_to_char( ibead, char_num )

!-----------------------------------------------------------------------
!     /*   make mopac input:  input.mop                               */
!-----------------------------------------------------------------------

!     /*   open the mopac prototype file   */
      open ( iounit,     file = 'mopac.dat'   )

!     /*   open the mopac input file   */
      char_file = ('./' // char_num // '/mopac.mop')
      open ( iounit_mopac, file = char_file )

      do i = 1, 3
         read ( iounit, '(a80)', iostat=ierr ) char_line
         write( iounit_mopac, '(a)' ) char_line
      end do

      do i = 1, natom

!        /*   read atomic number   */
         read ( iounit, *, iostat=ierr ) char_symbol(i)

!        /*   apply boundary condition: either free or periodic   */

         xi = x(i,ibead)
         yi = y(i,ibead)
         zi = z(i,ibead)

         call pbc_unfold_MPI &
     &      ( xi, yi, zi, mbox(1,i,ibead), mbox(2,i,ibead), &
     &        mbox(3,i,ibead) )

!        /*   change units au -> bohr   */

         xi = xi * bohr2ang
         yi = yi * bohr2ang
         zi = zi * bohr2ang

!        /*   write atomic number and Cartesian coordinates   */

         write( iounit_mopac, '(a4,f24.16,i2,f24.16,i2,f24.16,i2)' ) &
     &      char_symbol(i), xi, 1, yi, 1, zi, 1

      end do

!     /*   add a blank line at the end   */
      write( iounit_mopac, '(a)' ) ' '

!     /*   close files   */

      close( iounit )
      close( iounit_mopac )

!-----------------------------------------------------------------------
!     /*   run mopac                                                  */
!-----------------------------------------------------------------------

      char_command = mopac_command

      char_file  = ('./' // char_num // '/mopac.mop')
      char_line  = ('./' // char_num // '/mopac.scr')

      call system( char_command // ' ' // char_file // &
     &              ' > ' // char_line )

!     /*   remove unnecessary files
      call system('rm -f ./' // char_num // '/mopac.mop')
      call system('rm -f ./' // char_num // '/mopac.arc')
      call system('rm -f ./' // char_num // '/mopac.scr')

!-----------------------------------------------------------------------
!     /*   read mopac output:  potential                              */
!-----------------------------------------------------------------------

!     /*   open the mopac output file   */
      char_file = ('./' // char_num // '/mopac.out')
      open ( iounit_mopac, file = char_file )

      do

!        /*   read a line   */
         read ( iounit_mopac, '(a80)', iostat=ierr ) char_line

!        /*   error handling   */
         if ( ierr .ne. 0 ) then
            close ( iounit_mopac )
            go to 100
         end if

!        /*   see if the line matches   */
         if ( char_line(1:23) .eq. '          FINAL HEAT OF' )  exit

      end do

!     /*   go back one line   */
      backspace ( iounit_mopac )

!     /*   read potential in kcal/mol   */
      read ( iounit_mopac, '(a40,f12.5)', iostat=ierr ) &
     &   char_line, pot_kcal

!     /*   error handling   */
      if ( ierr .ne. 0 ) then
         close ( iounit_mopac )
         go to 100
      end if

!     /*   kcal/mol -> hartree   */
      pot(ibead) = pot_kcal / au2kcal

!     /*   close file   */
      close ( iounit_mopac )

!-----------------------------------------------------------------------
!     /*   read mopac output:  gradient                               */
!-----------------------------------------------------------------------

!     /*   open the mopac output file   */
      char_file = ('./' // char_num // '/mopac.out')
      open ( iounit_mopac, file = char_file )

      do

!        /*   read a line   */
         read ( iounit_mopac, '(a80)', iostat=ierr ) char_line

!        /*   seek for error   */
!        if ( ierr .ne. 0 ) exit

!        /*   see if the line matches   */
         if ( char_line(1:12) .eq. '   PARAMETER') exit

      end do

!     /*   if there was no error   */
      if ( ierr .eq. 0 ) then

!        /*   gradient kcal/mol/angstrom -> force hartree/bohr   */
         do i = 1, natom
            read ( iounit_mopac, '(a49,f13.5)', iostat=ierr ) &
     &         char_line, grad_kcal
            fx(i,ibead) = - grad_kcal  / au2kcal * bohr2ang
            read ( iounit_mopac, '(a49,f13.5)', iostat=ierr ) &
     &         char_line, grad_kcal
            fy(i,ibead) = - grad_kcal  / au2kcal * bohr2ang
            read ( iounit_mopac, '(a49,f13.5)', iostat=ierr ) &
     &         char_line, grad_kcal
            fz(i,ibead) = - grad_kcal  / au2kcal * bohr2ang

!           /*   error handling   */
            if ( ierr .ne. 0 ) then
               close ( iounit_mopac )
               go to 100
            end if

         end do

!     /*   if there was an error   */
      else

!        /*   error handling   */
         close ( iounit_mopac )
         go to 100

      end if

!-----------------------------------------------------------------------
!     /*   read mopac output:  dipole moment not read                 */
!-----------------------------------------------------------------------

      dipx(ibead) = 0.d0
      dipy(ibead) = 0.d0
      dipz(ibead) = 0.d0


!     /*   close file   */
      close ( iounit_mopac )

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz, natom, nbead )

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom

         xi = x(i,j)
         yi = y(i,j)
         zi = z(i,j)

         call pbc_unfold_MPI &
     &      ( xi, yi, zi, mbox(1,i,j), mbox(2,i,j), mbox(3,i,j) )

         vir(1,1) = vir(1,1) + fx(i,j)*xi
         vir(1,2) = vir(1,2) + fx(i,j)*yi
         vir(1,3) = vir(1,3) + fx(i,j)*zi
         vir(2,1) = vir(2,1) + fy(i,j)*xi
         vir(2,2) = vir(2,2) + fy(i,j)*yi
         vir(2,3) = vir(2,3) + fy(i,j)*zi
         vir(3,1) = vir(3,1) + fz(i,j)*xi
         vir(3,2) = vir(3,2) + fz(i,j)*yi
         vir(3,3) = vir(3,3) + fz(i,j)*zi

      end do
      end do

      return
      end





!***********************************************************************
      subroutine force_mais_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, &
     &   au_length, mopac_command, mbox, &
     &   iounit, iounit_mopac, nbead, nprocs, myrank, natom

      use mopac_variables, only : isymbol

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr, ibead, i, j

      real(8) :: bohr2ang, au2kcal, xi, yi, zi, &
     &           pot_kcal, grad_kcal

      character(len=80):: char_line, char_file, char_command
      character(len=3)::  char_num

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   constants                                                  */
!-----------------------------------------------------------------------

      bohr2ang   = au_length * 1.d+10
      au2kcal    = 627.5095d0

!-----------------------------------------------------------------------
!     /*   memory allocation and initialize directory                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         if ( .not. allocated( isymbol ) ) &
     &      allocate ( isymbol(natom) )

         do ibead = 1, nbead

!           /*   allocated bead only   */
            if ( mod( ibead-1, nprocs ) .ne. myrank ) cycle
            call int3_to_char( ibead, char_num )

!           /*   remove old directory and create new directory   */
            call system ('rm -f -r ./' // char_num )
            call system ('mkdir -p ./' // char_num )

!           /*   call MPI_barrier   */
            call my_mpi_barrier

         end do

         if ( myrank .eq. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'input.dat' )

!           /*   search for tag    */
            call search_tag ( '<mopac_command>', 15, iounit, ierr )

!           /*   read a line
            read ( iounit, *, iostat=ierr ) mopac_command

!           /*   file close   */
            close ( iounit )

            if ( ierr .ne. 0 ) then

!              /*   file open   */
               open ( iounit, file = 'input_default.dat' )

!              /*   search for tag    */
               call search_tag ( '<mopac_command>', 15, iounit, ierr )

!              /*   read a line
               read ( iounit, *, iostat=ierr ) mopac_command

!              /*   file close   */
               close ( iounit )

            end if

         end if

         call my_mpi_bcast_char_0 ( mopac_command, len(mopac_command) )

         iset = 1

      end if
!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!     /*   skip if `ibead is not my job'                              */
!-----------------------------------------------------------------------

      if ( mod( ibead-1, nprocs ) .ne. myrank ) cycle

!-----------------------------------------------------------------------
!     /*   make char_num according to myrank                          */
!-----------------------------------------------------------------------

      call int3_to_char( ibead, char_num )

!-----------------------------------------------------------------------
!     /*   make mopac input:  input.mop                               */
!-----------------------------------------------------------------------

!     /*   open the mopac prototype file   */
      open ( iounit,     file = 'mopac.dat'   )

!     /*   open the mopac input file   */
      char_file = ('./' // char_num // '/mopac.mop')
      open ( iounit_mopac, file = char_file )

      do i = 1, 3
         read ( iounit, '(a80)', iostat =ierr ) char_line
         write( iounit_mopac, '(a)' ) char_line
      end do

      do i = 1, natom

!        /*   read atomic number   */
         read ( iounit, *, iostat=ierr ) isymbol(i)

!        /*   apply boundary condition: either free or periodic   */

         xi = x(i,ibead)
         yi = y(i,ibead)
         zi = z(i,ibead)

         call pbc_unfold_MPI &
     &      ( xi, yi, zi, mbox(1,i,ibead), mbox(2,i,ibead), &
     &        mbox(3,i,ibead) )

!        /*   change units au -> bohr   */

         xi = xi * bohr2ang
         yi = yi * bohr2ang
         zi = zi * bohr2ang

!        /*   write atomic number and Cartesian coordinates   */

         write( iounit_mopac, '(i4,f24.16,i2,f24.16,i2,f24.16,i2)' ) &
     &      isymbol(i), xi, 1, yi, 1, zi, 1

      end do

!     /*   add a blank line at the end   */
      write( iounit_mopac, '(a)' ) ' '

!     /*   close files   */

      close( iounit )
      close( iounit_mopac )

!-----------------------------------------------------------------------
!     /*   run mopac                                                  */
!-----------------------------------------------------------------------

      char_command = mopac_command

      char_file  = ('./' // char_num // '/mopac.mop')
      char_line  = ('./' // char_num // '/mopac.scr')

      call system( char_command // ' ' // char_file // &
     &              ' > ' // char_line )

!     /*   remove unnecessary files
!      call system('rm -f ./' // char_num // '/mopac.mop')
!      call system('rm -f ./' // char_num // '/mopac.arc')
!      call system('rm -f ./' // char_num // '/mopac.scr')

!-----------------------------------------------------------------------
!     /*   read mopac output:  potential                              */
!-----------------------------------------------------------------------

!     /*   open the mopac output file   */
      char_file = ('./' // char_num // '/mopac.out')
      open ( iounit_mopac, file = char_file )

      do

!        /*   read a line   */
         read ( iounit_mopac, '(a80)', iostat=ierr ) char_line

!        /*   see if the line matches   */
         if ( char_line(1:23) .eq. '          FINAL HEAT OF' )  exit

      end do

!     /*   go back one line   */
      backspace ( iounit_mopac )

!     /*   read potential in kcal/mol   */
      read ( iounit_mopac, '(a40,f12.5)' ) char_line, pot_kcal

!     /*   kcal/mol -> hartree   */
      pot(ibead) = pot_kcal / au2kcal

!     /*   close file   */
      close ( iounit_mopac )

!-----------------------------------------------------------------------
!     /*   read mopac output:  gradient                               */
!-----------------------------------------------------------------------

!     /*   open the mopac output file   */
      char_file = ('./' // char_num // '/mopac.out')
      open ( iounit_mopac, file = char_file )

      do

!        /*   read a line   */
         read ( iounit_mopac, '(a80)', iostat=ierr ) char_line

!        /*   see if the line matches   */
         if ( char_line(1:12) .eq. '   PARAMETER') exit

      end do

!     /*   gradient kcal/mol/angstrom -> force hartree/bohr   */
      do i = 1, natom
         read ( iounit_mopac, '(a49,f13.5)' ) char_line, grad_kcal
         fx(i,ibead) = - grad_kcal  / au2kcal * bohr2ang
         read ( iounit_mopac, '(a49,f13.5)' ) char_line, grad_kcal
         fy(i,ibead) = - grad_kcal  / au2kcal * bohr2ang
         read ( iounit_mopac, '(a49,f13.5)' ) char_line, grad_kcal
         fz(i,ibead) = - grad_kcal  / au2kcal * bohr2ang
      end do

!     /*   close file   */
      close ( iounit_mopac )

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   subtract nuclear repulsion and add mais term               */
!-----------------------------------------------------------------------

      call force_mais_mod_MPI

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz, natom, nbead )

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom

         xi = x(i,j)
         yi = y(i,j)
         zi = z(i,j)

         call pbc_unfold_MPI &
     &      ( xi, yi, zi, mbox(1,i,j), mbox(2,i,j), mbox(3,i,j) )

         vir(1,1) = vir(1,1) + fx(i,j)*xi
         vir(1,2) = vir(1,2) + fx(i,j)*yi
         vir(1,3) = vir(1,3) + fx(i,j)*zi
         vir(2,1) = vir(2,1) + fy(i,j)*xi
         vir(2,2) = vir(2,2) + fy(i,j)*yi
         vir(2,3) = vir(2,3) + fy(i,j)*zi
         vir(3,1) = vir(3,1) + fz(i,j)*xi
         vir(3,2) = vir(3,2) + fz(i,j)*yi
         vir(3,3) = vir(3,3) + fz(i,j)*zi

      end do
      end do

      return
      end





!***********************************************************************
      subroutine force_mais_mod_MPI
!***********************************************************************
!=======================================================================
!
!     PM3-MAIS modification (tentative)
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, au_length, mbox, &
     &   iounit, iounit_mopac, nbead, nprocs, myrank, natom

      use mopac_variables, only : isymbol

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ni, nj, ibead, i, j, ig

      real(8) :: bohr2ang, au2ev, xi, yi, zi, xj, yj, zj, &
     &           xij, yij, zij, rij, rij2, rij3, a, b, c, re, re2, &
     &           factor, factor_1, factor_2, core2

      real(8):: core(20)

      real(8):: guesp1(20,2), guesp2(20,2), guesp3(20,2)

      real(8):: gmais1(20,20,3), gmais2(20,20,3), gmais3(20,20,3)

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   set mais parameters:  only H, O, Cl implemented!           */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         core(:)        =  0.d0

         guesp1(:,:)    =  0.d0

         gmais1(:,:,:)  =  0.d0

         core(  1 ) =  1.0d0
         core(  8 ) =  6.0d0
         core( 17 ) =  7.0d0

         guesp1(  1, 1 ) =  1.128750d0
         guesp2(  1, 1 ) =  5.096282d0
         guesp3(  1, 1 ) =  1.537465d0
         guesp1(  1, 2 ) = -1.060329d0
         guesp2(  1, 2 ) =  6.003788d0
         guesp3(  1, 2 ) =  1.570189d0
         guesp1(  8, 1 ) = -1.131128d0
         guesp2(  8, 1 ) =  6.002477d0
         guesp3(  8, 1 ) =  1.607311d0
         guesp1(  8, 2 ) =  1.137891d0
         guesp2(  8, 2 ) =  5.950512d0
         guesp3(  8, 2 ) =  1.598395d0
         guesp1( 17, 1 ) = -0.171591d0
         guesp2( 17, 1 ) =  6.000802d0
         guesp3( 17, 1 ) =  1.087502d0
         guesp1( 17, 2 ) = -0.013458d0
         guesp2( 17, 2 ) =  1.966618d0
         guesp3( 17, 2 ) =  2.292891d0

         gmais1(  1,  1, 1 ) =  0.000362d0
         gmais2(  1,  1, 1 ) =  0.385490d0
         gmais3(  1,  1, 1 ) =  6.230900d0
         gmais1(  1,  1, 2 ) =  0.009138d0
         gmais2(  1,  1, 2 ) =  0.227530d0
         gmais3(  1,  1, 2 ) =  2.211680d0
         gmais1(  1,  1, 3 ) =  0.007175d0
         gmais2(  1,  1, 3 ) =  3.013020d0
         gmais3(  1,  1, 3 ) =  2.225720d0
         gmais1(  1,  8, 1 ) =  0.038146d0
         gmais2(  1,  8, 1 ) =  0.564949d0
         gmais3(  1,  8, 1 ) =  4.128030d0
         gmais1(  1,  8, 2 ) = -0.042715d0
         gmais2(  1,  8, 2 ) =  0.518414d0
         gmais3(  1,  8, 2 ) =  4.058190d0
         gmais1(  1,  8, 3 ) = -0.039872d0
         gmais2(  1,  8, 3 ) =  0.342029d0
         gmais3(  1,  8, 3 ) =  0.529303d0
         gmais1(  1, 17, 1 ) = -0.247371d0
         gmais2(  1, 17, 1 ) =  1.520440d0
         gmais3(  1, 17, 1 ) =  2.176850d0
         gmais1(  1, 17, 2 ) =  0.245420d0
         gmais2(  1, 17, 2 ) =  1.649960d0
         gmais3(  1, 17, 2 ) =  2.223590d0
         gmais1(  1, 17, 3 ) = -0.009203d0
         gmais2(  1, 17, 3 ) =  0.121566d0
         gmais3(  1, 17, 3 ) =  0.841882d0
         gmais1(  8,  1, 1 ) =  0.038146d0
         gmais2(  8,  1, 1 ) =  0.564949d0
         gmais3(  8,  1, 1 ) =  4.128030d0
         gmais1(  8,  1, 2 ) = -0.042715d0
         gmais2(  8,  1, 2 ) =  0.518414d0
         gmais3(  8,  1, 2 ) =  4.058190d0
         gmais1(  8,  1, 3 ) = -0.039872d0
         gmais2(  8,  1, 3 ) =  0.342029d0
         gmais3(  8,  1, 3 ) =  0.529303d0
         gmais1(  8,  8, 1 ) = -6.797745d0
         gmais2(  8,  8, 1 ) =  0.326159d0
         gmais3(  8,  8, 1 ) =  1.211370d0
         gmais1(  8,  8, 2 ) =  6.912401d0
         gmais2(  8,  8, 2 ) =  0.320157d0
         gmais3(  8,  8, 2 ) =  1.209450d0
         gmais1(  8,  8, 3 ) =  0.074600d0
         gmais2(  8,  8, 3 ) =  1.268150d0
         gmais3(  8,  8, 3 ) =  2.579530d0
         gmais1( 17,  1, 1 ) =  0.017759d0
         gmais2( 17,  1, 1 ) =  1.022950d0
         gmais3( 17,  1, 1 ) =  4.543160d0
         gmais1( 17,  1, 2 ) = -0.001574d0
         gmais2( 17,  1, 2 ) =  0.369802d0
         gmais3( 17,  1, 2 ) =  7.538830d0
         gmais1( 17,  1, 3 ) = -0.065971d0
         gmais2( 17,  1, 3 ) =  0.591341d0
         gmais3( 17,  1, 3 ) =  1.989830d0
         gmais1( 17,  8, 1 ) =  0.017759d0
         gmais2( 17,  8, 1 ) =  1.022950d0
         gmais3( 17,  8, 1 ) =  4.543160d0
         gmais1( 17,  8, 2 ) = -0.001574d0
         gmais2( 17,  8, 2 ) =  0.369802d0
         gmais3( 17,  8, 2 ) =  7.538830d0
         gmais1( 17,  8, 3 ) = -0.065971d0
         gmais2( 17,  8, 3 ) =  0.591341d0
         gmais3( 17,  8, 3 ) =  1.989830d0
         gmais1( 17, 17, 1 ) =  0.000000d0
         gmais2( 17, 17, 1 ) =  0.000000d0
         gmais3( 17, 17, 1 ) =  0.000000d0
         gmais1( 17, 17, 2 ) =  0.000000d0
         gmais2( 17, 17, 2 ) =  0.000000d0
         gmais3( 17, 17, 2 ) =  0.000000d0
         gmais1( 17, 17, 3 ) =  0.000000d0
         gmais2( 17, 17, 3 ) =  0.000000d0
         gmais3( 17, 17, 3 ) =  0.000000d0

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   recalculate original nuclear repulsion                     */
!-----------------------------------------------------------------------

      bohr2ang   = au_length * 1.d+10
      au2ev      = 27.211608d0

!-----------------------------------------------------------------------
!     /*   main loop start                                            */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

         if ( mod( ibead-1, nprocs ) .ne. myrank ) cycle

         do i = 1, natom
         do j = i+1, natom

            ni = isymbol(i)
            nj = isymbol(j)

            if ( ni .gt. 20 ) cycle
            if ( nj .gt. 20 ) cycle

            core2 = core(ni)*core(nj)

            xi = x(i,ibead)
            yi = y(i,ibead)
            zi = z(i,ibead)

            call pbc_unfold_MPI &
     &         ( xi, yi, zi, mbox(1,i,ibead), mbox(2,i,ibead), &
     &           mbox(3,i,ibead) )

            xj = x(j,ibead)
            yj = y(j,ibead)
            zj = z(j,ibead)

            call pbc_unfold_MPI &
     &         ( xj, yj, zj, mbox(1,j,ibead), mbox(2,j,ibead), &
     &           mbox(3,j,ibead) )

            xij = xi - xj
            yij = yi - yj
            zij = zi - zj

            rij   = sqrt(xij*xij+yij*yij+zij*zij)
            rij2  = rij*rij
            rij3  = rij*rij2

            do ig = 1, 2

               a = core2 * guesp1(ni,ig) / au2ev / bohr2ang
               b = guesp2(ni,ig) * bohr2ang * bohr2ang
               c = guesp3(ni,ig) / bohr2ang

               re  = rij - c
               re2 = re*re

               factor = a * exp( - b*re2 )

               pot(ibead) = pot(ibead) - factor/rij

               factor_1 = - factor / rij3
               factor_2 = - factor / rij2 * 2.d0*b*re
               factor   = + factor_1 + factor_2

               fx(i,ibead) = fx(i,ibead) + xij * factor
               fy(i,ibead) = fy(i,ibead) + yij * factor
               fz(i,ibead) = fz(i,ibead) + zij * factor
               fx(j,ibead) = fx(j,ibead) - xij * factor
               fy(j,ibead) = fy(j,ibead) - yij * factor
               fz(j,ibead) = fz(j,ibead) - zij * factor

               a = core2 * guesp1(nj,ig) / au2ev / bohr2ang
               b = guesp2(nj,ig) * bohr2ang * bohr2ang
               c = guesp3(nj,ig) / bohr2ang

               re  = rij - c
               re2 = re*re

               factor = a * exp( - b*re2 )

               pot(ibead) = pot(ibead) - factor/rij

               factor_1 = - factor / rij3
               factor_2 = - factor / rij2 * 2.d0*b*re
               factor   = + factor_1 + factor_2

               fx(i,ibead) = fx(i,ibead) + xij * factor
               fy(i,ibead) = fy(i,ibead) + yij * factor
               fz(i,ibead) = fz(i,ibead) + zij * factor
               fx(j,ibead) = fx(j,ibead) - xij * factor
               fy(j,ibead) = fy(j,ibead) - yij * factor
               fz(j,ibead) = fz(j,ibead) - zij * factor

            end do

         end do
         end do

!-----------------------------------------------------------------------
!     /*   mais modification                                          */
!-----------------------------------------------------------------------

         do i = 1, natom
         do j = i+1, natom

            ni = isymbol(i)
            nj = isymbol(j)

            if ( ni .gt. 20 ) cycle
            if ( nj .gt. 20 ) cycle

            ni = isymbol(i)
            nj = isymbol(j)

            core2 = core(ni)*core(nj)

            xi = x(i,ibead)
            yi = y(i,ibead)
            zi = z(i,ibead)

            call pbc_unfold_MPI &
     &         ( xi, yi, zi, mbox(1,i,ibead), mbox(2,i,ibead), &
     &           mbox(3,i,ibead) )

            xj = x(j,ibead)
            yj = y(j,ibead)
            zj = z(j,ibead)

            call pbc_unfold_MPI &
     &         ( xj, yj, zj, mbox(1,j,ibead), mbox(2,j,ibead), &
     &           mbox(3,j,ibead) )

            xij = xi - xj
            yij = yi - yj
            zij = zi - zj

            rij   = sqrt(xij*xij+yij*yij+zij*zij)
            rij2  = rij*rij
            rij3  = rij*rij2

            do ig = 1, 3

               a = gmais1(ni,nj,ig)
               b = gmais2(ni,nj,ig) * bohr2ang * bohr2ang
               c = gmais3(ni,nj,ig) / bohr2ang

               re  = rij - c
               re2 = re*re

               factor = a * exp( - b * re2 )

               pot(ibead) = pot(ibead) + factor

               factor = factor * 2.d0*b*re / rij

               fx(i,ibead) = fx(i,ibead) + xij * factor
               fy(i,ibead) = fy(i,ibead) + yij * factor
               fz(i,ibead) = fz(i,ibead) + zij * factor
               fx(j,ibead) = fx(j,ibead) - xij * factor
               fy(j,ibead) = fy(j,ibead) - yij * factor
               fz(j,ibead) = fz(j,ibead) - zij * factor

            end do

         end do
         end do

      end do

      return
      end

