!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     energy and force from MOLPRO calculation
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine force_molpro
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, vir, fx, fy, fz, natom, nbead, &
     &   mbox, molpro_command, iounit

      use multistate_variables, only : &
     &   nstate

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   local integers   */
      integer          :: ibead, i, j, itest, ierr

!     /*   local characters   */
      character(len=3) :: char_num

!     /*   local integers   */
      integer, save    :: iset = 0

!     /*   local real numbers   */
      real(8)          :: xi, yi, zi

!     /*   local characters   */
      character(len=80) :: char_line

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      ierr = 0

      if ( iset .eq. 0 ) then

         do ibead = 1, nbead

!           /*   character   */
            call int3_to_char( ibead, char_num )

!           /*   remove old directory and create new directory   */
            call system ('mkdir -p ./' // char_num )

!           /*   copy data files   */
            call system ('cp -f *.dat ./' // char_num )

         end do

!-----------------------------------------------------------------------
!        /*   set molpro execution command                            */
!-----------------------------------------------------------------------

         open ( iounit, file = 'input.dat' )
         do
            read ( iounit, *, iostat=ierr )  char_line
            if ( char_line(1:16) .eq. '<molpro_command>' ) then
               read ( iounit, *, iostat=ierr ) molpro_command
               exit
            end if
            if ( ierr .ne. 0 ) exit
         end do
         close( iounit )

         if ( ierr .ne. 0 )  then
            open ( iounit, file = 'input_default.dat' )
            do
               read ( iounit, *, iostat=ierr )  char_line
               if ( char_line(1:16) .eq. '<molpro_command>' ) then
                  read ( iounit, *, iostat=ierr ) molpro_command
                  exit
               end if
            end do
            close( iounit )
         end if

         call error_handling ( ierr, 'subroutine force_molpro', 23 )

!-----------------------------------------------------------------------
!        /*   confirm molpro command                                  */
!-----------------------------------------------------------------------

         call system &
     &      ("echo '0' > test.out")
         call system &
     &      ("sleep 0.1")
         call system &
     &      ("which " // molpro_command // &
     &       " > /dev/null 2>&1 && echo '1' > test.out")

         open ( iounit, file = 'test.out' )

         read ( iounit, * ) itest

         close( iounit )

         if ( itest .eq. 0 ) then

            ierr = 1

            write( 6, '(a)' ) 'Error - Molpro command not found: ' // &
     &                         trim(molpro_command)
            write( 6, '(a)' )

         else

            ierr = 0

            write( 6, '(a)' ) 'Molpro command found: ' // &
     &                         trim(molpro_command)
            write( 6, '(a)' )

         end if

         call system('rm -f test.out')

         call error_handling ( ierr, 'subroutine force_molpro', 23 )

         iset = 1

      end if

!-----------------------------------------------------------------------
!        /*   make input                                              */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!        /*   character   */
         call int3_to_char( ibead, char_num )

!        /*   make input   */
         call iomolpro ( 1, ibead, char_num )

!        /*   remove uncessary files   */
         call system ('rm -f ' // char_num // '/*.xml' )
         call system ('rm -f ' // char_num // '/*.molpro' )

!        /*   run molpro   */
         call system ( 'cd ' // char_num // '; run_molpro.x; cd ..' )

!        /*   read output   */
         call iomolpro ( 2, ibead, char_num )

      end do

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

      if ( nstate .eq. 1 ) then

         do j = 1, nbead
         do i = 1, natom

            xi = x(i,j)
            yi = y(i,j)
            zi = z(i,j)

            call pbc_unfold &
     &         ( xi, yi, zi, mbox(1,i,j), mbox(2,i,j), mbox(3,i,j) )

            vir(1,1) = vir(1,1) + fx(i,j)*xi
            vir(1,2) = vir(1,2) + fx(i,j)*yi
            vir(1,3) = vir(1,3) + fx(i,j)*zi
            vir(2,1) = vir(2,1) + fy(i,j)*xi
            vir(2,2) = vir(2,2) + fy(i,j)*yi
            vir(2,3) = vir(2,3) + fy(i,j)*zi
            vir(3,1) = vir(3,1) + fz(i,j)*xi
            vir(3,2) = vir(3,2) + fz(i,j)*yi
            vir(3,3) = vir(3,3) + fz(i,j)*zi

         end do
         end do

      end if

      return
      end





!***********************************************************************
      subroutine iomolpro ( ioption, ibead, char_num )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, dipx, dipy, dipz, &
     &   natom, iounit

      use multistate_variables, only : &
     &   gxstate, gystate, gzstate, dxstate, dystate, dzstate, &
     &   dipxstate, dipystate, dipzstate, vstate, nstate

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   local integers   */
      integer          :: i, j, k, ibead, ioption

!     /*   local characters   */
      character(len=3) :: char_num

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 1 ) then

!        /*   file open   */
         open ( iounit, file = char_num // '/geometry.ini' )

!        /*   write position   */
         do i = 1, natom
            write(iounit,'(i8,6e24.16,3i4)') &
     &         1, x(i,ibead), y(i,ibead), z(i,ibead), &
     &         0.d0, 0.d0, 0.d0, 0, 0, 0
         end do

!        /*   file close   */
         close ( iounit )

      else if ( ioption .eq. 2 ) then

!        /*   file open   */
         open ( iounit, file = char_num // '/results.molpro' )

!        /*   single state   */
         if ( nstate .eq. 1 ) then

!           /*   read potential   */
            read( iounit, * ) pot(ibead)

!           /*   read gradient   */
            do i = 1, natom
               read( iounit, * ) fx(i,ibead), fy(i,ibead), fz(i,ibead)
            end do

!           /*   gradient -> force   */
            do i = 1, natom
               fx(i,ibead) = - fx(i,ibead)
               fy(i,ibead) = - fy(i,ibead)
               fz(i,ibead) = - fz(i,ibead)
            end do

!           /*   dipole moment   */
            read( iounit, * ) dipx(ibead), dipy(ibead), dipz(ibead)

!        /*   multiple states   */
         else if ( nstate .ge. 2 ) then

!           /*   zero clear   */
            vstate(:,:,ibead)    = 0.d0
            gxstate(:,:,:,ibead) = 0.d0
            gystate(:,:,:,ibead) = 0.d0
            gzstate(:,:,:,ibead) = 0.d0
            dxstate(:,:,:,ibead) = 0.d0
            dystate(:,:,:,ibead) = 0.d0
            dzstate(:,:,:,ibead) = 0.d0
            dipxstate(:,ibead)   = 0.d0
            dipystate(:,ibead)   = 0.d0
            dipzstate(:,ibead)   = 0.d0

!           /*   read potential   */
            do k = 1, nstate
            do j = 1, nstate
               read( iounit, * ) vstate(j,k,ibead)
            end do
            end do

!           /*   read gradient   */
            do k = 1, nstate
            do j = 1, nstate
            do i = 1, natom
               read( iounit, * ) gxstate(j,k,i,ibead), &
     &                           gystate(j,k,i,ibead), &
     &                           gzstate(j,k,i,ibead)
            end do
            end do
            end do

!           /*   read dipole moment   */
            do k = 1, nstate
               read( iounit, * ) &
     &            dipxstate(k,ibead), &
     &            dipystate(k,ibead), &
     &            dipzstate(k,ibead)
            end do

!           /*   read nonadiabatic coupling matrix elements   */
            do k = 1, nstate
            do j = 1, nstate
            do i = 1, natom
               read( iounit, * ) &
     &            dxstate(j,k,i,ibead), &
     &            dystate(j,k,i,ibead), &
     &            dzstate(j,k,i,ibead)
            end do
            end do
            end do

         end if

!        /*   file close   */
         close ( iounit )

      end if

      return
      end
