!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Mar 1, 2020 by M. Shiga
!      Description:     energy and force from tip4p force field
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      module tip4p_variables
!***********************************************************************

!     //   hydrogen labels
      integer, dimension(:,:), allocatable :: i_tip4p
      integer, dimension(:,:), allocatable :: j_tip4p

!     //   atomic positions
      real(8), dimension(:,:), allocatable :: x_tip4p
      real(8), dimension(:,:), allocatable :: y_tip4p
      real(8), dimension(:,:), allocatable :: z_tip4p

!     //   potentials
      real(8), dimension(:), allocatable :: pot_tip4p

!     //   forces
      real(8), dimension(:,:), allocatable :: fx_tip4p
      real(8), dimension(:,:), allocatable :: fy_tip4p
      real(8), dimension(:,:), allocatable :: fz_tip4p

!     //   virial
      real(8), dimension(:,:), allocatable :: vir_tip4p

!     //   tip4p constants
      real(8) :: gamma_tip4p

!     //   tip4p oxygen
      character(len=8) :: o_tip4p

!     //   tip4p oxygen
      character(len=8) :: h_tip4p

!     //   tip4p constants
      real(8), dimension(:), allocatable :: const_tip4p

!***********************************************************************
      end module tip4p_variables
!***********************************************************************





!***********************************************************************
      subroutine force_mm_tip4p_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   setup                                                      */
!-----------------------------------------------------------------------

      call force_mm_setup_MPI

!-----------------------------------------------------------------------
!     /*   main routine                                               */
!-----------------------------------------------------------------------

      call force_mm_main_tip4p_MPI

!-----------------------------------------------------------------------
!     /*   dipole moment                                              */
!-----------------------------------------------------------------------

      call mm_dipole_MPI

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

      call my_mpi_allreduce_md

      return
      end





!***********************************************************************
      subroutine force_mm_main_tip4p_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iboundary

      implicit none

!-----------------------------------------------------------------------
!     /*   linear bonds                                               */
!-----------------------------------------------------------------------

      call force_mm_lin_MPI

!-----------------------------------------------------------------------
!     /*   generalized linear bonds                                   */
!-----------------------------------------------------------------------

      call force_mm_genlin_MPI

!-----------------------------------------------------------------------
!     /*   angular bonds                                              */
!-----------------------------------------------------------------------

      call force_mm_angl_MPI

!-----------------------------------------------------------------------
!     /*   dihedral bonds                                             */
!-----------------------------------------------------------------------

      call force_mm_dih_MPI

!-----------------------------------------------------------------------
!     /*   improper bonds                                             */
!-----------------------------------------------------------------------

      call force_mm_improper_MPI

!-----------------------------------------------------------------------
!     /*   cmap of two dihedral bonds                                 */
!-----------------------------------------------------------------------

      call force_mm_cmap_MPI

!-----------------------------------------------------------------------
!     /*   lennard-jones                                              */
!-----------------------------------------------------------------------

      call force_mm_lj_MPI

!-----------------------------------------------------------------------
!     /*   lennard-jones pair                                         */
!-----------------------------------------------------------------------

      call force_mm_ljpair_MPI

!-----------------------------------------------------------------------
!     /*   buckingham                                                 */
!-----------------------------------------------------------------------

      call force_mm_buck_MPI

!-----------------------------------------------------------------------
!     /*   morse potential                                            */
!-----------------------------------------------------------------------

      call force_mm_morse_MPI

!-----------------------------------------------------------------------
!     /*   free     boundary  =  direct sum                           */
!     /*   periodic boundary  =  Ewald  sum                           */
!-----------------------------------------------------------------------

!     /*   tip4p   */
      call trans_tip4p_MPI( 1 )

      if ( iboundary .eq. 0 ) then

!        /*   direct sum   */
         call force_mm_coulomb_MPI

      else if ( iboundary .eq. 1 ) then

!        /*   Ewald sum   */
         call force_mm_ewald_MPI

      else if ( iboundary .eq. 2 ) then

!        /*   Ewald sum   */
         call force_mm_ewald_MPI

      end if

!     /*   tip4p   */
      call trans_tip4p_MPI( 2 )

      return
      end





!***********************************************************************
      subroutine trans_tip4p_MPI( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, vir, pot, species, natom, nbead, iounit, &
     &   myrank_main, nprocs_main, myrank_sub, nprocs_sub, myrank

      use tip4p_variables, only : &
     &   x_tip4p, y_tip4p, z_tip4p, fx_tip4p, fy_tip4p, fz_tip4p, &
     &   vir_tip4p, pot_tip4p, const_tip4p, gamma_tip4p, o_tip4p, &
     &   h_tip4p, i_tip4p, j_tip4p

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   integers
      integer :: i, j, k, l, imin_i, imin_j, ioption, ierr, noxygen

!     //   integer for initial settings
      integer, save :: iset = 0

!     //   oxygen-hydrogen cutoff distance
      real(8) :: rcut_tip4p = 4.d0

!     //   real numbers
      real(8) :: rcut2_tip4p, xik, yik, zik, rik, rik2, rmin_i, rmin_j, &
     &           xjk, yjk, zjk, xi, yi, zi, xj, yj, zj, xk, yk, zk, &
     &           c1, c2, c3

!     //   real numbers for tip4p scaling factor
      real(8) :: s_tip4p

!-----------------------------------------------------------------------
!     //   initial settings
!-----------------------------------------------------------------------

!     //   square of cutoff distance
      rcut2_tip4p = rcut_tip4p * rcut_tip4p

!     //   first time only
      if ( iset .eq. 0 ) then

!-----------------------------------------------------------------------
!        //   memory allocation
!-----------------------------------------------------------------------

!        //   memory allocation
         if ( .not. allocated( i_tip4p ) ) &
     &      allocate( i_tip4p(natom,nbead) )
         if ( .not. allocated( j_tip4p ) ) &
     &      allocate( j_tip4p(natom,nbead) )

!        //   memory allocation
         if ( .not. allocated( x_tip4p ) ) &
     &      allocate( x_tip4p(natom,nbead) )
         if ( .not. allocated( y_tip4p ) ) &
     &      allocate( y_tip4p(natom,nbead) )
         if ( .not. allocated( z_tip4p ) ) &
     &      allocate( z_tip4p(natom,nbead) )

!        //   memory allocation
         if ( .not. allocated( pot_tip4p ) ) &
     &      allocate( pot_tip4p(nbead) )

!        //   memory allocation
         if ( .not. allocated( fx_tip4p ) ) &
     &      allocate( fx_tip4p(natom,nbead) )
         if ( .not. allocated( fy_tip4p ) ) &
     &      allocate( fy_tip4p(natom,nbead) )
         if ( .not. allocated( fz_tip4p ) ) &
     &      allocate( fz_tip4p(natom,nbead) )

!        //   memory allocation
         if ( .not. allocated( vir_tip4p ) ) &
     &      allocate( vir_tip4p(3,3) )

!        //   memory allocation
         if ( .not. allocated( const_tip4p ) ) &
     &      allocate( const_tip4p(3) )

!-----------------------------------------------------------------------
!        //   read tip4p constants
!-----------------------------------------------------------------------

!        //   master rank
         if ( myrank .eq. 0 ) then

!           //   open file
            open ( iounit, file = 'mm.dat' )

!           //   tag
            call search_tag ( '<tip4p>', 7, iounit, ierr )

!           //   number of bonded charge pairs   */
            read( iounit, *, iostat=ierr ) o_tip4p, h_tip4p, gamma_tip4p

!           //   close file
            close( iounit )

!        //   master rank
         end if

!        //   broadcast
         call my_mpi_bcast_int_0( ierr )

!        //   stop on error
         call error_handling_MPI &
     &      ( ierr, 'subroutine force_mm_tip4p_MPI', 29 )

!        //   broadcast
         call my_mpi_bcast_char_0( o_tip4p, 8 )
         call my_mpi_bcast_char_0( h_tip4p, 8 )
         call my_mpi_bcast_real_0( gamma_tip4p )

!-----------------------------------------------------------------------
!        //   set tip4p constants
!-----------------------------------------------------------------------

!        //   s_tip4p = 0.15d0/(0.9572d0*2.d0*cos(104.52d0/360.d0*pi))

         s_tip4p = ( 1.d0 - gamma_tip4p ) / 2.d0

!        //   r_m =  c_1 * r_h1 + c_2 * r_h2 + c_3 * r_o

         const_tip4p(1) = s_tip4p
         const_tip4p(2) = s_tip4p
         const_tip4p(3) = 1.d0 - 2.d0 * s_tip4p

!-----------------------------------------------------------------------
!        //   i_tip4p and j_tip4p: hydrogen labels
!-----------------------------------------------------------------------

!        //   initialize
         i_tip4p(:,:) = 0
         j_tip4p(:,:) = 0

!        //   loop of beads
         do l = 1, nbead

!           //   only my bead
            if ( mod( l-1, nprocs_main ) .ne. myrank_main ) cycle

!           //   loop of atoms
            do k = 1, natom

!              //   only my atom
               if ( mod( k-1, nprocs_sub ) .ne. myrank_sub ) cycle

!              //   oxygen only
               if ( species(k)(1:8) .ne. o_tip4p(1:8) ) cycle

!              //   initialize
               rmin_i = rcut_tip4p
               rmin_j = rcut_tip4p

!              //   initialize
               imin_i = 0
               imin_j = 0

!              //   loop of atoms
               do i = 1, natom

!                 //   hydrogen only
                  if ( species(i)(1:8) .ne. h_tip4p(1:8) ) cycle

!                 //   oxygen-hydrogen vector
                  xik = x(i,l) - x(k,l)
                  yik = y(i,l) - y(k,l)
                  zik = z(i,l) - z(k,l)

!                 //   apply periodic boundary condition
                  call pbc_atom_MPI ( xik, yik, zik )

!                 //   square of oxygen-hydrogen distance
                  rik2 = xik*xik + yik*yik + zik*zik

!                 //   neglect ones beyond cutoff distance
                  if ( rik2 .gt. rcut2_tip4p ) cycle

!                 //   oxygen-hydrogen distance
                  rik = sqrt( rik2 )

!                 //   nearest hydrogen to oxygen
                  if      ( rik .lt. rmin_i ) then

!                     //   update list
                      imin_j = imin_i
                      imin_i = i

!                     //   update distance
                      rmin_j = rmin_i
                      rmin_i = rik

!                 //   second-nearest hydrogen to oxygen
                  else if ( rik .lt. rmin_j ) then

!                     //   update list
                      imin_j = i

!                     //   update distance
                      rmin_j = rik

!                 //   not near
                  end if

!              //   loop of atoms
               end do

!              //   first hydrogen
               i_tip4p(k,l) = imin_i

!              //   second hydrogen
               j_tip4p(k,l) = imin_j

!           //   loop of atoms
            end do

!        //   loop of beads
         end do

!        //   communicate
         call my_mpi_allreduce_int_2 ( i_tip4p, natom, nbead )
         call my_mpi_allreduce_int_2 ( j_tip4p, natom, nbead )

!        //   set complete
         iset = 1

!-----------------------------------------------------------------------
!        //   print tip4p information
!-----------------------------------------------------------------------

!        //   only master rank
         if ( myrank .eq. 0 ) then

!        //   open file
         open ( iounit, file = 'info_tip4p.out' )

!        //   titles
         write( iounit, '(a)' ) '--------------------------------'
         write( iounit, '(a)' ) '  number  bead     O     H     H'
         write( iounit, '(a)' ) '--------------------------------'

!        //   counter
         i = 0

!        //   initialize error flag
         ierr = 1

!        //   initialize number of oxygens
         noxygen = 1

!        //   loop of beads
         do l = 1, nbead

!        //   loop of atoms
         do k = 1, natom

!           //   oxygen only
            if ( species(k)(1:8) .ne. o_tip4p(1:8) ) cycle

!           //   counter
            i = i + 1

!           //   counter, bead, oxygen, hydrogen, hydrogen
            write( iounit, '(i8,4i6)' ) &
     &         i, l, k, i_tip4p(k,l), j_tip4p(k,l)

!           //   error flag
            ierr = ierr * min( i_tip4p(k,l)*j_tip4p(k,l), 1 )

!           //   number of oxygens
            noxygen = max( i, noxygen )

!        //   loop of atoms
         end do

!        //   loop of beads
         end do

!        //   error
         if ( ierr .eq. 0 ) then

            write( iounit, '(a)' )
            write( iounit, '(a)' ) &
     &         'Error found. Recheck input files.'
            write( iounit, '(a)' )

!        //   no error
         else

            write( iounit, '(a)' )
            write( iounit, '(a,i8)' ) &
     &         'Number of oxygen atoms:            ', noxygen
            write( iounit, '(a,i8)' ) &
     &         'Number of oxygen atoms per bead:   ', noxygen/nbead
            write( iounit, '(a,i8)' ) &
     &         'Number of hydrogen atoms:          ', noxygen*2
            write( iounit, '(a,i8)' ) &
     &         'Number of hydrogen atoms per bead: ', noxygen*2/nbead
            write( iounit, '(a)' ) 'No error.'
            write( iounit, '(a)' )

!        //   no error
         end if

!        //   close file
         close( iounit )

!        //   only master rank
         end if

!        //   synchronize all processes
         call my_mpi_barrier

!     //   first time only
      end if

!-----------------------------------------------------------------------
!     //   transform of coordinates
!-----------------------------------------------------------------------

!     //   transform of coordinates
      if  ( ioption .eq. 1 ) then

!-----------------------------------------------------------------------
!        //   save data
!-----------------------------------------------------------------------

!        //   save original coordinates
         x_tip4p(:,:) = x(:,:)
         y_tip4p(:,:) = y(:,:)
         z_tip4p(:,:) = z(:,:)

!        //   save original potential
         pot_tip4p(:) = pot(:)

!        //   save original forces
         fx_tip4p(:,:) = fx(:,:)
         fy_tip4p(:,:) = fy(:,:)
         fz_tip4p(:,:) = fz(:,:)

!        //   save original virial
         vir_tip4p(:,:) = vir(:,:)

!-----------------------------------------------------------------------
!        //   initialize forces and virial
!-----------------------------------------------------------------------

!        //   initialize potential
         pot(:) = 0.d0

!        //   initialize forces
         fx(:,:) = 0.d0
         fy(:,:) = 0.d0
         fz(:,:) = 0.d0

!        //   initialize virial
         vir(:,:) = 0.d0

!-----------------------------------------------------------------------
!        //   x, y, z: hydrogen at original position, oxygen at m-site
!-----------------------------------------------------------------------

!        //   constants
         c1 = const_tip4p(1)
         c2 = const_tip4p(2)
         c3 = const_tip4p(3)

!        //   loop of beads
         do l = 1, nbead

!           //   loop of atoms
            do k = 1, natom

!              //   for oxygen atoms
               if ( species(k)(1:8) .eq. o_tip4p(1:8) ) then

!                 //   oxygen atom k

                  xk = x_tip4p(k,l)
                  yk = y_tip4p(k,l)
                  zk = z_tip4p(k,l)

!                 //   hydrogen atom i

                  i = i_tip4p(k,l)

                  xik = x_tip4p(i,l) - xk
                  yik = y_tip4p(i,l) - yk
                  zik = z_tip4p(i,l) - zk

                  call pbc_atom_MPI ( xik, yik, zik )

                  xi = xik + xk
                  yi = yik + yk
                  zi = zik + zk

!                 //   hydrogen atom j

                  j = j_tip4p(k,l)

                  xjk = x_tip4p(j,l) - xk
                  yjk = y_tip4p(j,l) - yk
                  zjk = z_tip4p(j,l) - zk

                  call pbc_atom_MPI ( xjk, yjk, zjk )

                  xj = xjk + xk
                  yj = yjk + yk
                  zj = zjk + zk

!                 //   move from o site to m site

                  x(k,l) = c1*xi + c2*xj + c3*xk
                  y(k,l) = c1*yi + c2*yj + c3*yk
                  z(k,l) = c1*zi + c2*zj + c3*zk

!              //   for all other atoms
               end if

!           //   loop of atoms
            end do

!        //   loop of beads
         end do

!     //   transform of coordinates
      end if

!-----------------------------------------------------------------------
!     //   transform of forces
!-----------------------------------------------------------------------

!     //   transform of forces
      if  ( ioption .eq. 2 ) then

!        //   constants
         c1 = const_tip4p(1)
         c2 = const_tip4p(2)
         c3 = const_tip4p(3)

!-----------------------------------------------------------------------
!        //   f_tip4p: force on each atom
!-----------------------------------------------------------------------

!        //   loop of beads
         do l = 1, nbead

!           //   only my bead
            if ( mod( l-1, nprocs_main ) .ne. myrank_main ) cycle

!           //   loop of atoms
            do k = 1, natom

!              //   for m sites
               if ( species(k)(1:8) .eq. o_tip4p(1:8) ) then

!                 //   force correction to first hydrogen

                  i = i_tip4p(k,l)

                  fx_tip4p(i,l) = fx_tip4p(i,l) + c1*fx(k,l)
                  fy_tip4p(i,l) = fy_tip4p(i,l) + c1*fy(k,l)
                  fz_tip4p(i,l) = fz_tip4p(i,l) + c1*fz(k,l)

!                 //   force correction to second hydrogen

                  j = j_tip4p(k,l)

                  fx_tip4p(j,l) = fx_tip4p(j,l) + c2*fx(k,l)
                  fy_tip4p(j,l) = fy_tip4p(j,l) + c2*fy(k,l)
                  fz_tip4p(j,l) = fz_tip4p(j,l) + c2*fz(k,l)

!                 //   force correction to oxygen

                  fx_tip4p(k,l) = fx_tip4p(k,l) + c3*fx(k,l)
                  fy_tip4p(k,l) = fy_tip4p(k,l) + c3*fy(k,l)
                  fz_tip4p(k,l) = fz_tip4p(k,l) + c3*fz(k,l)

!              //   for all other atoms
               else

!                 //   forces
                  fx_tip4p(k,l) = fx_tip4p(k,l) + fx(k,l)
                  fy_tip4p(k,l) = fy_tip4p(k,l) + fy(k,l)
                  fz_tip4p(k,l) = fz_tip4p(k,l) + fz(k,l)

!              //   atoms
               end if

!           //   loop of atoms
            end do

!        //   loop of beads
         end do

!-----------------------------------------------------------------------
!        //   restore data
!-----------------------------------------------------------------------

!        //   restore coordinates
         x(:,:) = x_tip4p(:,:)
         y(:,:) = y_tip4p(:,:)
         z(:,:) = z_tip4p(:,:)

!        //   restore potential
         pot(:) = pot_tip4p(:) + pot(:)

!        //   restore forces
         fx(:,:) = fx_tip4p(:,:)
         fy(:,:) = fy_tip4p(:,:)
         fz(:,:) = fz_tip4p(:,:)

!        //   restore virial
         vir(:,:) = vir_tip4p(:,:) + vir(:,:)

!     //   transform of forces
      end if

      return
      end
