!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Mar 1, 2020 by M. Shiga
!      Description:     energy and force from lennard-jones pair
!
!///////////////////////////////////////////////////////////////////////



#ifdef nextver



!***********************************************************************
      subroutine force_mm_ljpair_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, boxinv, box, iboundary, &
     &   nbead, myrank_main, nprocs_main, myrank_sub, nprocs_sub

      use mm_variables, only : &
     &   rout_ljpair, rin_ljpair, eps_ljpair, sig_ljpair, bigboxinv, &
     &   bigbox, eps_ljbond, sig_ljbond, l_ljatom, n_list_ljpair, &
     &   nbox_ljpair, nljpair, epsrule_ljpair, sigrule_ljpair, &
     &   i_ljbond, j_ljbond, nljbond, i_ljpair, j_list_ljpair

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, n, jx, jy, jz, j2

      real(8) :: rout_ljpair2, xij, yij, zij, rij2, rij, rinv, eps, sig, &
     &           sr, sr2, sr6, sr12, u6, u12, uij, duij, fxi, fyi, fzi, &
     &           swf, dswf, aij, bij, cij, absa, absb, absc

      real(8) :: srmax = 2.d0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( nljpair .eq. 0 ) return

      rout_ljpair2 = rout_ljpair*rout_ljpair

!-----------------------------------------------------------------------
!     /*   extention of simulation box in real space sum              */
!-----------------------------------------------------------------------

      if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then

         absa = sqrt ( boxinv(1,1)*boxinv(1,1) &
     &               + boxinv(1,2)*boxinv(1,2) &
     &               + boxinv(1,3)*boxinv(1,3) )
         absb = sqrt ( boxinv(2,1)*boxinv(2,1) &
     &               + boxinv(2,2)*boxinv(2,2) &
     &               + boxinv(2,3)*boxinv(2,3) )
         absc = sqrt ( boxinv(3,1)*boxinv(3,1) &
     &               + boxinv(3,2)*boxinv(3,2) &
     &               + boxinv(3,3)*boxinv(3,3) )

         nbox_ljpair(1) = int(2.d0*rout_ljpair*absa) + 1
         nbox_ljpair(2) = int(2.d0*rout_ljpair*absb) + 1
         nbox_ljpair(3) = int(2.d0*rout_ljpair*absc) + 1

      end if

!-----------------------------------------------------------------------
!     /*   main loop : free boundary                                  */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) then

!        /*   make neighbor list   */
         call force_ljpair_makelist_MPI

         do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         do k = 1, nljpair-1

         if ( mod( k-1, nprocs_sub ) .ne. myrank_sub ) cycle

         do n = 1, n_list_ljpair(k,m)

            i = i_ljpair(k)
            j = j_list_ljpair(n,k,m)

            l = l_ljatom(j)

            if ( l .eq. 0 ) cycle

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            rij2 = xij*xij + yij*yij + zij*zij

            if ( rij2 .gt. rout_ljpair2 ) cycle

            rij     = sqrt(rij2)

            rinv    = 1.d0/rij

            call get_eps_ljpair &
     &         ( eps_ljpair(k), eps_ljpair(l), eps, epsrule_ljpair )

            if ( eps .eq. 0.d0 ) cycle

            call get_sig_ljpair &
     &         ( sig_ljpair(k), sig_ljpair(l), sig, sigrule_ljpair )

            sr      = sig*rinv

            if ( sr .gt. srmax ) cycle

            sr2     = sr*sr
            sr6     = sr2*sr2*sr2
            sr12    = sr6*sr6

            u6      = - 4.d0*eps*sr6
            u12     = + 4.d0*eps*sr12

!           /*   switching function   */
            call getswf( rij, rin_ljpair, rout_ljpair, swf, dswf )

!           /*   bare potential   */
            uij     = + u6 + u12

!           /*   bare potential gradient   */
            duij    = ( - 6.d0*u6*rinv - 12.d0*u12*rinv )*swf

!           /*   corrected potential   */
            pot(m)  = pot(m) + uij*swf

!           /*   corrected forces   */

            fxi = - uij*dswf*xij*rinv - duij*xij*rinv
            fyi = - uij*dswf*yij*rinv - duij*yij*rinv
            fzi = - uij*dswf*zij*rinv - duij*zij*rinv

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

         end do
         end do

         end do

!-----------------------------------------------------------------------
!     /*   main loop : periodic boundary with minimum image           */
!-----------------------------------------------------------------------

      else if(nbox_ljpair(1)*nbox_ljpair(2)*nbox_ljpair(3) .eq. 1) then

!        /*   make neighbor list   */
         call force_ljpair_makelist_MPI

         do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         do k = 1, nljpair-1

         if ( mod( k-1, nprocs_sub ) .ne. myrank_sub ) cycle

         do n = 1, n_list_ljpair(k,m)

            i = i_ljpair(k)
            j = j_list_ljpair(n,k,m)

            l = l_ljatom(j)

            if ( l .eq. 0 ) cycle

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            rij2 = xij*xij + yij*yij + zij*zij

            if ( rij2 .gt. rout_ljpair2 ) cycle

            rij     = sqrt(rij2)

            rinv    = 1.d0/rij

            call get_eps_ljpair &
     &         ( eps_ljpair(k), eps_ljpair(l), eps, epsrule_ljpair )

            if ( eps .eq. 0.d0 ) cycle

            call get_sig_ljpair &
     &         ( sig_ljpair(k), sig_ljpair(l), sig, sigrule_ljpair )

            sr      = sig*rinv

            if ( sr .gt. srmax ) cycle

            sr2     = sr*sr
            sr6     = sr2*sr2*sr2
            sr12    = sr6*sr6

            u6      = - 4.d0*eps*sr6
            u12     = + 4.d0*eps*sr12

!           /*   switching function   */
            call getswf( rij, rin_ljpair, rout_ljpair, swf, dswf )

!           /*   bare potential   */
            uij     = + u6 + u12

!           /*   bare potential gradient   */
            duij    = ( - 6.d0*u6*rinv - 12.d0*u12*rinv )*swf

!           /*   corrected potential   */
            pot(m)  = pot(m) + uij*swf

!           /*   corrected forces   */

            fxi = - uij*dswf*xij*rinv - duij*xij*rinv
            fyi = - uij*dswf*yij*rinv - duij*yij*rinv
            fzi = - uij*dswf*zij*rinv - duij*zij*rinv

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

         end do
         end do

         end do

!-----------------------------------------------------------------------
!     /*   main loop : periodic boundary                              */
!-----------------------------------------------------------------------

      else

         bigbox(:,1) = dble(nbox_ljpair(1))*box(:,1)
         bigbox(:,2) = dble(nbox_ljpair(2))*box(:,2)
         bigbox(:,3) = dble(nbox_ljpair(3))*box(:,3)

         call inv3 ( bigbox, bigboxinv )

         do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         do k = 1, nljpair

         if ( mod( k-1, nprocs_sub ) .ne. myrank_sub ) cycle

         do l = k, nljpair

            i = i_ljpair(k)
            j = i_ljpair(l)

            do jx = 0, nbox_ljpair(1)-1
            do jy = 0, nbox_ljpair(2)-1
            do jz = 0, nbox_ljpair(3)-1

               j2 = jx*jx + jy*jy + jz*jz

               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

               rij2 = xij*xij + yij*yij + zij*zij

               if ( rij2 .gt. rout_ljpair2 ) cycle

               rij     = sqrt(rij2)

               rinv    = 1.d0/rij

               call get_eps_ljpair &
     &            ( eps_ljpair(k), eps_ljpair(l), eps, epsrule_ljpair )

               if ( eps .eq. 0.d0 ) cycle

               call get_sig_ljpair &
     &            ( sig_ljpair(k), sig_ljpair(l), sig, sigrule_ljpair )

               sr      = sig*rinv

               if ( sr .gt. srmax ) cycle

               sr2     = sr*sr
               sr6     = sr2*sr2*sr2
               sr12    = sr6*sr6

               u6      = - 4.d0*eps*sr6
               u12     = + 4.d0*eps*sr12

!              /*   switching function   */
               call getswf( rij, rin_ljpair, rout_ljpair, swf, dswf )

!              /*   bare potential   */
               uij     = + u6 + u12

!              /*   bare potential gradient   */
               duij    = ( - 6.d0*u6*rinv - 12.d0*u12*rinv )*swf

!              /*   corrected potential   */
               pot(m)  = pot(m) + uij*swf*0.5d0

!              /*   corrected forces   */

               fxi = - uij*dswf*xij*rinv - duij*xij*rinv
               fyi = - uij*dswf*yij*rinv - duij*yij*rinv
               fzi = - uij*dswf*zij*rinv - duij*zij*rinv

               fx(i,m) = fx(i,m) + 0.5d0*fxi
               fy(i,m) = fy(i,m) + 0.5d0*fyi
               fz(i,m) = fz(i,m) + 0.5d0*fzi

               fx(j,m) = fx(j,m) - 0.5d0*fxi
               fy(j,m) = fy(j,m) - 0.5d0*fyi
               fz(j,m) = fz(j,m) - 0.5d0*fzi

               vir(1,1) = vir(1,1) + 0.5d0*fxi*xij
               vir(1,2) = vir(1,2) + 0.5d0*fxi*yij
               vir(1,3) = vir(1,3) + 0.5d0*fxi*zij
               vir(2,1) = vir(2,1) + 0.5d0*fyi*xij
               vir(2,2) = vir(2,2) + 0.5d0*fyi*yij
               vir(2,3) = vir(2,3) + 0.5d0*fyi*zij
               vir(3,1) = vir(3,1) + 0.5d0*fzi*xij
               vir(3,2) = vir(3,2) + 0.5d0*fzi*yij
               vir(3,3) = vir(3,3) + 0.5d0*fzi*zij

            end do
            end do
            end do

            if ( i .eq. j ) cycle

            do jx = 0, nbox_ljpair(1)-1
            do jy = 0, nbox_ljpair(2)-1
            do jz = 0, nbox_ljpair(3)-1

               j2 = jx*jx + jy*jy + jz*jz

               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

               xij = x(j,m) - x(i,m)
               yij = y(j,m) - y(i,m)
               zij = z(j,m) - z(i,m)

               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

               rij2 = xij*xij + yij*yij + zij*zij

               if ( rij2 .gt. rout_ljpair2 ) cycle

               rij     = sqrt(rij2)

               rinv    = 1.d0/rij

               call get_eps_ljpair &
     &            ( eps_ljpair(l), eps_ljpair(k), eps, epsrule_ljpair )

               if ( eps .eq. 0.d0 ) cycle

               call get_sig_ljpair &
     &            ( sig_ljpair(l), sig_ljpair(k), sig, sigrule_ljpair )

               sr      = sig*rinv

               if ( sr .gt. srmax ) cycle

               sr2     = sr*sr
               sr6     = sr2*sr2*sr2
               sr12    = sr6*sr6

               u6      = - 4.d0*eps*sr6
               u12     = + 4.d0*eps*sr12

!              /*   switching function   */
               call getswf( rij, rin_ljpair, rout_ljpair, swf, dswf )

!              /*   bare potential   */
               uij     = + u6 + u12

!              /*   bare potential gradient   */
               duij    = ( - 6.d0*u6*rinv - 12.d0*u12*rinv )*swf

!              /*   corrected potential   */
               pot(m)  = pot(m) + uij*swf*0.5d0

!              /*   corrected forces   */

               fxi = - uij*dswf*xij*rinv - duij*xij*rinv
               fyi = - uij*dswf*yij*rinv - duij*yij*rinv
               fzi = - uij*dswf*zij*rinv - duij*zij*rinv

               fx(j,m) = fx(j,m) + 0.5d0*fxi
               fy(j,m) = fy(j,m) + 0.5d0*fyi
               fz(j,m) = fz(j,m) + 0.5d0*fzi

               fx(i,m) = fx(i,m) - 0.5d0*fxi
               fy(i,m) = fy(i,m) - 0.5d0*fyi
               fz(i,m) = fz(i,m) - 0.5d0*fzi

               vir(1,1) = vir(1,1) + 0.5d0*fxi*xij
               vir(1,2) = vir(1,2) + 0.5d0*fxi*yij
               vir(1,3) = vir(1,3) + 0.5d0*fxi*zij
               vir(2,1) = vir(2,1) + 0.5d0*fyi*xij
               vir(2,2) = vir(2,2) + 0.5d0*fyi*yij
               vir(2,3) = vir(2,3) + 0.5d0*fyi*zij
               vir(3,1) = vir(3,1) + 0.5d0*fzi*xij
               vir(3,2) = vir(3,2) + 0.5d0*fzi*yij
               vir(3,3) = vir(3,3) + 0.5d0*fzi*zij

            end do
            end do
            end do

         end do
         end do

         end do

      end if

!-----------------------------------------------------------------------
!     /*   main loop: bonded pairs                                    */
!-----------------------------------------------------------------------

      do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         do k = 1, nljbond

            if ( mod( k-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i = i_ljbond(k)
            j = j_ljbond(k)

            n = l_ljatom(i)
            l = l_ljatom(j)

            if ( n .eq. 0 ) cycle
            if ( l .eq. 0 ) cycle

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            rij2 = xij*xij + yij*yij + zij*zij

            if ( rij2 .gt. rout_ljpair2 ) cycle

            rij     = sqrt(rij2)

            rinv    = 1.d0/rij

            call get_eps_ljpair &
     &         ( eps_ljpair(n), eps_ljpair(l), eps, epsrule_ljpair )

            if ( eps .eq. 0.d0 ) cycle

            call get_sig_ljpair &
     &         ( sig_ljpair(n), sig_ljpair(l), sig, sigrule_ljpair )

            eps     = - eps

            sr      = sig*rinv

            if ( sr .gt. srmax ) cycle

            sr2     = sr*sr
            sr6     = sr2*sr2*sr2
            sr12    = sr6*sr6

            u6      = - 4.d0*eps*sr6
            u12     = + 4.d0*eps*sr12

!           /*   switching function   */
            call getswf( rij, rin_ljpair, rout_ljpair, swf, dswf )

!           /*   bare potential   */
            uij     = + u6 + u12

!           /*   bare potential gradient   */
            duij    = ( - 6.d0*u6*rinv - 12.d0*u12*rinv )*swf

!           /*   corrected potential   */
            pot(m)  = pot(m) + uij*swf

!           /*   corrected forces   */

            fxi = - uij*dswf*xij*rinv - duij*xij*rinv
            fyi = - uij*dswf*yij*rinv - duij*yij*rinv
            fzi = - uij*dswf*zij*rinv - duij*zij*rinv

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

         end do

         do k = 1, nljbond

            if ( mod( k-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i = i_ljbond(k)
            j = j_ljbond(k)

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            rij2 = xij*xij + yij*yij + zij*zij

            if ( rij2 .gt. rout_ljpair2 ) cycle

            rij     = sqrt(rij2)

            rinv    = 1.d0/rij

            eps = + eps_ljbond(k)

            if ( eps .eq. 0.d0 ) cycle

            sig = + sig_ljbond(k)

            sr      = sig*rinv

            if ( sr .gt. srmax ) cycle

            sr2     = sr*sr
            sr6     = sr2*sr2*sr2
            sr12    = sr6*sr6

            u6      = - 4.d0*eps*sr6
            u12     = + 4.d0*eps*sr12

!           /*   switching function   */
            call getswf( rij, rin_ljpair, rout_ljpair, swf, dswf )

!           /*   bare potential   */
            uij     = + u6 + u12

!           /*   bare potential gradient   */
            duij    = ( - 6.d0*u6*rinv - 12.d0*u12*rinv )*swf

!           /*   corrected potential   */
            pot(m)  = pot(m) + uij*swf

!           /*   corrected forces   */

            fxi = - uij*dswf*xij*rinv - duij*xij*rinv
            fyi = - uij*dswf*yij*rinv - duij*yij*rinv
            fzi = - uij*dswf*zij*rinv - duij*zij*rinv

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_mm_ljpair_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, natom, myrank

      use mm_variables, only : &
     &   eps_ljpair, sig_ljpair, rin_ljpair, rout_ljpair, eps_ljbond, &
     &   sig_ljbond, epsrule_ljpair, sigrule_ljpair, &
     &   nljpair, nljbond, i_ljpair, i_ljbond, j_ljbond, l_ljatom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, l, ierr

!-----------------------------------------------------------------------
!     /*   combination rules                                          */
!-----------------------------------------------------------------------

!     //   default value
      ierr = 0

!     //   master rank
      if ( myrank .eq. 0 ) then

!        //   open file
         open ( iounit, file = 'mm.dat' )

!        /*   tag   */
         call search_tag ( '<ljrule>', 8, iounit, ierr )

!        /*   combination rules of lennard-jones pair interaction   */
         if ( ierr .eq. 0 ) then
            read( iounit, *, iostat=ierr ) &
     &         epsrule_ljpair, sigrule_ljpair
         end if

!        //   close file
         close( iounit )

         if ( ierr .ne. 0 ) then
            epsrule_ljpair = 'DEFAULT'
            sigrule_ljpair = 'DEFAULT'
         end if

!     //   master rank
      end if

!     //   broadcast
      call my_mpi_bcast_char_0 ( epsrule_ljpair, 12 )
      call my_mpi_bcast_char_0 ( sigrule_ljpair, 12 )

!-----------------------------------------------------------------------
!     /*   cut off distances                                          */
!-----------------------------------------------------------------------

!     //   default value
      ierr = 0

!     //   master rank
      if ( myrank .eq. 0 ) then

!        //   open file
         open ( iounit, file = 'mm.dat' )

!        /*   tag   */
         call search_tag ( '<ljcutoff>', 10, iounit, ierr )

!        /*   cut off distance, inside and outside   */
         read( iounit, *, iostat=ierr ) rin_ljpair, rout_ljpair

!        /*   close file   */
         close( iounit )

         if ( ierr .ne. 0 ) then
            rin_ljpair  = 20.d0
            rout_ljpair = 25.d0
         end if

!     //   master rank
      end if

!     //   broadcast
      call my_mpi_bcast_real_0 ( rin_ljpair )
      call my_mpi_bcast_real_0 ( rout_ljpair )

!-----------------------------------------------------------------------
!     /*   lennard-jones atoms                                        */
!-----------------------------------------------------------------------

!     //   default value
      ierr = 0

!     //   master rank
      if ( myrank .eq. 0 ) then

!        //   open file
         open ( iounit, file = 'mm.dat' )

!        /*   tag   */
         call search_tag ( '<ljpair>', 8, iounit, ierr )

!        /*   number of lennard jones atoms   */
         if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr ) nljpair

!        //   close file
         close( iounit )

!     //   master rank
      end if

!     //   broadcast
      call my_mpi_bcast_int_0( ierr )

      if ( ierr .ne. 0 ) then
         nljpair = 0
      else 
         call my_mpi_bcast_int_0( nljpair )
         if ( nljpair .gt. 0 ) then
            if ( .not. allocated(i_ljpair) ) &
     &         allocate( i_ljpair(nljpair) )
            if ( .not. allocated(eps_ljpair) ) &
     &         allocate( eps_ljpair(nljpair) )
            if ( .not. allocated(sig_ljpair) ) &
     &         allocate( sig_ljpair(nljpair) )
            if ( .not. allocated(l_ljatom) ) &
     &         allocate( l_ljatom(natom) )
         end if
      end if

!-----------------------------------------------------------------------
!     /*   bonded lennard jones pairs                                 */
!-----------------------------------------------------------------------

!     //   default value
      ierr = 0

!     //   master rank
      if ( myrank .eq. 0 ) then

!        /*   open file   */
         open ( iounit, file = 'mm.dat' )

!        /*   tag   */
         call search_tag ( '<ljbond>', 8, iounit, ierr )

!        /*   number of bonded lennard jones pairs   */
         if ( ierr .eq. 0 ) then
            read( iounit, *, iostat=ierr ) nljbond
         end if

!        /*   close file   */
         close( iounit )

!     //   master rank
      end if

!     //   broadcast
      call my_mpi_bcast_int_0( ierr )

      if ( ierr .ne. 0 ) then
         nljbond = 0
      else
         call my_mpi_bcast_int_0( nljbond )
         if ( nljbond .gt. 0 ) then
            if ( .not. allocated(i_ljbond) ) &
     &         allocate( i_ljbond(nljbond) )
            if ( .not. allocated(j_ljbond) ) &
     &         allocate( j_ljbond(nljbond) )
            if ( .not. allocated(eps_ljbond) ) &
     &         allocate( eps_ljbond(nljbond) )
            if ( .not. allocated(sig_ljbond) ) &
     &         allocate( sig_ljbond(nljbond) )
         end if
      end if

!-----------------------------------------------------------------------
!     /*   read non-bonded atoms                                      */
!-----------------------------------------------------------------------

!     //   default value
      ierr = 0

!     //   master rank
      if ( ( myrank .eq. 0 ) .and. ( nljpair .gt. 0 ) ) then

!        /*   open file   */
         open ( iounit, file = 'mm.dat' )

!        /*   tag   */
         call search_tag ( '<ljpair>', 8, iounit, ierr )

!        /*   skip a line   */
         read( iounit, *, iostat=ierr )

!           /*   atom, epsilon, sigma   */
            do l = 1, nljpair
               read( iounit, *, iostat=ierr ) &
     &            i_ljpair(l), eps_ljpair(l), sig_ljpair(l)
            end do

!        /*   close file   */
         close( iounit )

!     //   master rank
      end if

!     //   broadcast
      call my_mpi_bcast_int_0( ierr )

!     //   error handling
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_mm_ljpair_setup_MPI', 36 )

!     //   broadcast
      if ( nljpair .gt. 0 ) then
         call my_mpi_bcast_int_1 ( i_ljpair, nljpair )
         call my_mpi_bcast_real_1( eps_ljpair, nljpair )
         call my_mpi_bcast_real_1( sig_ljpair, nljpair )
      end if

!-----------------------------------------------------------------------
!     /*   read bonded pairs                                          */
!-----------------------------------------------------------------------

!     //   default value
      ierr = 0

!     //   master rank
      if ( ( myrank .eq. 0 ) .and. ( nljbond .gt. 0 ) ) then

!        /*   open file   */
         open ( iounit, file = 'mm.dat' )

!        /*   tag   */
         call search_tag ( '<ljbond>', 8, iounit, ierr )

!        /*   number of lennard jones   */
         read( iounit, *, iostat=ierr )

!        /*   atom 1, atom 2, epsilon, sigma, epsilon, sigma   */
         do l = 1, nljbond
            read( iounit, *, iostat=ierr ) &
     &         i_ljbond(l), j_ljbond(l), &
     &         eps_ljbond(l), sig_ljbond(l)
         end do

!        /*   close file   */
         close( iounit )

!     //   master rank
      end if

!     //   broadcast
      call my_mpi_bcast_int_0( ierr )

!     //   error handling
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_mm_ljpair_setup_MPI', 36 )

!     //   broadcast
      if ( nljbond .gt. 0 ) then
         call my_mpi_bcast_int_1 ( i_ljbond, nljbond )
         call my_mpi_bcast_int_1 ( j_ljbond, nljbond )
         call my_mpi_bcast_real_1 ( eps_ljbond, nljbond )
         call my_mpi_bcast_real_1 ( sig_ljbond, nljbond )
      end if

!-----------------------------------------------------------------------
!     /*   list of lj atom                                            */
!-----------------------------------------------------------------------

      ierr = 0

      if ( nljpair .gt. 0 ) l_ljatom(:) = 0

      do l = 1, nljpair

         i = i_ljpair(l)

         if ( ( i .lt. 1 ) .or. ( i .gt. natom ) ) then
            ierr = 1
         else
            l_ljatom(i) = l
         end if

      end do

!     //   error handling
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_mm_ljpair_setup_MPI', 36 )

      return
      end





!***********************************************************************
      subroutine force_ljpair_makelist_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, au_length, nbead, iounit, natom, nprocs_main, &
     &   myrank_main, nprocs_sub, myrank_sub, myrank

      use mm_variables, only : &
     &   x_list_ljpair, y_list_ljpair, z_list_ljpair, rcut_list_ljpair, &
     &   rcut2_list_ljpair, skin_ljpair, dmax_list_ljpair, rout_ljpair, &
     &   n_list_ljpair, j_list_ljpair, nmax_list_ljpair, nljpair, &
     &   i_ljpair

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize variables
      implicit none

!     //   visit flag
      integer, save :: iset = 0

!     //   integers
      integer :: i, j, k, l, m, n, ierr

!     //   real variables
      real(8) :: dx, dy, dz, d2, dmax, d2max, rx, ry, rz, r2

!c     //   real variables
!      real(8) :: bohr2ang = au_length * 1.d+10

!     //   real variables
      real(8) :: skin_ljpair_default = 4.d0

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

!     //   only first visit of this routine
      if ( iset .eq. 0 ) then

!        //   master rank only
         if ( myrank .eq. 0 ) then

!           //   open file
            open ( iounit, file = 'mm.dat' )

!           /*   tag   */
            call search_tag ( '<skin_ljpair>', 13, iounit, ierr )

!           /*   cut off distance   */
            read( iounit, *, iostat=ierr ) skin_ljpair

!           //   close file
            close( iounit )

!           //   default value
            if ( ierr .ne. 0 ) skin_ljpair = skin_ljpair_default

!        //   master rank only
         end if

!        //   communicate
         call my_mpi_bcast_int_0( ierr )

!        //   communicate
         call my_mpi_bcast_real_0( skin_ljpair )

!c        //   angstrom to bohr
!         skin_ljpair = skin_ljpair / bohr2ang

!        //   list cutoff radius
         rcut_list_ljpair = rout_ljpair + skin_ljpair

!        //   list cutoff radius squared
         rcut2_list_ljpair = rcut_list_ljpair * rcut_list_ljpair

!        //   maximum deviation allowed without updating neighbor list
         dmax_list_ljpair = 0.5d0 * skin_ljpair

!        //   memory allocation
         if ( .not. allocated(x_list_ljpair) ) &
     &      allocate( x_list_ljpair(natom,nbead) )
         if ( .not. allocated(y_list_ljpair) ) &
     &      allocate( y_list_ljpair(natom,nbead) )
         if ( .not. allocated(z_list_ljpair) ) &
     &      allocate( z_list_ljpair(natom,nbead) )

!        //   setup end
         iset = 1

!-----------------------------------------------------------------------
!     /*   deviation from last update of neighbor list                */
!-----------------------------------------------------------------------

!     //   from second visit to this routine
      else

!        //   initialize maximum deviation squared
         d2max = 0.d0

!        //   loop of beads and lj sites
         do k = 1, nbead
         do m = 1, nljpair

!           //   atom
            i = i_ljpair(m)

!           //   deviation
            dx = x(i,k) - x_list_ljpair(i,k)
            dy = y(i,k) - y_list_ljpair(i,k)
            dz = z(i,k) - z_list_ljpair(i,k)

!           //   apply boundary condition
            call pbc_atom_MPI( dx, dy, dz )

!           //   deviation squared
            d2 = dx*dx + dy*dy + dz*dz

!           //   maximum deviation squared
            d2max = max( d2, d2max )

!        //   loop of beads and lj sites
         end do
         end do

!        //   maximum deviation
         dmax = sqrt( d2max )

!        //   if maximum deviation is small, skip neighbor list update
         if ( dmax .lt. dmax_list_ljpair ) return

!     //   end of if statement
      end if

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( allocated( n_list_ljpair ) ) &
     &   deallocate( n_list_ljpair )
      if ( .not. allocated(n_list_ljpair) ) &
     &   allocate( n_list_ljpair(natom,nbead) )

!-----------------------------------------------------------------------
!     /*   number of atoms in neighbor list                           */
!-----------------------------------------------------------------------

!     //   clear
      n_list_ljpair(:,:) = 0

!     //   loop of beads
      do k = 1, nbead

!        /*   bead parallel   */
         if ( mod( k-1, nprocs_main ) .ne. myrank_main ) cycle

!        //   loop of site m
         do m = 1, nljpair

!           //   atom i
            i = i_ljpair(m)

!           /*   force parallel   */
            if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!           //   counter
            l = 0

!           //  loop of site n
            do n = m+1, nljpair

!              //   atom j
               j = i_ljpair(n)

!              //   separation of atoms i and j
               rx = x(j,k) - x(i,k)
               ry = y(j,k) - y(i,k)
               rz = z(j,k) - z(i,k)

!              //   apply boundary condition
               call pbc_atom_MPI( rx, ry, rz )

!              //   distance of atoms i and j squared
               r2 = rx*rx + ry*ry + rz*rz

!              //   if distance is smaller than list cutoff radius
               if ( r2 .lt. rcut2_list_ljpair ) then

!                 //   update counter
                  l = l + 1

!              //   end of if statement
               end if

!           //   loop of site n
            end do

!           //   number of j atoms in neighbor list for atom i, bead k
            n_list_ljpair(m,k) = l

!        //   loop of site m
         end do

!     //   loop of beads
      end do

!     //   communicate
      call my_mpi_allreduce_int_2( n_list_ljpair, nljpair, nbead )

!-----------------------------------------------------------------------
!     /*   maximum number of atoms                                    */
!-----------------------------------------------------------------------

!     //   counter
      nmax_list_ljpair = 0

!     //   loop of beads
      do k = 1, nbead

!        //   loop of site m
         do m = 1, nljpair

!           //   atom i
            i = i_ljpair(m)

!           //   maximum number of j atoms in neighbor list
            nmax_list_ljpair &
     &         = max( n_list_ljpair(m,k), nmax_list_ljpair )

!        //   loop of site m
         end do

!     //   loop of beads
      end do

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( allocated( j_list_ljpair ) ) deallocate( j_list_ljpair )
      allocate( j_list_ljpair(nmax_list_ljpair,nljpair,nbead) )

!-----------------------------------------------------------------------
!     /*   atoms in neighbor list                                     */
!-----------------------------------------------------------------------

!     //   clear
      j_list_ljpair(:,:,:) = 0

!     //   loop of beads
      do k = 1, nbead

!        /*   bead parallel   */
         if ( mod( k-1, nprocs_main ) .ne. myrank_main ) cycle

!        //   loop of site m
         do m = 1, nljpair

!           //   atom i
            i = i_ljpair(m)

!           /*   force parallel   */
            if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!           //   counter
            l = 0

!           //  loop of site n
            do n = m+1, nljpair

!              //   atom j
               j = i_ljpair(n)

!              //   separation of atoms i and j
               rx = x(j,k) - x(i,k)
               ry = y(j,k) - y(i,k)
               rz = z(j,k) - z(i,k)

!              //   apply boundary condition
               call pbc_atom_MPI( rx, ry, rz )

!              //   distance of atoms i and j squared
               r2 = rx*rx + ry*ry + rz*rz

!              //   if distance is smaller than list cutoff radius
               if ( r2 .lt. rcut2_list_ljpair ) then

!                 //   update counter
                  l = l + 1

!                 //   list of j atoms for atom i, bead k
                  j_list_ljpair(l,m,k) = j

               end if

!           //  loop of site m
            end do

!        //   loop of site n
         end do

!     //   loop of beads
      end do

!     //   communicate
      call my_mpi_allreduce_int_3 &
     &   ( j_list_ljpair, nmax_list_ljpair, natom, nbead )

!-----------------------------------------------------------------------
!     /*   update neighbor list                                       */
!-----------------------------------------------------------------------

      x_list_ljpair(:,:) = x(:,:)
      y_list_ljpair(:,:) = y(:,:)
      z_list_ljpair(:,:) = z(:,:)

      return
      end





!***********************************************************************
      subroutine force_ewald_makelist_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, au_length, natom, nbead, iounit, nprocs_main, &
     &   myrank_main, nprocs_sub, myrank_sub, myrank

      use mm_variables, only : &
     &   x_list_ewald, y_list_ewald, z_list_ewald, rcut_list_ewald, &
     &   rcut2_list_ewald, skin_ewald, dmax_list_ewald, rcut_ewald, &
     &   n_list_ewald, j_list_ewald, nmax_list_ewald, i_q, ncharge

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize variables
      implicit none

!     //   visit flag
      integer, save :: iset = 0

!     //   integers
      integer :: i, j, k, l, m, n, ierr

!     //   real variables
      real(8) :: dx, dy, dz, d2, dmax, d2max, rx, ry, rz, r2

!c     //   real variables
!      real(8) :: bohr2ang = au_length * 1.d+10

!     //   real variables
      real(8) :: skin_ewald_default = 4.d0

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

!     //   only first visit of this routine
      if ( iset .eq. 0 ) then

!        //   master rank only
         if ( myrank .eq. 0 ) then

!           //   open file
            open ( iounit, file = 'mm.dat' )

!           /*   tag   */
            call search_tag ( '<skin_ewald>', 12, iounit, ierr )

!           /*   cut off distance   */
            read( iounit, *, iostat=ierr ) skin_ewald

!           //   close file
            close( iounit )

!        //   master rank only
         end if

!        //   communicate
         call my_mpi_bcast_int_0( ierr )

!        //   default value
         if ( ierr .ne. 0 ) skin_ewald = skin_ewald_default

!        //   communicate
         call my_mpi_bcast_real_0( skin_ewald )

!c        //   angstrom to bohr
!         skin_ewald = skin_ewald / bohr2ang

!        //   list cutoff radius
         rcut_list_ewald = rcut_ewald + skin_ewald

!        //   list cutoff radius squared
         rcut2_list_ewald = rcut_list_ewald * rcut_list_ewald

!        //   maximum deviation allowed without updating neighbor list
         dmax_list_ewald = 0.5d0 * skin_ewald

!        //   memory allocation
         if ( .not. allocated(x_list_ewald) ) &
     &      allocate( x_list_ewald(natom,nbead) )
         if ( .not. allocated(y_list_ewald) ) &
     &      allocate( y_list_ewald(natom,nbead) )
         if ( .not. allocated(z_list_ewald) ) &
     &      allocate( z_list_ewald(natom,nbead) )

!        //   setup end
         iset = 1

!-----------------------------------------------------------------------
!     /*   deviation from last update of neighbor list                */
!-----------------------------------------------------------------------

!     //   from second visit to this routine
      else

!        //   initialize maximum deviation squared
         d2max = 0.d0

!        //   loop of beads and charges
         do k = 1, nbead
         do m = 1, ncharge

!           //   atom
            i = i_q(m)

!           //   deviation
            dx = x(i,k) - x_list_ewald(i,k)
            dy = y(i,k) - y_list_ewald(i,k)
            dz = z(i,k) - z_list_ewald(i,k)

!           //   apply boundary condition
            call pbc_atom_MPI( dx, dy, dz )

!           //   deviation squared
            d2 = dx*dx + dy*dy + dz*dz

!           //   maximum deviation squared
            d2max = max( d2, d2max )

!        //   loop of beads and charges
         end do
         end do

!        //   maximum deviation
         dmax = sqrt( d2max )

!        //   if maximum deviation is small, skip neighbor list update
         if ( dmax .lt. dmax_list_ewald ) return

!     //   end of if statement
      end if

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( allocated( n_list_ewald ) ) &
     &   deallocate( n_list_ewald )
      if ( .not. allocated(n_list_ewald) ) &
     &   allocate( n_list_ewald(ncharge,nbead) )

!-----------------------------------------------------------------------
!     /*   number of atoms in neighbor list                           */
!-----------------------------------------------------------------------

!     //   clear
      n_list_ewald(:,:) = 0

!     //   loop of beads
      do k = 1, nbead

!        /*   bead parallel   */
         if ( mod( k-1, nprocs_main ) .ne. myrank_main ) cycle

!        //   loop of charge m
         do m = 1, ncharge

!           //   atom i
            i = i_q(m)

!           /*   force parallel   */
            if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!           //   counter
            l = 0

!           //  loop of charge n
            do n = m+1, ncharge

!              //   atom j
               j = i_q(n)

!              //   separation of atoms i and j
               rx = x(j,k) - x(i,k)
               ry = y(j,k) - y(i,k)
               rz = z(j,k) - z(i,k)

!              //   apply boundary condition
               call pbc_atom_MPI( rx, ry, rz )

!              //   distance of atoms i and j squared
               r2 = rx*rx + ry*ry + rz*rz

!              //   if distance is smaller than list cutoff radius
               if ( r2 .lt. rcut2_list_ewald ) then

!                 //   update counter
                  l = l + 1

!              //   end of if statement
               end if

!           //   loop of charge n
            end do

!           //   number of j atoms in neighbor list for atom i, bead k
            n_list_ewald(i,k) = l

!        //   loop of charge m
         end do

!     //   loop of beads
      end do

!     //   communicate
      call my_mpi_allreduce_int_2( n_list_ewald, ncharge, nbead )

!-----------------------------------------------------------------------
!     /*   maximum number of atoms                                    */
!-----------------------------------------------------------------------

!     //   counter
      nmax_list_ewald = 0

!     //   loop of beads
      do k = 1, nbead

!        //   loop of charge m
         do m = 1, ncharge

!           //   atom i
            i = i_q(m)

!           //   maximum number of j atoms in neighbor list
            nmax_list_ewald &
     &         = max( n_list_ewald(i,k), nmax_list_ewald )

!        //   loop of charge m
         end do

!     //   loop of beads
      end do

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( allocated( j_list_ewald ) ) deallocate( j_list_ewald )
      allocate( j_list_ewald(nmax_list_ewald,ncharge,nbead) )

!-----------------------------------------------------------------------
!     /*   atoms in neighbor list                                     */
!-----------------------------------------------------------------------

!     //   clear
      j_list_ewald(:,:,:) = 0

!     //   loop of beads
      do k = 1, nbead

!        /*   bead parallel   */
         if ( mod( k-1, nprocs_main ) .ne. myrank_main ) cycle

!        //   loop of charge m
         do m = 1, ncharge

!           //   atom i
            i = i_q(m)

!           /*   force parallel   */
            if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!           //   counter
            l = 0

!           //  loop of charge n
            do n = m+1, ncharge

!              //   atom j
               j = i_q(n)

!              //   separation of atoms i and j
               rx = x(j,k) - x(i,k)
               ry = y(j,k) - y(i,k)
               rz = z(j,k) - z(i,k)

!              //   apply boundary condition
               call pbc_atom_MPI( rx, ry, rz )

!              //   distance of atoms i and j squared
               r2 = rx*rx + ry*ry + rz*rz

!              //   if distance is smaller than list cutoff radius
               if ( r2 .lt. rcut2_list_ewald ) then

!                 //   update counter
                  l = l + 1

!                 //   list of j atoms for atom i, bead k
                  j_list_ewald(l,i,k) = j

               end if

!           //  loop of charge n
            end do

!        //   loop of charge m
         end do

!     //   loop of beads
      end do

!     //   communicate
      call my_mpi_allreduce_int_3 &
     &   ( j_list_ewald, nmax_list_ewald, ncharge, nbead )

!-----------------------------------------------------------------------
!     /*   update neighbor list                                       */
!-----------------------------------------------------------------------

      x_list_ewald(:,:) = x(:,:)
      y_list_ewald(:,:) = y(:,:)
      z_list_ewald(:,:) = z(:,:)

      return
      end



#endif
