!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 25, 2018 by M. Shiga
!      Description:     energy and force from molecular mechanics
!
!///////////////////////////////////////////////////////////////////////



#ifndef nextver



!***********************************************************************
      subroutine force_mm_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   setup                                                      */
!-----------------------------------------------------------------------

      call force_mm_setup_MPI

!-----------------------------------------------------------------------
!     /*   main routine                                               */
!-----------------------------------------------------------------------

      call force_mm_main_MPI

!-----------------------------------------------------------------------
!     /*   dipole moment                                              */
!-----------------------------------------------------------------------

!      call mm_dipole_mol_MPI

      call mm_dipole_MPI

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

      call my_mpi_allreduce_md

      return
      end





!***********************************************************************
      subroutine force_mm_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iboundary

      use mm_variables, only : ewald_flag

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   only for the first access                                  */
!-----------------------------------------------------------------------

      if ( iset .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   linear bonds                                               */
!-----------------------------------------------------------------------

      call force_mm_lin_setup_MPI

!-----------------------------------------------------------------------
!     /*   general linear bonds                                       */
!-----------------------------------------------------------------------

      call force_mm_genlin_setup_MPI

!-----------------------------------------------------------------------
!     /*   angular bonds                                              */
!-----------------------------------------------------------------------

      call force_mm_angl_setup_MPI

!-----------------------------------------------------------------------
!     /*   dihedral bonds                                             */
!-----------------------------------------------------------------------

      call force_mm_dih_setup_MPI

!-----------------------------------------------------------------------
!     /*   improper bonds                                             */
!-----------------------------------------------------------------------

      call force_mm_improper_setup_MPI

!-----------------------------------------------------------------------
!     /*   cmap of two dihedral bonds                                 */
!-----------------------------------------------------------------------

      call force_mm_cmap_setup_MPI

!-----------------------------------------------------------------------
!     /*   lennard-jones                                              */
!-----------------------------------------------------------------------

      call force_mm_lj_setup_MPI

!-----------------------------------------------------------------------
!     /*   lennard-jones pair                                         */
!-----------------------------------------------------------------------

      call force_mm_ljpair_setup_MPI

!-----------------------------------------------------------------------
!     /*   buckingham                                                 */
!-----------------------------------------------------------------------

      call force_mm_buck_setup_MPI

!-----------------------------------------------------------------------
!     /*   morse potential                                            */
!-----------------------------------------------------------------------

      call force_mm_morse_setup_MPI

!-----------------------------------------------------------------------
!     /*   free     boundary  =  direct sum                           */
!     /*   periodic boundary  =  Ewald  sum                           */
!-----------------------------------------------------------------------

!     /*   free boundary   */
      if ( iboundary .eq. 0 ) then

!        /*   direct sum   */
         call force_mm_coulomb_setup_MPI

!     /*   periodic boundary   */
      else if ( iboundary .eq. 1 ) then

!        /*   Ewald sum   */
         call force_ewald_setup_MPI

!        /*   particle mesh Ewald   */
         if ( ewald_flag .eq. 1 ) call force_pmeewald_setup_MPI

!     /*   periodic boundary   */
      else if ( iboundary .eq. 2 ) then

!        /*   Ewald sum   */
         call force_ewald_setup_MPI

!        /*   particle mesh Ewald   */
         if ( ewald_flag .eq. 1 ) call force_pmeewald_setup_MPI

!     /*   end boundary condition   */
      end if

!-----------------------------------------------------------------------
!     /*   setup done                                                 */
!-----------------------------------------------------------------------

      iset = 1

      return
      end





!***********************************************************************
      subroutine force_mm_main_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iboundary

      use mm_variables, only : ewald_flag

      implicit none

!-----------------------------------------------------------------------
!     /*   linear bonds                                               */
!-----------------------------------------------------------------------

      call force_mm_lin_MPI

!-----------------------------------------------------------------------
!     /*   generalized linear bonds                                   */
!-----------------------------------------------------------------------

      call force_mm_genlin_MPI

!-----------------------------------------------------------------------
!     /*   angular bonds                                              */
!-----------------------------------------------------------------------

      call force_mm_angl_MPI

!-----------------------------------------------------------------------
!     /*   dihedral bonds                                             */
!-----------------------------------------------------------------------

      call force_mm_dih_MPI

!-----------------------------------------------------------------------
!     /*   improper bonds                                             */
!-----------------------------------------------------------------------

      call force_mm_improper_MPI

!-----------------------------------------------------------------------
!     /*   cmap of two dihedral bonds                                 */
!-----------------------------------------------------------------------

      call force_mm_cmap_MPI

!-----------------------------------------------------------------------
!     /*   lennard-jones                                              */
!-----------------------------------------------------------------------

      call force_mm_lj_MPI

!-----------------------------------------------------------------------
!     /*   lennard-jones pair                                         */
!-----------------------------------------------------------------------

      call force_mm_ljpair_MPI

!-----------------------------------------------------------------------
!     /*   buckingham                                                 */
!-----------------------------------------------------------------------

      call force_mm_buck_MPI

!-----------------------------------------------------------------------
!     /*   morse potential                                            */
!-----------------------------------------------------------------------

      call force_mm_morse_MPI

!-----------------------------------------------------------------------
!     /*   free     boundary  =  direct sum                           */
!     /*   periodic boundary  =  Ewald  sum                           */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) then

!        /*   direct sum   */
         call force_mm_coulomb_MPI

      else if ( iboundary .eq. 1 ) then

!        /*   Ewald sum   */
         if ( ewald_flag .ne. 1 ) call force_mm_ewald_MPI

!        /*   particle mesh Ewald   */
         if ( ewald_flag .eq. 1 ) call force_mm_pmeewald_MPI

      else if ( iboundary .eq. 2 ) then

!        /*   Ewald sum   */
         if ( ewald_flag .ne. 1 ) call force_mm_ewald_MPI

!        /*   particle mesh Ewald   */
         if ( ewald_flag .eq. 1 ) call force_mm_pmeewald_MPI

      end if

      return
      end





!***********************************************************************
      subroutine force_mm_lin_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, nbead, &
     &   myrank_main, nprocs_main, myrank_sub, nprocs_sub

      use mm_variables, only : &
     &   fc_lin, eq_lin, nlin, i_lin, j_lin

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: m, k, i, j

      real(8) :: xij, yij, zij, rij, dr, const, fxi, fyi, fzi

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( nlin .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         do k = 1, nlin

            if ( mod( k-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i = i_lin(k)
            j = j_lin(k)

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            dr = ( rij - eq_lin(k) )

            pot(m) = pot(m) + 0.5d0*fc_lin(k)*dr*dr

            const = - fc_lin(k)*dr/rij

            fxi = const*xij
            fyi = const*yij
            fzi = const*zij

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_mm_genlin_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, nbead, &
     &   myrank_main, nprocs_main, myrank_sub, nprocs_sub

      use mm_variables, only : &
     &   fc_genlin, eq_genlin, ngenlin, n_genlin, i_genlin, j_genlin

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: m, k, i, j

      real(8) :: xij, yij, zij, rij, dr, const, fxi, fyi, fzi

      real(8) :: drn = 1.d0
      real(8) :: drm = 0.d0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( ngenlin .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         do k = 1, ngenlin

            if ( mod( k-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i = i_genlin(k)
            j = j_genlin(k)

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            dr  = ( rij - eq_genlin(k) )

            if      ( n_genlin(k) .eq. 0 ) then
               drn    = 1.d0
               drm    = 0.d0
            else if ( n_genlin(k) .eq. 1 ) then
               drn    = dr
               drm    = 1.d0
            else if ( n_genlin(k) .eq. 2 ) then
               drn    = dr*dr
               drm    = dr
            else if ( n_genlin(k) .eq. 3 ) then
               drn    = dr*dr*dr
               drm    = dr*dr
            else if ( n_genlin(k) .eq. 4 ) then
               drn    = dr*dr*dr*dr
               drm    = dr*dr*dr
            else if ( n_genlin(k) .ge. 5 ) then
               drn    = dr**(n_genlin(k))
               drm    = dr**(n_genlin(k)-1)
            else if ( n_genlin(k) .le. -1 ) then
               drn    = dr**(n_genlin(k))
               drm    = dr**(n_genlin(k)-1)
            end if

            pot(m) = pot(m) + fc_genlin(k)*drn

            const = - n_genlin(k)*fc_genlin(k)*drm/rij

            fxi = const*xij
            fyi = const*yij
            fzi = const*zij

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_mm_angl_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, pi, nbead, &
     &   myrank_main, nprocs_main, myrank_sub, nprocs_sub

      use mm_variables, only : &
     &   fc_angl, eq_angl, i_angl, j_angl, k_angl, nangl

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, m, l

      real(8) :: xij, yij, zij, xkj, ykj, zkj, rij2, rkj2, rijk, &
     &           pijk, qijk, bijk, aijk, da, const, &
     &           fxi, fxj, fxk, fyi, fyj, fyk, fzi, fzj, fzk

      real(8) :: tiny_value = 1.d-4

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( nangl .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         do l = 1, nangl

            if ( mod( l-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i = i_angl(l)
            j = j_angl(l)
            k = k_angl(l)

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            xkj = x(k,m) - x(j,m)
            ykj = y(k,m) - y(j,m)
            zkj = z(k,m) - z(j,m)

            call pbc_atom_MPI ( xkj, ykj, zkj )

            rij2 = xij*xij + yij*yij + zij*zij
            rkj2 = xkj*xkj + ykj*ykj + zkj*zkj

            rijk = sqrt( rij2*rkj2 )

            pijk = xij*xkj + yij*ykj + zij*zkj

            qijk  = pijk/rijk

            qijk = max( qijk, -1.d0 )
            qijk = min( qijk,  1.d0 )

            bijk = acos( qijk )

            aijk = bijk*(180.d0/pi)

            da   = aijk - eq_angl(l)

            pot(m) = pot(m) + 0.5d0*fc_angl(l)*da*da

            if ( abs(bijk)    .lt. tiny_value ) cycle
            if ( abs(bijk-pi) .lt. tiny_value ) cycle
            if ( abs(bijk+pi) .lt. tiny_value ) cycle

            const = fc_angl(l)*da /sin(bijk) /rijk *(180.d0/pi)

            fxi = const*( xkj - pijk/rij2*xij )
            fxk = const*( xij - pijk/rkj2*xkj )
            fxj = - fxi - fxk

            fyi = const*( ykj - pijk/rij2*yij )
            fyk = const*( yij - pijk/rkj2*ykj )
            fyj = - fyi - fyk

            fzi = const*( zkj - pijk/rij2*zij )
            fzk = const*( zij - pijk/rkj2*zkj )
            fzj = - fzi - fzk

            fx(i,m) = fx(i,m) + fxi
            fx(j,m) = fx(j,m) + fxj
            fx(k,m) = fx(k,m) + fxk

            fy(i,m) = fy(i,m) + fyi
            fy(j,m) = fy(j,m) + fyj
            fy(k,m) = fy(k,m) + fyk

            fz(i,m) = fz(i,m) + fzi
            fz(j,m) = fz(j,m) + fzj
            fz(k,m) = fz(k,m) + fzk

            vir(1,1) = vir(1,1) + fxi*xij + fxk*xkj
            vir(1,2) = vir(1,2) + fxi*yij + fxk*ykj
            vir(1,3) = vir(1,3) + fxi*zij + fxk*zkj
            vir(2,1) = vir(2,1) + fyi*xij + fyk*xkj
            vir(2,2) = vir(2,2) + fyi*yij + fyk*ykj
            vir(2,3) = vir(2,3) + fyi*zij + fyk*zkj
            vir(3,1) = vir(3,1) + fzi*xij + fzk*xkj
            vir(3,2) = vir(3,2) + fzi*yij + fzk*ykj
            vir(3,3) = vir(3,3) + fzi*zij + fzk*zkj

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_mm_dih_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, nbead, &
     &   myrank_main, nprocs_main, myrank_sub, nprocs_sub

      use mm_variables, only : &
     &   v_dih, i_dih, j_dih, k_dih, l_dih, ndih, mu_dih, nu_dih

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, n, mu, nu, ii

      real(8) :: xij, yij, zij, xkj, ykj, zkj, xlj, ylj, zlj, &
     &           xijk, yijk, zijk, xjkl, yjkl, zjkl, rijk2, rjkl2, v, &
     &           rijkl2, rijk2inv, rjkl2inv, rijkl2inv, cos_phi, phi, &
     &           factor, fxi, fyi, fzi, fxj, fyj, fzj, fxk, fyk, fzk, &
     &           fxl, fyl, fzl

      real(8) :: tiny_value = 1.d-4

      real(8), dimension(0:12):: sinfactor

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( ndih .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         do n = 1, ndih

            if ( mod( n-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i = i_dih(n)
            j = j_dih(n)
            k = k_dih(n)
            l = l_dih(n)

!-----------------------------------------------------------------------

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            xkj = x(k,m) - x(j,m)
            ykj = y(k,m) - y(j,m)
            zkj = z(k,m) - z(j,m)

            call pbc_atom_MPI ( xkj, ykj, zkj )

            xlj = x(l,m) - x(j,m)
            ylj = y(l,m) - y(j,m)
            zlj = z(l,m) - z(j,m)

            call pbc_atom_MPI ( xlj, ylj, zlj )

!-----------------------------------------------------------------------

            xijk = yij*zkj - zij*ykj
            yijk = zij*xkj - xij*zkj
            zijk = xij*ykj - yij*xkj

            xjkl = ylj*zkj - zlj*ykj
            yjkl = zlj*xkj - xlj*zkj
            zjkl = xlj*ykj - ylj*xkj

            rijk2  = xijk*xijk + yijk*yijk + zijk*zijk
            rjkl2  = xjkl*xjkl + yjkl*yjkl + zjkl*zjkl

            rijkl2 = sqrt(rijk2*rjkl2)

            if ( abs(rijk2)  .lt. tiny_value ) cycle
            if ( abs(rjkl2)  .lt. tiny_value ) cycle
            if ( abs(rijkl2) .lt. tiny_value ) cycle

            rijk2inv  = 1.d0 / rijk2
            rjkl2inv  = 1.d0 / rjkl2
            rijkl2inv = 1.d0 / rijkl2

!-----------------------------------------------------------------------
!           /*   cos_phi = cos( phi )                                 */
!-----------------------------------------------------------------------

            cos_phi = ( xijk*xjkl + yijk*yjkl + zijk*zjkl ) * rijkl2inv

            cos_phi = max( cos_phi, -1.d0 )
            cos_phi = min( cos_phi,  1.d0 )

            phi = acos(cos_phi)

!-----------------------------------------------------------------------
!           /*   mu     =  cos(delta)  =  +1 or -1                    */
!           /*   delta  =  0 or pi                                    */
!-----------------------------------------------------------------------

            mu = mu_dih(n)
            nu = nu_dih(n)
            v  =  v_dih(n)

!-----------------------------------------------------------------------
!           /*   pot = 0.5 * v * ( cos ( nu*phi - delta ) + 1 )       */
!-----------------------------------------------------------------------

            pot(m)  = pot(m) + 0.5d0 * v * ( 1.d0 + cos(nu*phi)*mu )

!-----------------------------------------------------------------------
!           /*    sinfactor(n) = sin(n*phi) / sin(phi)                */
!-----------------------------------------------------------------------

            if      ( nu .eq. 0 ) then
               sinfactor(0) = 0.d0
            else if ( nu .eq. 1 ) then
               sinfactor(1) = 1.d0
            else if( nu .eq. 2 ) then
               sinfactor(2) = 2.d0*cos_phi
            else if( nu .eq. 3 ) then
               sinfactor(3) = 4.d0*cos_phi*cos_phi - 1.d0
            else if( nu .eq. 4 ) then
               sinfactor(4) = 4.d0*cos_phi*(2.d0*cos_phi*cos_phi - 1.d0)
            else if( nu .eq. 6 ) then
               sinfactor(6) = 2.d0 * ( 4.d0*cos_phi*cos_phi - 1.d0 ) &
     &                   * cos_phi * ( 4.d0*cos_phi*cos_phi - 3.d0 )
            else
               sinfactor(1) = 1.d0
               sinfactor(2) = 2.d0*cos_phi
               do ii = 3, nu
                  sinfactor(ii) = sinfactor(ii-2) &
     &                         - sin((ii-2)*phi)*sin(phi) &
     &                         + cos((ii-2)*phi)*cos(phi) &
     &                         + cos((ii-1)*phi)
               end do
            end if

            factor = - sinfactor(nu) * 0.5d0 * v * mu * nu

            fxi = factor * ( + ( ykj*zjkl - zkj*yjkl ) * rijkl2inv &
     &                    - ( ykj*zijk - zkj*yijk ) * cos_phi*rijk2inv )
            fyi = factor * ( + ( zkj*xjkl - xkj*zjkl ) * rijkl2inv &
     &                    - ( zkj*xijk - xkj*zijk ) * cos_phi*rijk2inv )
            fzi = factor * ( + ( xkj*yjkl - ykj*xjkl ) * rijkl2inv &
     &                    - ( xkj*yijk - ykj*xijk ) * cos_phi*rijk2inv )

            fxl = factor * ( + ( ykj*zijk - zkj*yijk ) * rijkl2inv &
     &                    - ( ykj*zjkl - zkj*yjkl ) * cos_phi*rjkl2inv )
            fyl = factor * ( + ( zkj*xijk - xkj*zijk ) * rijkl2inv &
     &                    - ( zkj*xjkl - xkj*zjkl ) * cos_phi*rjkl2inv )
            fzl = factor * ( + ( xkj*yijk - ykj*xijk ) * rijkl2inv &
     &                    - ( xkj*yjkl - ykj*xjkl ) * cos_phi*rjkl2inv )

            fxk = factor * ( - ( yij*zjkl - zij*yjkl ) * rijkl2inv &
     &                    - ( ylj*zijk - zlj*yijk ) * rijkl2inv &
     &                    + ( yij*zijk - zij*yijk ) * cos_phi*rijk2inv &
     &                    + ( ylj*zjkl - zlj*yjkl ) * cos_phi*rjkl2inv )
            fyk = factor * ( - ( zij*xjkl - xij*zjkl ) * rijkl2inv &
     &                    - ( zlj*xijk - xlj*zijk ) * rijkl2inv &
     &                    + ( zij*xijk - xij*zijk ) * cos_phi*rijk2inv &
     &                    + ( zlj*xjkl - xlj*zjkl ) * cos_phi*rjkl2inv )
            fzk = factor * ( - ( xij*yjkl - yij*xjkl ) * rijkl2inv &
     &                    - ( xlj*yijk - ylj*xijk ) * rijkl2inv &
     &                    + ( xij*yijk - yij*xijk ) * cos_phi*rijk2inv &
     &                    + ( xlj*yjkl - ylj*xjkl ) * cos_phi*rjkl2inv )

            fxj = - ( fxi + fxk + fxl )
            fyj = - ( fyi + fyk + fyl )
            fzj = - ( fzi + fzk + fzl )

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) + fxj
            fy(j,m) = fy(j,m) + fyj
            fz(j,m) = fz(j,m) + fzj

            fx(k,m) = fx(k,m) + fxk
            fy(k,m) = fy(k,m) + fyk
            fz(k,m) = fz(k,m) + fzk

            fx(l,m) = fx(l,m) + fxl
            fy(l,m) = fy(l,m) + fyl
            fz(l,m) = fz(l,m) + fzl

            vir(1,1) = vir(1,1) + fxi*xij + fxk*xkj + fxl*xlj
            vir(1,2) = vir(1,2) + fxi*yij + fxk*ykj + fxl*ylj
            vir(1,3) = vir(1,3) + fxi*zij + fxk*zkj + fxl*zlj
            vir(2,1) = vir(2,1) + fyi*xij + fyk*xkj + fyl*xlj
            vir(2,2) = vir(2,2) + fyi*yij + fyk*ykj + fyl*ylj
            vir(2,3) = vir(2,3) + fyi*zij + fyk*zkj + fyl*zlj
            vir(3,1) = vir(3,1) + fzi*xij + fzk*xkj + fzl*xlj
            vir(3,2) = vir(3,2) + fzi*yij + fzk*ykj + fzl*ylj
            vir(3,3) = vir(3,3) + fzi*zij + fzk*zkj + fzl*zlj

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_mm_coulomb_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, nbead, &
     &   myrank_main, nprocs_main, myrank_sub, nprocs_sub

      use mm_variables, only : &
     &   q, factor_bcp, ncharge, nbcp, i_q, i_bcp, j_bcp

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, n

      real(8) :: qi, qj, xij, yij, zij, rij, rinv, uij, duij, &
     &           fxi, fyi, fzi, factor

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( ncharge .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   main loop:  direct sum between all charges                 */
!-----------------------------------------------------------------------

      do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         n = 0

         do k = 1, ncharge-1
         do l = k+1, ncharge

            n = n + 1

            if ( mod( n-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i  = i_q(k)
            j  = i_q(l)

            qi = q(i)
            qj = q(j)

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            rinv = 1.d0/rij

            uij = + qi*qj*rinv

            pot(m) = pot(m) + uij

            duij = - uij*rinv

            fxi = - duij*xij*rinv
            fyi = - duij*yij*rinv
            fzi = - duij*zij*rinv

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

         end do
         end do

      end do

!-----------------------------------------------------------------------
!     /*   main loop:  subtract bonded charge pairs                   */
!-----------------------------------------------------------------------

      do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         do k = 1, nbcp

            if ( mod( k-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i  = i_bcp(k)
            j  = j_bcp(k)

            factor = factor_bcp(k)

            qi = q(i)
            qj = q(j)

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            rinv = 1.d0/rij

            uij = (factor - 1.d0) * qi*qj*rinv

            pot(m) = pot(m) + uij

            duij = - uij*rinv

            fxi = - duij*xij*rinv
            fyi = - duij*yij*rinv
            fzi = - duij*zij*rinv

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_mm_ewald_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use mm_variables, only : ncharge, nbox_ewald, ioption_ewald

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   set up Ewald parameters                                    */
!-----------------------------------------------------------------------

      if ( ncharge .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   real space contribution of Ewald sum                       */
!-----------------------------------------------------------------------

      if ( nbox_ewald(1)*nbox_ewald(2)*nbox_ewald(3) .eq. 1 ) then
         call force_ewald_rs_pair_MPI
      else
         call force_ewald_rs_MPI
      end if

!-----------------------------------------------------------------------
!     /*   Fourier space contribution of Ewald sum                    */
!-----------------------------------------------------------------------

      call force_ewald_fs_MPI

!-----------------------------------------------------------------------
!     /*   self contribution of Ewald sum                             */
!-----------------------------------------------------------------------

      call force_ewald_self_MPI

!-----------------------------------------------------------------------
!     /*   charged system contribution of Ewald sum                   */
!-----------------------------------------------------------------------

      call force_ewald_charge_MPI

!-----------------------------------------------------------------------
!     /*   dipole contribution of Ewald sum                           */
!-----------------------------------------------------------------------

      if ( ioption_ewald .eq. 1 ) call force_ewald_dipole_MPI

      return
      end





!***********************************************************************
      subroutine force_mm_pmeewald_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use mm_variables, only : ncharge, nbox_ewald, ioption_ewald

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   set up Ewald parameters                                    */
!-----------------------------------------------------------------------

      if ( ncharge .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   real space contribution of Ewald sum                       */
!-----------------------------------------------------------------------

      if ( nbox_ewald(1)*nbox_ewald(2)*nbox_ewald(3) .eq. 1 ) then
         call force_ewald_rs_pair_MPI
      else
         call force_ewald_rs_MPI
      end if

!-----------------------------------------------------------------------
!     /*   Fourier space contribution of Ewald sum                    */
!-----------------------------------------------------------------------

      call force_pmeewald_fs_MPI

!-----------------------------------------------------------------------
!     /*   self contribution of Ewald sum                             */
!-----------------------------------------------------------------------

      call force_ewald_self_MPI

!-----------------------------------------------------------------------
!     /*   charged system contribution of Ewald sum                   */
!-----------------------------------------------------------------------

      call force_ewald_charge_MPI

!-----------------------------------------------------------------------
!     /*   dipole contribution of Ewald sum                           */
!-----------------------------------------------------------------------

      if ( ioption_ewald .eq. 1 ) call force_ewald_dipole_MPI

      return
      end





!***********************************************************************
      subroutine force_ewald_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, box, boxinv, volume, iounit, natom

      use mm_variables, only : &
     &   s_ewald, q, factor_bcp, eps_ewald, ratio_ewald, alpha_ewald, &
     &   rcut_ewald, lmax_ewald, nbox_ewald, i_q, nbcp, i_bcp, j_bcp, &
     &   ncharge, ioption_ewald, ewald_type, pme_mesh_ewald, nfft_in, &
     &   ewald_flag

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, k, n, ierr

      real(8) :: snew, sold, sdif, absx, absy, absz, absa, absb, absc

      character(len=80) :: char_line

!-----------------------------------------------------------------------
!     /*   set up charges                                             */
!-----------------------------------------------------------------------

      open (iounit,file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<charges>', 9, iounit, ierr )

!        /*   number of charges   */
         if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr ) ncharge

      close(iounit)

      if ( ierr .ne. 0 ) ncharge = 0

      if ( .not. allocated( q ) ) &
     &   allocate( q(natom) )

      q(:) = 0.d0

      if ( ncharge .eq. 0 ) return

      if ( .not. allocated( i_q ) ) &
     &   allocate( i_q(ncharge) )

      open (iounit,file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<charges>', 9, iounit, ierr )

!        /*   number of charges   */
         read( iounit, *, iostat=ierr ) ncharge

!        /*   atom 1, atom 2, equilibrium, force constant   */
         do k = 1, ncharge
            read( iounit, *, iostat=ierr ) i, q(i)
            i_q(k) = i
         end do

      close(iounit)

!-----------------------------------------------------------------------
!     /*   set up bonded charge pairs                                 */
!-----------------------------------------------------------------------

      open (iounit,file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<nbcp>', 6, iounit, ierr )

!        /*   number of bonded charge pairs   */
         read( iounit, *, iostat=ierr ) nbcp

      close(iounit)

      if ( nbcp .ge. 1 ) then

         if ( .not. allocated(  i_bcp ) ) &
     &      allocate(  i_bcp(nbcp))
         if ( .not. allocated(  j_bcp ) ) &
     &      allocate(  j_bcp(nbcp))
         if ( .not. allocated(  factor_bcp ) ) &
     &      allocate(  factor_bcp(nbcp))

         open (iounit, file = 'mm.dat')

!           /*   tag   */
            call search_tag ( '<nbcp>', 6, iounit, ierr )

!           /*   number of bonded charge pairs   */
            read( iounit, *, iostat=ierr ) nbcp

            do k = 1, nbcp
               read( iounit, *, iostat=ierr ) &
     &            i_bcp(k), j_bcp(k), factor_bcp(k)
            end do

         close(iounit)

      end if

!-----------------------------------------------------------------------
!     /*                                                              */
!     /*   Ewald parameters                                           */
!     /*                                                              */
!     /*      eps_ewald    =  required accuracy                       */
!     /*      ratio_ewald  =  ratio of calculation time               */
!     /*                      real space / Fourier space              */
!     /*                                                              */
!     /*   ratio_ewald depends on the error function routine          */
!     /*                                                              */
!-----------------------------------------------------------------------

      open (iounit, file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<ewald>', 7, iounit, ierr )

         if ( ierr .eq. 0 ) then
            read( iounit, '(a)' ) char_line
            read( char_line, *, iostat=ierr ) &
     &         eps_ewald, ratio_ewald, ioption_ewald
            if ( ierr .ne. 0 ) then
               read( char_line, *, iostat=ierr ) &
     &            eps_ewald, ratio_ewald
               ioption_ewald = 0
               if ( ierr .ne. 0 ) then
                  eps_ewald    =  1.d-08
                  ratio_ewald  =  4.d+00
                  ioption_ewald =  0
               end if
            end if
         else
            eps_ewald    =  1.d-08
            ratio_ewald  =  4.d+00
            ioption_ewald =  0
         end if

      close(iounit)

!-----------------------------------------------------------------------
!     /*   parameter s:  solve exp(-s*s)/(s*s) = eps_ewald            */
!-----------------------------------------------------------------------

      snew = 0.d0

      do i = 1, 1000
         sold = snew
         snew = exp(-snew)*(snew+1.d0)/(eps_ewald+exp(-snew))
         sdif = abs(sold/snew - 1.d0)
         if ( sdif .lt. 1.d-15 ) exit
      end do

      s_ewald = sqrt(snew)

!-----------------------------------------------------------------------
!     /*   alpha:  exponent of fictitious Gaussian charge             */
!-----------------------------------------------------------------------

      alpha_ewald = (ratio_ewald*natom*pi**3/volume**2)**(1.d0/6.d0)

!-----------------------------------------------------------------------
!     /*   rcut:  cut off distance of real space sum                  */
!-----------------------------------------------------------------------

      rcut_ewald = s_ewald/alpha_ewald

!-----------------------------------------------------------------------
!     /*   lmax:  cut off in Fourier space sum                        */
!     /*          kmax = 2*pi/boxl*lmax                               */
!-----------------------------------------------------------------------

      absx = sqrt ( box(1,1)*box(1,1) &
     &            + box(2,1)*box(2,1) &
     &            + box(3,1)*box(3,1) )
      absy = sqrt ( box(1,2)*box(1,2) &
     &            + box(2,2)*box(2,2) &
     &            + box(3,2)*box(3,2) )
      absz = sqrt ( box(1,3)*box(1,3) &
     &            + box(2,3)*box(2,3) &
     &            + box(3,3)*box(3,3) )

      lmax_ewald(1) = int(s_ewald*absx*alpha_ewald/pi) + 1
      lmax_ewald(2) = int(s_ewald*absy*alpha_ewald/pi) + 1
      lmax_ewald(3) = int(s_ewald*absz*alpha_ewald/pi) + 1

!-----------------------------------------------------------------------
!     /*   extention of simulation box in real space sum              */
!-----------------------------------------------------------------------

      absa = sqrt ( boxinv(1,1)*boxinv(1,1) &
     &            + boxinv(1,2)*boxinv(1,2) &
     &            + boxinv(1,3)*boxinv(1,3) )
      absb = sqrt ( boxinv(2,1)*boxinv(2,1) &
     &            + boxinv(2,2)*boxinv(2,2) &
     &            + boxinv(2,3)*boxinv(2,3) )
      absc = sqrt ( boxinv(3,1)*boxinv(3,1) &
     &            + boxinv(3,2)*boxinv(3,2) &
     &            + boxinv(3,3)*boxinv(3,3) )

      nbox_ewald(1) = int(2.d0*rcut_ewald*absa) + 1
      nbox_ewald(2) = int(2.d0*rcut_ewald*absb) + 1
      nbox_ewald(3) = int(2.d0*rcut_ewald*absc) + 1

!-----------------------------------------------------------------------
!     /*   particle mesh ewald                                        */
!-----------------------------------------------------------------------

!     /*   read ewald_type: STANDARD or PME   */
      open ( iounit, file = 'mm.dat' )

!     /*   tag   */
      call search_tag ( '<ewald_type>', 12, iounit, ierr )

      if ( ierr .eq. 0 ) then
         read( iounit, *, iostat=ierr ) ewald_type
         if ( ierr .ne. 0 ) ewald_type = 'STANDARD'
      else
         ewald_type = 'STANDARD'
      end if

!     /*   read ewald_type: STANDARD or PME   */
      close( iounit )

!     /*   set flag   */

      if      ( ewald_type(1:4) .eq. 'PME ' ) then
         ewald_flag = 1
      else if ( ewald_type(1:9) .eq. 'STANDARD ' ) then
         ewald_flag = 0
      else
         ewald_flag = 0
      end if

!     /*   read number of FFT meshes per 1 bohr   */
      open ( iounit, file = 'mm.dat' )

!     /*   tag   */
      call search_tag ( '<pme_mesh_ewald>', 16, iounit, ierr )

      if ( ierr .eq. 0 ) then
         read( iounit, *, iostat=ierr ) pme_mesh_ewald
         if ( ierr .ne. 0 ) pme_mesh_ewald = 1.09d0
      else
         pme_mesh_ewald = 1.09d0
      end if

!     /*   read number of FFT meshes per 1 bohr   */
      close( iounit )

!     /*   set number of FFT meshes, nfft_in   */

      do i = 1, 3
         do n = 0, 20
            if ( pme_mesh_ewald*box(i,i) >= 2**n   .and. &
     &           pme_mesh_ewald*box(i,i) < 2**(n+1) ) exit
         end do
         nfft_in(i) = 2**(n+1)
      end do

      return
      end





!***********************************************************************
      subroutine force_ewald_rs_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pot, fx, fy, fz, vir, box, nbead, &
     &   myrank_main, nprocs_main, myrank_sub, nprocs_sub

      use mm_variables, only : &
     &   factor_bcp, q, rcut_ewald, alpha_ewald, bigbox, bigboxinv, &
     &   nbox_ewald, i_q, i_bcp, j_bcp, nbcp, ncharge

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, n, jx, jy, jz, j2

      real(8) :: qiqj, xij, yij, zij, aij, bij, cij, uij, duij, qi, qj, &
     &           r2, r, rinv, rinv2, rinv3, ar, erf_0, erf_1, erfc_ar, &
     &           pot_m, factor, fxi, fyi, fzi, rcut_ewald2, rij

!-----------------------------------------------------------------------
!     /*   parameters                                                 */
!-----------------------------------------------------------------------

      bigbox(:,1) = dble(nbox_ewald(1))*box(:,1)
      bigbox(:,2) = dble(nbox_ewald(2))*box(:,2)
      bigbox(:,3) = dble(nbox_ewald(3))*box(:,3)

      call inv3 ( bigbox, bigboxinv )

      rcut_ewald2 = rcut_ewald*rcut_ewald

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         pot_m = 0.d0

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         n = 0

         do k = 1, ncharge
         do l = 1, ncharge

            n = n + 1

            if ( mod( n-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i = i_q(k)
            j = i_q(l)

            qiqj = q(i)*q(j)

            if ( qiqj .eq. 0.d0 ) cycle

            do jx = 0, nbox_ewald(1)-1
            do jy = 0, nbox_ewald(2)-1
            do jz = 0, nbox_ewald(3)-1

               j2 = jx*jx + jy*jy + jz*jz

               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

               r2 = xij*xij + yij*yij + zij*zij

               if ( r2 .gt. rcut_ewald2 ) cycle

               r = sqrt(r2)

               rinv  = 1.d0/r
               rinv2 = rinv*rinv
               rinv3 = rinv*rinv2

               ar = alpha_ewald*r

               erfc_ar = 1.d0 - erf_0(ar)

               pot_m = pot_m + qiqj*erfc_ar*rinv

               factor = erfc_ar*rinv3 + alpha_ewald*erf_1(ar)*rinv2
               factor = qiqj*factor

               fxi = factor*xij
               fyi = factor*yij
               fzi = factor*zij

               fx(i,m) = fx(i,m) + fxi
               fy(i,m) = fy(i,m) + fyi
               fz(i,m) = fz(i,m) + fzi

               vir(1,1) = vir(1,1) + 0.5d0*fxi*xij
               vir(1,2) = vir(1,2) + 0.5d0*fxi*yij
               vir(1,3) = vir(1,3) + 0.5d0*fxi*zij
               vir(2,1) = vir(2,1) + 0.5d0*fyi*xij
               vir(2,2) = vir(2,2) + 0.5d0*fyi*yij
               vir(2,3) = vir(2,3) + 0.5d0*fyi*zij
               vir(3,1) = vir(3,1) + 0.5d0*fzi*xij
               vir(3,2) = vir(3,2) + 0.5d0*fzi*yij
               vir(3,3) = vir(3,3) + 0.5d0*fzi*zij

            end do
            end do
            end do

         end do
         end do

         pot(m) = pot(m) + 0.5d0*pot_m

      end do

!-----------------------------------------------------------------------
!     /*   main loop:  subtract bonded charge pairs                   */
!-----------------------------------------------------------------------

      do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         do k = 1, nbcp

            if ( mod( k-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i  = i_bcp(k)
            j  = j_bcp(k)

            factor = factor_bcp(k)

            qi = q(i)
            qj = q(j)

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            rinv = 1.d0/rij

            uij = (factor - 1.d0) * qi*qj*rinv

            pot(m) = pot(m) + uij

            duij = - uij*rinv

            fxi = - duij*xij*rinv
            fyi = - duij*yij*rinv
            fzi = - duij*zij*rinv

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_ewald_rs_pair_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pot, fx, fy, fz, vir, nbead, &
     &   myrank_main, nprocs_main, myrank_sub, nprocs_sub

      use mm_variables, only : &
     &   factor_bcp, q, rcut_ewald, alpha_ewald, i_q, i_bcp, j_bcp, &
     &   nbcp, ncharge

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, n

      real(8) :: rcut_ewald2, qiqj, xij, yij, zij, r2, r, rinv, &
     &           rinv2, rinv3, ar, erfc_ar, erf_0, erf_1, factor, uij, &
     &           duij, fxi, fyi, fzi, rij, qi, qj

!-----------------------------------------------------------------------
!     /*   parameters                                                 */
!-----------------------------------------------------------------------

      rcut_ewald2 = rcut_ewald*rcut_ewald

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         n = 0

         do k = 1, ncharge-1
         do l = k+1, ncharge

            n = n + 1

            if ( mod( n-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i  = i_q(k)
            j  = i_q(l)

            qiqj = q(i)*q(j)

            if ( qiqj .eq. 0.d0 ) cycle

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            r2 = xij*xij + yij*yij + zij*zij

            if ( r2 .gt. rcut_ewald2 ) cycle

            r = sqrt(r2)

            rinv  = 1.d0/r
            rinv2 = rinv*rinv
            rinv3 = rinv*rinv2

            ar = alpha_ewald*r

            erfc_ar = 1.d0 - erf_0(ar)

            pot(m) = pot(m) + qiqj*erfc_ar*rinv

            factor = erfc_ar*rinv3 + alpha_ewald*erf_1(ar)*rinv2

            factor = qiqj*factor

            fxi = factor*xij
            fyi = factor*yij
            fzi = factor*zij

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

         end do
         end do

      end do

!-----------------------------------------------------------------------
!     /*   main loop:  subtract bonded charge pairs                   */
!-----------------------------------------------------------------------

      do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         do k = 1, nbcp

            if ( mod( k-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i  = i_bcp(k)
            j  = j_bcp(k)

            factor = factor_bcp(k)

            qi = q(i)
            qj = q(j)

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            rinv = 1.d0/rij

            uij = (factor - 1.d0) * qi*qj*rinv

            pot(m) = pot(m) + uij

            duij = - uij*rinv

            fxi = - duij*xij*rinv
            fyi = - duij*yij*rinv
            fzi = - duij*zij*rinv

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_ewald_self_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, pot, nbead, &
     &   myrank_main, nprocs_main, myrank_sub, nprocs_sub

      use mm_variables, only : &
     &   alpha_ewald, q, i_q, ncharge

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, m, k

      real(8) :: factor, q2sum

!-----------------------------------------------------------------------
!     /*   parameters                                                 */
!-----------------------------------------------------------------------

      factor = alpha_ewald/sqrt(pi)

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         q2sum = 0.d0

         do k = 1, ncharge

            if ( mod( k-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i  = i_q(k)

            q2sum = q2sum + q(i)*q(i)

         end do

         pot(m) = pot(m) - q2sum*factor

      end do

      return
      end





!***********************************************************************
      subroutine force_ewald_charge_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, volume, pot, vir, nbead, &
     &   myrank_main, nprocs_main, myrank_sub

      use mm_variables, only : &
     &   alpha_ewald, q, i_q, ncharge

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, m, k

      real(8) :: factor, qsum

!-----------------------------------------------------------------------
!     /*   parameters                                                 */
!-----------------------------------------------------------------------

      factor = pi/(2.d0*volume*alpha_ewald*alpha_ewald)

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         if ( myrank_sub .ne. 0 ) cycle

         qsum = 0.d0

         do k = 1, ncharge

            i  = i_q(k)

            qsum = qsum + q(i)

         end do

         pot(m) = pot(m) - qsum*qsum*factor

         vir(1,1) = vir(1,1) - qsum*qsum*factor
         vir(2,2) = vir(2,2) - qsum*qsum*factor
         vir(3,3) = vir(3,3) - qsum*qsum*factor

      end do

      return
      end





!***********************************************************************
      subroutine force_ewald_fs_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pot, vir, fx, fy, fz, pi, boxinv, volume, &
     &   nbead, natom, &
     &   myrank_main, nprocs_main, myrank_sub, nprocs_sub

      use mm_variables, only : &
     &   eigax, eigay, eigaz, eigbx, eigby, eigbz, eigcx, eigcy, eigcz, &
     &   alpha_ewald, q, i_q, lmax_ewald, ncharge

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: m, k, i, l, n, l2, la, lb, lc

      real(8) :: ax, ay, az, bx, by, bz, cx, cy, cz, a2, b2, c2, &
     &           al2, bl2, cl2, factor_1, factor_2, factor_3, factor_4, &
     &           factor_5, factor_6, factor_7, factor_8, &
     &           gx, gy, gz, g2, g2max, qcos, qsin, fxi, fyi, fzi, &
     &           cos_gxyz, sin_gxyz, qexp2

      integer, save :: iset = 0
      integer, save :: lmax_ewald_save(3)

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         if ( .not. allocated( eigax ) ) &
     &      allocate( eigax(natom,-lmax_ewald(1):lmax_ewald(1)) )
         if ( .not. allocated( eigay ) ) &
     &      allocate( eigay(natom,-lmax_ewald(1):lmax_ewald(1)) )
         if ( .not. allocated( eigaz ) ) &
     &      allocate( eigaz(natom,-lmax_ewald(1):lmax_ewald(1)) )
         if ( .not. allocated( eigbx ) ) &
     &      allocate( eigbx(natom,-lmax_ewald(2):lmax_ewald(2)) )
         if ( .not. allocated( eigby ) ) &
     &      allocate( eigby(natom,-lmax_ewald(2):lmax_ewald(2)) )
         if ( .not. allocated( eigbz ) ) &
     &      allocate( eigbz(natom,-lmax_ewald(2):lmax_ewald(2)) )
         if ( .not. allocated( eigcx ) ) &
     &      allocate( eigcx(natom,-lmax_ewald(3):lmax_ewald(3)) )
         if ( .not. allocated( eigcy ) ) &
     &      allocate( eigcy(natom,-lmax_ewald(3):lmax_ewald(3)) )
         if ( .not. allocated( eigcz ) ) &
     &      allocate( eigcz(natom,-lmax_ewald(3):lmax_ewald(3)) )

         lmax_ewald_save(1) = lmax_ewald(1)
         lmax_ewald_save(2) = lmax_ewald(2)
         lmax_ewald_save(3) = lmax_ewald(3)

         iset = 1

      else

         if ( lmax_ewald(1) .ne. lmax_ewald_save(1) ) then

            if ( allocated( eigax ) ) &
     &         deallocate( eigax )
            if ( allocated( eigay ) ) &
     &         deallocate( eigay )
            if ( allocated( eigaz ) ) &
     &         deallocate( eigaz )

            if ( .not. allocated( eigax ) ) &
     &         allocate( eigax(natom,-lmax_ewald(1):lmax_ewald(1)) )
            if ( .not. allocated( eigay ) ) &
     &         allocate( eigay(natom,-lmax_ewald(1):lmax_ewald(1)) )
            if ( .not. allocated( eigaz ) ) &
     &         allocate( eigaz(natom,-lmax_ewald(1):lmax_ewald(1)) )

         end if

         if ( lmax_ewald(2) .ne. lmax_ewald_save(2) ) then

            if ( allocated( eigbx ) ) &
     &         deallocate( eigbx )
            if ( allocated( eigby ) ) &
     &         deallocate( eigby )
            if ( allocated( eigbz ) ) &
     &         deallocate( eigbz )

            if ( .not. allocated( eigbx ) ) &
     &         allocate( eigbx(natom,-lmax_ewald(2):lmax_ewald(2)) )
            if ( .not. allocated( eigby ) ) &
     &         allocate( eigby(natom,-lmax_ewald(2):lmax_ewald(2)) )
            if ( .not. allocated( eigbz ) ) &
     &         allocate( eigbz(natom,-lmax_ewald(2):lmax_ewald(2)) )

         end if

         if ( lmax_ewald(3) .ne. lmax_ewald_save(3) ) then

            if ( allocated( eigcx ) ) &
     &         deallocate( eigcx )
            if ( allocated( eigcy ) ) &
     &         deallocate( eigcy )
            if ( allocated( eigcz ) ) &
     &         deallocate( eigcz )

            if ( .not. allocated( eigcx ) ) &
     &         allocate( eigcx(natom,-lmax_ewald(3):lmax_ewald(3)) )
            if ( .not. allocated( eigcy ) ) &
     &         allocate( eigcy(natom,-lmax_ewald(3):lmax_ewald(3)) )
            if ( .not. allocated( eigcz ) ) &
     &         allocate( eigcz(natom,-lmax_ewald(3):lmax_ewald(3)) )

         end if

         lmax_ewald_save(1) = lmax_ewald(1)
         lmax_ewald_save(2) = lmax_ewald(2)
         lmax_ewald_save(3) = lmax_ewald(3)

      end if

!-----------------------------------------------------------------------
!     /*   loop of beads: start                                       */
!-----------------------------------------------------------------------

      do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

!-----------------------------------------------------------------------
!        /*   parameters                                              */
!-----------------------------------------------------------------------

         ax = 2.d0*pi*boxinv(1,1)
         ay = 2.d0*pi*boxinv(1,2)
         az = 2.d0*pi*boxinv(1,3)
         bx = 2.d0*pi*boxinv(2,1)
         by = 2.d0*pi*boxinv(2,2)
         bz = 2.d0*pi*boxinv(2,3)
         cx = 2.d0*pi*boxinv(3,1)
         cy = 2.d0*pi*boxinv(3,2)
         cz = 2.d0*pi*boxinv(3,3)

         a2 = ax*ax + ay*ay + az*az
         b2 = bx*bx + by*by + bz*bz
         c2 = cx*cx + cy*cy + cz*cz

         al2 = a2*lmax_ewald(1)**2
         bl2 = b2*lmax_ewald(2)**2
         cl2 = c2*lmax_ewald(3)**2

         g2max = min( al2, bl2, cl2 )

!-----------------------------------------------------------------------
!        /*   main loop                                               */
!-----------------------------------------------------------------------

         do k = 1, ncharge

            i  = i_q(k)

            eigax(i, 0)  = (1.d0,0.d0)
            eigay(i, 0)  = (1.d0,0.d0)
            eigaz(i, 0)  = (1.d0,0.d0)
            eigbx(i, 0)  = (1.d0,0.d0)
            eigby(i, 0)  = (1.d0,0.d0)
            eigbz(i, 0)  = (1.d0,0.d0)
            eigcx(i, 0)  = (1.d0,0.d0)
            eigcy(i, 0)  = (1.d0,0.d0)
            eigcz(i, 0)  = (1.d0,0.d0)

            eigax(i, 1)  = dcmplx ( cos(ax*x(i,m)), sin(ax*x(i,m)) )
            eigay(i, 1)  = dcmplx ( cos(ay*y(i,m)), sin(ay*y(i,m)) )
            eigaz(i, 1)  = dcmplx ( cos(az*z(i,m)), sin(az*z(i,m)) )
            eigbx(i, 1)  = dcmplx ( cos(bx*x(i,m)), sin(bx*x(i,m)) )
            eigby(i, 1)  = dcmplx ( cos(by*y(i,m)), sin(by*y(i,m)) )
            eigbz(i, 1)  = dcmplx ( cos(bz*z(i,m)), sin(bz*z(i,m)) )
            eigcx(i, 1)  = dcmplx ( cos(cx*x(i,m)), sin(cx*x(i,m)) )
            eigcy(i, 1)  = dcmplx ( cos(cy*y(i,m)), sin(cy*y(i,m)) )
            eigcz(i, 1)  = dcmplx ( cos(cz*z(i,m)), sin(cz*z(i,m)) )

            eigax(i,-1)  = dconjg ( eigax(i,1) )
            eigay(i,-1)  = dconjg ( eigay(i,1) )
            eigaz(i,-1)  = dconjg ( eigaz(i,1) )
            eigbx(i,-1)  = dconjg ( eigbx(i,1) )
            eigby(i,-1)  = dconjg ( eigby(i,1) )
            eigbz(i,-1)  = dconjg ( eigbz(i,1) )
            eigcx(i,-1)  = dconjg ( eigcx(i,1) )
            eigcy(i,-1)  = dconjg ( eigcy(i,1) )
            eigcz(i,-1)  = dconjg ( eigcz(i,1) )

            do l = 2, lmax_ewald(1)
               eigax(i, l)  = eigax(i,l-1)*eigax(i,1)
               eigay(i, l)  = eigay(i,l-1)*eigay(i,1)
               eigaz(i, l)  = eigaz(i,l-1)*eigaz(i,1)
               eigax(i,-l)  = dconjg ( eigax(i,l) )
               eigay(i,-l)  = dconjg ( eigay(i,l) )
               eigaz(i,-l)  = dconjg ( eigaz(i,l) )
            end do
            do l = 2, lmax_ewald(2)
               eigbx(i, l)  = eigbx(i,l-1)*eigbx(i,1)
               eigby(i, l)  = eigby(i,l-1)*eigby(i,1)
               eigbz(i, l)  = eigbz(i,l-1)*eigbz(i,1)
               eigbx(i,-l)  = dconjg ( eigbx(i,l) )
               eigby(i,-l)  = dconjg ( eigby(i,l) )
               eigbz(i,-l)  = dconjg ( eigbz(i,l) )
            end do
            do l = 2, lmax_ewald(3)
               eigcx(i, l)  = eigcx(i,l-1)*eigcx(i,1)
               eigcy(i, l)  = eigcy(i,l-1)*eigcy(i,1)
               eigcz(i, l)  = eigcz(i,l-1)*eigcz(i,1)
               eigcx(i,-l)  = dconjg ( eigcx(i,l) )
               eigcy(i,-l)  = dconjg ( eigcy(i,l) )
               eigcz(i,-l)  = dconjg ( eigcz(i,l) )
            end do

         end do

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

         factor_1 = (4.d0*pi)/(2.d0*volume)

         n = 0

         do la =              0, lmax_ewald(1)
         do lb = -lmax_ewald(2), lmax_ewald(2)
         do lc = -lmax_ewald(3), lmax_ewald(3)

            n = n + 1

            if ( mod( n-1, nprocs_sub ) .ne. myrank_sub ) cycle

            l2 = la*la + lb*lb + lc*lc

            if ( l2 .eq. 0 ) cycle

            if ( la .eq. 0 ) then
               factor_2 = 1.d0
            else
               factor_2 = 2.d0
            end if

            gx = ax*la + bx*lb + cx*lc
            gy = ay*la + by*lb + cy*lc
            gz = az*la + bz*lb + cz*lc

            g2 = gx*gx + gy*gy + gz*gz

            if ( g2 .gt. g2max ) cycle

            factor_3 = exp(-g2/(4.d0*alpha_ewald*alpha_ewald))/g2

            qcos = 0.d0
            qsin = 0.d0

            do k = 1, ncharge

               i  = i_q(k)

               cos_gxyz = dreal(eigax(i,la)*eigbx(i,lb)*eigcx(i,lc) &
     &                         *eigay(i,la)*eigby(i,lb)*eigcy(i,lc) &
     &                         *eigaz(i,la)*eigbz(i,lb)*eigcz(i,lc))
               sin_gxyz = dimag(eigax(i,la)*eigbx(i,lb)*eigcx(i,lc) &
     &                         *eigay(i,la)*eigby(i,lb)*eigcy(i,lc) &
     &                         *eigaz(i,la)*eigbz(i,lb)*eigcz(i,lc))

               qcos = qcos + q(i)*cos_gxyz
               qsin = qsin + q(i)*sin_gxyz

            end do

            qexp2 = qcos*qcos + qsin*qsin

            pot(m) = pot(m) + factor_1*factor_2*factor_3*qexp2

            do k = 1, ncharge

               i  = i_q(k)

               cos_gxyz = dreal(eigax(i,la)*eigbx(i,lb)*eigcx(i,lc) &
     &                         *eigay(i,la)*eigby(i,lb)*eigcy(i,lc) &
     &                         *eigaz(i,la)*eigbz(i,lb)*eigcz(i,lc))
               sin_gxyz = dimag(eigax(i,la)*eigbx(i,lb)*eigcx(i,lc) &
     &                         *eigay(i,la)*eigby(i,lb)*eigcy(i,lc) &
     &                         *eigaz(i,la)*eigbz(i,lb)*eigcz(i,lc))

               factor_4 = sin_gxyz*qcos - cos_gxyz*qsin

               factor_5 = 2.d0*q(i)*factor_1*factor_2*factor_3*factor_4

               fxi = factor_5*gx
               fyi = factor_5*gy
               fzi = factor_5*gz

               fx(i,m) = fx(i,m) + fxi
               fy(i,m) = fy(i,m) + fyi
               fz(i,m) = fz(i,m) + fzi

            end do

            factor_6 = factor_1*factor_2*factor_3*qexp2
            factor_7 = 1.d0/(4.d0*alpha_ewald*alpha_ewald)
            factor_8 = 2.d0 * ( 1.d0 + factor_7*g2 ) / g2

            vir(1,1) = vir(1,1) + factor_6 * ( 1.d0 - factor_8*gx*gx )
            vir(1,2) = vir(1,2) - factor_6 * factor_8*gx*gy
            vir(1,3) = vir(1,3) - factor_6 * factor_8*gx*gz
            vir(2,1) = vir(2,1) - factor_6 * factor_8*gy*gx
            vir(2,2) = vir(2,2) + factor_6 * ( 1.d0 - factor_8*gy*gy )
            vir(2,3) = vir(2,3) - factor_6 * factor_8*gy*gz
            vir(3,1) = vir(3,1) - factor_6 * factor_8*gz*gx
            vir(3,2) = vir(3,2) - factor_6 * factor_8*gz*gy
            vir(3,3) = vir(3,3) + factor_6 * ( 1.d0 - factor_8*gz*gz )

         end do
         end do
         end do

!-----------------------------------------------------------------------
!     /*   loop of beads: end                                         */
!-----------------------------------------------------------------------

      end do

      return
      end





!***********************************************************************
      subroutine mm_dipole_MPI
!***********************************************************************
!=======================================================================
!
!     calculate dipole moment
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, dipx, dipy, dipz, mbox, nbead, &
     &   myrank_main, nprocs_main, myrank_sub, nprocs_sub

      use mm_variables, only : &
     &   q, i_q, ncharge

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: m, i, k, m1, m2, m3

      real(8) :: xb, yb, zb

!-----------------------------------------------------------------------
!     /*   calculate mm dipole moment                                 */
!-----------------------------------------------------------------------

      do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         dipx(m) = 0.d0
         dipy(m) = 0.d0
         dipz(m) = 0.d0

         do k = 1, ncharge

            if ( mod( k-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i  = i_q(k)

            xb = x(i,m)
            yb = y(i,m)
            zb = z(i,m)

            m1 = mbox(1,i,m)
            m2 = mbox(2,i,m)
            m3 = mbox(3,i,m)

            call pbc_unfold_MPI( xb, yb, zb, m1, m2, m3 )

            dipx(m) = dipx(m) + q(i)*xb
            dipy(m) = dipy(m) + q(i)*yb
            dipz(m) = dipz(m) + q(i)*zb

         end do

      end do

      return
      end





!***********************************************************************
      subroutine mm_dipole_mol_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, ux, uy, uz, dipx, dipy, dipz, iounit, &
     &   nbead, myrank, &
     &   myrank_main, nprocs_main, myrank_sub, nprocs_sub

      use mm_variables, only : &
     &   q, natom_per_dip, natom_dip_max, list_atom_dip, &
     &   nmol_dip, list_atom_dip

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

!     /*   flag for initial setups   */
      integer, save :: iset = 0

!     /*   integer   */
      integer :: ierr = 0

!     /*   integer   */
      integer :: i, j, k, l, m, n

!     /*   real numbers   */
      real(8) :: xk, yk, zk, qk

!-----------------------------------------------------------------------
!     //   this is carried out only once
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

!-----------------------------------------------------------------------
!     //   psuedo do loop
!-----------------------------------------------------------------------

      do

!-----------------------------------------------------------------------
!     //   read number of molecules
!-----------------------------------------------------------------------

!     /*   parent process   */
      if ( myrank .eq. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'mm.dat' )

!        /*   search for tag    */
         call search_tag ( '<dipoles>', 9, iounit, ierr )

!        /*   number of polar molecules   */
         read ( iounit, *, iostat = ierr ) nmol_dip

!        /*   file close   */
         close( iounit )

!     /*   parent process   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0 ( ierr )

!-----------------------------------------------------------------------
!     //   warning: on error, atomic dipole is applied for pbc
!-----------------------------------------------------------------------

      if ( ierr .ne. 0 ) exit

!-----------------------------------------------------------------------
!     //   communicate
!-----------------------------------------------------------------------

      call my_mpi_bcast_int_0( nmol_dip )

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

!     /*   number of atoms in a molecule   */
      if ( .not. allocated( natom_per_dip ) ) &
     &   allocate( natom_per_dip(nmol_dip) )

!-----------------------------------------------------------------------
!     //   number of atoms per molecule
!-----------------------------------------------------------------------

!     /*   parent process   */
      if ( myrank .eq. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'mm.dat' )

!        /*   search for tag    */
         call search_tag ( '<dipoles>', 9, iounit, ierr )

!        /*   number of molecules   */
         read ( iounit, *, iostat = ierr )

!        /*   maximum number of atoms per polar molecule   */
         natom_dip_max = 0

!        /*   loop of molecules   */
         do i = 1, nmol_dip

!           /*   number of atoms per polar molecule   */
            read ( iounit, *, iostat = ierr ) natom_per_dip(i)

!           /*   maximum number of atoms per polar molecule   */

            if ( natom_per_dip(i) .gt. natom_dip_max ) then
               natom_dip_max = natom_per_dip(i)
            end if

!        /*   loop of molecules   */
         end do

!        /*   file close   */
         close( iounit )

!     /*   parent process   */
      end if

!-----------------------------------------------------------------------
!     //   communicate
!-----------------------------------------------------------------------

      call my_mpi_bcast_int_1( natom_per_dip, nmol_dip )
      call my_mpi_bcast_int_0( natom_dip_max )

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

!     /*   list of atoms in a molecule   */
      if ( .not. allocated( list_atom_dip ) ) &
     &   allocate( list_atom_dip(natom_dip_max,nmol_dip) )

!-----------------------------------------------------------------------
!     //   number of atoms per molecule
!-----------------------------------------------------------------------

!     /*   parent process   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'mm.dat' )

!     /*   search for tag    */
      call search_tag ( '<dipoles>', 9, iounit, ierr )

!     /*   number of molecules   */
      read ( iounit, *, iostat = ierr )

!     /*   loop of molecules   */
      do i = 1, nmol_dip

!        /*   read list of atoms   */
         read ( iounit, *, iostat = ierr ) &
     &      j, ( list_atom_dip(k,i), k = 1, natom_per_dip(i) )

!     /*   loop of molecules   */
      end do

!     /*   file close   */
      close( iounit )

!     /*   parent process   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0 ( ierr )

!-----------------------------------------------------------------------
!     //   communicate
!-----------------------------------------------------------------------

      call my_mpi_bcast_int_2( list_atom_dip, natom_dip_max, nmol_dip )

!-----------------------------------------------------------------------
!     //   exit loop
!-----------------------------------------------------------------------

      exit

!-----------------------------------------------------------------------
!     //   pseudo do loop
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     //   set done
!-----------------------------------------------------------------------

      iset = 1

!-----------------------------------------------------------------------
!     //   this is carried out only once
!-----------------------------------------------------------------------

      end if

!-----------------------------------------------------------------------
!     //   calculate mm dipole moment
!-----------------------------------------------------------------------

!     /*   read correctly   */
      if ( ierr .eq. 0 ) then

!     /*   loop of beads   */
      do l = 1, nbead

!        /*   only my bead   */
         if ( mod( l-1, nprocs_main ) .ne. myrank_main ) cycle

!        /*   atoms   */
         n = 0

!        /*   loop of polar molecules   */
         do j = 1, nmol_dip

!        /*   loop of atoms per polar molecules    */
         do i = 1, natom_per_dip(j)

!           /*   atoms  */
            n = n + 1

!           /*   only my bead   */
            if ( mod( n-1, nprocs_sub ) .ne. myrank_sub ) cycle

!           /*   i-th atom   */
            k = list_atom_dip(i,j)

!           /*   first atom   */
            m = list_atom_dip(1,j)

!           /*   relative position of i-th atom to the first atom   */

            xk = x(k,l) - ux(m,1)
            yk = y(k,l) - uy(m,1)
            zk = z(k,l) - uz(m,1)

!           /*   apply the boundary condition   */
            call pbc_atom_MPI ( xk, yk, zk )

!           /*   relative position of i-th atom to the first atom   */

            xk = xk + ux(m,1)
            yk = yk + uy(m,1)
            zk = zk + uz(m,1)

!           /*   atomic charge   */
            qk = q(k)

!           /*   dipole moment   */

            dipx(l) = dipx(l) + qk*xk
            dipy(l) = dipy(l) + qk*yk
            dipz(l) = dipz(l) + qk*zk

!        /*   loop of atoms per polar molecules    */
         end do

!        /*   loop of polar molecules   */
         end do

!     /*   loop of beads   */
      end do

!     /*   read correctly   */
      end if

!-----------------------------------------------------------------------
!     //   if read incorrect, calculate mm atomic dipole moment
!-----------------------------------------------------------------------

!     /*   read incorrect   */
      if ( ierr .ne. 0 ) then

!        /*   mm atomic dipole moment   */
         call mm_dipole_MPI

!     /*   read incorrect   */
      end if

!-----------------------------------------------------------------------
!     //   end of routine
!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      subroutine force_mm_lin_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iounit

      use mm_variables, only : eq_lin, fc_lin, i_lin, j_lin, nlin

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer k, ierr

!-----------------------------------------------------------------------
!     /*   set up                                                     */
!-----------------------------------------------------------------------

      open (iounit,file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<linear_bonds>', 14, iounit, ierr )

!        /*   number of interatomic bonds   */
         if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr ) nlin

      close(iounit)

      if ( ierr .ne. 0 ) nlin = 0

      if ( nlin .eq. 0 ) return

      if ( .not. allocated( i_lin ) ) &
     &   allocate( i_lin(nlin))
      if ( .not. allocated( j_lin ) ) &
     &   allocate( j_lin(nlin))
      if ( .not. allocated(eq_lin ) ) &
     &   allocate(eq_lin(nlin))
      if ( .not. allocated(fc_lin ) ) &
     &   allocate(fc_lin(nlin))

      open (iounit,file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<linear_bonds>', 14, iounit, ierr )

!        /*   number of interatomic bonds   */
         read( iounit, *, iostat=ierr ) nlin

!        /*   atom 1, atom 2, equilibrium, force constant   */
         do k = 1, nlin
            read( iounit, *, iostat=ierr ) &
     &         i_lin(k), j_lin(k), eq_lin(k), fc_lin(k)
         end do

      close(iounit)

      return
      end





!***********************************************************************
      subroutine force_mm_genlin_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit

      use mm_variables, only : &
     &   eq_genlin, fc_genlin, i_genlin, j_genlin, n_genlin, ngenlin

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer k, ierr

!-----------------------------------------------------------------------
!     /*   set up                                                     */
!-----------------------------------------------------------------------

      open (iounit,file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<genlin_bonds>', 14, iounit, ierr )

!        /*   number of generalized linear bonds   */
         if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr ) ngenlin

      close(iounit)

      if ( ierr .ne. 0 ) ngenlin = 0

      if ( ngenlin .eq. 0 ) return

      if ( .not. allocated( i_genlin ) ) &
     &   allocate( i_genlin(ngenlin))
      if ( .not. allocated( j_genlin ) ) &
     &   allocate( j_genlin(ngenlin))
      if ( .not. allocated( n_genlin ) ) &
     &   allocate( n_genlin(ngenlin))
      if ( .not. allocated(eq_genlin ) ) &
     &   allocate(eq_genlin(ngenlin))
      if ( .not. allocated(fc_genlin ) ) &
     &   allocate(fc_genlin(ngenlin))

      open (iounit,file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<genlin_bonds>', 14, iounit, ierr )

!        /*   number of generalized linear bonds   */
         read( iounit, *, iostat=ierr ) ngenlin

!        /*   atom 1, atom 2, power, equilibrium, force constant   */
         do k = 1, ngenlin
            read( iounit, *, iostat=ierr ) &
     &         i_genlin(k),  j_genlin(k), &
     &         n_genlin(k), eq_genlin(k), fc_genlin(k)
         end do

      close(iounit)

      return
      end





!***********************************************************************
      subroutine force_mm_angl_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit

      use mm_variables, only : &
     &   eq_angl, fc_angl, i_angl, j_angl, k_angl, nangl

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer l, ierr

!-----------------------------------------------------------------------
!     /*   set up                                                     */
!-----------------------------------------------------------------------

      open (iounit,file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<angular_bonds>', 15, iounit, ierr )

!        /*   number of angular bonds   */
         if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr ) nangl

      close(iounit)

      if ( ierr .ne. 0 ) nangl = 0

      if ( nangl .eq. 0 ) return

      if ( .not. allocated( i_angl ) ) &
     &   allocate( i_angl(nangl))
      if ( .not. allocated( j_angl ) ) &
     &   allocate( j_angl(nangl))
      if ( .not. allocated( k_angl ) ) &
     &   allocate( k_angl(nangl))
      if ( .not. allocated(eq_angl ) ) &
     &   allocate(eq_angl(nangl))
      if ( .not. allocated(fc_angl ) ) &
     &   allocate(fc_angl(nangl))

      open (iounit,file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<angular_bonds>', 15, iounit, ierr )

!        /*   number of angular bonds   */
         read( iounit, *, iostat=ierr ) nangl

!        /*   atom 1, atom 2, atom 3, equilibrium, force constant  */
         do l = 1, nangl
            read( iounit, *, iostat=ierr ) &
     &         i_angl(l), j_angl(l), k_angl(l), eq_angl(l), fc_angl(l)
         end do

      close(iounit)

      return
      end





!***********************************************************************
      subroutine force_mm_dih_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit

      use mm_variables, only : &
     &   v_dih, i_dih, j_dih, k_dih, l_dih, nu_dih, mu_dih, ndih

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer l, ierr

!-----------------------------------------------------------------------
!     /*   set up                                                     */
!-----------------------------------------------------------------------

      open (iounit,file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<dihedral_bonds>', 16, iounit, ierr )

!        /*   number of dihedral bonds   */
         if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr ) ndih

      close(iounit)

      if ( ierr .ne. 0 ) ndih = 0

      if ( ndih .eq. 0 ) return

      if ( .not. allocated( i_dih ) ) &
     &   allocate( i_dih(ndih))
      if ( .not. allocated( j_dih ) ) &
     &   allocate( j_dih(ndih))
      if ( .not. allocated( k_dih ) ) &
     &   allocate( k_dih(ndih))
      if ( .not. allocated( l_dih ) ) &
     &   allocate( l_dih(ndih))
      if ( .not. allocated( v_dih ) ) &
     &   allocate( v_dih(ndih))
      if ( .not. allocated(nu_dih ) ) &
     &   allocate(nu_dih(ndih))
      if ( .not. allocated(mu_dih ) ) &
     &   allocate(mu_dih(ndih))

      open (iounit,file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<dihedral_bonds>', 16, iounit, ierr )

!        /*   number of dihedral bonds   */
         read( iounit, *, iostat=ierr ) ndih

!        /*   atom 1, atom 2, atom 3, atom 4, v, nu, mu   */
         do l = 1, ndih
            read( iounit, *, iostat=ierr ) &
     &         i_dih(l), j_dih(l), k_dih(l), l_dih(l), &
     &         v_dih(l), nu_dih(l), mu_dih(l)
         end do

!        /*   check data   */
!         ierr = 0
!         do l = 1, ndih
!            if ( nu_dih(l) .lt.  1 ) ierr = 1
!            if ( mu_dih(l) .lt. -1 ) ierr = 1
!            if ( mu_dih(l) .gt.  1 ) ierr = 1
!         end do

         call error_handling_MPI( ierr, 'subroutine force_mm_MPI', 23 )

      close(iounit)

      return
      end





!***********************************************************************
      subroutine force_mm_lj_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit

      use mm_variables, only : &
     &   eps_lj, sig_lj, rin_lj, rout_lj, i_lj, j_lj, nlj

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer l, ierr

!-----------------------------------------------------------------------
!     /*   set up                                                     */
!-----------------------------------------------------------------------

      open (iounit,file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<lennard-jones>', 15, iounit, ierr )

!        /*   number of lennard-jones interaction   */
         if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr ) nlj

      close(iounit)

      if ( ierr .ne. 0 ) nlj = 0

      if ( nlj .eq. 0 ) return

      if ( .not. allocated(  i_lj ) ) &
     &   allocate(  i_lj(nlj))
      if ( .not. allocated(  j_lj ) ) &
     &   allocate(  j_lj(nlj))
      if ( .not. allocated(eps_lj ) ) &
     &   allocate(eps_lj(nlj))
      if ( .not. allocated(sig_lj ) ) &
     &   allocate(sig_lj(nlj))

      open (iounit, file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<lennard-jones>', 15, iounit, ierr )

!        /*   number of lennard-jones interaction   */
         read( iounit, *, iostat=ierr ) nlj

!        /*   cut off distance, inside and outside   */
         read( iounit, *, iostat=ierr ) rin_lj, rout_lj

!        /*   atom 1, atom 2, epsilon, sigma   */
         do l = 1, nlj
            read( iounit, *, iostat=ierr ) &
     &         i_lj(l), j_lj(l), eps_lj(l), sig_lj(l)
         end do

      close(iounit)

      return
      end





!***********************************************************************
      subroutine force_mm_buck_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit

      use mm_variables, only : &
     &   a_buck, b_buck, c_buck, rin_buck, rout_buck, &
     &   i_buck, j_buck, nbuck

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer l, ierr

!-----------------------------------------------------------------------
!     /*   set up                                                     */
!-----------------------------------------------------------------------

      open (iounit,file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<buckingham>', 12, iounit, ierr )

!        /*   number of buckingham interaction   */
         if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr ) nbuck

      close(iounit)

      if ( ierr .ne. 0 ) nbuck = 0

      if ( nbuck .eq. 0 ) return

      if ( .not. allocated(i_buck ) ) &
     &   allocate(i_buck(nbuck))
      if ( .not. allocated(j_buck ) ) &
     &   allocate(j_buck(nbuck))
      if ( .not. allocated(a_buck ) ) &
     &   allocate(a_buck(nbuck))
      if ( .not. allocated(b_buck ) ) &
     &   allocate(b_buck(nbuck))
      if ( .not. allocated(c_buck ) ) &
     &   allocate(c_buck(nbuck))

      open (iounit, file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<buckingham>', 12, iounit, ierr )

!        /*   number of lennard-jones interaction   */
         read( iounit, *, iostat=ierr ) nbuck

!        /*   cut off distance, inside and outside   */
         read( iounit, *, iostat=ierr ) rin_buck, rout_buck

!        /*   atom 1, atom 2, epsilon, sigma   */
         do l = 1, nbuck
            read( iounit, *, iostat=ierr ) &
     &         i_buck(l), j_buck(l), a_buck(l), b_buck(l), c_buck(l)
         end do

      close(iounit)

      return
      end





!***********************************************************************
      subroutine force_mm_coulomb_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, natom

      use mm_variables, only : &
     &   factor_bcp, q, i_q, i_bcp, j_bcp, ncharge, nbcp

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer i, k, ierr

!-----------------------------------------------------------------------
!     /*   set up                                                     */
!-----------------------------------------------------------------------

      open (iounit,file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<charges>', 9, iounit, ierr )

!        /*   number of charges   */
         if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr ) ncharge

      close(iounit)

      if ( ierr .ne. 0 ) ncharge = 0

      if ( .not. allocated( q ) ) &
     &   allocate( q(natom) )

      q(:) = 0.d0

      if ( ncharge .eq. 0 ) return

      if ( .not. allocated( i_q ) ) &
     &   allocate( i_q(ncharge) )

      open (iounit,file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<charges>', 9, iounit, ierr )

!        /*   number of charges   */
         read( iounit, *, iostat=ierr ) ncharge

!        /*   atom 1, atom 2, equilibrium, force constant   */
         do k = 1, ncharge
            read( iounit, *, iostat=ierr ) i, q(i)
            i_q(k) = i
         end do

      close(iounit)

      open (iounit,file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<nbcp>', 6, iounit, ierr )

!        /*   number of bonded charge pairs   */
         read( iounit, *, iostat=ierr ) nbcp

      close(iounit)

      if ( nbcp .eq. 0 ) return

      if ( .not. allocated(  i_bcp ) ) &
     &   allocate(  i_bcp(nbcp))
      if ( .not. allocated(  j_bcp ) ) &
     &   allocate(  j_bcp(nbcp))
      if ( .not. allocated(  factor_bcp ) ) &
     &   allocate(  factor_bcp(nbcp))

      open (iounit, file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<nbcp>', 6, iounit, ierr )

!        /*   number of bonded charge pairs   */
         read( iounit, *, iostat=ierr ) nbcp

         do k = 1, nbcp
            read( iounit, *, iostat=ierr ) &
     &         i_bcp(k), j_bcp(k), factor_bcp(k)
         end do

      close(iounit)

      return
      end





!***********************************************************************
      subroutine force_mm_improper_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit

      use mm_variables, only : &
     &   eq_improper, fc_improper, i_improper, j_improper, &
     &   k_improper, l_improper, nimproper

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer l, ierr

!-----------------------------------------------------------------------
!     /*   set up                                                     */
!-----------------------------------------------------------------------

      open (iounit,file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<improper_bonds>', 16, iounit, ierr )

!        /*   number of dihedral bonds   */
         if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr ) nimproper

      close(iounit)

      if ( ierr .ne. 0 ) nimproper = 0

      if ( nimproper .eq. 0 ) return

      if ( .not. allocated(  i_improper ) ) &
     &   allocate(  i_improper(nimproper) )
      if ( .not. allocated(  j_improper ) ) &
     &   allocate(  j_improper(nimproper) )
      if ( .not. allocated(  k_improper ) ) &
     &   allocate(  k_improper(nimproper) )
      if ( .not. allocated(  l_improper ) ) &
     &   allocate(  l_improper(nimproper) )
      if ( .not. allocated( eq_improper ) ) &
     &   allocate( eq_improper(nimproper) )
      if ( .not. allocated( fc_improper ) ) &
     &   allocate( fc_improper(nimproper) )

      open (iounit,file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<improper_bonds>', 16, iounit, ierr )

!        /*   number of dihedral bonds   */
         read( iounit, *, iostat=ierr ) nimproper

!        /*   atom 1, atom 2, atom 3, atom 4, equilibrium, constant   */
         do l = 1, nimproper
            read( iounit, *, iostat=ierr ) &
     &         i_improper(l), j_improper(l), &
     &         k_improper(l), l_improper(l), &
     &         eq_improper(l), fc_improper(l)
         end do

         call error_handling_MPI( ierr, 'subroutine force_mm_MPI', 23 )

      close(iounit)

      return
      end





!***********************************************************************
      subroutine force_mm_improper_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, x, y, z, fx, fy, fz, pot, vir, nbead, &
     &   myrank_main, nprocs_main, myrank_sub, nprocs_sub

      use mm_variables, only : &
     &   eq_improper, fc_improper, i_improper, j_improper, &
     &   k_improper, l_improper, nimproper

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, n

      real(8) :: tiny_value = 1.d-4

      real(8) :: xij, yij, zij, xkj, ykj, zkj, xlj, ylj, zlj, &
     &           xijk, yijk, zijk, xjkl, yjkl, zjkl, rijk2, rjkl2, &
     &           rijkl2, rijk2inv, rjkl2inv, rijkl2inv, cos_phi, phi, &
     &           fxi, fyi, fzi, fxj, fyj, fzj, fxk, fyk, fzk, &
     &           fxl, fyl, fzl

      real(8) :: dphi, factor_1, factor_2, factor_3, factor_4, &
     &           px1, py1, pz1, px2, py2, pz2, px3, py3, pz3, &
     &           px4, py4, pz4, px5, py5, pz5, px6, py6, pz6

      real(8) :: ax, ay, az, a1, a2, xkl, ykl, zkl, xki, yki, zki

      real(8) :: daxdxi, daxdyi, daxdzi, daydxi, daydyi, daydzi, &
     &           dazdxi, dazdyi, dazdzi, dadxi, dadyi, dadzi, &
     &           daxdxj, daxdyj, daxdzj, daydxj, daydyj, daydzj, &
     &           dazdxj, dazdyj, dazdzj, dadxj, dadyj, dadzj, &
     &           daxdxl, daxdyl, daxdzl, daydxl, daydyl, daydzl, &
     &           dazdxl, dazdyl, dazdzl, dadxl, dadyl, dadzl

      real(8) :: f1, sin_phi, sign_phi

!      real(8)::  daxdxk, daxdyk, daxdzk, daydxk, daydyk, daydzk,
!     &           dazdxk, dazdyk, dazdzk, dadxk, dadyk, dadzk

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( nimproper .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         do n = 1, nimproper

            if ( mod( n-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i = i_improper(n)
            j = j_improper(n)
            k = k_improper(n)
            l = l_improper(n)

!-----------------------------------------------------------------------

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            xkj = x(k,m) - x(j,m)
            ykj = y(k,m) - y(j,m)
            zkj = z(k,m) - z(j,m)

            call pbc_atom_MPI ( xkj, ykj, zkj )

            xlj = x(l,m) - x(j,m)
            ylj = y(l,m) - y(j,m)
            zlj = z(l,m) - z(j,m)

            call pbc_atom_MPI ( xlj, ylj, zlj )

!-----------------------------------------------------------------------

            xijk = yij*zkj - zij*ykj
            yijk = zij*xkj - xij*zkj
            zijk = xij*ykj - yij*xkj

            xjkl = ylj*zkj - zlj*ykj
            yjkl = zlj*xkj - xlj*zkj
            zjkl = xlj*ykj - ylj*xkj

            rijk2  = xijk*xijk + yijk*yijk + zijk*zijk
            rjkl2  = xjkl*xjkl + yjkl*yjkl + zjkl*zjkl

            rijkl2 = sqrt(rijk2*rjkl2)

            if ( abs(rijk2)  .lt. tiny_value ) cycle
            if ( abs(rjkl2)  .lt. tiny_value ) cycle
            if ( abs(rijkl2) .lt. tiny_value ) cycle

            rijk2inv  = 1.d0 / rijk2
            rjkl2inv  = 1.d0 / rjkl2
            rijkl2inv = 1.d0 / rijkl2

!-----------------------------------------------------------------------

            factor_1 = fc_improper(n) * (180.d0/pi) * (180.d0/pi)

            cos_phi = ( xijk*xjkl + yijk*yjkl + zijk*zjkl ) * rijkl2inv

            cos_phi = max( cos_phi, -1.d0 )
            cos_phi = min( cos_phi,  1.d0 )

            phi = acos(cos_phi)

            sign_phi = ( yijk*zjkl - zijk*yjkl ) * xkj &
     &               + ( zijk*xjkl - xijk*zjkl ) * ykj &
     &               + ( xijk*yjkl - yijk*xjkl ) * zkj

            sign_phi = sign( 1.d0, sign_phi )

            phi = phi * sign_phi

            factor_2  = eq_improper(n) * (pi/180.d0)

            dphi = phi - factor_2

!-----------------------------------------------------------------------

            pot(m)  = pot(m) + 0.5d0 * factor_1 * dphi * dphi

!-----------------------------------------------------------------------

            if ( ( abs(phi)    .gt. tiny_value ) .and. &
     &           ( abs(phi-pi) .gt. tiny_value ) .and. &
     &           ( abs(phi+pi) .gt. tiny_value ) ) then

!-----------------------------------------------------------------------

               factor_3 = sin(phi)

               factor_4 = factor_1 * dphi / factor_3

               px1 = yijk*zij - zijk*yij
               py1 = zijk*xij - xijk*zij
               pz1 = xijk*yij - yijk*xij

               px2 = yjkl*zij - zjkl*yij
               py2 = zjkl*xij - xjkl*zij
               pz2 = xjkl*yij - yjkl*xij

               px3 = yijk*zkj - zijk*ykj
               py3 = zijk*xkj - xijk*zkj
               pz3 = xijk*ykj - yijk*xkj

               px4 = yjkl*zkj - zjkl*ykj
               py4 = zjkl*xkj - xjkl*zkj
               pz4 = xjkl*ykj - yjkl*xkj

               px5 = yijk*zlj - zijk*ylj
               py5 = zijk*xlj - xijk*zlj
               pz5 = xijk*ylj - yijk*xlj

               px6 = yjkl*zlj - zjkl*ylj
               py6 = zjkl*xlj - xjkl*zlj
               pz6 = xjkl*ylj - yjkl*xlj

               fxi = factor_4 * ( - px4*rijkl2inv &
     &                            + px3*rijk2inv*cos_phi )
               fyi = factor_4 * ( - py4*rijkl2inv &
     &                            + py3*rijk2inv*cos_phi )
               fzi = factor_4 * ( - pz4*rijkl2inv &
     &                            + pz3*rijk2inv*cos_phi )

               fxk = factor_4 * ( + px2*rijkl2inv &
     &                            + px5*rijkl2inv &
     &                            - px1*rijk2inv*cos_phi &
     &                            - px6*rjkl2inv*cos_phi )
               fyk = factor_4 * ( + py2*rijkl2inv &
     &                            + py5*rijkl2inv &
     &                            - py1*rijk2inv*cos_phi &
     &                            - py6*rjkl2inv*cos_phi )
               fzk = factor_4 * ( + pz2*rijkl2inv &
     &                            + pz5*rijkl2inv &
     &                            - pz1*rijk2inv*cos_phi &
     &                            - pz6*rjkl2inv*cos_phi )

               fxl = factor_4 * ( - px3*rijkl2inv &
     &                            + px4*rjkl2inv*cos_phi )
               fyl = factor_4 * ( - py3*rijkl2inv &
     &                            + py4*rjkl2inv*cos_phi )
               fzl = factor_4 * ( - pz3*rijkl2inv &
     &                            + pz4*rjkl2inv*cos_phi )

               fxj = - ( fxi + fxk + fxl )
               fyj = - ( fyi + fyk + fyl )
               fzj = - ( fzi + fzk + fzl )

!-----------------------------------------------------------------------

            else

!-----------------------------------------------------------------------

               xki = - xij + xkj
               yki = - yij + ykj
               zki = - zij + zkj

               xkl = - xlj + xkj
               ykl = - ylj + ykj
               zkl = - zlj + zkj

               ax = yijk*zjkl - zijk*yjkl
               ay = zijk*xjkl - xijk*zjkl
               az = xijk*yjkl - yijk*xjkl

               a2 = ax*ax + ay*ay + az*az

               a1 = sqrt( a2 )

               sin_phi = a1 / rijkl2

               sin_phi = max( sin_phi, -1.d0 )
               sin_phi = min( sin_phi,  1.d0 )

               phi = sign_phi * asin( sin_phi )

               if ( cos_phi .lt. 0.d0 ) phi = pi - phi

               daxdxi = - zjkl * zkj - ykj * yjkl
               daxdyi = + yjkl * xkj
               daxdzi = + zjkl * xkj

               daydxi = + xjkl * ykj
               daydyi = - xjkl * xkj - zkj * zjkl
               daydzi = + zjkl * ykj

               dazdxi = + xjkl * zkj
               dazdyi = + yjkl * zkj
               dazdzi = - yjkl * ykj - xkj * xjkl

               daxdxj = - yijk * ykl + zjkl * zki &
     &                  + yjkl * yki - zijk * zkl
               daxdyj = + yijk * xkl - yjkl * xki
               daxdzj = + zijk * xkl - zjkl * xki

               daydxj = + xijk * ykl - xjkl * yki
               daydyj = - zijk * zkl + xjkl * xki &
     &                  + zjkl * zki - xijk * xkl
               daydzj = + zijk * ykl - zjkl * yki

               dazdxj = + xijk * zkl - xjkl * zki
               dazdyj = + yijk * zkl - yjkl * zki
               dazdzj = - xijk * xkl + yjkl * yki &
     &                  + xjkl * xki - yijk * ykl

!               daxdxk = - yjkl * yij + zijk * zlj
!                        + yijk * ylj - zjkl * zij
!               daxdyk = + yjkl * xij - yijk * xlj
!               daxdzk = + zjkl * xij - zijk * xlj
!
!               daydxk = + xjkl * yij - xijk * ylj
!               daydyk = - zjkl * zij + xijk * xlj
!                        + zijk * zlj - xjkl * xij
!               daydzk = + zjkl * yij - zijk * ylj
!
!               dazdxk = + xjkl * zij - xijk * zlj
!               dazdyk = + yjkl * zij - yijk * zlj
!               dazdzk = - xjkl * xij + yijk * ylj
!                        + xijk * xlj - yjkl * yij

               daxdxl = + zijk * zkj + ykj * yijk
               daxdyl = - yijk * xkj
               daxdzl = - zijk * xkj

               daydxl = - xijk * ykj
               daydyl = + xijk * xkj + zkj * zijk
               daydzl = - zijk * ykj

               dazdxl = - xijk * zkj
               dazdyl = - yijk * zkj
               dazdzl = + yijk * ykj + xkj * xijk

               dadxi = ax/a1*daxdxi + ay/a1*daydxi + az/a1*dazdxi
               dadyi = ax/a1*daxdyi + ay/a1*daydyi + az/a1*dazdyi
               dadzi = ax/a1*daxdzi + ay/a1*daydzi + az/a1*dazdzi

               dadxj = ax/a1*daxdxj + ay/a1*daydxj + az/a1*dazdxj
               dadyj = ax/a1*daxdyj + ay/a1*daydyj + az/a1*dazdyj
               dadzj = ax/a1*daxdzj + ay/a1*daydzj + az/a1*dazdzj

!               dadxk = ax/a1*daxdxk + ay/a1*daydxk + az/a1*dazdxk
!               dadyk = ax/a1*daxdyk + ay/a1*daydyk + az/a1*dazdyk
!               dadzk = ax/a1*daxdzk + ay/a1*daydzk + az/a1*dazdzk

               dadxl = ax/a1*daxdxl + ay/a1*daydxl + az/a1*dazdxl
               dadyl = ax/a1*daxdyl + ay/a1*daydyl + az/a1*dazdyl
               dadzl = ax/a1*daxdzl + ay/a1*daydzl + az/a1*dazdzl

               f1 = - sign_phi/cos_phi * (180.d0/pi) * (180.d0/pi) &
     &              * fc_improper(n) * dphi

               fxi = + f1 * ( dadxi / rijkl2 &
     &                + sin_phi * ( + yijk*zkj - zijk*ykj ) * rijk2inv )

               fyi = + f1 * ( dadyi / rijkl2 &
     &                + sin_phi * ( + zijk*xkj - xijk*zkj ) * rijk2inv )

               fzi = + f1 * ( dadzi / rijkl2 &
     &                + sin_phi * ( + xijk*ykj - yijk*xkj ) * rijk2inv )

               fxj = + f1 * ( dadxj / rijkl2 &
     &                + sin_phi * ( - yijk*zki + zijk*yki ) * rijk2inv &
     &                - sin_phi * ( + yjkl*zkl - zjkl*ykl ) * rjkl2inv )

               fyj = + f1 * ( dadyj / rijkl2 &
     &                + sin_phi * ( - zijk*xki + xijk*zki ) * rijk2inv &
     &                - sin_phi * ( + zjkl*xkl - xjkl*zkl ) * rjkl2inv )

               fzj = + f1 * ( dadzj / rijkl2 &
     &                + sin_phi * ( - xijk*yki + yijk*xki ) * rijk2inv &
     &                - sin_phi * ( + xjkl*ykl - yjkl*xkl ) * rjkl2inv )

!              fxk = - f1 * ( dadxk / rijkl2
!     &                - sin_phi * ( - yjkl*zlj + zjkl*ylj ) * rjkl2inv
!     &                + sin_phi * ( + yijk*zij - zijk*yij ) * rijk2inv )
!
!              fyk = - f1 * ( dadyk / rijkl2
!     &                - sin_phi * ( - zjkl*xlj + xjkl*zlj ) * rjkl2inv
!     &                + sin_phi * ( + zijk*xij - xijk*zij ) * rijk2inv )
!
!              fzk = - f1 * ( dadzk / rijkl2
!     &                - sin_phi * ( - xjkl*ylj + yjkl*xlj ) * rjkl2inv
!     &                + sin_phi * ( + xijk*yij - yijk*xij ) * rijk2inv )

               fxl = + f1 * ( dadxl / rijkl2 &
     &                + sin_phi * ( + yjkl*zkj - zjkl*ykj ) * rjkl2inv )

               fyl = + f1 * ( dadyl / rijkl2 &
     &                + sin_phi * ( + zjkl*xkj - xjkl*zkj ) * rjkl2inv )

               fzl = + f1 * ( dadzl / rijkl2 &
     &                + sin_phi * ( + xjkl*ykj - yjkl*xkj ) * rjkl2inv )

               fxk = - fxi - fxj - fxl
               fyk = - fyi - fyj - fyl
               fzk = - fzi - fzj - fzl

!-----------------------------------------------------------------------

            end if

!-----------------------------------------------------------------------

            fx(i,m) = fx(i,m) + fxi
            fx(j,m) = fx(j,m) + fxj
            fx(k,m) = fx(k,m) + fxk
            fx(l,m) = fx(l,m) + fxl

            fy(i,m) = fy(i,m) + fyi
            fy(j,m) = fy(j,m) + fyj
            fy(k,m) = fy(k,m) + fyk
            fy(l,m) = fy(l,m) + fyl

            fz(i,m) = fz(i,m) + fzi
            fz(j,m) = fz(j,m) + fzj
            fz(k,m) = fz(k,m) + fzk
            fz(l,m) = fz(l,m) + fzl

!-----------------------------------------------------------------------

            vir(1,1) = vir(1,1) + fxi*xij + fxk*xkj + fxl*xlj
            vir(1,2) = vir(1,2) + fxi*yij + fxk*ykj + fxl*ylj
            vir(1,3) = vir(1,3) + fxi*zij + fxk*zkj + fxl*zlj
            vir(2,1) = vir(2,1) + fyi*xij + fyk*xkj + fyl*xlj
            vir(2,2) = vir(2,2) + fyi*yij + fyk*ykj + fyl*ylj
            vir(2,3) = vir(2,3) + fyi*zij + fyk*zkj + fyl*zlj
            vir(3,1) = vir(3,1) + fzi*xij + fzk*xkj + fzl*xlj
            vir(3,2) = vir(3,2) + fzi*yij + fzk*ykj + fzl*ylj
            vir(3,3) = vir(3,3) + fzi*zij + fzk*zkj + fzl*zlj

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_mm_lj_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, boxinv, box, iboundary, nbead, &
     &   myrank_main, nprocs_main, myrank_sub, nprocs_sub

      use mm_variables, only : &
     &   rout_lj, rin_lj, eps_lj, sig_lj, bigbox, bigboxinv, &
     &   i_lj, j_lj, nlj, nbox_lj

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, l, m, jx, jy, jz, j2

      real(8) :: rout_lj2, xij, yij, zij, rij2, rij, rinv, eps, sig, &
     &           sr, sr2, sr6, sr12, u6, u12, uij, duij, fxi, fyi, fzi, &
     &           swf, dswf, aij, bij, cij, absa, absb, absc

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( nlj .eq. 0 ) return

      rout_lj2 = rout_lj*rout_lj

!-----------------------------------------------------------------------
!     /*   extention of simulation box in real space sum              */
!-----------------------------------------------------------------------

      if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then

         absa = sqrt ( boxinv(1,1)*boxinv(1,1) &
     &               + boxinv(1,2)*boxinv(1,2) &
     &               + boxinv(1,3)*boxinv(1,3) )
         absb = sqrt ( boxinv(2,1)*boxinv(2,1) &
     &               + boxinv(2,2)*boxinv(2,2) &
     &               + boxinv(2,3)*boxinv(2,3) )
         absc = sqrt ( boxinv(3,1)*boxinv(3,1) &
     &               + boxinv(3,2)*boxinv(3,2) &
     &               + boxinv(3,3)*boxinv(3,3) )

         nbox_lj(1) = int(2.d0*rout_lj*absa) + 1
         nbox_lj(2) = int(2.d0*rout_lj*absb) + 1
         nbox_lj(3) = int(2.d0*rout_lj*absc) + 1

      end if

!-----------------------------------------------------------------------
!     /*   main loop : free boundary                                  */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) then

         do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         do l = 1, nlj

            if ( mod( l-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i = i_lj(l)
            j = j_lj(l)

            if ( i .eq. j ) cycle

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            rij2 = xij*xij + yij*yij + zij*zij

            if ( rij2 .gt. rout_lj2 ) cycle

            rij     = sqrt(rij2)

            rinv    = 1.d0/rij

            eps     = eps_lj(l)
            sig     = sig_lj(l)

            sr      = sig*rinv
            sr2     = sr*sr
            sr6     = sr2*sr2*sr2
            sr12    = sr6*sr6

            u6      = - 4.d0*eps*sr6
            u12     = + 4.d0*eps*sr12

!           /*   switching function   */
            call getswf( rij, rin_lj, rout_lj, swf, dswf )

!           /*   bare potential   */
            uij     = + u6 + u12

!           /*   bare potential gradient   */
            duij    = ( - 6.d0*u6*rinv - 12.d0*u12*rinv )*swf

!           /*   corrected potential   */
            pot(m)  = pot(m) + uij*swf

!           /*   corrected forces   */

            fxi = - uij*dswf*xij*rinv - duij*xij*rinv
            fyi = - uij*dswf*yij*rinv - duij*yij*rinv
            fzi = - uij*dswf*zij*rinv - duij*zij*rinv

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

         end do

         end do

!-----------------------------------------------------------------------
!     /*   main loop : periodic boundary with minimum image           */
!-----------------------------------------------------------------------

      else if ( nbox_lj(1)*nbox_lj(2)*nbox_lj(3) .eq. 1 ) then

         do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         do l = 1, nlj

            if ( mod( l-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i = i_lj(l)
            j = j_lj(l)

            if ( i .eq. j ) cycle

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            rij2 = xij*xij + yij*yij + zij*zij

            if ( rij2 .gt. rout_lj2 ) cycle

            rij     = sqrt(rij2)

            rinv    = 1.d0/rij

            eps     = eps_lj(l)
            sig     = sig_lj(l)

            sr      = sig*rinv
            sr2     = sr*sr
            sr6     = sr2*sr2*sr2
            sr12    = sr6*sr6

            u6      = - 4.d0*eps*sr6
            u12     = + 4.d0*eps*sr12

!           /*   switching function   */
            call getswf( rij, rin_lj, rout_lj, swf, dswf )

!           /*   bare potential   */
            uij     = + u6 + u12

!           /*   bare potential gradient   */
            duij    = ( - 6.d0*u6*rinv - 12.d0*u12*rinv )*swf

!           /*   corrected potential   */
            pot(m)  = pot(m) + uij*swf

!           /*   corrected forces   */

            fxi = - uij*dswf*xij*rinv - duij*xij*rinv
            fyi = - uij*dswf*yij*rinv - duij*yij*rinv
            fzi = - uij*dswf*zij*rinv - duij*zij*rinv

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

         end do

         end do

!-----------------------------------------------------------------------
!     /*   main loop : periodic boundary                              */
!-----------------------------------------------------------------------

      else

         bigbox(:,1) = dble(nbox_lj(1))*box(:,1)
         bigbox(:,2) = dble(nbox_lj(2))*box(:,2)
         bigbox(:,3) = dble(nbox_lj(3))*box(:,3)

         call inv3 ( bigbox, bigboxinv )

         do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         do l = 1, nlj

            if ( mod( l-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i = i_lj(l)
            j = j_lj(l)

            do jx = 0, nbox_lj(1)-1
            do jy = 0, nbox_lj(2)-1
            do jz = 0, nbox_lj(3)-1

               j2 = jx*jx + jy*jy + jz*jz

               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

               rij2 = xij*xij + yij*yij + zij*zij

               if ( rij2 .gt. rout_lj2 ) cycle

               rij     = sqrt(rij2)

               rinv    = 1.d0/rij

               eps     = eps_lj(l)
               sig     = sig_lj(l)

               sr      = sig*rinv
               sr2     = sr*sr
               sr6     = sr2*sr2*sr2
               sr12    = sr6*sr6

               u6      = - 4.d0*eps*sr6
               u12     = + 4.d0*eps*sr12

!              /*   switching function   */
               call getswf( rij, rin_lj, rout_lj, swf, dswf )

!              /*   bare potential   */
               uij     = + u6 + u12

!              /*   bare potential gradient   */
               duij    = ( - 6.d0*u6*rinv - 12.d0*u12*rinv )*swf

!              /*   corrected potential   */
               pot(m)  = pot(m) + uij*swf*0.5d0

!              /*   corrected forces   */

               fxi = - uij*dswf*xij*rinv - duij*xij*rinv
               fyi = - uij*dswf*yij*rinv - duij*yij*rinv
               fzi = - uij*dswf*zij*rinv - duij*zij*rinv

               fx(i,m) = fx(i,m) + 0.5d0*fxi
               fy(i,m) = fy(i,m) + 0.5d0*fyi
               fz(i,m) = fz(i,m) + 0.5d0*fzi

               fx(j,m) = fx(j,m) - 0.5d0*fxi
               fy(j,m) = fy(j,m) - 0.5d0*fyi
               fz(j,m) = fz(j,m) - 0.5d0*fzi

               vir(1,1) = vir(1,1) + 0.5d0*fxi*xij
               vir(1,2) = vir(1,2) + 0.5d0*fxi*yij
               vir(1,3) = vir(1,3) + 0.5d0*fxi*zij
               vir(2,1) = vir(2,1) + 0.5d0*fyi*xij
               vir(2,2) = vir(2,2) + 0.5d0*fyi*yij
               vir(2,3) = vir(2,3) + 0.5d0*fyi*zij
               vir(3,1) = vir(3,1) + 0.5d0*fzi*xij
               vir(3,2) = vir(3,2) + 0.5d0*fzi*yij
               vir(3,3) = vir(3,3) + 0.5d0*fzi*zij

            end do
            end do
            end do

            if ( i .eq. j ) cycle

            i = j_lj(l)
            j = i_lj(l)

            do jx = 0, nbox_lj(1)-1
            do jy = 0, nbox_lj(2)-1
            do jz = 0, nbox_lj(3)-1

               j2 = jx*jx + jy*jy + jz*jz

               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

               rij2 = xij*xij + yij*yij + zij*zij

               if ( rij2 .gt. rout_lj2 ) cycle

               rij     = sqrt(rij2)

               rinv    = 1.d0/rij

               eps     = eps_lj(l)
               sig     = sig_lj(l)

               sr      = sig*rinv
               sr2     = sr*sr
               sr6     = sr2*sr2*sr2
               sr12    = sr6*sr6

               u6      = - 4.d0*eps*sr6
               u12     = + 4.d0*eps*sr12

!              /*   switching function   */
               call getswf( rij, rin_lj, rout_lj, swf, dswf )

!              /*   bare potential   */
               uij     = + u6 + u12

!              /*   bare potential gradient   */
               duij    = ( - 6.d0*u6*rinv - 12.d0*u12*rinv )*swf

!              /*   corrected potential   */
               pot(m)  = pot(m) + uij*swf*0.5d0

!              /*   corrected forces   */

               fxi = - uij*dswf*xij*rinv - duij*xij*rinv
               fyi = - uij*dswf*yij*rinv - duij*yij*rinv
               fzi = - uij*dswf*zij*rinv - duij*zij*rinv

               fx(i,m) = fx(i,m) + 0.5d0*fxi
               fy(i,m) = fy(i,m) + 0.5d0*fyi
               fz(i,m) = fz(i,m) + 0.5d0*fzi

               fx(j,m) = fx(j,m) - 0.5d0*fxi
               fy(j,m) = fy(j,m) - 0.5d0*fyi
               fz(j,m) = fz(j,m) - 0.5d0*fzi

               vir(1,1) = vir(1,1) + 0.5d0*fxi*xij
               vir(1,2) = vir(1,2) + 0.5d0*fxi*yij
               vir(1,3) = vir(1,3) + 0.5d0*fxi*zij
               vir(2,1) = vir(2,1) + 0.5d0*fyi*xij
               vir(2,2) = vir(2,2) + 0.5d0*fyi*yij
               vir(2,3) = vir(2,3) + 0.5d0*fyi*zij
               vir(3,1) = vir(3,1) + 0.5d0*fzi*xij
               vir(3,2) = vir(3,2) + 0.5d0*fzi*yij
               vir(3,3) = vir(3,3) + 0.5d0*fzi*zij

            end do
            end do
            end do

         end do

         end do

      end if

      return
      end





!***********************************************************************
      subroutine force_mm_buck_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, boxinv, box, iboundary, nbead, &
     &   myrank_main, nprocs_main, myrank_sub, nprocs_sub

      use mm_variables, only : &
     &   rout_buck, rin_buck, a_buck, b_buck, c_buck, bigbox, bigboxinv, &
     &   i_buck, j_buck, nbuck, nbox_buck

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, l, m, jx, jy, jz, j2

      real(8) :: rout_buck2, xij, yij, zij, rij2, rij, rinv, a, b, c, &
     &           rinv2, rinv6, u6, ue, uij, duij, fxi, fyi, fzi, &
     &           swf, dswf, aij, bij, cij, absa, absb, absc

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( nbuck .eq. 0 ) return

      rout_buck2 = rout_buck*rout_buck

!-----------------------------------------------------------------------
!     /*   extention of simulation box in real space sum              */
!-----------------------------------------------------------------------

      if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then

         absa = sqrt ( boxinv(1,1)*boxinv(1,1) &
     &               + boxinv(1,2)*boxinv(1,2) &
     &               + boxinv(1,3)*boxinv(1,3) )
         absb = sqrt ( boxinv(2,1)*boxinv(2,1) &
     &               + boxinv(2,2)*boxinv(2,2) &
     &               + boxinv(2,3)*boxinv(2,3) )
         absc = sqrt ( boxinv(3,1)*boxinv(3,1) &
     &               + boxinv(3,2)*boxinv(3,2) &
     &               + boxinv(3,3)*boxinv(3,3) )

         nbox_buck(1) = int(2.d0*rout_buck*absa) + 1
         nbox_buck(2) = int(2.d0*rout_buck*absb) + 1
         nbox_buck(3) = int(2.d0*rout_buck*absc) + 1

      end if

!-----------------------------------------------------------------------
!     /*   main loop : free boundary                                  */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) then

         do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         do l = 1, nbuck

            if ( mod( l-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i = i_buck(l)
            j = j_buck(l)

            if ( i .eq. j ) cycle

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            rij2 = xij*xij + yij*yij + zij*zij

            if ( rij2 .gt. rout_buck2 ) cycle

            rij     = sqrt(rij2)

            rinv    = 1.d0 / rij

            a     = a_buck(l)
            b     = b_buck(l)
            c     = c_buck(l)

            rinv2 = rinv * rinv
            rinv6 = rinv2 * rinv2 * rinv2

            u6     = - c*rinv**6
            ue     = + a*exp(-b*rij)

!           /*   switching function   */
            call getswf( rij, rin_buck, rout_buck, swf, dswf )

!           /*   bare potential   */
            uij     = + u6 + ue

!           /*   bare potential gradient   */
            duij    = ( - 6.d0*u6*rinv - b*ue ) * swf

!           /*   corrected potential   */
            pot(m)  = pot(m) + uij * swf

!           /*   corrected forces   */

            fxi = - uij*dswf*xij*rinv - duij*xij*rinv
            fyi = - uij*dswf*yij*rinv - duij*yij*rinv
            fzi = - uij*dswf*zij*rinv - duij*zij*rinv

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

         end do

         end do

!-----------------------------------------------------------------------
!     /*   main loop : periodic boundary with minimum image           */
!-----------------------------------------------------------------------

      else if ( nbox_buck(1)*nbox_buck(2)*nbox_buck(3) .eq. 1 ) then

         do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         do l = 1, nbuck

            if ( mod( l-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i = i_buck(l)
            j = j_buck(l)

            if ( i .eq. j ) cycle

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            rij2 = xij*xij + yij*yij + zij*zij

            if ( rij2 .gt. rout_buck2 ) cycle

            rij     = sqrt(rij2)

            rinv    = 1.d0/rij

            a     = a_buck(l)
            b     = b_buck(l)
            c     = c_buck(l)

            rinv2 = rinv * rinv
            rinv6 = rinv2 * rinv2 * rinv2

            u6     = - c*rinv**6
            ue     = + a*exp(-b*rij)

!           /*   switching function   */
            call getswf( rij, rin_buck, rout_buck, swf, dswf )

!           /*   bare potential   */
            uij     = + u6 + ue

!           /*   bare potential gradient   */
            duij    = ( - 6.d0*u6*rinv - b*ue ) * swf

!           /*   corrected potential   */
            pot(m)  = pot(m) + uij * swf

!           /*   corrected forces   */

            fxi = - uij*dswf*xij*rinv - duij*xij*rinv
            fyi = - uij*dswf*yij*rinv - duij*yij*rinv
            fzi = - uij*dswf*zij*rinv - duij*zij*rinv

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

         end do

         end do

!-----------------------------------------------------------------------
!     /*   main loop : periodic boundary                              */
!-----------------------------------------------------------------------

      else

         bigbox(:,1) = dble(nbox_buck(1))*box(:,1)
         bigbox(:,2) = dble(nbox_buck(2))*box(:,2)
         bigbox(:,3) = dble(nbox_buck(3))*box(:,3)

         call inv3 ( bigbox, bigboxinv )

         do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         do l = 1, nbuck

            if ( mod( l-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i = i_buck(l)
            j = j_buck(l)

            do jx = 0, nbox_buck(1)-1
            do jy = 0, nbox_buck(2)-1
            do jz = 0, nbox_buck(3)-1

               j2 = jx*jx + jy*jy + jz*jz

               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

               rij2 = xij*xij + yij*yij + zij*zij

               if ( rij2 .gt. rout_buck2 ) cycle

               rij     = sqrt(rij2)

               rinv    = 1.d0/rij

               a     = a_buck(l)
               b     = b_buck(l)
               c     = c_buck(l)

               rinv2 = rinv * rinv
               rinv6 = rinv2 * rinv2 * rinv2

               u6     = - c*rinv**6
               ue     = + a*exp(-b*rij)

!              /*   switching function   */
               call getswf( rij, rin_buck, rout_buck, swf, dswf )

!              /*   bare potential   */
               uij     = + u6 + ue

!              /*   bare potential gradient   */
               duij    = ( - 6.d0*u6*rinv - b*ue ) * swf

!              /*   corrected potential   */
               pot(m)  = pot(m) + uij * swf * 0.5d0

!              /*   corrected forces   */

               fxi = - uij*dswf*xij*rinv - duij*xij*rinv
               fyi = - uij*dswf*yij*rinv - duij*yij*rinv
               fzi = - uij*dswf*zij*rinv - duij*zij*rinv

               fx(i,m) = fx(i,m) + 0.5d0*fxi
               fy(i,m) = fy(i,m) + 0.5d0*fyi
               fz(i,m) = fz(i,m) + 0.5d0*fzi

               fx(j,m) = fx(j,m) - 0.5d0*fxi
               fy(j,m) = fy(j,m) - 0.5d0*fyi
               fz(j,m) = fz(j,m) - 0.5d0*fzi

               vir(1,1) = vir(1,1) + 0.5d0*fxi*xij
               vir(1,2) = vir(1,2) + 0.5d0*fxi*yij
               vir(1,3) = vir(1,3) + 0.5d0*fxi*zij
               vir(2,1) = vir(2,1) + 0.5d0*fyi*xij
               vir(2,2) = vir(2,2) + 0.5d0*fyi*yij
               vir(2,3) = vir(2,3) + 0.5d0*fyi*zij
               vir(3,1) = vir(3,1) + 0.5d0*fzi*xij
               vir(3,2) = vir(3,2) + 0.5d0*fzi*yij
               vir(3,3) = vir(3,3) + 0.5d0*fzi*zij

            end do
            end do
            end do

            if ( i .eq. j ) cycle

            i = j_buck(l)
            j = i_buck(l)

            do jx = 0, nbox_buck(1)-1
            do jy = 0, nbox_buck(2)-1
            do jz = 0, nbox_buck(3)-1

               j2 = jx*jx + jy*jy + jz*jz

               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

               rij2 = xij*xij + yij*yij + zij*zij

               if ( rij2 .gt. rout_buck2 ) cycle

               rij     = sqrt(rij2)

               rinv    = 1.d0/rij

               a     = a_buck(l)
               b     = b_buck(l)
               c     = c_buck(l)

               rinv2 = rinv * rinv
               rinv6 = rinv2 * rinv2 * rinv2

               u6     = - c*rinv**6
               ue     = + a*exp(-b*rij)

!              /*   switching function   */
               call getswf( rij, rin_buck, rout_buck, swf, dswf )

!              /*   bare potential   */
               uij     = + u6 + ue

!              /*   bare potential gradient   */
               duij    = ( - 6.d0*u6*rinv - b*ue ) * swf

!              /*   corrected potential   */
               pot(m)  = pot(m) + uij * swf * 0.5d0

!              /*   corrected forces   */

               fxi = - uij*dswf*xij*rinv - duij*xij*rinv
               fyi = - uij*dswf*yij*rinv - duij*yij*rinv
               fzi = - uij*dswf*zij*rinv - duij*zij*rinv

               fx(i,m) = fx(i,m) + 0.5d0*fxi
               fy(i,m) = fy(i,m) + 0.5d0*fyi
               fz(i,m) = fz(i,m) + 0.5d0*fzi

               fx(j,m) = fx(j,m) - 0.5d0*fxi
               fy(j,m) = fy(j,m) - 0.5d0*fyi
               fz(j,m) = fz(j,m) - 0.5d0*fzi

               vir(1,1) = vir(1,1) + 0.5d0*fxi*xij
               vir(1,2) = vir(1,2) + 0.5d0*fxi*yij
               vir(1,3) = vir(1,3) + 0.5d0*fxi*zij
               vir(2,1) = vir(2,1) + 0.5d0*fyi*xij
               vir(2,2) = vir(2,2) + 0.5d0*fyi*yij
               vir(2,3) = vir(2,3) + 0.5d0*fyi*zij
               vir(3,1) = vir(3,1) + 0.5d0*fzi*xij
               vir(3,2) = vir(3,2) + 0.5d0*fzi*yij
               vir(3,3) = vir(3,3) + 0.5d0*fzi*zij

            end do
            end do
            end do

         end do

         end do

      end if

      return
      end





!***********************************************************************
      subroutine force_mm_morse_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pot, fx, fy, fz, vir, nbead, &
     &   myrank_main, nprocs_main, myrank_sub, nprocs_sub

      use mm_variables, only : &
     &   depth_morse, alpha_morse, eq_morse, i_morse, j_morse, nmorse

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, m

      real(8) :: alpha, const, depth, dr, factor, expfactor, &
     &           fxi, fyi, fzi, rij, xij, yij, zij

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( nmorse .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         do k = 1, nmorse

            if ( mod( k-1, nprocs_sub ) .ne. myrank_sub ) cycle

            i     =      i_morse(k)
            j     =      j_morse(k)
            depth =  depth_morse(k)
            alpha =  alpha_morse(k)

            if ( i .eq. j ) cycle

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI ( xij, yij, zij )

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            dr  = ( rij - eq_morse(k) )

            expfactor = exp(-alpha*dr)

            factor = 1.d0 - expfactor

            pot(m) = pot(m) + depth*( factor*factor - 1.d0 )

            const = - 2.d0*depth*factor*alpha*expfactor/rij

            fxi = const*xij
            fyi = const*yij
            fzi = const*zij

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_mm_morse_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, myrank

      use mm_variables, only : &
     &   depth_morse, alpha_morse, eq_morse, i_morse, j_morse, nmorse

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ierr, k

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

      open (iounit,file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<morse>', 7, iounit, ierr )

!        /*   number of morse interactions   */
         read( iounit, *, iostat=ierr ) nmorse

         if ( ierr .ne. 0 ) nmorse = 0

      close(iounit)

      end if

      call my_mpi_bcast_int_0( nmorse )

      if ( nmorse .eq. 0 ) return

      if ( .not. allocated(     i_morse ) ) &
     &   allocate(     i_morse(nmorse) )
      if ( .not. allocated(     j_morse ) ) &
     &   allocate(     j_morse(nmorse) )
      if ( .not. allocated(    eq_morse ) ) &
     &   allocate(    eq_morse(nmorse) )
      if ( .not. allocated( depth_morse ) ) &
     &   allocate( depth_morse(nmorse) )
      if ( .not. allocated( alpha_morse ) ) &
     &   allocate( alpha_morse(nmorse) )

      if ( myrank .eq. 0 ) then

      open (iounit,file = 'mm.dat')

!        /*   tag   */
         call search_tag ( '<morse>', 7, iounit, ierr )

!        /*   number of morse interactions   */
         read( iounit, *, iostat=ierr ) nmorse

!        /*   atom 1, atom 2, equilibrium, depth, exponent  */
         do k = 1, nmorse
            read( iounit, *, iostat=ierr ) &
     &         i_morse(k), j_morse(k), eq_morse(k), &
     &         depth_morse(k), alpha_morse(k)
         end do

      close(iounit)

      end if

      call my_mpi_bcast_int_1 ( i_morse,     nmorse )
      call my_mpi_bcast_int_1 ( j_morse,     nmorse )
      call my_mpi_bcast_real_1( eq_morse,    nmorse )
      call my_mpi_bcast_real_1( depth_morse, nmorse )
      call my_mpi_bcast_real_1( alpha_morse, nmorse )

      return
      end





!***********************************************************************
      subroutine force_ewald_dipole_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables from PIMD                                 */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, volume, pi, mbox, nbead, &
     &   nprocs_main, myrank_main, myrank_sub

      use mm_variables, only : i_q, ncharge, q

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   reset   */
      implicit none

!     /*   integers   */
      integer :: i, j, k, m1, m2, m3

!     /*   real numbers   */
      real(8) :: dx, dy, dz, d2, factor, xi, yi, zi

!-----------------------------------------------------------------------
!     /*   shared variables from PIMD                                 */
!-----------------------------------------------------------------------

!     /*   constant   */
      factor = 2.d0 * pi / ( 3.d0 * volume )

!     /*   loop of beads   */
      do j = 1, nbead

!        /*   parallel calculation   */
         if ( mod( j-1, nprocs_main ) .ne. myrank_main ) cycle

!        /*   parallel calculation   */
         if ( myrank_sub .ne. 0 ) cycle

!        /*   initialize   */
         dx = 0.d0
         dy = 0.d0
         dz = 0.d0

!        /*   loop of charges   */
         do k = 1, ncharge

!           /*   atom   */
            i  = i_q(k)

!           /*   coordinates   */
            xi = x(i,j)
            yi = y(i,j)
            zi = z(i,j)

!           /*   box number   */
            m1 = mbox(1,i,j)
            m2 = mbox(2,i,j)
            m3 = mbox(3,i,j)

!           /*   apply periodic boundary condition   */
            call pbc_unfold_MPI ( xi, yi, zi, m1, m2, m3 )

!           /*   dipole moment   */
            dx = dx + q(i) * xi
            dy = dy + q(i) * yi
            dz = dz + q(i) * zi

!        /*   loop of atoms   */
         end do

!        /*   square of dipole moment   */
         d2 = dx*dx + dy*dy + dz*dz

!        /*   potential   */
         pot(j) = pot(j) + factor * d2

!        /*   loop of charges   */
         do k = 1, ncharge

!           /*   atom   */
            i  = i_q(k)

!           /*   coordinates   */
            xi = x(i,j)
            yi = y(i,j)
            zi = z(i,j)

!           /*   box number   */
            m1 = mbox(1,i,j)
            m2 = mbox(2,i,j)
            m3 = mbox(3,i,j)

!           /*   apply periodic boundary condition   */
            call pbc_unfold_MPI ( xi, yi, zi, m1, m2, m3 )

!           /*   forces   */
            fx(i,j) = fx(i,j) - 2.d0 * factor * q(i) * dx
            fy(i,j) = fy(i,j) - 2.d0 * factor * q(i) * dy
            fz(i,j) = fz(i,j) - 2.d0 * factor * q(i) * dz

         end do

!        /*   virial   */
         vir(1,1) = vir(1,1) - 2.d0 * factor * dx * dx + factor * d2
         vir(1,2) = vir(1,2) - 2.d0 * factor * dx * dy
         vir(1,3) = vir(1,3) - 2.d0 * factor * dx * dz
         vir(2,1) = vir(2,1) - 2.d0 * factor * dy * dx
         vir(2,2) = vir(2,2) - 2.d0 * factor * dy * dy + factor * d2
         vir(2,3) = vir(2,3) - 2.d0 * factor * dy * dz
         vir(3,1) = vir(3,1) - 2.d0 * factor * dz * dx
         vir(3,2) = vir(3,2) - 2.d0 * factor * dz * dy
         vir(3,3) = vir(3,3) - 2.d0 * factor * dz * dz + factor * d2

!     /*   loop of beads   */
      end do

      return
      end





!***********************************************************************
      subroutine force_mm_cmap_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   au2kcal, iounit, myrank

      use mm_variables, only : &
     &   xgrid_cmap, ygrid_cmap, vgrid_cmap, v2grid_cmap, &
     &   i_cmap, j_cmap, k_cmap, l_cmap, ikind_cmap, ncmap, mgrid_cmap, &
     &   ngrid_cmap, nkind_cmap, nbuff_cmap

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer i, j, k, l, ierr

!-----------------------------------------------------------------------
!     /*   set up                                                     */
!-----------------------------------------------------------------------

!     /*   flag   */
      ierr = 0

!     /*   master rank   */
      if ( myrank .eq. 0 ) then

!     /*   use mm file   */
      open ( iounit, file = 'mm.dat' )

!        /*   tag   */
         call search_tag ( '<ncmap>', 7, iounit, ierr )

!        /*   number of cmap dihedrals   */
         if ( ierr .eq. 0 ) read( iounit, *, iostat=ierr ) ncmap

!     /*   use mm file   */
      close( iounit )

!     /*   master rank   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   no cmap if not found   */
      if ( ierr .ne. 0 ) ncmap = 0

!     /*   communicate   */
      call my_mpi_bcast_int_0( ncmap )

!     /*   return if no cmap   */
      if ( ncmap .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( .not. allocated( i_cmap ) ) allocate( i_cmap(2,ncmap) )
      if ( .not. allocated( j_cmap ) ) allocate( j_cmap(2,ncmap) )
      if ( .not. allocated( k_cmap ) ) allocate( k_cmap(2,ncmap) )
      if ( .not. allocated( l_cmap ) ) allocate( l_cmap(2,ncmap) )

      if ( .not. allocated( ikind_cmap ) ) allocate( ikind_cmap(ncmap) )

!-----------------------------------------------------------------------
!     /*   set up cmap dihedrals                                      */
!-----------------------------------------------------------------------

!     /*   flag   */
      ierr = 0

!     /*   master rank   */
      if ( myrank .eq. 0 ) then

!     /*   use mm file   */
      open ( iounit, file = 'mm.dat' )

!        /*   tag   */
         call search_tag ( '<ncmap>', 7, iounit, ierr )

!        /*   number of cmap dihedrals   */
         read( iounit, *, iostat=ierr ) ncmap

!        /*   4 atoms of phi, 4 atoms of psi, and cmap kind   */
         do l = 1, ncmap
            read( iounit, *, iostat=ierr ) &
     &         i_cmap(1,l), j_cmap(1,l), k_cmap(1,l), l_cmap(1,l), &
     &         i_cmap(2,l), j_cmap(2,l), k_cmap(2,l), l_cmap(2,l), &
     &         ikind_cmap(l)
         end do

!     /*   use mm file   */
      close( iounit )

!     /*   master rank   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   stop on error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_mm_cmap_setup_MPI', 34 )

!     /*   communicate   */
      call my_mpi_bcast_int_2( i_cmap, 2, ncmap )
      call my_mpi_bcast_int_2( j_cmap, 2, ncmap )
      call my_mpi_bcast_int_2( k_cmap, 2, ncmap )
      call my_mpi_bcast_int_2( l_cmap, 2, ncmap )
      call my_mpi_bcast_int_1( ikind_cmap, ncmap )

!-----------------------------------------------------------------------
!     /*   set up cmap kinds and grids                                */
!-----------------------------------------------------------------------

!c     /*   flag   */
!      ierr = 0
!
!c     /*   master rank   */
!      if ( myrank .eq. 0 ) then
!
!c     /*   use mm file   */
!      open ( iounit, file = 'mm.dat' )
!
!c        /*   tag   */
!         call search_tag ( '<ngrid_cmap>', 6, iounit, ierr )
!
!c        /*   number of grids of cmap   */
!         read( iounit, *, iostat=ierr ) ngrid_cmap
!
!c     /*   use mm file   */
!      close( iounit )
!
!c     /*   master rank   */
!      end if
!
!c     /*   communicate   */
!      call my_mpi_bcast_int_0( ierr )
!
!c     /*   stop on error   */
!      call error_handling_MPI
!     &   ( ierr, 'subroutine force_mm_cmap_setup_MPI', 34 )
!
!c     /*   communicate   */
!      call my_mpi_bcast_int_0( ngrid_cmap )

!     /*   number of grids including buffer   */
      mgrid_cmap = ngrid_cmap + nbuff_cmap

!-----------------------------------------------------------------------
!     /*   set up cmap kinds and grids                                */
!-----------------------------------------------------------------------

!     /*   flag   */
      ierr = 0

!     /*   master rank   */
      if ( myrank .eq. 0 ) then

!     /*   reset number of kinds of cmap   */
      nkind_cmap = 0

!     /*   use mm file   */
      open ( iounit, file = 'mm.dat' )

!        /*   tag   */
         call search_tag ( '<nkind_cmap>', 6, iounit, ierr )

!        /*   number of kinds of cmap   */
         read( iounit, *, iostat=ierr ) nkind_cmap

!     /*   use mm file   */
      close( iounit )

!     /*   master rank   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   stop on error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_mm_cmap_setup_MPI', 34 )

!     /*   communicate   */
      call my_mpi_bcast_int_0( nkind_cmap )

!     /*   check   */
      if ( maxval(ikind_cmap(:)) .gt. nkind_cmap ) ierr = 1

!     /*   stop on error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_mm_cmap_setup_MPI', 34 )

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( .not. allocated( vgrid_cmap ) ) &
     &   allocate( vgrid_cmap(mgrid_cmap,mgrid_cmap,nkind_cmap) )

      if ( .not. allocated( v2grid_cmap ) ) &
     &   allocate( v2grid_cmap(mgrid_cmap,mgrid_cmap,nkind_cmap) )

      if ( .not. allocated( xgrid_cmap ) ) &
     &   allocate( xgrid_cmap(mgrid_cmap) )

      if ( .not. allocated( ygrid_cmap ) ) &
     &   allocate( ygrid_cmap(mgrid_cmap) )

!-----------------------------------------------------------------------
!     /*   read cmap grids                                            */
!-----------------------------------------------------------------------

!     /*   flag   */
      ierr = 0

!     /*   master rank   */
      if ( myrank .eq. 0 ) then

!     /*   use mm file   */
      open ( iounit, file = 'mm.dat' )

!        /*   tag   */
         call search_tag ( '<nkind_cmap>', 6, iounit, ierr )

!        /*   number of kinds of cmap   */
         read( iounit, *, iostat=ierr ) nkind_cmap

!        /*   loop of cmap kinds   */
         do k = 1, nkind_cmap

!           /*   cmap kind   */
            read( iounit, *, iostat=ierr ) l

!           /*   loop of cmap grid values   */
            do j = 1, ngrid_cmap

!c              /*   grid data   */
!               read( iounit, *, iostat=ierr )
!     &            ( vgrid_cmap(i,j,k), i = 1, ngrid_cmap )

!c              /*   grid data   */
               read( iounit, *, iostat=ierr ) &
     &            ( vgrid_cmap(j,i,k), i = 1, ngrid_cmap )

!           /*   loop of cmap grids   */
            end do

!        /*   loop of cmap kinds   */
         end do

!     /*   use mm file   */
      close( iounit )

!     /*   master rank   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   stop on error   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_mm_cmap_setup_MPI', 34 )

!     /*   communicate   */
      call my_mpi_bcast_real_3 &
     &   ( vgrid_cmap, ngrid_cmap, ngrid_cmap, nkind_cmap )

!     /*   change units to au   */
      vgrid_cmap(:,:,:) = vgrid_cmap(:,:,:) / au2kcal

!-----------------------------------------------------------------------
!     /*   set up cmap grids                                          */
!-----------------------------------------------------------------------

!     /*   loop of cmap grids   */

      do i = 1, mgrid_cmap
         xgrid_cmap(i) = -180.d0 + dble(i-1)/dble(ngrid_cmap)*360.d0
         ygrid_cmap(i) = -180.d0 + dble(i-1)/dble(ngrid_cmap)*360.d0
      end do

!     /*   copy cmap grids to buffer region   */

      do k = 1, nkind_cmap
      do j = 1, ngrid_cmap
      do i = 1, nbuff_cmap
         vgrid_cmap(ngrid_cmap+i,j,k) = vgrid_cmap(i,j,k)
      end do
      end do
      end do

!     /*   copy cmap grids to buffer region   */

      do k = 1, nkind_cmap
      do j = 1, nbuff_cmap
      do i = 1, mgrid_cmap
         vgrid_cmap(i,ngrid_cmap+j,k) = vgrid_cmap(i,j,k)
      end do
      end do
      end do

!-----------------------------------------------------------------------
!     /*   prepare for bicubic spline interpolation                   */
!-----------------------------------------------------------------------

      do k = 1, nkind_cmap

         call splie2_cmap( ygrid_cmap(:), vgrid_cmap(:,:,k), &
     &                     v2grid_cmap(:,:,k), mgrid_cmap, mgrid_cmap )

      end do

      return
      end





!***********************************************************************
      subroutine force_mm_cmap_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, x, y, z, fx, fy, fz, pot, vir, nbead, &
     &   myrank_main, nprocs_main, myrank_sub, nprocs_sub

      use mm_variables, only : &
     &   xgrid_cmap, ygrid_cmap, vgrid_cmap, v2grid_cmap, &
     &   i_cmap, j_cmap, k_cmap, l_cmap, ncmap, mgrid_cmap

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, n, i1or2

      real(8) :: xij, yij, zij, xkj, ykj, zkj, xlj, ylj, zlj, xijk, &
     &           yijk, zijk, xjkl, yjkl, zjkl, rijk2, rjkl2, rijkl2, &
     &           rijk2inv, rjkl2inv, rijkl2inv, cos_phi, sign_phi, &
     &           v, phi(2), dvdphi(2), fxi, fyi, fzi, fxj, fyj, fzj, &
     &           fxk, fyk, fzk, fxl, fyl, fzl, factor, sin_phi, f1, &
     &           xki, yki, zki, xkl, ykl, zkl, ax, ay, az, a1, a2, &
     &           daxdxi, daxdyi, daxdzi, daydxi, daydyi, daydzi, &
     &           dazdxi, dazdyi, dazdzi, dadxi, dadyi, dadzi, &
     &           daxdxj, daxdyj, daxdzj, daydxj, daydyj, daydzj, &
     &           dazdxj, dazdyj, dazdzj, dadxj, dadyj, dadzj, &
     &           daxdxl, daxdyl, daxdzl, daydxl, daydyl, daydzl, &
     &           dazdxl, dazdyl, dazdzl, dadxl, dadyl, dadzl, &
     &           phi1, phi2, p1c, p2c, dvdphi1, dvdphi2

      real(8) :: tiny_value = 1.d-4

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( ncmap .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         do n = 1, ncmap

            if ( mod( n-1, nprocs_sub ) .ne. myrank_sub ) cycle

            do i1or2 = 1, 2

               i = i_cmap(i1or2,n)
               j = j_cmap(i1or2,n)
               k = k_cmap(i1or2,n)
               l = l_cmap(i1or2,n)

               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

               call pbc_atom_MPI ( xij, yij, zij )

               xkj = x(k,m) - x(j,m)
               ykj = y(k,m) - y(j,m)
               zkj = z(k,m) - z(j,m)

               call pbc_atom_MPI ( xkj, ykj, zkj )

               xlj = x(l,m) - x(j,m)
               ylj = y(l,m) - y(j,m)
               zlj = z(l,m) - z(j,m)

               call pbc_atom_MPI ( xlj, ylj, zlj )

               xijk = yij*zkj - zij*ykj
               yijk = zij*xkj - xij*zkj
               zijk = xij*ykj - yij*xkj

               xjkl = ylj*zkj - zlj*ykj
               yjkl = zlj*xkj - xlj*zkj
               zjkl = xlj*ykj - ylj*xkj

               rijk2  = xijk*xijk + yijk*yijk + zijk*zijk
               rjkl2  = xjkl*xjkl + yjkl*yjkl + zjkl*zjkl

               rijkl2 = sqrt(rijk2*rjkl2)

               if ( abs(rijk2)  .lt. tiny_value ) cycle
               if ( abs(rjkl2)  .lt. tiny_value ) cycle
               if ( abs(rijkl2) .lt. tiny_value ) cycle

               rijk2inv  = 1.d0 / rijk2
               rjkl2inv  = 1.d0 / rjkl2
               rijkl2inv = 1.d0 / rijkl2

               cos_phi = ( xijk*xjkl + yijk*yjkl + zijk*zjkl ) &
     &                 * rijkl2inv

               cos_phi = max( cos_phi, -1.d0 )
               cos_phi = min( cos_phi,  1.d0 )

               phi(i1or2) = acos(cos_phi)

               sign_phi = ( yijk*zjkl - zijk*yjkl ) * xkj &
     &                  + ( zijk*xjkl - xijk*zjkl ) * ykj &
     &                  + ( xijk*yjkl - yijk*xjkl ) * zkj

               sign_phi = sign( 1.d0, sign_phi )

               phi(i1or2) = phi(i1or2) * sign_phi

            end do

            p1c = 0.5d0 * ( xgrid_cmap(1) + xgrid_cmap(mgrid_cmap) )
            p2c = 0.5d0 * ( ygrid_cmap(1) + ygrid_cmap(mgrid_cmap) )

            phi1 = phi(1) * 180.d0 / pi
            phi2 = phi(2) * 180.d0 / pi

            phi1 = phi1 - nint( (phi1-p1c) / 360.d0 ) * 360.d0
            phi2 = phi2 - nint( (phi2-p2c) / 360.d0 ) * 360.d0

            call splin2_cmap( xgrid_cmap(:), ygrid_cmap(:), &
     &                        vgrid_cmap(:,:,n), v2grid_cmap(:,:,n), &
     &                        mgrid_cmap, mgrid_cmap, &
     &                        phi1, phi2, v, dvdphi1, dvdphi2 )

            dvdphi(1) = dvdphi1 * 180.d0 / pi
            dvdphi(2) = dvdphi2 * 180.d0 / pi

            pot(m)  = pot(m) + v

            do i1or2 = 1, 2

               i = i_cmap(i1or2,n)
               j = j_cmap(i1or2,n)
               k = k_cmap(i1or2,n)
               l = l_cmap(i1or2,n)

               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

               call pbc_atom_MPI ( xij, yij, zij )

               xkj = x(k,m) - x(j,m)
               ykj = y(k,m) - y(j,m)
               zkj = z(k,m) - z(j,m)

               call pbc_atom_MPI ( xkj, ykj, zkj )

               xlj = x(l,m) - x(j,m)
               ylj = y(l,m) - y(j,m)
               zlj = z(l,m) - z(j,m)

               call pbc_atom_MPI ( xlj, ylj, zlj )

               xijk = yij*zkj - zij*ykj
               yijk = zij*xkj - xij*zkj
               zijk = xij*ykj - yij*xkj

               xjkl = ylj*zkj - zlj*ykj
               yjkl = zlj*xkj - xlj*zkj
               zjkl = xlj*ykj - ylj*xkj

               rijk2  = xijk*xijk + yijk*yijk + zijk*zijk
               rjkl2  = xjkl*xjkl + yjkl*yjkl + zjkl*zjkl

               rijkl2 = sqrt(rijk2*rjkl2)

               if ( abs(rijk2)  .lt. tiny_value ) cycle
               if ( abs(rjkl2)  .lt. tiny_value ) cycle
               if ( abs(rijkl2) .lt. tiny_value ) cycle

               rijk2inv  = 1.d0 / rijk2
               rjkl2inv  = 1.d0 / rjkl2
               rijkl2inv = 1.d0 / rijkl2

               cos_phi = ( xijk*xjkl + yijk*yjkl + zijk*zjkl ) &
     &                 * rijkl2inv

               cos_phi = max( cos_phi, -1.d0 )
               cos_phi = min( cos_phi,  1.d0 )

               phi(i1or2) = acos(cos_phi)

               sign_phi = ( yijk*zjkl - zijk*yjkl ) * xkj &
     &                  + ( zijk*xjkl - xijk*zjkl ) * ykj &
     &                  + ( xijk*yjkl - yijk*xjkl ) * zkj

               sign_phi = sign( 1.d0, sign_phi )

               phi(i1or2) = phi(i1or2) * sign_phi

               if ( ( abs(phi(i1or2))    .gt. tiny_value ) .and. &
     &              ( abs(phi(i1or2)-pi) .gt. tiny_value ) .and. &
     &              ( abs(phi(i1or2)+pi) .gt. tiny_value ) ) then

                  sin_phi = sin( phi(i1or2) )

                  factor = dvdphi(i1or2) / sin_phi

                  fxi = factor * ( + ( ykj*zjkl - zkj*yjkl ) * rijkl2inv &
     &                    - ( ykj*zijk - zkj*yijk ) * cos_phi*rijk2inv )
                  fyi = factor * ( + ( zkj*xjkl - xkj*zjkl ) * rijkl2inv &
     &                    - ( zkj*xijk - xkj*zijk ) * cos_phi*rijk2inv )
                  fzi = factor * ( + ( xkj*yjkl - ykj*xjkl ) * rijkl2inv &
     &                    - ( xkj*yijk - ykj*xijk ) * cos_phi*rijk2inv )

                  fxl = factor * ( + ( ykj*zijk - zkj*yijk ) * rijkl2inv &
     &                    - ( ykj*zjkl - zkj*yjkl ) * cos_phi*rjkl2inv )
                  fyl = factor * ( + ( zkj*xijk - xkj*zijk ) * rijkl2inv &
     &                    - ( zkj*xjkl - xkj*zjkl ) * cos_phi*rjkl2inv )
                  fzl = factor * ( + ( xkj*yijk - ykj*xijk ) * rijkl2inv &
     &                    - ( xkj*yjkl - ykj*xjkl ) * cos_phi*rjkl2inv )

                  fxk = factor * ( - ( yij*zjkl - zij*yjkl ) * rijkl2inv &
     &                    - ( ylj*zijk - zlj*yijk ) * rijkl2inv &
     &                    + ( yij*zijk - zij*yijk ) * cos_phi*rijk2inv &
     &                    + ( ylj*zjkl - zlj*yjkl ) * cos_phi*rjkl2inv )
                  fyk = factor * ( - ( zij*xjkl - xij*zjkl ) * rijkl2inv &
     &                    - ( zlj*xijk - xlj*zijk ) * rijkl2inv &
     &                    + ( zij*xijk - xij*zijk ) * cos_phi*rijk2inv &
     &                    + ( zlj*xjkl - xlj*zjkl ) * cos_phi*rjkl2inv )
                  fzk = factor * ( - ( xij*yjkl - yij*xjkl ) * rijkl2inv &
     &                    - ( xlj*yijk - ylj*xijk ) * rijkl2inv &
     &                    + ( xij*yijk - yij*xijk ) * cos_phi*rijk2inv &
     &                    + ( xlj*yjkl - ylj*xjkl ) * cos_phi*rjkl2inv )

                  fxj = - ( fxi + fxk + fxl )
                  fyj = - ( fyi + fyk + fyl )
                  fzj = - ( fzi + fzk + fzl )

               else

                  xki = - xij + xkj
                  yki = - yij + ykj
                  zki = - zij + zkj

                  xkl = - xlj + xkj
                  ykl = - ylj + ykj
                  zkl = - zlj + zkj

                  ax = yijk*zjkl - zijk*yjkl
                  ay = zijk*xjkl - xijk*zjkl
                  az = xijk*yjkl - yijk*xjkl

                  a2 = ax*ax + ay*ay + az*az

                  a1 = sqrt( a2 )

                  sin_phi = a1 / rijkl2

                  sin_phi = max( sin_phi, -1.d0 )
                  sin_phi = min( sin_phi,  1.d0 )

                  phi = sign_phi * asin( sin_phi )

                  if ( cos_phi .lt. 0.d0 ) phi = pi - phi

                  daxdxi = - zjkl * zkj - ykj * yjkl
                  daxdyi = + yjkl * xkj
                  daxdzi = + zjkl * xkj

                  daydxi = + xjkl * ykj
                  daydyi = - xjkl * xkj - zkj * zjkl
                  daydzi = + zjkl * ykj

                  dazdxi = + xjkl * zkj
                  dazdyi = + yjkl * zkj
                  dazdzi = - yjkl * ykj - xkj * xjkl

                  daxdxj = - yijk * ykl + zjkl * zki &
     &                     + yjkl * yki - zijk * zkl
                  daxdyj = + yijk * xkl - yjkl * xki
                  daxdzj = + zijk * xkl - zjkl * xki

                  daydxj = + xijk * ykl - xjkl * yki
                  daydyj = - zijk * zkl + xjkl * xki &
     &                     + zjkl * zki - xijk * xkl
                  daydzj = + zijk * ykl - zjkl * yki

                  dazdxj = + xijk * zkl - xjkl * zki
                  dazdyj = + yijk * zkl - yjkl * zki
                  dazdzj = - xijk * xkl + yjkl * yki &
     &                     + xjkl * xki - yijk * ykl

!                  daxdxk = - yjkl * yij + zijk * zlj
!                           + yijk * ylj - zjkl * zij
!                  daxdyk = + yjkl * xij - yijk * xlj
!                  daxdzk = + zjkl * xij - zijk * xlj
!
!                  daydxk = + xjkl * yij - xijk * ylj
!                  daydyk = - zjkl * zij + xijk * xlj
!                           + zijk * zlj - xjkl * xij
!                  daydzk = + zjkl * yij - zijk * ylj
!
!                  dazdxk = + xjkl * zij - xijk * zlj
!                  dazdyk = + yjkl * zij - yijk * zlj
!                  dazdzk = - xjkl * xij + yijk * ylj
!                           + xijk * xlj - yjkl * yij

                  daxdxl = + zijk * zkj + ykj * yijk
                  daxdyl = - yijk * xkj
                  daxdzl = - zijk * xkj

                  daydxl = - xijk * ykj
                  daydyl = + xijk * xkj + zkj * zijk
                  daydzl = - zijk * ykj

                  dazdxl = - xijk * zkj
                  dazdyl = - yijk * zkj
                  dazdzl = + yijk * ykj + xkj * xijk

                  dadxi = ax/a1*daxdxi + ay/a1*daydxi + az/a1*dazdxi
                  dadyi = ax/a1*daxdyi + ay/a1*daydyi + az/a1*dazdyi
                  dadzi = ax/a1*daxdzi + ay/a1*daydzi + az/a1*dazdzi

                  dadxj = ax/a1*daxdxj + ay/a1*daydxj + az/a1*dazdxj
                  dadyj = ax/a1*daxdyj + ay/a1*daydyj + az/a1*dazdyj
                  dadzj = ax/a1*daxdzj + ay/a1*daydzj + az/a1*dazdzj

!                  dadxk = ax/a1*daxdxk + ay/a1*daydxk + az/a1*dazdxk
!                  dadyk = ax/a1*daxdyk + ay/a1*daydyk + az/a1*dazdyk
!                  dadzk = ax/a1*daxdzk + ay/a1*daydzk + az/a1*dazdzk

                  dadxl = ax/a1*daxdxl + ay/a1*daydxl + az/a1*dazdxl
                  dadyl = ax/a1*daxdyl + ay/a1*daydyl + az/a1*dazdyl
                  dadzl = ax/a1*daxdzl + ay/a1*daydzl + az/a1*dazdzl

                  f1 = - sign_phi / cos_phi * dvdphi(i1or2)

                  fxi = + f1 * ( dadxi / rijkl2 &
     &                + sin_phi * ( + yijk*zkj - zijk*ykj ) * rijk2inv )

                  fyi = + f1 * ( dadyi / rijkl2 &
     &                + sin_phi * ( + zijk*xkj - xijk*zkj ) * rijk2inv )

                  fzi = + f1 * ( dadzi / rijkl2 &
     &                + sin_phi * ( + xijk*ykj - yijk*xkj ) * rijk2inv )

                  fxj = + f1 * ( dadxj / rijkl2 &
     &                + sin_phi * ( - yijk*zki + zijk*yki ) * rijk2inv &
     &                - sin_phi * ( + yjkl*zkl - zjkl*ykl ) * rjkl2inv )

                  fyj = + f1 * ( dadyj / rijkl2 &
     &                + sin_phi * ( - zijk*xki + xijk*zki ) * rijk2inv &
     &                - sin_phi * ( + zjkl*xkl - xjkl*zkl ) * rjkl2inv )

                  fzj = + f1 * ( dadzj / rijkl2 &
     &                + sin_phi * ( - xijk*yki + yijk*xki ) * rijk2inv &
     &                - sin_phi * ( + xjkl*ykl - yjkl*xkl ) * rjkl2inv )

!                 fxk = - f1 * ( dadxk / rijkl2
!     &                - sin_phi * ( - yjkl*zlj + zjkl*ylj ) * rjkl2inv
!     &                + sin_phi * ( + yijk*zij - zijk*yij ) * rijk2inv )
!
!                 fyk = - f1 * ( dadyk / rijkl2
!     &                - sin_phi * ( - zjkl*xlj + xjkl*zlj ) * rjkl2inv
!     &                + sin_phi * ( + zijk*xij - xijk*zij ) * rijk2inv )
!
!                 fzk = - f1 * ( dadzk / rijkl2
!     &                - sin_phi * ( - xjkl*ylj + yjkl*xlj ) * rjkl2inv
!     &                + sin_phi * ( + xijk*yij - yijk*xij ) * rijk2inv )

                  fxl = + f1 * ( dadxl / rijkl2 &
     &                + sin_phi * ( + yjkl*zkj - zjkl*ykj ) * rjkl2inv )

                  fyl = + f1 * ( dadyl / rijkl2 &
     &                + sin_phi * ( + zjkl*xkj - xjkl*zkj ) * rjkl2inv )

                  fzl = + f1 * ( dadzl / rijkl2 &
     &                + sin_phi * ( + xjkl*ykj - yjkl*xkj ) * rjkl2inv )

                  fxk = - fxi - fxj - fxl
                  fyk = - fyi - fyj - fyl
                  fzk = - fzi - fzj - fzl

               end if

               fx(i,m) = fx(i,m) + fxi
               fx(j,m) = fx(j,m) + fxj
               fx(k,m) = fx(k,m) + fxk
               fx(l,m) = fx(l,m) + fxl

               fy(i,m) = fy(i,m) + fyi
               fy(j,m) = fy(j,m) + fyj
               fy(k,m) = fy(k,m) + fyk
               fy(l,m) = fy(l,m) + fyl

               fz(i,m) = fz(i,m) + fzi
               fz(j,m) = fz(j,m) + fzj
               fz(k,m) = fz(k,m) + fzk
               fz(l,m) = fz(l,m) + fzl

               vir(1,1) = vir(1,1) + fxi*xij + fxk*xkj + fxl*xlj
               vir(1,2) = vir(1,2) + fxi*yij + fxk*ykj + fxl*ylj
               vir(1,3) = vir(1,3) + fxi*zij + fxk*zkj + fxl*zlj
               vir(2,1) = vir(2,1) + fyi*xij + fyk*xkj + fyl*xlj
               vir(2,2) = vir(2,2) + fyi*yij + fyk*ykj + fyl*ylj
               vir(2,3) = vir(2,3) + fyi*zij + fyk*zkj + fyl*zlj
               vir(3,1) = vir(3,1) + fzi*xij + fzk*xkj + fzl*xlj
               vir(3,2) = vir(3,2) + fzi*yij + fzk*ykj + fzl*ylj
               vir(3,3) = vir(3,3) + fzi*zij + fzk*zkj + fzl*zlj

            end do

         end do

      end do

      return
      end





!***********************************************************************
      subroutine splie2_cmap( x2a, ya, y2a, m, n )
!***********************************************************************

      implicit none

      integer :: j, k, m, n

      real(8) :: x2a(n), ya(m,n), y2a(m,n), ytmp(n), y2tmp(n)

      do j = 1, m

         do k = 1, n
            ytmp(k) = ya(j,k)
         end do

         call spline_cmap( x2a, ytmp, n, y2tmp )

         do k = 1, n
            y2a(j,k) = y2tmp(k)
         end do

      end do

      return
      end





!***********************************************************************
      subroutine splin2_cmap &
     &   ( x1a, x2a, ya, y2a, m, n, x1, x2, y, dydx1, dydx2 )
!***********************************************************************

      implicit none

      integer :: j, k, m, n

      real(8) :: x1a(m), x2a(n), ya(m,n), y2a(m,n)
      real(8) :: ytmp(m), y2tmp(m), yytmp(m), z2tmp(m), zztmp(m)
      real(8) :: x1, x2, y, dydx1, dydx2, dy2dx1dx2

      do j = 1, m

         do k = 1, n
            ytmp(k) = ya(j,k)
            y2tmp(k) = y2a(j,k)
         end do

         call splint_cmap( x2a, ytmp, y2tmp, n, x2, yytmp(j), zztmp(j) )

      end do

      call spline_cmap( x1a, yytmp, m, y2tmp )
      call splint_cmap( x1a, yytmp, y2tmp, m, x1, y, dydx1 )

      call spline_cmap( x1a, zztmp, m, z2tmp )
      call splint_cmap( x1a, zztmp, z2tmp, m, x1, dydx2, dy2dx1dx2 )

      return
      end





!***********************************************************************
      subroutine spline_cmap( x, y, n, y2 )
!***********************************************************************

      implicit none

      integer :: i, k, n

      real(8) :: x(n), y(n), y2(n), u(n), sig, p

      y2(1) = 0.d0
      u(1)  = 0.d0

      do i = 2, n-1

         sig = ( x(i) - x(i-1) ) / ( x(i+1) - x(i-1) )

         p = sig * y2(i-1) + 2.d0

         y2(i) = ( sig - 1.d0 ) / p

         u(i) = ( 6.d0 * ( ( y(i+1) - y(i) ) / ( x(i+1) - x(i) ) &
     &                   - ( y(i) - y(i-1) ) / ( x(i) - x(i-1) ) ) &
     &                   / ( x(i+1) - x(i-1) ) &
     &          - sig * u(i-1) ) / p

      end do

      y2(n) = 0.d0

      do k = n-1, 1, -1
         y2(k) = y2(k) * y2(k+1) + u(k)
      end do

      return
      end





!***********************************************************************
      subroutine splint_cmap( xa, ya, y2a, n, x, y, dydx )
!***********************************************************************

      implicit none

      integer :: khi, klo, n

      real(8) :: xa(n), ya(n), y2a(n), x, y, h, a, b, dydx, hinv

      klo = int( ( x - xa(1) ) / ( xa(n) - xa(1) ) * ( n - 1 ) ) + 1
      khi = klo + 1

      h = xa(khi) - xa(klo)

      hinv = 1.d0 / h

      a = ( xa(khi) - x ) * hinv
      b = ( x - xa(klo) ) * hinv

      y = a * ya(klo) + b * ya(khi) &
     &  + ( (a*a*a-a) * y2a(klo) + (b*b*b-b) * y2a(khi) ) * (h*h) / 6.d0

      dydx = ( - ya(klo) + ya(khi) ) * hinv &
     &     + ( ( 1.d0 - 3.d0*a*a ) * y2a(klo) &
     &       + ( 3.d0*b*b - 1.d0 ) * y2a(khi) ) * h / 6.d0

      return
      end



#endif
