!***********************************************************************
      module metalwater_variables
!***********************************************************************
!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

!     //   number of metal atoms
      integer :: nmetal

!     //   number of water molecules
      integer :: nwater

!     //   number of oxygen atoms
      integer :: noxygen

!     //   number of hydrogen atoms
      integer :: nhydrogen

!     //   number of metal kinds
      integer :: nkind_metal

!     //   metal atoms
      logical, dimension(:), allocatable :: is_metal

!     //   oxygen atoms
      logical, dimension(:), allocatable :: is_oxygen

!     //   hydrogen atoms
      logical, dimension(:), allocatable :: is_hydrogen

!     //   1st hydrogen atom attached to oxygen
      integer, dimension(:), allocatable :: j_bond

!     //   2nd hydrogen atom attached to oxygen
      integer, dimension(:), allocatable :: k_bond

!     //   coordination number
      real(8), dimension(:), allocatable :: cn

!     //   generalized coordination number (GCN)
      real(8), dimension(:), allocatable :: gcn

!     //   gradient of GCN
      real(8), dimension(:,:), allocatable :: gcnx
      real(8), dimension(:,:), allocatable :: gcny
      real(8), dimension(:,:), allocatable :: gcnz

!     //   surface normal vector (SNV)
      real(8), dimension(:), allocatable :: sx
      real(8), dimension(:), allocatable :: sy
      real(8), dimension(:), allocatable :: sz

!     //   gradient of SNV
      real(8), dimension(:,:), allocatable :: sxx
      real(8), dimension(:,:), allocatable :: sxy
      real(8), dimension(:,:), allocatable :: sxz
      real(8), dimension(:,:), allocatable :: syx
      real(8), dimension(:,:), allocatable :: syy
      real(8), dimension(:,:), allocatable :: syz
      real(8), dimension(:,:), allocatable :: szx
      real(8), dimension(:,:), allocatable :: szy
      real(8), dimension(:,:), allocatable :: szz

!     //   metal-metal coordination number
      real(8) :: rin_cn
      real(8) :: rout_cn

!     //   metal-oxygen distance
      real(8) :: rin_mo
      real(8) :: rout_mo

!     //   metal-hydrogen distance
      real(8) :: rin_mh
      real(8) :: rout_mh

!     //   metal-metal surface normal vector
      real(8) :: rin_sf
      real(8) :: rout_sf

!     //   GCN parameters
      real(8), dimension(:,:), allocatable :: param_r_O
      real(8), dimension(:,:), allocatable :: param_bn
      real(8), dimension(:,:), allocatable :: param_bp
      real(8), dimension(:,:), allocatable :: param_b_O
      real(8), dimension(:,:), allocatable :: param_b_H
      real(8), dimension(:,:), allocatable :: param_a1
      real(8), dimension(:,:), allocatable :: param_a2
      real(8), dimension(:,:), allocatable :: param_a3
      real(8), dimension(:,:), allocatable :: param_a4
      real(8), dimension(:,:), allocatable :: param_eps
      real(8), dimension(:,:), allocatable :: param_a_O
      real(8), dimension(:,:), allocatable :: param_a_H
      real(8), dimension(:,:), allocatable :: param_c_O
      real(8), dimension(:),   allocatable :: param_cnmax

!     //   metal-metal interaction
      real(8) :: potential_gal
      integer :: iprint_gal = 0

!***********************************************************************
      end module metalwater_variables
!***********************************************************************





!***********************************************************************
      subroutine force_metalwater
!***********************************************************************
!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only: iounit
      use metalwater_variables, only: potential_gal, iprint_gal

!-----------------------------------------------------------------------
!     //   force calculations
!-----------------------------------------------------------------------

!     //   TIP4P/F for water
      call force_metalwater_tip4p

!     //   EAM for metals
      call force_metalwater_eam

!     //   GAL for metal-water interaction
      call force_metalwater_gal

!-----------------------------------------------------------------------
!     //   print net GAL potential
!-----------------------------------------------------------------------

      if ( iprint_gal .eq. 1 ) then
         open ( iounit, file = 'gal.out', access='append' )
         write( iounit, '(f16.8)' ) potential_gal
         close( iounit )
      end if

      return
      end





!***********************************************************************
      subroutine force_metalwater_gal_setup
!***********************************************************************
!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   natom, au_length, au2kcal, species, box, volume, iboundary

      use metalwater_variables, only : &
     &   is_metal, gcn, param_a_H, param_b_H, param_a_O, param_b_O, &
     &   cn, param_c_O, gcnx, gcny, gcnz, sx, sy, sz, sxx, sxy, sxz, &
     &   syx, syy, syz, szx, szy, szz, param_eps, param_bp, param_bn, &
     &   param_a1, param_a2, param_a3, param_a4, param_r_O, rin_sf, &
     &   rout_sf, param_cnmax, rin_cn, rout_cn, rin_mo, rout_mo, &
     &   rin_mh, rout_mh

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   real numbers
      real(8) :: bohr2ang = au_length * 1.d+10

!     //   real numbers
      real(8) :: ax, ay, az, bx, by, bz, cx, cy, cz, absa, absb, &
     &           absc, rcut, eps0, eps1, eps2

!     //   integers
      integer :: i, ierr

!-----------------------------------------------------------------------
!     //   cutoff parameters
!-----------------------------------------------------------------------

!     //   coordination number
      rin_cn  =  2.7890 / bohr2ang
      rout_cn =  3.2200 / bohr2ang

!     //   metal-oxygen distance
      rin_mo  =  5.5000 / bohr2ang
      rout_mo =  6.0000 / bohr2ang

!     //   metal-hydrogen distance
      rin_mh  =  5.5000 / bohr2ang
      rout_mh =  6.0000 / bohr2ang

!     //   surface normal vector
      rin_sf  =  5.5000 / bohr2ang
      rout_sf =  6.0000 / bohr2ang

!-----------------------------------------------------------------------
!     //   check cutoff parameters
!-----------------------------------------------------------------------

!     //   periodic boundary
      if ( iboundary .ne. 0 ) then

!        /*   vector product of lattice vectors b, c   */
         ax = box(2,2)*box(3,3) - box(2,3)*box(3,2)
         ay = box(3,2)*box(1,3) - box(3,3)*box(1,2)
         az = box(1,2)*box(2,3) - box(1,3)*box(2,2)

!        /*   vector product of lattice vectors c, a   */
         bx = box(2,3)*box(3,1) - box(2,1)*box(3,3)
         by = box(3,3)*box(1,1) - box(3,1)*box(1,3)
         bz = box(1,3)*box(2,1) - box(1,1)*box(2,3)

!        /*   vector product of lattice vectors a, b   */
         cx = box(2,1)*box(3,2) - box(2,2)*box(3,1)
         cy = box(3,1)*box(1,2) - box(3,2)*box(1,1)
         cz = box(1,1)*box(2,2) - box(1,2)*box(2,1)

!        /*   distance between parallel planes   */
         absa = volume / sqrt( ax*ax + ay*ay + az*az )
         absb = volume / sqrt( bx*bx + by*by + bz*bz )
         absc = volume / sqrt( cx*cx + cy*cy + cz*cz )

!        /*   error flag   */
         ierr = 0

!        //   maximum cutoff distance
         rcut = max( rout_cn, rout_mo, rout_mh, rout_sf )

!        /*   error flag   */
         if ( int(2.d0*rcut/absa) .gt. 1 ) ierr = 1
         if ( int(2.d0*rcut/absb) .gt. 1 ) ierr = 1
         if ( int(2.d0*rcut/absc) .gt. 1 ) ierr = 1

!        /*   error handling   */
         call error_handling &
     &      ( ierr, 'subroutine force_metalwater_gal_setup', 37 )

!     //   periodic boundary
      end if

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

!     //   coordination number
      if ( .not. allocated(cn)   ) allocate( cn(natom) )

!     //   generalized coordination number (GCN)
      if ( .not. allocated(gcn)  ) allocate( gcn(natom) )

!     //   gradient of GCN
      if ( .not. allocated(gcnx) ) allocate( gcnx(natom,natom) )
      if ( .not. allocated(gcny) ) allocate( gcny(natom,natom) )
      if ( .not. allocated(gcnz) ) allocate( gcnz(natom,natom) )

!     //   surface normal vector
      if ( .not. allocated(sx) ) allocate( sx(natom) )
      if ( .not. allocated(sy) ) allocate( sy(natom) )
      if ( .not. allocated(sz) ) allocate( sz(natom) )

!     //   gradient of SNV
      if ( .not. allocated(sxx) ) allocate( sxx(natom,natom) )
      if ( .not. allocated(sxy) ) allocate( sxy(natom,natom) )
      if ( .not. allocated(sxz) ) allocate( sxz(natom,natom) )
      if ( .not. allocated(syx) ) allocate( syx(natom,natom) )
      if ( .not. allocated(syy) ) allocate( syy(natom,natom) )
      if ( .not. allocated(syz) ) allocate( syz(natom,natom) )
      if ( .not. allocated(szx) ) allocate( szx(natom,natom) )
      if ( .not. allocated(szy) ) allocate( szy(natom,natom) )
      if ( .not. allocated(szz) ) allocate( szz(natom,natom) )

!     //   GAL parameters
      if ( .not. allocated(param_r_O) ) allocate( param_r_O(0:0,natom) )
      if ( .not. allocated(param_bn ) ) allocate( param_bn (0:1,natom) )
      if ( .not. allocated(param_bp ) ) allocate( param_bp (0:1,natom) )
      if ( .not. allocated(param_b_O) ) allocate( param_b_O(0:1,natom) )
      if ( .not. allocated(param_b_H) ) allocate( param_b_H(0:1,natom) )
      if ( .not. allocated(param_a1 ) ) allocate( param_a1 (0:2,natom) )
      if ( .not. allocated(param_a2 ) ) allocate( param_a2 (0:2,natom) )
      if ( .not. allocated(param_a3 ) ) allocate( param_a3 (0:2,natom) )
      if ( .not. allocated(param_a4 ) ) allocate( param_a4 (0:2,natom) )
      if ( .not. allocated(param_eps) ) allocate( param_eps(0:2,natom) )
      if ( .not. allocated(param_a_O) ) allocate( param_a_O(0:1,natom) )
      if ( .not. allocated(param_a_H) ) allocate( param_a_H(0:1,natom) )
      if ( .not. allocated(param_c_O) ) allocate( param_c_O(0:0,natom) )
      if ( .not. allocated(param_cnmax))  allocate( param_cnmax(natom) )

!-----------------------------------------------------------------------
!     //   parameters
!-----------------------------------------------------------------------

!     //   error status
      ierr = 0

!     //   initialize
      param_r_O(0:0,:) = 0.d0
      param_bn (0:1,:) = 0.d0
      param_bp (0:1,:) = 0.d0
      param_b_O(0:1,:) = 0.d0
      param_b_H(0:1,:) = 0.d0
      param_a1 (0:2,:) = 0.d0
      param_a2 (0:2,:) = 0.d0
      param_a3 (0:2,:) = 0.d0
      param_a4 (0:2,:) = 0.d0
      param_eps(0:2,:) = 0.d0
      param_a_O(0:1,:) = 0.d0
      param_a_H(0:1,:) = 0.d0
      param_c_O(0:0,:) = 0.d0
      param_cnmax(:)   = 0.d0

!     //   loop of atoms
      do i = 1, natom

!        //   for metals only
         if ( .not. is_metal(i) ) cycle

!        //   atomic species
         if ( species(i)(1:2) .eq. 'Pt' ) then

!            //   Paul Clabaut, Thesis (2021)
!            param_r_O(0,i) =      2.01 / bohr2ang
!            param_bn(0,i)  =      0.57 * bohr2ang * bohr2ang
!            param_bn(1,i)  =      0.01 * bohr2ang * bohr2ang
!            param_bp(0,i)  =      0.22 * bohr2ang * bohr2ang
!            param_bp(1,i)  =      0.00 * bohr2ang * bohr2ang
!            param_b_O(0,i) =      3.26 * bohr2ang
!            param_b_O(1,i) =     -0.01 * bohr2ang
!            param_b_H(0,i) =      3.33 * bohr2ang
!            param_b_H(1,i) =      0.09 * bohr2ang
!            param_a1(0,i)  =     49.82 / au2kcal
!            param_a1(1,i)  =    -17.70 / au2kcal
!            param_a1(2,i)  =      1.89 / au2kcal
!            param_a2(0,i)  =     55.91 / au2kcal
!            param_a2(1,i)  =    -18.02 / au2kcal
!            param_a2(2,i)  =      1.50 / au2kcal
!            param_a3(0,i)  =    -92.20 / au2kcal
!            param_a3(1,i)  =     12.11 / au2kcal
!            param_a3(2,i)  =      0.06 / au2kcal
!            param_a4(0,i)  =      6.40 / au2kcal
!            param_a4(1,i)  =      0.00 / au2kcal
!            param_a4(2,i)  =     -0.10 / au2kcal
!            param_eps(0,i) =     55.60 / au2kcal
!            param_eps(1,i) =     -3.63 / au2kcal
!            param_eps(2,i) =      0.03 / au2kcal
!            param_a_O(0,i) =    301.57 / au2kcal
!            param_a_O(1,i) =   1742.28 / au2kcal
!            param_a_H(0,i) =   9856.14 / au2kcal
!            param_a_H(1,i) =   1166.11 / au2kcal
!            param_c_O(0,i) =    917.00 / bohr2ang**6 / au2kcal
!            param_cnmax(i) =     12.00

!           //    Clabaut, et al. J. Chem. Phys. 157, 194705 (2022)
            param_r_O(0,i) =     2.177 / bohr2ang
            param_bn(0,i)  =     0.166 * bohr2ang * bohr2ang
            param_bn(1,i)  =    0.0905 * bohr2ang * bohr2ang
            param_bp(0,i)  =     0.362 * bohr2ang * bohr2ang
            param_bp(1,i)  =   -0.0004 * bohr2ang * bohr2ang
            param_b_O(0,i) =     3.171 * bohr2ang
            param_b_O(1,i) =   -0.0069 * bohr2ang
            param_b_H(0,i) =     2.949 * bohr2ang
            param_b_H(1,i) =    0.0585 * bohr2ang
            param_a1(0,i)  =    -65.49 / au2kcal
            param_a1(1,i)  =     27.73 / au2kcal
            param_a1(2,i)  =     -2.12 / au2kcal
            param_a2(0,i)  =    -45.18 / au2kcal
            param_a2(1,i)  =     20.57 / au2kcal
            param_a2(2,i)  =     -1.74 / au2kcal
            param_a3(0,i)  =    -88.65 / au2kcal
            param_a3(1,i)  =     28.11 / au2kcal
            param_a3(2,i)  =     -2.05 / au2kcal
            param_a4(0,i)  =     29.54 / au2kcal
            param_a4(1,i)  =      0.00 / au2kcal
            param_a4(2,i)  =     -0.56 / au2kcal
            param_eps(0,i) =     0.632 / au2kcal
            param_eps(1,i) =    -18.21 / au2kcal
            param_eps(2,i) =    200.00 / au2kcal
            param_a_O(0,i) =  16757.08 / au2kcal
            param_a_O(1,i) =    723.45 / au2kcal
            param_a_H(0,i) =   2211.10 / au2kcal
            param_a_H(1,i) =    446.63 / au2kcal
            param_c_O(0,i) =    1589.0 / bohr2ang**6 / au2kcal
            param_cnmax(i) =     12.00

!        //   atomic species
         else if ( species(i)(1:2) .eq. 'Ag' ) then

!            //   Paul Clabaut, Thesis (2021)
!            param_r_O(0,i) =      2.02 / bohr2ang
!            param_bn(0,i)  =      0.38 * bohr2ang * bohr2ang
!            param_bn(1,i)  =      0.00 * bohr2ang * bohr2ang
!            param_bp(0,i)  =      0.13 * bohr2ang * bohr2ang
!            param_bp(1,i)  =     -0.00 * bohr2ang * bohr2ang
!            param_b_O(0,i) =      3.45 * bohr2ang
!            param_b_O(1,i) =     -0.10 * bohr2ang
!            param_b_H(0,i) =      3.05 * bohr2ang
!            param_b_H(1,i) =      0.03 * bohr2ang
!            param_a1(0,i)  =   -136.67 / au2kcal
!            param_a1(1,i)  =     45.33 / au2kcal
!            param_a1(2,i)  =     -3.18 / au2kcal
!            param_a2(0,i)  =      8.17 / au2kcal
!            param_a2(1,i)  =      2.16 / au2kcal
!            param_a2(2,i)  =     -0.38 / au2kcal
!            param_a3(0,i)  =    -20.36 / au2kcal
!            param_a3(1,i)  =      1.68 / au2kcal
!            param_a3(2,i)  =      0.19 / au2kcal
!            param_a4(0,i)  =     -2.52 / au2kcal
!            param_a4(1,i)  =      0.00 / au2kcal
!            param_a4(2,i)  =      0.06 / au2kcal
!            param_eps(0,i) =      8.53 / au2kcal
!            param_eps(1,i) =      1.82 / au2kcal
!            param_eps(2,i) =     -0.20 / au2kcal
!            param_a_O(0,i) =   8278.92 / au2kcal
!            param_a_O(1,i) =   -587.78 / au2kcal
!            param_a_H(0,i) =   2517.03 / au2kcal
!            param_a_H(1,i) =    370.92 / au2kcal
!            param_c_O(0,i) =    930.00 / bohr2ang**6 / au2kcal
!            param_cnmax(i) =     12.00

!           //    Clabaut, et al. J. Chem. Phys. 157, 194705 (2022)
            param_r_O(0,i) =     2.378 / bohr2ang
            param_bn(0,i)  =     0.262 * bohr2ang * bohr2ang
            param_bn(1,i)  =    0.0582 * bohr2ang * bohr2ang
            param_bp(0,i)  =     0.349 * bohr2ang * bohr2ang
            param_bp(1,i)  =   -0.0247 * bohr2ang * bohr2ang
            param_b_O(0,i) =     3.178 * bohr2ang
            param_b_O(1,i) =   -0.0249 * bohr2ang
            param_b_H(0,i) =     3.314 * bohr2ang
            param_b_H(1,i) =    0.1065 * bohr2ang
            param_a1(0,i)  =    -55.47 / au2kcal
            param_a1(1,i)  =     21.78 / au2kcal
            param_a1(2,i)  =     -1.70 / au2kcal
            param_a2(0,i)  =     36.37 / au2kcal
            param_a2(1,i)  =     -8.86 / au2kcal
            param_a2(2,i)  =      0.53 / au2kcal
            param_a3(0,i)  =    -47.49 / au2kcal
            param_a3(1,i)  =     12.19 / au2kcal
            param_a3(2,i)  =     -0.71 / au2kcal
            param_a4(0,i)  =      7.82 / au2kcal
            param_a4(1,i)  =      0.00 / au2kcal
            param_a4(2,i)  =     -0.13 / au2kcal
            param_eps(0,i) =     0.415 / au2kcal
            param_eps(1,i) =    -11.53 / au2kcal
            param_eps(2,i) =     76.32 / au2kcal
            param_a_O(0,i) =  12833.18 / au2kcal
            param_a_O(1,i) =      0.00 / au2kcal
            param_a_H(0,i) =   5544.04 / au2kcal
            param_a_H(1,i) =   2378.06 / au2kcal
            param_c_O(0,i) =    1690.0 / bohr2ang**6 / au2kcal
            param_cnmax(i) =     12.00

!        //   atomic species
         else if ( species(i)(1:2) .eq. 'Au' ) then

!            //   Paul Clabaut, Thesis (2021)
!            param_r_O(0,i) =      2.04 / bohr2ang
!            param_bn(0,i)  =      0.43 * bohr2ang * bohr2ang
!            param_bn(1,i)  =      0.00 * bohr2ang * bohr2ang
!            param_bp(0,i)  =      0.09 * bohr2ang * bohr2ang
!            param_bp(1,i)  =      0.00 * bohr2ang * bohr2ang
!            param_b_O(0,i) =      3.49 * bohr2ang
!            param_b_O(1,i) =     -0.09 * bohr2ang
!            param_b_H(0,i) =      4.69 * bohr2ang
!            param_b_H(1,i) =     -0.18 * bohr2ang
!            param_a1(0,i)  =   -122.35 / au2kcal
!            param_a1(1,i)  =     43.43 / au2kcal
!            param_a1(2,i)  =     -3.20 / au2kcal
!            param_a2(0,i)  =     10.64 / au2kcal
!            param_a2(1,i)  =     -1.50 / au2kcal
!            param_a2(2,i)  =      0.07 / au2kcal
!            param_a3(0,i)  =    -13.00 / au2kcal
!            param_a3(1,i)  =     -0.26 / au2kcal
!            param_a3(2,i)  =      0.26 / au2kcal
!            param_a4(0,i)  =      0.58 / au2kcal
!            param_a4(1,i)  =      0.00 / au2kcal
!            param_a4(2,i)  =     -0.01 / au2kcal
!            param_eps(0,i) =      0.54 / au2kcal
!            param_eps(1,i) =      3.47 / au2kcal
!            param_eps(2,i) =     -0.28 / au2kcal
!            param_a_O(0,i) =   7553.69 / au2kcal
!            param_a_O(1,i) =   -175.69 / au2kcal
!            param_a_H(0,i) =  32286.66 / au2kcal
!            param_a_H(1,i) =  -3619.43 / au2kcal
!            param_c_O(0,i) =    918.00 / bohr2ang**6 / au2kcal
!            param_cnmax(i) =     12.00

!           //    Clabaut, et al. J. Chem. Phys. 157, 194705 (2022)
            param_r_O(0,i) =     2.224 / bohr2ang
            param_bn(0,i)  =     0.318 * bohr2ang * bohr2ang
            param_bn(1,i)  =    0.0392 * bohr2ang * bohr2ang
            param_bp(0,i)  =     0.178 * bohr2ang * bohr2ang
            param_bp(1,i)  =   -0.0039 * bohr2ang * bohr2ang
            param_b_O(0,i) =     3.462 * bohr2ang
            param_b_O(1,i) =   -0.0615 * bohr2ang
            param_b_H(0,i) =     2.954 * bohr2ang
            param_b_H(1,i) =    0.0163 * bohr2ang
            param_a1(0,i)  =    -65.92 / au2kcal
            param_a1(1,i)  =     33.50 / au2kcal
            param_a1(2,i)  =     -2.85 / au2kcal
            param_a2(0,i)  =     48.35 / au2kcal
            param_a2(1,i)  =    -18.39 / au2kcal
            param_a2(2,i)  =      1.57 / au2kcal
            param_a3(0,i)  =    -31.93 / au2kcal
            param_a3(1,i)  =      5.50 / au2kcal
            param_a3(2,i)  =     -0.09 / au2kcal
            param_a4(0,i)  =     -1.64 / au2kcal
            param_a4(1,i)  =      0.00 / au2kcal
            param_a4(2,i)  =      0.07 / au2kcal
            param_eps(0,i) =    -0.230 / au2kcal
            param_eps(1,i) =      1.63 / au2kcal
            param_eps(2,i) =     12.89 / au2kcal
            param_a_O(0,i) =  12928.33 / au2kcal
            param_a_O(1,i) =      0.00 / au2kcal
            param_a_H(0,i) =   2580.41 / au2kcal
            param_a_H(1,i) =     70.45 / au2kcal
            param_c_O(0,i) =    1632.0 / bohr2ang**6 / au2kcal
            param_cnmax(i) =     12.00

!        //   atomic species
         else if ( species(i)(1:2) .eq. 'Co' ) then

!            //   Paul Clabaut, Thesis (2021)
!            param_r_O(0,i) =      1.99 / bohr2ang
!            param_bn(0,i)  =      0.61 * bohr2ang * bohr2ang
!            param_bn(1,i)  =      0.00 * bohr2ang * bohr2ang
!            param_bp(0,i)  =      0.13 * bohr2ang * bohr2ang
!            param_bp(1,i)  =      0.00 * bohr2ang * bohr2ang
!            param_b_O(0,i) =      3.24 * bohr2ang
!            param_b_O(1,i) =      0.16 * bohr2ang
!            param_b_H(0,i) =      3.89 * bohr2ang
!            param_b_H(1,i) =     -0.04 * bohr2ang
!            param_a1(0,i)  =   -459.60 / au2kcal
!            param_a1(1,i)  =    134.32 / au2kcal
!            param_a1(2,i)  =     -8.66 / au2kcal
!            param_a2(0,i)  =     12.17 / au2kcal
!            param_a2(1,i)  =     -0.14 / au2kcal
!            param_a2(2,i)  =     -0.07 / au2kcal
!            param_a3(0,i)  =    -21.48 / au2kcal
!            param_a3(1,i)  =     -0.54 / au2kcal
!            param_a3(2,i)  =      0.37 / au2kcal
!            param_a4(0,i)  =      9.02 / au2kcal
!            param_a4(1,i)  =      0.00 / au2kcal
!            param_a4(2,i)  =     -0.14 / au2kcal
!            param_eps(0,i) =    -17.34 / au2kcal
!            param_eps(1,i) =     16.06 / au2kcal
!            param_eps(2,i) =     -1.65 / au2kcal
!            param_a_O(0,i) = -49271.02 / au2kcal
!            param_a_O(1,i) =  16707.66 / au2kcal
!            param_a_H(0,i) =  10171.40 / au2kcal
!            param_a_H(1,i) =      2.84 / au2kcal
!            param_c_O(0,i) =    907.00 / bohr2ang**6 / au2kcal
!            param_cnmax(i) =     12.00

!           //    Clabaut, et al. J. Chem. Phys. 157, 194705 (2022)
            param_r_O(0,i) =     2.098 / bohr2ang
            param_bn(0,i)  =     0.041 * bohr2ang * bohr2ang
            param_bn(1,i)  =    0.1271 * bohr2ang * bohr2ang
            param_bp(0,i)  =     0.342 * bohr2ang * bohr2ang
            param_bp(1,i)  =   -0.0184 * bohr2ang * bohr2ang
            param_b_O(0,i) =     3.436 * bohr2ang
            param_b_O(1,i) =    0.0226 * bohr2ang
            param_b_H(0,i) =     2.967 * bohr2ang
            param_b_H(1,i) =    0.0401 * bohr2ang
            param_a1(0,i)  =   -330.41 / au2kcal
            param_a1(1,i)  =    109.88 / au2kcal
            param_a1(2,i)  =     -7.93 / au2kcal
            param_a2(0,i)  =    101.68 / au2kcal
            param_a2(1,i)  =    -32.73 / au2kcal
            param_a2(2,i)  =      2.47 / au2kcal
            param_a3(0,i)  =    -49.87 / au2kcal
            param_a3(1,i)  =      9.83 / au2kcal
            param_a3(2,i)  =     -0.50 / au2kcal
            param_a4(0,i)  =      4.24 / au2kcal
            param_a4(1,i)  =      0.00 / au2kcal
            param_a4(2,i)  =     -0.08 / au2kcal
            param_eps(0,i) =    -0.031 / au2kcal
            param_eps(1,i) =    -10.40 / au2kcal
            param_eps(2,i) =     92.80 / au2kcal
            param_a_O(0,i) =   8198.02 / au2kcal
            param_a_O(1,i) =   2336.13 / au2kcal
            param_a_H(0,i) =   2937.29 / au2kcal
            param_a_H(1,i) =    112.83 / au2kcal
            param_c_O(0,i) =    1243.0 / bohr2ang**6 / au2kcal
            param_cnmax(i) =     12.00

!        //   atomic species
         else if ( species(i)(1:2) .eq. 'Cu' ) then

!            //   Paul Clabaut, Thesis (2021)
!            param_r_O(0,i) =     2.00 / bohr2ang
!            param_bn(0,i)  =     0.63 * bohr2ang * bohr2ang
!            param_bn(1,i)  =     0.00 * bohr2ang * bohr2ang
!            param_bp(0,i)  =     0.23 * bohr2ang * bohr2ang
!            param_bp(1,i)  =     0.00 * bohr2ang * bohr2ang
!            param_b_O(0,i) =     3.25 * bohr2ang
!            param_b_O(1,i) =    -0.08 * bohr2ang
!            param_b_H(0,i) =     4.02 * bohr2ang
!            param_b_H(1,i) =    -0.02 * bohr2ang
!            param_a1(0,i)  =  -234.36 / au2kcal
!            param_a1(1,i)  =    70.10 / au2kcal
!            param_a1(2,i)  =    -4.56 / au2kcal
!            param_a2(0,i)  =    46.69 / au2kcal
!            param_a2(1,i)  =    -9.71 / au2kcal
!            param_a2(2,i)  =     0.50 / au2kcal
!            param_a3(0,i)  =   -47.26 / au2kcal
!            param_a3(1,i)  =    10.90 / au2kcal
!            param_a3(2,i)  =    -0.58 / au2kcal
!            param_a4(0,i)  =   -10.75 / au2kcal
!            param_a4(1,i)  =     0.00 / au2kcal
!            param_a4(2,i)  =     0.20 / au2kcal
!            param_eps(0,i) =    37.83 / au2kcal
!            param_eps(1,i) =    -0.12 / au2kcal
!            param_eps(2,i) =     0.14 / au2kcal
!            param_a_O(0,i) =  3928.51 / au2kcal
!            param_a_O(1,i) =    25.42 / au2kcal
!            param_a_H(0,i) =  8578.84 / au2kcal
!            param_a_H(1,i) =   869.16 / au2kcal
!            param_c_O(0,i) =   899.00 / bohr2ang**6 / au2kcal
!            param_cnmax(i) =    12.00

!           //    Clabaut, et al. J. Chem. Phys. 157, 194705 (2022)
            param_r_O(0,i) =     2.120 / bohr2ang
            param_bn(0,i)  =     0.115 * bohr2ang * bohr2ang
            param_bn(1,i)  =    0.1286 * bohr2ang * bohr2ang
            param_bp(0,i)  =     0.325 * bohr2ang * bohr2ang
            param_bp(1,i)  =   -0.0098 * bohr2ang * bohr2ang
            param_b_O(0,i) =     3.447 * bohr2ang
            param_b_O(1,i) =   -0.0390 * bohr2ang
            param_b_H(0,i) =     3.522 * bohr2ang
            param_b_H(1,i) =    0.0069 * bohr2ang
            param_a1(0,i)  =    -79.80 / au2kcal
            param_a1(1,i)  =     33.36 / au2kcal
            param_a1(2,i)  =     -2.56 / au2kcal
            param_a2(0,i)  =     70.98 / au2kcal
            param_a2(1,i)  =    -19.54 / au2kcal
            param_a2(2,i)  =      1.33 / au2kcal
            param_a3(0,i)  =   -100.15 / au2kcal
            param_a3(1,i)  =     22.26 / au2kcal
            param_a3(2,i)  =     -1.14 / au2kcal
            param_a4(0,i)  =     20.14 / au2kcal
            param_a4(1,i)  =      0.00 / au2kcal
            param_a4(2,i)  =     -0.37 / au2kcal
            param_eps(0,i) =     0.298 / au2kcal
            param_eps(1,i) =    -10.78 / au2kcal
            param_eps(2,i) =     84.35 / au2kcal
            param_a_O(0,i) =   9901.00 / au2kcal
            param_a_O(1,i) =      0.00 / au2kcal
            param_a_H(0,i) =   6506.64 / au2kcal
            param_a_H(1,i) =     14.71 / au2kcal
            param_c_O(0,i) =    1250.0 / bohr2ang**6 / au2kcal
            param_cnmax(i) =     12.00

!        //   atomic species
         else if ( species(i)(1:2) .eq. 'Ni' ) then

!            //   Paul Clabaut, Thesis (2021)
!            param_r_O(0,i) =     1.99 / bohr2ang
!            param_bn(0,i)  =     0.21 * bohr2ang * bohr2ang
!            param_bn(1,i)  =     0.00 * bohr2ang * bohr2ang
!            param_bp(0,i)  =     0.28 * bohr2ang * bohr2ang
!            param_bp(1,i)  =     0.00 * bohr2ang * bohr2ang
!            param_b_O(0,i) =     3.23 * bohr2ang
!            param_b_O(1,i) =     0.15 * bohr2ang
!            param_b_H(0,i) =     3.79 * bohr2ang
!            param_b_H(1,i) =    -0.06 * bohr2ang
!            param_a1(0,i)  =  -334.81 / au2kcal
!            param_a1(1,i)  =   107.09 / au2kcal
!            param_a1(2,i)  =    -7.59 / au2kcal
!            param_a2(0,i)  =    16.16 / au2kcal
!            param_a2(1,i)  =    -5.61 / au2kcal
!            param_a2(2,i)  =     0.63 / au2kcal
!            param_a3(0,i)  =   -25.22 / au2kcal
!            param_a3(1,i)  =    -1.03 / au2kcal
!            param_a3(2,i)  =     0.37 / au2kcal
!            param_a4(0,i)  =    16.89 / au2kcal
!            param_a4(1,i)  =     0.00 / au2kcal
!            param_a4(2,i)  =    -0.32 / au2kcal
!            param_eps(0,i) =   137.58 / au2kcal
!            param_eps(1,i) =   -38.61 / au2kcal
!            param_eps(2,i) =     2.71 / au2kcal
!            param_a_O(0,i) =  2702.40 / au2kcal
!            param_a_O(1,i) =  4253.73 / au2kcal
!            param_a_H(0,i) =  7048.14 / au2kcal
!            param_a_H(1,i) =  -395.93 / au2kcal
!            param_c_O(0,i) =   903.00 / bohr2ang**6 / au2kcal
!            param_cnmax(i) =    12.00

!           //    Clabaut, et al. J. Chem. Phys. 157, 194705 (2022)
            param_r_O(0,i) =     2.287 / bohr2ang
            param_bn(0,i)  =     0.204 * bohr2ang * bohr2ang
            param_bn(1,i)  =    0.1133 * bohr2ang * bohr2ang
            param_bp(0,i)  =     0.324 * bohr2ang * bohr2ang
            param_bp(1,i)  =    0.0317 * bohr2ang * bohr2ang
            param_b_O(0,i) =     3.070 * bohr2ang
            param_b_O(1,i) =    0.0325 * bohr2ang
            param_b_H(0,i) =     2.941 * bohr2ang
            param_b_H(1,i) =    0.0881 * bohr2ang
            param_a1(0,i)  =   -194.44 / au2kcal
            param_a1(1,i)  =     73.65 / au2kcal
            param_a1(2,i)  =     -5.88 / au2kcal
            param_a2(0,i)  =     95.91 / au2kcal
            param_a2(1,i)  =    -35.97 / au2kcal
            param_a2(2,i)  =      2.91 / au2kcal
            param_a3(0,i)  =   -109.66 / au2kcal
            param_a3(1,i)  =     32.08 / au2kcal
            param_a3(2,i)  =     -2.20 / au2kcal
            param_a4(0,i)  =     10.60 / au2kcal
            param_a4(1,i)  =      0.00 / au2kcal
            param_a4(2,i)  =     -0.20 / au2kcal
            param_eps(0,i) =    -4.023 / au2kcal
            param_eps(1,i) =     44.87 / au2kcal
            param_eps(2,i) =     64.15 / au2kcal
            param_a_O(0,i) =   4527.04 / au2kcal
            param_a_O(1,i) =   2518.75 / au2kcal
            param_a_H(0,i) =   1805.49 / au2kcal
            param_a_H(1,i) =    352.13 / au2kcal
            param_c_O(0,i) =    1241.0 / bohr2ang**6 / au2kcal
            param_cnmax(i) =     12.00

!        //   atomic species
         else if ( species(i)(1:2) .eq. 'Pd' ) then

!            //   Paul Clabaut, Thesis (2021)
!            param_r_O(0,i) =     2.03 / bohr2ang
!            param_bn(0,i)  =     0.46 * bohr2ang * bohr2ang
!            param_bn(1,i)  =     0.01 * bohr2ang * bohr2ang
!            param_bp(0,i)  =     0.26 * bohr2ang * bohr2ang
!            param_bp(1,i)  =     0.00 * bohr2ang * bohr2ang
!            param_b_O(0,i) =     3.24 * bohr2ang
!            param_b_O(1,i) =    -0.18 * bohr2ang
!            param_b_H(0,i) =     3.94 * bohr2ang
!            param_b_H(1,i) =    -0.07 * bohr2ang
!            param_a1(0,i)  =  -150.34 / au2kcal
!            param_a1(1,i)  =    46.85 / au2kcal
!            param_a1(2,i)  =    -3.06 / au2kcal
!            param_a2(0,i)  =    12.07 / au2kcal
!            param_a2(1,i)  =    -2.78 / au2kcal
!            param_a2(2,i)  =     0.21 / au2kcal
!            param_a3(0,i)  =   -35.32 / au2kcal
!            param_a3(1,i)  =     4.23 / au2kcal
!            param_a3(2,i)  =    -0.02 / au2kcal
!            param_a4(0,i)  =    -6.92 / au2kcal
!            param_a4(1,i)  =     0.00 / au2kcal
!            param_a4(2,i)  =     0.11 / au2kcal
!            param_eps(0,i) =     0.40 / au2kcal
!            param_eps(1,i) =    29.59 / au2kcal
!            param_eps(2,i) =    -3.39 / au2kcal
!            param_a_O(0,i) =  6082.77 / au2kcal
!            param_a_O(1,i) =  -662.05 / au2kcal
!            param_a_H(0,i) = 11756.76 / au2kcal
!            param_a_H(1,i) =  -816.35 / au2kcal
!            param_c_O(0,i) =   872.00 / bohr2ang**6 / au2kcal
!            param_cnmax(i) =    12.00

!           //    Clabaut, et al. J. Chem. Phys. 157, 194705 (2022)
            param_r_O(0,i) =     2.142 / bohr2ang
            param_bn(0,i)  =     0.051 * bohr2ang * bohr2ang
            param_bn(1,i)  =    0.1071 * bohr2ang * bohr2ang
            param_bp(0,i)  =     0.367 * bohr2ang * bohr2ang
            param_bp(1,i)  =   -0.0111 * bohr2ang * bohr2ang
            param_b_O(0,i) =     3.531 * bohr2ang
            param_b_O(1,i) =   -0.0018 * bohr2ang
            param_b_H(0,i) =     3.348 * bohr2ang
            param_b_H(1,i) =    0.0841 * bohr2ang
            param_a1(0,i)  =   -105.59 / au2kcal
            param_a1(1,i)  =     38.54 / au2kcal
            param_a1(2,i)  =     -2.81 / au2kcal
            param_a2(0,i)  =     32.12 / au2kcal
            param_a2(1,i)  =    -10.33 / au2kcal
            param_a2(2,i)  =      0.82 / au2kcal
            param_a3(0,i)  =    -88.67 / au2kcal
            param_a3(1,i)  =     28.30 / au2kcal
            param_a3(2,i)  =     -2.12 / au2kcal
            param_a4(0,i)  =     16.39 / au2kcal
            param_a4(1,i)  =      0.00 / au2kcal
            param_a4(2,i)  =     -0.35 / au2kcal
            param_eps(0,i) =     0.766 / au2kcal
            param_eps(1,i) =    -17.65 / au2kcal
            param_eps(2,i) =    130.56 / au2kcal
            param_a_O(0,i) =  30386.07 / au2kcal
            param_a_O(1,i) =    259.78 / au2kcal
            param_a_H(0,i) =   3219.69 / au2kcal
            param_a_H(1,i) =   1544.04 / au2kcal
            param_c_O(0,i) =    1229.0 / bohr2ang**6 / au2kcal
            param_cnmax(i) =     12.00

!        //   atomic species
         else

!           //   error flag
            ierr = ierr + 1

!        //   atomic species
         end if

!        //   correction: Stephan N. Steinmann, private communication
         eps2 = - param_eps(0,i)
         eps1 = - param_eps(1,i)
         eps0 = - param_eps(2,i)
         param_eps(0,i) = eps0
         param_eps(1,i) = eps1
         param_eps(2,i) = eps2

!     //   loop of atoms
      end do

!     /*   error handling   */
      call error_handling &
     &   ( ierr, 'subroutine force_metalwater_gal_setup', 37 )

      return
      end





!***********************************************************************
      subroutine force_metalwater_common_setup
!***********************************************************************
!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, species, natom, iounit, au_length

      use metalwater_variables, only : &
     &   is_metal, nmetal, nkind_metal, nwater, noxygen, nhydrogen, &
     &   is_hydrogen, is_oxygen, j_bond, k_bond

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   OH bond cutoff
      real(8) :: rcut_qtip4p = 2.45663000E+00

!     //   real numbers
      real(8) :: dx, dy, dz, r2

!     //   integers
      integer :: i, j, k, l, ierr

!     //   integers
      integer :: na_metal(50)

!     //   integers
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     //   initialize
!-----------------------------------------------------------------------

!     /*   for initial access   */
      if ( iset .eq. 0 ) then

!        /*   set complete   */
         iset = 1

!     /*   otherwise   */
      else

!        /*   skip   */
         return

!     /*   for initial access   */
      end if

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

!     //   metal atom or not
      if ( .not. allocated(is_metal) ) allocate( is_metal(natom) )

!     //   hydrogen atom or not
      if ( .not. allocated(is_hydrogen) ) allocate( is_hydrogen(natom) )

!     //   oxygen atom or not
      if ( .not. allocated(is_oxygen) ) allocate( is_oxygen(natom) )

!     //   1st hydrogen atom
      if ( .not. allocated(j_bond) ) allocate( j_bond(natom) )

!     //   2nd hydrogen atom
      if ( .not. allocated(k_bond) ) allocate( k_bond(natom) )

!-----------------------------------------------------------------------
!     //   metal atoms
!-----------------------------------------------------------------------

!     //   number of atoms per metal
      na_metal(:) = 0

!     //   initialize
      is_metal(1:natom) = .false.

!     //   loop of atoms
      do i = 1, natom

!        //   look for metal atoms
         if ( species(i)(1:2) .eq. 'Li' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Be' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Na' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Mg' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Al' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'K ' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Ca' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Sc' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Ti' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'V ' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Cr' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Mn' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Fe' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Co' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Ni' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Cu' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Zn' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Ga' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Ge' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Rb' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Sr' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Y ' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Zr' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Nb' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Mo' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Tc' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Ru' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Pd' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Ag' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Cd' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'In' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Sn' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Sb' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Cs' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Ba' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Hf' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Ta' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'W ' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Re' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Os' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Ir' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Pt' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Au' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Hg' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Tl' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Pb' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Bi' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Po' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Fr' ) is_metal(i) = .true.
         if ( species(i)(1:2) .eq. 'Ra' ) is_metal(i) = .true.

!        //   count number of atoms
         if ( species(i)(1:2) .eq. 'Li' ) na_metal( 1) = na_metal( 1)+1
         if ( species(i)(1:2) .eq. 'Be' ) na_metal( 2) = na_metal( 2)+1
         if ( species(i)(1:2) .eq. 'Na' ) na_metal( 3) = na_metal( 3)+1
         if ( species(i)(1:2) .eq. 'Mg' ) na_metal( 4) = na_metal( 4)+1
         if ( species(i)(1:2) .eq. 'Al' ) na_metal( 5) = na_metal( 5)+1
         if ( species(i)(1:2) .eq. 'K ' ) na_metal( 6) = na_metal( 6)+1
         if ( species(i)(1:2) .eq. 'Ca' ) na_metal( 7) = na_metal( 7)+1
         if ( species(i)(1:2) .eq. 'Sc' ) na_metal( 8) = na_metal( 8)+1
         if ( species(i)(1:2) .eq. 'Ti' ) na_metal( 9) = na_metal( 9)+1
         if ( species(i)(1:2) .eq. 'V ' ) na_metal(10) = na_metal(10)+1
         if ( species(i)(1:2) .eq. 'Cr' ) na_metal(11) = na_metal(11)+1
         if ( species(i)(1:2) .eq. 'Mn' ) na_metal(12) = na_metal(12)+1
         if ( species(i)(1:2) .eq. 'Fe' ) na_metal(13) = na_metal(13)+1
         if ( species(i)(1:2) .eq. 'Co' ) na_metal(14) = na_metal(14)+1
         if ( species(i)(1:2) .eq. 'Ni' ) na_metal(15) = na_metal(15)+1
         if ( species(i)(1:2) .eq. 'Cu' ) na_metal(16) = na_metal(16)+1
         if ( species(i)(1:2) .eq. 'Zn' ) na_metal(17) = na_metal(17)+1
         if ( species(i)(1:2) .eq. 'Ga' ) na_metal(18) = na_metal(18)+1
         if ( species(i)(1:2) .eq. 'Ge' ) na_metal(19) = na_metal(19)+1
         if ( species(i)(1:2) .eq. 'Rb' ) na_metal(20) = na_metal(20)+1
         if ( species(i)(1:2) .eq. 'Sr' ) na_metal(21) = na_metal(21)+1
         if ( species(i)(1:2) .eq. 'Y ' ) na_metal(22) = na_metal(22)+1
         if ( species(i)(1:2) .eq. 'Zr' ) na_metal(23) = na_metal(23)+1
         if ( species(i)(1:2) .eq. 'Nb' ) na_metal(24) = na_metal(24)+1
         if ( species(i)(1:2) .eq. 'Mo' ) na_metal(25) = na_metal(25)+1
         if ( species(i)(1:2) .eq. 'Tc' ) na_metal(26) = na_metal(26)+1
         if ( species(i)(1:2) .eq. 'Ru' ) na_metal(27) = na_metal(27)+1
         if ( species(i)(1:2) .eq. 'Pd' ) na_metal(28) = na_metal(28)+1
         if ( species(i)(1:2) .eq. 'Ag' ) na_metal(29) = na_metal(29)+1
         if ( species(i)(1:2) .eq. 'Cd' ) na_metal(30) = na_metal(30)+1
         if ( species(i)(1:2) .eq. 'In' ) na_metal(31) = na_metal(31)+1
         if ( species(i)(1:2) .eq. 'Sn' ) na_metal(32) = na_metal(32)+1
         if ( species(i)(1:2) .eq. 'Sb' ) na_metal(33) = na_metal(33)+1
         if ( species(i)(1:2) .eq. 'Cs' ) na_metal(34) = na_metal(34)+1
         if ( species(i)(1:2) .eq. 'Ba' ) na_metal(35) = na_metal(35)+1
         if ( species(i)(1:2) .eq. 'Hf' ) na_metal(36) = na_metal(36)+1
         if ( species(i)(1:2) .eq. 'Ta' ) na_metal(37) = na_metal(37)+1
         if ( species(i)(1:2) .eq. 'W ' ) na_metal(38) = na_metal(38)+1
         if ( species(i)(1:2) .eq. 'Re' ) na_metal(39) = na_metal(39)+1
         if ( species(i)(1:2) .eq. 'Os' ) na_metal(40) = na_metal(40)+1
         if ( species(i)(1:2) .eq. 'Ir' ) na_metal(41) = na_metal(41)+1
         if ( species(i)(1:2) .eq. 'Pt' ) na_metal(42) = na_metal(42)+1
         if ( species(i)(1:2) .eq. 'Au' ) na_metal(43) = na_metal(43)+1
         if ( species(i)(1:2) .eq. 'Hg' ) na_metal(44) = na_metal(44)+1
         if ( species(i)(1:2) .eq. 'Tl' ) na_metal(45) = na_metal(45)+1
         if ( species(i)(1:2) .eq. 'Pb' ) na_metal(46) = na_metal(46)+1
         if ( species(i)(1:2) .eq. 'Bi' ) na_metal(47) = na_metal(47)+1
         if ( species(i)(1:2) .eq. 'Po' ) na_metal(48) = na_metal(48)+1
         if ( species(i)(1:2) .eq. 'Fr' ) na_metal(49) = na_metal(49)+1
         if ( species(i)(1:2) .eq. 'Ra' ) na_metal(50) = na_metal(50)+1

!     //   loop of atoms
      end do

!     //   number of metals
      nmetal = 0

!     //   number of metal kinds
      nkind_metal = 0

!     //   elements
      do i = 1, 50

!        //   count number of metals
         nmetal = nmetal + na_metal(i)

!        //   count number of metal kinds
         if ( na_metal(i) .ne. 0 ) nkind_metal = nkind_metal + 1

!     //   elements
      end do

!-----------------------------------------------------------------------
!     //   hydrogen atoms
!-----------------------------------------------------------------------

!     //   hydrogen atom or not
      is_hydrogen(1:natom) = .false.

!     //   oxygen atom or not
      is_oxygen(1:natom)   = .false.

!     //   loop of atoms
      do i = 1, natom

!        //   look for hydrogen
         if ( species(i)(1:2) .eq. 'H ' ) is_hydrogen(i) = .true.
         if ( species(i)(1:2) .eq. 'D ' ) is_hydrogen(i) = .true.
         if ( species(i)(1:2) .eq. 'T ' ) is_hydrogen(i) = .true.
         if ( species(i)(1:2) .eq. 'Mu' ) is_hydrogen(i) = .true.

!        //   look for oxygen
         if ( species(i)(1:2) .eq. 'O ' ) is_oxygen(i)   = .true.

!     //   loop of atoms
      end do

!-----------------------------------------------------------------------
!     //   water molecules
!-----------------------------------------------------------------------

!     //   initialize
      k = 0
      l = 0

!     //   loop of atoms
      do i = 1, natom

!        //   look for oxygen
         if ( is_oxygen(i) )   k = k + 1

!        //   look for hydrogen
         if ( is_hydrogen(i) ) l = l + 1

!     //   loop of atoms
      end do

!     //   number of oxygen atoms
      noxygen    = k

!     //   number of hydrogen atoms
      nhydrogen  = l

!     //   number of water molecules
      nwater     = k

!     //   check
      ierr = 0
      if ( (nmetal+noxygen+nhydrogen) .ne. natom ) ierr = 1
      if ( (2*noxygen) .ne. nhydrogen            ) ierr = 1

!     /*   error handling   */
      call error_handling &
     &   ( ierr, 'subroutine force_metalwater_common_setup', 40 )

!-----------------------------------------------------------------------
!     //   water bonds
!-----------------------------------------------------------------------

!     //   initialize
      j_bond(:) = 0
      k_bond(:) = 0

!     //   error flag
      ierr = 0

!     //   loop of oxygen atoms
      do i = 1, natom

!        //   oxygen atoms only
         if ( .not. is_oxygen(i) ) cycle

!        //   counter
         l = 0

!        //   loop of hydrogen atoms
         do j = 1, natom

!           //   oxygen atoms only
            if ( .not. is_hydrogen(j) ) cycle

!           //   distance
            dx = x(i,1) - x(j,1)
            dy = y(i,1) - y(j,1)
            dz = z(i,1) - z(j,1)

!           //   apply periodic boundary
            call pbc_atom( dx, dy, dz )

!           //   distance squared
            r2 = dx*dx + dy*dy + dz*dz

!           //   within cutoff
            if ( r2 .lt. rcut_qtip4p*rcut_qtip4p ) then

!              //   counter
               l = l + 1

!              //   1st hydrogen
               if ( l .eq. 1 ) then

                  j_bond(i) = j

!              //   2nd hydrogen
               else if ( l .eq. 2 ) then

                  k_bond(i) = j

!              //   hydrogen
               end if

!           //   within cutoff
            end if

         end do

!        //   check error
         if ( l .ne. 2 ) ierr = ierr + 1

      end do

!     /*   error handling   */
      call error_handling &
     &   ( ierr, 'subroutine force_metalwater_common_setup', 40 )

      return
      end





!***********************************************************************
      subroutine force_metalwater_gal
!***********************************************************************
!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, natom, pot, nbead, iounit, au_length, &
     &   au2kcal, vir

      use metalwater_variables, only : &
     &   is_metal, is_hydrogen, rin_cn, rout_cn, cn, gcn, &
     &   param_a_H, param_b_H, param_a_O, param_b_O, param_c_O, &
     &   gcnx, gcny, gcnz, is_oxygen, sx, sy, sz, sxx, sxy, sxz, &
     &   syx, syy, syz, szx, szy, szz, param_eps, param_bp, param_bn, &
     &   j_bond, k_bond, param_a1, param_a2, param_a3, param_a4, &
     &   param_r_O, rin_mo, rout_mo, rin_mh, rout_mh, param_cnmax, &
     &   rout_sf, rin_sf, potential_gal

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   integers
      integer :: i, j, k, l, m, n

!     //   integers
      integer, save :: iset = 0

!     //   real numbers
      real(8) :: xij, yij, zij, rij, rij2, rijinv, xjk, yjk, zjk, rjk, &
     &           rjk2, fij, dfij, fjk, dfjk, a_H, b_H, bx, by, bz, &
     &           xik, yik, zik, rik, rik2, fik, dfik, ax, ay, az, p1, &
     &           p2, p3, p4, p5, p6, br1, a_O, b_O, c_O, br2, br3, &
     &           br4, br5, br6, rij2inv, rij5inv, rij6inv, rij7inv, &
     &           rij8inv, s2, sinv, s3inv, sx_i, sy_i, sz_i, sxx_ii, &
     &           sxy_ii, sxz_ii, syx_ii, syy_ii, syz_ii, szx_ii, &
     &           szy_ii, szz_ii, sxx_ij, sxy_ij, sxz_ij, syx_ij, &
     &           syy_ij, syz_ij, szx_ij, szy_ij, szz_ij, xp, yp, zp, &
     &           xn, yn, zn, rp2, rn2, eps, bp, bn, rs, cx, cy, cz, &
     &           rxi, ryi, rzi, rxj, ryj, rzj, rxk, ryk, rzk, bpn, &
     &           abx, aby, abz, dx, dy, dz, d, ds, dinv, d3inv, d2, &
     &           a1, a2, a3, a4, c1, c2, c3, c4, cxi, cyi, czi, cxj, &
     &           cyj, czj, cxk, cyk, czk, cxl, cyl, czl, ac, dac, ef, &
     &           a1x, a1y, a1z, a2x, a2y, a2z, a3x, a3y, a3z, a, ainv, &
     &           a4x, a4y, a4z, rinv_mo, xil, yil, zil, rilinv, ril2, &
     &           ril, xln, yln, zln, rln2, fil, dfil, cxn, cyn, czn, &
     &           xin, yin, zin, rin, rin2, rininv, fin, dfin, eil, &
     &           fil2, ein, g, h, gxi, gyi, gzi, gxl, gyl, gzl, &
     &           gxn, gyn, gzn, hxi, hyi, hzi, hxj, hyj, hzj, hxk, &
     &           eil2, hyk, hzk, hxl, hyl, hzl, hxn, hyn, hzn, fxi, &
     &           fyi, fzi, fxk, fyk, fzk, fxj, fyj, fzj, fxl, fyl, &
     &           fzl, fxn, fyn, fzn

!     //   real numbers
      real(8) :: tiny = 1.d-10

!-----------------------------------------------------------------------
!     //   initialize
!-----------------------------------------------------------------------

!     /*   for initial access   */
      if ( iset .eq. 0 ) then

!        /*   identify atoms   */
         call force_metalwater_common_setup

!        /*   prepare gal   */
         call force_metalwater_gal_setup

!        /*   set complete   */
         iset = 1

!     /*   for initial access   */
      end if

!-----------------------------------------------------------------------
!     //   start
!-----------------------------------------------------------------------

!     //   net GAL potential
      potential_gal = 0.d0
      do m = 1, nbead
         potential_gal = potential_gal - pot(m)
      end do

!     //   loop of beads
      do m = 1, nbead

!-----------------------------------------------------------------------
!        //   initialize
!-----------------------------------------------------------------------

!        //   initialize CN
         cn(:) = 0.d0

!        //   initialize GCN
         gcn(:)  = 0.d0

!        //   initialize GCN derivative
         gcnx(:,:) = 0.d0
         gcny(:,:) = 0.d0
         gcnz(:,:) = 0.d0

!-----------------------------------------------------------------------
!        //   prepare coordination number
!-----------------------------------------------------------------------

!        //   loop of metal atoms
         do i = 1, natom

!           //   for metal only
            if ( .not. is_metal(i) ) cycle

!           //   loop of metal atoms
            do j = 1, natom

!              //   for metal only
               if ( ( j .eq. i ) .or. ( .not. is_metal(j) ) ) cycle

!              //   distance squared
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              //   apply periodic boundary if needed
               call pbc_atom( xij, yij, zij )

!              //   distance squared
               rij2 = xij*xij + yij*yij + zij*zij

!              //   skip if distance longer than outer cutoff
               if ( rij2 .gt. rout_cn*rout_cn ) cycle

!              //   distance
               rij  = sqrt(rij2)

!              //   damping function
               call getswf ( rij, rin_cn, rout_cn, fij, dfij )

!              //   add contribution to coordination number
               cn(i) = cn(i) + fij

!           //   loop of metal atoms
            end do

!        //   loop of metal atoms
         end do

!-----------------------------------------------------------------------
!        //   prepare GCN
!-----------------------------------------------------------------------

!        //   loop of metal atoms
         do i = 1, natom

!           //   for metal only
            if ( .not. is_metal(i) ) cycle

!           //   loop of metal atoms
            do j = 1, natom

!              //   for metal only
               if ( ( j .eq. i ) .or. ( .not. is_metal(j) ) ) cycle

!              //   distance squared
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              //   apply periodic boundary if needed
               call pbc_atom( xij, yij, zij )

!              //   distance squared
               rij2 = xij*xij + yij*yij + zij*zij

!              //   skip if distance longer than outer cutoff
               if ( rij2 .gt. rout_cn*rout_cn ) cycle

!              //   distance
               rij  = sqrt(rij2)

!              //   damping function
               call getswf ( rij, rin_cn, rout_cn, fij, dfij )

!              //   add contribution to generalized coordination number
               gcn(i) = gcn(i) + cn(j) * fij / param_cnmax(j)

!           //   loop of metal atoms
            end do

!        //   loop of metal atoms
         end do

!-----------------------------------------------------------------------
!        //   prepare GCN derivative
!-----------------------------------------------------------------------

!        //   loop of metal atoms
         do i = 1, natom

!           //   for metal only
            if ( .not. is_metal(i) ) cycle

!           //   loop of metal atoms
            do j = 1, natom

!              //   for metal only
               if ( ( j .eq. i ) .or. ( .not. is_metal(j) ) ) cycle

!              //   distance squared
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              //   apply periodic boundary if needed
               call pbc_atom( xij, yij, zij )

!              //   distance squared
               rij2 = xij*xij + yij*yij + zij*zij

!              //   distance shorter than inner cutoff
               if ( rij2 .lt. rin_cn*rin_cn ) then

!                 //   damping function
                  fij  = 1.d0
                  dfij = 0.d0

!              //   distance longer than outer cutoff
               else if ( rij2 .gt. rout_cn*rout_cn ) then

!                 //   damping function
                  fij  = 0.d0
                  dfij = 0.d0

!              //   distance in between inner and outer cutoff
               else

!                 //   distance
                  rij  = sqrt(rij2)

!                 //   damping function
                  call getswf ( rij, rin_cn, rout_cn, fij, dfij )

!                 //   part 1
                  p1 = ( cn(j) + fij ) * dfij / rij / param_cnmax(j)

!                 //   add contribution to its derivative
                  gcnx(i,i) = gcnx(i,i) + p1 * xij
                  gcny(i,i) = gcny(i,i) + p1 * yij
                  gcnz(i,i) = gcnz(i,i) + p1 * zij

!                 //   part 2
                  p2 = cn(j) * dfij / rij / param_cnmax(j)

!                 //   add contribution to its derivative
                  gcnx(i,j) = gcnx(i,j) - p2 * xij
                  gcny(i,j) = gcny(i,j) - p2 * yij
                  gcnz(i,j) = gcnz(i,j) - p2 * zij

!              //   distance
               end if

!              //   loop of metal atoms
               do k = 1, natom

!                 //   for metal only
                  if ( ( k .eq. j ) .or. ( .not. is_metal(k) ) ) cycle

!                 //   distance squared
                  xjk = x(j,m) - x(k,m)
                  yjk = y(j,m) - y(k,m)
                  zjk = z(j,m) - z(k,m)

!                 //   apply periodic boundary if needed
                  call pbc_atom( xjk, yjk, zjk )

!                 //   distance squared
                  rjk2 = xjk*xjk + yjk*yjk + zjk*zjk

!                 //   distance in between inner and outer cutoff
                  if ( rjk2 .lt. 4.d0*rout_cn*rout_cn ) then

!                    //   distance
                     rjk  = sqrt(rjk2)

!                    //   damping function
                     call getswf ( rjk, rin_cn, rout_cn, fjk, dfjk )

!                    //   part 1
                     p1 = dfjk * fij / rjk / param_cnmax(j)

!                    //   add contribution to its gradient
                     gcnx(i,j) = gcnx(i,j) + p1 * xjk
                     gcny(i,j) = gcny(i,j) + p1 * yjk
                     gcnz(i,j) = gcnz(i,j) + p1 * zjk

!                    //   for metal only
                     if ( k .eq. i ) cycle

!                    //   distance squared
                     xik = x(i,m) - x(k,m)
                     yik = y(i,m) - y(k,m)
                     zik = z(i,m) - z(k,m)

!                    //   apply periodic boundary if needed
                     call pbc_atom( xik, yik, zik )

!                    //   distance squared
                     rik2 = xik*xik + yik*yik + zik*zik

!                    //   distance shorter than outer cutoff distance
                     if ( rik2 .lt. rout_cn*rout_cn ) then

!                       //   distance
                        rik  = sqrt(rik2)

!                       //   damping function
                        call getswf ( rik, rin_cn, rout_cn, fik, dfik )

!                       //   part 1
                        p1 = dfjk * fik / rjk / param_cnmax(j)

!                       //   add contribution to its gradient
                        gcnx(i,j) = gcnx(i,j) + p1 * xjk
                        gcny(i,j) = gcny(i,j) + p1 * yjk
                        gcnz(i,j) = gcnz(i,j) + p1 * zjk

!                    //   distance shorter than outer cutoff distance
                     end if

!                 //   distance
                  end if

!              //   loop of metal atoms
               end do

!           //   loop of metal atoms
            end do

!        //   loop of metal atoms
         end do

!-----------------------------------------------------------------------
!        //   metal-hydrogen interaction
!-----------------------------------------------------------------------

!        //   loop of hydrogen atoms
         do i = 1, natom

!           //   for hydrogen only
            if ( .not. is_hydrogen(i) ) cycle

!           //   loop of metal atoms
            do j = 1, natom

!              //   for metal only
               if ( .not. is_metal(j) ) cycle

!              //   distance squared
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              //   apply periodic boundary if needed
               call pbc_atom( xij, yij, zij )

!              //   distance squared
               rij2 = xij*xij + yij*yij + zij*zij

!              //   skip if distance longer than twice outer cutoff
               if ( rij2 .gt. rout_mh*rout_mh ) cycle

!              //   distance
               rij  = sqrt( rij2 )

!              //   damping function
               call getswf ( rij, rin_mh, rout_mh, fij, dfij )

!              //   inverse of r
               rijinv  = 1.d0 / rij

!              //   parameter weighted by GCN
               a_H = param_a_H(0,j) + gcn(j) * param_a_H(1,j)
               b_H = param_b_H(0,j) + gcn(j) * param_b_H(1,j)

!              //   part 1
               p1 = exp( - b_H * rij )

!              //   add contribution to potential
               pot(m) = pot(m) + a_H * p1 * fij

!              //   part 2
               p2 = a_H * ( b_H * fij - dfij ) * p1 * rijinv

!              //   force on atom i
               fxi = p2 * xij
               fyi = p2 * yij
               fzi = p2 * zij

!              //   add contribution to force
               fx(i,m) = fx(i,m) + fxi
               fy(i,m) = fy(i,m) + fyi
               fz(i,m) = fz(i,m) + fzi

!              //   add contribution to force
               fx(j,m) = fx(j,m) - fxi
               fy(j,m) = fy(j,m) - fyi
               fz(j,m) = fz(j,m) - fzi

!              //   add contribution to virial
               vir(1,1) = vir(1,1) + fxi*xij
               vir(1,2) = vir(1,2) + fxi*yij
               vir(1,3) = vir(1,3) + fxi*zij
               vir(2,1) = vir(2,1) + fyi*xij
               vir(2,2) = vir(2,2) + fyi*yij
               vir(2,3) = vir(2,3) + fyi*zij
               vir(3,1) = vir(3,1) + fzi*xij
               vir(3,2) = vir(3,2) + fzi*yij
               vir(3,3) = vir(3,3) + fzi*zij

!              //   GCN corrected a_H
               ax = gcnx(j,j) * param_a_H(1,j)
               ay = gcny(j,j) * param_a_H(1,j)
               az = gcnz(j,j) * param_a_H(1,j)

!              //   GCN corrected b_H
               bx = gcnx(j,j) * param_b_H(1,j)
               by = gcny(j,j) * param_b_H(1,j)
               bz = gcnz(j,j) * param_b_H(1,j)

!              //   force on atom j
               fxj = - ( ax - a_H * bx * rij ) * p1 * fij
               fyj = - ( ay - a_H * by * rij ) * p1 * fij
               fzj = - ( az - a_H * bz * rij ) * p1 * fij

!              //   add contribution to force
               fx(j,m) = fx(j,m) + fxj
               fy(j,m) = fy(j,m) + fyj
               fz(j,m) = fz(j,m) + fzj

!              //   loop of metal atoms
               do k = 1, natom

!                 //   for metal only
                  if ( ( k .eq. j ) .or. ( .not. is_metal(k) ) ) cycle

!                 //   distance squared
                  xjk = x(j,m) - x(k,m)
                  yjk = y(j,m) - y(k,m)
                  zjk = z(j,m) - z(k,m)

!                 //   apply periodic boundary if needed
                  call pbc_atom( xjk, yjk, zjk )

!                 //   distance squared
                  rjk2 = xjk*xjk + yjk*yjk + zjk*zjk

!                 //   skip if distance longer than twice outer cutoff
                  if ( rjk2 .gt. 4.d0*rout_cn*rout_cn ) cycle

!                 //   GCN corrected a_H
                  ax = gcnx(j,k) * param_a_H(1,j)
                  ay = gcny(j,k) * param_a_H(1,j)
                  az = gcnz(j,k) * param_a_H(1,j)

!                 //   GCN corrected b_H
                  bx = gcnx(j,k) * param_b_H(1,j)
                  by = gcny(j,k) * param_b_H(1,j)
                  bz = gcnz(j,k) * param_b_H(1,j)

!                 //   force on k atom
                  fxk = - ( ax - a_H * bx * rij ) * p1 * fij
                  fyk = - ( ay - a_H * by * rij ) * p1 * fij
                  fzk = - ( az - a_H * bz * rij ) * p1 * fij

!                 //   add contribution to force
                  fx(k,m) = fx(k,m) + fxk
                  fy(k,m) = fy(k,m) + fyk
                  fz(k,m) = fz(k,m) + fzk

!                 //   add contribution to virial
                  vir(1,1) = vir(1,1) - fxk*xjk
                  vir(1,2) = vir(1,2) - fxk*yjk
                  vir(1,3) = vir(1,3) - fxk*zjk
                  vir(2,1) = vir(2,1) - fyk*xjk
                  vir(2,2) = vir(2,2) - fyk*yjk
                  vir(2,3) = vir(2,3) - fyk*zjk
                  vir(3,1) = vir(3,1) - fzk*xjk
                  vir(3,2) = vir(3,2) - fzk*yjk
                  vir(3,3) = vir(3,3) - fzk*zjk

!              //   loop of metal atoms
               end do

!           //   loop of metal atoms
            end do

!        //   loop of hydrogen atoms
         end do

!-----------------------------------------------------------------------
!        //   metal-oxygen Tang-Toennies term
!-----------------------------------------------------------------------

!        //   loop of oxygen atoms
         do i = 1, natom

!           //   for oxygen only
            if ( .not. is_oxygen(i) ) cycle

!           //   loop of metal atoms
            do j = 1, natom

!              //   for metal only
               if ( .not. is_metal(j) ) cycle

!              //   parameter weighted by GCN
               a_O = param_a_O(0,j) + gcn(j) * param_a_O(1,j)
               b_O = param_b_O(0,j) + gcn(j) * param_b_O(1,j)
               c_O = param_c_O(0,j)

!              //   distance squared
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              //   apply periodic boundary if needed
               call pbc_atom( xij, yij, zij )

!              //   distance squared
               rij2 = xij*xij + yij*yij + zij*zij

!              //   skip if distance longer than twice outer cutoff
               if ( rij2 .gt. rout_mo*rout_mo ) cycle

!              //   distance
               rij  = sqrt( rij2 )

!              //   damping function
               call getswf ( rij, rin_mo, rout_mo, fij, dfij )

!              //   inverse of r
               rijinv  = 1.d0 / rij

!              //   part 1
               p1 = exp( - b_O * rij )

!              //   add contribution to potential
               pot(m) = pot(m) + a_O * p1 * fij

!              //   part 2
               p2 = a_O * ( b_O * fij - dfij ) * rijinv * p1

!              //   force on atom i
               fxi = p2 * xij
               fyi = p2 * yij
               fzi = p2 * zij

!              //   add contribution to force
               fx(i,m) = fx(i,m) + fxi
               fy(i,m) = fy(i,m) + fyi
               fz(i,m) = fz(i,m) + fzi

!              //   add contribution to force
               fx(j,m) = fx(j,m) - fxi
               fy(j,m) = fy(j,m) - fyi
               fz(j,m) = fz(j,m) - fzi

!              //   add contribution to virial
               vir(1,1) = vir(1,1) + fxi*xij
               vir(1,2) = vir(1,2) + fxi*yij
               vir(1,3) = vir(1,3) + fxi*zij
               vir(2,1) = vir(2,1) + fyi*xij
               vir(2,2) = vir(2,2) + fyi*yij
               vir(2,3) = vir(2,3) + fyi*zij
               vir(3,1) = vir(3,1) + fzi*xij
               vir(3,2) = vir(3,2) + fzi*yij
               vir(3,3) = vir(3,3) + fzi*zij

!              //   GCN corrected a_O
               ax = gcnx(j,j) * param_a_O(1,j)
               ay = gcny(j,j) * param_a_O(1,j)
               az = gcnz(j,j) * param_a_O(1,j)

!              //   GCN corrected b_O
               bx = gcnx(j,j) * param_b_O(1,j)
               by = gcny(j,j) * param_b_O(1,j)
               bz = gcnz(j,j) * param_b_O(1,j)

!              //   force on atom j
               fxj = - ( ax - a_O * bx * rij ) * p1 * fij
               fyj = - ( ay - a_O * by * rij ) * p1 * fij
               fzj = - ( az - a_O * bz * rij ) * p1 * fij

!              //   add contribution to force
               fx(j,m) = fx(j,m) + fxj
               fy(j,m) = fy(j,m) + fyj
               fz(j,m) = fz(j,m) + fzj

!              //   preparation
               br1 = b_O * rij
               br2 = br1 * br1
               br3 = br1 * br2
               br4 = br1 * br3
               br5 = br1 * br4
               br6 = br1 * br5

!              //   preparation
               rij2inv = rijinv * rijinv
               rij6inv = rij2inv * rij2inv * rij2inv
               rij8inv = rij2inv * rij6inv

!              //   part 3
               p3 = 1.d0 + br1 + br2/2.d0 + br3/6.d0 + br4/24.d0 &
     &            + br5/120.d0 + br6/720.d0

!              //   add contribution to potential
               pot(m) = pot(m) &
     &            - ( 1.d0 - p3 * p1 ) * c_O * rij6inv * fij

!              //   part 4
               p4 = ( 1.d0       * (      - br1 ) &
     &              + br1        * ( 1.d0 - br1 ) &
     &              + br2/2.d0   * ( 2.d0 - br1 ) &
     &              + br3/6.d0   * ( 3.d0 - br1 ) &
     &              + br4/24.d0  * ( 4.d0 - br1 ) &
     &              + br5/120.d0 * ( 5.d0 - br1 ) &
     &              + br6/720.d0 * ( 6.d0 - br1 ) ) &
     &              * p1 * c_O * rij8inv * fij

!              //   part 5
               p5 = ( 1.d0 - p3 * p1 ) * 6.d0 * c_O * rij8inv * fij &
     &            - ( 1.d0 - p3 * p1 ) * c_O * rij6inv * dfij * rijinv

!              //   force on atom i
               fxi = - ( p4 + p5 ) * xij
               fyi = - ( p4 + p5 ) * yij
               fzi = - ( p4 + p5 ) * zij

!              //   add contribution to force
               fx(i,m) = fx(i,m) + fxi
               fy(i,m) = fy(i,m) + fyi
               fz(i,m) = fz(i,m) + fzi

!              //   add contribution to force
               fx(j,m) = fx(j,m) - fxi
               fy(j,m) = fy(j,m) - fyi
               fz(j,m) = fz(j,m) - fzi

!              //   add contribution to virial
               vir(1,1) = vir(1,1) + fxi*xij
               vir(1,2) = vir(1,2) + fxi*yij
               vir(1,3) = vir(1,3) + fxi*zij
               vir(2,1) = vir(2,1) + fyi*xij
               vir(2,2) = vir(2,2) + fyi*yij
               vir(2,3) = vir(2,3) + fyi*zij
               vir(3,1) = vir(3,1) + fzi*xij
               vir(3,2) = vir(3,2) + fzi*yij
               vir(3,3) = vir(3,3) + fzi*zij

!              //   part 6
               p6 = p4 * rij2 / b_O

!              //   force on atom j
               fxj = - p6 * bx
               fyj = - p6 * by
               fzj = - p6 * bz

!              //   add contribution to force
               fx(j,m) = fx(j,m) + fxj
               fy(j,m) = fy(j,m) + fyj
               fz(j,m) = fz(j,m) + fzj

!              //   loop of metal atoms
               do k = 1, natom

!                 //   for metal only
                  if ( ( k .eq. j ) .or. ( .not. is_metal(k) ) ) cycle

!                 //   distance squared
                  xjk = x(j,m) - x(k,m)
                  yjk = y(j,m) - y(k,m)
                  zjk = z(j,m) - z(k,m)

!                 //   apply periodic boundary if needed
                  call pbc_atom( xjk, yjk, zjk )

!                 //   distance squared
                  rjk2 = xjk*xjk + yjk*yjk + zjk*zjk

!                 //   skip if distance longer than twice outer cutoff
                  if ( rjk2 .gt. 4.d0*rout_cn*rout_cn ) cycle

!                 //   GCN corrected a_O
                  ax = gcnx(j,k) * param_a_O(1,j)
                  ay = gcny(j,k) * param_a_O(1,j)
                  az = gcnz(j,k) * param_a_O(1,j)

!                 //   GCN corrected b_O
                  bx = gcnx(j,k) * param_b_O(1,j)
                  by = gcny(j,k) * param_b_O(1,j)
                  bz = gcnz(j,k) * param_b_O(1,j)

                  fxk = - ( ax - a_O * bx * rij ) * p1 * fij - p6 * bx
                  fyk = - ( ay - a_O * by * rij ) * p1 * fij - p6 * by
                  fzk = - ( az - a_O * bz * rij ) * p1 * fij - p6 * bz

!                 //   add contribution to force
                  fx(k,m) = fx(k,m) + fxk
                  fy(k,m) = fy(k,m) + fyk
                  fz(k,m) = fz(k,m) + fzk

!                 //   add contribution to virial
                  vir(1,1) = vir(1,1) - fxk*xjk
                  vir(1,2) = vir(1,2) - fxk*yjk
                  vir(1,3) = vir(1,3) - fxk*zjk
                  vir(2,1) = vir(2,1) - fyk*xjk
                  vir(2,2) = vir(2,2) - fyk*yjk
                  vir(2,3) = vir(2,3) - fyk*zjk
                  vir(3,1) = vir(3,1) - fzk*xjk
                  vir(3,2) = vir(3,2) - fzk*yjk
                  vir(3,3) = vir(3,3) - fzk*zjk

!              //   loop of metal atoms
               end do

!           //   loop of metal atoms
            end do

!        //   loop of oxygen atoms
         end do

!-----------------------------------------------------------------------
!        //   normal vector
!-----------------------------------------------------------------------

!        //   initialized
         sx(:) = 0.d0
         sy(:) = 0.d0
         sz(:) = 0.d0

!        //   initialized
         sxx(:,:) = 0.d0
         sxy(:,:) = 0.d0
         sxz(:,:) = 0.d0
         syx(:,:) = 0.d0
         syy(:,:) = 0.d0
         syz(:,:) = 0.d0
         szx(:,:) = 0.d0
         szy(:,:) = 0.d0
         szz(:,:) = 0.d0

!        //   loop of metal atoms
         do i = 1, natom

!           //   for metal only
            if ( .not. is_metal(i) ) cycle

!           //   surface vector
            sx_i = 0.d0
            sy_i = 0.d0
            sz_i = 0.d0

!           //   surface vector derivative
            sxx_ii = 0.d0
            sxy_ii = 0.d0
            sxz_ii = 0.d0
            syx_ii = 0.d0
            syy_ii = 0.d0
            syz_ii = 0.d0
            szx_ii = 0.d0
            szy_ii = 0.d0
            szz_ii = 0.d0

!           //   loop of metal atoms
            do j = 1, natom

!              //   for metal only
               if ( ( j .eq. i ) .or. ( .not. is_metal(j) ) ) cycle

!              //   distance squared
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              //   apply periodic boundary if needed
               call pbc_atom( xij, yij, zij )

!              //   distance squared
               rij2 = xij*xij + yij*yij + zij*zij

!              //   skip if distance longer than outer cutoff
               if ( rij2 .gt. rout_sf*rout_sf ) cycle

!              //   distance
               rij  = sqrt(rij2)

!              //   damping function
               call getswf ( rij, rin_sf, rout_sf, fij, dfij )

!              //   preparation
               rijinv  = 1.d0 / rij
               rij2inv = rijinv * rijinv
               rij5inv = rijinv * rij2inv * rij2inv
               rij6inv = rijinv * rij5inv
               rij7inv = rij2inv * rij5inv

!              //   surface vector
               sx_i = sx_i + xij * rij5inv * fij
               sy_i = sy_i + yij * rij5inv * fij
               sz_i = sz_i + zij * rij5inv * fij

!              //   surface vector derivative
               sxx_ii = sxx_ii - ( 5.d0*xij*xij*rij7inv - rij5inv ) *fij
               sxy_ii = sxy_ii - ( 5.d0*xij*yij*rij7inv ) *fij
               sxz_ii = sxz_ii - ( 5.d0*xij*zij*rij7inv ) *fij
               syx_ii = syx_ii - ( 5.d0*yij*xij*rij7inv ) *fij
               syy_ii = syy_ii - ( 5.d0*yij*yij*rij7inv - rij5inv ) *fij
               syz_ii = syz_ii - ( 5.d0*yij*zij*rij7inv ) *fij
               szx_ii = szx_ii - ( 5.d0*zij*xij*rij7inv ) *fij
               szy_ii = szy_ii - ( 5.d0*zij*yij*rij7inv ) *fij
               szz_ii = szz_ii - ( 5.d0*zij*zij*rij7inv - rij5inv ) *fij

!              //   surface vector derivative
               sxx_ii = sxx_ii + xij*xij*rij6inv*dfij
               sxy_ii = sxy_ii + xij*yij*rij6inv*dfij
               sxz_ii = sxz_ii + xij*zij*rij6inv*dfij
               syx_ii = syx_ii + yij*xij*rij6inv*dfij
               syy_ii = syy_ii + yij*yij*rij6inv*dfij
               syz_ii = syz_ii + yij*zij*rij6inv*dfij
               szx_ii = szx_ii + zij*xij*rij6inv*dfij
               szy_ii = szy_ii + zij*yij*rij6inv*dfij
               szz_ii = szz_ii + zij*zij*rij6inv*dfij

!           //   loop of metal atoms
            end do

!           //   norm of surface vector
            s2 = sx_i*sx_i + sy_i*sy_i + sz_i*sz_i

!           //   on error
            if ( s2 .lt. tiny ) cycle

!           //   preparation
            sinv  = 1.d0 / sqrt(s2)
            s3inv = sinv * sinv * sinv

!           //   surface vector
            sx(i) = sx_i * sinv
            sy(i) = sy_i * sinv
            sz(i) = sz_i * sinv

!           //   surface vector derivatives
            sxx(i,i) = sxx_ii * sinv &
     &             - sxx_ii * sx_i*sx_i * s3inv &
     &             - syx_ii * sx_i*sy_i * s3inv &
     &             - szx_ii * sx_i*sz_i * s3inv

!           //   surface vector derivatives
            sxy(i,i) = sxy_ii * sinv &
     &             - sxy_ii * sx_i*sx_i * s3inv &
     &             - syy_ii * sx_i*sy_i * s3inv &
     &             - szy_ii * sx_i*sz_i * s3inv

!           //   surface vector derivatives
            sxz(i,i) = sxz_ii * sinv &
     &             - sxz_ii * sx_i*sx_i * s3inv &
     &             - szz_ii * sx_i*sz_i * s3inv &
     &             - syz_ii * sx_i*sy_i * s3inv

!           //   surface vector derivatives
            syx(i,i) = syx_ii * sinv &
     &             - syx_ii * sy_i*sy_i * s3inv &
     &             - sxx_ii * sy_i*sx_i * s3inv &
     &             - szx_ii * sy_i*sz_i * s3inv

!           //   surface vector derivatives
            syy(i,i) = syy_ii * sinv &
     &             - syy_ii * sy_i*sy_i * s3inv &
     &             - szy_ii * sy_i*sz_i * s3inv &
     &             - sxy_ii * sy_i*sx_i * s3inv

!           //   surface vector derivatives
            syz(i,i) = syz_ii * sinv &
     &             - syz_ii * sy_i*sy_i * s3inv &
     &             - szz_ii * sy_i*sz_i * s3inv &
     &             - sxz_ii * sy_i*sx_i * s3inv

!           //   surface vector derivatives
            szx(i,i) = szx_ii * sinv &
     &             - szx_ii * sz_i*sz_i * s3inv &
     &             - sxx_ii * sz_i*sx_i * s3inv &
     &             - syx_ii * sz_i*sy_i * s3inv

!           //   surface vector derivatives
            szy(i,i) = szy_ii * sinv &
     &             - szy_ii * sz_i*sz_i * s3inv &
     &             - syy_ii * sz_i*sy_i * s3inv &
     &             - sxy_ii * sz_i*sx_i * s3inv

!           //   surface vector derivatives
            szz(i,i) = szz_ii * sinv &
     &             - szz_ii * sz_i*sz_i * s3inv &
     &             - sxz_ii * sz_i*sx_i * s3inv &
     &             - syz_ii * sz_i*sy_i * s3inv

!           //   loop of metal atoms
            do j = 1, natom

!              //   for metal only
               if ( ( j .eq. i ) .or. ( .not. is_metal(j) ) ) cycle

!              //   distance squared
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              //   apply periodic boundary if needed
               call pbc_atom( xij, yij, zij )

!              //   distance squared
               rij2 = xij*xij + yij*yij + zij*zij

!              //   skip if distance longer than outer cutoff
               if ( rij2 .gt. rout_sf*rout_sf ) cycle

!              //   distance
               rij  = sqrt(rij2)

!              //   damping function
               call getswf ( rij, rin_sf, rout_sf, fij, dfij )

!              //   preparation
               rijinv  = 1.d0 / rij
               rij2inv = rijinv * rijinv
               rij5inv = rijinv * rij2inv * rij2inv
               rij6inv = rijinv * rij5inv
               rij7inv = rij2inv * rij5inv

!              //   surface vector derivative
               sxx_ij = + ( 5.d0*xij*xij*rij7inv - rij5inv ) *fij
               sxy_ij = + ( 5.d0*xij*yij*rij7inv ) *fij
               sxz_ij = + ( 5.d0*xij*zij*rij7inv ) *fij
               syx_ij = + ( 5.d0*yij*xij*rij7inv ) *fij
               syy_ij = + ( 5.d0*yij*yij*rij7inv - rij5inv ) *fij
               syz_ij = + ( 5.d0*yij*zij*rij7inv ) *fij
               szx_ij = + ( 5.d0*zij*xij*rij7inv ) *fij
               szy_ij = + ( 5.d0*zij*yij*rij7inv ) *fij
               szz_ij = + ( 5.d0*zij*zij*rij7inv - rij5inv ) *fij

!              //   surface vector derivative
               sxx_ij = sxx_ij - xij*xij*rij6inv*dfij
               sxy_ij = sxy_ij - xij*yij*rij6inv*dfij
               sxz_ij = sxz_ij - xij*zij*rij6inv*dfij
               syx_ij = syx_ij - yij*xij*rij6inv*dfij
               syy_ij = syy_ij - yij*yij*rij6inv*dfij
               syz_ij = syz_ij - yij*zij*rij6inv*dfij
               szx_ij = szx_ij - zij*xij*rij6inv*dfij
               szy_ij = szy_ij - zij*yij*rij6inv*dfij
               szz_ij = szz_ij - zij*zij*rij6inv*dfij

!              //   surface vector derivatives
               sxx(i,j) = sxx_ij * sinv &
        &             - sxx_ij * sx_i*sx_i * s3inv &
        &             - syx_ij * sx_i*sy_i * s3inv &
        &             - szx_ij * sx_i*sz_i * s3inv

!              //   surface vector derivatives
               sxy(i,j) = sxy_ij * sinv &
        &             - sxy_ij * sx_i*sx_i * s3inv &
        &             - syy_ij * sx_i*sy_i * s3inv &
        &             - szy_ij * sx_i*sz_i * s3inv

!              //   surface vector derivatives
               sxz(i,j) = sxz_ij * sinv &
        &             - sxz_ij * sx_i*sx_i * s3inv &
        &             - szz_ij * sx_i*sz_i * s3inv &
        &             - syz_ij * sx_i*sy_i * s3inv

!              //   surface vector derivatives
               syx(i,j) = syx_ij * sinv &
        &             - syx_ij * sy_i*sy_i * s3inv &
        &             - sxx_ij * sy_i*sx_i * s3inv &
        &             - szx_ij * sy_i*sz_i * s3inv

!              //   surface vector derivatives
               syy(i,j) = syy_ij * sinv &
        &             - syy_ij * sy_i*sy_i * s3inv &
        &             - szy_ij * sy_i*sz_i * s3inv &
        &             - sxy_ij * sy_i*sx_i * s3inv

!              //   surface vector derivatives
               syz(i,j) = syz_ij * sinv &
        &             - syz_ij * sy_i*sy_i * s3inv &
        &             - szz_ij * sy_i*sz_i * s3inv &
        &             - sxz_ij * sy_i*sx_i * s3inv

!              //   surface vector derivatives
               szx(i,j) = szx_ij * sinv &
        &             - szx_ij * sz_i*sz_i * s3inv &
        &             - sxx_ij * sz_i*sx_i * s3inv &
        &             - syx_ij * sz_i*sy_i * s3inv

!              //   surface vector derivatives
               szy(i,j) = szy_ij * sinv &
        &             - szy_ij * sz_i*sz_i * s3inv &
        &             - syy_ij * sz_i*sy_i * s3inv &
        &             - sxy_ij * sz_i*sx_i * s3inv

!              //   surface vector derivatives
               szz(i,j) = szz_ij * sinv &
        &             - szz_ij * sz_i*sz_i * s3inv &
        &             - sxz_ij * sz_i*sx_i * s3inv &
        &             - syz_ij * sz_i*sy_i * s3inv

!           //   loop of metal atoms
            end do

!        //   loop of metal atoms
         end do

!-----------------------------------------------------------------------
!        //   Gaussian term
!-----------------------------------------------------------------------

!        //   loop of oxygen atoms
         do i = 1, natom

!           //   for oxygen only
            if ( .not. is_oxygen(i) ) cycle

!           //   loop of metal atoms
            do j = 1, natom

!              //   for metal only
               if ( .not. is_metal(j) ) cycle

!              //   distance squared
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              //   apply periodic boundary if needed
               call pbc_atom( xij, yij, zij )

!              //   distance squared
               rij2 = xij*xij + yij*yij + zij*zij

!              //   inner product
               s2 = sx(j)*sx(j) + sy(j)*sy(j) + sz(j)*sz(j)

!              //   skip if distance longer than outer cutoff
               if ( ( rij2 .gt. rout_mo*rout_mo ) .or. &
     &              ( s2 .lt. tiny ) ) cycle

!              //   distance
               rij  = sqrt( rij2 )

!              //   damping function
               call getswf ( rij, rin_mo, rout_mo, fij, dfij )

!              //   inverse of r
               rijinv  = 1.d0 / rij

!              //   inner product
               rs = xij*sx(j) + yij*sy(j) + zij*sz(j)

!              //   parallel component
               xp = rs * sx(j)
               yp = rs * sy(j)
               zp = rs * sz(j)

!              //   apply periodic boundary if needed
               call pbc_atom( xp, yp, zp )

!              //   square
               rp2 = xp*xp + yp*yp + zp*zp

!              //   normal component
               xn = xij - xp
               yn = yij - yp
               zn = zij - zp

!              //   apply periodic boundary if needed
               call pbc_atom( xn, yn, zn )

!              //   square
               rn2 = xn*xn + yn*yn + zn*zn

!              //   parameter weighted by GCN
               bp  = param_bp(0,j) + gcn(j) * param_bp(1,j)
               bn  = param_bn(0,j) + gcn(j) * param_bn(1,j)

!              //   parameter weighted by GCN
               bpn = bp - bn

!              //   parameter weighted by GCN
               eps = param_eps(0,j) + gcn(j) * param_eps(1,j) &
     &             + gcn(j) * gcn(j) * param_eps(2,j)

!              //   part 1
               p1 = exp( - bp*rp2 - bn*rn2 )
!               p1 = exp( - bpn*rp2 - bn*rij2 )

!              //   energy
               p2 = eps * p1

!              //   energy with damping
               p3 = p2 * fij

!              //   add to energy
               pot(m) = pot(m) + p3

!              //   GCN corrected bp
               ax = gcnx(j,j) * param_bp(1,j)
               ay = gcny(j,j) * param_bp(1,j)
               az = gcnz(j,j) * param_bp(1,j)

!              //   GCN corrected bn
               bx = gcnx(j,j) * param_bn(1,j)
               by = gcny(j,j) * param_bn(1,j)
               bz = gcnz(j,j) * param_bn(1,j)

!              //   GCN corrected bp
               abx = ax - bx
               aby = ay - by
               abz = az - bz

!              //   GCN corrected epsilon
               cx = gcnx(j,j) * param_eps(1,j) &
    &             + 2.d0 * gcnx(j,j) * gcn(j) * param_eps(2,j)
               cy = gcny(j,j) * param_eps(1,j) &
    &             + 2.d0 * gcny(j,j) * gcn(j) * param_eps(2,j)
               cz = gcnz(j,j) * param_eps(1,j) &
    &             + 2.d0 * gcnz(j,j) * gcn(j) * param_eps(2,j)

!              //   preparation
               rxi = 2.d0 * rs * sx(j)
               ryi = 2.d0 * rs * sy(j)
               rzi = 2.d0 * rs * sz(j)

!              //   preparation
               rxj = xij * sxx(j,j) + yij * syx(j,j) + zij * szx(j,j)
               ryj = xij * sxy(j,j) + yij * syy(j,j) + zij * szy(j,j)
               rzj = xij * sxz(j,j) + yij * syz(j,j) + zij * szz(j,j)

!              //   preparation
               rxj = 2.d0 * rs * ( rxj - sx(j) )
               ryj = 2.d0 * rs * ( ryj - sy(j) )
               rzj = 2.d0 * rs * ( rzj - sz(j) )

!              //   part 5
               p5 = eps * p1 * bp
               p6 = eps * p1 * bn

!              //   force on atom i
               fxi = + p3 * ( bpn * rxi + 2.d0 * bn * xij ) &
     &               - p2 * dfij * xij * rijinv
               fyi = + p3 * ( bpn * ryi + 2.d0 * bn * yij ) &
     &               - p2 * dfij * yij * rijinv
               fzi = + p3 * ( bpn * rzi + 2.d0 * bn * zij ) &
     &               - p2 * dfij * zij * rijinv

!              //   add to forces
               fx(i,m) = fx(i,m) + fxi
               fy(i,m) = fy(i,m) + fyi
               fz(i,m) = fz(i,m) + fzi

!              //   add contribution to virial
               vir(1,1) = vir(1,1) + fxi*xij
               vir(1,2) = vir(1,2) + fxi*yij
               vir(1,3) = vir(1,3) + fxi*zij
               vir(2,1) = vir(2,1) + fyi*xij
               vir(2,2) = vir(2,2) + fyi*yij
               vir(2,3) = vir(2,3) + fyi*zij
               vir(3,1) = vir(3,1) + fzi*xij
               vir(3,2) = vir(3,2) + fzi*yij
               vir(3,3) = vir(3,3) + fzi*zij

!              //   force on atom j
               fxj = - cx * p1 * fij &
     &               + p3 * ( bpn * rxj - 2.d0 * bn * xij ) &
     &               + p3 * ( abx * rp2 + bx * rij2 ) &
     &               + p2 * dfij * xij * rijinv
               fyj = - cy * p1 * fij &
     &               + p3 * ( bpn * ryj - 2.d0 * bn * yij ) &
     &               + p3 * ( aby * rp2 + by * rij2 ) &
     &               + p2 * dfij * yij * rijinv
               fzj = - cz * p1 * fij &
     &               + p3 * ( bpn * rzj - 2.d0 * bn * zij ) &
     &               + p3 * ( abz * rp2 + bz * rij2 ) &
     &               + p2 * dfij * zij * rijinv

!              //   add to forces
               fx(j,m) = fx(j,m) + fxj
               fy(j,m) = fy(j,m) + fyj
               fz(j,m) = fz(j,m) + fzj

!              //   loop of metal atoms
               do k = 1, natom

!                 //   for metal only
                  if ( ( k .eq. j ) .or. ( .not. is_metal(k) ) ) cycle

!                 //   distance squared
                  xjk = x(j,m) - x(k,m)
                  yjk = y(j,m) - y(k,m)
                  zjk = z(j,m) - z(k,m)

!                 //   apply periodic boundary if needed
                  call pbc_atom( xjk, yjk, zjk )

!                 //   distance squared
                  rjk2 = xjk*xjk + yjk*yjk + zjk*zjk

!                 //   skip if distance longer than twice outer cutoff
                  if ( ( rjk2 .gt. rout_sf*rout_sf ) .and. &
    &                  ( rjk2 .gt. 4.d0*rout_cn*rout_cn ) ) cycle

!                 //   GCN corrected bp
                  ax = gcnx(j,k) * param_bp(1,j)
                  ay = gcny(j,k) * param_bp(1,j)
                  az = gcnz(j,k) * param_bp(1,j)

!                 //   GCN corrected bn
                  bx = gcnx(j,k) * param_bn(1,j)
                  by = gcny(j,k) * param_bn(1,j)
                  bz = gcnz(j,k) * param_bn(1,j)

!                 //   GCN corrected bp
                  abx = ax - bx
                  aby = ay - by
                  abz = az - bz

!                 //   GCN corrected epsilon
                  cx = gcnx(j,k) * param_eps(1,j) &
    &                + 2.d0 * gcnx(j,k) * gcn(j) * param_eps(2,j)
                  cy = gcny(j,k) * param_eps(1,j) &
    &                + 2.d0 * gcny(j,k) * gcn(j) * param_eps(2,j)
                  cz = gcnz(j,k) * param_eps(1,j) &
    &                + 2.d0 * gcnz(j,k) * gcn(j) * param_eps(2,j)

!                 //   preparation
                  rxk = xij * sxx(j,k) + yij * syx(j,k) + zij * szx(j,k)
                  ryk = xij * sxy(j,k) + yij * syy(j,k) + zij * szy(j,k)
                  rzk = xij * sxz(j,k) + yij * syz(j,k) + zij * szz(j,k)

!                 //   preparation
                  rxk = 2.d0 * rs * rxk
                  ryk = 2.d0 * rs * ryk
                  rzk = 2.d0 * rs * rzk

                  fxk = - cx * p1 * fij &
     &                  + p3 * bpn * rxk &
     &                  + p3 * ( abx * rp2 + bx * rij2 )

                  fyk = - cy * p1 * fij &
     &                  + p3 * bpn * ryk &
     &                  + p3 * ( aby * rp2 + by * rij2 )

                  fzk = - cz * p1 * fij &
     &                  + p3 * bpn * rzk &
     &                  + p3 * ( abz * rp2 + bz * rij2 )

!                 //   add contribution to force
                  fx(k,m) = fx(k,m) + fxk
                  fy(k,m) = fy(k,m) + fyk
                  fz(k,m) = fz(k,m) + fzk

!                 //   add contribution to virial
                  vir(1,1) = vir(1,1) - fxk*xjk
                  vir(1,2) = vir(1,2) - fxk*yjk
                  vir(1,3) = vir(1,3) - fxk*zjk
                  vir(2,1) = vir(2,1) - fyk*xjk
                  vir(2,2) = vir(2,2) - fyk*yjk
                  vir(2,3) = vir(2,3) - fyk*zjk
                  vir(3,1) = vir(3,1) - fzk*xjk
                  vir(3,2) = vir(3,2) - fzk*yjk
                  vir(3,3) = vir(3,3) - fzk*zjk

!              //   loop of metal atoms
               end do

!           //   loop of metal atoms
            end do

!        //   loop of oxygen atoms
         end do

!-----------------------------------------------------------------------
!        //   angular term
!-----------------------------------------------------------------------

!        //   loop of oxygen atoms
         do i = 1, natom

!           //   for oxygen only
            if ( .not. is_oxygen(i) ) cycle

!-----------------------------------------------------------------------
!           //   denominator of pairwise prefactor
!-----------------------------------------------------------------------

!           //   exponential sum
            a  = 0.d0

!           //   derivative of exponential sum
            ax = 0.d0
            ay = 0.d0
            az = 0.d0

!           //   loop of metal atoms
            do l = 1, natom

!              //   for metal only
               if ( .not. is_metal(l) ) cycle

!              //   distance squared
               xil = x(i,m) - x(l,m)
               yil = y(i,m) - y(l,m)
               zil = z(i,m) - z(l,m)

!              //   apply periodic boundary if needed
               call pbc_atom( xil, yil, zil )

!              //   distance squared
               ril2 = xil*xil + yil*yil + zil*zil

!              //   skip if distance longer than outer cutoff
               if ( ril2 .gt. rout_mo*rout_mo ) cycle

!              //   distance
               ril = sqrt( ril2 )

!              //   damping function
               call getswf ( ril, rin_mo, rout_mo, fil, dfil )

!              //   inverse of distance
               rilinv = 1.d0 / ril

!              //   inverse of distance parameter
               rinv_mo = 1.d0 / param_r_O(0,l)

!              //   exponential factor
               eil = exp( - ril * rinv_mo )

!              //   exponential sum
               a  = a  + eil * fil

!              //   i-derivative of (eil*fil)
               bx = xil * rilinv * eil * ( dfil - rinv_mo * fil )
               by = yil * rilinv * eil * ( dfil - rinv_mo * fil )
               bz = zil * rilinv * eil * ( dfil - rinv_mo * fil )

!              //   i-derivative of sum_l (eil*fil)
               ax = ax + bx
               ay = ay + by
               az = az + bz

!           //   loop of metal atoms
            end do

!           //   return on error
            if ( a .eq. 0.d0 ) cycle

!           //   inverse of exponential sum
            ainv  = 1.d0 / a

!-----------------------------------------------------------------------
!           //   water dipole vector
!-----------------------------------------------------------------------

!           //   bonded hydrogen atoms
            j = j_bond(i)
            k = k_bond(i)

!           //   distance
            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom( xij, yij, zij )

!           //   distance
            xik = x(i,m) - x(k,m)
            yik = y(i,m) - y(k,m)
            zik = z(i,m) - z(k,m)

            call pbc_atom( xik, yik, zik )

!           //   vector of water dipole
            dx = - ( xij + xik )
            dy = - ( yij + yik )
            dz = - ( zij + zik )

!           //   vector norm squared
            d2 = dx*dx + dy*dy + dz*dz

!           //   vector norm
            d  = sqrt( d2 )

!           //   preparation
            dinv = 1.d0 / d
            d3inv = dinv * dinv * dinv

!-----------------------------------------------------------------------
!           //   pairwise prefactor: i, j, k, l
!-----------------------------------------------------------------------

!           //   loop of metal atoms
            do l = 1, natom

!              //   for metal only
               if ( .not. is_metal(l) ) cycle

!              //   distance squared
               xil = x(i,m) - x(l,m)
               yil = y(i,m) - y(l,m)
               zil = z(i,m) - z(l,m)

!              //   apply periodic boundary if needed
               call pbc_atom( xil, yil, zil )

!              //   distance squared
               ril2 = xil*xil + yil*yil + zil*zil

!              //   inner product
               s2 = sx(l)*sx(l) + sy(l)*sy(l) + sz(l)*sz(l)

!              //   skip if distance longer than outer cutoff
               if ( ( ril2 .gt. rout_mo*rout_mo ) .or. &
     &              ( s2 .lt. tiny ) ) cycle

!              //   skip if distance longer than outer cutoff
               if ( ril2 .gt. rout_mo*rout_mo ) cycle

!              //   distance
               ril = sqrt( ril2 )

!              //   damping function
               call getswf ( ril, rin_mo, rout_mo, fil, dfil )

!              //   inverse of distance
               rilinv = 1.d0 / ril

!              //   inverse of distance parameter
               rinv_mo = 1.d0 / param_r_O(0,l)

!              //   exponential factor
               eil  = exp( - rinv_mo * ril )

!              //   preparation
               eil2 = eil * eil
               fil2 = fil * fil

!              //   energy prefactor
               ef = eil2 * fil2 * ainv

!              //   preparation
               p1 = 2.d0 * fil * rilinv * eil2 * ainv &
      &                  * ( fil * rinv_mo - dfil )

!              //   preparation
               p2 = ef * ainv

!              //   i-derivative of (eil*fil)
               bx = xil * rilinv * eil * ( dfil - rinv_mo * fil )
               by = yil * rilinv * eil * ( dfil - rinv_mo * fil )
               bz = zil * rilinv * eil * ( dfil - rinv_mo * fil )

!              //   gradient
               gxi = - p1 * xil - p2 * ax
               gyi = - p1 * yil - p2 * ay
               gzi = - p1 * zil - p2 * az

!              //   gradient
               gxl = + p1 * xil + p2 * bx
               gyl = + p1 * yil + p2 * by
               gzl = + p1 * zil + p2 * bz

!-----------------------------------------------------------------------
!              //   angular dependent factor: i, j, k, l
!-----------------------------------------------------------------------

!              //   parameter weighted by GCN
               a1 = param_a1(0,l) + gcn(l) * param_a1(1,l) &
     &            + gcn(l) * gcn(l) * param_a1(2,l)
               a2 = param_a2(0,l) + gcn(l) * param_a2(1,l) &
     &            + gcn(l) * gcn(l) * param_a2(2,l)
               a3 = param_a3(0,l) + gcn(l) * param_a3(1,l) &
     &            + gcn(l) * gcn(l) * param_a3(2,l)
               a4 = param_a4(0,l) + gcn(l) * param_a4(1,l) &
     &            + gcn(l) * gcn(l) * param_a4(2,l)

!              //   GCN corrected bp
               a1x = gcnx(l,l) * param_a1(1,l) &
     &             + 2.d0 * gcnx(l,l) * gcn(l) * param_a1(2,l)
               a1y = gcny(l,l) * param_a1(1,l) &
     &             + 2.d0 * gcny(l,l) * gcn(l) * param_a1(2,l)
               a1z = gcnz(l,l) * param_a1(1,l) &
     &             + 2.d0 * gcnz(l,l) * gcn(l) * param_a1(2,l)
               a2x = gcnx(l,l) * param_a2(1,l) &
     &             + 2.d0 * gcnx(l,l) * gcn(l) * param_a2(2,l)
               a2y = gcny(l,l) * param_a2(1,l) &
     &             + 2.d0 * gcny(l,l) * gcn(l) * param_a2(2,l)
               a2z = gcnz(l,l) * param_a2(1,l) &
     &             + 2.d0 * gcnz(l,l) * gcn(l) * param_a2(2,l)
               a3x = gcnx(l,l) * param_a3(1,l) &
     &             + 2.d0 * gcnx(l,l) * gcn(l) * param_a3(2,l)
               a3y = gcny(l,l) * param_a3(1,l) &
     &             + 2.d0 * gcny(l,l) * gcn(l) * param_a3(2,l)
               a3z = gcnz(l,l) * param_a3(1,l) &
     &             + 2.d0 * gcnz(l,l) * gcn(l) * param_a3(2,l)
               a4x = gcnx(l,l) * param_a4(1,l) &
     &             + 2.d0 * gcnx(l,l) * gcn(l) * param_a4(2,l)
               a4y = gcny(l,l) * param_a4(1,l) &
     &             + 2.d0 * gcny(l,l) * gcn(l) * param_a4(2,l)
               a4z = gcnz(l,l) * param_a4(1,l) &
     &             + 2.d0 * gcnz(l,l) * gcn(l) * param_a4(2,l)

!              //   inner product
               ds = dx*sx(l) + dy*sy(l) + dz*sz(l)

!              //   cos(theta)
               c1 = ds * dinv

!              //   cos(n*theta)
               c2 = 2.d0*c1*c1       - 1.d0
               c3 = 4.d0*c1*c1*c1    - 3.d0*c1
               c4 = 8.d0*c1*c1*c1*c1 - 8.d0*c1*c1 + 1.d0

!              //   linear combination of cosines
               ac = a1*c1 + a2*c2 + a3*c3 + a4*c4

!              //   linear combination of cosines
               dac = a1 + a2 * ( 4.d0*c1 ) &
     &             + a3 * ( 12.d0*c1*c1 - 3.d0 ) &
     &             + a4 * ( 32.d0*c1*c1*c1 - 16.d0*c1 )

!              //   derivatives of cosines
               cxi = - 2.d0 * sx(l) * dinv + 2.d0 * ds * dx * d3inv
               cyi = - 2.d0 * sy(l) * dinv + 2.d0 * ds * dy * d3inv
               czi = - 2.d0 * sz(l) * dinv + 2.d0 * ds * dz * d3inv

!              //   derivatives of cosines
               cxj = + sx(l) * dinv - ds * dx * d3inv
               cyj = + sy(l) * dinv - ds * dy * d3inv
               czj = + sz(l) * dinv - ds * dz * d3inv

!              //   derivatives of cosines
               cxk = + sx(l) * dinv - ds * dx * d3inv
               cyk = + sy(l) * dinv - ds * dy * d3inv
               czk = + sz(l) * dinv - ds * dz * d3inv

!              //   derivatives of cosines
               cxl = (dx*sxx(l,l) + dy*syx(l,l) + dz*szx(l,l)) * dinv
               cyl = (dx*sxy(l,l) + dy*syy(l,l) + dz*szy(l,l)) * dinv
               czl = (dx*sxz(l,l) + dy*syz(l,l) + dz*szz(l,l)) * dinv

!              //   gradient
               hxi = + cxi * dac
               hyi = + cyi * dac
               hzi = + czi * dac

!              //   gradient
               hxj = + cxj * dac
               hyj = + cyj * dac
               hzj = + czj * dac

!              //   gradient
               hxk = + cxk * dac
               hyk = + cyk * dac
               hzk = + czk * dac

!              //   gradient
               hxl = + cxl * dac + a1x*c1 + a2x*c2 + a3x*c3 + a4x*c4
               hyl = + cyl * dac + a1y*c1 + a2y*c2 + a3y*c3 + a4y*c4
               hzl = + czl * dac + a1z*c1 + a2z*c2 + a3z*c3 + a4z*c4

!-----------------------------------------------------------------------
!              //   energy: all
!-----------------------------------------------------------------------

!              //   two factors
               g = ef
               h = ac

!              //   add to energy
               pot(m) = pot(m) + g * h

!-----------------------------------------------------------------------
!              //   atomic forces
!-----------------------------------------------------------------------

               fxi = - gxi * h - g * hxi
               fyi = - gyi * h - g * hyi
               fzi = - gzi * h - g * hzi

               fxj = - g * hxj
               fyj = - g * hyj
               fzj = - g * hzj

               fxk = - g * hxk
               fyk = - g * hyk
               fzk = - g * hzk

               fxl = - gxl * h - g * hxl
               fyl = - gyl * h - g * hyl
               fzl = - gzl * h - g * hzl

!              //   atom i
               fx(i,m) = fx(i,m) + fxi
               fy(i,m) = fy(i,m) + fyi
               fz(i,m) = fz(i,m) + fzi

!              //   atom j
               fx(j,m) = fx(j,m) + fxj
               fy(j,m) = fy(j,m) + fyj
               fz(j,m) = fz(j,m) + fzj

!              //   atom k
               fx(k,m) = fx(k,m) + fxk
               fy(k,m) = fy(k,m) + fyk
               fz(k,m) = fz(k,m) + fzk

!              //   atom l
               fx(l,m) = fx(l,m) + fxl
               fy(l,m) = fy(l,m) + fyl
               fz(l,m) = fz(l,m) + fzl

!              //   add contribution to virial
               vir(1,1) = vir(1,1) - fxj*xij - fxk*xik - fxl*xil
               vir(1,2) = vir(1,2) - fxj*yij - fxk*yik - fxl*yil
               vir(1,3) = vir(1,3) - fxj*zij - fxk*zik - fxl*zil
               vir(2,1) = vir(2,1) - fyj*xij - fyk*xik - fyl*xil
               vir(2,2) = vir(2,2) - fyj*yij - fyk*yik - fyl*yil
               vir(2,3) = vir(2,3) - fyj*zij - fyk*zik - fyl*zil
               vir(3,1) = vir(3,1) - fzj*xij - fzk*xik - fzl*xil
               vir(3,2) = vir(3,2) - fzj*yij - fzk*yik - fzl*yil
               vir(3,3) = vir(3,3) - fzj*zij - fzk*zik - fzl*zil

!-----------------------------------------------------------------------
!              //   forces on n
!-----------------------------------------------------------------------

!              //   loop of metal atoms
               do n = 1, natom

!                 //   for metal only
                  if ( ( n .eq. l ) .or. ( .not. is_metal(n) ) ) cycle

!                 //   distance squared
                  xln = x(l,m) - x(n,m)
                  yln = y(l,m) - y(n,m)
                  zln = z(l,m) - z(n,m)

!                 //   apply periodic boundary if needed
                  call pbc_atom( xln, yln, zln )

!                 //   distance squared
                  rln2 = xln*xln + yln*yln + zln*zln

!                 //   distance squared
                  xin = x(i,m) - x(n,m)
                  yin = y(i,m) - y(n,m)
                  zin = z(i,m) - z(n,m)

!                 //   apply periodic boundary if needed
                  call pbc_atom( xin, yin, zin )

!                 //   distance squared
                  rin2 = xin*xin + yin*yin + zin*zin

!                 //   skip if distance longer than outer cutoff
                  if ( ( rln2 .gt. rout_sf*rout_sf ) .and. &
     &                 ( rin2 .gt. rout_mo*rout_mo ) ) cycle

!-----------------------------------------------------------------------
!                 //   pairwise prefactor: n
!-----------------------------------------------------------------------

!                 //   distance
                  rin = sqrt( rin2 )

!                 //   damping function
                  call getswf ( rin, rin_mo, rout_mo, fin, dfin )

!                 //   inverse of distance
                  rininv = 1.d0 / rin

!                 //   inverse of distance parameter
                  rinv_mo = 1.d0 / param_r_O(0,l)

!                 //   exponential factor
                  ein = exp( - rinv_mo * rin )

!                 //   n-derivative of (ein*fin)
                  bx = xin * rininv * ein * ( dfin - rinv_mo * fin )
                  by = yin * rininv * ein * ( dfin - rinv_mo * fin )
                  bz = zin * rininv * ein * ( dfin - rinv_mo * fin )

!                 //   preparation
                  p2 = ef * ainv

!                 //   gradient
                  gxn = + p2 * bx
                  gyn = + p2 * by
                  gzn = + p2 * bz

!-----------------------------------------------------------------------
!                 //   angular dependent factor: n
!-----------------------------------------------------------------------

!                 //   GCN corrected bp
                  a1x = gcnx(l,n) * param_a1(1,l) &
     &                + 2.d0 * gcnx(l,n) * gcn(l) * param_a1(2,l)
                  a1y = gcny(l,n) * param_a1(1,l) &
     &                + 2.d0 * gcny(l,n) * gcn(l) * param_a1(2,l)
                  a1z = gcnz(l,n) * param_a1(1,l) &
     &                + 2.d0 * gcnz(l,n) * gcn(l) * param_a1(2,l)
                  a2x = gcnx(l,n) * param_a2(1,l) &
     &                + 2.d0 * gcnx(l,n) * gcn(l) * param_a2(2,l)
                  a2y = gcny(l,n) * param_a2(1,l) &
     &                + 2.d0 * gcny(l,n) * gcn(l) * param_a2(2,l)
                  a2z = gcnz(l,n) * param_a2(1,l) &
     &                + 2.d0 * gcnz(l,n) * gcn(l) * param_a2(2,l)
                  a3x = gcnx(l,n) * param_a3(1,l) &
     &                + 2.d0 * gcnx(l,n) * gcn(l) * param_a3(2,l)
                  a3y = gcny(l,n) * param_a3(1,l) &
     &                + 2.d0 * gcny(l,n) * gcn(l) * param_a3(2,l)
                  a3z = gcnz(l,n) * param_a3(1,l) &
     &                + 2.d0 * gcnz(l,n) * gcn(l) * param_a3(2,l)
                  a4x = gcnx(l,n) * param_a4(1,l) &
     &                + 2.d0 * gcnx(l,n) * gcn(l) * param_a4(2,l)
                  a4y = gcny(l,n) * param_a4(1,l) &
     &                + 2.d0 * gcny(l,n) * gcn(l) * param_a4(2,l)
                  a4z = gcnz(l,n) * param_a4(1,l) &
     &                + 2.d0 * gcnz(l,n) * gcn(l) * param_a4(2,l)

!                 //   derivatives of cosines
                  cxn = (dx*sxx(l,n) + dy*syx(l,n) + dz*szx(l,n)) * dinv
                  cyn = (dx*sxy(l,n) + dy*syy(l,n) + dz*szy(l,n)) * dinv
                  czn = (dx*sxz(l,n) + dy*syz(l,n) + dz*szz(l,n)) * dinv

!                 //   gradient
                  hxn = + cxn * dac + a1x*c1 + a2x*c2 + a3x*c3 + a4x*c4
                  hyn = + cyn * dac + a1y*c1 + a2y*c2 + a3y*c3 + a4y*c4
                  hzn = + czn * dac + a1z*c1 + a2z*c2 + a3z*c3 + a4z*c4

!-----------------------------------------------------------------------
!                 //   atomic forces
!-----------------------------------------------------------------------

!                 //   force on n
                  fxn = - gxn * h - g * hxn
                  fyn = - gyn * h - g * hyn
                  fzn = - gzn * h - g * hzn

!                 //   add contribution to forces
                  fx(n,m) = fx(n,m) + fxn
                  fy(n,m) = fy(n,m) + fyn
                  fz(n,m) = fz(n,m) + fzn

!                 //   add contribution to virial
                  vir(1,1) = vir(1,1) - fxn*xin
                  vir(1,2) = vir(1,2) - fxn*yin
                  vir(1,3) = vir(1,3) - fxn*zin
                  vir(2,1) = vir(2,1) - fyn*xin
                  vir(2,2) = vir(2,2) - fyn*yin
                  vir(2,3) = vir(2,3) - fyn*zin
                  vir(3,1) = vir(3,1) - fzn*xin
                  vir(3,2) = vir(3,2) - fzn*yin
                  vir(3,3) = vir(3,3) - fzn*zin

!              //   loop of metal atoms
               end do

!           //   loop of metal atoms
            end do

!        //   loop of oxygen atoms
         end do

!     //   loop of beads
      end do

!     //   net GAL potential
      potential_gal = 0.d0
      do m = 1, nbead
         potential_gal = potential_gal - pot(m)
      end do

      return
      end





!***********************************************************************
      subroutine force_metalwater_tip4p
!***********************************************************************
!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   integers
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     //   initialize
!-----------------------------------------------------------------------

!     /*   for initial access   */
      if ( iset .eq. 0 ) then

!        /*   identify atoms   */
         call force_metalwater_common_setup

!        /*   prepare tip4p   */
         call force_metalwater_tip4p_setup

!        /*   set complete   */
         iset = 1

!     /*   for initial access   */
      end if

!-----------------------------------------------------------------------
!     //   run tip4p calculation
!-----------------------------------------------------------------------

      call force_mm_main_tip4p

      return
      end





!***********************************************************************
      subroutine force_metalwater_tip4p_setup
!***********************************************************************
!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   natom, iounit, iboundary

      use mm_variables, only : &
     &   i_genlin, j_genlin, n_genlin, eq_genlin, fc_genlin, i_angl, &
     &   j_angl, k_angl, eq_angl, fc_angl, i_lj, j_lj, eps_lj, sig_lj, &
     &   rin_lj, rout_lj, q, i_q, i_bcp, j_bcp, factor_bcp, &
     &   nlin, ngenlin, nangl, ndih, nimproper, ncmap, nlj, nbuck, &
     &   ncharge, nbcp, nmorse, ewald_flag

      use metalwater_variables, only : &
     &   j_bond, k_bond, nwater, is_oxygen, is_hydrogen

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   integers
      integer :: i, j, k, l, m, n

!     //   real numbers
      real(8) :: eq_genlin_qtip4p   =  0.17799329E+01
      real(8) :: fc2_genlin_qtip4p  =  0.27096233E+00
      real(8) :: fc3_genlin_qtip4p  = -3.27926318E-01
      real(8) :: fc4_genlin_qtip4p  =  2.31505033E-01
      real(8) :: eq_angl_qtip4p     =  0.10740000E+03
      real(8) :: fc_angl_qtip4p     =  0.42645755E-04
      real(8) :: qO_qtip4p          = -0.11128000E+01
      real(8) :: qH_qtip4p          =  0.55640000E+00
      real(8) :: eps_lj_qtip4p      =  0.29513472E-03
      real(8) :: sig_lj_qtip4p      =  0.59694554E+01
      real(8) :: rin_lj_qtip4p      =  0.15390000E+02
      real(8) :: rout_lj_qtip4p     =  0.18225000E+02
      real(8) :: gamma_qtip4p       =  0.73612000E+00

!-----------------------------------------------------------------------
!     //   memory allocation
!-----------------------------------------------------------------------

      nlin      = 0

      ngenlin   = nwater*6

      if ( .not. allocated(i_genlin) ) allocate( i_genlin(ngenlin) )
      if ( .not. allocated(j_genlin) ) allocate( j_genlin(ngenlin) )
      if ( .not. allocated(n_genlin) ) allocate( n_genlin(ngenlin) )
      if ( .not. allocated(eq_genlin) ) allocate( eq_genlin(ngenlin) )
      if ( .not. allocated(fc_genlin) ) allocate( fc_genlin(ngenlin) )

      nangl     = nwater

      if ( .not. allocated(i_angl) ) allocate( i_angl(nangl) )
      if ( .not. allocated(j_angl) ) allocate( j_angl(nangl) )
      if ( .not. allocated(k_angl) ) allocate( k_angl(nangl) )
      if ( .not. allocated(eq_angl) ) allocate( eq_angl(nangl) )
      if ( .not. allocated(fc_angl) ) allocate( fc_angl(nangl) )

      ndih      = 0

      nimproper = 0
      ncmap     = 0

      nlj       = nwater*(nwater-1)/2

      if ( .not. allocated(i_lj) ) allocate( i_lj(nlj) )
      if ( .not. allocated(j_lj) ) allocate( j_lj(nlj) )
      if ( .not. allocated(eps_lj) ) allocate( eps_lj(nlj) )
      if ( .not. allocated(sig_lj) ) allocate( sig_lj(nlj) )

      nbuck     = 0

      ncharge   = 3*nwater

      if ( .not. allocated(q) ) allocate( q(natom) )
      if ( .not. allocated(i_q) ) allocate( i_q(ncharge) )

      nbcp      = 3*nwater

      if ( .not. allocated(i_bcp) ) allocate( i_bcp(nbcp))
      if ( .not. allocated(j_bcp) ) allocate( j_bcp(nbcp) )
      if ( .not. allocated(factor_bcp) ) allocate( factor_bcp(nbcp) )

      nmorse    = 0

!-----------------------------------------------------------------------
!     //   water angles, charges
!-----------------------------------------------------------------------

      q(:) = 0.d0

      j = 0

      do i = 1, natom

         if ( is_oxygen(i) ) then

            j = j + 1
            i_q(j) = i
            q(i) = qO_qtip4p

         end if

         if ( is_hydrogen(i) ) then

            j = j + 1
            i_q(j) = i
            q(i) = qH_qtip4p

         end if

      end do

      k = 0
      l = 0
      m = 0
      n = 0

      do i = 1, natom

         if ( .not. is_oxygen(i) ) cycle

         k = k + 1

         i_genlin(k)    = i
         j_genlin(k)    = j_bond(i)
         n_genlin(k)    = 2
         eq_genlin(k)   = eq_genlin_qtip4p
         fc_genlin(k)   = fc2_genlin_qtip4p

         k = k + 1

         i_genlin(k)    = i
         j_genlin(k)    = j_bond(i)
         n_genlin(k)    = 3
         eq_genlin(k)   = eq_genlin_qtip4p
         fc_genlin(k)   = fc3_genlin_qtip4p

         k = k + 1

         i_genlin(k)    = i
         j_genlin(k)    = j_bond(i)
         n_genlin(k)    = 4
         eq_genlin(k)   = eq_genlin_qtip4p
         fc_genlin(k)   = fc4_genlin_qtip4p

         k = k + 1

         i_genlin(k)    = i
         j_genlin(k)    = k_bond(i)
         n_genlin(k)    = 2
         eq_genlin(k)   = eq_genlin_qtip4p
         fc_genlin(k)   = fc2_genlin_qtip4p

         k = k + 1

         i_genlin(k)    = i
         j_genlin(k)    = k_bond(i)
         n_genlin(k)    = 3
         eq_genlin(k)   = eq_genlin_qtip4p
         fc_genlin(k)   = fc3_genlin_qtip4p

         k = k + 1

         i_genlin(k)    = i
         j_genlin(k)    = k_bond(i)
         n_genlin(k)    = 4
         eq_genlin(k)   = eq_genlin_qtip4p
         fc_genlin(k)   = fc4_genlin_qtip4p

         l = l + 1

         i_angl(l)      = j_bond(i)
         j_angl(l)      = i
         k_angl(l)      = k_bond(i)
         eq_angl(l)     = eq_angl_qtip4p
         fc_angl(l)     = fc_angl_qtip4p

         m = m + 1

         i_bcp(m)       = i
         j_bcp(m)       = j_bond(i)
         factor_bcp(m)  = 0.d0

         m = m + 1

         i_bcp(m)       = i
         j_bcp(m)       = k_bond(i)
         factor_bcp(m)  = 0.d0

         m = m + 1

         i_bcp(m)       = j_bond(i)
         j_bcp(m)       = k_bond(i)
         factor_bcp(m)  = 0.d0

         do j = i+1, natom

            if ( .not. is_oxygen(j) ) cycle

            n = n + 1

            i_lj(n)     = i
            j_lj(n)     = j
            eps_lj(n)   = eps_lj_qtip4p
            sig_lj(n)   = sig_lj_qtip4p

         end do

      end do

      rin_lj  = rin_lj_qtip4p
      rout_lj = rout_lj_qtip4p

!-----------------------------------------------------------------------
!     //   tip4p parameters
!-----------------------------------------------------------------------

      open ( iounit, file = 'mm.dat' )

      write( iounit, '(a)' ) '<genlin_bonds>'
      write( iounit, '(i8)' ) ngenlin
      do i = 1, ngenlin
         write( iounit, '(3i8,2e24.16)' ) i_genlin(i), j_genlin(i), &
     &       n_genlin(i), eq_genlin(i), fc_genlin(i)
      end do
      write( iounit, '(a)' )

      write( iounit, '(a)' ) '<angular_bonds>'
      write( iounit, '(i8)' ) nangl
      do i = 1, nangl
         write( iounit, '(3i8,2e24.16)' ) i_angl(i), j_angl(i), &
     &       k_angl(i), eq_angl(i), fc_angl(i)
      end do
      write( iounit, '(a)' )

      write( iounit, '(a)' ) '<dihedral_bonds>'
      write( iounit, '(i8)' ) ndih
      write( iounit, '(a)' )

      write( iounit, '(a)' ) '<improper_bonds>'
      write( iounit, '(i8)' ) nimproper
      write( iounit, '(a)' )

      write( iounit, '(a)' ) '<charges>'
      write( iounit, '(i8)' ) ncharge
      do i = 1, ncharge
         write( iounit, '(i8,e24.16)' ) i_q(i), q(i_q(i))
      end do
      write( iounit, '(a)' )

      write( iounit, '(a)' ) '<nbcp>'
      write( iounit, '(i8)' ) nbcp
      do i = 1, nbcp
         write( iounit, '(2i8,e24.16)' ) &
     &      i_bcp(i), j_bcp(i), factor_bcp(i)
      end do
      write( iounit, '(a)' )

      write( iounit, '(a)' ) '<lennard-jones>'
      write( iounit, '(i8)' ) nlj
      write( iounit, '(2e24.16)' ) rin_lj, rout_lj
      do i = 1, nlj
         write( iounit, '(2i8,2e24.16)' ) &
     &      i_lj(i), j_lj(i), eps_lj(i), sig_lj(i)
      end do
      write( iounit, '(a)' )

      write( iounit, '(a)' ) '<tip4p>'
      write( iounit, '(a,e24.16)' ) 'O H ', gamma_qtip4p

      close( iounit )

!-----------------------------------------------------------------------
!     /*   free     boundary  =  direct sum                           */
!     /*   periodic boundary  =  Ewald  sum                           */
!-----------------------------------------------------------------------

!     /*   free boundary   */
      if ( iboundary .eq. 0 ) then

!        /*   direct sum   */
         call force_mm_coulomb_setup

!     /*   periodic boundary   */
      else if ( iboundary .eq. 1 ) then

!        /*   Ewald sum   */
         call force_ewald_setup

!        /*   particle mesh Ewald   */
         if ( ewald_flag .eq. 1 ) call force_pmeewald_setup

!     /*   periodic boundary   */
      else if ( iboundary .eq. 2 ) then

!        /*   Ewald sum   */
         call force_ewald_setup

!        /*   particle mesh Ewald   */
         if ( ewald_flag .eq. 1 ) call force_pmeewald_setup

!     /*   end boundary condition   */
      end if

      return
      end





!***********************************************************************
      subroutine force_metalwater_eam
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, vir, pot, au_length, au_charge, volume, &
     &   box, au_energy, natom, nbead, iounit, iboundary

      use mm_variables, only : &
     &   srho_eam, dfdrho_eam, rcut_eam, bigbox, bigboxinv, rcut_eam2, &
     &   neam, nbox_eam, ikind_eam

      use metalwater_variables, only : &
     &   is_metal

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: m, i, j, k, l, jx, jy, jz, j2

      real(8) :: xij, yij, zij, rij, rinv, srho, phir_eam, &
     &           rhor_eam, rhor_grad_eam, frho_eam, frho_grad_eam, &
     &           phir_grad_eam, fxi, fyi, fzi, ax, ay, az, &
     &           drhoirdr, drhojrdr, dphirdr, dfdrhoi, dfdrhoj, rij2, &
     &           absa, absb, absc, aij, bij, cij, bx, by, bz, cx, cy, cz

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   for initial access   */
      if ( iset .eq. 0 ) then

!        /*   identify atoms   */
         call force_metalwater_common_setup

!        /*   read eam files   */
         call force_metalwater_eam_setup

!        /*   set complete   */
         iset = 1

!     /*   for initial access   */
      end if

!     /*   return if no eam   */
      if ( neam .eq. 0 ) return

!     /*   cut off distance squared   */
      rcut_eam2 = rcut_eam*rcut_eam

!-----------------------------------------------------------------------
!     /*   extention of simulation box in real space sum              */
!-----------------------------------------------------------------------

!     /*   free boundary   */
      if ( iboundary .eq. 0 ) then

!        /*   number of replicated boxes   */
         nbox_eam(1) = 1
         nbox_eam(2) = 1
         nbox_eam(3) = 1

!     /*   periodic boundary   */
      else

!        /*   vector product of lattice vectors b, c   */
         ax = box(2,2)*box(3,3) - box(2,3)*box(3,2)
         ay = box(3,2)*box(1,3) - box(3,3)*box(1,2)
         az = box(1,2)*box(2,3) - box(1,3)*box(2,2)

!        /*   vector product of lattice vectors c, a   */
         bx = box(2,3)*box(3,1) - box(2,1)*box(3,3)
         by = box(3,3)*box(1,1) - box(3,1)*box(1,3)
         bz = box(1,3)*box(2,1) - box(1,1)*box(2,3)

!        /*   vector product of lattice vectors a, b   */
         cx = box(2,1)*box(3,2) - box(2,2)*box(3,1)
         cy = box(3,1)*box(1,2) - box(3,2)*box(1,1)
         cz = box(1,1)*box(2,2) - box(1,2)*box(2,1)

!        /*   distance between parallel planes   */
         absa = volume / sqrt( ax*ax + ay*ay + az*az )
         absb = volume / sqrt( bx*bx + by*by + bz*bz )
         absc = volume / sqrt( cx*cx + cy*cy + cz*cz )

!        /*   number of replicated boxes   */
         nbox_eam(1) = int(2.d0*rcut_eam/absa) + 1
         nbox_eam(2) = int(2.d0*rcut_eam/absb) + 1
         nbox_eam(3) = int(2.d0*rcut_eam/absc) + 1

!     /*   boundary condition   */
      end if

!-----------------------------------------------------------------------
!     /*   first loop                                                 */
!-----------------------------------------------------------------------

!     /*   free boundary or minimum image convention   */
      if ( nbox_eam(1)*nbox_eam(2)*nbox_eam(3) .eq. 1 ) then

!        /*   loop of beads   */
         do m = 1, nbead

!           /*   srho_eam = sum of electron density rho_eam   */
            srho_eam(:) = 0.d0

!           /*   loop of atom pairs   */
            do i = 1, natom

!              //   metal atom only
               if ( .not. is_metal(i) ) cycle

!           /*   loop of atom pairs   */
            do j = 1, natom

!              //   metal atom only
               if ( ( .not. is_metal(j) ) .or. ( j .eq. i ) ) cycle

!              /*   interatomic distance   */
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              /*   apply free or periodic boundary   */
               call pbc_atom ( xij, yij, zij )

!              /*   interatomic distance squared   */
               rij2 = xij*xij + yij*yij + zij*zij

!              /*   neglect beyond cutoff distance   */
               if ( rij2 .gt. rcut_eam2 ) cycle

!              /*   interatomic distance   */
               rij = sqrt( rij2 )

!              /*   ikind_eam = species number   */
               k = ikind_eam(i)
               l = ikind_eam(j)

!              /*   sum of electron density   */
               srho_eam(i) = srho_eam(i) + rhor_eam(rij,l)
               srho_eam(j) = srho_eam(j) + rhor_eam(rij,k)

!           /*   loop of atom pairs   */
            end do
            end do

!-----------------------------------------------------------------------
!           /*   second loop                                          */
!-----------------------------------------------------------------------

!           /*   loop of atoms   */
            do i = 1, natom

!              /*   initialize   */
               dfdrho_eam(i) = 0.d0

!              //   metal atom only
               if ( .not. is_metal(i) ) cycle

!              /*   sum of electron density   */
               srho = srho_eam(i)

!              /*   species number   */
               k    = ikind_eam(i)

!              /*   embedding potential   */
               pot(m)  =  pot(m) + frho_eam(srho,k)

!              /*   gradient of embedding potential   */
               dfdrho_eam(i) =  frho_grad_eam(srho,k)

!           /*   loop of atoms   */
            end do

!-----------------------------------------------------------------------
!           /*   third loop                                           */
!-----------------------------------------------------------------------

!           /*   loop of atom pairs   */
            do i = 1, natom

!              //   metal atom only
               if ( .not. is_metal(i) ) cycle

!           /*   loop of atom pairs   */
            do j = 1, natom

!              //   metal atom only
               if ( ( .not. is_metal(j) ) .or. ( j .eq. i ) ) cycle

!              /*   interatomic distance   */
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              /*   apply free or periodic boundary   */
               call pbc_atom ( xij, yij, zij )

!              /*   interatomic distance squared   */
               rij2 = xij*xij + yij*yij + zij*zij

!              /*   neglect beyond cutoff distance   */
               if ( rij2 .gt. rcut_eam2 ) cycle

!              /*   interatomic distance   */
               rij = sqrt( rij2 )

!              /*   inverse of interatomic distance   */
               rinv = 1.d0/rij

!              /*   species number   */
               k = ikind_eam(i)
               l = ikind_eam(j)

!              /*   electron density   */
               drhoirdr =  rhor_grad_eam(rij,k)
               drhojrdr =  rhor_grad_eam(rij,l)

!              /*   gradient of pair potential   */
               dphirdr  =  phir_grad_eam(rij,k,l)

!              /*   gradient of embedding potential   */
               dfdrhoi  =  dfdrho_eam(i)
               dfdrhoj  =  dfdrho_eam(j)

!-----------------------------------------------------------------------
!              /*   pair potential   */
!-----------------------------------------------------------------------

               pot(m)   =  pot(m) + phir_eam(rij,k,l)

!-----------------------------------------------------------------------
!              /*   forces   */
!-----------------------------------------------------------------------

               fxi = - dfdrhoi * drhojrdr * xij * rinv &
     &               - dfdrhoj * drhoirdr * xij * rinv &
     &               - dphirdr * xij * rinv

               fyi = - dfdrhoi * drhojrdr * yij * rinv &
     &               - dfdrhoj * drhoirdr * yij * rinv &
     &               - dphirdr * yij * rinv

               fzi = - dfdrhoi * drhojrdr * zij * rinv &
     &               - dfdrhoj * drhoirdr * zij * rinv &
     &               - dphirdr * zij * rinv

!-----------------------------------------------------------------------
!              /*   total force and virial   */
!-----------------------------------------------------------------------

               fx(i,m) = fx(i,m) + fxi
               fy(i,m) = fy(i,m) + fyi
               fz(i,m) = fz(i,m) + fzi

               fx(j,m) = fx(j,m) - fxi
               fy(j,m) = fy(j,m) - fyi
               fz(j,m) = fz(j,m) - fzi

               vir(1,1) = vir(1,1) + fxi*xij
               vir(1,2) = vir(1,2) + fxi*yij
               vir(1,3) = vir(1,3) + fxi*zij
               vir(2,1) = vir(2,1) + fyi*xij
               vir(2,2) = vir(2,2) + fyi*yij
               vir(2,3) = vir(2,3) + fyi*zij
               vir(3,1) = vir(3,1) + fzi*xij
               vir(3,2) = vir(3,2) + fzi*yij
               vir(3,3) = vir(3,3) + fzi*zij

!           /*   loop of atom pairs   */
            end do
            end do

!        /*   loop of beads   */
         end do

!-----------------------------------------------------------------------
!     /*   first loop                                                 */
!-----------------------------------------------------------------------

!     /*   periodic boundary   */
      else

!        /*   replicated boxes   */
         bigbox(:,1) = dble(nbox_eam(1))*box(:,1)
         bigbox(:,2) = dble(nbox_eam(2))*box(:,2)
         bigbox(:,3) = dble(nbox_eam(3))*box(:,3)

!        /*   inverse of box matrix   */
         call inv3 ( bigbox, bigboxinv )

!        /*   loop of beads   */
         do m = 1, nbead

!           /*   srho_eam = sum of electron density rho_eam   */
            srho_eam(:) = 0.d0

!           /*   loop of atom pairs   */
            do i = 1, natom

!              //   metal atom only
               if ( .not. is_metal(i) ) cycle

!           /*   loop of atom pairs   */
            do j = 1, natom

!              //   metal atom only
               if ( .not. is_metal(j) ) cycle

!           /*   loop of replicated boxes   */
            do jx = 0, nbox_eam(1)-1
            do jy = 0, nbox_eam(2)-1
            do jz = 0, nbox_eam(3)-1

!              /*   square of box index   */
               j2 = jx*jx + jy*jy + jz*jz

!              /*   skip same atom   */
               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

!              /*   interatomic distance of i and j in same box   */
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              /*   distance of i and j in different box  */
               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

!              /*   vector in big box   */
               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

!              /*   apply periodic boundary in big box   */
               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

!              /*   distance of nearest i and j   */
               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

!              /*   interatomic distance squared   */
               rij2 = xij*xij + yij*yij + zij*zij

!              /*   neglect beyond cutoff distance   */
               if ( rij2 .gt. rcut_eam2 ) cycle

!              /*   interatomic distance   */
               rij = sqrt( rij2 )

!              /*   ikind_eam = species number   */
               k = ikind_eam(i)
               l = ikind_eam(j)

!              /*   sum of electron density   */
               srho_eam(i) = srho_eam(i) + rhor_eam(rij,l)

!           /*   loop of replicated boxes   */
            end do
            end do
            end do

!           /*   loop of atom pairs   */
            end do
            end do

!-----------------------------------------------------------------------
!           /*   second loop                                          */
!-----------------------------------------------------------------------

!           /*   loop of atom pairs   */
            do i = 1, natom

!              /*   initialize   */
               dfdrho_eam(i) = 0.d0

!              //   metal atom only
               if ( .not. is_metal(i) ) cycle

!              /*   sum of electron density   */
               srho = srho_eam(i)

!              /*   species number   */
               k    = ikind_eam(i)

!              /*   embedding potential   */
               pot(m)  =  pot(m) + frho_eam(srho,k)

!              /*   gradient of embedding potential   */
               dfdrho_eam(i) =  frho_grad_eam(srho,k)

!           /*   loop of atoms   */
            end do

!-----------------------------------------------------------------------
!           /*   third loop                                           */
!-----------------------------------------------------------------------

!           /*   loop of atom pairs   */
            do i = 1, natom

!              //   metal atom only
               if ( .not. is_metal(i) ) cycle

!           /*   loop of atom pairs   */
            do j = 1, natom

!              //   metal atom only
               if ( .not. is_metal(j) ) cycle

!           /*   loop of replicated boxes   */
            do jx = 0, nbox_eam(1)-1
            do jy = 0, nbox_eam(2)-1
            do jz = 0, nbox_eam(3)-1

!              /*   square of box index   */
               j2 = jx*jx + jy*jy + jz*jz

!              /*   skip same atom   */
               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

!              /*   interatomic distance of i and j in same box   */
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              /*   distance of i and j in different box  */
               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

!              /*   vector in big box   */
               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

!              /*   apply periodic boundary in big box   */
               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

!              /*   distance of nearest i and j   */
               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

!              /*    interatomic distance squared   */
               rij2 = xij*xij + yij*yij + zij*zij

!              /*   neglect beyond cutoff distance   */
               if ( rij2 .gt. rcut_eam2 ) cycle

!              /*   interatomic distance   */
               rij = sqrt( rij2 )

!              /*   inverse of interatomic distance   */
               rinv = 1.d0/rij

!              /*   species number   */
               k = ikind_eam(i)
               l = ikind_eam(j)

!              /*   electron density   */
               drhoirdr =  rhor_grad_eam(rij,k)
               drhojrdr =  rhor_grad_eam(rij,l)

!              /*   gradient of pair potential   */
               dphirdr  =  phir_grad_eam(rij,k,l)

!              /*   gradient of embedding potential   */
               dfdrhoi  =  dfdrho_eam(i)
               dfdrhoj  =  dfdrho_eam(j)

!-----------------------------------------------------------------------
!              /*   pair potential   */
!-----------------------------------------------------------------------

               pot(m)   =  pot(m) + 0.5d0*phir_eam(rij,k,l)

!-----------------------------------------------------------------------
!              /*   forces   */
!-----------------------------------------------------------------------

               fxi = - dfdrhoi * drhojrdr * xij * rinv &
     &               - dfdrhoj * drhoirdr * xij * rinv &
     &               - dphirdr * xij * rinv

               fyi = - dfdrhoi * drhojrdr * yij * rinv &
     &               - dfdrhoj * drhoirdr * yij * rinv &
     &               - dphirdr * yij * rinv

               fzi = - dfdrhoi * drhojrdr * zij * rinv &
     &               - dfdrhoj * drhoirdr * zij * rinv &
     &               - dphirdr * zij * rinv

!-----------------------------------------------------------------------
!              /*   total force and virial   */
!-----------------------------------------------------------------------

               fx(i,m) = fx(i,m) + fxi
               fy(i,m) = fy(i,m) + fyi
               fz(i,m) = fz(i,m) + fzi

               vir(1,1) = vir(1,1) + 0.5d0*fxi*xij
               vir(1,2) = vir(1,2) + 0.5d0*fxi*yij
               vir(1,3) = vir(1,3) + 0.5d0*fxi*zij
               vir(2,1) = vir(2,1) + 0.5d0*fyi*xij
               vir(2,2) = vir(2,2) + 0.5d0*fyi*yij
               vir(2,3) = vir(2,3) + 0.5d0*fyi*zij
               vir(3,1) = vir(3,1) + 0.5d0*fzi*xij
               vir(3,2) = vir(3,2) + 0.5d0*fzi*yij
               vir(3,3) = vir(3,3) + 0.5d0*fzi*zij

!           /*   loop of replicated boxes   */
            end do
            end do
            end do

!           /*   loop of atom pairs   */
            end do
            end do

!        /*   loop of beads   */
         end do

!     /*   boundary condition   */
      end if

      return
      end





!***********************************************************************
      subroutine force_metalwater_eam_setup
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   au_length, au_charge, au_energy, ikind, nkind, natom, iounit

      use mm_variables, only : &
     &   xref_eam, yref_eam, y2ref_eam, srho_eam, dfdrho_eam, rcut_eam, &
     &   neam, nref_eam, ikind_eam, iphir_eam, irhor_eam, ifrho_eam

      use metalwater_variables, only : &
     &   nkind_metal

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, ierr, nrhor_eam, nfrho_eam, nphir_eam

!-----------------------------------------------------------------------
!     /*   number of tables                                           */
!-----------------------------------------------------------------------

!     /*   eam   */
      nrhor_eam = nkind_metal
      nfrho_eam = nkind_metal
      nphir_eam = nkind_metal*(nkind_metal+1)/2

!     /*   neam = total number of eam reference tables   */
      neam = nrhor_eam + nfrho_eam + nphir_eam

!-----------------------------------------------------------------------
!     /*   read eam data                                              */
!-----------------------------------------------------------------------

!     /*   open file   */
      open (iounit,file = 'eam.dat')

!     /*   tag   */
      call search_tag ( '<nref_eam>', 10, iounit, ierr )

!     /*   number of reference data points in the table   */
      if ( neam .ne. 0 ) read( iounit, *, iostat=ierr ) nref_eam

!     /*   tag   */
      call search_tag ( '<rcut_eam>', 10, iounit, ierr )

!     /*   cut off distance   */
      if ( neam .ne. 0 ) read( iounit, *, iostat=ierr ) rcut_eam

!     /*   angstrom --> bohr   */
      rcut_eam = rcut_eam / au_length * 1.d-10

!     /*   close file   */
      close(iounit)

!     /*   on error, no eam   */
      if ( ierr .ne. 0 ) neam = 0

!     /*   error handling   */
      call error_handling &
     &   ( ierr, 'subroutine force_metalwater_eam_setup', 37 )

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   species number   */
      if ( .not. allocated( ikind_eam ) ) &
     &   allocate( ikind_eam(natom) )

!     /*   sum of electron density rho_eam   */
      if ( .not. allocated( srho_eam ) ) &
     &   allocate( srho_eam(natom) )

!     /*   gradient of embedding potential   */
      if ( .not. allocated( dfdrho_eam ) ) &
     &   allocate( dfdrho_eam(natom) )

!     /*   table number of rho-r, f-rho, phi-r   */
      if ( .not. allocated( irhor_eam ) ) &
     &   allocate( irhor_eam(nkind) )
      if ( .not. allocated( ifrho_eam ) ) &
     &   allocate( ifrho_eam(nkind) )
      if ( .not. allocated( iphir_eam ) ) &
     &   allocate( iphir_eam(nkind,nkind) )

!     /*   table rho-r, f-rho, phi-r   */
      if ( .not. allocated( xref_eam ) ) &
     &   allocate( xref_eam(nref_eam,neam) )
      if ( .not. allocated( yref_eam ) ) &
     &   allocate( yref_eam(nref_eam,neam) )

!     /*   spline dimensions for rho-r, f-rho, phi-r   */
      if ( .not. allocated( y2ref_eam ) ) &
     &   allocate( y2ref_eam(nref_eam,neam) )

!-----------------------------------------------------------------------
!     /*   make species number                                        */
!-----------------------------------------------------------------------

      ikind_eam(:) = ikind(:)

!-----------------------------------------------------------------------
!     /*   read eam table:  r[angstrom] - rho                         */
!-----------------------------------------------------------------------

!     /*   open file   */
      open (iounit,file = 'eam.dat')

!     /*   tag   */
      call search_tag ( '<rhor_eam>', 10, iounit, ierr )

!     /*   loop of table   */
      do i = 1, nrhor_eam

!        /*   read species for the table   */
         read( iounit, *, iostat=ierr ) k

!        /*   read table for species k   */
         do j = 1, nref_eam
            read( iounit, *, iostat=ierr ) &
     &         xref_eam(j,i), yref_eam(j,i)
         end do

!        /*   check error   */
         if ( ierr .ne. 0 ) exit

!        /*   i-th table is for species k   */
         irhor_eam(k) = i

!        /*   angstrom --> bohr   */
         do j = 1, nref_eam
            xref_eam(j,i) = xref_eam(j,i) / au_length * 1.d-10
         end do

!     /*   loop of table   */
      end do

!     /*   close file   */
      close(iounit)

!-----------------------------------------------------------------------
!     /*   read eam table:  rho - f[electron volt]                    */
!-----------------------------------------------------------------------

!     /*   open file   */
      open (iounit,file = 'eam.dat')

!        /*   tag   */
         call search_tag ( '<frho_eam>', 10, iounit, ierr )

!        /*   loop of table   */
         do i = nrhor_eam+1, nrhor_eam+nfrho_eam

!           /*   read species for the table   */
            read( iounit, *, iostat=ierr ) k

!           /*   read table for species k   */
            do j = 1, nref_eam
               read( iounit, *, iostat=ierr ) &
     &            xref_eam(j,i), yref_eam(j,i)
            end do

!           /*   check error   */
            if ( ierr .ne. 0 ) exit

!           /*   electron volt --> hartree   */
            do j = 1, nref_eam
               yref_eam(j,i) = yref_eam(j,i)
               yref_eam(j,i) = yref_eam(j,i) * au_charge / au_energy
            end do

!           /*   i-th table is for species k   */
            ifrho_eam(k) = i

!        /*   loop of table   */
         end do

!     /*   close file   */
      close(iounit)

!-----------------------------------------------------------------------
!     /*   read eam table:  r[angstrom] - phi[electron volt]          */
!-----------------------------------------------------------------------

!     /*   open file   */
      open (iounit,file = 'eam.dat')

!     /*   tag   */
      call search_tag ( '<phir_eam>', 10, iounit, ierr )

!     /*   loop of table   */
      do i = nrhor_eam+nfrho_eam+1, nrhor_eam+nfrho_eam+nphir_eam

!        /*   read species for the table   */
         read( iounit, *, iostat=ierr ) k, l

!        /*   read table for species k-l   */
         do j = 1, nref_eam
            read( iounit, *, iostat=ierr ) &
     &         xref_eam(j,i), yref_eam(j,i)
         end do

!        /*   check error   */
         if ( ierr .ne. 0 ) exit

!        /*   angstrom --> bohr,  electron volt --> hartree   */
         do j = 1, nref_eam
            xref_eam(j,i) = xref_eam(j,i) / au_length * 1.d-10
            yref_eam(j,i) = yref_eam(j,i) * au_charge / au_energy
         end do

!        /*   i-th table is for species pair k-l   */
         iphir_eam(k,l) = i
         iphir_eam(l,k) = i

!     /*   loop of table   */
      end do

!     /*   close file   */
      close(iounit)

!-----------------------------------------------------------------------
!     /*   initialization of spline                                   */
!-----------------------------------------------------------------------

      do i = 1, neam
         call spline_init_eam &
     &      ( xref_eam(:,i), yref_eam(:,i), y2ref_eam(:,i), nref_eam )
      end do

      return
      end