!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 25, 2018 by M. Shiga
!      Description:     energy and force from QM/MM electronic embedding
!
!///////////////////////////////////////////////////////////////////////



#ifndef nextver



!***********************************************************************
      subroutine force_me
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables from PIMD                                 */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, fx, fy, fz, vir, dipx, dipy, dipz, iounit, nbead, istep

      use qmmm_variables, only : &
     &   pot_a, fx_a, fy_a, fz_a, vir_a, dipx_a, dipy_a, dipz_a, &
     &   pot_b, fx_b, fy_b, fz_b, vir_b, dipx_b, dipy_b, dipz_b, &
     &   natom_p, natom_s, natom_l, iprint_qmmm

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then
         call setup_me
         call force_me_setup
         call setup_best
         iset = 1
      end if

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      pot_a(:)   = 0.d0
      pot_b(:)   = 0.d0

      fx_a(:,:)  = 0.d0
      fy_a(:,:)  = 0.d0
      fz_a(:,:)  = 0.d0
      fx_b(:,:)  = 0.d0
      fy_b(:,:)  = 0.d0
      fz_b(:,:)  = 0.d0

      vir_a(:,:) = 0.d0
      vir_b(:,:) = 0.d0

      dipx_a(:)  = 0.d0
      dipy_a(:)  = 0.d0
      dipz_a(:)  = 0.d0
      dipx_b(:)  = 0.d0
      dipy_b(:)  = 0.d0
      dipz_b(:)  = 0.d0

!-----------------------------------------------------------------------
!     /*   qm part from SMASH                                         */
!-----------------------------------------------------------------------

      if ( natom_p+natom_l .gt. 0 ) call force_me_qmpart

!-----------------------------------------------------------------------
!     /*   mm part from PIMD                                          */
!-----------------------------------------------------------------------

      if ( natom_s .gt. 0 ) call force_me_mmpart

!-----------------------------------------------------------------------
!     /*   sum                                                        */
!-----------------------------------------------------------------------

      pot(:)  = pot_a(:) + pot_b(:)

      fx(:,:) = fx_a(:,:) + fx_b(:,:)
      fy(:,:) = fy_a(:,:) + fy_b(:,:)
      fz(:,:) = fz_a(:,:) + fz_b(:,:)

      dipx(:) = dipx_a(:) + dipx_b(:)
      dipy(:) = dipy_a(:) + dipy_b(:)
      dipz(:) = dipz_a(:) + dipz_b(:)

      vir(:,:) = vir_a(:,:) + vir_b(:,:)

!-----------------------------------------------------------------------
!     /*   print to file                                              */
!-----------------------------------------------------------------------

      if ( iprint_qmmm .gt. 0 ) then
      if ( mod(istep,iprint_qmmm) .eq. 0 ) then

      open ( iounit, file = 'qmmm.out', access = 'append' )

      do i = 1, nbead
         write( iounit, '(i8,2f16.8)' ) istep, pot_a(i), pot_b(i)
      end do

      close( iounit )

      end if
      end if

      return
      end





!***********************************************************************
      subroutine force_me_qmpart
!***********************************************************************

#ifdef smash

      use common_variables, only : qmmm_potential

      if ( qmmm_potential(1:6) .eq. 'SMASH ' ) then

         call force_me_qmpart_smash

      else

         call force_me_qmpart_pimd

      end if

#else

      call force_me_qmpart_pimd

#endif

      return
      end





!***********************************************************************
      subroutine force_me_qmpart_pimd
!***********************************************************************
!=======================================================================
!
!     this subroutine runs external calls
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables from PIMD                                 */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, nbead, mbox, qmmm_potential, box, iboundary, &
     &   species, iounit, pimd_command

      use qmmm_variables, only : &
     &   fx_a, fy_a, fz_a, dipx_a, dipy_a, dipz_a, pot_a, vir_a, &
     &   r_link, i_link, j_link, natom_p, natom_l, species_link, &
     &   qmmm_dat_dir, qmmm_scr_dir, natom_s, layer

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   reset   */
      implicit none

!     /*   integers   */
      integer :: ibead, i, j, l, m1, m2, m3

!     /*   real numbers   */
      real(8) :: xi, yi, zi, xj, yj, zj

!     /*   characters   */
      character(len=3)  :: char_num

!     /*   flag for initial setting   */
      integer, save :: iset = 0

!     /*   coordinates in qm calculation   */
      real(8), dimension(3,natom_p+natom_l) :: coord

!     /*   energy in qm calculation   */
      real(8) :: escf

!     /*   forces in qm calculation   */
      real(8), dimension(3,natom_p+natom_l) :: grad_s

!     /*   dipole moment in qm calculation   */
      real(8) :: dipx_s, dipy_s, dipz_s

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   visit first time   */
      if ( iset .eq. 0 ) then

!        /*   loop of beads   */
         do ibead = 1, nbead

!           /*   bead number   */
            call int3_to_char( ibead, char_num )

!           /*   make subdirectories for pimd execution   */
            call system( 'mkdir -p ' // trim(qmmm_scr_dir) // '/' // &
     &                   char_num )

!           /*   copy files for pimd execution   */
            call system( 'cp -f ' // trim(qmmm_dat_dir) // '/* ' // &
     &                   trim(qmmm_scr_dir) // '/' // char_num )

!           /*   copy files for pimd execution   */
            call system( 'cp -f input_default.dat ' // &
     &                    trim(qmmm_scr_dir) // '/' // char_num )

!           /*   edit input.dat   */
            open ( iounit, file = trim(qmmm_scr_dir) // '/' &
     &                    // char_num // '/input.dat' )

            write( iounit, '(a)'  ) '<input_style>'
            write( iounit, '(a)'  ) 'NEW'
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<method>'
            write( iounit, '(a)'  ) 'STATIC'
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<natom>'
            write( iounit, '(i8)' ) natom_p + natom_l
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<ipotential>'
            write( iounit, '(a)'  ) qmmm_potential
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<iboundary>'
            write( iounit, '(i1)' ) iboundary
            if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then
               write( iounit, '(3e24.16)' ) box(1,1:3)
               write( iounit, '(3e24.16)' ) box(2,1:3)
               write( iounit, '(3e24.16)' ) box(3,1:3)
            end if
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<iprint_dip>'
            write( iounit, '(i8)' ) 1
            write( iounit, '(a)'  )

            write( iounit, '(a)'  ) '<pimd_command>'
            write( iounit, '(a)'  ) pimd_command
            write( iounit, '(a)'  )

!           /*   close input file   */
            close( iounit )

         end do

         iset = 1

!     /*   visit first time  */
      end if

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!     /*   coordinates in the frame of the first step (nstep=0)       */
!-----------------------------------------------------------------------

!     //   real atoms

      j = 0

      do i = 1, natom_p + natom_s

         if ( layer(i)(1:1) .eq. 'A' ) then

            j = j + 1

            xi = x(i,ibead)
            yi = y(i,ibead)
            zi = z(i,ibead)

            m1 = mbox(1,i,ibead)
            m2 = mbox(2,i,ibead)
            m3 = mbox(3,i,ibead)

            call pbc_unfold ( xi, yi, zi, m1, m2, m3 )

            coord(1,j) = xi
            coord(2,j) = yi
            coord(3,j) = zi

         end if

      end do

!     //   link atoms

      do l = 1, natom_l

         j = j + 1

         xi = x(i_link(l),ibead)
         yi = y(i_link(l),ibead)
         zi = z(i_link(l),ibead)

         m1 = mbox(1,i_link(l),ibead)
         m2 = mbox(2,i_link(l),ibead)
         m3 = mbox(3,i_link(l),ibead)

         call pbc_unfold ( xi, yi, zi, m1, m2, m3 )

         xj = x(j_link(l),ibead)
         yj = y(j_link(l),ibead)
         zj = z(j_link(l),ibead)

         m1 = mbox(1,j_link(l),ibead)
         m2 = mbox(2,j_link(l),ibead)
         m3 = mbox(3,j_link(l),ibead)

         call pbc_unfold ( xj, yj, zj, m1, m2, m3 )

         coord(1,j) = xi + r_link(l) * ( xj - xi )
         coord(2,j) = yi + r_link(l) * ( yj - yi )
         coord(3,j) = zi + r_link(l) * ( zj - zi )

      end do

!-----------------------------------------------------------------------
!     /*   print structure.dat                                        */
!-----------------------------------------------------------------------

      call int3_to_char( ibead, char_num )

      open ( iounit, file = trim(qmmm_scr_dir) // '/' // char_num // &
     &                      '/structure.dat' )

      write( iounit, '(i8)' ) natom_p + natom_l
      write( iounit, '(a)' ) 'BOHR'

      j = 0

      do i = 1, natom_p + natom_s

         if ( layer(i)(1:1) .eq. 'A' ) then

            j = j + 1

            write( iounit, '(a,3e24.16,i2)' ) &
     &         species(i), coord(1:3,j), 1

         end if

      end do

      do i = 1, natom_l

         j = j + 1

         write( iounit, '(a,3e24.16,i2)' ) &
     &      species_link(i), coord(1:3,j), 1

      end do

      close( iounit )

!-----------------------------------------------------------------------
!     /*   print geometry.ini                                         */
!-----------------------------------------------------------------------

      open ( iounit, file = trim(qmmm_scr_dir) // '/' // char_num // &
     &                      '/geometry.ini' )

      do j = 1, natom_p + natom_l

         write( iounit, '(i8,6e24.16,3i4)' ) &
     &      0, coord(1:3,j), 0.d0, 0.d0, 0.d0, 0, 0, 0

      end do

      close( iounit )

!-----------------------------------------------------------------------
!     /*   run PIMD                                                   */
!-----------------------------------------------------------------------

      call system( 'cd ' // trim(qmmm_scr_dir) // '/' // char_num // &
     &             '; ' // pimd_command // '> monitor.out; cd ../..' )

!-----------------------------------------------------------------------
!     /*   read output file                                           */
!-----------------------------------------------------------------------

      open ( iounit, file = trim(qmmm_scr_dir) // '/' // &
     &                      char_num // '/forces.out' )

      read ( iounit, * ) escf

      do j = 1, natom_p + natom_l

         read ( iounit, * ) grad_s(1:3,j)

      end do

      close( iounit )

      open ( iounit, file = trim(qmmm_scr_dir) // '/' // &
     &                      char_num // '/dipole.out' )

      read ( iounit, * ) dipx_s, dipy_s, dipz_s

      close( iounit )

!-----------------------------------------------------------------------
!     /*   potential                                                  */
!-----------------------------------------------------------------------

      pot_a(ibead) = pot_a(ibead) + escf

!-----------------------------------------------------------------------
!     /*   force                                                      */
!-----------------------------------------------------------------------

!     /*   real atoms   */

      j = 0

      do i = 1, natom_p + natom_s

         if ( layer(i)(1:1) .eq. 'A' ) then

            j = j + 1

            fx_a(i,ibead) = fx_a(i,ibead) + grad_s(1,j)
            fy_a(i,ibead) = fy_a(i,ibead) + grad_s(2,j)
            fz_a(i,ibead) = fz_a(i,ibead) + grad_s(3,j)

         end if

      end do

!     /*   link atoms   */

      do l = 1, natom_l

         j = j + 1

         fx_a(i_link(l),ibead) = fx_a(i_link(l),ibead) &
     &      + ( 1.d0 - r_link(l) ) * grad_s(1,j)
         fy_a(i_link(l),ibead) = fy_a(i_link(l),ibead) &
     &      + ( 1.d0 - r_link(l) ) * grad_s(2,j)
         fz_a(i_link(l),ibead) = fz_a(i_link(l),ibead) &
     &      + ( 1.d0 - r_link(l) ) * grad_s(3,j)

         fx_a(j_link(l),ibead) = fx_a(j_link(l),ibead) &
     &      + r_link(l) * grad_s(1,j)
         fy_a(j_link(l),ibead) = fy_a(j_link(l),ibead) &
     &      + r_link(l) * grad_s(2,j)
         fz_a(j_link(l),ibead) = fz_a(j_link(l),ibead) &
     &      + r_link(l) * grad_s(3,j)

      end do

!-----------------------------------------------------------------------
!     /*   dipole moment                                              */
!-----------------------------------------------------------------------

      dipx_a(ibead) = dipx_a(ibead) + dipx_s
      dipy_a(ibead) = dipy_a(ibead) + dipy_s
      dipz_a(ibead) = dipz_a(ibead) + dipz_s

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

!     /*   real atoms   */

      j = 0

      do i = 1, natom_p + natom_s

         if ( layer(i)(1:1) .eq. 'A' ) then

            j = j + 1

            xi = coord(1,j)
            yi = coord(2,j)
            zi = coord(3,j)

            vir_a(1,1) = vir_a(1,1) + grad_s(1,j)*xi
            vir_a(1,2) = vir_a(1,2) + grad_s(1,j)*yi
            vir_a(1,3) = vir_a(1,3) + grad_s(1,j)*zi
            vir_a(2,1) = vir_a(2,1) + grad_s(2,j)*xi
            vir_a(2,2) = vir_a(2,2) + grad_s(2,j)*yi
            vir_a(2,3) = vir_a(2,3) + grad_s(2,j)*zi
            vir_a(3,1) = vir_a(3,1) + grad_s(3,j)*xi
            vir_a(3,2) = vir_a(3,2) + grad_s(3,j)*yi
            vir_a(3,3) = vir_a(3,3) + grad_s(3,j)*zi

         end if

      end do

!     /*   link atoms   */

      do l = 1, natom_l

         j = j + 1

         xi = coord(1,j)
         yi = coord(2,j)
         zi = coord(3,j)

         vir_a(1,1) = vir_a(1,1) + grad_s(1,j)*xi
         vir_a(1,2) = vir_a(1,2) + grad_s(1,j)*yi
         vir_a(1,3) = vir_a(1,3) + grad_s(1,j)*zi
         vir_a(2,1) = vir_a(2,1) + grad_s(2,j)*xi
         vir_a(2,2) = vir_a(2,2) + grad_s(2,j)*yi
         vir_a(2,3) = vir_a(2,3) + grad_s(2,j)*zi
         vir_a(3,1) = vir_a(3,1) + grad_s(3,j)*xi
         vir_a(3,2) = vir_a(3,2) + grad_s(3,j)*yi
         vir_a(3,3) = vir_a(3,3) + grad_s(3,j)*zi

      end do

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

      return
      end





!#######################################################################
#ifdef smash
!#######################################################################

!***********************************************************************
      subroutine force_me_qmpart_smash
!***********************************************************************
!=======================================================================
!
!     this subroutine runs SMASH via internal calls
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables from PIMD                                 */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, nbead, istep, mbox

      use qmmm_variables, only : &
     &   fx_a, fy_a, fz_a, dipx_a, dipy_a, dipz_a, pot_a, vir_a, &
     &   r_link, i_link, j_link, natom_p, natom_s, natom_l, layer

!-----------------------------------------------------------------------
!     /*   shared variables from SMASH                                */
!-----------------------------------------------------------------------

      use modparallel, only : &
     &   master, nproc1, nproc2, myrank1, myrank2, mpi_comm1, mpi_comm2
      use modwarn, only : nwarn
      use modmemory, only : memusedmax
      use modjob, only : runtype, scftype
!cc      use modecp, only : flagecp
      use modiofile, only : input, icheck, version
      use modmolecule, only : coord
      use modenergy, only : escf, emp2
      use modguess, only : guess

!-----------------------------------------------------------------------
!     /*   shared variables from PIMD-SMASH interface                 */
!-----------------------------------------------------------------------

      use smash_variables, only : grad_s, dipx_s, dipy_s, dipz_s, &
     &   znuc2_s, natom_smash

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   reset   */
      implicit none

!     /*   integers   */
      integer :: ibead, i, j, l, m1, m2, m3

!     /*   real numbers   */
      real(8) :: xi, yi, zi, xj, yj, zj

!     /*   flag for initial setting   */
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   visited: first time   */
      if ( iset .eq. 0 ) then

!        /*   number of atoms in SMASH   */
         natom_smash = natom_p + natom_l

!        /*   memory allocatation of gradient   */
         if ( .not. allocated( grad_s ) ) &
     &      allocate( grad_s(3,natom_smash) )

!        /*   point charges of qm images   */
         if ( .not. allocated( znuc2_s ) ) &
     &      allocate( znuc2_s(natom_smash) )

!        /*   reset all SMASH energies to zero   */
         escf = 0.d0
         emp2 = 0.d0

!        /*   initialize mpi environment   */
         call setparallel

!        /*   open SMASH output file   */
         if ( master ) open( 99, file = 'smash.out' )

!c        /*   open SMASH input file   */
!         if ( master ) open( 98, file = 'smash.dat' )

!        /*   version of SMASH   */
         version = '2.2.0'

!        /*   print title of SMASH   */

         if ( master ) then
            write( 99, &
     &      '(" *******************************************",/, &
     &        "            SMASH Version ",a10/, &
     &        "           written by K. ISHIMURA",/, &
     &        " *******************************************",/)') &
     &      version
         end if

!        /*   SMASH timer routine   */
         call tstamp(0)

!        /*   initialize parallel calculation of SMASH   */
         call parallelinfo

!        /*   read input file and set details   */
         call setdetails(mpi_comm1)

!        /*   set indispensible data for PIMD-SMASH   */
         call force_smash_start

!c        /*   open SMASH output file   */
!        if ( master ) close( 98 )

!        /*   setup finished  */
         iset = 1

!     /*   visit: not first time   */
      else

!        /*   initial guess from previous step   */
         guess = 'PIMD'

!        /*   open SMASH output file   */
         if ( master ) open( 99, file = 'smash.out' )

!        /*   print title of SMASH   */

         if ( master ) then
            write( 99, &
     &      '(" *******************************************",/, &
     &        "            SMASH Version ",a10/, &
     &        "           written by K. ISHIMURA",/, &
     &        " *******************************************",/)') &
     &      version
         end if

!        /*   SMASH timer routine   */
         call tstamp(0)

!     /*   visit   */
      end if

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!     /*   coordinates in the frame of the first step (nstep=0)       */
!-----------------------------------------------------------------------

!     //   real atoms

      j = 0

      do i = 1, natom_p + natom_s

         if ( layer(i)(1:1) .eq. 'A' ) then

            j = j + 1

            xi = x(i,ibead)
            yi = y(i,ibead)
            zi = z(i,ibead)

            m1 = mbox(1,i,ibead)
            m2 = mbox(2,i,ibead)
            m3 = mbox(3,i,ibead)

            call pbc_unfold ( xi, yi, zi, m1, m2, m3 )

            coord(1,j) = xi
            coord(2,j) = yi
            coord(3,j) = zi

         end if

      end do

!     //   link atoms

      do l = 1, natom_l

         j = j + 1

         xi = x(i_link(l),ibead)
         yi = y(i_link(l),ibead)
         zi = z(i_link(l),ibead)

         m1 = mbox(1,i_link(l),ibead)
         m2 = mbox(2,i_link(l),ibead)
         m3 = mbox(3,i_link(l),ibead)

         call pbc_unfold ( xi, yi, zi, m1, m2, m3 )

         xj = x(j_link(l),ibead)
         yj = y(j_link(l),ibead)
         zj = z(j_link(l),ibead)

         m1 = mbox(1,j_link(l),ibead)
         m2 = mbox(2,j_link(l),ibead)
         m3 = mbox(3,j_link(l),ibead)

         call pbc_unfold ( xj, yj, zj, m1, m2, m3 )

         coord(1,j) = xi + r_link(l) * ( xj - xi )
         coord(2,j) = yi + r_link(l) * ( yj - yi )
         coord(3,j) = zi + r_link(l) * ( zj - zi )

      end do

!-----------------------------------------------------------------------
!     /*   print to SMASH output                                      */
!-----------------------------------------------------------------------

      if ( master ) then
         write( 99, '(a)' ) &
     &   "#####################################################"
         write( 99, '(a,i8,a,i8)' ) &
     &   "## STEP: ", istep, "  ## BEAD", ibead
         write( 99, '(a)' ) &
     &   "#####################################################"
      end if

!-----------------------------------------------------------------------
!     /*   run SMASH                                                  */
!-----------------------------------------------------------------------

      if ( scftype == 'RHF' ) then

         if ( runtype == 'ENERGY' ) then

            call calcrenergy &
     &         (nproc1, nproc2, myrank1, myrank2, mpi_comm1, mpi_comm2)

         else if (runtype == 'GRADIENT' ) then

            call calcrgradient &
     &         (nproc1, nproc2, myrank1, myrank2, mpi_comm1, mpi_comm2)

         else

           if (master) then
              write ( 99, &
     &           '(" Error! This program does not support runtype= ", &
     &           a16,".")') runtype
              call iabort
           end if

         end if

      else if ( scftype == 'UHF' ) then

         if ( runtype == 'ENERGY' ) then

            call calcuenergy &
     &         (nproc1, nproc2, myrank1, myrank2, mpi_comm1, mpi_comm2)

         else if ( runtype == 'GRADIENT' ) then

            call calcugradient &
     &         (nproc1, nproc2, myrank1, myrank2, mpi_comm1, mpi_comm2)

         end if

      else

         if (master) then
            write( 99, '(" Error! SCFtype=",a16," is not supported.")') &
     &      scftype
         end if

         call iabort

      end if

!-----------------------------------------------------------------------
!     /*   close input.dat and checkpoint files                       */
!-----------------------------------------------------------------------

!     /*   close and remove input.dat.number   */
!cc      if ( master ) close ( unit=input, status='DELETE' )

!     /*   close checkpoint files if is there   */
!cc      if ( master .and. (check /= '') ) close(unit=icheck)

!     /*   SMASH memory check   */
      call memcheck

!     /*   SMASH timer routine   */
      call tstamp(2)

      if (master) then
         write( 99, &
     &     '(" Used memory :",1x,i6," MB")') memusedmax/125000
         write( 99, &
     &     '(" Your calculation finished with",i3," warning(s).") &
     &     ') nwarn
      end if

!-----------------------------------------------------------------------
!     /*   SMASH potential                                            */
!-----------------------------------------------------------------------

      pot_a(ibead) = pot_a(ibead) + escf + emp2

!-----------------------------------------------------------------------
!     /*   SMASH force ( - gradient )                                 */
!-----------------------------------------------------------------------

!     /*   real atoms   */

      j = 0

      do i = 1, natom_p + natom_s

         if ( layer(i)(1:1) .eq. 'A' ) then

            j = j + 1

            fx_a(i,ibead) = fx_a(i,ibead) - grad_s(1,j)
            fy_a(i,ibead) = fy_a(i,ibead) - grad_s(2,j)
            fz_a(i,ibead) = fz_a(i,ibead) - grad_s(3,j)

         end if

      end do

!     /*   link atoms   */

      do l = 1, natom_l

         j = j + 1

         fx_a(i_link(l),ibead) = fx_a(i_link(l),ibead) &
     &      - ( 1.d0 - r_link(l) ) * grad_s(1,j)
         fy_a(i_link(l),ibead) = fy_a(i_link(l),ibead) &
     &      - ( 1.d0 - r_link(l) ) * grad_s(2,j)
         fz_a(i_link(l),ibead) = fz_a(i_link(l),ibead) &
     &      - ( 1.d0 - r_link(l) ) * grad_s(3,j)

         fx_a(j_link(l),ibead) = fx_a(j_link(l),ibead) &
     &      - r_link(l) * grad_s(1,j)
         fy_a(j_link(l),ibead) = fy_a(j_link(l),ibead) &
     &      - r_link(l) * grad_s(2,j)
         fz_a(j_link(l),ibead) = fz_a(j_link(l),ibead) &
     &      - r_link(l) * grad_s(3,j)

      end do

!-----------------------------------------------------------------------
!     /*   SMASH dipole moment - SCF level                            */
!-----------------------------------------------------------------------

      dipx_a(ibead) = dipx_a(ibead) + dipx_s
      dipy_a(ibead) = dipy_a(ibead) + dipy_s
      dipz_a(ibead) = dipz_a(ibead) + dipz_s

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

!     /*   real atoms   */

      j = 0

      do i = 1, natom_p + natom_s

         if ( layer(i)(1:1) .eq. 'A' ) then

            j = j + 1

            xi = coord(1,j)
            yi = coord(2,j)
            zi = coord(3,j)

            vir_a(1,1) = vir_a(1,1) - grad_s(1,j)*xi
            vir_a(1,2) = vir_a(1,2) - grad_s(1,j)*yi
            vir_a(1,3) = vir_a(1,3) - grad_s(1,j)*zi
            vir_a(2,1) = vir_a(2,1) - grad_s(2,j)*xi
            vir_a(2,2) = vir_a(2,2) - grad_s(2,j)*yi
            vir_a(2,3) = vir_a(2,3) - grad_s(2,j)*zi
            vir_a(3,1) = vir_a(3,1) - grad_s(3,j)*xi
            vir_a(3,2) = vir_a(3,2) - grad_s(3,j)*yi
            vir_a(3,3) = vir_a(3,3) - grad_s(3,j)*zi

         end if

      end do

!     /*   link atoms   */

      do l = 1, natom_l

         j = j + 1

         xi = coord(1,j)
         yi = coord(2,j)
         zi = coord(3,j)

         vir_a(1,1) = vir_a(1,1) - grad_s(1,j)*xi
         vir_a(1,2) = vir_a(1,2) - grad_s(1,j)*yi
         vir_a(1,3) = vir_a(1,3) - grad_s(1,j)*zi
         vir_a(2,1) = vir_a(2,1) - grad_s(2,j)*xi
         vir_a(2,2) = vir_a(2,2) - grad_s(2,j)*yi
         vir_a(2,3) = vir_a(2,3) - grad_s(2,j)*zi
         vir_a(3,1) = vir_a(3,1) - grad_s(3,j)*xi
         vir_a(3,2) = vir_a(3,2) - grad_s(3,j)*yi
         vir_a(3,3) = vir_a(3,3) - grad_s(3,j)*zi

      end do

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   close file                                                 */
!-----------------------------------------------------------------------

      if ( master ) close( 99 )

      return
      end

!#######################################################################
#else
!#######################################################################

!***********************************************************************
      subroutine force_me_qmpart_smash
!***********************************************************************

      implicit none

      write( 6, '(a)' ) &
     &   'Error - SMASH must be installed.'
      write( 6, '(a)' )

      call error_handling ( 1, 'subroutine force_me_qmpart_smash', 32 )

      return
      end

!#######################################################################
#endif
!#######################################################################





!***********************************************************************
      subroutine force_me_mmpart
!***********************************************************************
!=======================================================================
!
!     this subroutine runs SMASH via internal calls
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   reset   */
      implicit none

!-----------------------------------------------------------------------
!     /*   main routine                                               */
!-----------------------------------------------------------------------

      call force_me_main

!-----------------------------------------------------------------------
!     /*   dipole moment                                              */
!-----------------------------------------------------------------------

      call me_dipole( 0 )

      return
      end





!***********************************************************************
      subroutine force_me_setup
!***********************************************************************

      call force_mm_setup

      return
      end





!***********************************************************************
      subroutine force_me_main
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : iboundary

      implicit none

!-----------------------------------------------------------------------
!     /*   linear bonds                                               */
!-----------------------------------------------------------------------

      call force_me_lin( 0 )

!-----------------------------------------------------------------------
!     /*   generalized linear bonds                                   */
!-----------------------------------------------------------------------

      call force_me_genlin( 0 )

!-----------------------------------------------------------------------
!     /*   angular bonds                                              */
!-----------------------------------------------------------------------

      call force_me_angl( 0 )

!-----------------------------------------------------------------------
!     /*   dihedral bonds                                             */
!-----------------------------------------------------------------------

      call force_me_dih( 0 )

!-----------------------------------------------------------------------
!     /*   improper bonds                                             */
!-----------------------------------------------------------------------

      call force_me_improper( 0 )

!-----------------------------------------------------------------------
!     /*   cmap of two dihedral bonds                                 */
!-----------------------------------------------------------------------

      call force_me_cmap( 0 )

!-----------------------------------------------------------------------
!     /*   lennard-jones                                              */
!-----------------------------------------------------------------------

      call force_me_lj( 0 )

!-----------------------------------------------------------------------
!     /*   lennard-jones pair                                         */
!-----------------------------------------------------------------------

      call force_me_ljpair( 0 )

!-----------------------------------------------------------------------
!     /*   morse potential                                            */
!-----------------------------------------------------------------------

      call force_me_morse( 0 )

!-----------------------------------------------------------------------
!     /*   free     boundary  =  direct sum                           */
!     /*   periodic boundary  =  Ewald  sum                           */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) then

!        /*   direct sum   */
         call force_me_coulomb( 0 )

      else if ( iboundary .eq. 1 ) then

!        /*   Ewald sum   */
         call force_me_ewald
         call force_me_coulomb_subtract( 0 )

      else if ( iboundary .eq. 2 ) then

!        /*   Ewald sum   */
         call force_me_ewald
         call force_me_coulomb_subtract( 0 )

      end if

      return
      end





!***********************************************************************
      subroutine force_me_lin( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, nbead

      use mm_variables, only : &
     &   fc_lin, eq_lin, nlin, i_lin, j_lin

      use qmmm_variables, only : &
     &   layer, fx_b, fy_b, fz_b, pot_b, vir_b

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: m, k, i, j, ioption

      real(8) :: xij, yij, zij, rij, dr, const, fxi, fyi, fzi

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( nlin .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         do k = 1, nlin

            i = i_lin(k)
            j = j_lin(k)

            if ( ioption .eq. 0 ) then
               if ( ( layer(i)(1:1) .eq. 'A' ) .and. &
     &              ( layer(j)(1:1) .eq. 'A' ) ) cycle
            end if

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom ( xij, yij, zij )

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            dr = ( rij - eq_lin(k) )

            pot_b(m) = pot_b(m) + 0.5d0*fc_lin(k)*dr*dr

            const = - fc_lin(k)*dr/rij

            fxi = const*xij
            fyi = const*yij
            fzi = const*zij

            fx_b(i,m) = fx_b(i,m) + fxi
            fy_b(i,m) = fy_b(i,m) + fyi
            fz_b(i,m) = fz_b(i,m) + fzi

            fx_b(j,m) = fx_b(j,m) - fxi
            fy_b(j,m) = fy_b(j,m) - fyi
            fz_b(j,m) = fz_b(j,m) - fzi

            vir_b(1,1) = vir_b(1,1) + fxi*xij
            vir_b(1,2) = vir_b(1,2) + fxi*yij
            vir_b(1,3) = vir_b(1,3) + fxi*zij
            vir_b(2,1) = vir_b(2,1) + fyi*xij
            vir_b(2,2) = vir_b(2,2) + fyi*yij
            vir_b(2,3) = vir_b(2,3) + fyi*zij
            vir_b(3,1) = vir_b(3,1) + fzi*xij
            vir_b(3,2) = vir_b(3,2) + fzi*yij
            vir_b(3,3) = vir_b(3,3) + fzi*zij

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_me_genlin( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, nbead

      use mm_variables, only : &
     &   fc_genlin, eq_genlin, ngenlin, n_genlin, i_genlin, j_genlin

      use qmmm_variables, only : &
     &   layer, fx_b, fy_b, fz_b, pot_b, vir_b

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: m, k, i, j, ioption

      real(8) :: xij, yij, zij, rij, dr, const, fxi, fyi, fzi

      real(8) :: drn = 1.d0
      real(8) :: drm = 0.d0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( ngenlin .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         do k = 1, ngenlin

            i = i_genlin(k)
            j = j_genlin(k)

            if ( ioption .eq. 0 ) then
               if ( ( layer(i)(1:1) .eq. 'A' ) .and. &
     &              ( layer(j)(1:1) .eq. 'A' ) ) cycle
            end if

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom ( xij, yij, zij )

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            dr  = ( rij - eq_genlin(k) )

            if      ( n_genlin(k) .eq. 0 ) then
               drn    = 1.d0
               drm    = 0.d0
            else if ( n_genlin(k) .eq. 1 ) then
               drn    = dr
               drm    = 1.d0
            else if ( n_genlin(k) .eq. 2 ) then
               drn    = dr*dr
               drm    = dr
            else if ( n_genlin(k) .eq. 3 ) then
               drn    = dr*dr*dr
               drm    = dr*dr
            else if ( n_genlin(k) .eq. 4 ) then
               drn    = dr*dr*dr*dr
               drm    = dr*dr*dr
            else if ( n_genlin(k) .ge. 5 ) then
               drn    = dr**(n_genlin(k))
               drm    = dr**(n_genlin(k)-1)
            else if ( n_genlin(k) .le. -1 ) then
               drn    = dr**(n_genlin(k))
               drm    = dr**(n_genlin(k)-1)
            end if

            pot_b(m) = pot_b(m) + fc_genlin(k)*drn

            const = - n_genlin(k)*fc_genlin(k)*drm/rij

            fxi = const*xij
            fyi = const*yij
            fzi = const*zij

            fx_b(i,m) = fx_b(i,m) + fxi
            fy_b(i,m) = fy_b(i,m) + fyi
            fz_b(i,m) = fz_b(i,m) + fzi

            fx_b(j,m) = fx_b(j,m) - fxi
            fy_b(j,m) = fy_b(j,m) - fyi
            fz_b(j,m) = fz_b(j,m) - fzi

            vir_b(1,1) = vir_b(1,1) + fxi*xij
            vir_b(1,2) = vir_b(1,2) + fxi*yij
            vir_b(1,3) = vir_b(1,3) + fxi*zij
            vir_b(2,1) = vir_b(2,1) + fyi*xij
            vir_b(2,2) = vir_b(2,2) + fyi*yij
            vir_b(2,3) = vir_b(2,3) + fyi*zij
            vir_b(3,1) = vir_b(3,1) + fzi*xij
            vir_b(3,2) = vir_b(3,2) + fzi*yij
            vir_b(3,3) = vir_b(3,3) + fzi*zij

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_me_angl( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pi, nbead

      use mm_variables, only : &
     &   fc_angl, eq_angl, i_angl, j_angl, k_angl, nangl

      use qmmm_variables, only : &
     &   layer, fx_b, fy_b, fz_b, pot_b, vir_b

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, m, l, ioption

      real(8) :: xij, yij, zij, xkj, ykj, zkj, rij2, rkj2, rijk, &
     &           pijk, qijk, bijk, aijk, da, const, &
     &           fxi, fxj, fxk, fyi, fyj, fyk, fzi, fzj, fzk

      real(8) :: tiny_value = 1.d-4

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( nangl .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         do l = 1, nangl

            i = i_angl(l)
            j = j_angl(l)
            k = k_angl(l)

            if ( ioption .eq. 0 ) then
               if ( ( layer(i)(1:1) .eq. 'A' ) .and. &
     &              ( layer(j)(1:1) .eq. 'A' ) .and. &
     &              ( layer(k)(1:1) .eq. 'A' ) ) cycle
            end if

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom ( xij, yij, zij )

            xkj = x(k,m) - x(j,m)
            ykj = y(k,m) - y(j,m)
            zkj = z(k,m) - z(j,m)

            call pbc_atom ( xkj, ykj, zkj )

            rij2 = xij*xij + yij*yij + zij*zij
            rkj2 = xkj*xkj + ykj*ykj + zkj*zkj

            rijk = sqrt( rij2*rkj2 )

            pijk = xij*xkj + yij*ykj + zij*zkj

            qijk  = pijk/rijk

            qijk = max( qijk, -1.d0 )
            qijk = min( qijk,  1.d0 )

            bijk = acos( qijk )

            aijk = bijk*(180.d0/pi)

            da   = aijk - eq_angl(l)

            pot_b(m) = pot_b(m) + 0.5d0*fc_angl(l)*da*da

            if ( abs(bijk)    .lt. tiny_value ) cycle
            if ( abs(bijk-pi) .lt. tiny_value ) cycle
            if ( abs(bijk+pi) .lt. tiny_value ) cycle

            const = fc_angl(l)*da /sin(bijk) /rijk *(180.d0/pi)

            fxi = const*( xkj - pijk/rij2*xij )
            fxk = const*( xij - pijk/rkj2*xkj )
            fxj = - fxi - fxk

            fyi = const*( ykj - pijk/rij2*yij )
            fyk = const*( yij - pijk/rkj2*ykj )
            fyj = - fyi - fyk

            fzi = const*( zkj - pijk/rij2*zij )
            fzk = const*( zij - pijk/rkj2*zkj )
            fzj = - fzi - fzk

            fx_b(i,m) = fx_b(i,m) + fxi
            fx_b(j,m) = fx_b(j,m) + fxj
            fx_b(k,m) = fx_b(k,m) + fxk

            fy_b(i,m) = fy_b(i,m) + fyi
            fy_b(j,m) = fy_b(j,m) + fyj
            fy_b(k,m) = fy_b(k,m) + fyk

            fz_b(i,m) = fz_b(i,m) + fzi
            fz_b(j,m) = fz_b(j,m) + fzj
            fz_b(k,m) = fz_b(k,m) + fzk

            vir_b(1,1) = vir_b(1,1) + fxi*xij + fxk*xkj
            vir_b(1,2) = vir_b(1,2) + fxi*yij + fxk*ykj
            vir_b(1,3) = vir_b(1,3) + fxi*zij + fxk*zkj
            vir_b(2,1) = vir_b(2,1) + fyi*xij + fyk*xkj
            vir_b(2,2) = vir_b(2,2) + fyi*yij + fyk*ykj
            vir_b(2,3) = vir_b(2,3) + fyi*zij + fyk*zkj
            vir_b(3,1) = vir_b(3,1) + fzi*xij + fzk*xkj
            vir_b(3,2) = vir_b(3,2) + fzi*yij + fzk*ykj
            vir_b(3,3) = vir_b(3,3) + fzi*zij + fzk*zkj

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_me_dih( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, nbead

      use mm_variables, only : &
     &   v_dih, i_dih, j_dih, k_dih, l_dih, ndih, mu_dih, nu_dih

      use qmmm_variables, only : &
     &   layer, fx_b, fy_b, fz_b, pot_b, vir_b

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, n, mu, nu, ii, ioption

      real(8) :: xij, yij, zij, xkj, ykj, zkj, xlj, ylj, zlj, &
     &           xijk, yijk, zijk, xjkl, yjkl, zjkl, rijk2, rjkl2, v, &
     &           rijkl2, rijk2inv, rjkl2inv, rijkl2inv, cos_phi, phi, &
     &           factor, fxi, fyi, fzi, fxj, fyj, fzj, fxk, fyk, fzk, &
     &           fxl, fyl, fzl

      real(8) :: tiny_value = 1.d-4

      real(8), dimension(0:12):: sinfactor

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( ndih .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         do n = 1, ndih

            i = i_dih(n)
            j = j_dih(n)
            k = k_dih(n)
            l = l_dih(n)

            if ( ioption .eq. 0 ) then
               if ( ( layer(i)(1:1) .eq. 'A' ) .and. &
     &              ( layer(j)(1:1) .eq. 'A' ) .and. &
     &              ( layer(k)(1:1) .eq. 'A' ) .and. &
     &              ( layer(l)(1:1) .eq. 'A' ) ) cycle
            end if

!-----------------------------------------------------------------------

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom ( xij, yij, zij )

            xkj = x(k,m) - x(j,m)
            ykj = y(k,m) - y(j,m)
            zkj = z(k,m) - z(j,m)

            call pbc_atom ( xkj, ykj, zkj )

            xlj = x(l,m) - x(j,m)
            ylj = y(l,m) - y(j,m)
            zlj = z(l,m) - z(j,m)

            call pbc_atom ( xlj, ylj, zlj )

!-----------------------------------------------------------------------

            xijk = yij*zkj - zij*ykj
            yijk = zij*xkj - xij*zkj
            zijk = xij*ykj - yij*xkj

            xjkl = ylj*zkj - zlj*ykj
            yjkl = zlj*xkj - xlj*zkj
            zjkl = xlj*ykj - ylj*xkj

            rijk2  = xijk*xijk + yijk*yijk + zijk*zijk
            rjkl2  = xjkl*xjkl + yjkl*yjkl + zjkl*zjkl

            rijkl2 = sqrt(rijk2*rjkl2)

            if ( abs(rijk2)  .lt. tiny_value ) cycle
            if ( abs(rjkl2)  .lt. tiny_value ) cycle
            if ( abs(rijkl2) .lt. tiny_value ) cycle

            rijk2inv  = 1.d0 / rijk2
            rjkl2inv  = 1.d0 / rjkl2
            rijkl2inv = 1.d0 / rijkl2

!-----------------------------------------------------------------------
!           /*   cos_phi = cos( phi )                                 */
!-----------------------------------------------------------------------

            cos_phi = ( xijk*xjkl + yijk*yjkl + zijk*zjkl ) * rijkl2inv

            cos_phi = max( cos_phi, -1.d0 )
            cos_phi = min( cos_phi,  1.d0 )

            phi = acos(cos_phi)

!-----------------------------------------------------------------------
!           /*   mu     =  cos(delta)  =  +1 or -1                    */
!           /*   delta  =  0 or pi                                    */
!-----------------------------------------------------------------------

            mu = mu_dih(n)
            nu = nu_dih(n)
            v  =  v_dih(n)

!-----------------------------------------------------------------------
!           /*   pot = 0.5 * v * ( cos ( nu*phi - delta ) + 1 )       */
!-----------------------------------------------------------------------

            pot_b(m)  = pot_b(m) + 0.5d0 * v * ( 1.d0 + cos(nu*phi)*mu )

!-----------------------------------------------------------------------
!           /*    sinfactor(n) = sin(n*phi) / sin(phi)                */
!-----------------------------------------------------------------------

            if      ( nu .eq. 0 ) then
               sinfactor(0) = 0.d0
            else if ( nu .eq. 1 ) then
               sinfactor(1) = 1.d0
            else if( nu .eq. 2 ) then
               sinfactor(2) = 2.d0*cos_phi
            else if( nu .eq. 3 ) then
               sinfactor(3) = 4.d0*cos_phi*cos_phi - 1.d0
            else if( nu .eq. 4 ) then
               sinfactor(4) = 4.d0*cos_phi*(2.d0*cos_phi*cos_phi - 1.d0)
            else if( nu .eq. 6 ) then
               sinfactor(6) = 2.d0 * ( 4.d0*cos_phi*cos_phi - 1.d0 ) &
     &                   * cos_phi * ( 4.d0*cos_phi*cos_phi - 3.d0 )
            else
               sinfactor(1) = 1.d0
               sinfactor(2) = 2.d0*cos_phi
               do ii = 3, nu
                  sinfactor(ii) = sinfactor(ii-2) &
     &                         - sin((ii-2)*phi)*sin(phi) &
     &                         + cos((ii-2)*phi)*cos(phi) &
     &                         + cos((ii-1)*phi)
               end do
            end if

            factor = - sinfactor(nu) * 0.5d0 * v * mu * nu

            fxi = factor * ( + ( ykj*zjkl - zkj*yjkl ) * rijkl2inv &
     &                    - ( ykj*zijk - zkj*yijk ) * cos_phi*rijk2inv )
            fyi = factor * ( + ( zkj*xjkl - xkj*zjkl ) * rijkl2inv &
     &                    - ( zkj*xijk - xkj*zijk ) * cos_phi*rijk2inv )
            fzi = factor * ( + ( xkj*yjkl - ykj*xjkl ) * rijkl2inv &
     &                    - ( xkj*yijk - ykj*xijk ) * cos_phi*rijk2inv )

            fxl = factor * ( + ( ykj*zijk - zkj*yijk ) * rijkl2inv &
     &                    - ( ykj*zjkl - zkj*yjkl ) * cos_phi*rjkl2inv )
            fyl = factor * ( + ( zkj*xijk - xkj*zijk ) * rijkl2inv &
     &                    - ( zkj*xjkl - xkj*zjkl ) * cos_phi*rjkl2inv )
            fzl = factor * ( + ( xkj*yijk - ykj*xijk ) * rijkl2inv &
     &                    - ( xkj*yjkl - ykj*xjkl ) * cos_phi*rjkl2inv )

            fxk = factor * ( - ( yij*zjkl - zij*yjkl ) * rijkl2inv &
     &                    - ( ylj*zijk - zlj*yijk ) * rijkl2inv &
     &                    + ( yij*zijk - zij*yijk ) * cos_phi*rijk2inv &
     &                    + ( ylj*zjkl - zlj*yjkl ) * cos_phi*rjkl2inv )
            fyk = factor * ( - ( zij*xjkl - xij*zjkl ) * rijkl2inv &
     &                    - ( zlj*xijk - xlj*zijk ) * rijkl2inv &
     &                    + ( zij*xijk - xij*zijk ) * cos_phi*rijk2inv &
     &                    + ( zlj*xjkl - xlj*zjkl ) * cos_phi*rjkl2inv )
            fzk = factor * ( - ( xij*yjkl - yij*xjkl ) * rijkl2inv &
     &                    - ( xlj*yijk - ylj*xijk ) * rijkl2inv &
     &                    + ( xij*yijk - yij*xijk ) * cos_phi*rijk2inv &
     &                    + ( xlj*yjkl - ylj*xjkl ) * cos_phi*rjkl2inv )

            fxj = - ( fxi + fxk + fxl )
            fyj = - ( fyi + fyk + fyl )
            fzj = - ( fzi + fzk + fzl )

            fx_b(i,m) = fx_b(i,m) + fxi
            fy_b(i,m) = fy_b(i,m) + fyi
            fz_b(i,m) = fz_b(i,m) + fzi

            fx_b(j,m) = fx_b(j,m) + fxj
            fy_b(j,m) = fy_b(j,m) + fyj
            fz_b(j,m) = fz_b(j,m) + fzj

            fx_b(k,m) = fx_b(k,m) + fxk
            fy_b(k,m) = fy_b(k,m) + fyk
            fz_b(k,m) = fz_b(k,m) + fzk

            fx_b(l,m) = fx_b(l,m) + fxl
            fy_b(l,m) = fy_b(l,m) + fyl
            fz_b(l,m) = fz_b(l,m) + fzl

            vir_b(1,1) = vir_b(1,1) + fxi*xij + fxk*xkj + fxl*xlj
            vir_b(1,2) = vir_b(1,2) + fxi*yij + fxk*ykj + fxl*ylj
            vir_b(1,3) = vir_b(1,3) + fxi*zij + fxk*zkj + fxl*zlj
            vir_b(2,1) = vir_b(2,1) + fyi*xij + fyk*xkj + fyl*xlj
            vir_b(2,2) = vir_b(2,2) + fyi*yij + fyk*ykj + fyl*ylj
            vir_b(2,3) = vir_b(2,3) + fyi*zij + fyk*zkj + fyl*zlj
            vir_b(3,1) = vir_b(3,1) + fzi*xij + fzk*xkj + fzl*xlj
            vir_b(3,2) = vir_b(3,2) + fzi*yij + fzk*ykj + fzl*ylj
            vir_b(3,3) = vir_b(3,3) + fzi*zij + fzk*zkj + fzl*zlj

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_me_coulomb( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, nbead, mbox

      use mm_variables, only : &
     &   q, factor_bcp, ncharge, nbcp, i_q, i_bcp, j_bcp

      use qmmm_variables, only : &
     &   layer, fx_b, fy_b, fz_b, pot_b, vir_b

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, m1, m2, m3, ioption

      real(8) :: qi, qj, xij, yij, zij, rij, rinv, uij, duij, &
     &           fxi, fyi, fzi, factor, xi, yi, zi, xj, yj, zj

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( ncharge .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   main loop:  add A-B and B-B contributions                  */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         do m = 1, nbead

         do k = 1, ncharge-1
         do l = k+1, ncharge

            i  = i_q(k)
            j  = i_q(l)

            if ( ( layer(i)(1:1) .eq. 'A' ) .and. &
     &           ( layer(j)(1:1) .eq. 'A' ) ) cycle

            qi = q(i)
            qj = q(j)

            xi = x(i,m)
            yi = y(i,m)
            zi = z(i,m)

            m1 = mbox(1,i,m)
            m2 = mbox(2,i,m)
            m3 = mbox(3,i,m)

            call pbc_unfold( xi, yi, zi, m1, m2, m3 )

            xj = x(j,m)
            yj = y(j,m)
            zj = z(j,m)

            m1 = mbox(1,j,m)
            m2 = mbox(2,j,m)
            m3 = mbox(3,j,m)

            call pbc_unfold( xj, yj, zj, m1, m2, m3 )

            xij = xi - xj
            yij = yi - yj
            zij = zi - zj

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            rinv = 1.d0/rij

            uij = + qi*qj*rinv

            duij = - uij*rinv

            fxi = - duij*xij*rinv
            fyi = - duij*yij*rinv
            fzi = - duij*zij*rinv

            pot_b(m) = pot_b(m) + uij

            fx_b(i,m) = fx_b(i,m) + fxi
            fy_b(i,m) = fy_b(i,m) + fyi
            fz_b(i,m) = fz_b(i,m) + fzi

            fx_b(j,m) = fx_b(j,m) - fxi
            fy_b(j,m) = fy_b(j,m) - fyi
            fz_b(j,m) = fz_b(j,m) - fzi

            vir_b(1,1) = vir_b(1,1) + fxi*xij
            vir_b(1,2) = vir_b(1,2) + fxi*yij
            vir_b(1,3) = vir_b(1,3) + fxi*zij
            vir_b(2,1) = vir_b(2,1) + fyi*xij
            vir_b(2,2) = vir_b(2,2) + fyi*yij
            vir_b(2,3) = vir_b(2,3) + fyi*zij
            vir_b(3,1) = vir_b(3,1) + fzi*xij
            vir_b(3,2) = vir_b(3,2) + fzi*yij
            vir_b(3,3) = vir_b(3,3) + fzi*zij

         end do
         end do

         end do

!-----------------------------------------------------------------------
!        /*  subtract bonded charge pairs in B                        */
!-----------------------------------------------------------------------

         do m = 1, nbead
         do k = 1, nbcp

            i  = i_bcp(k)
            j  = j_bcp(k)

            if ( ( layer(i)(1:1) .eq. 'A' ) .or. &
     &           ( layer(j)(1:1) .eq. 'A' ) ) cycle

            factor = factor_bcp(k)

            qi = q(i)
            qj = q(j)

            xi = x(i,m)
            yi = y(i,m)
            zi = z(i,m)

            m1 = mbox(1,i,m)
            m2 = mbox(2,i,m)
            m3 = mbox(3,i,m)

            call pbc_unfold( xi, yi, zi, m1, m2, m3 )

            xj = x(j,m)
            yj = y(j,m)
            zj = z(j,m)

            m1 = mbox(1,j,m)
            m2 = mbox(2,j,m)
            m3 = mbox(3,j,m)

            call pbc_unfold( xj, yj, zj, m1, m2, m3 )

            xij = xi - xj
            yij = yi - yj
            zij = zi - zj

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            rinv = 1.d0/rij

            uij = (factor - 1.d0) * qi*qj*rinv

            pot_b(m) = pot_b(m) + uij

            duij = - uij*rinv

            fxi = - duij*xij*rinv
            fyi = - duij*yij*rinv
            fzi = - duij*zij*rinv

            fx_b(i,m) = fx_b(i,m) + fxi
            fy_b(i,m) = fy_b(i,m) + fyi
            fz_b(i,m) = fz_b(i,m) + fzi

            fx_b(j,m) = fx_b(j,m) - fxi
            fy_b(j,m) = fy_b(j,m) - fyi
            fz_b(j,m) = fz_b(j,m) - fzi

            vir_b(1,1) = vir_b(1,1) + fxi*xij
            vir_b(1,2) = vir_b(1,2) + fxi*yij
            vir_b(1,3) = vir_b(1,3) + fxi*zij
            vir_b(2,1) = vir_b(2,1) + fyi*xij
            vir_b(2,2) = vir_b(2,2) + fyi*yij
            vir_b(2,3) = vir_b(2,3) + fyi*zij
            vir_b(3,1) = vir_b(3,1) + fzi*xij
            vir_b(3,2) = vir_b(3,2) + fzi*yij
            vir_b(3,3) = vir_b(3,3) + fzi*zij

         end do
         end do

      end if

      return
      end





!***********************************************************************
      subroutine force_me_coulomb_subtract( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, nbead, mbox

      use mm_variables, only : &
     &   q, factor_bcp, ncharge, nbcp, i_q, i_bcp, j_bcp

      use qmmm_variables, only : &
     &   layer, fx_b, fy_b, fz_b, pot_b, vir_b

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, m1, m2, m3, ioption

      real(8) :: qi, qj, xij, yij, zij, rij, rinv, uij, duij, &
     &           fxi, fyi, fzi, factor, xi, yi, zi, xj, yj, zj

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( ncharge .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   main loop:  subtract A-A contribution                      */
!-----------------------------------------------------------------------

      if ( ioption .eq. 0 ) then

         do m = 1, nbead

         do k = 1, ncharge-1
         do l = k+1, ncharge

            i  = i_q(k)
            j  = i_q(l)

            if ( ( layer(i)(1:1) .eq. 'B' ) .or. &
     &           ( layer(j)(1:1) .eq. 'B' ) ) cycle

            qi = q(i)
            qj = q(j)

            xi = x(i,m)
            yi = y(i,m)
            zi = z(i,m)

            m1 = mbox(1,i,m)
            m2 = mbox(2,i,m)
            m3 = mbox(3,i,m)

            call pbc_unfold( xi, yi, zi, m1, m2, m3 )

            xj = x(j,m)
            yj = y(j,m)
            zj = z(j,m)

            m1 = mbox(1,j,m)
            m2 = mbox(2,j,m)
            m3 = mbox(3,j,m)

            call pbc_unfold( xj, yj, zj, m1, m2, m3 )

            xij = xi - xj
            yij = yi - yj
            zij = zi - zj

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            rinv = 1.d0/rij

            uij = + qi*qj*rinv

            duij = - uij*rinv

            fxi = - duij*xij*rinv
            fyi = - duij*yij*rinv
            fzi = - duij*zij*rinv

            pot_b(m) = pot_b(m) - uij

            fx_b(i,m) = fx_b(i,m) - fxi
            fy_b(i,m) = fy_b(i,m) - fyi
            fz_b(i,m) = fz_b(i,m) - fzi

            fx_b(j,m) = fx_b(j,m) + fxi
            fy_b(j,m) = fy_b(j,m) + fyi
            fz_b(j,m) = fz_b(j,m) + fzi

            vir_b(1,1) = vir_b(1,1) - fxi*xij
            vir_b(1,2) = vir_b(1,2) - fxi*yij
            vir_b(1,3) = vir_b(1,3) - fxi*zij
            vir_b(2,1) = vir_b(2,1) - fyi*xij
            vir_b(2,2) = vir_b(2,2) - fyi*yij
            vir_b(2,3) = vir_b(2,3) - fyi*zij
            vir_b(3,1) = vir_b(3,1) - fzi*xij
            vir_b(3,2) = vir_b(3,2) - fzi*yij
            vir_b(3,3) = vir_b(3,3) - fzi*zij

         end do
         end do

         end do

!-----------------------------------------------------------------------
!        /*  add bonded charge pairs in A                             */
!-----------------------------------------------------------------------

         do m = 1, nbead
         do k = 1, nbcp

            i  = i_bcp(k)
            j  = j_bcp(k)

            if ( ( layer(i)(1:1) .eq. 'B' ) .or. &
     &           ( layer(j)(1:1) .eq. 'B' ) ) cycle

            factor = factor_bcp(k)

            qi = q(i)
            qj = q(j)

            xi = x(i,m)
            yi = y(i,m)
            zi = z(i,m)

            m1 = mbox(1,i,m)
            m2 = mbox(2,i,m)
            m3 = mbox(3,i,m)

            call pbc_unfold( xi, yi, zi, m1, m2, m3 )

            xj = x(j,m)
            yj = y(j,m)
            zj = z(j,m)

            m1 = mbox(1,j,m)
            m2 = mbox(2,j,m)
            m3 = mbox(3,j,m)

            call pbc_unfold( xj, yj, zj, m1, m2, m3 )

            xij = xi - xj
            yij = yi - yj
            zij = zi - zj

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            rinv = 1.d0/rij

            uij = (factor - 1.d0) * qi*qj*rinv

            pot_b(m) = pot_b(m) - uij

            duij = - uij*rinv

            fxi = - duij*xij*rinv
            fyi = - duij*yij*rinv
            fzi = - duij*zij*rinv

            fx_b(i,m) = fx_b(i,m) - fxi
            fy_b(i,m) = fy_b(i,m) - fyi
            fz_b(i,m) = fz_b(i,m) - fzi

            fx_b(j,m) = fx_b(j,m) + fxi
            fy_b(j,m) = fy_b(j,m) + fyi
            fz_b(j,m) = fz_b(j,m) + fzi

            vir_b(1,1) = vir_b(1,1) - fxi*xij
            vir_b(1,2) = vir_b(1,2) - fxi*yij
            vir_b(1,3) = vir_b(1,3) - fxi*zij
            vir_b(2,1) = vir_b(2,1) - fyi*xij
            vir_b(2,2) = vir_b(2,2) - fyi*yij
            vir_b(2,3) = vir_b(2,3) - fyi*zij
            vir_b(3,1) = vir_b(3,1) - fzi*xij
            vir_b(3,2) = vir_b(3,2) - fzi*yij
            vir_b(3,3) = vir_b(3,3) - fzi*zij

         end do
         end do

      end if

      return
      end





!***********************************************************************
      subroutine force_me_ewald
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use mm_variables, only : ncharge, nbox_ewald, ioption_ewald

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!-----------------------------------------------------------------------
!     /*   set up Ewald parameters                                    */
!-----------------------------------------------------------------------

      if ( ncharge .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   real space contribution of Ewald sum                       */
!-----------------------------------------------------------------------

      if ( nbox_ewald(1)*nbox_ewald(2)*nbox_ewald(3) .eq. 1 ) then
         call force_me_ewald_rs_pair
      else
         call force_me_ewald_rs
      end if

!-----------------------------------------------------------------------
!     /*   Fourier space contribution of Ewald sum                    */
!-----------------------------------------------------------------------

      call force_me_ewald_fs

!-----------------------------------------------------------------------
!     /*   self contribution of Ewald sum                             */
!-----------------------------------------------------------------------

      call force_me_ewald_self

!-----------------------------------------------------------------------
!     /*   charged system contribution of Ewald sum                   */
!-----------------------------------------------------------------------

      call force_me_ewald_charge

!-----------------------------------------------------------------------
!     /*   dipole contribution of Ewald sum                           */
!-----------------------------------------------------------------------

      if ( ioption_ewald .eq. 1 ) call force_me_ewald_dipole

      return
      end





!***********************************************************************
      subroutine force_me_ewald_setup
!***********************************************************************

      call force_ewald_setup

      return
      end





!***********************************************************************
      subroutine force_me_ewald_rs
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, box, nbead

      use mm_variables, only : &
     &   q, rcut_ewald, alpha_ewald, bigbox, bigboxinv, &
     &   nbox_ewald, i_q, ncharge, factor_bcp, i_bcp, j_bcp, nbcp

      use qmmm_variables, only : &
     &   fx_b, fy_b, fz_b, pot_b, vir_b

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, jx, jy, jz, j2

      real(8) :: qiqj, xij, yij, zij, aij, bij, cij, &
     &           r2, r, rinv, rinv2, rinv3, ar, erf_0, erf_1, erfc_ar, &
     &           pot_m, factor, fxi, fyi, fzi, rcut_ewald2

      real(8) :: duij, qi, qj, rij, uij

!-----------------------------------------------------------------------
!     /*   parameters                                                 */
!-----------------------------------------------------------------------

      bigbox(:,1) = dble(nbox_ewald(1))*box(:,1)
      bigbox(:,2) = dble(nbox_ewald(2))*box(:,2)
      bigbox(:,3) = dble(nbox_ewald(3))*box(:,3)

      call inv3 ( bigbox, bigboxinv )

      rcut_ewald2 = rcut_ewald*rcut_ewald

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         pot_m = 0.d0

         do k = 1, ncharge
         do l = 1, ncharge

            i = i_q(k)
            j = i_q(l)

            qiqj = q(i)*q(j)

            if ( qiqj .eq. 0.d0 ) cycle

            do jx = 0, nbox_ewald(1)-1
            do jy = 0, nbox_ewald(2)-1
            do jz = 0, nbox_ewald(3)-1

               j2 = jx*jx + jy*jy + jz*jz

               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

               r2 = xij*xij + yij*yij + zij*zij

               if ( r2 .gt. rcut_ewald2 ) cycle

               r = sqrt(r2)

               rinv  = 1.d0/r
               rinv2 = rinv*rinv
               rinv3 = rinv*rinv2

               ar = alpha_ewald*r

               erfc_ar = 1.d0 - erf_0(ar)

               pot_m = pot_m + qiqj*erfc_ar*rinv

               factor = erfc_ar*rinv3 + alpha_ewald*erf_1(ar)*rinv2
               factor = qiqj*factor

               fxi = factor*xij
               fyi = factor*yij
               fzi = factor*zij

               fx_b(i,m) = fx_b(i,m) + fxi
               fy_b(i,m) = fy_b(i,m) + fyi
               fz_b(i,m) = fz_b(i,m) + fzi

               vir_b(1,1) = vir_b(1,1) + 0.5d0*fxi*xij
               vir_b(1,2) = vir_b(1,2) + 0.5d0*fxi*yij
               vir_b(1,3) = vir_b(1,3) + 0.5d0*fxi*zij
               vir_b(2,1) = vir_b(2,1) + 0.5d0*fyi*xij
               vir_b(2,2) = vir_b(2,2) + 0.5d0*fyi*yij
               vir_b(2,3) = vir_b(2,3) + 0.5d0*fyi*zij
               vir_b(3,1) = vir_b(3,1) + 0.5d0*fzi*xij
               vir_b(3,2) = vir_b(3,2) + 0.5d0*fzi*yij
               vir_b(3,3) = vir_b(3,3) + 0.5d0*fzi*zij

            end do
            end do
            end do

         end do
         end do

         pot_b(m) = pot_b(m) + 0.5d0*pot_m

      end do

!-----------------------------------------------------------------------
!     /*   main loop:  subtract bonded charge pairs                   */
!-----------------------------------------------------------------------

      do m = 1, nbead

         do k = 1, nbcp

            i  = i_bcp(k)
            j  = j_bcp(k)

            factor = factor_bcp(k)

            qi = q(i)
            qj = q(j)

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom( xij, yij, zij )

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            rinv = 1.d0/rij

            uij = (factor - 1.d0) * qi*qj*rinv

            pot_b(m) = pot_b(m) + uij

            duij = - uij*rinv

            fxi = - duij*xij*rinv
            fyi = - duij*yij*rinv
            fzi = - duij*zij*rinv

            fx_b(i,m) = fx_b(i,m) + fxi
            fy_b(i,m) = fy_b(i,m) + fyi
            fz_b(i,m) = fz_b(i,m) + fzi

            fx_b(j,m) = fx_b(j,m) - fxi
            fy_b(j,m) = fy_b(j,m) - fyi
            fz_b(j,m) = fz_b(j,m) - fzi

            vir_b(1,1) = vir_b(1,1) + fxi*xij
            vir_b(1,2) = vir_b(1,2) + fxi*yij
            vir_b(1,3) = vir_b(1,3) + fxi*zij
            vir_b(2,1) = vir_b(2,1) + fyi*xij
            vir_b(2,2) = vir_b(2,2) + fyi*yij
            vir_b(2,3) = vir_b(2,3) + fyi*zij
            vir_b(3,1) = vir_b(3,1) + fzi*xij
            vir_b(3,2) = vir_b(3,2) + fzi*yij
            vir_b(3,3) = vir_b(3,3) + fzi*zij

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_me_ewald_rs_pair
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, nbead

      use mm_variables, only : &
     &   q, rcut_ewald, alpha_ewald, i_q, ncharge, &
     &   factor_bcp, i_bcp, j_bcp, nbcp

      use qmmm_variables, only : &
     &   fx_b, fy_b, fz_b, pot_b, vir_b

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m

      real(8) :: rcut_ewald2, qiqj, xij, yij, zij, r2, r, rinv, &
     &           rinv2, rinv3, ar, erfc_ar, erf_0, erf_1, factor, &
     &           fxi, fyi, fzi, qi, qj, rij, uij, duij

!-----------------------------------------------------------------------
!     /*   parameters                                                 */
!-----------------------------------------------------------------------

      rcut_ewald2 = rcut_ewald*rcut_ewald

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         do k = 1, ncharge-1
         do l = k+1, ncharge

            i  = i_q(k)
            j  = i_q(l)

            qiqj = q(i)*q(j)

            if ( qiqj .eq. 0.d0 ) cycle

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom( xij, yij, zij )

            r2 = xij*xij + yij*yij + zij*zij

            if ( r2 .gt. rcut_ewald2 ) cycle

            r = sqrt(r2)

            rinv  = 1.d0/r
            rinv2 = rinv*rinv
            rinv3 = rinv*rinv2

            ar = alpha_ewald*r

            erfc_ar = 1.d0 - erf_0(ar)

            pot_b(m) = pot_b(m) + qiqj*erfc_ar*rinv

            factor = erfc_ar*rinv3 + alpha_ewald*erf_1(ar)*rinv2

            factor = qiqj*factor

            fxi = factor*xij
            fyi = factor*yij
            fzi = factor*zij

            fx_b(i,m) = fx_b(i,m) + fxi
            fy_b(i,m) = fy_b(i,m) + fyi
            fz_b(i,m) = fz_b(i,m) + fzi

            fx_b(j,m) = fx_b(j,m) - fxi
            fy_b(j,m) = fy_b(j,m) - fyi
            fz_b(j,m) = fz_b(j,m) - fzi

            vir_b(1,1) = vir_b(1,1) + fxi*xij
            vir_b(1,2) = vir_b(1,2) + fxi*yij
            vir_b(1,3) = vir_b(1,3) + fxi*zij
            vir_b(2,1) = vir_b(2,1) + fyi*xij
            vir_b(2,2) = vir_b(2,2) + fyi*yij
            vir_b(2,3) = vir_b(2,3) + fyi*zij
            vir_b(3,1) = vir_b(3,1) + fzi*xij
            vir_b(3,2) = vir_b(3,2) + fzi*yij
            vir_b(3,3) = vir_b(3,3) + fzi*zij

         end do
         end do

      end do

!-----------------------------------------------------------------------
!     /*   main loop:  subtract bonded charge pairs                   */
!-----------------------------------------------------------------------

      do m = 1, nbead

         do k = 1, nbcp

            i  = i_bcp(k)
            j  = j_bcp(k)

            factor = factor_bcp(k)

            qi = q(i)
            qj = q(j)

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom( xij, yij, zij )

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            rinv = 1.d0/rij

            uij = (factor - 1.d0) * qi*qj*rinv

            pot_b(m) = pot_b(m) + uij

            duij = - uij*rinv

            fxi = - duij*xij*rinv
            fyi = - duij*yij*rinv
            fzi = - duij*zij*rinv

            fx_b(i,m) = fx_b(i,m) + fxi
            fy_b(i,m) = fy_b(i,m) + fyi
            fz_b(i,m) = fz_b(i,m) + fzi

            fx_b(j,m) = fx_b(j,m) - fxi
            fy_b(j,m) = fy_b(j,m) - fyi
            fz_b(j,m) = fz_b(j,m) - fzi

            vir_b(1,1) = vir_b(1,1) + fxi*xij
            vir_b(1,2) = vir_b(1,2) + fxi*yij
            vir_b(1,3) = vir_b(1,3) + fxi*zij
            vir_b(2,1) = vir_b(2,1) + fyi*xij
            vir_b(2,2) = vir_b(2,2) + fyi*yij
            vir_b(2,3) = vir_b(2,3) + fyi*zij
            vir_b(3,1) = vir_b(3,1) + fzi*xij
            vir_b(3,2) = vir_b(3,2) + fzi*yij
            vir_b(3,3) = vir_b(3,3) + fzi*zij

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_me_ewald_self
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : pi, nbead

      use mm_variables, only : alpha_ewald, q, i_q, ncharge

      use qmmm_variables, only : pot_b

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, m, k

      real(8) :: factor, q2sum

!-----------------------------------------------------------------------
!     /*   parameters                                                 */
!-----------------------------------------------------------------------

      factor = alpha_ewald/sqrt(pi)

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         q2sum = 0.d0

         do k = 1, ncharge

            i  = i_q(k)

            q2sum = q2sum + q(i)*q(i)

         end do

         pot_b(m) = pot_b(m) - q2sum*factor

      end do

      return
      end





!***********************************************************************
      subroutine force_me_ewald_charge
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : pi, volume, nbead

      use mm_variables, only : alpha_ewald, q, i_q, ncharge

      use qmmm_variables, only : pot_b, vir_b

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, m, k

      real(8) :: factor, qsum

!-----------------------------------------------------------------------
!     /*   parameters                                                 */
!-----------------------------------------------------------------------

      factor = pi/(2.d0*volume*alpha_ewald*alpha_ewald)

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         qsum = 0.d0

         do k = 1, ncharge

            i  = i_q(k)

            qsum = qsum + q(i)

         end do

         pot_b(m) = pot_b(m) - qsum*qsum*factor

         vir_b(1,1) = vir_b(1,1) - qsum*qsum*factor
         vir_b(2,2) = vir_b(2,2) - qsum*qsum*factor
         vir_b(3,3) = vir_b(3,3) - qsum*qsum*factor

      end do

      return
      end





!***********************************************************************
      subroutine force_me_ewald_fs
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, pi, boxinv, volume, nbead, natom

      use mm_variables, only : &
     &   eigax, eigay, eigaz, eigbx, eigby, eigbz, eigcx, eigcy, eigcz, &
     &   alpha_ewald, q, i_q, lmax_ewald, ncharge

      use qmmm_variables, only : &
     &   fx_b, fy_b, fz_b, pot_b, vir_b

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: m, k, i, l, l2, la, lb, lc

      real(8) :: ax, ay, az, bx, by, bz, cx, cy, cz, a2, b2, c2, &
     &           al2, bl2, cl2, factor_1, factor_2, factor_3, factor_4, &
     &           factor_5, factor_6, factor_7, factor_8, &
     &           gx, gy, gz, g2, g2max, qcos, qsin, fxi, fyi, fzi, &
     &           cos_gxyz, sin_gxyz, qexp2

      integer, save :: iset = 0
      integer, save :: lmax_ewald_save(3)

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

         if ( .not. allocated( eigax ) ) &
     &      allocate( eigax(natom,-lmax_ewald(1):lmax_ewald(1)) )
         if ( .not. allocated( eigay ) ) &
     &      allocate( eigay(natom,-lmax_ewald(1):lmax_ewald(1)) )
         if ( .not. allocated( eigaz ) ) &
     &      allocate( eigaz(natom,-lmax_ewald(1):lmax_ewald(1)) )
         if ( .not. allocated( eigbx ) ) &
     &      allocate( eigbx(natom,-lmax_ewald(2):lmax_ewald(2)) )
         if ( .not. allocated( eigby ) ) &
     &      allocate( eigby(natom,-lmax_ewald(2):lmax_ewald(2)) )
         if ( .not. allocated( eigbz ) ) &
     &      allocate( eigbz(natom,-lmax_ewald(2):lmax_ewald(2)) )
         if ( .not. allocated( eigcx ) ) &
     &      allocate( eigcx(natom,-lmax_ewald(3):lmax_ewald(3)) )
         if ( .not. allocated( eigcy ) ) &
     &      allocate( eigcy(natom,-lmax_ewald(3):lmax_ewald(3)) )
         if ( .not. allocated( eigcz ) ) &
     &      allocate( eigcz(natom,-lmax_ewald(3):lmax_ewald(3)) )

         lmax_ewald_save(1) = lmax_ewald(1)
         lmax_ewald_save(2) = lmax_ewald(2)
         lmax_ewald_save(3) = lmax_ewald(3)

         iset = 1

      else

         if ( lmax_ewald(1) .ne. lmax_ewald_save(1) ) then

            if ( allocated( eigax ) ) &
     &         deallocate( eigax )
            if ( allocated( eigay ) ) &
     &         deallocate( eigay )
            if ( allocated( eigaz ) ) &
     &         deallocate( eigaz )

            if ( .not. allocated( eigax ) ) &
     &         allocate( eigax(natom,-lmax_ewald(1):lmax_ewald(1)) )
            if ( .not. allocated( eigay ) ) &
     &         allocate( eigay(natom,-lmax_ewald(1):lmax_ewald(1)) )
            if ( .not. allocated( eigaz ) ) &
     &         allocate( eigaz(natom,-lmax_ewald(1):lmax_ewald(1)) )

         end if

         if ( lmax_ewald(2) .ne. lmax_ewald_save(2) ) then

            if ( allocated( eigbx ) ) &
     &         deallocate( eigbx )
            if ( allocated( eigby ) ) &
     &         deallocate( eigby )
            if ( allocated( eigbz ) ) &
     &         deallocate( eigbz )

            if ( .not. allocated( eigbx ) ) &
     &         allocate( eigbx(natom,-lmax_ewald(2):lmax_ewald(2)) )
            if ( .not. allocated( eigby ) ) &
     &         allocate( eigby(natom,-lmax_ewald(2):lmax_ewald(2)) )
            if ( .not. allocated( eigbz ) ) &
     &         allocate( eigbz(natom,-lmax_ewald(2):lmax_ewald(2)) )

         end if

         if ( lmax_ewald(3) .ne. lmax_ewald_save(3) ) then

            if ( allocated( eigcx ) ) &
     &         deallocate( eigcx )
            if ( allocated( eigcy ) ) &
     &         deallocate( eigcy )
            if ( allocated( eigcz ) ) &
     &         deallocate( eigcz )

            if ( .not. allocated( eigcx ) ) &
     &         allocate( eigcx(natom,-lmax_ewald(3):lmax_ewald(3)) )
            if ( .not. allocated( eigcy ) ) &
     &         allocate( eigcy(natom,-lmax_ewald(3):lmax_ewald(3)) )
            if ( .not. allocated( eigcz ) ) &
     &         allocate( eigcz(natom,-lmax_ewald(3):lmax_ewald(3)) )

         end if

         lmax_ewald_save(1) = lmax_ewald(1)
         lmax_ewald_save(2) = lmax_ewald(2)
         lmax_ewald_save(3) = lmax_ewald(3)

      end if

!-----------------------------------------------------------------------
!     /*   loop of beads: start                                       */
!-----------------------------------------------------------------------

      do m = 1, nbead

!-----------------------------------------------------------------------
!        /*   parameters                                              */
!-----------------------------------------------------------------------

         ax = 2.d0*pi*boxinv(1,1)
         ay = 2.d0*pi*boxinv(1,2)
         az = 2.d0*pi*boxinv(1,3)
         bx = 2.d0*pi*boxinv(2,1)
         by = 2.d0*pi*boxinv(2,2)
         bz = 2.d0*pi*boxinv(2,3)
         cx = 2.d0*pi*boxinv(3,1)
         cy = 2.d0*pi*boxinv(3,2)
         cz = 2.d0*pi*boxinv(3,3)

         a2 = ax*ax + ay*ay + az*az
         b2 = bx*bx + by*by + bz*bz
         c2 = cx*cx + cy*cy + cz*cz

         al2 = a2*lmax_ewald(1)**2
         bl2 = b2*lmax_ewald(2)**2
         cl2 = c2*lmax_ewald(3)**2

         g2max = min( al2, bl2, cl2 )

!-----------------------------------------------------------------------
!        /*   main loop                                               */
!-----------------------------------------------------------------------

         do k = 1, ncharge

            i  = i_q(k)

            eigax(i, 0)  = (1.d0,0.d0)
            eigay(i, 0)  = (1.d0,0.d0)
            eigaz(i, 0)  = (1.d0,0.d0)
            eigbx(i, 0)  = (1.d0,0.d0)
            eigby(i, 0)  = (1.d0,0.d0)
            eigbz(i, 0)  = (1.d0,0.d0)
            eigcx(i, 0)  = (1.d0,0.d0)
            eigcy(i, 0)  = (1.d0,0.d0)
            eigcz(i, 0)  = (1.d0,0.d0)

            eigax(i, 1)  = dcmplx ( cos(ax*x(i,m)), sin(ax*x(i,m)) )
            eigay(i, 1)  = dcmplx ( cos(ay*y(i,m)), sin(ay*y(i,m)) )
            eigaz(i, 1)  = dcmplx ( cos(az*z(i,m)), sin(az*z(i,m)) )
            eigbx(i, 1)  = dcmplx ( cos(bx*x(i,m)), sin(bx*x(i,m)) )
            eigby(i, 1)  = dcmplx ( cos(by*y(i,m)), sin(by*y(i,m)) )
            eigbz(i, 1)  = dcmplx ( cos(bz*z(i,m)), sin(bz*z(i,m)) )
            eigcx(i, 1)  = dcmplx ( cos(cx*x(i,m)), sin(cx*x(i,m)) )
            eigcy(i, 1)  = dcmplx ( cos(cy*y(i,m)), sin(cy*y(i,m)) )
            eigcz(i, 1)  = dcmplx ( cos(cz*z(i,m)), sin(cz*z(i,m)) )

            eigax(i,-1)  = dconjg ( eigax(i,1) )
            eigay(i,-1)  = dconjg ( eigay(i,1) )
            eigaz(i,-1)  = dconjg ( eigaz(i,1) )
            eigbx(i,-1)  = dconjg ( eigbx(i,1) )
            eigby(i,-1)  = dconjg ( eigby(i,1) )
            eigbz(i,-1)  = dconjg ( eigbz(i,1) )
            eigcx(i,-1)  = dconjg ( eigcx(i,1) )
            eigcy(i,-1)  = dconjg ( eigcy(i,1) )
            eigcz(i,-1)  = dconjg ( eigcz(i,1) )

            do l = 2, lmax_ewald(1)
               eigax(i, l)  = eigax(i,l-1)*eigax(i,1)
               eigay(i, l)  = eigay(i,l-1)*eigay(i,1)
               eigaz(i, l)  = eigaz(i,l-1)*eigaz(i,1)
               eigax(i,-l)  = dconjg ( eigax(i,l) )
               eigay(i,-l)  = dconjg ( eigay(i,l) )
               eigaz(i,-l)  = dconjg ( eigaz(i,l) )
            end do
            do l = 2, lmax_ewald(2)
               eigbx(i, l)  = eigbx(i,l-1)*eigbx(i,1)
               eigby(i, l)  = eigby(i,l-1)*eigby(i,1)
               eigbz(i, l)  = eigbz(i,l-1)*eigbz(i,1)
               eigbx(i,-l)  = dconjg ( eigbx(i,l) )
               eigby(i,-l)  = dconjg ( eigby(i,l) )
               eigbz(i,-l)  = dconjg ( eigbz(i,l) )
            end do
            do l = 2, lmax_ewald(3)
               eigcx(i, l)  = eigcx(i,l-1)*eigcx(i,1)
               eigcy(i, l)  = eigcy(i,l-1)*eigcy(i,1)
               eigcz(i, l)  = eigcz(i,l-1)*eigcz(i,1)
               eigcx(i,-l)  = dconjg ( eigcx(i,l) )
               eigcy(i,-l)  = dconjg ( eigcy(i,l) )
               eigcz(i,-l)  = dconjg ( eigcz(i,l) )
            end do

         end do

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

         factor_1 = (4.d0*pi)/(2.d0*volume)

         do la =              0, lmax_ewald(1)
         do lb = -lmax_ewald(2), lmax_ewald(2)
         do lc = -lmax_ewald(3), lmax_ewald(3)

            l2 = la*la + lb*lb + lc*lc

            if ( l2 .eq. 0 ) cycle

            if ( la .eq. 0 ) then
               factor_2 = 1.d0
            else
               factor_2 = 2.d0
            end if

            gx = ax*la + bx*lb + cx*lc
            gy = ay*la + by*lb + cy*lc
            gz = az*la + bz*lb + cz*lc

            g2 = gx*gx + gy*gy + gz*gz

            if ( g2 .gt. g2max ) cycle

            factor_3 = exp(-g2/(4.d0*alpha_ewald*alpha_ewald))/g2

            qcos = 0.d0
            qsin = 0.d0

            do k = 1, ncharge

               i  = i_q(k)

               cos_gxyz = dreal(eigax(i,la)*eigbx(i,lb)*eigcx(i,lc) &
     &                         *eigay(i,la)*eigby(i,lb)*eigcy(i,lc) &
     &                         *eigaz(i,la)*eigbz(i,lb)*eigcz(i,lc))
               sin_gxyz = dimag(eigax(i,la)*eigbx(i,lb)*eigcx(i,lc) &
     &                         *eigay(i,la)*eigby(i,lb)*eigcy(i,lc) &
     &                         *eigaz(i,la)*eigbz(i,lb)*eigcz(i,lc))

               qcos = qcos + q(i)*cos_gxyz
               qsin = qsin + q(i)*sin_gxyz

            end do

            qexp2 = qcos*qcos + qsin*qsin

            pot_b(m) = pot_b(m) + factor_1*factor_2*factor_3*qexp2

            do k = 1, ncharge

               i  = i_q(k)

               cos_gxyz = dreal(eigax(i,la)*eigbx(i,lb)*eigcx(i,lc) &
     &                         *eigay(i,la)*eigby(i,lb)*eigcy(i,lc) &
     &                         *eigaz(i,la)*eigbz(i,lb)*eigcz(i,lc))
               sin_gxyz = dimag(eigax(i,la)*eigbx(i,lb)*eigcx(i,lc) &
     &                         *eigay(i,la)*eigby(i,lb)*eigcy(i,lc) &
     &                         *eigaz(i,la)*eigbz(i,lb)*eigcz(i,lc))

               factor_4 = sin_gxyz*qcos - cos_gxyz*qsin

               factor_5 = 2.d0*q(i)*factor_1*factor_2*factor_3*factor_4

               fxi = factor_5*gx
               fyi = factor_5*gy
               fzi = factor_5*gz

               fx_b(i,m) = fx_b(i,m) + fxi
               fy_b(i,m) = fy_b(i,m) + fyi
               fz_b(i,m) = fz_b(i,m) + fzi

            end do

            factor_6 = factor_1*factor_2*factor_3*qexp2
            factor_7 = 1.d0/(4.d0*alpha_ewald*alpha_ewald)
            factor_8 = 2.d0 * ( 1.d0 + factor_7*g2 ) / g2

            vir_b(1,1) = vir_b(1,1) &
     &                 + factor_6 * ( 1.d0 - factor_8*gx*gx )
            vir_b(1,2) = vir_b(1,2) &
     &                  - factor_6 * factor_8*gx*gy
            vir_b(1,3) = vir_b(1,3) &
     &                  - factor_6 * factor_8*gx*gz
            vir_b(2,1) = vir_b(2,1) &
     &                  - factor_6 * factor_8*gy*gx
            vir_b(2,2) = vir_b(2,2) &
     &                  + factor_6 * ( 1.d0 - factor_8*gy*gy )
            vir_b(2,3) = vir_b(2,3) &
     &                  - factor_6 * factor_8*gy*gz
            vir_b(3,1) = vir_b(3,1) &
     &                  - factor_6 * factor_8*gz*gx
            vir_b(3,2) = vir_b(3,2) &
     &                  - factor_6 * factor_8*gz*gy
            vir_b(3,3) = vir_b(3,3) &
     &                  + factor_6 * ( 1.d0 - factor_8*gz*gz )

         end do
         end do
         end do

!-----------------------------------------------------------------------
!     /*   loop of beads: end                                         */
!-----------------------------------------------------------------------

      end do

      return
      end





!***********************************************************************
      subroutine me_dipole( ioption )
!***********************************************************************
!=======================================================================
!
!     calculate dipole moment
!
!=======================================================================

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, nbead, mbox

      use mm_variables, only : &
     &   q, i_q, ncharge

      use qmmm_variables, only : &
     &   layer, dipx_b, dipy_b, dipz_b

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: m, i, k, m1, m2, m3, ioption

      real(8) :: xb, yb, zb

!-----------------------------------------------------------------------
!     /*   calculate mm dipole moment                                 */
!-----------------------------------------------------------------------

      do m = 1, nbead

         dipx_b(m) = 0.d0
         dipy_b(m) = 0.d0
         dipz_b(m) = 0.d0

         do k = 1, ncharge

            i  = i_q(k)

            if ( ioption .eq. 0 ) then
               if ( layer(i)(1:1) .eq. 'A' ) cycle
            end if

            xb = x(i,m)
            yb = y(i,m)
            zb = z(i,m)

            m1 = mbox(1,i,m)
            m2 = mbox(2,i,m)
            m3 = mbox(3,i,m)

            call pbc_unfold( xb, yb, zb, m1, m2, m3 )

            dipx_b(m) = dipx_b(m) + q(i)*xb
            dipy_b(m) = dipy_b(m) + q(i)*yb
            dipz_b(m) = dipz_b(m) + q(i)*zb

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_me_improper( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, x, y, z, nbead

      use mm_variables, only : &
     &   eq_improper, fc_improper, i_improper, j_improper, &
     &   k_improper, l_improper, nimproper

      use qmmm_variables, only : &
     &   layer, fx_b, fy_b, fz_b, pot_b, vir_b

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, n, ioption

      real(8) :: tiny_value = 1.d-4

      real(8) :: xij, yij, zij, xkj, ykj, zkj, xlj, ylj, zlj, &
     &           xijk, yijk, zijk, xjkl, yjkl, zjkl, rijk2, rjkl2, &
     &           rijkl2, rijk2inv, rjkl2inv, rijkl2inv, cos_phi, phi, &
     &           fxi, fyi, fzi, fxj, fyj, fzj, fxk, fyk, fzk, &
     &           fxl, fyl, fzl

      real(8) :: dphi, factor_1, factor_2, factor_3, factor_4, &
     &           px1, py1, pz1, px2, py2, pz2, px3, py3, pz3, &
     &           px4, py4, pz4, px5, py5, pz5, px6, py6, pz6

      real(8) :: ax, ay, az, a1, a2, xkl, ykl, zkl, xki, yki, zki

      real(8) :: daxdxi, daxdyi, daxdzi, daydxi, daydyi, daydzi, &
     &           dazdxi, dazdyi, dazdzi, dadxi, dadyi, dadzi, &
     &           daxdxj, daxdyj, daxdzj, daydxj, daydyj, daydzj, &
     &           dazdxj, dazdyj, dazdzj, dadxj, dadyj, dadzj, &
     &           daxdxl, daxdyl, daxdzl, daydxl, daydyl, daydzl, &
     &           dazdxl, dazdyl, dazdzl, dadxl, dadyl, dadzl

      real(8) :: f1, sin_phi, sign_phi

!      real(8)::  daxdxk, daxdyk, daxdzk, daydxk, daydyk, daydzk,
!     &           dazdxk, dazdyk, dazdzk, dadxk, dadyk, dadzk

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( nimproper .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         do n = 1, nimproper

            i = i_improper(n)
            j = j_improper(n)
            k = k_improper(n)
            l = l_improper(n)

            if ( ioption .eq. 0 ) then
               if ( ( layer(i)(1:1) .eq. 'A' ) .and. &
     &              ( layer(j)(1:1) .eq. 'A' ) .and. &
     &              ( layer(k)(1:1) .eq. 'A' ) .and. &
     &              ( layer(l)(1:1) .eq. 'A' ) ) cycle
            end if

!-----------------------------------------------------------------------

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom ( xij, yij, zij )

            xkj = x(k,m) - x(j,m)
            ykj = y(k,m) - y(j,m)
            zkj = z(k,m) - z(j,m)

            call pbc_atom ( xkj, ykj, zkj )

            xlj = x(l,m) - x(j,m)
            ylj = y(l,m) - y(j,m)
            zlj = z(l,m) - z(j,m)

            call pbc_atom ( xlj, ylj, zlj )

!-----------------------------------------------------------------------

            xijk = yij*zkj - zij*ykj
            yijk = zij*xkj - xij*zkj
            zijk = xij*ykj - yij*xkj

            xjkl = ylj*zkj - zlj*ykj
            yjkl = zlj*xkj - xlj*zkj
            zjkl = xlj*ykj - ylj*xkj

            rijk2  = xijk*xijk + yijk*yijk + zijk*zijk
            rjkl2  = xjkl*xjkl + yjkl*yjkl + zjkl*zjkl

            rijkl2 = sqrt(rijk2*rjkl2)

            if ( abs(rijk2)  .lt. tiny_value ) cycle
            if ( abs(rjkl2)  .lt. tiny_value ) cycle
            if ( abs(rijkl2) .lt. tiny_value ) cycle

            rijk2inv  = 1.d0 / rijk2
            rjkl2inv  = 1.d0 / rjkl2
            rijkl2inv = 1.d0 / rijkl2

!-----------------------------------------------------------------------

            factor_1 = fc_improper(n) * (180.d0/pi) * (180.d0/pi)

            cos_phi = ( xijk*xjkl + yijk*yjkl + zijk*zjkl ) * rijkl2inv

            cos_phi = max( cos_phi, -1.d0 )
            cos_phi = min( cos_phi,  1.d0 )

            phi = acos(cos_phi)

            sign_phi = ( yijk*zjkl - zijk*yjkl ) * xkj &
     &               + ( zijk*xjkl - xijk*zjkl ) * ykj &
     &               + ( xijk*yjkl - yijk*xjkl ) * zkj

            sign_phi = sign( 1.d0, sign_phi )

            phi = phi * sign_phi

            factor_2  = eq_improper(n) * (pi/180.d0)

            dphi = phi - factor_2

!-----------------------------------------------------------------------

            pot_b(m)  = pot_b(m) + 0.5d0 * factor_1 * dphi * dphi

!-----------------------------------------------------------------------

            if ( ( abs(phi)    .gt. tiny_value ) .and. &
     &           ( abs(phi-pi) .gt. tiny_value ) .and. &
     &           ( abs(phi+pi) .gt. tiny_value ) ) then

!-----------------------------------------------------------------------

               factor_3 = sin(phi)

               factor_4 = factor_1 * dphi / factor_3

               px1 = yijk*zij - zijk*yij
               py1 = zijk*xij - xijk*zij
               pz1 = xijk*yij - yijk*xij

               px2 = yjkl*zij - zjkl*yij
               py2 = zjkl*xij - xjkl*zij
               pz2 = xjkl*yij - yjkl*xij

               px3 = yijk*zkj - zijk*ykj
               py3 = zijk*xkj - xijk*zkj
               pz3 = xijk*ykj - yijk*xkj

               px4 = yjkl*zkj - zjkl*ykj
               py4 = zjkl*xkj - xjkl*zkj
               pz4 = xjkl*ykj - yjkl*xkj

               px5 = yijk*zlj - zijk*ylj
               py5 = zijk*xlj - xijk*zlj
               pz5 = xijk*ylj - yijk*xlj

               px6 = yjkl*zlj - zjkl*ylj
               py6 = zjkl*xlj - xjkl*zlj
               pz6 = xjkl*ylj - yjkl*xlj

               fxi = factor_4 * ( - px4*rijkl2inv &
     &                            + px3*rijk2inv*cos_phi )
               fyi = factor_4 * ( - py4*rijkl2inv &
     &                            + py3*rijk2inv*cos_phi )
               fzi = factor_4 * ( - pz4*rijkl2inv &
     &                            + pz3*rijk2inv*cos_phi )

               fxk = factor_4 * ( + px2*rijkl2inv &
     &                            + px5*rijkl2inv &
     &                            - px1*rijk2inv*cos_phi &
     &                            - px6*rjkl2inv*cos_phi )
               fyk = factor_4 * ( + py2*rijkl2inv &
     &                            + py5*rijkl2inv &
     &                            - py1*rijk2inv*cos_phi &
     &                            - py6*rjkl2inv*cos_phi )
               fzk = factor_4 * ( + pz2*rijkl2inv &
     &                            + pz5*rijkl2inv &
     &                            - pz1*rijk2inv*cos_phi &
     &                            - pz6*rjkl2inv*cos_phi )

               fxl = factor_4 * ( - px3*rijkl2inv &
     &                            + px4*rjkl2inv*cos_phi )
               fyl = factor_4 * ( - py3*rijkl2inv &
     &                            + py4*rjkl2inv*cos_phi )
               fzl = factor_4 * ( - pz3*rijkl2inv &
     &                            + pz4*rjkl2inv*cos_phi )

               fxj = - ( fxi + fxk + fxl )
               fyj = - ( fyi + fyk + fyl )
               fzj = - ( fzi + fzk + fzl )

!-----------------------------------------------------------------------

            else

!-----------------------------------------------------------------------

               xki = - xij + xkj
               yki = - yij + ykj
               zki = - zij + zkj

               xkl = - xlj + xkj
               ykl = - ylj + ykj
               zkl = - zlj + zkj

               ax = yijk*zjkl - zijk*yjkl
               ay = zijk*xjkl - xijk*zjkl
               az = xijk*yjkl - yijk*xjkl

               a2 = ax*ax + ay*ay + az*az

               a1 = sqrt( a2 )

               sin_phi = a1 / rijkl2

               sin_phi = max( sin_phi, -1.d0 )
               sin_phi = min( sin_phi,  1.d0 )

               phi = sign_phi * asin( sin_phi )

               if ( cos_phi .lt. 0.d0 ) phi = pi - phi

               daxdxi = - zjkl * zkj - ykj * yjkl
               daxdyi = + yjkl * xkj
               daxdzi = + zjkl * xkj

               daydxi = + xjkl * ykj
               daydyi = - xjkl * xkj - zkj * zjkl
               daydzi = + zjkl * ykj

               dazdxi = + xjkl * zkj
               dazdyi = + yjkl * zkj
               dazdzi = - yjkl * ykj - xkj * xjkl

               daxdxj = - yijk * ykl + zjkl * zki &
     &                  + yjkl * yki - zijk * zkl
               daxdyj = + yijk * xkl - yjkl * xki
               daxdzj = + zijk * xkl - zjkl * xki

               daydxj = + xijk * ykl - xjkl * yki
               daydyj = - zijk * zkl + xjkl * xki &
     &                  + zjkl * zki - xijk * xkl
               daydzj = + zijk * ykl - zjkl * yki

               dazdxj = + xijk * zkl - xjkl * zki
               dazdyj = + yijk * zkl - yjkl * zki
               dazdzj = - xijk * xkl + yjkl * yki &
     &                  + xjkl * xki - yijk * ykl

!               daxdxk = - yjkl * yij + zijk * zlj
!                        + yijk * ylj - zjkl * zij
!               daxdyk = + yjkl * xij - yijk * xlj
!               daxdzk = + zjkl * xij - zijk * xlj
!
!               daydxk = + xjkl * yij - xijk * ylj
!               daydyk = - zjkl * zij + xijk * xlj
!                        + zijk * zlj - xjkl * xij
!               daydzk = + zjkl * yij - zijk * ylj
!
!               dazdxk = + xjkl * zij - xijk * zlj
!               dazdyk = + yjkl * zij - yijk * zlj
!               dazdzk = - xjkl * xij + yijk * ylj
!                        + xijk * xlj - yjkl * yij

               daxdxl = + zijk * zkj + ykj * yijk
               daxdyl = - yijk * xkj
               daxdzl = - zijk * xkj

               daydxl = - xijk * ykj
               daydyl = + xijk * xkj + zkj * zijk
               daydzl = - zijk * ykj

               dazdxl = - xijk * zkj
               dazdyl = - yijk * zkj
               dazdzl = + yijk * ykj + xkj * xijk

               dadxi = ax/a1*daxdxi + ay/a1*daydxi + az/a1*dazdxi
               dadyi = ax/a1*daxdyi + ay/a1*daydyi + az/a1*dazdyi
               dadzi = ax/a1*daxdzi + ay/a1*daydzi + az/a1*dazdzi

               dadxj = ax/a1*daxdxj + ay/a1*daydxj + az/a1*dazdxj
               dadyj = ax/a1*daxdyj + ay/a1*daydyj + az/a1*dazdyj
               dadzj = ax/a1*daxdzj + ay/a1*daydzj + az/a1*dazdzj

!               dadxk = ax/a1*daxdxk + ay/a1*daydxk + az/a1*dazdxk
!               dadyk = ax/a1*daxdyk + ay/a1*daydyk + az/a1*dazdyk
!               dadzk = ax/a1*daxdzk + ay/a1*daydzk + az/a1*dazdzk

               dadxl = ax/a1*daxdxl + ay/a1*daydxl + az/a1*dazdxl
               dadyl = ax/a1*daxdyl + ay/a1*daydyl + az/a1*dazdyl
               dadzl = ax/a1*daxdzl + ay/a1*daydzl + az/a1*dazdzl

               f1 = - sign_phi/cos_phi * (180.d0/pi) * (180.d0/pi) &
     &              * fc_improper(n) * dphi

               fxi = + f1 * ( dadxi / rijkl2 &
     &                + sin_phi * ( + yijk*zkj - zijk*ykj ) * rijk2inv )

               fyi = + f1 * ( dadyi / rijkl2 &
     &                + sin_phi * ( + zijk*xkj - xijk*zkj ) * rijk2inv )

               fzi = + f1 * ( dadzi / rijkl2 &
     &                + sin_phi * ( + xijk*ykj - yijk*xkj ) * rijk2inv )

               fxj = + f1 * ( dadxj / rijkl2 &
     &                + sin_phi * ( - yijk*zki + zijk*yki ) * rijk2inv &
     &                - sin_phi * ( + yjkl*zkl - zjkl*ykl ) * rjkl2inv )

               fyj = + f1 * ( dadyj / rijkl2 &
     &                + sin_phi * ( - zijk*xki + xijk*zki ) * rijk2inv &
     &                - sin_phi * ( + zjkl*xkl - xjkl*zkl ) * rjkl2inv )

               fzj = + f1 * ( dadzj / rijkl2 &
     &                + sin_phi * ( - xijk*yki + yijk*xki ) * rijk2inv &
     &                - sin_phi * ( + xjkl*ykl - yjkl*xkl ) * rjkl2inv )

!              fxk = - f1 * ( dadxk / rijkl2
!     &                - sin_phi * ( - yjkl*zlj + zjkl*ylj ) * rjkl2inv
!     &                + sin_phi * ( + yijk*zij - zijk*yij ) * rijk2inv )
!
!              fyk = - f1 * ( dadyk / rijkl2
!     &                - sin_phi * ( - zjkl*xlj + xjkl*zlj ) * rjkl2inv
!     &                + sin_phi * ( + zijk*xij - xijk*zij ) * rijk2inv )
!
!              fzk = - f1 * ( dadzk / rijkl2
!     &                - sin_phi * ( - xjkl*ylj + yjkl*xlj ) * rjkl2inv
!     &                + sin_phi * ( + xijk*yij - yijk*xij ) * rijk2inv )

               fxl = + f1 * ( dadxl / rijkl2 &
     &                + sin_phi * ( + yjkl*zkj - zjkl*ykj ) * rjkl2inv )

               fyl = + f1 * ( dadyl / rijkl2 &
     &                + sin_phi * ( + zjkl*xkj - xjkl*zkj ) * rjkl2inv )

               fzl = + f1 * ( dadzl / rijkl2 &
     &                + sin_phi * ( + xjkl*ykj - yjkl*xkj ) * rjkl2inv )

               fxk = - fxi - fxj - fxl
               fyk = - fyi - fyj - fyl
               fzk = - fzi - fzj - fzl

!-----------------------------------------------------------------------

            end if

!-----------------------------------------------------------------------

            fx_b(i,m) = fx_b(i,m) + fxi
            fx_b(j,m) = fx_b(j,m) + fxj
            fx_b(k,m) = fx_b(k,m) + fxk
            fx_b(l,m) = fx_b(l,m) + fxl

            fy_b(i,m) = fy_b(i,m) + fyi
            fy_b(j,m) = fy_b(j,m) + fyj
            fy_b(k,m) = fy_b(k,m) + fyk
            fy_b(l,m) = fy_b(l,m) + fyl

            fz_b(i,m) = fz_b(i,m) + fzi
            fz_b(j,m) = fz_b(j,m) + fzj
            fz_b(k,m) = fz_b(k,m) + fzk
            fz_b(l,m) = fz_b(l,m) + fzl

!-----------------------------------------------------------------------

            vir_b(1,1) = vir_b(1,1) + fxi*xij + fxk*xkj + fxl*xlj
            vir_b(1,2) = vir_b(1,2) + fxi*yij + fxk*ykj + fxl*ylj
            vir_b(1,3) = vir_b(1,3) + fxi*zij + fxk*zkj + fxl*zlj
            vir_b(2,1) = vir_b(2,1) + fyi*xij + fyk*xkj + fyl*xlj
            vir_b(2,2) = vir_b(2,2) + fyi*yij + fyk*ykj + fyl*ylj
            vir_b(2,3) = vir_b(2,3) + fyi*zij + fyk*zkj + fyl*zlj
            vir_b(3,1) = vir_b(3,1) + fzi*xij + fzk*xkj + fzl*xlj
            vir_b(3,2) = vir_b(3,2) + fzi*yij + fzk*ykj + fzl*ylj
            vir_b(3,3) = vir_b(3,3) + fzi*zij + fzk*zkj + fzl*zlj

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_me_cmap( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pi, x, y, z, nbead

      use mm_variables, only : &
     &   xgrid_cmap, ygrid_cmap, vgrid_cmap, v2grid_cmap, &
     &   i_cmap, j_cmap, k_cmap, l_cmap, ncmap, mgrid_cmap

      use qmmm_variables, only : &
     &   layer, fx_b, fy_b, fz_b, pot_b, vir_b

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, m, n, i1or2, ioption

      real(8) :: xij, yij, zij, xkj, ykj, zkj, xlj, ylj, zlj, xijk, &
     &           yijk, zijk, xjkl, yjkl, zjkl, rijk2, rjkl2, rijkl2, &
     &           rijk2inv, rjkl2inv, rijkl2inv, cos_phi, sign_phi, &
     &           v, phi(2), dvdphi(2), fxi, fyi, fzi, fxj, fyj, fzj, &
     &           fxk, fyk, fzk, fxl, fyl, fzl, factor, sin_phi, f1, &
     &           xki, yki, zki, xkl, ykl, zkl, ax, ay, az, a1, a2, &
     &           daxdxi, daxdyi, daxdzi, daydxi, daydyi, daydzi, &
     &           dazdxi, dazdyi, dazdzi, dadxi, dadyi, dadzi, &
     &           daxdxj, daxdyj, daxdzj, daydxj, daydyj, daydzj, &
     &           dazdxj, dazdyj, dazdzj, dadxj, dadyj, dadzj, &
     &           daxdxl, daxdyl, daxdzl, daydxl, daydyl, daydzl, &
     &           dazdxl, dazdyl, dazdzl, dadxl, dadyl, dadzl, &
     &           phi1, phi2, p1c, p2c, dvdphi1, dvdphi2

      real(8) :: tiny_value = 1.d-4

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( ncmap .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         do n = 1, ncmap

            if ( ioption .eq. 0 ) then
               if ( ( layer(i_cmap(1,n))(1:1) .eq. 'A' ) .and. &
     &              ( layer(j_cmap(1,n))(1:1) .eq. 'A' ) .and. &
     &              ( layer(k_cmap(1,n))(1:1) .eq. 'A' ) .and. &
     &              ( layer(l_cmap(1,n))(1:1) .eq. 'A' ) .and. &
     &              ( layer(i_cmap(2,n))(1:1) .eq. 'A' ) .and. &
     &              ( layer(j_cmap(2,n))(1:1) .eq. 'A' ) .and. &
     &              ( layer(k_cmap(2,n))(1:1) .eq. 'A' ) .and. &
     &              ( layer(l_cmap(2,n))(1:1) .eq. 'A' ) ) cycle
            end if

            do i1or2 = 1, 2

               i = i_cmap(i1or2,n)
               j = j_cmap(i1or2,n)
               k = k_cmap(i1or2,n)
               l = l_cmap(i1or2,n)

               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

               call pbc_atom ( xij, yij, zij )

               xkj = x(k,m) - x(j,m)
               ykj = y(k,m) - y(j,m)
               zkj = z(k,m) - z(j,m)

               call pbc_atom ( xkj, ykj, zkj )

               xlj = x(l,m) - x(j,m)
               ylj = y(l,m) - y(j,m)
               zlj = z(l,m) - z(j,m)

               call pbc_atom ( xlj, ylj, zlj )

               xijk = yij*zkj - zij*ykj
               yijk = zij*xkj - xij*zkj
               zijk = xij*ykj - yij*xkj

               xjkl = ylj*zkj - zlj*ykj
               yjkl = zlj*xkj - xlj*zkj
               zjkl = xlj*ykj - ylj*xkj

               rijk2  = xijk*xijk + yijk*yijk + zijk*zijk
               rjkl2  = xjkl*xjkl + yjkl*yjkl + zjkl*zjkl

               rijkl2 = sqrt(rijk2*rjkl2)

               if ( abs(rijk2)  .lt. tiny_value ) cycle
               if ( abs(rjkl2)  .lt. tiny_value ) cycle
               if ( abs(rijkl2) .lt. tiny_value ) cycle

               rijk2inv  = 1.d0 / rijk2
               rjkl2inv  = 1.d0 / rjkl2
               rijkl2inv = 1.d0 / rijkl2

               cos_phi = ( xijk*xjkl + yijk*yjkl + zijk*zjkl ) &
     &                 * rijkl2inv

               cos_phi = max( cos_phi, -1.d0 )
               cos_phi = min( cos_phi,  1.d0 )

               phi(i1or2) = acos(cos_phi)

               sign_phi = ( yijk*zjkl - zijk*yjkl ) * xkj &
     &                  + ( zijk*xjkl - xijk*zjkl ) * ykj &
     &                  + ( xijk*yjkl - yijk*xjkl ) * zkj

               sign_phi = sign( 1.d0, sign_phi )

               phi(i1or2) = phi(i1or2) * sign_phi

            end do

            p1c = 0.5d0 * ( xgrid_cmap(1) + xgrid_cmap(mgrid_cmap) )
            p2c = 0.5d0 * ( ygrid_cmap(1) + ygrid_cmap(mgrid_cmap) )

            phi1 = phi(1) * 180.d0 / pi
            phi2 = phi(2) * 180.d0 / pi

            phi1 = phi1 - nint( (phi1-p1c) / 360.d0 ) * 360.d0
            phi2 = phi2 - nint( (phi2-p2c) / 360.d0 ) * 360.d0

            call splin2_cmap( xgrid_cmap(:), ygrid_cmap(:), &
     &                        vgrid_cmap(:,:,n), v2grid_cmap(:,:,n), &
     &                        mgrid_cmap, mgrid_cmap, &
     &                        phi1, phi2, v, dvdphi1, dvdphi2 )

            dvdphi(1) = dvdphi1 * 180.d0 / pi
            dvdphi(2) = dvdphi2 * 180.d0 / pi

            pot_b(m)  = pot_b(m) + v

            do i1or2 = 1, 2

               i = i_cmap(i1or2,n)
               j = j_cmap(i1or2,n)
               k = k_cmap(i1or2,n)
               l = l_cmap(i1or2,n)

               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

               call pbc_atom ( xij, yij, zij )

               xkj = x(k,m) - x(j,m)
               ykj = y(k,m) - y(j,m)
               zkj = z(k,m) - z(j,m)

               call pbc_atom ( xkj, ykj, zkj )

               xlj = x(l,m) - x(j,m)
               ylj = y(l,m) - y(j,m)
               zlj = z(l,m) - z(j,m)

               call pbc_atom ( xlj, ylj, zlj )

               xijk = yij*zkj - zij*ykj
               yijk = zij*xkj - xij*zkj
               zijk = xij*ykj - yij*xkj

               xjkl = ylj*zkj - zlj*ykj
               yjkl = zlj*xkj - xlj*zkj
               zjkl = xlj*ykj - ylj*xkj

               rijk2  = xijk*xijk + yijk*yijk + zijk*zijk
               rjkl2  = xjkl*xjkl + yjkl*yjkl + zjkl*zjkl

               rijkl2 = sqrt(rijk2*rjkl2)

               if ( abs(rijk2)  .lt. tiny_value ) cycle
               if ( abs(rjkl2)  .lt. tiny_value ) cycle
               if ( abs(rijkl2) .lt. tiny_value ) cycle

               rijk2inv  = 1.d0 / rijk2
               rjkl2inv  = 1.d0 / rjkl2
               rijkl2inv = 1.d0 / rijkl2

               cos_phi = ( xijk*xjkl + yijk*yjkl + zijk*zjkl ) &
     &                 * rijkl2inv

               cos_phi = max( cos_phi, -1.d0 )
               cos_phi = min( cos_phi,  1.d0 )

               phi(i1or2) = acos(cos_phi)

               sign_phi = ( yijk*zjkl - zijk*yjkl ) * xkj &
     &                  + ( zijk*xjkl - xijk*zjkl ) * ykj &
     &                  + ( xijk*yjkl - yijk*xjkl ) * zkj

               sign_phi = sign( 1.d0, sign_phi )

               phi(i1or2) = phi(i1or2) * sign_phi

               if ( ( abs(phi(i1or2))    .gt. tiny_value ) .and. &
     &              ( abs(phi(i1or2)-pi) .gt. tiny_value ) .and. &
     &              ( abs(phi(i1or2)+pi) .gt. tiny_value ) ) then

                  sin_phi = sin( phi(i1or2) )

                  factor = dvdphi(i1or2) / sin_phi

                  fxi = factor * ( + ( ykj*zjkl - zkj*yjkl ) * rijkl2inv &
     &                    - ( ykj*zijk - zkj*yijk ) * cos_phi*rijk2inv )
                  fyi = factor * ( + ( zkj*xjkl - xkj*zjkl ) * rijkl2inv &
     &                    - ( zkj*xijk - xkj*zijk ) * cos_phi*rijk2inv )
                  fzi = factor * ( + ( xkj*yjkl - ykj*xjkl ) * rijkl2inv &
     &                    - ( xkj*yijk - ykj*xijk ) * cos_phi*rijk2inv )

                  fxl = factor * ( + ( ykj*zijk - zkj*yijk ) * rijkl2inv &
     &                    - ( ykj*zjkl - zkj*yjkl ) * cos_phi*rjkl2inv )
                  fyl = factor * ( + ( zkj*xijk - xkj*zijk ) * rijkl2inv &
     &                    - ( zkj*xjkl - xkj*zjkl ) * cos_phi*rjkl2inv )
                  fzl = factor * ( + ( xkj*yijk - ykj*xijk ) * rijkl2inv &
     &                    - ( xkj*yjkl - ykj*xjkl ) * cos_phi*rjkl2inv )

                  fxk = factor * ( - ( yij*zjkl - zij*yjkl ) * rijkl2inv &
     &                    - ( ylj*zijk - zlj*yijk ) * rijkl2inv &
     &                    + ( yij*zijk - zij*yijk ) * cos_phi*rijk2inv &
     &                    + ( ylj*zjkl - zlj*yjkl ) * cos_phi*rjkl2inv )
                  fyk = factor * ( - ( zij*xjkl - xij*zjkl ) * rijkl2inv &
     &                    - ( zlj*xijk - xlj*zijk ) * rijkl2inv &
     &                    + ( zij*xijk - xij*zijk ) * cos_phi*rijk2inv &
     &                    + ( zlj*xjkl - xlj*zjkl ) * cos_phi*rjkl2inv )
                  fzk = factor * ( - ( xij*yjkl - yij*xjkl ) * rijkl2inv &
     &                    - ( xlj*yijk - ylj*xijk ) * rijkl2inv &
     &                    + ( xij*yijk - yij*xijk ) * cos_phi*rijk2inv &
     &                    + ( xlj*yjkl - ylj*xjkl ) * cos_phi*rjkl2inv )

                  fxj = - ( fxi + fxk + fxl )
                  fyj = - ( fyi + fyk + fyl )
                  fzj = - ( fzi + fzk + fzl )

               else

                  xki = - xij + xkj
                  yki = - yij + ykj
                  zki = - zij + zkj

                  xkl = - xlj + xkj
                  ykl = - ylj + ykj
                  zkl = - zlj + zkj

                  ax = yijk*zjkl - zijk*yjkl
                  ay = zijk*xjkl - xijk*zjkl
                  az = xijk*yjkl - yijk*xjkl

                  a2 = ax*ax + ay*ay + az*az

                  a1 = sqrt( a2 )

                  sin_phi = a1 / rijkl2

                  sin_phi = max( sin_phi, -1.d0 )
                  sin_phi = min( sin_phi,  1.d0 )

                  phi = sign_phi * asin( sin_phi )

                  if ( cos_phi .lt. 0.d0 ) phi = pi - phi

                  daxdxi = - zjkl * zkj - ykj * yjkl
                  daxdyi = + yjkl * xkj
                  daxdzi = + zjkl * xkj

                  daydxi = + xjkl * ykj
                  daydyi = - xjkl * xkj - zkj * zjkl
                  daydzi = + zjkl * ykj

                  dazdxi = + xjkl * zkj
                  dazdyi = + yjkl * zkj
                  dazdzi = - yjkl * ykj - xkj * xjkl

                  daxdxj = - yijk * ykl + zjkl * zki &
     &                     + yjkl * yki - zijk * zkl
                  daxdyj = + yijk * xkl - yjkl * xki
                  daxdzj = + zijk * xkl - zjkl * xki

                  daydxj = + xijk * ykl - xjkl * yki
                  daydyj = - zijk * zkl + xjkl * xki &
     &                     + zjkl * zki - xijk * xkl
                  daydzj = + zijk * ykl - zjkl * yki

                  dazdxj = + xijk * zkl - xjkl * zki
                  dazdyj = + yijk * zkl - yjkl * zki
                  dazdzj = - xijk * xkl + yjkl * yki &
     &                     + xjkl * xki - yijk * ykl

!                  daxdxk = - yjkl * yij + zijk * zlj
!                           + yijk * ylj - zjkl * zij
!                  daxdyk = + yjkl * xij - yijk * xlj
!                  daxdzk = + zjkl * xij - zijk * xlj
!
!                  daydxk = + xjkl * yij - xijk * ylj
!                  daydyk = - zjkl * zij + xijk * xlj
!                           + zijk * zlj - xjkl * xij
!                  daydzk = + zjkl * yij - zijk * ylj
!
!                  dazdxk = + xjkl * zij - xijk * zlj
!                  dazdyk = + yjkl * zij - yijk * zlj
!                  dazdzk = - xjkl * xij + yijk * ylj
!                           + xijk * xlj - yjkl * yij

                  daxdxl = + zijk * zkj + ykj * yijk
                  daxdyl = - yijk * xkj
                  daxdzl = - zijk * xkj

                  daydxl = - xijk * ykj
                  daydyl = + xijk * xkj + zkj * zijk
                  daydzl = - zijk * ykj

                  dazdxl = - xijk * zkj
                  dazdyl = - yijk * zkj
                  dazdzl = + yijk * ykj + xkj * xijk

                  dadxi = ax/a1*daxdxi + ay/a1*daydxi + az/a1*dazdxi
                  dadyi = ax/a1*daxdyi + ay/a1*daydyi + az/a1*dazdyi
                  dadzi = ax/a1*daxdzi + ay/a1*daydzi + az/a1*dazdzi

                  dadxj = ax/a1*daxdxj + ay/a1*daydxj + az/a1*dazdxj
                  dadyj = ax/a1*daxdyj + ay/a1*daydyj + az/a1*dazdyj
                  dadzj = ax/a1*daxdzj + ay/a1*daydzj + az/a1*dazdzj

!                  dadxk = ax/a1*daxdxk + ay/a1*daydxk + az/a1*dazdxk
!                  dadyk = ax/a1*daxdyk + ay/a1*daydyk + az/a1*dazdyk
!                  dadzk = ax/a1*daxdzk + ay/a1*daydzk + az/a1*dazdzk

                  dadxl = ax/a1*daxdxl + ay/a1*daydxl + az/a1*dazdxl
                  dadyl = ax/a1*daxdyl + ay/a1*daydyl + az/a1*dazdyl
                  dadzl = ax/a1*daxdzl + ay/a1*daydzl + az/a1*dazdzl

                  f1 = - sign_phi / cos_phi * dvdphi(i1or2)

                  fxi = + f1 * ( dadxi / rijkl2 &
     &                + sin_phi * ( + yijk*zkj - zijk*ykj ) * rijk2inv )

                  fyi = + f1 * ( dadyi / rijkl2 &
     &                + sin_phi * ( + zijk*xkj - xijk*zkj ) * rijk2inv )

                  fzi = + f1 * ( dadzi / rijkl2 &
     &                + sin_phi * ( + xijk*ykj - yijk*xkj ) * rijk2inv )

                  fxj = + f1 * ( dadxj / rijkl2 &
     &                + sin_phi * ( - yijk*zki + zijk*yki ) * rijk2inv &
     &                - sin_phi * ( + yjkl*zkl - zjkl*ykl ) * rjkl2inv )

                  fyj = + f1 * ( dadyj / rijkl2 &
     &                + sin_phi * ( - zijk*xki + xijk*zki ) * rijk2inv &
     &                - sin_phi * ( + zjkl*xkl - xjkl*zkl ) * rjkl2inv )

                  fzj = + f1 * ( dadzj / rijkl2 &
     &                + sin_phi * ( - xijk*yki + yijk*xki ) * rijk2inv &
     &                - sin_phi * ( + xjkl*ykl - yjkl*xkl ) * rjkl2inv )

!                 fxk = - f1 * ( dadxk / rijkl2
!     &                - sin_phi * ( - yjkl*zlj + zjkl*ylj ) * rjkl2inv
!     &                + sin_phi * ( + yijk*zij - zijk*yij ) * rijk2inv )
!
!                 fyk = - f1 * ( dadyk / rijkl2
!     &                - sin_phi * ( - zjkl*xlj + xjkl*zlj ) * rjkl2inv
!     &                + sin_phi * ( + zijk*xij - xijk*zij ) * rijk2inv )
!
!                 fzk = - f1 * ( dadzk / rijkl2
!     &                - sin_phi * ( - xjkl*ylj + yjkl*xlj ) * rjkl2inv
!     &                + sin_phi * ( + xijk*yij - yijk*xij ) * rijk2inv )

                  fxl = + f1 * ( dadxl / rijkl2 &
     &                + sin_phi * ( + yjkl*zkj - zjkl*ykj ) * rjkl2inv )

                  fyl = + f1 * ( dadyl / rijkl2 &
     &                + sin_phi * ( + zjkl*xkj - xjkl*zkj ) * rjkl2inv )

                  fzl = + f1 * ( dadzl / rijkl2 &
     &                + sin_phi * ( + xjkl*ykj - yjkl*xkj ) * rjkl2inv )

                  fxk = - fxi - fxj - fxl
                  fyk = - fyi - fyj - fyl
                  fzk = - fzi - fzj - fzl

               end if

               fx_b(i,m) = fx_b(i,m) + fxi
               fx_b(j,m) = fx_b(j,m) + fxj
               fx_b(k,m) = fx_b(k,m) + fxk
               fx_b(l,m) = fx_b(l,m) + fxl

               fy_b(i,m) = fy_b(i,m) + fyi
               fy_b(j,m) = fy_b(j,m) + fyj
               fy_b(k,m) = fy_b(k,m) + fyk
               fy_b(l,m) = fy_b(l,m) + fyl

               fz_b(i,m) = fz_b(i,m) + fzi
               fz_b(j,m) = fz_b(j,m) + fzj
               fz_b(k,m) = fz_b(k,m) + fzk
               fz_b(l,m) = fz_b(l,m) + fzl

               vir_b(1,1) = vir_b(1,1) + fxi*xij + fxk*xkj + fxl*xlj
               vir_b(1,2) = vir_b(1,2) + fxi*yij + fxk*ykj + fxl*ylj
               vir_b(1,3) = vir_b(1,3) + fxi*zij + fxk*zkj + fxl*zlj
               vir_b(2,1) = vir_b(2,1) + fyi*xij + fyk*xkj + fyl*xlj
               vir_b(2,2) = vir_b(2,2) + fyi*yij + fyk*ykj + fyl*ylj
               vir_b(2,3) = vir_b(2,3) + fyi*zij + fyk*zkj + fyl*zlj
               vir_b(3,1) = vir_b(3,1) + fzi*xij + fzk*xkj + fzl*xlj
               vir_b(3,2) = vir_b(3,2) + fzi*yij + fzk*ykj + fzl*ylj
               vir_b(3,3) = vir_b(3,3) + fzi*zij + fzk*zkj + fzl*zlj

            end do

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_me_lj( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, boxinv, box, &
     &   iboundary, nbead

      use mm_variables, only : &
     &   rout_lj, rin_lj, eps_lj, sig_lj, bigbox, bigboxinv, &
     &   i_lj, j_lj, nlj, nbox_lj

      use qmmm_variables, only : &
     &   layer, fx_b, fy_b, fz_b, pot_b, vir_b

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, l, m, jx, jy, jz, j2, ioption

      real(8) :: rout_lj2, xij, yij, zij, rij2, rij, rinv, eps, sig, &
     &           sr, sr2, sr6, sr12, u6, u12, uij, duij, fxi, fyi, fzi, &
     &           swf, dswf, aij, bij, cij, absa, absb, absc

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( nlj .eq. 0 ) return

      rout_lj2 = rout_lj*rout_lj

!-----------------------------------------------------------------------
!     /*   extention of simulation box in real space sum              */
!-----------------------------------------------------------------------

      if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then

         absa = sqrt ( boxinv(1,1)*boxinv(1,1) &
     &               + boxinv(1,2)*boxinv(1,2) &
     &               + boxinv(1,3)*boxinv(1,3) )
         absb = sqrt ( boxinv(2,1)*boxinv(2,1) &
     &               + boxinv(2,2)*boxinv(2,2) &
     &               + boxinv(2,3)*boxinv(2,3) )
         absc = sqrt ( boxinv(3,1)*boxinv(3,1) &
     &               + boxinv(3,2)*boxinv(3,2) &
     &               + boxinv(3,3)*boxinv(3,3) )

         nbox_lj(1) = int(2.d0*rout_lj*absa) + 1
         nbox_lj(2) = int(2.d0*rout_lj*absb) + 1
         nbox_lj(3) = int(2.d0*rout_lj*absc) + 1

      end if

!-----------------------------------------------------------------------
!     /*   main loop : free boundary                                  */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) then

         do m = 1, nbead

         do l = 1, nlj

            i = i_lj(l)
            j = j_lj(l)

            if ( ioption .eq. 0 ) then
               if ( ( layer(i)(1:1) .eq. 'A' ) .and. &
     &              ( layer(j)(1:1) .eq. 'A' ) ) cycle
            end if

            if ( i .eq. j ) cycle

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom ( xij, yij, zij )

            rij2 = xij*xij + yij*yij + zij*zij

            if ( rij2 .gt. rout_lj2 ) cycle

            rij     = sqrt(rij2)

            rinv    = 1.d0/rij

            eps     = eps_lj(l)
            sig     = sig_lj(l)

            sr      = sig*rinv
            sr2     = sr*sr
            sr6     = sr2*sr2*sr2
            sr12    = sr6*sr6

            u6      = - 4.d0*eps*sr6
            u12     = + 4.d0*eps*sr12

!           /*   switching function   */
            call getswf( rij, rin_lj, rout_lj, swf, dswf )

!           /*   bare potential   */
            uij     = + u6 + u12

!           /*   bare potential gradient   */
            duij    = ( - 6.d0*u6*rinv - 12.d0*u12*rinv )*swf

!           /*   corrected potential   */
            pot_b(m)  = pot_b(m) + uij*swf

!           /*   corrected forces   */

            fxi = - uij*dswf*xij*rinv - duij*xij*rinv
            fyi = - uij*dswf*yij*rinv - duij*yij*rinv
            fzi = - uij*dswf*zij*rinv - duij*zij*rinv

            fx_b(i,m) = fx_b(i,m) + fxi
            fy_b(i,m) = fy_b(i,m) + fyi
            fz_b(i,m) = fz_b(i,m) + fzi

            fx_b(j,m) = fx_b(j,m) - fxi
            fy_b(j,m) = fy_b(j,m) - fyi
            fz_b(j,m) = fz_b(j,m) - fzi

            vir_b(1,1) = vir_b(1,1) + fxi*xij
            vir_b(1,2) = vir_b(1,2) + fxi*yij
            vir_b(1,3) = vir_b(1,3) + fxi*zij
            vir_b(2,1) = vir_b(2,1) + fyi*xij
            vir_b(2,2) = vir_b(2,2) + fyi*yij
            vir_b(2,3) = vir_b(2,3) + fyi*zij
            vir_b(3,1) = vir_b(3,1) + fzi*xij
            vir_b(3,2) = vir_b(3,2) + fzi*yij
            vir_b(3,3) = vir_b(3,3) + fzi*zij

         end do

         end do

!-----------------------------------------------------------------------
!     /*   main loop : periodic boundary with minimum image           */
!-----------------------------------------------------------------------

      else if ( nbox_lj(1)*nbox_lj(2)*nbox_lj(3) .eq. 1 ) then

         do m = 1, nbead

         do l = 1, nlj

            i = i_lj(l)
            j = j_lj(l)

            if ( ioption .eq. 0 ) then
               if ( ( layer(i)(1:1) .eq. 'A' ) .and. &
     &              ( layer(j)(1:1) .eq. 'A' ) ) cycle
            end if

            if ( i .eq. j ) cycle

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom ( xij, yij, zij )

            rij2 = xij*xij + yij*yij + zij*zij

            if ( rij2 .gt. rout_lj2 ) cycle

            rij     = sqrt(rij2)

            rinv    = 1.d0/rij

            eps     = eps_lj(l)
            sig     = sig_lj(l)

            sr      = sig*rinv
            sr2     = sr*sr
            sr6     = sr2*sr2*sr2
            sr12    = sr6*sr6

            u6      = - 4.d0*eps*sr6
            u12     = + 4.d0*eps*sr12

!           /*   switching function   */
            call getswf( rij, rin_lj, rout_lj, swf, dswf )

!           /*   bare potential   */
            uij     = + u6 + u12

!           /*   bare potential gradient   */
            duij    = ( - 6.d0*u6*rinv - 12.d0*u12*rinv )*swf

!           /*   corrected potential   */
            pot_b(m)  = pot_b(m) + uij*swf

!           /*   corrected forces   */

            fxi = - uij*dswf*xij*rinv - duij*xij*rinv
            fyi = - uij*dswf*yij*rinv - duij*yij*rinv
            fzi = - uij*dswf*zij*rinv - duij*zij*rinv

            fx_b(i,m) = fx_b(i,m) + fxi
            fy_b(i,m) = fy_b(i,m) + fyi
            fz_b(i,m) = fz_b(i,m) + fzi

            fx_b(j,m) = fx_b(j,m) - fxi
            fy_b(j,m) = fy_b(j,m) - fyi
            fz_b(j,m) = fz_b(j,m) - fzi

            vir_b(1,1) = vir_b(1,1) + fxi*xij
            vir_b(1,2) = vir_b(1,2) + fxi*yij
            vir_b(1,3) = vir_b(1,3) + fxi*zij
            vir_b(2,1) = vir_b(2,1) + fyi*xij
            vir_b(2,2) = vir_b(2,2) + fyi*yij
            vir_b(2,3) = vir_b(2,3) + fyi*zij
            vir_b(3,1) = vir_b(3,1) + fzi*xij
            vir_b(3,2) = vir_b(3,2) + fzi*yij
            vir_b(3,3) = vir_b(3,3) + fzi*zij

         end do

         end do

!-----------------------------------------------------------------------
!     /*   main loop : periodic boundary                              */
!-----------------------------------------------------------------------

      else

         bigbox(:,1) = dble(nbox_lj(1))*box(:,1)
         bigbox(:,2) = dble(nbox_lj(2))*box(:,2)
         bigbox(:,3) = dble(nbox_lj(3))*box(:,3)

         call inv3 ( bigbox, bigboxinv )

         do m = 1, nbead

         do l = 1, nlj

            i = i_lj(l)
            j = j_lj(l)

            if ( ioption .eq. 0 ) then
               if ( ( layer(i)(1:1) .eq. 'A' ) .and. &
     &              ( layer(j)(1:1) .eq. 'A' ) ) cycle
            end if

            do jx = 0, nbox_lj(1)-1
            do jy = 0, nbox_lj(2)-1
            do jz = 0, nbox_lj(3)-1

               j2 = jx*jx + jy*jy + jz*jz

               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

               rij2 = xij*xij + yij*yij + zij*zij

               if ( rij2 .gt. rout_lj2 ) cycle

               rij     = sqrt(rij2)

               rinv    = 1.d0/rij

               eps     = eps_lj(l)
               sig     = sig_lj(l)

               sr      = sig*rinv
               sr2     = sr*sr
               sr6     = sr2*sr2*sr2
               sr12    = sr6*sr6

               u6      = - 4.d0*eps*sr6
               u12     = + 4.d0*eps*sr12

!              /*   switching function   */
               call getswf( rij, rin_lj, rout_lj, swf, dswf )

!              /*   bare potential   */
               uij     = + u6 + u12

!              /*   bare potential gradient   */
               duij    = ( - 6.d0*u6*rinv - 12.d0*u12*rinv )*swf

!              /*   corrected potential   */
               pot_b(m)  = pot_b(m) + uij*swf*0.5d0

!              /*   corrected forces   */

               fxi = - uij*dswf*xij*rinv - duij*xij*rinv
               fyi = - uij*dswf*yij*rinv - duij*yij*rinv
               fzi = - uij*dswf*zij*rinv - duij*zij*rinv

               fx_b(i,m) = fx_b(i,m) + 0.5d0*fxi
               fy_b(i,m) = fy_b(i,m) + 0.5d0*fyi
               fz_b(i,m) = fz_b(i,m) + 0.5d0*fzi

               fx_b(j,m) = fx_b(j,m) - 0.5d0*fxi
               fy_b(j,m) = fy_b(j,m) - 0.5d0*fyi
               fz_b(j,m) = fz_b(j,m) - 0.5d0*fzi

               vir_b(1,1) = vir_b(1,1) + 0.5d0*fxi*xij
               vir_b(1,2) = vir_b(1,2) + 0.5d0*fxi*yij
               vir_b(1,3) = vir_b(1,3) + 0.5d0*fxi*zij
               vir_b(2,1) = vir_b(2,1) + 0.5d0*fyi*xij
               vir_b(2,2) = vir_b(2,2) + 0.5d0*fyi*yij
               vir_b(2,3) = vir_b(2,3) + 0.5d0*fyi*zij
               vir_b(3,1) = vir_b(3,1) + 0.5d0*fzi*xij
               vir_b(3,2) = vir_b(3,2) + 0.5d0*fzi*yij
               vir_b(3,3) = vir_b(3,3) + 0.5d0*fzi*zij

            end do
            end do
            end do

            if ( i .eq. j ) cycle

            i = j_lj(l)
            j = i_lj(l)

            if ( ioption .eq. 0 ) then
               if ( ( layer(i)(1:1) .eq. 'A' ) .and. &
     &              ( layer(j)(1:1) .eq. 'A' ) ) cycle
            end if

            do jx = 0, nbox_lj(1)-1
            do jy = 0, nbox_lj(2)-1
            do jz = 0, nbox_lj(3)-1

               j2 = jx*jx + jy*jy + jz*jz

               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

               rij2 = xij*xij + yij*yij + zij*zij

               if ( rij2 .gt. rout_lj2 ) cycle

               rij     = sqrt(rij2)

               rinv    = 1.d0/rij

               eps     = eps_lj(l)
               sig     = sig_lj(l)

               sr      = sig*rinv
               sr2     = sr*sr
               sr6     = sr2*sr2*sr2
               sr12    = sr6*sr6

               u6      = - 4.d0*eps*sr6
               u12     = + 4.d0*eps*sr12

!              /*   switching function   */
               call getswf( rij, rin_lj, rout_lj, swf, dswf )

!              /*   bare potential   */
               uij     = + u6 + u12

!              /*   bare potential gradient   */
               duij    = ( - 6.d0*u6*rinv - 12.d0*u12*rinv )*swf

!              /*   corrected potential   */
               pot_b(m)  = pot_b(m) + uij*swf*0.5d0

!              /*   corrected forces   */

               fxi = - uij*dswf*xij*rinv - duij*xij*rinv
               fyi = - uij*dswf*yij*rinv - duij*yij*rinv
               fzi = - uij*dswf*zij*rinv - duij*zij*rinv

               fx_b(i,m) = fx_b(i,m) + 0.5d0*fxi
               fy_b(i,m) = fy_b(i,m) + 0.5d0*fyi
               fz_b(i,m) = fz_b(i,m) + 0.5d0*fzi

               fx_b(j,m) = fx_b(j,m) - 0.5d0*fxi
               fy_b(j,m) = fy_b(j,m) - 0.5d0*fyi
               fz_b(j,m) = fz_b(j,m) - 0.5d0*fzi

               vir_b(1,1) = vir_b(1,1) + 0.5d0*fxi*xij
               vir_b(1,2) = vir_b(1,2) + 0.5d0*fxi*yij
               vir_b(1,3) = vir_b(1,3) + 0.5d0*fxi*zij
               vir_b(2,1) = vir_b(2,1) + 0.5d0*fyi*xij
               vir_b(2,2) = vir_b(2,2) + 0.5d0*fyi*yij
               vir_b(2,3) = vir_b(2,3) + 0.5d0*fyi*zij
               vir_b(3,1) = vir_b(3,1) + 0.5d0*fzi*xij
               vir_b(3,2) = vir_b(3,2) + 0.5d0*fzi*yij
               vir_b(3,3) = vir_b(3,3) + 0.5d0*fzi*zij

            end do
            end do
            end do

         end do

         end do

      end if

      return
      end





!***********************************************************************
      subroutine force_me_morse( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, nbead

      use mm_variables, only : &
     &   depth_morse, alpha_morse, eq_morse, i_morse, j_morse, nmorse

      use qmmm_variables, only : &
     &   layer, fx_b, fy_b, fz_b, pot_b, vir_b

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, m, ioption

      real(8) :: alpha, const, depth, dr, factor, expfactor, &
     &           fxi, fyi, fzi, rij, xij, yij, zij

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( nmorse .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   main loop                                                  */
!-----------------------------------------------------------------------

      do m = 1, nbead

         do k = 1, nmorse

            i     =      i_morse(k)
            j     =      j_morse(k)
            depth =  depth_morse(k)
            alpha =  alpha_morse(k)

            if ( ioption .eq. 0 ) then
               if ( ( layer(i)(1:1) .eq. 'A' ) .and. &
     &              ( layer(j)(1:1) .eq. 'A' ) ) cycle
            end if

            if ( i .eq. j ) cycle

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom ( xij, yij, zij )

            rij = sqrt( xij*xij + yij*yij + zij*zij )

            dr  = ( rij - eq_morse(k) )

            expfactor = exp(-alpha*dr)

            factor = 1.d0 - expfactor

            pot_b(m) = pot_b(m) + depth*( factor*factor - 1.d0 )

            const = - 2.d0*depth*factor*alpha*expfactor/rij

            fxi = const*xij
            fyi = const*yij
            fzi = const*zij

            fx_b(i,m) = fx_b(i,m) + fxi
            fy_b(i,m) = fy_b(i,m) + fyi
            fz_b(i,m) = fz_b(i,m) + fzi

            fx_b(j,m) = fx_b(j,m) - fxi
            fy_b(j,m) = fy_b(j,m) - fyi
            fz_b(j,m) = fz_b(j,m) - fzi

            vir_b(1,1) = vir_b(1,1) + fxi*xij
            vir_b(1,2) = vir_b(1,2) + fxi*yij
            vir_b(1,3) = vir_b(1,3) + fxi*zij
            vir_b(2,1) = vir_b(2,1) + fyi*xij
            vir_b(2,2) = vir_b(2,2) + fyi*yij
            vir_b(2,3) = vir_b(2,3) + fyi*zij
            vir_b(3,1) = vir_b(3,1) + fzi*xij
            vir_b(3,2) = vir_b(3,2) + fzi*yij
            vir_b(3,3) = vir_b(3,3) + fzi*zij

         end do

      end do

      return
      end





!***********************************************************************
      subroutine force_me_ljpair( ioption )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, boxinv, box, iboundary, natom, nbead

      use mm_variables, only : &
     &   rout_ljpair, rin_ljpair, eps_ljpair, sig_ljpair, bigboxinv, &
     &   bigbox, nbox_ljpair, nljpair, epsrule_ljpair, sigrule_ljpair

      use qmmm_variables, only : &
     &   layer, fx_b, fy_b, fz_b, pot_b, vir_b

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, m, jx, jy, jz, j2, ioption

      real(8) :: rout_ljpair2, xij, yij, zij, rij2, rij, rinv, eps, sig, &
     &           sr, sr2, sr6, sr12, u6, u12, uij, duij, fxi, fyi, fzi, &
     &           swf, dswf, aij, bij, cij, absa, absb, absc

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( nljpair .eq. 0 ) return

      rout_ljpair2 = rout_ljpair*rout_ljpair

!-----------------------------------------------------------------------
!     /*   extention of simulation box in real space sum              */
!-----------------------------------------------------------------------

      if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then

         absa = sqrt ( boxinv(1,1)*boxinv(1,1) &
     &               + boxinv(1,2)*boxinv(1,2) &
     &               + boxinv(1,3)*boxinv(1,3) )
         absb = sqrt ( boxinv(2,1)*boxinv(2,1) &
     &               + boxinv(2,2)*boxinv(2,2) &
     &               + boxinv(2,3)*boxinv(2,3) )
         absc = sqrt ( boxinv(3,1)*boxinv(3,1) &
     &               + boxinv(3,2)*boxinv(3,2) &
     &               + boxinv(3,3)*boxinv(3,3) )

         nbox_ljpair(1) = int(2.d0*rout_ljpair*absa) + 1
         nbox_ljpair(2) = int(2.d0*rout_ljpair*absb) + 1
         nbox_ljpair(3) = int(2.d0*rout_ljpair*absc) + 1

      end if

!-----------------------------------------------------------------------
!     /*   main loop : free boundary                                  */
!-----------------------------------------------------------------------

      if ( iboundary .eq. 0 ) then

         do m = 1, nbead

         do i = 1, natom-1
         do j = i+1, natom

            if ( ioption .eq. 0 ) then
               if ( ( layer(i)(1:1) .eq. 'A' ) .and. &
     &              ( layer(j)(1:1) .eq. 'A' ) ) cycle
            end if

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom ( xij, yij, zij )

            rij2 = xij*xij + yij*yij + zij*zij

            if ( rij2 .gt. rout_ljpair2 ) cycle

            rij     = sqrt(rij2)

            rinv    = 1.d0/rij

            call get_eps_ljpair &
     &         ( eps_ljpair(i), eps_ljpair(j), eps, epsrule_ljpair )

            call get_sig_ljpair &
     &         ( sig_ljpair(i), sig_ljpair(j), sig, sigrule_ljpair )

            sr      = sig*rinv
            sr2     = sr*sr
            sr6     = sr2*sr2*sr2
            sr12    = sr6*sr6

            u6      = - 4.d0*eps*sr6
            u12     = + 4.d0*eps*sr12

!           /*   switching function   */
            call getswf( rij, rin_ljpair, rout_ljpair, swf, dswf )

!           /*   bare potential   */
            uij     = + u6 + u12

!           /*   bare potential gradient   */
            duij    = ( - 6.d0*u6*rinv - 12.d0*u12*rinv )*swf

!           /*   corrected potential   */
            pot_b(m)  = pot_b(m) + uij*swf

!           /*   corrected forces   */

            fxi = - uij*dswf*xij*rinv - duij*xij*rinv
            fyi = - uij*dswf*yij*rinv - duij*yij*rinv
            fzi = - uij*dswf*zij*rinv - duij*zij*rinv

            fx_b(i,m) = fx_b(i,m) + fxi
            fy_b(i,m) = fy_b(i,m) + fyi
            fz_b(i,m) = fz_b(i,m) + fzi

            fx_b(j,m) = fx_b(j,m) - fxi
            fy_b(j,m) = fy_b(j,m) - fyi
            fz_b(j,m) = fz_b(j,m) - fzi

            vir_b(1,1) = vir_b(1,1) + fxi*xij
            vir_b(1,2) = vir_b(1,2) + fxi*yij
            vir_b(1,3) = vir_b(1,3) + fxi*zij
            vir_b(2,1) = vir_b(2,1) + fyi*xij
            vir_b(2,2) = vir_b(2,2) + fyi*yij
            vir_b(2,3) = vir_b(2,3) + fyi*zij
            vir_b(3,1) = vir_b(3,1) + fzi*xij
            vir_b(3,2) = vir_b(3,2) + fzi*yij
            vir_b(3,3) = vir_b(3,3) + fzi*zij

         end do
         end do

         end do

!-----------------------------------------------------------------------
!     /*   main loop : periodic boundary with minimum image           */
!-----------------------------------------------------------------------

      else if(nbox_ljpair(1)*nbox_ljpair(2)*nbox_ljpair(3) .eq. 1) then

         do m = 1, nbead

         do i = 1, natom-1
         do j = i+1, natom

            if ( ioption .eq. 0 ) then
               if ( ( layer(i)(1:1) .eq. 'A' ) .and. &
     &              ( layer(j)(1:1) .eq. 'A' ) ) cycle
            end if

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom ( xij, yij, zij )

            rij2 = xij*xij + yij*yij + zij*zij

            if ( rij2 .gt. rout_ljpair2 ) cycle

            rij     = sqrt(rij2)

            rinv    = 1.d0/rij

            call get_eps_ljpair &
     &         ( eps_ljpair(i), eps_ljpair(j), eps, epsrule_ljpair )

            call get_sig_ljpair &
     &         ( sig_ljpair(i), sig_ljpair(j), sig, sigrule_ljpair )

            sr      = sig*rinv
            sr2     = sr*sr
            sr6     = sr2*sr2*sr2
            sr12    = sr6*sr6

            u6      = - 4.d0*eps*sr6
            u12     = + 4.d0*eps*sr12

!           /*   switching function   */
            call getswf( rij, rin_ljpair, rout_ljpair, swf, dswf )

!           /*   bare potential   */
            uij     = + u6 + u12

!           /*   bare potential gradient   */
            duij    = ( - 6.d0*u6*rinv - 12.d0*u12*rinv )*swf

!           /*   corrected potential   */
            pot_b(m)  = pot_b(m) + uij*swf

!           /*   corrected forces   */

            fxi = - uij*dswf*xij*rinv - duij*xij*rinv
            fyi = - uij*dswf*yij*rinv - duij*yij*rinv
            fzi = - uij*dswf*zij*rinv - duij*zij*rinv

            fx_b(i,m) = fx_b(i,m) + fxi
            fy_b(i,m) = fy_b(i,m) + fyi
            fz_b(i,m) = fz_b(i,m) + fzi

            fx_b(j,m) = fx_b(j,m) - fxi
            fy_b(j,m) = fy_b(j,m) - fyi
            fz_b(j,m) = fz_b(j,m) - fzi

            vir_b(1,1) = vir_b(1,1) + fxi*xij
            vir_b(1,2) = vir_b(1,2) + fxi*yij
            vir_b(1,3) = vir_b(1,3) + fxi*zij
            vir_b(2,1) = vir_b(2,1) + fyi*xij
            vir_b(2,2) = vir_b(2,2) + fyi*yij
            vir_b(2,3) = vir_b(2,3) + fyi*zij
            vir_b(3,1) = vir_b(3,1) + fzi*xij
            vir_b(3,2) = vir_b(3,2) + fzi*yij
            vir_b(3,3) = vir_b(3,3) + fzi*zij

         end do
         end do

         end do

!-----------------------------------------------------------------------
!     /*   main loop : periodic boundary                              */
!-----------------------------------------------------------------------

      else

         bigbox(:,1) = dble(nbox_ljpair(1))*box(:,1)
         bigbox(:,2) = dble(nbox_ljpair(2))*box(:,2)
         bigbox(:,3) = dble(nbox_ljpair(3))*box(:,3)

         call inv3 ( bigbox, bigboxinv )

         do m = 1, nbead

         do i = 1, natom
         do j = i, natom

            if ( ioption .eq. 0 ) then
               if ( ( layer(i)(1:1) .eq. 'A' ) .and. &
     &              ( layer(j)(1:1) .eq. 'A' ) ) cycle
            end if

            do jx = 0, nbox_ljpair(1)-1
            do jy = 0, nbox_ljpair(2)-1
            do jz = 0, nbox_ljpair(3)-1

               j2 = jx*jx + jy*jy + jz*jz

               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

               rij2 = xij*xij + yij*yij + zij*zij

               if ( rij2 .gt. rout_ljpair2 ) cycle

               rij     = sqrt(rij2)

               rinv    = 1.d0/rij

               call get_eps_ljpair &
     &            ( eps_ljpair(i), eps_ljpair(j), eps, epsrule_ljpair )

               call get_sig_ljpair &
     &            ( sig_ljpair(i), sig_ljpair(j), sig, sigrule_ljpair )

               sr      = sig*rinv
               sr2     = sr*sr
               sr6     = sr2*sr2*sr2
               sr12    = sr6*sr6

               u6      = - 4.d0*eps*sr6
               u12     = + 4.d0*eps*sr12

!              /*   switching function   */
               call getswf( rij, rin_ljpair, rout_ljpair, swf, dswf )

!              /*   bare potential   */
               uij     = + u6 + u12

!              /*   bare potential gradient   */
               duij    = ( - 6.d0*u6*rinv - 12.d0*u12*rinv )*swf

!              /*   corrected potential   */
               pot_b(m)  = pot_b(m) + uij*swf*0.5d0

!              /*   corrected forces   */

               fxi = - uij*dswf*xij*rinv - duij*xij*rinv
               fyi = - uij*dswf*yij*rinv - duij*yij*rinv
               fzi = - uij*dswf*zij*rinv - duij*zij*rinv

               fx_b(i,m) = fx_b(i,m) + 0.5d0*fxi
               fy_b(i,m) = fy_b(i,m) + 0.5d0*fyi
               fz_b(i,m) = fz_b(i,m) + 0.5d0*fzi

               fx_b(j,m) = fx_b(j,m) - 0.5d0*fxi
               fy_b(j,m) = fy_b(j,m) - 0.5d0*fyi
               fz_b(j,m) = fz_b(j,m) - 0.5d0*fzi

               vir_b(1,1) = vir_b(1,1) + 0.5d0*fxi*xij
               vir_b(1,2) = vir_b(1,2) + 0.5d0*fxi*yij
               vir_b(1,3) = vir_b(1,3) + 0.5d0*fxi*zij
               vir_b(2,1) = vir_b(2,1) + 0.5d0*fyi*xij
               vir_b(2,2) = vir_b(2,2) + 0.5d0*fyi*yij
               vir_b(2,3) = vir_b(2,3) + 0.5d0*fyi*zij
               vir_b(3,1) = vir_b(3,1) + 0.5d0*fzi*xij
               vir_b(3,2) = vir_b(3,2) + 0.5d0*fzi*yij
               vir_b(3,3) = vir_b(3,3) + 0.5d0*fzi*zij

            end do
            end do
            end do

            if ( i .eq. j ) cycle

            do jx = 0, nbox_ljpair(1)-1
            do jy = 0, nbox_ljpair(2)-1
            do jz = 0, nbox_ljpair(3)-1

               j2 = jx*jx + jy*jy + jz*jz

               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

               xij = x(j,m) - x(i,m)
               yij = y(j,m) - y(i,m)
               zij = z(j,m) - z(i,m)

               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

               rij2 = xij*xij + yij*yij + zij*zij

               if ( rij2 .gt. rout_ljpair2 ) cycle

               rij     = sqrt(rij2)

               rinv    = 1.d0/rij

               call get_eps_ljpair &
     &            ( eps_ljpair(j), eps_ljpair(i), eps, epsrule_ljpair )

               call get_sig_ljpair &
     &            ( sig_ljpair(j), sig_ljpair(i), sig, sigrule_ljpair )

               sr      = sig*rinv
               sr2     = sr*sr
               sr6     = sr2*sr2*sr2
               sr12    = sr6*sr6

               u6      = - 4.d0*eps*sr6
               u12     = + 4.d0*eps*sr12

!              /*   switching function   */
               call getswf( rij, rin_ljpair, rout_ljpair, swf, dswf )

!              /*   bare potential   */
               uij     = + u6 + u12

!              /*   bare potential gradient   */
               duij    = ( - 6.d0*u6*rinv - 12.d0*u12*rinv )*swf

!              /*   corrected potential   */
               pot_b(m)  = pot_b(m) + uij*swf*0.5d0

!              /*   corrected forces   */

               fxi = - uij*dswf*xij*rinv - duij*xij*rinv
               fyi = - uij*dswf*yij*rinv - duij*yij*rinv
               fzi = - uij*dswf*zij*rinv - duij*zij*rinv

               fx_b(j,m) = fx_b(j,m) + 0.5d0*fxi
               fy_b(j,m) = fy_b(j,m) + 0.5d0*fyi
               fz_b(j,m) = fz_b(j,m) + 0.5d0*fzi

               fx_b(i,m) = fx_b(i,m) - 0.5d0*fxi
               fy_b(i,m) = fy_b(i,m) - 0.5d0*fyi
               fz_b(i,m) = fz_b(i,m) - 0.5d0*fzi

               vir_b(1,1) = vir_b(1,1) + 0.5d0*fxi*xij
               vir_b(1,2) = vir_b(1,2) + 0.5d0*fxi*yij
               vir_b(1,3) = vir_b(1,3) + 0.5d0*fxi*zij
               vir_b(2,1) = vir_b(2,1) + 0.5d0*fyi*xij
               vir_b(2,2) = vir_b(2,2) + 0.5d0*fyi*yij
               vir_b(2,3) = vir_b(2,3) + 0.5d0*fyi*zij
               vir_b(3,1) = vir_b(3,1) + 0.5d0*fzi*xij
               vir_b(3,2) = vir_b(3,2) + 0.5d0*fzi*yij
               vir_b(3,3) = vir_b(3,3) + 0.5d0*fzi*zij

            end do
            end do
            end do

         end do
         end do

         end do

      end if

      return
      end





!***********************************************************************
      subroutine force_me_ljpair_setup
!***********************************************************************

      call force_mm_ljpair_setup

      return
      end





!***********************************************************************
      subroutine force_me_ewald_dipole
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables from PIMD                                 */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, volume, pi, mbox, nbead

      use mm_variables, only : i_q, ncharge, q

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   reset   */
      implicit none

!     /*   integers   */
      integer :: i, j, k, m1, m2, m3

!     /*   real numbers   */
      real(8) :: dx, dy, dz, d2, factor, xi, yi, zi

!-----------------------------------------------------------------------
!     /*   shared variables from PIMD                                 */
!-----------------------------------------------------------------------

!     /*   constant   */
      factor = 2.d0 * pi / ( 3.d0 * volume )

!     /*   loop of beads   */
      do j = 1, nbead

!        /*   initialize   */
         dx = 0.d0
         dy = 0.d0
         dz = 0.d0

!        /*   loop of charges   */
         do k = 1, ncharge

!           /*   atom   */
            i  = i_q(k)

!           /*   coordinates   */
            xi = x(i,j)
            yi = y(i,j)
            zi = z(i,j)

!           /*   box number   */
            m1 = mbox(1,i,j)
            m2 = mbox(2,i,j)
            m3 = mbox(3,i,j)

!           /*   apply periodic boundary condition   */
            call pbc_unfold ( xi, yi, zi, m1, m2, m3 )

!           /*   dipole moment   */
            dx = dx + q(i) * xi
            dy = dy + q(i) * yi
            dz = dz + q(i) * zi

!        /*   loop of atoms   */
         end do

!        /*   square of dipole moment   */
         d2 = dx*dx + dy*dy + dz*dz

!        /*   potential   */
         pot(j) = pot(j) + factor * d2

!        /*   loop of charges   */
         do k = 1, ncharge

!           /*   atom   */
            i  = i_q(k)

!           /*   coordinates   */
            xi = x(i,j)
            yi = y(i,j)
            zi = z(i,j)

!           /*   box number   */
            m1 = mbox(1,i,j)
            m2 = mbox(2,i,j)
            m3 = mbox(3,i,j)

!           /*   apply periodic boundary condition   */
            call pbc_unfold ( xi, yi, zi, m1, m2, m3 )

!           /*   forces   */
            fx(i,j) = fx(i,j) - 2.d0 * factor * q(i) * dx
            fy(i,j) = fy(i,j) - 2.d0 * factor * q(i) * dy
            fz(i,j) = fz(i,j) - 2.d0 * factor * q(i) * dz

         end do

!        /*   virial   */
         vir(1,1) = vir(1,1) - 2.d0 * factor * dx * dx + factor * d2
         vir(1,2) = vir(1,2) - 2.d0 * factor * dx * dy
         vir(1,3) = vir(1,3) - 2.d0 * factor * dx * dz
         vir(2,1) = vir(2,1) - 2.d0 * factor * dy * dx
         vir(2,2) = vir(2,2) - 2.d0 * factor * dy * dy + factor * d2
         vir(2,3) = vir(2,3) - 2.d0 * factor * dy * dz
         vir(3,1) = vir(3,1) - 2.d0 * factor * dz * dx
         vir(3,2) = vir(3,2) - 2.d0 * factor * dz * dy
         vir(3,3) = vir(3,3) - 2.d0 * factor * dz * dz + factor * d2

!     /*   loop of beads   */
      end do

      return
      end



#endif
