!///////////////////////////////////////////////////////////////////////
!
!      Author:          B. Thomsen, M. Shiga
!      Last updated:    Jan 28, 2020 by B. Thomsen
!      Description:     energy and force from DFTB calculation 
!                       with DFTB+ in static library implementation
!
!///////////////////////////////////////////////////////////////////////

!#######################################################################
#ifdef dftblib
!#######################################################################

!***********************************************************************
      module dftblib_variables
!***********************************************************************

      use dftbplus

!     /*   atomic charges   */
      real(8), dimension(:,:), allocatable :: coord_dftb
      real(8), dimension(:,:), allocatable :: q_dftb
      
!     /* DFTB+ calculation handle */
      type(TDftbPlus), save :: dftbp

!***********************************************************************
      end module dftblib_variables
!***********************************************************************





!***********************************************************************
      subroutine force_libdftb_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, dipx, dipy, dipz, box, volume, &
     &   iounit, mbox, iounit_dftb, nbead, natom, iboundary, myrank,  &
     &   nprocs

      use dftblib_variables, only : coord_dftb, q_dftb, dftbp

      use dftbplus

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      character(len=3)::  char_num

      real(8) :: xi, yi, zi

      real(8) :: stress_tensor(3,3)

      integer :: ierr, ibead, i, j, m1, m2, m3, output_opt

      integer, save :: iset = 0

!     /* DFTB+ calculation input */
      type(TDftbPlusInput) :: input

!-----------------------------------------------------------------------
!     /*   initialize directory                                       */
!-----------------------------------------------------------------------

!     /*   error flag   */
      ierr = 0

!     /*   initial settings   */
      if ( iset .eq. 0 ) then

         if ( nprocs .gt. nbead ) then

            if( myrank .eq. 0 ) then

               write( 6, '(a)' ) &
     &            "Error - Too many MPI processes for DFTB."
               write( 6, '(a)' ) &
     &            "The number of MPI processes must be equal to " // &
     &            "or lower than the number of beads."

            end if

!           /*   error termination   */
            call error_handling_MPI &
     &         ( 1, 'subroutine force_libdftb_MPI', 28 )

         end if

         call read_int1_MPI ( output_opt , '<dftb_lib_output>',  &
     &                        17, iounit)

!        /*   loop of beads   */
         do ibead = 1, min(nbead,nprocs)

!           /*   allocated bead only   */
            if ( mod(ibead-1,nprocs) .ne. myrank ) cycle

            if( output_opt .eq. 1 ) then

               call int3_to_char( ibead, char_num )
               call system ('rm -f -r ./' // char_num )
               call system ('mkdir -p ./' // char_num )
               open ( iounit_dftb, file = './' // char_num //  &
     &              '/output.dftb', status = 'unknown' )
               call TDftbPlus_init( dftbp, iounit_dftb )

            else if( (output_opt .eq. 0) .and. (myrank .eq. 0) ) then

               call int3_to_char( 1, char_num )
               call system ('rm -f -r ./' // char_num )
               call system ('mkdir -p ./' // char_num )
               open ( iounit_dftb, file = './' // char_num //  &
     &              '/output.dftb', status = 'unknown' )
               call TDftbPlus_init( dftbp, iounit_dftb )

            else

               open ( iounit_dftb, file  = '/dev/null' )
               call TDftbPlus_init( dftbp, iounit_dftb )

            end if

!           /*   read dftb input from file   */
            call dftbp%getInputFromFile("dftb.dat", input)
            call dftbp%setupCalculator(input)

!           /*   check if number of atoms are the same in dftb and   */
!           /*   pimd input                                          */
            if (dftbp%nrOfAtoms() /= natom) then

               write( 6, '(a)' ) &
     &            "Error - Mismatch of number of atoms in" &
     &            // " structure.dat and dftb.dat files."

               call TDftbPlus_destruct(dftbp)

!              /*   error termination   */
               call error_handling_MPI &
     &            ( 1, 'subroutine force_libdftb_MPI', 28 )

            end if

!           /*   memory allocation   */
            if ( .not. allocated( coord_dftb ) ) &
     &         allocate( coord_dftb(3,natom) )

            if ( .not. allocated( q_dftb ) ) &
     &         allocate( q_dftb(natom,nbead))

         end do

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   stress   */
      stress_tensor(:,:) = 0.d0

!     /*   dipole moment   */
      dipx(:) = 0.d0
      dipy(:) = 0.d0
      dipz(:) = 0.d0

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

      if ( mod(ibead-1,nprocs) .ne. myrank ) cycle

!-----------------------------------------------------------------------
!     /*   Set geometry in DFT+                                       */
!-----------------------------------------------------------------------

!     /*   Transfer coordinates to local storage for sending to DFTB+ */

      do i = 1, natom
         coord_dftb(1,i) = x(i,ibead)
         coord_dftb(2,i) = y(i,ibead)
         coord_dftb(3,i) = z(i,ibead)
      end do

!     /*   Set both new coordinates and latice vectors if we are      */
!     /*   using PBC, otherwise set only the coordinates              */

      if (( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 )) then
         call dftbp%setGeometry(coord_dftb, box)
      else
         call dftbp%setGeometry(coord_dftb)
      end if

!-----------------------------------------------------------------------
!     /* Get Calculate Energy and Forces with DFT+                    */
!-----------------------------------------------------------------------

      call dftbp%getEnergy(pot(ibead))
      call dftbp%getGradients(coord_dftb)

      do i=1, natom
         fx(i,ibead) = -coord_dftb(1,i)
         fy(i,ibead) = -coord_dftb(2,i)
         fz(i,ibead) = -coord_dftb(3,i)
      end do

      if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then
         call dftbp%getStressTensor(stress_tensor)
      end if

      call dftbp%getGrossCharges(q_dftb(:,ibead))

!-----------------------------------------------------------------------
!     /*   read dftb output:  dipole moment not read                  */
!-----------------------------------------------------------------------

!     /*   loop of atoms   */
      do i = 1, natom

!        /*   coordinates   */
         xi = x(i,ibead)
         yi = y(i,ibead)
         zi = z(i,ibead)

!        /*   original box index   */
         m1 = mbox(1,i,ibead)
         m2 = mbox(2,i,ibead)
         m3 = mbox(3,i,ibead)

!        /*   apply periodic boundary   */
         call pbc_unfold_MPI( xi, yi, zi, m1, m2, m3 )

!        /*   dipole moment   */
         dipx(ibead) = dipx(ibead) + q_dftb(i,ibead) * xi
         dipy(ibead) = dipy(ibead) + q_dftb(i,ibead) * yi
         dipz(ibead) = dipz(ibead) + q_dftb(i,ibead) * zi

!     /*   loop of atoms   */
      end do

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz, natom, nbead )

!     /*   stress   */
      call my_mpi_allreduce_real_2_main ( stress_tensor, 3, 3 )

!     /*   dipole   */
      call my_mpi_allreduce_real_1 ( dipx, nbead )
      call my_mpi_allreduce_real_1 ( dipy, nbead )
      call my_mpi_allreduce_real_1 ( dipz, nbead )

!     /*   charges   */
      call my_mpi_allreduce_real_2 ( q_dftb, natom, nbead )


!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do j = 1, nbead

!        /*   add stress contribution to virial   */
         vir(1,1) = vir(1,1) + stress_tensor(1,1) * volume / nbead
         vir(1,2) = vir(1,2) + stress_tensor(1,2) * volume / nbead
         vir(1,3) = vir(1,3) + stress_tensor(1,3) * volume / nbead
         vir(2,1) = vir(2,1) + stress_tensor(2,1) * volume / nbead
         vir(2,2) = vir(2,2) + stress_tensor(2,2) * volume / nbead
         vir(2,3) = vir(2,3) + stress_tensor(2,3) * volume / nbead
         vir(3,1) = vir(3,1) + stress_tensor(3,1) * volume / nbead
         vir(3,2) = vir(3,2) + stress_tensor(3,2) * volume / nbead
         vir(3,3) = vir(3,3) + stress_tensor(3,3) * volume / nbead

!     /*   loop of beads   */
      end do

      return
      end

!***********************************************************************
      subroutine finalize_libdftb_MPI
!***********************************************************************

      use dftblib_variables, only : dftbp

      use dftbplus

      implicit none

      call TDftbPlus_destruct(dftbp)

      return
      end

!#######################################################################
#else
!#######################################################################

!***********************************************************************
      subroutine force_libdftb_MPI
!***********************************************************************

      use common_variables, only : myrank

      implicit none

      if ( myrank .eq. 0 ) then
         write( 6, '(a)' ) &
     &      'Error - libdftb is not compiled.'
         write( 6, '(a)' ) &
     &      'Try compiling with the option -Dlibdftb.'
         write( 6, '(a)' )
      end if

      call error_handling_MPI( 1, 'subroutine force_libdftb_MPI', 28 )

      return
      end

!***********************************************************************
      subroutine finalize_libdftb_MPI
!***********************************************************************

      implicit none

      write( 6, '(a)' ) &
     &   'Error - libdftb is not compiled.'
      write( 6, '(a)' ) &
     &   'Try compiling with the option -Dlibdftb.'
      write( 6, '(a)' )

      call error_handling_MPI ( 1, 'subroutine finalize_libdftb_MPI' &
     &                         , 31 )

      return
      end

!#######################################################################
#endif
!#######################################################################
