!///////////////////////////////////////////////////////////////////////
!
!      Author:          B. Thomsen, M. Shiga
!      Last updated:    Feb 21, 2020 by B. Thomsen
!      Description:     energy and force from a calculation 
!                       with CP2K in a library implementation
!
!///////////////////////////////////////////////////////////////////////

!#######################################################################
#ifdef libcp2kmpi_old
!#######################################################################

!***********************************************************************
      subroutine force_libcp2k_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, dipx, dipy, dipz, &
     &   box, volume, iounit, nbead, natom, iboundary, &
     &   myrank, mpi_comm_sub, myrank_main, nprocs_main, &
     &   iounit_null

      use analysis_variables, only : iprint_dip

!     /*   Modules, variables and functions used from CP2K   */
      use f77_interface
      use kinds, only : default_string_length
      use cp_error_handling, only: warning_counter
      use input_cp2k, only : create_cp2k_root_section
      use input_section_types, only : section_type, section_release

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      character(len=3) :: char_num

      real(8), dimension(:), allocatable, save :: coord_cp2k

      character(len = default_string_length) :: char_dip = "[DIPOLE]"

      real(8) :: stress_tensor(3,3)
      real(8) :: dipole(3) = 0.0

      integer :: ierr, ibead, itest, output_opt, i, j

      integer, dimension(:), allocatable, save :: env_bead_map

      integer, save :: iset = 0

!     /*   For the cp2k input   */
      type(section_type), pointer :: input_declaration

!-----------------------------------------------------------------------
!     /*   initialize directory                                       */
!-----------------------------------------------------------------------

!     /*   error flag   */
      ierr = 0

!     /*   initial settings   */
      if ( iset .eq. 0 ) then

         open( iounit_null, file='/dev/null', status='replace',  &
     &        action='write' )

         allocate( env_bead_map(nbead) )

         call init_cp2k( .FALSE., ierr )

         if ( ierr .ne. 0 ) then

            if ( myrank .eq. 0 ) then
               write( 6, '(a)' ) "Error - Failed to initialize CP2K."
               write( 6, '(a)' )
            end if

            call error_handling_MPI &
     &         ( ierr, 'subroutine force_libcp2k_MPI', 28 )

         end if

         call read_int1_MPI ( output_opt, '<cp2k_lib_output>', &
     &                        17, iounit )

!        /*   Setup directories for the calculations if needed   */
         if ( ( myrank .eq. 0 ) .and. ( output_opt .eq. 1 ) ) then
            do ibead = 1, nbead
               call int3_to_char( ibead, char_num )
               call system ('rm -f -r ./' // char_num )
               call system ('mkdir -p ./' // char_num )
            end do
         else if ( ( myrank .eq. 0 ) .and. ( output_opt .eq. 0 ) ) then
            call int3_to_char( 1, char_num )
            call system ('rm -f -r ./' // char_num )
            call system ('mkdir -p ./' // char_num )
         end if

!        /*   call MPI_barrier   */
         call my_mpi_barrier

         do ibead = 1, nbead

!           /*   allocated bead only   */
            if ( mod(ibead-1,nprocs_main) .ne. myrank_main ) cycle

!           /*   Initialize   */
            env_bead_map(ibead) = -1
            NULLIFY ( input_declaration )
            call create_cp2k_root_section( input_declaration )

            call int3_to_char( ibead, char_num )
            if ( (output_opt .eq. 1) .or. (output_opt .eq. 0) ) then

!              /*   output level 1: print the data for all beads   */
               call create_force_env(env_bead_map(ibead),  &
     &                 input_declaration, './cp2k.dat',  &
     &                 './' // char_num // '/cp2k.out', &
     &                 mpi_comm = mpi_comm_sub, ierr=ierr)

            else if ( (output_opt .eq. 0) .and. (ibead .eq. 1) ) then

!              /*   output level 0: print the data for the 1st bead   */
               call create_force_env( env_bead_map(ibead),  &
     &                 input_declaration, './cp2k.dat',  &
     &                 './' // char_num // '/cp2k.out', &
     &                 mpi_comm = mpi_comm_sub, ierr=ierr)

            else

!              /*   Any other output level removes all output data   */
               call create_force_env( env_bead_map(ibead),  &
     &                 input_declaration, './cp2k.dat',  &
     &                 output_unit=iounit_null, owns_out_unit = .FALSE., &
     &                 mpi_comm = mpi_comm_sub, ierr=ierr)


            end if

!           /*   Finalize   */
            call section_release( input_declaration )

!           /*   confirm same number of atoms in dftb and pimd input  */
            call get_natom( env_bead_map(ibead), itest, ierr )

            if (itest /= natom) then
               write( 6, '(a)' ) &
     &            "Error - Mismatch of number of atoms in" &
     &            // " structure.dat and cp2k.dat files."
               write( 6, '(a)' )
               call finalize_cp2k( .FALSE. , ierr )
!              /*   error termination   */
               call error_handling_MPI &
     &            ( 1, 'subroutine force_dftb_MPI', 25 )
            end if

         end do

!        /*   memory allocation   */
         if ( .not. allocated( coord_cp2k ) ) &
     &      allocate( coord_cp2k(natom*3) )

         iset = 1
      end if

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   stress   */
      stress_tensor(:,:) = 0.d0

!     /*   dipole moment   */
      dipx(:) = 0.d0
      dipy(:) = 0.d0
      dipz(:) = 0.d0

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!     /*   allocated bead only   */
      if ( mod(ibead-1,nprocs_main) .ne. myrank_main ) cycle

!-----------------------------------------------------------------------
!     /*   Set geometry in DFT+                                       */
!-----------------------------------------------------------------------

!     /*   Transfer coordinates to local storage for sending to CP2K  */
      do i = 1, natom
         coord_cp2k(3*i-2) = x(i,ibead)
         coord_cp2k(3*i-1) = y(i,ibead)
         coord_cp2k(3*i)   = z(i,ibead)
      end do

!     /*   Set both new coordinates and latice vectors if we are      */
!     /*   using PBC, otherwise set only the coordinates              */

      call set_pos( env_bead_map(ibead), coord_cp2k, natom*3, ierr )

      if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then
         call set_cell( env_bead_map(ibead), box, ierr )
      end if

!-----------------------------------------------------------------------
!     /*   Calculate Energy and Forces                                */
!-----------------------------------------------------------------------

      call calc_energy_force( env_bead_map(ibead), .TRUE., ierr )
      
      if( ( warning_counter .gt. 0 ) .and. ( myrank_main .eq. 0 )) then
         call int3_to_char( ibead, char_num )      
         write( 6, '(a)' ) &
     &         "Error - A warning was triggered for bead " // char_num &
     &         // " this might be due to lack of HF convergence"
         call finalize_cp2k( .FALSE. , ierr )
!        /*   error termination   */
         call error_handling_MPI( 1, 'subroutine force_cp2k_MPI', 25 )
      end if

!-----------------------------------------------------------------------
!     /*   Get Energy, Forces and Stress Tensor                       */
!-----------------------------------------------------------------------

      call get_energy( env_bead_map(ibead), pot(ibead), ierr )
      call get_force( env_bead_map(ibead), coord_cp2k, natom*3, ierr )

      do i = 1, natom
         fx(i,ibead) = coord_cp2k(3*i-2)
         fy(i,ibead) = coord_cp2k(3*i-1)
         fz(i,ibead) = coord_cp2k(3*i)
      end do

      if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then
         call get_stress_tensor( env_bead_map(ibead), &
     &                           stress_tensor, ierr )
      end if

!-----------------------------------------------------------------------
!     /*   Get Dipole if Requested                                    */
!-----------------------------------------------------------------------

      if ( iprint_dip .gt. 0 ) then
         call get_result_r1( env_bead_map(ibead), char_dip, 3, &
     &                       dipole, ierr=ierr)
         dipx(ibead) = dipole(1)
         dipy(ibead) = dipole(2)
         dipz(ibead) = dipole(3)
      end if

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1_main ( pot, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2_main ( fx, natom, nbead )
      call my_mpi_allreduce_real_2_main ( fy, natom, nbead )
      call my_mpi_allreduce_real_2_main ( fz, natom, nbead )

!     /*   stress   */
      call my_mpi_allreduce_real_2_main ( stress_tensor, 3, 3 )

!     /*   dipole   */
      call my_mpi_allreduce_real_1_main ( dipx, nbead )
      call my_mpi_allreduce_real_1_main ( dipy, nbead )
      call my_mpi_allreduce_real_1_main ( dipz, nbead )

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do j = 1, nbead

!        /*   add stress contribution to virial   */
         vir(1,1) = vir(1,1) + stress_tensor(1,1) * volume / nbead
         vir(1,2) = vir(1,2) + stress_tensor(1,2) * volume / nbead
         vir(1,3) = vir(1,3) + stress_tensor(1,3) * volume / nbead
         vir(2,1) = vir(2,1) + stress_tensor(2,1) * volume / nbead
         vir(2,2) = vir(2,2) + stress_tensor(2,2) * volume / nbead
         vir(2,3) = vir(2,3) + stress_tensor(2,3) * volume / nbead
         vir(3,1) = vir(3,1) + stress_tensor(3,1) * volume / nbead
         vir(3,2) = vir(3,2) + stress_tensor(3,2) * volume / nbead
         vir(3,3) = vir(3,3) + stress_tensor(3,3) * volume / nbead

!     /*   loop of beads   */
      end do

      return
      end





!***********************************************************************
      subroutine finalize_libcp2k_MPI
!***********************************************************************

!     /* CP2K routines and variables*/
      use f77_interface, only : finalize_cp2k, default_para_env
      use reference_manager, only : print_all_references,  &
     &                              print_format_journal, &
     &                              collect_citations_from_ranks

!     /* PIMD output unit */
      use common_variables, only : myrank, iounit_cp2k, iounit_null

      implicit none

      integer :: ierr

      call collect_citations_from_ranks(default_para_env)

      if ( myrank .eq. 0 ) then
         open ( iounit_cp2k, file = 'cp2k_citations.out', &
     &          status= 'replace' )
         call print_all_references( sorted=.TRUE., cited_only=.TRUE., &
     &          FORMAT=print_format_journal, unit=iounit_cp2k )
         close( iounit_cp2k )
      endif

      call finalize_cp2k( .FALSE., ierr )

      close( iounit_null )

      return
      end

!#######################################################################
#endif
!#######################################################################
