!///////////////////////////////////////////////////////////////////////
!
!      Author:          B. Thomsen, M. Shiga
!      Last updated:    Jun 2, 2023 by B. Thomsen
!      Description:     energy and force from CP2K calculation
!                       in static library implementation
!
!///////////////////////////////////////////////////////////////////////

!#######################################################################
#ifdef libcp2kmpi
!#######################################################################

!***********************************************************************
      subroutine force_libcp2k_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, dipx, dipy, dipz, &
     &   box, volume, iounit, nbead, natom, iboundary, &
     &   myrank, mpi_comm_sub, myrank_main, nprocs_main, &
     &   iounit_null, np_beads, q_es, iprint_charge, istep

      use analysis_variables, only : iprint_dip

!     /*   Modules, variables and functions used from CP2K   */
      use f77_interface, only : create_force_env, get_result_r1, &
     &    set_pos, calc_energy_force, get_energy, get_force, &
     &    get_stress_tensor, set_cell, init_cp2k, get_natom, &
     &    finalize_cp2k, calc_energy
      use kinds, only : default_string_length
      use input_cp2k, only : create_cp2k_root_section
      use input_section_types, only : section_type, section_release
      use message_passing, only : mp_comm_type

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      character(len=3) :: char_num

      real(8), dimension(:), allocatable, save :: coord_cp2k

      character(len = default_string_length) :: char_desc

      real(8) :: stress_tensor(3,3)
      real(8) :: dipole(3) = 0.0
      real(8) :: charges(natom)

      integer :: ierr, ibead, jbead, itest, output_opt, i, j

      integer, dimension(:), allocatable, save :: env_bead_map

      integer, save :: iset = 0
      integer, save :: icalc_grad = 1

      TYPE(mp_comm_type) :: comm_forcp2k

!     /*   For the cp2k input   */
      type(section_type), pointer :: input_declaration

!     /*   Zero charges before running next step   */
      q_es(:,:,:) = 0.0d0

!-----------------------------------------------------------------------
!     /*   initialize directory                                       */
!-----------------------------------------------------------------------

!     /*   error flag   */
      ierr = 0

!     /*   initial settings   */
      if ( iset .eq. 0 ) then

         call read_int1_MPI ( icalc_grad, '<cp2k_lib_calcforce>', &
     &                        20, iounit )

         if( myrank .eq. 0 ) then
            write( 6, '(a,i3,a)' ) &
     &         "The electronic gradients and stress tensor will be " &
     &         // "evaluated every ", icalc_grad, " steps."
         end if

         call read_int1_MPI ( iprint_charge, '<cp2k_lib_chargeout>', &
     &                        20, iounit )

         if( iprint_charge .eq. 1 .and. myrank .eq. 0 ) then
            write( 6, '(a)' ) &
     &         "Mulliken and Hirshfeld Charges will be " &
     &         // "collected from CP2K."
         end if

         open( iounit_null, file='/dev/null', status='replace',  &
     &        action='write' )

         allocate( env_bead_map(nbead) )

         call init_cp2k( .FALSE., ierr )

         if ( ierr .ne. 0 ) then

            write( 6, '(a)' ) "Error - Failed to initialize CP2K."

            call error_handling_MPI &
     &         ( ierr, 'subroutine force_libcp2k_MPI', 28 )

         end if

         call read_int1_MPI ( output_opt, '<cp2k_lib_output>', &
     &                        17, iounit )

!        /*   Setup directories for the calculations if needed   */
         if ( ( myrank .eq. 0 ) .and. ( output_opt .eq. 1 ) ) then
            do ibead = 1, np_beads
               call int3_to_char( ibead, char_num )
               call system ('rm -f -r ./' // char_num )
               call system ('mkdir -p ./' // char_num )
            end do
         else if ( ( myrank .eq. 0 ) .and. ( output_opt .eq. 0 ) ) then
            call int3_to_char( 1, char_num )
            call system ('rm -f -r ./' // char_num )
            call system ('mkdir -p ./' // char_num )
         end if

!        /*   call MPI_barrier   */
         call my_mpi_barrier

         do ibead = 1, np_beads

!           /*   allocated bead only   */
            if ( mod(ibead-1,nprocs_main) .ne. myrank_main ) cycle

            call comm_forcp2k%set_handle(mpi_comm_sub)

!           /*   Initialize   */
            env_bead_map(ibead) = -1
            NULLIFY ( input_declaration )
            call create_cp2k_root_section( input_declaration )

            call int3_to_char( ibead, char_num )
            if ( (output_opt .eq. 1) .or. (output_opt .eq. 0) ) then

!              /*   output level 1: print the data for all beads   */
               call create_force_env(env_bead_map(ibead),  &
     &            input_declaration, './cp2k.dat',  &
     &            './' // char_num // '/cp2k.out', &
     &            mpi_comm = comm_forcp2k, ierr=ierr)

            else if ( (output_opt .eq. 0) .and. (ibead .eq. 1) ) then

!              /*   output level 0: print the data for the 1st bead   */
               call create_force_env( env_bead_map(ibead),  &
     &            input_declaration, './cp2k.dat',  &
     &            './' // char_num // '/cp2k.out', &
     &            mpi_comm = comm_forcp2k, ierr=ierr)

            else

!              /*   Any other output level removes all output data   */
               call create_force_env( env_bead_map(ibead),  &
     &            input_declaration, './cp2k.dat',  &
     &            output_unit=iounit_null, owns_out_unit = .FALSE., &
     &            mpi_comm = comm_forcp2k, ierr=ierr)


            end if

!           /*   Finalize   */
            call section_release( input_declaration )

!           /*   confirm same number of atoms in cp2k and pimd input  */
            call get_natom( env_bead_map(ibead), itest, ierr )

            if (itest /= natom) then

               write( 6, '(a)' ) &
     &            "Error - Mismatch of number of atoms in" &
     &            // " structure.dat and cp2k.dat files."
               write( 6, '(a)' )

               call finalize_cp2k( .FALSE. , ierr )

               ierr = 1
               exit

            end if

         end do

         do ibead = 1, np_beads
            do jbead = ibead, nbead, np_beads
               env_bead_map(jbead) = env_bead_map(ibead)
            end do
         end do

!        /*   error termination   */
         call error_handling_MPI &
     &      ( ierr, 'subroutine force_libcp2k_MPI', 28 )

!        /*   memory allocation   */
         if ( .not. allocated( coord_cp2k ) ) &
     &      allocate( coord_cp2k(natom*3) )

         iset = 1
      end if

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   stress   */
      stress_tensor(:,:) = 0.d0

!     /*   charges  */
      q_es(:,:,:) = 0.d0

!     /*   dipole moment   */
      dipx(:) = 0.d0
      dipy(:) = 0.d0
      dipz(:) = 0.d0

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!     /*   allocated bead only   */
      if ( mod(ibead-1,nprocs_main) .ne. myrank_main ) cycle

!-----------------------------------------------------------------------
!     /*   Prepare geometry transfer                                  */
!-----------------------------------------------------------------------

!     /*   Transfer coordinates to local storage for sending to CP2K  */
      do i = 1, natom
         coord_cp2k(3*i-2) = x(i,ibead)
         coord_cp2k(3*i-1) = y(i,ibead)
         coord_cp2k(3*i)   = z(i,ibead)
      end do

!     /*   Set latice vectors if we are using PBC   */
      if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then
         call set_cell( env_bead_map(ibead), box, ierr )
         if ( ierr .ne. 0 ) then
            ierr = 1
            exit
         end if
      end if

!-----------------------------------------------------------------------
!     /*   Calculate Energy and Forces                                */
!-----------------------------------------------------------------------

!     /*   Check if we are calculating the gradient or not   */
      if (icalc_grad .gt. 0 .and. mod(istep, icalc_grad) .eq. 0) then
!
         call set_pos( env_bead_map(ibead), coord_cp2k, natom*3, ierr )
         if ( ierr .ne. 0 ) then
            ierr = 2
            exit
         end if

         call calc_energy_force( env_bead_map(ibead), .TRUE., ierr )
         if ( ierr .ne. 0 ) then
            ierr = 3
            exit
         end if

!-----------------------------------------------------------------------
!     /*   Get Energy, Forces and Stress Tensor                       */
!-----------------------------------------------------------------------

         call get_energy( env_bead_map(ibead), pot(ibead), ierr )
         if ( ierr .ne. 0 ) then
            ierr = 4
            exit
         end if

         call get_force( env_bead_map(ibead), coord_cp2k, natom*3, ierr)
         if ( ierr .ne. 0 ) then
            ierr = 5
            exit
         end if

         do i = 1, natom
            fx(i,ibead) = coord_cp2k(3*i-2)
            fy(i,ibead) = coord_cp2k(3*i-1)
            fz(i,ibead) = coord_cp2k(3*i)
        end do

        if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then

           call get_stress_tensor( env_bead_map(ibead), &
     &                             stress_tensor, ierr )

           if ( ierr .ne. 0 ) then
              ierr = 6
              exit
           end if

        end if

!-----------------------------------------------------------------------
!     /*   Calculate only the energy                                  */
!-----------------------------------------------------------------------
      else

!        /*   If we are just calculating energy the call passes the   */
!        /*   coordinates directly and gets the energy back directly  */

         call calc_energy(env_bead_map(ibead), coord_cp2k, natom*3, &
        &                 pot(ibead), ierr)

         if ( ierr .ne. 0 ) then
            ierr = 7
            exit
         end if

!       /*   Set forces to zero just to be safe   */
         do i = 1, natom
            fx(i,ibead) = 0.0
            fy(i,ibead) = 0.0
            fz(i,ibead) = 0.0
        end do

      endif

!-----------------------------------------------------------------------
!     /*   Get Charges From CP2K                                      */
!-----------------------------------------------------------------------

      if (iprint_charge .gt. 0) then

         char_desc = "[MULLIKEN-CHARGES]"

         call get_result_r1( env_bead_map(ibead), char_desc, &
     &                       natom, charges, ierr=ierr)

         do i = 1, natom
            q_es(1,i,ibead) = charges(i)
         end do

         char_desc = "[HIRSHFELD-CHARGES]"

         call get_result_r1( env_bead_map(ibead), char_desc, &
     &                       natom, charges, ierr=ierr)

         do i = 1, natom
            q_es(2,i,ibead) = charges(i)
         end do

      end if

!-----------------------------------------------------------------------
!     /*   Get Dipole if Requested                                    */
!-----------------------------------------------------------------------

      if ( iprint_dip .gt. 0 ) then

         char_desc = "[DIPOLE]"

         call get_result_r1( env_bead_map(ibead), char_desc, 3, &
     &                       dipole, ierr=ierr)

         if ( ierr .ne. 0 ) then
            ierr = 6
            exit
         end if

         dipx(ibead) = dipole(1)
         dipy(ibead) = dipole(2)
         dipz(ibead) = dipole(3)

      end if

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   error termination                                          */
!-----------------------------------------------------------------------

      if ( ierr .ne. 0 ) then
         write( 6, '(a,i2,a)' ) 'Error - CP2K error code', ierr, '.'
      end if

      call error_handling_MPI( ierr, 'subroutine force_cp2k_MPI', 25 )

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1_main ( pot, nbead )

!     /*   charges */
      call my_mpi_allreduce_real_3_main ( q_es, 2, natom, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2_main ( fx, natom, nbead )
      call my_mpi_allreduce_real_2_main ( fy, natom, nbead )
      call my_mpi_allreduce_real_2_main ( fz, natom, nbead )

!     /*   stress   */
      call my_mpi_allreduce_real_2_main ( stress_tensor, 3, 3 )

!     /*   dipole   */
      call my_mpi_allreduce_real_1_main ( dipx, nbead )
      call my_mpi_allreduce_real_1_main ( dipy, nbead )
      call my_mpi_allreduce_real_1_main ( dipz, nbead )

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do j = 1, nbead

!        /*   add stress contribution to virial   */
         vir(1,1) = vir(1,1) + stress_tensor(1,1) * volume / nbead
         vir(1,2) = vir(1,2) + stress_tensor(1,2) * volume / nbead
         vir(1,3) = vir(1,3) + stress_tensor(1,3) * volume / nbead
         vir(2,1) = vir(2,1) + stress_tensor(2,1) * volume / nbead
         vir(2,2) = vir(2,2) + stress_tensor(2,2) * volume / nbead
         vir(2,3) = vir(2,3) + stress_tensor(2,3) * volume / nbead
         vir(3,1) = vir(3,1) + stress_tensor(3,1) * volume / nbead
         vir(3,2) = vir(3,2) + stress_tensor(3,2) * volume / nbead
         vir(3,3) = vir(3,3) + stress_tensor(3,3) * volume / nbead

!     /*   loop of beads   */
      end do

      return
      end





!***********************************************************************
      subroutine finalize_libcp2k_MPI
!***********************************************************************

!     /* CP2K routines and variables*/
      use f77_interface, only : finalize_cp2k, default_para_env
      use reference_manager, only : print_cited_references,  &
     &                              collect_citations_from_ranks
!     /* PIMD output unit */
      use common_variables, only : myrank, iounit_cp2k, iounit_null

      implicit none

      integer :: ierr

      call collect_citations_from_ranks(default_para_env)

      if ( myrank .eq. 0 ) then
         open ( iounit_cp2k, file = 'cp2k_citations.out', &
     &          status= 'replace' )
         call print_cited_references( iounit_cp2k )
         close( iounit_cp2k )
      endif

      call finalize_cp2k( .FALSE., ierr )

      close( iounit_null )

      return
      end

!#######################################################################
#else
#ifndef libcp2kmpi_old
!#######################################################################

!***********************************************************************
      subroutine force_libcp2k_MPI
!***********************************************************************

      implicit none

      write( 6, '(a)' ) &
     &   'Error - libcp2k with MPI is not compiled.'
      write( 6, '(a)' ) &
     &   'Try compiling with the option -Dlibcp2kmpi.'
      write( 6, '(a)' )

      call error_handling_MPI ( 1, 'subroutine force_libcp2k_MPI', 28 )

      return
      end





!***********************************************************************
      subroutine finalize_libcp2k_MPI
!***********************************************************************

      implicit none

      write( 6, '(a)' ) &
     &   'Error - libcp2k with MPI is not compiled.'
      write( 6, '(a)' ) &
     &   'Try compiling with the option -Dlibcp2kmpi.'
      write( 6, '(a)' )

      call error_handling_MPI ( 1, 'subroutine finalize_libcp2k_MPI',  &
     &                          31 )

      return
      end

!#######################################################################
#endif
#endif
!#######################################################################
