!///////////////////////////////////////////////////////////////////////
!
!      Author:          B. Thomsen, M. Shiga
!      Last updated:    Apr 19, 2022 by M. Shiga
!      Description:     energy and force from CP2K calculation
!                       in static library implementation
!
!///////////////////////////////////////////////////////////////////////

!#######################################################################
#ifdef libcp2k
!#######################################################################

!***********************************************************************
      subroutine force_libcp2k
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, dipx, dipy, dipz, q_es, &
     &   box, volume, iounit, nbead, natom, iboundary, iounit_null, &
     &   istep, iprint_charge

      use analysis_variables, only : iprint_dip

!     /*   Modules, variables and functions used from CP2K   */
      use f77_interface, only : create_force_env, get_result_r1, &
     &    set_pos, calc_energy_force, get_energy, get_force, &
     &    get_stress_tensor, set_cell, init_cp2k, get_natom, &
     &    finalize_cp2k
      use kinds, only : default_string_length
      use input_cp2k, only : create_cp2k_root_section
      use input_section_types, only : section_type, section_release

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      character(len=3) :: char_num

      real(8), dimension(:), allocatable, save :: coord_cp2k

      character(len = default_string_length) :: char_desc

      real(8) :: stress_tensor(3,3)
      real(8) :: dipole(3) = 0.0
      real(8) :: charge(natom)

      integer :: ierr, ibead, itest, output_opt, i, j

      integer, dimension(:), allocatable, save :: env_bead_map

      integer, save :: iset = 0
      integer, save :: icalc_grad = 1

!     /*   For the cp2k input   */
      type(section_type), pointer :: input_declaration

!-----------------------------------------------------------------------
!     /*   initialize directory                                       */
!-----------------------------------------------------------------------

!     /*   error flag   */
      ierr = 0

!     /*   initial settings   */
      if ( iset .eq. 0 ) then

         call read_int1 ( icalc_grad, '<cp2k_lib_calcforce>', &
     &                        20, iounit )

          write( 6, '(a,i3,a)' ) &
     &      "The electronic gradients and stress tensor will be " &
     &      // "evaluated every", icalc_grad, " steps."

         call read_int1 ( iprint_charge, '<cp2k_lib_chargeout>', &
     &                        20, iounit )

         if( iprint_charge .eq. 1) then
            write(6, '(a)') &
     &         "Mulliken and Hirshfeld Charges will be " &
     &         // "collected from CP2K."
         end if

         open(iounit_null,file='/dev/null',status='replace', &
     &        action='write')

         allocate( env_bead_map(nbead) )

         call init_cp2k( .FALSE., ierr )

         if ( ierr .ne. 0 ) then

            write( 6, '(a)' ) "Error - Failed to initialize CP2K."

            call error_handling &
     &         ( ierr, 'subroutine force_libcp2k', 24 )

         end if

         call read_int1 ( output_opt, '<cp2k_lib_output>', &
     &                    17, iounit )

         do ibead = 1, nbead

!           /*   Initialize   */
            env_bead_map(ibead) = -1
            NULLIFY (input_declaration)
            call create_cp2k_root_section(input_declaration)

            if ( (output_opt .eq. 1) .or. (output_opt .eq. 0) ) then

!              /*   output level 1: print the data for all beads   */
               call int3_to_char( ibead, char_num )
               call system ('rm -f -r ./' // char_num )
               call system ('mkdir -p ./' // char_num )
               call create_force_env( env_bead_map(ibead),  &
     &                 input_declaration, './cp2k.dat',  &
     &                 './' // char_num // '/cp2k.out', ierr=ierr )

            else if ( (output_opt .eq. 0) .and. (ibead .eq. 1) ) then

!              /*   output level 0: print the data for the 1st bead   */
               call int3_to_char( ibead, char_num )
               call system ('rm -f -r ./' // char_num )
               call system ('mkdir -p ./' // char_num )
               call create_force_env(env_bead_map(ibead),  &
     &                 input_declaration, './cp2k.dat',  &
     &                 './' // char_num // '/cp2k.out', ierr=ierr)

            else

!              /*   Any other output level removes all output data   */
               call create_force_env(env_bead_map(ibead),  &
     &               input_declaration, './cp2k.dat',  &
     &               output_unit=iounit_null, owns_out_unit = .FALSE., &
     &               ierr=ierr)
            end if

!           /*   Finalize   */
            call section_release(input_declaration)

!           /*   confirm same number of atoms in cp2k and pimd input  */
            call get_natom( env_bead_map(ibead), itest, ierr )

            if (itest /= natom) then

               write( 6, '(a)' ) &
     &            "Error - Mismatch of number of atoms in" &
     &            // " structure.dat and cp2k.dat files."

               call finalize_cp2k( .FALSE. , ierr )

               ierr = 1
               exit

            end if

         end do

!        /*   error termination   */
         call error_handling( ierr, 'subroutine force_libcp2k', 24 )

!        /*   memory allocation   */
         if ( .not. allocated( coord_cp2k ) ) &
     &      allocate( coord_cp2k(natom*3) )

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   stress   */
      stress_tensor(:,:) = 0.d0

!     /*   charges  */
      q_es(:,:,:) = 0.d0

!     /*   dipole moment   */
      dipx(:) = 0.d0
      dipy(:) = 0.d0
      dipz(:) = 0.d0

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!     /*   Set geometry in CP2K                                       */
!-----------------------------------------------------------------------

!     /*   Transfer coordinates to local storage for sending to CP2K */
      do i = 1, natom
         coord_cp2k(3*i-2) = x(i,ibead)
         coord_cp2k(3*i-1) = y(i,ibead)
         coord_cp2k(3*i)   = z(i,ibead)
      end do

!     /*   Set both new coordinates and latice vectors if we are using */
!     /*   PBC, otherwise set only the coordinates                     */

      call set_pos( env_bead_map(ibead), coord_cp2k, natom*3, ierr )

      if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then
         call set_cell( env_bead_map(ibead), box, ierr )
         if ( ierr .ne. 0 ) then
            ierr = 1
            exit
         end if
      end if

!-----------------------------------------------------------------------
!     /* Calculate Energy and Forces                                  */
!-----------------------------------------------------------------------

!     /* Check if we are calculating the gradient or not */
      if (icalc_grad .gt. 0 .and. mod(istep, icalc_grad) .eq. 0) then

         call set_pos( env_bead_map(ibead), coord_cp2k, natom*3, ierr )

         if ( ierr .ne. 0 ) then
               ierr = 2
               exit
         end if

         call calc_energy_force( env_bead_map(ibead), .TRUE., ierr )

         if ( ierr .ne. 0 ) then
            ierr = 3
            exit
         end if

!-----------------------------------------------------------------------
!     /*   Get Energy, Forces and Stress Tensor                       */
!-----------------------------------------------------------------------

         call get_energy( env_bead_map(ibead), pot(ibead), ierr )

         if ( ierr .ne. 0 ) then
             ierr = 4
             exit
         end if

         call get_force( env_bead_map(ibead), coord_cp2k, natom*3, ierr)

         if ( ierr .ne. 0 ) then
             ierr = 5
             exit
         end if

         do i = 1, natom
            fx(i,ibead) = coord_cp2k(3*i-2)
            fy(i,ibead) = coord_cp2k(3*i-1)
            fz(i,ibead) = coord_cp2k(3*i)
         end do

         if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then

            call get_stress_tensor( env_bead_map(ibead), &
     &                              stress_tensor, ierr )

            if ( ierr .ne. 0 ) then
               ierr = 6
               exit
            end if

         end if

!-----------------------------------------------------------------------
!     /*   Calculate only the energy                                  */
!-----------------------------------------------------------------------

      else

!     /*   If we are just calculating the energy the call passes the  */
!     /*   coordinates directly and gets the energy back directly     */

         call calc_energy(env_bead_map(ibead), coord_cp2k, natom*3, &
     &                    pot(ibead), ierr)

         if ( ierr .ne. 0 ) then
            ierr = 7
            exit
         end if

!       /*   Set forces to zero just to be safe   */
        do i = 1, natom
           fx(i,ibead) = 0.0
           fy(i,ibead) = 0.0
           fz(i,ibead) = 0.0
        end do

      end if

!-----------------------------------------------------------------------
!     /* Get Charges From CP2K                                        */
!-----------------------------------------------------------------------

      if (iprint_charge .gt. 0) then

            char_desc = "[MULLIKEN-CHARGES]"

            call get_result_r1( env_bead_map(ibead), char_desc, &
        &                       natom, charges, ierr=ierr)

            do i = 1, natom
               q_es(1,i,ibead) = charges(i)
            end do

            char_desc = "[HIRSHFELD-CHARGES]"
            call get_result_r1( env_bead_map(ibead), char_desc, &
        &                       natom, charges, ierr=ierr)

            do i = 1, natom
               q_es(2,i,ibead) = charges(i)
            end do

         end if

!-----------------------------------------------------------------------
!     /* Get Dipole if Requested                                      */
!-----------------------------------------------------------------------

      if ( iprint_dip .gt. 0 ) then

         char_desc = "[DIPOLE]"
         call get_result_r1( env_bead_map(ibead), char_desc, 3, &
     &                       dipole, ierr=ierr)

         if ( ierr .ne. 0 ) then
            ierr = 6
            exit
         end if

         dipx(ibead) = dipole(1)
         dipy(ibead) = dipole(2)
         dipz(ibead) = dipole(3)

      end if

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

!     /*   error termination   */
      if ( ierr .ne. 0 ) then
         write( 6, '(a,i2,a)' ) 'Error - CP2K error code', ierr, '.'
      end if

      call error_handling( ierr, 'subroutine force_libcp2k', 24 )

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do j = 1, nbead

!        /*   add stress contribution to virial   */
         vir(1,1) = vir(1,1) + stress_tensor(1,1) * volume / nbead
         vir(1,2) = vir(1,2) + stress_tensor(1,2) * volume / nbead
         vir(1,3) = vir(1,3) + stress_tensor(1,3) * volume / nbead
         vir(2,1) = vir(2,1) + stress_tensor(2,1) * volume / nbead
         vir(2,2) = vir(2,2) + stress_tensor(2,2) * volume / nbead
         vir(2,3) = vir(2,3) + stress_tensor(2,3) * volume / nbead
         vir(3,1) = vir(3,1) + stress_tensor(3,1) * volume / nbead
         vir(3,2) = vir(3,2) + stress_tensor(3,2) * volume / nbead
         vir(3,3) = vir(3,3) + stress_tensor(3,3) * volume / nbead

!     /*   loop of beads   */
      end do

      return
      end





!***********************************************************************
      subroutine finalize_libcp2k
!***********************************************************************

!     /* CP2K routines and variables*/
      use f77_interface, only : finalize_cp2k
      USE reference_manager, only : print_all_references,  &
     &                              print_format_journal

!     /* PIMD output unit */
      use common_variables, only : iounit_cp2k, iounit_null

      implicit none

      integer :: ierr

      open ( iounit_cp2k, file = 'cp2k_citations.out',  &
     &           status= 'replace' )
      call print_all_references(sorted=.TRUE., cited_only=.TRUE., &
     &                    FORMAT=print_format_journal, unit=iounit_cp2k)
      close( iounit_cp2k )

      call finalize_cp2k( .FALSE. , ierr )

      close ( iounit_null )

      end

!#######################################################################
#else
!#######################################################################

!***********************************************************************
      subroutine force_libcp2k
!***********************************************************************

      implicit none

      write( 6, '(a)' ) &
     &   'Error - libcp2k is not compiled.'
      write( 6, '(a)' ) &
     &   'Try compiling with the option -Dlibcp2k.'
      write( 6, '(a)' )

      call error_handling ( 1, 'subroutine force_libcp2k', 24 )

      return
      end





!***********************************************************************
      subroutine finalize_libcp2k
!***********************************************************************

      implicit none

      write( 6, '(a)' ) &
     &   'Error - libcp2k is not compiled.'
      write( 6, '(a)' ) &
     &   'Try compiling with the option -Dlibcp2k.'
      write( 6, '(a)' )

      call error_handling ( 1, 'subroutine finalize_libcp2k', 27 )

      return
      end

!#######################################################################
#endif
!#######################################################################
