!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Dec 27, 2022 by M. Shiga
!      Description:     calculate energy and force
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine force_hydrogen_silveragoldman
!***********************************************************************
!=======================================================================
!
!     H2 potential by Silvera, Goldman, J. Chem. Phys. 69, 4209 (1978).
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   au_length, fx, fy, fz, pi, pot, vir, x, y, z, natom, nbead

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   parameters   */
      real(8), parameter :: alpha =   1.713d0
      real(8), parameter :: beta  =   1.5671d0
      real(8), parameter :: gamma =   0.00993d0
      real(8), parameter :: c6    =   12.14d0
      real(8), parameter :: c8    =   215.2d0
      real(8), parameter :: c9    =   143.1d0
      real(8), parameter :: c10   =   4813.9d0
      real(8), parameter :: rm0   =   3.41d0

!      real(8), parameter :: eps_s =   32.2 kelvin
!      real(8), parameter :: rm_s  =   3.44 angstrom
!      real(8), parameter :: eps_p =   34.3 kelvin
!      real(8), parameter :: rm_p  =   3.41 angstrom

!     /*   integers   */
      integer :: i, j, m

!     /*   real numbers   */
      real(8) ::  e1, e1grad, e2, e2grad, f0, fc, fcgrad, fxi, fyi, fzi, &
     &            q, qgrad, r, r2, rinv, r2inv, r6inv, r8inv, r9inv, &
     &            r10inv, rc, rm, v, vgrad, xij, yij, zij

!-----------------------------------------------------------------------
!     /*   unit conversion                                            */
!-----------------------------------------------------------------------

      rm = rm0 * 1.d-10 / au_length

!-----------------------------------------------------------------------
!     /*   start main loop                                            */
!-----------------------------------------------------------------------

      do m = 1, nbead

         do i = 1, natom-1
         do j = i+1, natom

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom( xij, yij, zij )

            r2 = xij*xij + yij*yij + zij*zij

            r = sqrt( r2 )

            rinv = 1.d0 / r

            r2inv  = rinv  * rinv
            r6inv  = r2inv * r2inv * r2inv
            r8inv  = r6inv * r2inv
            r9inv  = r8inv * rinv
            r10inv = r8inv * r2inv

            rc = 1.28d0 * rm

            fc = 1.d0
            fcgrad = 0.d0

            if ( r .lt. rc ) then
               q = rc*rinv - 1.d0
               qgrad = - rc*r2inv
               fc = exp( - q*q )
               fcgrad = - 2.d0*q*fc*qgrad
            end if

            e1 = exp( alpha - beta*r - gamma*r2 )

            e2 = - c6*r6inv - c8*r8inv + c9*r9inv - c10*r10inv

            v = e1 + e2 * fc

            e1grad = ( - beta - 2.d0*gamma*r ) * e1

            e2grad = ( 6.d0*c6*r6inv + 8.d0*c8*r8inv &
     &               - 9.d0*c9*r9inv + 10.d0*c10*r10inv ) * rinv

            pot(m) = pot(m) + v

            vgrad = e1grad + e2grad*fc + e2*fcgrad

            f0 = - vgrad * rinv

            fxi = f0 * xij
            fyi = f0 * yij
            fzi = f0 * zij

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

         end do
         end do

      end do

      return
      end

