!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Sep 6, 2022 by M. Shiga
!      Description:     harmonic oscillators bath model
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      module hob_variables
!***********************************************************************

      integer:: nref_hob

      real(8), dimension(:), allocatable :: xref_hob
      real(8), dimension(:), allocatable :: yref_hob
      real(8), dimension(:), allocatable :: y2ref_hob

      integer:: imode_hob

!***********************************************************************
      end module hob_variables
!***********************************************************************




!***********************************************************************
      subroutine force_hob
!***********************************************************************
!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, au_length, physmass, speedlight_SI, &
     &   au_time, pi, natom, iounit, nbead, iboundary

      use nma_variables, only : &
     &   hess, eigval, eigvec, r, qr

      use hob_variables, only : &
     &   imode_hob

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

!     //   initialize
      implicit none

!     //   cutoff parameter to remove translation and rotation
      real(8), parameter :: cutoff_cminv = 1.5d0

!     //   unit conversion factor
      real(8) :: bohr2ang = au_length * 1.d+10

!     //   real numbers
      real(8) :: rx, ry, rz, vr, factor, dv, dvdq, qri, cminv

!     //   real functions
      real(8) :: vmode_hob, vmode_hob_grad

!     //   flag
      integer, save :: iset = 0

!     //   integers
      integer :: i, j, k, l, n, ij, kl, ierr

!     //   potential at origin
      real(8), save :: pot_origin

!     //   characters
      character(len=80) :: char
      character(len=8)  :: lengthunit
      character(len=8)  :: symbol

!-----------------------------------------------------------------------
!     //   start initial setup
!-----------------------------------------------------------------------

!     //   on first entry
      if ( iset .eq. 0 ) then

!-----------------------------------------------------------------------
!        //   memory allocation
!-----------------------------------------------------------------------

!        //   number of modes
         n = 3*natom

!        //   coordinates
         if ( .not. allocated(r) ) allocate( r(n) )

!        //   hessian
         if ( .not. allocated(hess) ) allocate( hess(n,n) )

!        //   modes
         if ( .not. allocated(qr) ) allocate( qr(n) )

!        //   eigenvalues
         if ( .not. allocated(eigval) ) allocate( eigval(n) )

!        //   eigenvectors
         if ( .not. allocated(eigvec) ) allocate( eigvec(n,n) )

!-----------------------------------------------------------------------
!        //   save coordinates at 1
!-----------------------------------------------------------------------

         do i = 1, 3*natom, 3
            l = (i+2) / 3
            r(i+0) = x(l,1)
            r(i+1) = y(l,1)
            r(i+2) = z(l,1)
         end do

!-----------------------------------------------------------------------
!        //   read potential at the origin
!-----------------------------------------------------------------------

!        //   open file
         open ( iounit, file = 'hob.dat' )

!        //   read line
         do
            read ( iounit, *, iostat=ierr ) char
            if ( ierr .ne. 0 ) exit
            if ( char(1:8) .eq. '<energy>' ) exit
         end do

!        //   read line
         read ( iounit, *, iostat=ierr ) pot_origin

!        //   close file
         close( iounit )

!        //   error check
         call error_handling( ierr, 'subroutine force_hob', 20 )

!-----------------------------------------------------------------------
!        //   read length unit at the origin
!-----------------------------------------------------------------------

!        //   open file
         open ( iounit, file = 'hob.dat' )

!        //   read line
         do
            read ( iounit, *, iostat=ierr ) char
            if ( ierr .ne. 0 ) exit
            if ( char(1:12) .eq. '<lengthunit>' ) exit
         end do

!        //   read line
         read ( iounit, *, iostat=ierr ) lengthunit

!        //   close file
         close( iounit )

!        //   error check
         call error_handling( ierr, 'subroutine force_hob', 20 )

!-----------------------------------------------------------------------
!        //   read coordinates at the origin
!-----------------------------------------------------------------------

!        //   open file
         open ( iounit, file = 'hob.dat' )

!        //   read line
         do
            read ( iounit, *, iostat=ierr ) char
            if ( ierr .ne. 0 ) exit
            if ( char(1:13) .eq. '<coordinates>' ) exit
         end do

!        //   read data
         do i = 1, natom
            read ( iounit, *, iostat=ierr ) &
     &         symbol, x(i,1), y(i,1), z(i,1)
         end do

!        //   close file
         close( iounit )

!        //   error check
         call error_handling( ierr, 'subroutine force_hob', 20 )

!        //   change units to bohr
         if ( lengthunit(1:8) .eq. 'ANGSTROM' ) then
            do i = 1, natom 
               x(i,1) = x(i,1) / bohr2ang
               y(i,1) = y(i,1) / bohr2ang
               z(i,1) = z(i,1) / bohr2ang
            end do
         end if

!-----------------------------------------------------------------------
!        //   read hessian
!-----------------------------------------------------------------------

!        //   open file
         open ( iounit, file = 'hob.dat' )

!        //   read line
         do
            read ( iounit, *, iostat=ierr ) char
            if ( ierr .ne. 0 ) exit
            if ( char(1:9) .eq. '<hessian>' ) exit
         end do

!        //   read data
         do i = 1, 3*natom
         do j = 1, 3*natom
            read ( iounit, *, iostat=ierr ) k, l, hess(i,j)
         end do
         end do

!        //   close file
         close( iounit )

!        //   error check
         call error_handling( ierr, 'subroutine force_hob', 20 )

!        //   symmetrize hessian
         do i = 1, 3*natom
         do j = i, 3*natom
            hess(i,j) = 0.5d0 * ( hess(i,j) + hess(j,i) )
            hess(j,i) = hess(i,j)
         end do
         end do

!-----------------------------------------------------------------------
!        //   read reactive mode
!-----------------------------------------------------------------------

         call vmode_hob_setup

!-----------------------------------------------------------------------
!        //   project out translation and rotation
!-----------------------------------------------------------------------

         call project_out_nma( 1 )

!-----------------------------------------------------------------------
!        //   mass weighted hessian
!-----------------------------------------------------------------------

         ij = 0

         do i = 1, natom
         do j = 1, 3

            ij = ij + 1

            kl = 0

            do k = 1, natom
            do l = 1, 3

               kl = kl + 1

               factor = 1.d0 / sqrt(physmass(i)*physmass(k))

               hess(ij,kl) = hess(ij,kl) * factor

            end do
            end do

         end do
         end do

!-----------------------------------------------------------------------
!        //   diagonalize mass weighted hessian
!-----------------------------------------------------------------------

         call ddiag ( hess, eigval, eigvec, n )

!-----------------------------------------------------------------------
!        //   warning: remove rotation and translation
!-----------------------------------------------------------------------

         factor = 2.d0 * pi * speedlight_SI * au_time * 100.d0

         j = 0

         do i = 1, 3*natom

            cminv = 0.d0

            if ( eigval(i) .gt. 0 ) then
               cminv = + sqrt(abs(eigval(i))) / factor
            else
               cminv = - sqrt(abs(eigval(i))) / factor
            end if

!           //   find low frequency modes
            if ( abs(cminv) .lt. cutoff_cminv ) then

               j = j + 1

               eigval(i)   = 0.d0
               eigvec(:,i) = 0.d0

            end if

         end do

         ierr = 0

         if ( iboundary .eq. 0 ) then
            if ( ( natom .eq. 1 ) .and. ( j .ne. 3 ) ) ierr = 1
            if ( ( natom .eq. 2 ) .and. ( j .ne. 5 ) ) ierr = 1
            if ( ( natom .ge. 3 ) .and. ( j .ne. 6 ) ) ierr = 1
         end if
         if ( ( iboundary .eq. 1 ) .and. ( j .ne. 3 ) ) ierr = 1
         if ( ( iboundary .eq. 2 ) .and. ( j .ne. 3 ) ) ierr = 1

!        //   error check
         call error_handling( ierr, 'subroutine force_hob', 20 )

!-----------------------------------------------------------------------
!        //   recover coordinates at 1
!-----------------------------------------------------------------------

         do i = 1, 3*natom, 3
            l = (i+2) / 3
            rx     = x(l,1)
            ry     = y(l,1)
            rz     = z(l,1)
            x(l,1) = r(i+0)
            y(l,1) = r(i+1)
            z(l,1) = r(i+2)
            r(i+0) = rx
            r(i+1) = ry
            r(i+2) = rz
         end do

!-----------------------------------------------------------------------
!        //   reset flag
!-----------------------------------------------------------------------

         iset = 1

!-----------------------------------------------------------------------
!     //   end if initial setup
!-----------------------------------------------------------------------

!     //   on first entry
      end if

!-----------------------------------------------------------------------
!     //   potential and forces
!-----------------------------------------------------------------------

!     //   loop of beads
      do k = 1, nbead

!        //   potential at origin
         vr = pot_origin

!        //   loop of modes
         do i = 1, 3*natom

!           //   mode at origin
            qr(i) = 0.d0

!           //   loops of atoms
            do j = 1, 3*natom, 3

!              //   atom l
               l = (j+2) / 3

!              //   relative coordinates
               rx = x(l,k) - r(j+0)
               ry = y(l,k) - r(j+1)
               rz = z(l,k) - r(j+2)

!              //   apply boundary condition
               call pbc_atom( rx, ry, rz )

!              //   scaling factor
               factor = sqrt( physmass(l) )

!              //   modes
               qr(i) = qr(i) + factor * rx * eigvec(j+0,i)
               qr(i) = qr(i) + factor * ry * eigvec(j+1,i)
               qr(i) = qr(i) + factor * rz * eigvec(j+2,i)

!           //   loops of atoms
            end do

!cc            write( 6,'(i6,f16.4)' ) i, qr(i)

!           //   potential
            vr = vr + 0.5d0 * qr(i) * eigval(i) * qr(i)

!           //   loop of atoms
            do j = 1, 3*natom, 3

!              //   atom j
               l = (j+2) / 3

!              //   scaling factor
               factor = sqrt( physmass(l) ) * eigval(i) * qr(i)

!              //   force
               fx(l,k) = fx(l,k) - factor * eigvec(j+0,i)
               fy(l,k) = fy(l,k) - factor * eigvec(j+1,i)
               fz(l,k) = fz(l,k) - factor * eigvec(j+2,i)

!           //   loop of atoms
            end do

!        //   loops of modes
         end do

!-----------------------------------------------------------------------
!        //   reactive mode
!-----------------------------------------------------------------------

!        //   reactive mode
         i = imode_hob

!        //   mode at origin
         qri = 0.d0

!        //   loops of atoms
         do j = 1, 3*natom, 3

!           //   atom l
            l = (j+2) / 3

!           //   relative coordinates
            rx = x(l,k) - r(j+0)
            ry = y(l,k) - r(j+1)
            rz = z(l,k) - r(j+2)

!           //   apply boundary condition
            call pbc_atom( rx, ry, rz )

!           //   scaling factor
            factor = sqrt( physmass(l) )

!           //   modes
            qri = qri + factor * rx * eigvec(j+0,i)
            qri = qri + factor * ry * eigvec(j+1,i)
            qri = qri + factor * rz * eigvec(j+2,i)

!        //   loops of atoms
         end do

!        //   spline function
         dv   = vmode_hob( qri )
         dvdq = vmode_hob_grad( qri )

!        //   subtract and add potential
         vr = vr - pot_origin - 0.5d0 * qri * eigval(i) * qri + dv

!        //   loop of atoms
         do j = 1, 3*natom, 3

!           //   atom j
            l = (j+2) / 3

!           //   scaling factor
            factor = sqrt( physmass(l) ) * ( - eigval(i) * qri + dvdq )

!           //   subtract force
            fx(l,k) = fx(l,k) - factor * eigvec(j+0,i)
            fy(l,k) = fy(l,k) - factor * eigvec(j+1,i)
            fz(l,k) = fz(l,k) - factor * eigvec(j+2,i)

!        //   loop of atoms
         end do

!        //   potential of bead k
         pot(k) = pot(k) + vr

!     //   loop of beads
      end do

      return
      end





!***********************************************************************
      subroutine vmode_hob_setup
!***********************************************************************
!-----------------------------------------------------------------------
!     //   shared variables
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit

      use hob_variables, only : &
     &   xref_hob, yref_hob, y2ref_hob, nref_hob, imode_hob

!-----------------------------------------------------------------------
!     //   local variables
!-----------------------------------------------------------------------

      implicit none

      integer :: i, ierr

      character(len=80) :: char

!-----------------------------------------------------------------------
!     //   read reactive mode
!-----------------------------------------------------------------------

!     //   open file
      open ( iounit, file = 'hob.dat' )

!     //   read line
      do
         read ( iounit, *, iostat=ierr ) char
         if ( ierr .ne. 0 ) exit
         if ( char(1:7) .eq. '<imode>' ) exit
      end do

!     //   read line
      read ( iounit, *, iostat=ierr ) imode_hob

!     //   close file
      close( iounit )

!     //   error check
      call error_handling( ierr, 'subroutine vmode_hob_setup', 26 )

!-----------------------------------------------------------------------
!     //   read number of grid points
!-----------------------------------------------------------------------

!     //   open file
      open ( iounit, file = 'hob.dat' )

!     //   read line
      do
         read ( iounit, *, iostat=ierr ) char
         if ( ierr .ne. 0 ) exit
         if ( char(1:7) .eq. '<imode>' ) exit
      end do

!     //   read line
      read ( iounit, *, iostat=ierr ) 
      read ( iounit, *, iostat=ierr ) nref_hob

!     //   close file
      close( iounit )

!     //   error check
      call error_handling( ierr, 'subroutine vmode_hob_setup', 26 )

!-----------------------------------------------------------------------
!     //   memory allocation of grid points
!-----------------------------------------------------------------------

      allocate( xref_hob(nref_hob) )
      allocate( yref_hob(nref_hob) )
      allocate( y2ref_hob(nref_hob) )

!-----------------------------------------------------------------------
!     //   read grid points
!-----------------------------------------------------------------------

!     //   open file
      open ( iounit, file = 'hob.dat' )

!     //   read line
      do
         read ( iounit, *, iostat=ierr ) char
         if ( ierr .ne. 0 ) exit
         if ( char(1:7) .eq. '<imode>' ) exit
      end do

!     //   read line
      read ( iounit, *, iostat=ierr ) 
      read ( iounit, *, iostat=ierr )

      do i = 1, nref_hob
         read( iounit, *, iostat=ierr ) xref_hob(i), yref_hob(i)
      end do

!     //   close file
      close( iounit )

!     //   error check
      call error_handling( ierr, 'subroutine vmode_hob_setup', 26 )

!-----------------------------------------------------------------------
!     //   spline interpolation
!-----------------------------------------------------------------------

      call spline_init_eam( xref_hob, yref_hob, y2ref_hob, nref_hob )

      return
      end





!***********************************************************************
      real(8) function vmode_hob( x )
!***********************************************************************

      use hob_variables, only : xref_hob, yref_hob, y2ref_hob, nref_hob

      implicit none

      real(8) :: x

!-----------------------------------------------------------------------

      if ( ( x .lt. xref_hob(1) ) .or. &
     &     ( x .gt. xref_hob(nref_hob) ) ) then

         vmode_hob = 0.d0

      else

         vmode_hob = getspline_eam &
     &      ( xref_hob(:), yref_hob(:), y2ref_hob(:), nref_hob, x )

      end if

!-----------------------------------------------------------------------

      return
      contains
      include 'getspline_eam_func.F90'
      end





!***********************************************************************
      real(8) function vmode_hob_grad( x )
!***********************************************************************

      use hob_variables, only : xref_hob, yref_hob, y2ref_hob, nref_hob

      implicit none

      real(8) :: x

!-----------------------------------------------------------------------

      if ( ( x .lt. xref_hob(1) ) .or. &
     &     ( x .gt. xref_hob(nref_hob) ) ) then

         vmode_hob_grad = 0.d0

      else

         vmode_hob_grad = getspline_grad_eam &
     &      ( xref_hob(:), yref_hob(:), y2ref_hob(:), nref_hob, x )

      end if

!-----------------------------------------------------------------------

      return
      contains
      include 'getspline_grad_eam_func.F90'
      end

