!///////////////////////////////////////////////////////////////////////
!
!      Authors:         S. Miura, H. Suno, M. Shiga
!      Last updated:    Jan 12, 2023 by M. Shiga
!      Description:     calculate energy and force
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine force_helium_aziz1991_MPI
!***********************************************************************
!=======================================================================
!
!     LM2M2 potential for helium:
!     Aziz, Slaman, J. Chem. Phys. 94, 8047 (1991)
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   au_length, fx, fy, fz, pi, pot, vir, x, y, z, natom, nbead, &
     &   myrank_main, nprocs_main, myrank_sub, nprocs_sub

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8), parameter :: alpha = -10.70203539d0
      real(8), parameter :: beta  =  -1.90740649d0
      real(8), parameter :: c6    =   1.34687065d0
      real(8), parameter :: c8    =   0.41308398d0
      real(8), parameter :: c10   =   0.17060159d0
      real(8), parameter :: d     =   1.4088d0
      real(8), parameter :: x0    =   1.003535949d0
      real(8), parameter :: x1    =   1.454790369d0
      real(8), parameter :: a0    =   2.9695d0
      real(8), parameter :: a1    =   1.89635353d5
      real(8), parameter :: a2    =   0.0026d0
      real(8), parameter :: a3    =   0.3474011413d-4

      integer :: i, j, k, m

      real(8) :: a4, a5, a6, a8, a10, b, b6, b8, b10, e1, e2, e3, e4, &
     &   e5, f, f0, fxi, fyi, fzi, g, r, r0, rinv, r0inv, s, s2, sinv, &
     &   s2inv, s6inv, s8inv, s10inv, v, vgrad, xij, yij, zij

!-----------------------------------------------------------------------
!     /*   parameters                                                 */
!-----------------------------------------------------------------------

      r0 = a0 * 1.d-10 / au_length

      r0inv = 1.d0 / r0

      b = 2.d0 * pi / ( x1 - x0 )

      a4  = a3 * a1
      a5  = a3 * a2
      a6  = a3 * c6
      a8  = a3 * c8
      a10 = a3 * c10

      b6  = 6.d0  * a6
      b8  = 8.d0  * a8
      b10 = 10.d0 * a10

!-----------------------------------------------------------------------
!     /*   start main loop                                            */
!-----------------------------------------------------------------------

      do m = 1, nbead

         if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

         k = 0

         do i = 1, natom-1
         do j = i+1, natom

            k = k + 1

            if ( mod( k-1, nprocs_sub ) .ne. myrank_sub ) cycle

            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

            call pbc_atom_MPI( xij, yij, zij )

            r = sqrt( xij*xij + yij*yij + zij*zij )

            rinv = 1.d0 / r

            s = r * r0inv

            s2 = s * s

            sinv   = rinv * r0
            s2inv  = sinv  * sinv
            s6inv  = s2inv * s2inv * s2inv
            s8inv  = s6inv * s2inv
            s10inv = s8inv * s2inv

            e1 = exp( alpha*s + beta*s2 )

            v = a4 * e1

            vgrad = a4 * ( alpha + 2.d0*beta*s ) * e1 * r0inv

            e2 = a6*s6inv + a8*s8inv + a10*s10inv

            e3 = ( b6*s6inv + b8*s8inv + b10*s10inv ) * sinv

            if ( s .ge. d ) then

               v = v - e2

               vgrad = vgrad + e3 * r0inv

            else

               g = d*sinv - 1.d0

               f = exp( - g*g )

               e4 = 2.d0*g*f * d*s2inv

               v = v - e2 * f

               vgrad = vgrad + ( e3*f - e2*e4 ) * r0inv

            end if

            if ( ( s .ge. x0 ) .and. ( s .le. x1 ) ) then

               e5 = b * ( s - x0 ) - 0.5d0 * pi

               v = v + a5 * ( sin(e5) + 1.d0 )

               vgrad = vgrad + ( a5 * b * cos(e5) ) * r0inv

            end if

            pot(m) = pot(m) + v

            f0 = - vgrad * rinv

            fxi = f0 * xij
            fyi = f0 * yij
            fzi = f0 * zij

            fx(i,m) = fx(i,m) + fxi
            fy(i,m) = fy(i,m) + fyi
            fz(i,m) = fz(i,m) + fzi

            fx(j,m) = fx(j,m) - fxi
            fy(j,m) = fy(j,m) - fyi
            fz(j,m) = fz(j,m) - fzi

            vir(1,1) = vir(1,1) + fxi*xij
            vir(1,2) = vir(1,2) + fxi*yij
            vir(1,3) = vir(1,3) + fxi*zij
            vir(2,1) = vir(2,1) + fyi*xij
            vir(2,2) = vir(2,2) + fyi*yij
            vir(2,3) = vir(2,3) + fyi*zij
            vir(3,1) = vir(3,1) + fzi*xij
            vir(3,2) = vir(3,2) + fzi*yij
            vir(3,3) = vir(3,3) + fzi*zij

         end do
         end do

      end do

      call my_mpi_allreduce_md

      return
      end





!***********************************************************************
      subroutine force_helium_aziz1992_MPI
!***********************************************************************
!=======================================================================
!
!     HFD-B2 potential for helium:
!     Aziz, Slaman, Koide, Allnat, Meath, Mol. Phys. 77, 321 (1992)
!
!=======================================================================
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   au_length, boltz, fx, fy, fz, pi, pot, vir, x, y, z, boxinv, &
     &   natom, nbead, iboundary, myrank_main, nprocs_main, myrank_sub, &
     &   nprocs_sub

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8), parameter ::  coef6          =  1.34920045d0
      real(8), parameter ::  coef8          =  0.41365922d0
      real(8), parameter ::  coef10         =  0.17078164d0
      real(8), parameter ::  acoef          =  1.9221529d+5
      real(8), parameter ::  alphap         = 10.73520708d0
      real(8), parameter ::  betap          = -1.89296514d0
      real(8), parameter ::  dcoef          =  1.4135d0
      real(8), parameter ::  epsilon_kelvin = 10.94d0
      real(8), parameter ::  rm_angstrom    =  2.970d0

      integer :: ibead, i, j, k

      real(8) :: rxij, ryij, rzij, rij, rij2, rcut2, reducex, reducex2, &
     &           rxinv, rxinv2, rxinv6, rxinv7, rxinv8, rxinv9, rxinv10, &
     &           rxinv11, dummy, dumarg, pf, dfdx, pg, factf, dgdx, pe, &
     &           dedx, vij, dvdr, wij, fij, fxij, fyij, fzij, epsilon, &
     &           rm, absa, absb, absc, rcut

!-----------------------------------------------------------------------
!     /*   cut off                                                    */
!-----------------------------------------------------------------------

      rcut = 0.d0

      if ( iboundary .ne. 0 ) then

         absa = sqrt ( boxinv(1,1)*boxinv(1,1) &
     &               + boxinv(1,2)*boxinv(1,2) &
     &               + boxinv(1,3)*boxinv(1,3) )
         absb = sqrt ( boxinv(2,1)*boxinv(2,1) &
     &               + boxinv(2,2)*boxinv(2,2) &
     &               + boxinv(2,3)*boxinv(2,3) )
         absc = sqrt ( boxinv(3,1)*boxinv(3,1) &
     &               + boxinv(3,2)*boxinv(3,2) &
     &               + boxinv(3,3)*boxinv(3,3) )

         rcut = 0.5d0 / max( absa, absb, absc )

      end if

      rcut2 = rcut * rcut

!-----------------------------------------------------------------------
!     /*   unit conversion                                            */
!-----------------------------------------------------------------------

      epsilon = epsilon_kelvin * boltz
      rm      = rm_angstrom / ( au_length * 1.d+10 )

      factf   = epsilon / rm

!-----------------------------------------------------------------------
!     /*   start main loop                                            */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

         if ( mod( ibead-1, nprocs_main ) .ne. myrank_main ) cycle

         k = 0

         do i = 1, natom-1
         do j = i+1, natom

            k = k + 1

            if ( mod( k-1, nprocs_sub ) .ne. myrank_sub ) cycle

            rxij = x(i,ibead) - x(j,ibead)
            ryij = y(i,ibead) - y(j,ibead)
            rzij = z(i,ibead) - z(j,ibead)

            call pbc_atom_MPI( rxij, ryij, rzij )

            rij2 = rxij*rxij + ryij*ryij + rzij*rzij

            if ( ( iboundary .ne. 0 ) .and. ( rij2 .ge. rcut2 ) ) cycle

            rij = sqrt( rij2 )

            reducex  = rij / rm
            reducex2 = reducex * reducex
            rxinv    = 1.d0 / reducex
            rxinv2   = rxinv * rxinv
            rxinv6   = rxinv2 * rxinv2 * rxinv2
            rxinv7   = rxinv6 * rxinv
            rxinv8   = rxinv7 * rxinv
            rxinv9   = rxinv8 * rxinv
            rxinv10  = rxinv9 * rxinv
            rxinv11  = rxinv10 * rxinv

            pf = 1.d0
            dfdx = 0.d0

            if ( reducex .lt. dcoef ) then
               dummy = dcoef*rxinv - 1.d0
               dumarg = dummy*dummy
               pf = exp(-dumarg)
               dfdx = 2.d0*dcoef*rxinv2*dummy*pf
            end if

            pg = coef6*rxinv6 + coef8*rxinv8 + coef10*rxinv10

            dgdx = - 6.d0*coef6*rxinv7 - 8.d0*coef8*rxinv9 &
     &             - 10.d0*coef10*rxinv11

            dumarg = -alphap*reducex + betap*reducex2

            pe = acoef*exp(dumarg)

            dedx = (-alphap + 2.d0*betap*reducex)*pe

            vij = pe - pg*pf

            pot(ibead) = pot(ibead) + vij * epsilon

            dvdr = dedx - dgdx*pf - pg*dfdx

            wij = rij*dvdr * factf

            fij = - wij / rij2

            fxij = fij*rxij
            fyij = fij*ryij
            fzij = fij*rzij

            fx(i,ibead) = fx(i,ibead) + fxij
            fy(i,ibead) = fy(i,ibead) + fyij
            fz(i,ibead) = fz(i,ibead) + fzij

            fx(j,ibead) = fx(j,ibead) - fxij
            fy(j,ibead) = fy(j,ibead) - fyij
            fz(j,ibead) = fz(j,ibead) - fzij

            vir(1,1) = vir(1,1) + fxij*rxij
            vir(1,2) = vir(1,2) + fxij*ryij
            vir(1,3) = vir(1,3) + fxij*rzij
            vir(2,1) = vir(2,1) + fyij*rxij
            vir(2,2) = vir(2,2) + fyij*ryij
            vir(2,3) = vir(2,3) + fyij*rzij
            vir(3,1) = vir(3,1) + fzij*rxij
            vir(3,2) = vir(3,2) + fzij*ryij
            vir(3,3) = vir(3,3) + fzij*rzij

         end do
         end do

      end do

      call my_mpi_allreduce_md

      return
      end


