!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     energy and force from GAMESS calculation
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine force_gamess_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, dipx, dipy, dipz, vir, &
     &   au2debye, gamess_command, iounit, iounit_gamess, nbead, &
     &   natom, mbox, myrank, nprocs

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: ibead, i, j, match, ierr, itest

      real(8) :: xi, yi, zi

      character(len=80):: char_line, char_file, char_dummy(3)

      character(len=3) :: char_num

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      ierr = 0

      if ( iset .eq. 0 ) then

         do ibead = 1, nbead

!           /*   allocated bead only   */
            if ( mod( ibead-1, nprocs ) .ne. myrank ) cycle
            call int3_to_char( ibead, char_num )

!           /*   remove old directory and create new directory   */
            call system ('rm -f -r ./' // char_num )
            call system ('mkdir -p ./' // char_num )

!           /*   call MPI_barrier   */
            call my_mpi_barrier

         end do

         if ( myrank .eq. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'input.dat' )

!           /*   search for tag    */
            call search_tag ( '<gamess_command>', 16, iounit, ierr )

!           /*   read a line   */
            read ( iounit, *, iostat=ierr ) gamess_command

!           /*   file close   */
            close ( iounit )

            if ( ierr .ne. 0 ) then

!              /*   file open   */
               open ( iounit, file = 'input_default.dat' )

!              /*   search for tag    */
               call search_tag ( '<gamess_command>', 16, iounit, ierr )

!              /*   read a line
               read ( iounit, *, iostat=ierr ) gamess_command

!              /*   file close   */
               close( iounit )

            end if

!-----------------------------------------------------------------------
!        /*   confirm gamess command                                  */
!-----------------------------------------------------------------------

            call system &
     &         ("echo '0' > test.out")
            call system &
     &         ("sleep 0.1")
            call system &
     &         ("which rungms > /dev/null 2>&1 && echo '1' > test.out")

            open ( iounit, file = 'test.out' )

            read ( iounit, * ) itest

            close( iounit )

            if ( itest .eq. 0 ) then

               ierr = 1

               write( 6, '(a)' ) 'Error - Gamess command not found: ' // &
     &                           'rungms'
               write( 6, '(a)' )

            else

               ierr = 0

               write( 6, '(a)' ) 'Gamess command found: ' // &
     &                           'rungms'
               write( 6, '(a)' )

            end if

            call system('rm -f test.out')

         end if

         call error_handling_MPI &
     &       ( ierr, 'subroutine force_gamess_MPI', 27 )

         call my_mpi_bcast_char_0 &
     &      ( gamess_command, len(gamess_command) )

         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!     /*   skip if `ibead is not my job'                              */
!-----------------------------------------------------------------------

      if ( mod( ibead-1, nprocs ) .ne. myrank ) cycle

!-----------------------------------------------------------------------
!     /*   make char_num according to myrank                          */
!-----------------------------------------------------------------------

      call int3_to_char( ibead, char_num )

!-----------------------------------------------------------------------
!     /*   make gamess input:  input.gamess                           */
!-----------------------------------------------------------------------

!     /*   open the gamess prototype file   */
      open ( iounit, file = 'gamess.dat'   )

!     /*   open the gamess input file   */
      char_file = ('./' // char_num // '/gamess.inp')
      open ( iounit_gamess, file = char_file )

      do

!        /*   read a line   */
         read ( iounit, '(a80)', iostat=ierr ) char_line

!        /*   exit at the end of the line   */
         if ( ierr .ne. 0 ) exit

!        /*   see if the line matches   */
         match = index( char_line(1:2), 'C1' )

!        /*   if matched   */
         if ( match .ge. 1 ) then

!           /*   write a copy of the line   */
            write( iounit_gamess, '(a)' ) char_line

            do i = 1, natom

!              /*   read atomic number   */
               read ( iounit, * ) char_dummy(1), char_dummy(2)

!              /*   boundary condition   */

               xi = x(i,ibead)
               yi = y(i,ibead)
               zi = z(i,ibead)

               call pbc_unfold_MPI &
     &            ( xi, yi, zi, mbox(1,i,ibead), mbox(2,i,ibead), &
     &              mbox(3,i,ibead) )

!              /*   write atomic number and cartesian coordinates   */
               write( iounit_gamess, '(a12,a4,3e20.12)' ) &
     &            char_dummy(1), char_dummy(2), xi, yi, zi

            end do

!        /*   if not matched   */
         else

!           /*   write a copy of the line   */
            write( iounit_gamess, '(a)' ) char_line

         end if

      end do

!     /*   close files   */
      close( iounit )
      close( iounit_gamess )

!-----------------------------------------------------------------------
!     /*   run gamess                                                 */
!-----------------------------------------------------------------------

      call system( &
     &   'cd ' // char_num // '; ' // &
     &   gamess_command // ' ' // char_num // ' > gamess.out; ' // &
     &   'cd ..' )

!-----------------------------------------------------------------------
!     /*   read gamess output:  potential                             */
!-----------------------------------------------------------------------

!     /*   open the gamess output file   */
      char_file = ('./' // char_num // '/gamess.out')
      open ( iounit_gamess, file = char_file )

      do

!        /*   read a line   */
         read ( iounit_gamess, '(a80)', iostat=ierr ) char_line

!        /*   error handling   */
         if ( ierr .ne. 0 ) exit

!        /*   see if the line matches   */
         match = index( char_line(24:37), 'TOTAL ENERGY =' )

!        /*   if matched   */
         if ( match .ge. 1 ) then

!           /*  go back one line   */
            backspace( iounit_gamess )

!           /*   read the potential data   */
            read ( iounit_gamess, *, iostat=ierr ) &
     &         char_dummy(1), char_dummy(2), char_dummy(3), pot(ibead)

         end if

      end do

!     /*   close file   */
      close( iounit_gamess )

!-----------------------------------------------------------------------
!     /*   read gamess output:  potential gradient                    */
!-----------------------------------------------------------------------

!     /*   open the gamess output file   */
      char_file = ('./' // char_num // '/gamess.out')
      open ( iounit_gamess, file = char_file )

      do

!        /*   read a line   */
         read ( iounit_gamess, '(a80)', iostat=ierr ) char_line

!        /*   error handling   */
         if ( ierr .ne. 0 ) exit

!        /*   see if the line matches   */
         match = index( char_line(26:47), 'GRADIENT OF THE ENERGY' )

!        /*   if matched   */
         if ( match .ge. 1 ) then

!           /*   read a line   */
            read ( iounit_gamess, '(a80)', iostat=ierr ) char_line
            read ( iounit_gamess, '(a80)', iostat=ierr ) char_line
            read ( iounit_gamess, '(a80)', iostat=ierr ) char_line

!           /*   count number of lines need to be read   */
            do i = 1, natom

!              /*   read potential gradient data from gamess output   */
               read ( iounit_gamess, *, iostat=ierr ) &
     &            char_dummy(1), char_dummy(2), &
     &            fx(i,ibead), fy(i,ibead), fz(i,ibead)

            end do

!           /*   change sign:  gradient -> force   */

            do i = 1, natom
               fx(i,ibead) = - fx(i,ibead)
               fy(i,ibead) = - fy(i,ibead)
               fz(i,ibead) = - fz(i,ibead)
            end do

         end if

      end do

!     /*   close file   */
      close( iounit_gamess )

!-----------------------------------------------------------------------
!     /*   read gamess output:  dipole moment                         */
!-----------------------------------------------------------------------

!     /*   open the gamess output file   */
      char_file = ('./' // char_num // '/gamess.out')
      open ( iounit_gamess, file = char_file )

      do

!        /*   read a line   */
         read ( iounit_gamess, '(a80)', iostat=ierr ) char_line

!        /*   error handling   */
         if ( ierr .ne. 0 ) exit

!        /*   see if the line matches   */
         match = index( char_line(11:31), 'ELECTROSTATIC MOMENTS' )

!        /*   if matched   */
         if ( match .ge. 1 ) then

!           /*   read a line   */
            read ( iounit_gamess, '(a80)', iostat=ierr ) char_line
            read ( iounit_gamess, '(a80)', iostat=ierr ) char_line
            read ( iounit_gamess, '(a80)', iostat=ierr ) char_line
            read ( iounit_gamess, '(a80)', iostat=ierr ) char_line
            read ( iounit_gamess, '(a80)', iostat=ierr ) char_line

!           /*   read potential gradient data from gamess output   */
            read ( iounit_gamess, *, iostat=ierr ) &
     &         dipx(ibead), dipy(ibead), dipz(ibead)

!           /*   change unit:  debye -> au   */
            dipx(ibead) = dipx(ibead) / au2debye
            dipy(ibead) = dipy(ibead) / au2debye
            dipz(ibead) = dipz(ibead) / au2debye

         end if

      end do

!     /*   close file   */
      close( iounit_gamess )

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz, natom, nbead )

!     /*   dipole moment   */
      call my_mpi_allreduce_real_1 ( dipx, nbead )
      call my_mpi_allreduce_real_1 ( dipy, nbead )
      call my_mpi_allreduce_real_1 ( dipz, nbead )

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

      do j = 1, nbead
      do i = 1, natom

         xi = x(i,j)
         yi = y(i,j)
         zi = z(i,j)

         call pbc_unfold_MPI &
     &     ( xi, yi, zi, mbox(1,i,j), mbox(2,i,j), mbox(3,i,j) )

         vir(1,1) = vir(1,1) + fx(i,j)*xi
         vir(1,2) = vir(1,2) + fx(i,j)*yi
         vir(1,3) = vir(1,3) + fx(i,j)*zi
         vir(2,1) = vir(2,1) + fy(i,j)*xi
         vir(2,2) = vir(2,2) + fy(i,j)*yi
         vir(2,3) = vir(2,3) + fy(i,j)*zi
         vir(3,1) = vir(3,1) + fz(i,j)*xi
         vir(3,2) = vir(3,2) + fz(i,j)*yi
         vir(3,3) = vir(3,3) + fz(i,j)*zi

      end do
      end do

      return
      end

