!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, H. Kimizuka
!      Last updated:    Jun 4, 2022 by M. Shiga
!      Description:     add external force
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine force_external_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   mechanical force                                           */
!-----------------------------------------------------------------------

      call force_mech_MPI

      call force_freeze_MPI

      call force_mech_add_MPI

!-----------------------------------------------------------------------
!     /*   best for oniom or qmmm                                     */
!-----------------------------------------------------------------------

      call force_best_MPI

      call force_best_add_MPI

      return
      end





!***********************************************************************
      subroutine force_freeze_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   fx, fy, fz, atom_change, natom

      use mech_variables, only : &
     &   fx_mech, fy_mech, fz_mech, nfreeze_mech

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

!        /*   setup mechanical force   */
         call force_freeze_setup_MPI

!        /*   set complete   */
         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   see existence                                              */
!-----------------------------------------------------------------------

      if ( nfreeze_mech .le. 0 ) return

!-----------------------------------------------------------------------
!     /*   freeze atoms                                               */
!-----------------------------------------------------------------------

!     /*   loop of fixed atoms   */
      do i = 1, natom

!        /*   frozen atoms   */
         if ( atom_change(i)(1:7) .ne. 'FREEZE ' ) cycle

!        /*   mechanical force   */
         fx_mech(i,:) = - fx(i,:)
         fy_mech(i,:) = - fy(i,:)
         fz_mech(i,:) = - fz(i,:)

!     /*   loop of fixed atoms   */
      end do

      return
      end





!***********************************************************************
      subroutine force_freeze_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, myrank_world

      use mech_variables, only : &
     &   nfreeze_mech

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: ierr = 0

!     /*   integers   */
      integer :: i, j, k, nline

!     /*   initial setting   */
      integer, save :: iset = 0

!     /*   characters   */
      character(len=9) :: char

!-----------------------------------------------------------------------
!     /*   external force                                             */
!-----------------------------------------------------------------------

      if ( iset .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   default values                                             */
!-----------------------------------------------------------------------

!     /*   number of fixed atoms   */
      nfreeze_mech = 0

!-----------------------------------------------------------------------
!     /*   set up                                                     */
!-----------------------------------------------------------------------

!     /*   master rank   */
      if ( myrank_world .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   tag   */
      call search_tag ( '<atom_change>', 13, iounit, ierr )

!     /*   if no error  */
      if ( ierr .eq. 0 ) then

!        /*   fixed atoms   */
         read ( iounit, *, iostat=ierr ) nline

!        /*   lines   */
         do k = 1, nline

            read ( iounit, *, iostat=ierr ) char, i, j

!           /*   read correctly   */
            if ( ierr .eq. 0 ) then

!            /*   fixed atoms   */
            if ( char(1:7) .eq. 'FREEZE ' ) then

!               /*   fixed atoms are present   */
                nfreeze_mech = nfreeze_mech + max( 0, j-i+1 )

!            /*   fixed atoms   */
             end if

!            /*   read correctly   */
             end if

!        /*   lines   */
         end do

!     /*   if no error  */
      end if

!     /*   file close   */
      close( iounit )

!     /*   master rank   */
      end if

!-----------------------------------------------------------------------
!     /*   communications                                             */
!-----------------------------------------------------------------------

      call my_mpi_bcast_int_0 ( ierr )

      call my_mpi_bcast_int_0 ( nfreeze_mech )
 
!-----------------------------------------------------------------------
!     /*   print information                                          */
!-----------------------------------------------------------------------

      if ( nfreeze_mech .ge. 1 ) then

         if ( ierr .ne. 0 ) then
            if ( myrank_world .eq. 0 ) then
               write( 6, '(a)' ) 'Error - FREEZE input.'
               write( 6, '(a)' ) 
            end if
         else
            if ( myrank_world .eq. 0 ) then
               write( 6, '(a)' ) 'Frozen atoms are present.'
               write( 6, '(a)' ) 
            end if
         end if

      else

         ierr = 0

      end if

!-----------------------------------------------------------------------
!     /*   error termination                                          */
!-----------------------------------------------------------------------

      call error_handling_MPI &
     &   ( ierr, 'subroutine force_freeze_setup_MPI', 33 )

!-----------------------------------------------------------------------
!     /*   set end                                                    */
!-----------------------------------------------------------------------

      iset = 1

      return
      end





!***********************************************************************
      subroutine force_mech_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, istep, nbead, iounit, myrank

      use mech_variables, only : &
     &   pot_mech, iprint_mech, mech_type

      use common_variables, only : &
     &   method, istep, istep_start, istep_hmc

      use hmc_variables, only : &
     &   jstep

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: m
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      if ( iset .eq. 0 ) then

!        /*   setup mechanical force   */
         call force_mech_setup_MPI

!        /*   set complete   */
         iset = 1

      end if

!-----------------------------------------------------------------------
!     /*   mechanical force                                           */
!-----------------------------------------------------------------------

      if      ( mech_type(1:5) .eq. 'NONE ' ) then

         return

      else if ( mech_type(1:5) .eq. 'EFEI ' ) then

         call force_mech_efei_MPI

      else if ( mech_type(1:5) .eq. 'AFIR ' ) then

         call force_mech_afir_MPI

      else if ( mech_type(1:8) .eq. 'CLUSTER ' ) then

         call force_mech_cluster_MPI

      else if ( mech_type(1:6) .eq. 'GROUP ' ) then

         call force_mech_group_MPI

      else if ( mech_type(1:5) .eq. 'ZONE ' ) then

         call force_mech_zone_MPI

      else if ( mech_type(1:4) .eq. 'ESF ' ) then

         call force_mech_esf_MPI

      else

         call error_handling_MPI( 1, 'subroutine force_mech_MPI', 25 )

      end if

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      if ( method(1:6) .eq. 'PIHMC ' ) then

         if ( myrank .eq. 0 ) then
         if ( iprint_mech .gt. 0 ) then
         if ( mod(istep,iprint_mech) .eq. 0 ) then

            if ( istep .eq. istep_start ) then

               open ( iounit, file = 'mech.out', access = 'append' )

               do m = 1, nbead
                  write( iounit, '(i8,3e24.16)' ) &
     &               istep, pot(m), pot_mech(m), pot(m)+pot_mech(m)
               end do

               close( iounit )

            else if ( jstep .eq. istep_hmc ) then

               open ( iounit, file = 'mech.out', access = 'append' )

               do m = 1, nbead
                  write( iounit, '(i8,3e24.16)' ) &
     &               istep, pot(m), pot_mech(m), pot(m)+pot_mech(m)
               end do

               close( iounit )

            end if

         end if
         end if
         end if

      else

         if ( myrank .eq. 0 ) then
         if ( iprint_mech .gt. 0 ) then
         if ( mod(istep,iprint_mech) .eq. 0 ) then

            open ( iounit, file = 'mech.out', access = 'append' )

            do m = 1, nbead
               write( iounit, '(i8,3e24.16)' ) &
     &            istep, pot(m), pot_mech(m), pot(m)+pot_mech(m)
            end do

            close( iounit )

         end if
         end if
         end if

      end if

      return
      end





!***********************************************************************
      subroutine force_mech_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, natom, nbead, myrank

      use mech_variables, only : &
     &   pot_mech, fx_mech, fy_mech, fz_mech, vir_mech, iprint_mech, &
     &   mech_type

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer i, j, ierr
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   external force                                             */
!-----------------------------------------------------------------------

      if ( iset .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   external force                                             */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

!        /*   file open   */
         open (iounit, file = 'input.dat')

!        /*   tag   */
         call search_tag ( '<mech_type>', 11, iounit, ierr )

!        /*   if no error  */
         if ( ierr .eq. 0 ) then

!           /*   option   */
            read( iounit, *, iostat=ierr ) mech_type

         end if

!        /*   file close   */
         close(iounit)

!        /*   if error is found, read default value   */
         if ( ierr .ne. 0 ) then

!           /*   file open   */
            open (iounit, file = 'input_default.dat')

!           /*   tag   */
            call search_tag ( '<mech_type>', 11, iounit, ierr )

!           /*   option   */
            read( iounit, *, iostat=ierr ) mech_type

!           /*   file close   */
            close(iounit)

         end if

      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   error termination   */
      if ( ierr .ne. 0 ) then
         if ( myrank .eq. 0 ) then
            write( 6, '(a)' ) 'Error - <mech_type> read incorrectly.'
            write( 6, '(a)' )
         end if
      end if

!     /*   error handling  */
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_mech_setup_MPI', 31 )

!-----------------------------------------------------------------------
!     /*   communications                                             */
!-----------------------------------------------------------------------

      call my_mpi_bcast_char_0 ( mech_type, 8 )

!-----------------------------------------------------------------------
!     /*   print interval                                             */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

!        /*   file open   */
         open (iounit, file = 'input.dat')

!        /*   tag   */
         call search_tag ( '<iprint_mech>', 13, iounit, ierr )

!        /*   if no error  */
         if ( ierr .eq. 0 ) then

!           /*   option   */
            read( iounit, *, iostat=ierr ) iprint_mech

         end if

!        /*   file close   */
         close(iounit)

!        /*   if error is found, read default value   */
         if ( ierr .ne. 0 ) then

!           /*   file open   */
            open (iounit, file = 'input_default.dat')

!           /*   tag   */
            call search_tag ( '<iprint_mech>', 13, iounit, ierr )

!           /*   option   */
            read( iounit, *, iostat=ierr ) iprint_mech

!           /*   file close   */
            close(iounit)

         end if

      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   error termination   */
      if ( ierr .ne. 0 ) then
         if ( myrank .eq. 0 ) then
            write( 6, '(a)' ) 'Error - <iprint_type> read incorrectly.'
            write( 6, '(a)' )
         end if
      end if

!     /*   error handling  */
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_mech_setup_MPI', 31 )

!-----------------------------------------------------------------------
!     /*   communications                                             */
!-----------------------------------------------------------------------

      call my_mpi_bcast_int_0 ( iprint_mech )

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( .not. allocated( pot_mech ) ) &
     &   allocate( pot_mech(nbead) )

      if ( .not. allocated( fx_mech ) ) &
     &   allocate( fx_mech(natom,nbead) )
      if ( .not. allocated( fy_mech ) ) &
     &   allocate( fy_mech(natom,nbead) )
      if ( .not. allocated( fz_mech ) ) &
     &   allocate( fz_mech(natom,nbead) )

      if ( .not. allocated( vir_mech ) ) &
     &   allocate( vir_mech(3,3) )

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   mechanical potential   */
      do i = 1, nbead
         pot_mech(i) = 0.d0
      end do

!     /*   mechanical force   */
      do i = 1, nbead
      do j = 1, natom
         fx_mech(j,i) = 0.d0
         fy_mech(j,i) = 0.d0
         fz_mech(j,i) = 0.d0
      end do
      end do

!     /*   mechanical virial   */
      do i = 1, 3
      do j = 1, 3
         vir_mech(j,i) = 0.d0
      end do
      end do

!-----------------------------------------------------------------------
!     /*   read parameters                                            */
!-----------------------------------------------------------------------

      if      ( mech_type(1:5) .eq. 'NONE ' ) then

         continue

      else if ( mech_type(1:5) .eq. 'EFEI ' ) then

         call force_mech_efei_setup_MPI

      else if ( mech_type(1:5) .eq. 'AFIR ' ) then

         call force_mech_afir_setup_MPI

      else if ( mech_type(1:8) .eq. 'CLUSTER ' ) then

         call force_mech_cluster_setup_MPI

      else if ( mech_type(1:6) .eq. 'GROUP ' ) then

         call force_mech_group_setup_MPI

      else if ( mech_type(1:5) .eq. 'ZONE ' ) then

         call force_mech_zone_setup_MPI

      else if ( mech_type(1:4) .eq. 'ESF ' ) then

         call force_mech_esf_setup_MPI

      else

         call error_handling_MPI &
     &      ( 1, 'subroutine force_mech_setup_MPI', 31 )

      end if

!-----------------------------------------------------------------------
!     /*   set end                                                    */
!-----------------------------------------------------------------------

      iset = 1

      return
      end





!***********************************************************************
      subroutine force_mech_efei_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, myrank

      use mech_variables, only : &
     &   fc_mech, i_mech, j_mech

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer ierr

!     /*   integers   */
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   external force                                             */
!-----------------------------------------------------------------------

      if ( iset .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   set up                                                     */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

!        /*   file open   */
         open (iounit, file = 'input.dat')

!        /*   tag   */
         call search_tag ( '<efei>', 6, iounit, ierr )

!        /*   if no error  */
         if ( ierr .eq. 0 ) then

!           /*   memory allocation   */
            if ( .not. allocated(  i_mech ) ) &
     &         allocate (  i_mech(1) )
            if ( .not. allocated(  j_mech ) ) &
     &         allocate (  j_mech(1) )
            if ( .not. allocated( fc_mech ) ) &
     &         allocate ( fc_mech(1) )

!           /*   atom 1, atom 2, force   */
            read( iounit, *, iostat=ierr ) &
     &         i_mech(1), j_mech(1), fc_mech(1)

         end if

!        /*   file close   */
         close(iounit)

!        /*   if error is found, read default value   */
         if ( ierr .ne. 0 ) then

!           /*   file open   */
            open (iounit, file = 'input_default.dat')

!           /*   tag   */
            call search_tag ( '<efei>', 6, iounit, ierr )

!           /*   memory allocation   */
            if ( .not. allocated(  i_mech ) ) &
     &         allocate (  i_mech(1) )
            if ( .not. allocated(  j_mech ) ) &
     &         allocate (  j_mech(1) )
            if ( .not. allocated( fc_mech ) ) &
     &         allocate ( fc_mech(1) )

!           /*   atom 1, atom 2, force   */
            read( iounit, *, iostat=ierr ) &
     &         i_mech(1), j_mech(1), fc_mech(1)

!           /*   file close   */
            close(iounit)

         end if

      end if

!-----------------------------------------------------------------------
!     /*   communications                                             */
!-----------------------------------------------------------------------

      call my_mpi_bcast_int_0 ( ierr )

!-----------------------------------------------------------------------
!     /*   error termination                                          */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) 'Error - EFEI input.'
         write( 6, '(a)' ) 
      end if
      end if

      call error_handling_MPI &
     &   ( ierr, 'subroutine force_mech_efei_setup_MPI', 36 )

!-----------------------------------------------------------------------
!     /*   communications                                             */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 ) then
         if ( .not. allocated(  i_mech ) ) &
     &      allocate (  i_mech(1) )
         if ( .not. allocated(  j_mech ) ) &
     &      allocate (  j_mech(1) )
         if ( .not. allocated( fc_mech ) ) &
     &      allocate ( fc_mech(1) )
      end if

      call my_mpi_bcast_int_1 (  i_mech, 1 )
      call my_mpi_bcast_int_1 (  j_mech, 1 )
      call my_mpi_bcast_real_1( fc_mech, 1 )

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

         write( 6, '(a)' ) 'External force explicitly included.'
         write( 6, '(a)' ) 

      end if

!-----------------------------------------------------------------------
!     /*   set end                                                    */
!-----------------------------------------------------------------------

      iset = 1

      return
      end





!***********************************************************************
      subroutine force_mech_afir_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, myrank

      use mech_variables, only : &
     &   fc_mech, i_mech, j_mech, nafir_mech

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: k, ierr

!     /*   integers   */
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   external force                                             */
!-----------------------------------------------------------------------

      if ( iset .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   set up                                                     */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   tag   */
         call search_tag ( '<afir>', 6, iounit, ierr )

!        /*   atom 1, atom 2, force   */
         read( iounit, *, iostat=ierr ) nafir_mech

!        /*   if no error  */
         if ( ierr .eq. 0 ) then

!           /*   memory allocation   */
            if ( .not. allocated(  i_mech ) ) &
     &         allocate (  i_mech(nafir_mech) )
            if ( .not. allocated(  j_mech ) ) &
     &         allocate (  j_mech(nafir_mech) )
            if ( .not. allocated( fc_mech ) ) &
     &         allocate ( fc_mech(nafir_mech) )

!           /*   atom 1, atom 2, force   */
            do k = 1, nafir_mech
               read( iounit, *, iostat=ierr ) &
     &            i_mech(k), j_mech(k), fc_mech(k)
            end do

         end if

!        /*   file close   */
         close( iounit )

!        /*   if error is found, read default value   */
         if ( ierr .ne. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'input_default.dat' )

!           /*   tag   */
            call search_tag ( '<afir>', 6, iounit, ierr )

!           /*   atom 1, atom 2, force   */
            read( iounit, *, iostat=ierr ) nafir_mech

!           /*   memory allocation   */
            if ( .not. allocated(  i_mech ) ) &
     &         allocate (  i_mech(nafir_mech) )
            if ( .not. allocated(  j_mech ) ) &
     &         allocate (  j_mech(nafir_mech) )
            if ( .not. allocated( fc_mech ) ) &
     &         allocate ( fc_mech(nafir_mech) )

!           /*   atom 1, atom 2, force   */
            do k = 1, nafir_mech
               read( iounit, *, iostat=ierr ) &
     &            i_mech(k), j_mech(k), fc_mech(k)
            end do

!           /*   file close   */
            close( iounit )

         end if

      end if

!-----------------------------------------------------------------------
!     /*   communications                                             */
!-----------------------------------------------------------------------

      call my_mpi_bcast_int_0 ( ierr )

!-----------------------------------------------------------------------
!     /*   error termination                                          */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) 'Error - AFIR input.'
         write( 6, '(a)' ) 
      end if
      end if

      call error_handling_MPI &
     &   ( ierr, 'subroutine force_mech_afir_setup_MPI', 36 )

!-----------------------------------------------------------------------
!     /*   communications                                             */
!-----------------------------------------------------------------------

      call my_mpi_bcast_int_0 ( nafir_mech )

      if ( myrank .ne. 0 ) then
         if ( .not. allocated(  i_mech ) ) &
     &      allocate (  i_mech(nafir_mech) )
         if ( .not. allocated(  j_mech ) ) &
     &      allocate (  j_mech(nafir_mech) )
         if ( .not. allocated( fc_mech ) ) &
     &      allocate ( fc_mech(nafir_mech) )
      end if

      call my_mpi_bcast_int_1 (  i_mech, nafir_mech )
      call my_mpi_bcast_int_1 (  j_mech, nafir_mech )
      call my_mpi_bcast_real_1( fc_mech, nafir_mech )

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

         write( 6, '(a)' ) 'External force explicitly included.'
         write( 6, '(a)' ) 

      end if

!-----------------------------------------------------------------------
!     /*   set end                                                    */
!-----------------------------------------------------------------------

      iset = 1

      return
      end





!***********************************************************************
      subroutine force_mech_cluster_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, myrank

      use mech_variables, only : &
     &   fc_mech, r_mech

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer ierr

!     /*   initial setting   */
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   external force                                             */
!-----------------------------------------------------------------------

      if ( iset .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   set up                                                     */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

!        /*   file open   */
         open (iounit, file = 'input.dat')

!        /*   tag   */
         call search_tag ( '<cluster>', 9, iounit, ierr )

!        /*   if no error  */
         if ( ierr .eq. 0 ) then

!           /*   memory allocation   */
            if ( .not. allocated(  r_mech ) ) &
     &         allocate (  r_mech(1) )
            if ( .not. allocated( fc_mech ) ) &
     &         allocate ( fc_mech(1) )

!           /*   force   */
            read( iounit, *, iostat=ierr ) &
     &         r_mech(1), fc_mech(1)

         end if

!        /*   file close   */
         close(iounit)

!        /*   if error is found, read default value   */
         if ( ierr .ne. 0 ) then

!           /*   file open   */
            open (iounit, file = 'input_default.dat')

!           /*   tag   */
            call search_tag ( '<cluster>', 9, iounit, ierr )

!           /*   memory allocation   */
            if ( .not. allocated(  r_mech ) ) &
     &         allocate (  r_mech(1) )
            if ( .not. allocated( fc_mech ) ) &
     &         allocate ( fc_mech(1) )

!           /*   atom 1, atom 2, force   */
            read( iounit, *, iostat=ierr ) &
     &         r_mech(1), fc_mech(1)

!           /*   file close   */
            close(iounit)

         end if

      end if

!-----------------------------------------------------------------------
!     /*   communications                                             */
!-----------------------------------------------------------------------

      call my_mpi_bcast_int_0 ( ierr )

!-----------------------------------------------------------------------
!     /*   error termination                                          */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) 'Error - CLUSTER input.'
         write( 6, '(a)' ) 
      end if
      end if

      call error_handling_MPI &
     &   ( ierr, 'subroutine force_mech_cluster_setup_MPI', 39 )

!-----------------------------------------------------------------------
!     /*   communications                                             */
!-----------------------------------------------------------------------

      if ( myrank .ne. 0 ) then
         if ( .not. allocated(  r_mech ) ) &
     &      allocate (  r_mech(1) )
         if ( .not. allocated( fc_mech ) ) &
     &      allocate ( fc_mech(1) )
      end if

      call my_mpi_bcast_real_1(  r_mech, 1 )
      call my_mpi_bcast_real_1( fc_mech, 1 )

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

         write( 6, '(a)' ) 'Cluster boundary condition.'
         write( 6, '(a)' ) 

      end if

!-----------------------------------------------------------------------
!     /*   set end                                                    */
!-----------------------------------------------------------------------

      iset = 1

      return
      end





!***********************************************************************
      subroutine force_mech_efei_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, natom, nbead, myrank, nprocs

      use mech_variables, only : &
     &   pot_mech, fx_mech, fy_mech, fz_mech, fc_mech, vir_mech, &
     &   i_mech, j_mech

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j, m

!     /*   real numbers   */
      real(8) :: xij, yij, zij, rij, rij2, fxi, fyi, fzi

!     /*   mechanical potential   */
      do i = 1, nbead
         pot_mech(i) = 0.d0
      end do

!     /*   mechanical force   */
      do i = 1, nbead
      do j = 1, natom
         fx_mech(j,i) = 0.d0
         fy_mech(j,i) = 0.d0
         fz_mech(j,i) = 0.d0
      end do
      end do

!     /*   mechanical virial   */
      do i = 1, 3
      do j = 1, 3
         vir_mech(j,i) = 0.d0
      end do
      end do

!     /*   loop of beads   */
      do m = 1, nbead

!        /*   select bead   */
         if ( mod(m-1,nprocs) .ne. myrank ) cycle

!        /*   pair of atoms   */
         i = i_mech(1)
         j = j_mech(1)

!        /*   distance   */
         xij = x(i,m) - x(j,m)
         yij = y(i,m) - y(j,m)
         zij = z(i,m) - z(j,m)

!        /*   periodic boundary condition   */
         call pbc_atom_MPI ( xij, yij, zij )

!        /*   distance   */
         rij2 = xij*xij + yij*yij + zij*zij

!        /*   interatomic distance   */
         rij  = sqrt(rij2)

!        /*   mechanical potential   */

         pot_mech(m) = pot_mech(m) - fc_mech(1)*rij

!        /*   mechanical force   */

         fxi = fc_mech(1)*xij/rij
         fyi = fc_mech(1)*yij/rij
         fzi = fc_mech(1)*zij/rij

         fx_mech(i,m) = fx_mech(i,m) + fxi
         fy_mech(i,m) = fy_mech(i,m) + fyi
         fz_mech(i,m) = fz_mech(i,m) + fzi

         fx_mech(j,m) = fx_mech(j,m) - fxi
         fy_mech(j,m) = fy_mech(j,m) - fyi
         fz_mech(j,m) = fz_mech(j,m) - fzi

!        /*   mechanical virial   */

         vir_mech(1,1) = vir_mech(1,1) + fxi*xij
         vir_mech(1,2) = vir_mech(1,2) + fxi*yij
         vir_mech(1,3) = vir_mech(1,3) + fxi*zij
         vir_mech(2,1) = vir_mech(2,1) + fyi*xij
         vir_mech(2,2) = vir_mech(2,2) + fyi*yij
         vir_mech(2,3) = vir_mech(2,3) + fyi*zij
         vir_mech(3,1) = vir_mech(3,1) + fzi*xij
         vir_mech(3,2) = vir_mech(3,2) + fzi*yij
         vir_mech(3,3) = vir_mech(3,3) + fzi*zij

!     /*   loop of beads   */
      end do

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot_mech, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx_mech, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy_mech, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz_mech, natom, nbead )

!     /*   virial   */
      call my_mpi_allreduce_real_2 ( vir_mech, 3, 3 )

      return
      end





!***********************************************************************
      subroutine force_mech_afir_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, natom, nbead, myrank, nprocs

      use mech_variables, only : &
     &   pot_mech, fx_mech, fy_mech, fz_mech, fc_mech, vir_mech, &
     &   i_mech, j_mech, nafir_mech

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j, k, m

!     /*   real numbers   */
      real(8) :: xij, yij, zij, rij, rij2, fxi, fyi, fzi

!     /*   mechanical potential   */
      do i = 1, nbead
         pot_mech(i) = 0.d0
      end do

!     /*   mechanical force   */
      do i = 1, nbead
      do j = 1, natom
         fx_mech(j,i) = 0.d0
         fy_mech(j,i) = 0.d0
         fz_mech(j,i) = 0.d0
      end do
      end do

!     /*   mechanical virial   */
      do i = 1, 3
      do j = 1, 3
         vir_mech(j,i) = 0.d0
      end do
      end do

!     /*   loop of beads   */
      do m = 1, nbead

!        /*   select bead   */
         if ( mod(m-1,nprocs) .ne. myrank ) cycle

!        /*   loop of mechanical forces   */
         do k = 1, nafir_mech

!           /*   pair of atoms   */
            i = i_mech(k)
            j = j_mech(k)

!           /*   distance   */
            xij = x(i,m) - x(j,m)
            yij = y(i,m) - y(j,m)
            zij = z(i,m) - z(j,m)

!           /*   periodic boundary condition   */
            call pbc_atom_MPI ( xij, yij, zij )

!           /*   distance   */
            rij2 = xij*xij + yij*yij + zij*zij

!           /*   interatomic distance   */
            rij  = sqrt(rij2)

!           /*   mechanical potential (negative of efei)  */
            pot_mech(m) = pot_mech(m) + fc_mech(1)*rij

!           /*   mechanical force (negative of efei)   */

            fxi = - fc_mech(k) * xij / rij
            fyi = - fc_mech(k) * yij / rij
            fzi = - fc_mech(k) * zij / rij

            fx_mech(i,m) = fx_mech(i,m) + fxi
            fy_mech(i,m) = fy_mech(i,m) + fyi
            fz_mech(i,m) = fz_mech(i,m) + fzi

            fx_mech(j,m) = fx_mech(j,m) - fxi
            fy_mech(j,m) = fy_mech(j,m) - fyi
            fz_mech(j,m) = fz_mech(j,m) - fzi

!           /*   mechanical virial (negative of efei)   */

            vir_mech(1,1) = vir_mech(1,1) + fxi*xij
            vir_mech(1,2) = vir_mech(1,2) + fxi*yij
            vir_mech(1,3) = vir_mech(1,3) + fxi*zij
            vir_mech(2,1) = vir_mech(2,1) + fyi*xij
            vir_mech(2,2) = vir_mech(2,2) + fyi*yij
            vir_mech(2,3) = vir_mech(2,3) + fyi*zij
            vir_mech(3,1) = vir_mech(3,1) + fzi*xij
            vir_mech(3,2) = vir_mech(3,2) + fzi*yij
            vir_mech(3,3) = vir_mech(3,3) + fzi*zij

!        /*   loop of mechanical forces   */
         end do

!     /*   loop of beads   */
      end do

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot_mech, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx_mech, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy_mech, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz_mech, natom, nbead )

!     /*   virial   */
      call my_mpi_allreduce_real_2 ( vir_mech, 3, 3 )

      return
      end





!***********************************************************************
      subroutine force_mech_cluster_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, physmass, natom, nbead, myrank, nprocs

      use mech_variables, only : &
     &   pot_mech, fx_mech, fy_mech, fz_mech, vir_mech, fc_mech, r_mech

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j, m

!     /*   real numbers   */
      real(8) :: rx, ry, rz, r2, r2_cluster, fc_cluster, xg, yg, zg, &
     &           fxi, fyi, fzi, pm, pminv, factor, r1, r1_cluster, &
     &           dr1, dr2

!     /*   mechanical potential   */
      do i = 1, nbead
         pot_mech(i) = 0.d0
      end do

!     /*   mechanical force   */
      do i = 1, nbead
      do j = 1, natom
         fx_mech(j,i) = 0.d0
         fy_mech(j,i) = 0.d0
         fz_mech(j,i) = 0.d0
      end do
      end do

!     /*   mechanical virial   */
      do i = 1, 3
      do j = 1, 3
         vir_mech(j,i) = 0.d0
      end do
      end do

!     /*   loop of beads   */
      do m = 1, nbead

!        /*   select bead   */
         if ( mod(m-1,nprocs) .ne. myrank ) cycle

!        //   square of cluster radius
         r2_cluster = r_mech(1) * r_mech(1)

!        //   force constant
         fc_cluster = fc_mech(1)

!        //   center of mass position

         xg = 0.d0
         yg = 0.d0
         zg = 0.d0

         pm = 0.d0

         do i = 1, natom

            xg = xg + physmass(i) * x(i,m)
            yg = yg + physmass(i) * y(i,m)
            zg = zg + physmass(i) * z(i,m)

            pm = pm + physmass(i)

         end do

         pminv = 1.d0 / pm

         xg = xg * pminv
         yg = yg * pminv
         zg = zg * pminv

!        //   loop of atoms
         do i = 1, natom

!           //   relative distance from center of mass

            rx = x(i,m) - xg
            ry = y(i,m) - yg
            rz = z(i,m) - zg

!           //   periodic boundary condition
            call pbc_atom_MPI ( rx, ry, rz )

!           //   distance squared
            r2 = rx*rx + ry*ry + rz*rz

!           //   applied beyond cluster radius
            if ( r2 .gt. r2_cluster ) then

!              //   distance
               r1 = sqrt( r2 )

!              //   cluster radius
               r1_cluster = r_mech(1)

!              //   difference
               dr1 = r1 - r1_cluster

!              //   difference squared
               dr2 = dr1 * dr1

!              //   potential
               pot_mech(m) = pot_mech(m) + 0.5d0 * fc_cluster * dr2

!              //   force
               fxi = - fc_cluster * rx * dr1 / r1
               fyi = - fc_cluster * ry * dr1 / r1
               fzi = - fc_cluster * rz * dr1 / r1

!              //   force on atom i
               fx_mech(i,m) = fx_mech(i,m) + fxi
               fy_mech(i,m) = fy_mech(i,m) + fyi
               fz_mech(i,m) = fz_mech(i,m) + fzi

!              //   force on atom j
               do j = 1, natom
                  factor = physmass(j) * pminv
                  fx_mech(j,m) = fx_mech(j,m) - fxi * factor
                  fy_mech(j,m) = fy_mech(j,m) - fyi * factor
                  fz_mech(j,m) = fz_mech(j,m) - fzi * factor
               end do

!              //   virial
               vir_mech(1,1) = vir_mech(1,1) + fxi*rx
               vir_mech(1,2) = vir_mech(1,2) + fxi*ry
               vir_mech(1,3) = vir_mech(1,3) + fxi*rz
               vir_mech(2,1) = vir_mech(2,1) + fyi*rx
               vir_mech(2,2) = vir_mech(2,2) + fyi*ry
               vir_mech(2,3) = vir_mech(2,3) + fyi*rz
               vir_mech(3,1) = vir_mech(3,1) + fzi*rx
               vir_mech(3,2) = vir_mech(3,2) + fzi*ry
               vir_mech(3,3) = vir_mech(3,3) + fzi*rz

!           //   applied beyond cluster radius
            end if

!        //   loop of atoms
         end do

!     /*   loop of beads   */
      end do

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot_mech, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx_mech, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy_mech, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz_mech, natom, nbead )

!     /*   virial   */
      call my_mpi_allreduce_real_2 ( vir_mech, 3, 3 )

      return
      end





!***********************************************************************
      subroutine force_mech_add_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, fx, fy, fz, vir, natom, nbead

      use mech_variables, only : &
     &   pot_mech, fx_mech, fy_mech, fz_mech, vir_mech, mech_type, &
     &   nfreeze_mech

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j

!-----------------------------------------------------------------------
!     /*   skip if mechanical force is zero                           */
!-----------------------------------------------------------------------

      if ( ( nfreeze_mech .le. 0 ) .and. &
     &     ( mech_type(1:5) .eq. 'NONE ' ) ) return

!-----------------------------------------------------------------------
!     /*   add contributions                                          */
!-----------------------------------------------------------------------

      do i = 1, nbead
         pot(i) = pot(i) + pot_mech(i)
      end do

      do i = 1, nbead
      do j = 1, natom
         fx(j,i) = fx(j,i) + fx_mech(j,i)
         fy(j,i) = fy(j,i) + fy_mech(j,i)
         fz(j,i) = fz(j,i) + fz_mech(j,i)
      end do
      end do

      do i = 1, 3
      do j = 1, 3
         vir(j,i) = vir(j,i) + vir_mech(j,i)
      end do
      end do

      return
      end





!***********************************************************************
      subroutine force_mech_group_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, iounit_tmp, myrank

      use mech_variables, only : &
     &   group_mech, ngroup_mech

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer ierr
      integer :: i, j

!     /*   integers   */
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   external force                                             */
!-----------------------------------------------------------------------

      if ( iset .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   set up                                                     */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

!        /*   file open   */
         open (iounit, file = 'input.dat')

!        /*   tag   */
         call search_tag ( '<group>', 7, iounit, ierr )

!        /*   if no error  */
         if ( ierr .eq. 0 ) then

!           /*   Number of groups   */
            read( iounit, *, iostat=ierr ) &
     &         ngroup_mech

!           /*   memory allocation   */
            if ( .not. allocated( group_mech ) ) &
     &         allocate ( group_mech(ngroup_mech) )

            do i = 1, ngroup_mech
               read ( iounit, *, iostat=ierr ) &
     &            group_mech(i)%label, &
     &            group_mech(i)%natom_each, &
     &            group_mech(i)%cfx, &
     &            group_mech(i)%cfy, &
     &            group_mech(i)%cfz, &
     &            group_mech(i)%flag_pot

!              /*   memory allocation   */
               if ( .not. allocated( group_mech(i)%id ) ) &
     &            allocate (group_mech(i)%id(group_mech(i)%natom_each))

               open (iounit_tmp, file = group_mech(i)%label)

               do j = 1, group_mech(i)%natom_each
                  read ( iounit_tmp, *, iostat=ierr ) &
     &               group_mech(i)%id(j)
               enddo

               close(iounit_tmp)

            enddo

         end if

!        /*   file close   */
         close(iounit)

!        /*   if error is found, read default value   */
         if ( ierr .ne. 0 ) then

!           /*   file open   */
            open (iounit, file = 'input_default.dat')

!           /*   tag   */
            call search_tag ( '<group>', 7, iounit, ierr )

!           /*   Number of groups   */
            read( iounit, *, iostat=ierr ) &
     &         ngroup_mech

!           /*   memory allocation   */
            if ( .not. allocated( group_mech ) ) &
     &         allocate ( group_mech(ngroup_mech) )

            do i = 1, ngroup_mech
               read ( iounit, *, iostat=ierr ) &
     &            group_mech(i)%label, &
     &            group_mech(i)%natom_each, &
     &            group_mech(i)%cfx, &
     &            group_mech(i)%cfy, &
     &            group_mech(i)%cfz, &
     &            group_mech(i)%flag_pot

!              /*   memory allocation   */
               if ( .not. allocated( group_mech(i)%id ) ) &
     &            allocate (group_mech(i)%id(group_mech(i)%natom_each))

               open (iounit_tmp, file = group_mech(i)%label)

               do j = 1, group_mech(i)%natom_each
                  read ( iounit_tmp, *, iostat=ierr ) &
     &               group_mech(i)%id(j)
               enddo

               close(iounit_tmp)

            enddo

!           /*   file close   */
            close(iounit)

         end if

      end if

!-----------------------------------------------------------------------
!     /*   communications                                             */
!-----------------------------------------------------------------------

      call my_mpi_bcast_int_0 ( ierr )

!-----------------------------------------------------------------------
!     /*   error termination                                          */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) 'Error - GROUP input.'
         write( 6, '(a)' ) 
      end if
      end if

      call error_handling_MPI &
     &   ( ierr, 'subroutine force_mech_group_setup_MPI', 37 )

!-----------------------------------------------------------------------
!     /*   communications                                             */
!-----------------------------------------------------------------------

      call my_mpi_bcast_int_0 ( ngroup_mech )

      if ( myrank .ne. 0 ) then
         if ( .not. allocated( group_mech ) ) &
     &      allocate ( group_mech(ngroup_mech) )
      end if

      do i = 1, ngroup_mech
         call my_mpi_bcast_int_0 ( group_mech(i)%natom_each )
         call my_mpi_bcast_real_0 ( group_mech(i)%cfx )
         call my_mpi_bcast_real_0 ( group_mech(i)%cfy )
         call my_mpi_bcast_real_0 ( group_mech(i)%cfz )
         call my_mpi_bcast_int_0 ( group_mech(i)%flag_pot )
      enddo

      if ( myrank .ne. 0 ) then
         do i = 1, ngroup_mech
            if ( .not. allocated( group_mech(i)%id ) ) &
     &         allocate ( group_mech(i)%id(group_mech(i)%natom_each) )
         enddo
      end if

      do i = 1, ngroup_mech
         call my_mpi_bcast_int_1 ( group_mech(i)%id, &
     &                             group_mech(i)%natom_each )
      enddo

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then

         write( 6, '(a)' ) 'External force explicitly included.'
         write( 6, '(a)' ) 

      end if

!-----------------------------------------------------------------------
!     /*   set end                                                    */
!-----------------------------------------------------------------------

      iset = 1

      return
      end





!***********************************************************************
      subroutine force_mech_group_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, natom, nbead, mbox, myrank, nprocs

      use mech_variables, only : &
     &   pot_mech, fx_mech, fy_mech, fz_mech, vir_mech, &
     &   group_mech, ngroup_mech

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, ii, j, ibead
      integer :: m1, m2, m3

!     /*   real numbers   */
      real(8) :: rx, ry, rz, cfx, cfy, cfz, w

!     /*   mechanical potential   */
      do i = 1, nbead
         pot_mech(i) = 0.d0
      end do

!     /*   mechanical force   */
      do i = 1, nbead
      do j = 1, natom
         fx_mech(j,i) = 0.d0
         fy_mech(j,i) = 0.d0
         fz_mech(j,i) = 0.d0
      end do
      end do

!     /*   mechanical virial   */
      do i = 1, 3
      do j = 1, 3
         vir_mech(j,i) = 0.d0
      end do
      end do

!     /*   loop of beads   */
      do ibead = 1, nbead

!        /*   select bead   */
         if ( mod(ibead-1,nprocs) .ne. myrank ) cycle

         do i = 1, ngroup_mech

            do ii = 1, group_mech(i)%natom_each

               j = group_mech(i)%id(ii)

!              /*   mechanical force   */
               cfx = group_mech(i)%cfx
               cfy = group_mech(i)%cfy
               cfz = group_mech(i)%cfz

               fx_mech(j,ibead) = fx_mech(j,ibead) + cfx
               fy_mech(j,ibead) = fy_mech(j,ibead) + cfy
               fz_mech(j,ibead) = fz_mech(j,ibead) + cfz

               rx = x(j,ibead)
               ry = y(j,ibead)
               rz = z(j,ibead)

               m1 = mbox(1,j,ibead)
               m2 = mbox(2,j,ibead)
               m3 = mbox(3,j,ibead)

               call pbc_unfold_MPI(rx, ry, rz, m1, m2, m3)

!              /*   mechanical potential   */
               if ( group_mech(i)%flag_pot == 1 ) then 

                  w = -(rx*cfx + ry*cfy + rz*cfz)
                  pot_mech(ibead) = pot_mech(ibead) + w

               endif

            enddo

         enddo

!     /*   loop of beads   */
      end do

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot_mech, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx_mech, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy_mech, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz_mech, natom, nbead )

!     /*   virial   */
      call my_mpi_allreduce_real_2 ( vir_mech, 3, 3 )

      return
      end





!***********************************************************************
      subroutine force_mech_zone_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, myrank

      use mech_variables, only : &
     &   fc_mech, r_mech, req_mech, i_mech, j_mech, k_mech, l_mech, &
     &   nu_mech, mu_mech, nzone_mech, zone_mech, itype_zone_mech

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: k, ierr

!     /*   initial setting   */
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   external force                                             */
!-----------------------------------------------------------------------

      if ( iset .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   read number of forces                                      */
!-----------------------------------------------------------------------

!     /*   master rank   */
      if ( myrank .eq. 0 ) then

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   tag   */
         call search_tag ( '<zone>', 6, iounit, ierr )

!        /*   if no error  */
         if ( ierr .eq. 0 ) then

!           /*   number of types   */
            read( iounit, *, iostat=ierr ) nzone_mech

!           /*   file close   */
            close ( iounit )

!        /*   if error is found, read default value   */
         else

!           /*   file close   */
            close ( iounit )

!           /*   file open   */
            open ( iounit, file = 'input_default.dat' )

!           /*   tag   */
            call search_tag ( '<zone>', 6, iounit, ierr )

!           /*   number of types   */
            read( iounit, *, iostat=ierr ) nzone_mech

!           /*   file close   */
            close ( iounit )

!        /*   if error is found, read default value   */
         end if

!     /*   master rank   */
      end if

!     /*   communication   */
      call my_mpi_bcast_int_0 ( ierr )

!     /*   error handling   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_mech_zone_setup_MPI', 36 )

!     /*   communication   */
      call my_mpi_bcast_int_0 ( nzone_mech )

!     /*   no forces   */
      if ( nzone_mech .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   memory allocation   */
      if ( .not. allocated( itype_zone_mech ) ) &
     &   allocate (  itype_zone_mech(nzone_mech) )
      if ( .not. allocated(  i_mech ) ) &
     &   allocate (  i_mech(nzone_mech) )
      if ( .not. allocated(  j_mech ) ) &
     &   allocate (  j_mech(nzone_mech) )
      if ( .not. allocated(  k_mech ) ) &
     &   allocate (  k_mech(nzone_mech) )
      if ( .not. allocated(  l_mech ) ) &
     &   allocate (  l_mech(nzone_mech) )
      if ( .not. allocated(  nu_mech ) ) &
     &   allocate (  nu_mech(nzone_mech,2) )
      if ( .not. allocated(  mu_mech ) ) &
     &   allocate (  mu_mech(nzone_mech,2) )
      if ( .not. allocated( req_mech ) ) &
     &   allocate ( req_mech(nzone_mech,2) )
      if ( .not. allocated( fc_mech ) ) &
     &   allocate ( fc_mech(nzone_mech) )
      if ( .not. allocated(  r_mech ) ) &
     &   allocate (  r_mech(2*nzone_mech) )
      if ( .not. allocated(  zone_mech ) ) &
     &   allocate (  zone_mech(nzone_mech) )

!-----------------------------------------------------------------------
!     /*   read data                                                  */
!-----------------------------------------------------------------------

!     /*   master rank   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   tag   */
      call search_tag ( '<zone>', 6, iounit, ierr )

!     /*   number of types   */
      read( iounit, *, iostat=ierr )

!     /*   loop of mechanical forces   */
      do k = 1, nzone_mech

!        /*   read a line   */
         read( iounit, *, iostat=ierr ) zone_mech(k)

!        /*   back a line   */
         backspace( iounit )

!        /*   zone type: DIST    */
         if      ( ( zone_mech(k)(1:6)    .eq. '1     ' ) .or. &
     &             ( zone_mech(k)(1:6) .eq. 'DIST  ' ) ) then

!            /*   number   */
             itype_zone_mech(k) = 1

!            /*   read data   */
             read( iounit, *, iostat=ierr ) zone_mech(k), &
     &          i_mech(k), j_mech(k), &
     &          r_mech(2*k-1), r_mech(2*k), fc_mech(k)

!        /*   zone type: ANGL   */
         else if ( ( zone_mech(k)(1:6) .eq. '2     ' ) .or. &
     &             ( zone_mech(k)(1:6) .eq. 'ANGL  ' ) ) then

!            /*   number   */
             itype_zone_mech(k) = 2

!            /*   read data   */
             read( iounit, *, iostat=ierr ) zone_mech(k), &
     &          i_mech(k), j_mech(k), k_mech(k), &
     &          r_mech(2*k-1), r_mech(2*k), fc_mech(k)

!cc
!            /*   this code is unfinished   */
             ierr = 1
!cc

!        /*   zone type: DIH   */
         else if ( ( zone_mech(k)(1:6) .eq. '3     ' ) .or. &
     &             ( zone_mech(k)(1:6) .eq. 'DIH   ' ) ) then

!            /*   number   */
             itype_zone_mech(k) = 3

!            /*   read data   */
             read( iounit, *, iostat=ierr ) zone_mech(k), &
     &          i_mech(k), j_mech(k), k_mech(k), l_mech(k), &
     &          r_mech(2*k-1), r_mech(2*k), fc_mech(k)

!cc
!            /*   this code is unfinished   */
             ierr = 1
!cc

!        /*   zone type: DIFF   */
         else if ( ( zone_mech(k)(1:6) .eq. '4     ' ) .or. &
     &             ( zone_mech(k)(1:6) .eq. 'DIFF  ' ) ) then

!            /*   number   */
             itype_zone_mech(k) = 4

!            /*   read data   */
             read( iounit, *, iostat=ierr ) zone_mech(k), &
     &          i_mech(k), j_mech(k), k_mech(k), &
     &          r_mech(2*k-1), r_mech(2*k), fc_mech(k)

!cc
!            /*   this code is unfinished   */
             ierr = 1
!cc

!        /*   zone type: CN   */
          else if ( ( zone_mech(k)(1:6) .eq. '5     ' ) .or. &
     &              ( zone_mech(k)(1:6) .eq. 'CN    ' ) ) then

!            /*   number   */
             itype_zone_mech(k) = 5

!            /*   read data   */
             read( iounit, *, iostat=ierr ) zone_mech(k), &
     &          i_mech(k), j_mech(k), nu_mech(k,1), mu_mech(k,1), &
     &          req_mech(k,1), &
     &          r_mech(2*k-1), r_mech(2*k), fc_mech(k)

!        /*   zone type: DCN   */
          else if ( ( zone_mech(k)(1:6) .eq. '6     ' ) .or. &
     &              ( zone_mech(k)(1:6) .eq. 'DCN   ' ) ) then

!            /*   number   */
             itype_zone_mech(k) = 6

!            /*   read data   */
             read( iounit, *, iostat=ierr ) zone_mech(k), &
     &          i_mech(k), j_mech(k), nu_mech(k,1), mu_mech(k,1), &
     &          req_mech(k,1), &
     &          k_mech(k), l_mech(k), nu_mech(k,2), mu_mech(k,2), &
     &          req_mech(k,2), &
     &          r_mech(2*k-1), r_mech(2*k), fc_mech(k)

!cc
!            /*   this code is unfinished   */
             ierr = 1
!cc

!        /*   zone type: XYZ   */
         else if ( ( zone_mech(k)(1:6) .eq. '7     ' ) .or. &
     &             ( zone_mech(k)(1:6) .eq. 'XYZ   ' ) ) then

!            /*   number   */
             itype_zone_mech(k) = 7

!            /*   read data   */
             read( iounit, *, iostat=ierr ) zone_mech(k), &
     &          i_mech(k), j_mech(k), &
     &          r_mech(2*k-1), r_mech(2*k), fc_mech(k)

!cc
!            /*   this code is unfinished   */
!cc             ierr = 1
!cc

!        /*   zone type: DXYZ   */
          else if ( ( zone_mech(k)(1:6) .eq. '8     ' ) .or. &
     &              ( zone_mech(k)(1:6) .eq. 'DXYZ  ' ) ) then

!            /*   number   */
             itype_zone_mech(k) = 8

!            /*   read data   */
             read( iounit, *, iostat=ierr ) zone_mech(k), &
     &          i_mech(k), j_mech(k), k_mech(k), &
     &          r_mech(2*k-1), r_mech(2*k), fc_mech(k)

!cc
!            /*   this code is unfinished   */
             ierr = 1
!cc

!        /*   zone type: other   */
         else

!            /*   error flag   */
             ierr = 1

!            /*   exit */
             exit

!        /*   zone type   */
         end if

!     /*   loop of mechanical forces   */
      end do

!     /*   file close   */
      close ( iounit )

!     /*   master rank   */
      end if

!-----------------------------------------------------------------------
!     /*   communication                                              */
!-----------------------------------------------------------------------

      call my_mpi_bcast_int_1 ( i_mech, nzone_mech )
      call my_mpi_bcast_int_1 ( j_mech, nzone_mech )
      call my_mpi_bcast_int_1 ( k_mech, nzone_mech )
      call my_mpi_bcast_int_1 ( l_mech, nzone_mech )
      call my_mpi_bcast_int_2 ( nu_mech, nzone_mech, 2 )
      call my_mpi_bcast_int_2 ( mu_mech, nzone_mech, 2 )
      call my_mpi_bcast_real_1( fc_mech, nzone_mech )
      call my_mpi_bcast_real_2( req_mech, nzone_mech, 2 )
      call my_mpi_bcast_real_1( r_mech, 2*nzone_mech )
      call my_mpi_bcast_char_1( zone_mech, len(zone_mech), nzone_mech )
      call my_mpi_bcast_int_1 ( itype_zone_mech, nzone_mech )

!-----------------------------------------------------------------------
!     /*   error termination                                          */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) 'Error - ZONE input.'
         write( 6, '(a)' ) 
      end if
      end if

      call error_handling_MPI &
     &   ( ierr, 'subroutine force_mech_zone_setup_MPI', 36 )

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then
         write( 6, '(a)' ) 'Zone of confinement defined.'
         write( 6, '(a)' ) 
      end if

!-----------------------------------------------------------------------
!     /*   set end                                                    */
!-----------------------------------------------------------------------

      iset = 1

      return
      end





!***********************************************************************
      subroutine force_mech_zone_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   natom, nbead

      use mech_variables, only : &
     &   pot_mech, fx_mech, fy_mech, fz_mech, vir_mech, nzone_mech, &
     &   itype_zone_mech

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j, k

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   mechanical potential   */
      do i = 1, nbead
         pot_mech(i) = 0.d0
      end do

!     /*   mechanical force   */
      do i = 1, nbead
      do j = 1, natom
         fx_mech(j,i) = 0.d0
         fy_mech(j,i) = 0.d0
         fz_mech(j,i) = 0.d0
      end do
      end do

!     /*   mechanical virial   */
      do i = 1, 3
      do j = 1, 3
         vir_mech(j,i) = 0.d0
      end do
      end do

!-----------------------------------------------------------------------
!     /*   compute force                                              */
!-----------------------------------------------------------------------

!     /*   loop of zones   */
      do k = 1, nzone_mech

!        /*   mech type: DIST   */
         if ( itype_zone_mech(k) .eq. 1 ) call lin_mech_MPI(k)

!        /*   mech type: ANGL ... unfinished   */
!cc         if ( itype_zone_mech(k) .eq. 2 ) call angl_mech_MPI(k)

!        /*   mech type: DIH ... unfinished   */
!cc         if ( itype_zone_mech(k) .eq. 3 ) call dih_mech_MPI(k)

!        /*   mech type: DIFF ... unfinished   */
!cc         if ( itype_zone_mech(k) .eq. 4 ) call diff_mech_MPI(k)

!        /*   mech type: CN   */
         if ( itype_zone_mech(k) .eq. 5 ) call cord_mech_MPI(k)

!        /*   mech type: DCN ... unfinished   */
!cc         if ( itype_zone_mech(k) .eq. 6 ) call dcord_mech_MPI(k)

!        /*   mech type: XYZ ... unfinished   */
         if ( itype_zone_mech(k) .eq. 7 ) call xyz_mech_MPI(k)

!        /*   mech type: DXYZ ... unfinished   */
!cc         if ( itype_zone_mech(k) .eq. 8 ) call dxyz_mech_MPI(k)

!     /*   loop of zones   */
      end do

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot_mech, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx_mech, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy_mech, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz_mech, natom, nbead )

!     /*   virial   */
      call my_mpi_allreduce_real_2 ( vir_mech, 3, 3 )

      return
      end





!***********************************************************************
      subroutine lin_mech_MPI( k )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, nbead, myrank, nprocs

      use mech_variables, only : &
     &   pot_mech, fx_mech, fy_mech, fz_mech, fc_mech, vir_mech, &
     &   r_mech, i_mech, j_mech

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j, k, m

!     /*   real numbers   */
      real(8) :: xij, yij, zij, rij, rij2, fxi, fyi, fzi, dr

!-----------------------------------------------------------------------
!     /*   compute force                                              */
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do m = 1, nbead

!        /*   select bead   */
         if ( mod(m-1,nprocs) .ne. myrank ) cycle

!        /*   pair of atoms   */
         i = i_mech(k)
         j = j_mech(k)

!        /*   distance   */
         xij = x(i,m) - x(j,m)
         yij = y(i,m) - y(j,m)
         zij = z(i,m) - z(j,m)

!        /*   periodic boundary condition   */
         call pbc_atom_MPI ( xij, yij, zij )

!        /*   distance   */
         rij2 = xij*xij + yij*yij + zij*zij

!        /*   interatomic distance   */
         rij  = sqrt(rij2)

!        /*   deviation from the maximum of zone   */
         dr = max( rij-r_mech(2*k), 0.d0 )

!        /*   deviation from the minimum of zone   */
         dr = min( rij-r_mech(2*k-1), dr )

!        /*   mechanical potential   */
         pot_mech(m) = pot_mech(m) + 0.5d0*fc_mech(k)*dr*dr

!        /*   mechanical force   */
         fxi = - fc_mech(k) * dr / rij * xij
         fyi = - fc_mech(k) * dr / rij * yij
         fzi = - fc_mech(k) * dr / rij * zij

!        /*   i atom   */
         fx_mech(i,m) = fx_mech(i,m) + fxi
         fy_mech(i,m) = fy_mech(i,m) + fyi
         fz_mech(i,m) = fz_mech(i,m) + fzi

!        /*   j atom   */
         fx_mech(j,m) = fx_mech(j,m) - fxi
         fy_mech(j,m) = fy_mech(j,m) - fyi
         fz_mech(j,m) = fz_mech(j,m) - fzi

!        /*   mechanical virial   */
         vir_mech(1,1) = vir_mech(1,1) + fxi*xij
         vir_mech(1,2) = vir_mech(1,2) + fxi*yij
         vir_mech(1,3) = vir_mech(1,3) + fxi*zij
         vir_mech(2,1) = vir_mech(2,1) + fyi*xij
         vir_mech(2,2) = vir_mech(2,2) + fyi*yij
         vir_mech(2,3) = vir_mech(2,3) + fyi*zij
         vir_mech(3,1) = vir_mech(3,1) + fzi*xij
         vir_mech(3,2) = vir_mech(3,2) + fzi*yij
         vir_mech(3,3) = vir_mech(3,3) + fzi*zij

!     /*   loop of beads   */
      end do

      return
      end






!***********************************************************************
      subroutine cord_mech_MPI( m )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, natom, nbead, ikind, myrank, nprocs

      use mech_variables, only : &
     &   fx_mech, fy_mech, fz_mech, req_mech, pot_mech, fc_mech, &
     &   r_mech, vir_mech, nu_mech, mu_mech, i_mech, j_mech

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: nu, mu, i, j, m, n

      real(8) :: req, cn, dcn, xij, yij, zij, rij, fa, fb, dfa, dfb, &
     &           dcndr, dcndxi, dcndyi, dcndzi, fxi, fyi, fzi

!-----------------------------------------------------------------------
!     /*   parameters                                                 */
!-----------------------------------------------------------------------

      nu  = nu_mech(m,1)
      mu  = mu_mech(m,1)
      req = req_mech(m,1)

!-----------------------------------------------------------------------
!     /*   main loop start                                            */
!-----------------------------------------------------------------------

      do n = 1, nbead

!        /*   select bead   */
         if ( mod(n-1,nprocs) .ne. myrank ) cycle

!-----------------------------------------------------------------------
!        /*   calculate coordination number                           */
!-----------------------------------------------------------------------

!        /*   coordination number   */
         cn = 0.d0

!        /*   loop of atom pairs   */
         do i = 1, natom-1
         do j = i+1, natom

!           /*   specified pair of species   */
            if ( ( ( ikind(i) .eq. i_mech(m) ) .and. &
     &             ( ikind(j) .eq. j_mech(m) ) ) .or. &
     &           ( ( ikind(i) .eq. j_mech(m) ) .and. &
     &             ( ikind(j) .eq. i_mech(m) ) ) ) then

!              /*   interatomic distance   */
               xij = x(i,n) - x(j,n)
               yij = y(i,n) - y(j,n)
               zij = z(i,n) - z(j,n)

!              /*   apply periodic boundary   */
               call pbc_atom_MPI ( xij, yij, zij )

!              /*   interatomic distance   */
               rij = sqrt( xij*xij + yij*yij + zij*zij )

!              /*   coordination number   */
               cn = cn + ( 1.d0 - (rij/req)**nu ) &
     &                 / ( 1.d0 - (rij/req)**mu )

!           /*   specified pair of species   */
            end if

!        /*   loop of atom pairs   */
         end do
         end do

!-----------------------------------------------------------------------
!        /*   calculate potential and forces of harmonic term         */
!-----------------------------------------------------------------------

!        /*   deviation from the maximum of zone   */
         dcn = max( cn-r_mech(2*m), 0.d0 )

!        /*   deviation from the minimum of zone   */
         dcn = min( cn-r_mech(2*m-1), dcn )

!        /*   potential   */
         pot_mech(n) = pot_mech(n) + 0.5d0*fc_mech(m)*dcn*dcn

!        /*   loop of atom pairs   */
         do i = 1, natom-1
         do j = i+1, natom

!           /*   specified pair of species   */
            if ( ( ( ikind(i) .eq. i_mech(m) ) .and. &
     &             ( ikind(j) .eq. j_mech(m) ) ) .or. &
     &           ( ( ikind(i) .eq. j_mech(m) ) .and. &
     &             ( ikind(j) .eq. i_mech(m) ) ) ) then

!              /*   interatomic distance   */
               xij = x(i,n) - x(j,n)
               yij = y(i,n) - y(j,n)
               zij = z(i,n) - z(j,n)

!              /*   apply periodic boundary   */
               call pbc_atom_MPI ( xij, yij, zij )

!              /*   interatomic distance   */
               rij = sqrt( xij*xij + yij*yij + zij*zij )

!              /*   function   */
               fa  = 1.d0 - (rij/req)**nu
               fb  = 1.d0 - (rij/req)**mu

               dfa = - nu * (rij/req)**(nu-1) /req
               dfb = - mu * (rij/req)**(mu-1) /req

               dcndr  = ( dfa*fb - fa*dfb ) / ( fb*fb )

               dcndxi = dcndr * (xij/rij)
               dcndyi = dcndr * (yij/rij)
               dcndzi = dcndr * (zij/rij)

!              /*   force   */
               fxi = - fc_mech(m) * dcn * dcndxi
               fyi = - fc_mech(m) * dcn * dcndyi
               fzi = - fc_mech(m) * dcn * dcndzi

!              /*   i atom   */
               fx_mech(i,n) = fx_mech(i,n) + fxi
               fy_mech(i,n) = fy_mech(i,n) + fyi
               fz_mech(i,n) = fz_mech(i,n) + fzi

!              /*   j atom   */
               fx_mech(j,n) = fx_mech(j,n) - fxi
               fy_mech(j,n) = fy_mech(j,n) - fyi
               fz_mech(j,n) = fz_mech(j,n) - fzi

!              /*   mechanical virial   */
               vir_mech(1,1) = vir_mech(1,1) + fxi*xij
               vir_mech(1,2) = vir_mech(1,2) + fxi*yij
               vir_mech(1,3) = vir_mech(1,3) + fxi*zij
               vir_mech(2,1) = vir_mech(2,1) + fyi*xij
               vir_mech(2,2) = vir_mech(2,2) + fyi*yij
               vir_mech(2,3) = vir_mech(2,3) + fyi*zij
               vir_mech(3,1) = vir_mech(3,1) + fzi*xij
               vir_mech(3,2) = vir_mech(3,2) + fzi*yij
               vir_mech(3,3) = vir_mech(3,3) + fzi*zij

!           /*   specified pair of species   */
            end if

!        /*   loop of atom pairs   */
         end do
         end do

!-----------------------------------------------------------------------
!     /*   main loop end                                              */
!-----------------------------------------------------------------------

      end do

      return
      end





!***********************************************************************
      subroutine force_mech_esf_MPI
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, nbead, mbox, natom, nprocs, myrank

      use mech_variables, only : &
     &   q_mech, pot_mech, fx_mech, fy_mech, fz_mech, vir_mech, &
     &   ex_mech, ey_mech, ez_mech, q_mech

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: m, i, j, m1, m2, m3

      real(8) :: xb, yb, zb, qb, fxb, fyb, fzb

!-----------------------------------------------------------------------
!     /*   initialization                                             */
!-----------------------------------------------------------------------

!     /*   mechanical potential   */
      do i = 1, nbead
         pot_mech(i) = 0.d0
      end do

!     /*   mechanical force   */
      do i = 1, nbead
      do j = 1, natom
         fx_mech(j,i) = 0.d0
         fy_mech(j,i) = 0.d0
         fz_mech(j,i) = 0.d0
      end do
      end do

!     /*   mechanical virial   */
      do i = 1, 3
      do j = 1, 3
         vir_mech(j,i) = 0.d0
      end do
      end do

!-----------------------------------------------------------------------
!     /*   calculate mm dipole moment                                 */
!-----------------------------------------------------------------------

      do m = 1, nbead

!        /*   select bead   */
         if ( mod(m-1,nprocs) .ne. myrank ) cycle

         do i = 1, natom

            xb = x(i,m)
            yb = y(i,m)
            zb = z(i,m)

            m1 = mbox(1,i,m)
            m2 = mbox(2,i,m)
            m3 = mbox(3,i,m)

            call pbc_unfold_MPI( xb, yb, zb, m1, m2, m3 )

            qb = q_mech(i)

            fxb = qb * ex_mech
            fyb = qb * ey_mech
            fzb = qb * ez_mech

            fx_mech(i,m) = fxb
            fy_mech(i,m) = fyb
            fz_mech(i,m) = fzb

            pot_mech(m) = pot_mech(m) - fxb * xb - fyb * yb - fzb * zb

            vir_mech(1,1) = vir_mech(1,1) + fxb * xb
            vir_mech(1,2) = vir_mech(1,2) + fxb * yb
            vir_mech(1,3) = vir_mech(1,3) + fxb * zb
            vir_mech(2,1) = vir_mech(2,1) + fyb * xb
            vir_mech(2,2) = vir_mech(2,2) + fyb * yb
            vir_mech(2,3) = vir_mech(2,3) + fyb * zb
            vir_mech(3,1) = vir_mech(3,1) + fzb * xb
            vir_mech(3,2) = vir_mech(3,2) + fzb * yb
            vir_mech(3,3) = vir_mech(3,3) + fzb * zb

         end do

      end do

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot_mech, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx_mech, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy_mech, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz_mech, natom, nbead )

!     /*   virial   */
      call my_mpi_allreduce_real_2 ( vir_mech, 3, 3 )

      return
      end





!***********************************************************************
      subroutine force_mech_esf_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   iounit, natom, myrank


      use mech_variables, only : &
     &   q_mech, ex_mech, ey_mech, ez_mech

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: ierr, i, j

!     /*   initial setting   */
      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   external force                                             */
!-----------------------------------------------------------------------

      if ( iset .ne. 0 ) return

!-----------------------------------------------------------------------
!     /*   set up                                                     */
!-----------------------------------------------------------------------

!     /*   memory allocation   */
      if ( .not. allocated(  q_mech ) ) allocate (  q_mech(natom) )

!     /*   master rank   */
      if ( myrank .eq. 0 ) then

!     /*   file open   */
      open ( iounit, file = 'input.dat' )

!     /*   tag   */
      call search_tag ( '<esf>', 5, iounit, ierr )

!     /*   if no error  */
      if ( ierr .eq. 0 ) then

!        /*   electrostatic field   */
         read( iounit, *, iostat=ierr ) ex_mech, ey_mech, ez_mech

!        /*   electrostatic field   */
         do i = 1, natom
            read( iounit, *, iostat=ierr ) j, q_mech(j)
         end do

!     /*   if no error  */
      end if

!     /*   file close   */
      close( iounit )

!     /*   master rank   */
      end if

!-----------------------------------------------------------------------
!     /*   error termination                                          */
!-----------------------------------------------------------------------

!     /*   communication   */
      call my_mpi_bcast_int_0 ( ierr )

      if ( myrank .eq. 0 ) then
      if ( ierr .ne. 0 ) then
         write( 6, '(a)' ) 'Error - ESF input.'
         write( 6, '(a)' ) 
      end if
      end if

      call error_handling_MPI &
     &   ( ierr, 'subroutine force_mech_esf_setup_MPI', 35 )

!     /*   communication   */
      call my_mpi_bcast_real_0 ( ex_mech )
      call my_mpi_bcast_real_0 ( ey_mech )
      call my_mpi_bcast_real_0 ( ez_mech )
      call my_mpi_bcast_real_1 ( q_mech, natom )

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then
         write( 6, '(a)' ) 'External electrostatic field applied.'
         write( 6, '(a)' ) 
      end if

!-----------------------------------------------------------------------
!     /*   set end                                                    */
!-----------------------------------------------------------------------

      iset = 1

      return
      end





!***********************************************************************
      subroutine xyz_mech_MPI( k )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, natom, nbead, mbox, ikind, myrank, nprocs

      use mech_variables, only : &
     &   pot_mech, fx_mech, fy_mech, fz_mech, fc_mech, vir_mech, &
     &   r_mech, i_mech, j_mech

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, k, m, m1, m2, m3

!     /*   real numbers   */
      real(8) :: xi, yi, zi, dx, dy, dz, dr2, fxi, fyi, fzi
      real(8) :: xmax, xmin, ymax, ymin, zmax, zmin

!     /*   real numbers   */
      real(8) :: huge = 1.d+38

!-----------------------------------------------------------------------
!     /*   range                                                      */
!-----------------------------------------------------------------------

!     //   initial values
      xmin = - huge
      xmax = + huge
      ymin = - huge
      ymax = + huge
      zmin = - huge
      zmax = + huge

!     //   read values
      if ( i_mech(k) .eq. 1 ) xmin = r_mech(2*k-1)
      if ( i_mech(k) .eq. 1 ) xmax = r_mech(2*k)
      if ( i_mech(k) .eq. 2 ) ymin = r_mech(2*k-1)
      if ( i_mech(k) .eq. 2 ) ymax = r_mech(2*k)
      if ( i_mech(k) .eq. 3 ) zmin = r_mech(2*k-1)
      if ( i_mech(k) .eq. 3 ) zmax = r_mech(2*k)

!-----------------------------------------------------------------------
!     /*   compute force                                              */
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do m = 1, nbead

!        /*   select bead   */
         if ( mod(m-1,nprocs) .ne. myrank ) cycle

!        //   loop of atoms
         do i = 1, natom

!           //   for specific kind
            if ( ikind(i) .eq. j_mech(k) ) then

!              /*   coordinates   */
               xi = x(i,m)
               yi = y(i,m)
               zi = z(i,m)

!              /*   box   */
               m1 = mbox(1,i,m)
               m2 = mbox(2,i,m)
               m3 = mbox(3,i,m)

!              /*   unfold coordinates   */
               call pbc_unfold_MPI( xi, yi, zi, m1, m2, m3 )

!              /*   distance from wall   */
               dx = min( max( xi-xmax, 0.d0 ), xi-xmin )
               dy = min( max( yi-ymax, 0.d0 ), yi-ymin )
               dz = min( max( zi-zmax, 0.d0 ), zi-zmin )

!              //   distance squared
               dr2 = dx*dx + dy*dy + dz*dz

!              /*   mechanical potential   */
               pot_mech(m) = pot_mech(m) + 0.5d0*fc_mech(k)*dr2

!              /*   mechanical force   */
               fxi = - fc_mech(k) * dx
               fyi = - fc_mech(k) * dy
               fzi = - fc_mech(k) * dz

!              /*   i atom   */
               fx_mech(i,m) = fx_mech(i,m) + fxi
               fy_mech(i,m) = fy_mech(i,m) + fyi
               fz_mech(i,m) = fz_mech(i,m) + fzi

!              /*   mechanical virial   */
               vir_mech(1,1) = vir_mech(1,1) + fxi*dx
               vir_mech(1,2) = vir_mech(1,2) + fxi*dy
               vir_mech(1,3) = vir_mech(1,3) + fxi*dz
               vir_mech(2,1) = vir_mech(2,1) + fyi*dx
               vir_mech(2,2) = vir_mech(2,2) + fyi*dy
               vir_mech(2,3) = vir_mech(2,3) + fyi*dz
               vir_mech(3,1) = vir_mech(3,1) + fzi*dx
               vir_mech(3,2) = vir_mech(3,2) + fzi*dy
               vir_mech(3,3) = vir_mech(3,3) + fzi*dz

!           //   for specific kind
            end if

!        //   loop of atoms
         end do

!     /*   loop of beads   */
      end do

      return
      end

