!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, H. Kimizuka
!      Last updated:    Jan 23, 2025 by M. Shiga
!      Description:     energy and force from embedded atom method
!
!///////////////////////////////////////////////////////////////////////
!**********************************************************************
      subroutine force_eam_XMPI
!**********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, vir_bead, pot, au_length, au_charge, &
     &   volume, box, au_energy, vir_bead, natom, iounit, iboundary

      use mm_variables, only : &
     &   srho_eam, dfdrho_eam, rcut_eam, bigbox, bigboxinv, rcut_eam2, &
     &   neam, nbox_eam, ikind_eam, n_list, j_list

      use XMPI_variables, only : &
     &   jstart_bead, jend_bead, jstart_atom, jend_atom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: m, i, j, k, l, n, jx, jy, jz, j2

      real(8) :: xij, yij, zij, rij, rinv, srho, phir_eam, ax, ay, az, &
     &           rhor_eam, rhor_grad_eam, frho_eam, frho_grad_eam, &
     &           phir_grad_eam, bx, by, bz, cx, cy, cz, &
     &           drhoirdr, drhojrdr, dphirdr, dfdrhoi, dfdrhoj, &
     &           absa, absb, absc, aij, bij, cij, rij2, fxi, fyi, fzi

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   for initial access   */
      if ( iset .eq. 0 ) then

!        /*   read eam files   */
         call force_eam_setup_MPI

!        /*   set complete   */
         iset = 1

!     /*   for initial access   */
      end if

!     /*   return if no eam   */
      if ( neam .eq. 0 ) return

!     /*   cut off distance squared   */
      rcut_eam2 = rcut_eam*rcut_eam

!-----------------------------------------------------------------------
!     /*   extention of simulation box in real space sum              */
!-----------------------------------------------------------------------

!     /*   free boundary   */
      if ( iboundary .eq. 0 ) then

!        /*   number of replicated boxes   */
         nbox_eam(1) = 1
         nbox_eam(2) = 1
         nbox_eam(3) = 1

!     /*   periodic boundary   */
      else

!        /*   vector product of lattice vectors b, c   */
         ax = box(2,2)*box(3,3) - box(2,3)*box(3,2)
         ay = box(3,2)*box(1,3) - box(3,3)*box(1,2)
         az = box(1,2)*box(2,3) - box(1,3)*box(2,2)

!        /*   vector product of lattice vectors c, a   */
         bx = box(2,3)*box(3,1) - box(2,1)*box(3,3)
         by = box(3,3)*box(1,1) - box(3,1)*box(1,3)
         bz = box(1,3)*box(2,1) - box(1,1)*box(2,3)

!        /*   vector product of lattice vectors a, b   */
         cx = box(2,1)*box(3,2) - box(2,2)*box(3,1)
         cy = box(3,1)*box(1,2) - box(3,2)*box(1,1)
         cz = box(1,1)*box(2,2) - box(1,2)*box(2,1)

!        /*   distance between parallel planes   */
         absa = volume / sqrt( ax*ax + ay*ay + az*az )
         absb = volume / sqrt( bx*bx + by*by + bz*bz )
         absc = volume / sqrt( cx*cx + cy*cy + cz*cz )

!        /*   number of replicated boxes   */
         nbox_eam(1) = int(2.d0*rcut_eam/absa) + 1
         nbox_eam(2) = int(2.d0*rcut_eam/absb) + 1
         nbox_eam(3) = int(2.d0*rcut_eam/absc) + 1

!     /*   boundary condition   */
      end if

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

      do m = jstart_bead, jend_bead
         vir_bead(:,:,m) = 0.d0
      end do

!-----------------------------------------------------------------------
!     /*   first loop                                                 */
!-----------------------------------------------------------------------

!     /*   free boundary or minimum image convention   */
      if ( nbox_eam(1)*nbox_eam(2)*nbox_eam(3) .eq. 1 ) then

!        /*   make neighbor list   */
         call force_eam_makelist_XMPI

!        /*   loop of beads   */
         do m = jstart_bead, jend_bead

!           /*   srho_eam = sum of electron density rho_eam   */
            srho_eam(:) = 0.d0

!           /*   loop of atom pairs   */
            do i = jstart_atom, jend_atom

!           /*   loop of atom pairs   */
            do n = 1, n_list(i-jstart_atom+1,m-jstart_bead+1)

!              /*   atom in neighbor list   */
               j = j_list(n,i-jstart_atom+1,m-jstart_bead+1)

!              /*   interatomic distance   */
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              /*   apply free or periodic boundary   */
               call pbc_atom_MPI ( xij, yij, zij )

!              /*   interatomic distance squared   */
               rij2 = xij*xij + yij*yij + zij*zij

!              /*   neglect beyond cutoff distance   */
               if ( rij2 .gt. rcut_eam2 ) cycle

!              /*   interatomic distance   */
               rij = sqrt( rij2 )

!              /*   ikind_eam = species number   */
               k = ikind_eam(i)
               l = ikind_eam(j)

!              /*   sum of electron density   */
               srho_eam(i) = srho_eam(i) + rhor_eam(rij,l)
               srho_eam(j) = srho_eam(j) + rhor_eam(rij,k)

!           /*   loop of atom pairs   */
            end do

!           /*   loop of atom pairs   */
            end do

!           /*   communication   */
            call my_mpi_allreduce_real_1_sub( srho_eam, natom )

!-----------------------------------------------------------------------
!           /*   second loop                                          */
!-----------------------------------------------------------------------

!           /*   initialize   */
            dfdrho_eam(:) = 0.d0

!           /*   loop of atoms   */
            do i = jstart_atom, jend_atom

!              /*   sum of electron density   */
               srho = srho_eam(i)

!              /*   species number   */
               k    = ikind_eam(i)

!              /*   embedding potential   */
               pot(m)  =  pot(m) + frho_eam(srho,k)

!              /*   gradient of embedding potential   */
               dfdrho_eam(i) =  frho_grad_eam(srho,k)

!           /*   loop of atoms   */
            end do

!           /*   communication   */
            call my_mpi_allreduce_real_1_sub( dfdrho_eam, natom )

!-----------------------------------------------------------------------
!           /*   third loop                                           */
!-----------------------------------------------------------------------

!           /*   loop of atom pairs   */
            do i = jstart_atom, jend_atom

!           /*   loop of atom pairs   */
            do n = 1, n_list(i-jstart_atom+1,m-jstart_bead+1)

!              /*   atom in neighbor list   */
               j = j_list(n,i-jstart_atom+1,m-jstart_bead+1)

!              /*   interatomic distance   */
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              /*   apply free or periodic boundary   */
               call pbc_atom_MPI ( xij, yij, zij )

!              /*   interatomic distance squared   */
               rij2 = xij*xij + yij*yij + zij*zij

!              /*   neglect beyond cutoff distance   */
               if ( rij2 .gt. rcut_eam2 ) cycle

!              /*   interatomic distance   */
               rij = sqrt( rij2 )

!              /*   inverse of interatomic distance   */
               rinv = 1.d0/rij

!              /*   species number   */
               k = ikind_eam(i)
               l = ikind_eam(j)

!              /*   electron density   */
               drhoirdr =  rhor_grad_eam(rij,k)
               drhojrdr =  rhor_grad_eam(rij,l)

!              /*   gradient of pair potential   */
               dphirdr  =  phir_grad_eam(rij,k,l)

!              /*   gradient of embedding potential   */
               dfdrhoi  =  dfdrho_eam(i)
               dfdrhoj  =  dfdrho_eam(j)

!-----------------------------------------------------------------------
!              /*   pair potential   */
!-----------------------------------------------------------------------

               pot(m)   =  pot(m) + phir_eam(rij,k,l)

!-----------------------------------------------------------------------
!              /*   forces   */
!-----------------------------------------------------------------------

               fxi = - dfdrhoi * drhojrdr * xij * rinv &
     &               - dfdrhoj * drhoirdr * xij * rinv &
     &               - dphirdr * xij * rinv

               fyi = - dfdrhoi * drhojrdr * yij * rinv &
     &               - dfdrhoj * drhoirdr * yij * rinv &
     &               - dphirdr * yij * rinv

               fzi = - dfdrhoi * drhojrdr * zij * rinv &
     &               - dfdrhoj * drhoirdr * zij * rinv &
     &               - dphirdr * zij * rinv

!-----------------------------------------------------------------------
!              /*   total force and virial   */
!-----------------------------------------------------------------------

               fx(i,m) = fx(i,m) + fxi
               fy(i,m) = fy(i,m) + fyi
               fz(i,m) = fz(i,m) + fzi

               fx(j,m) = fx(j,m) - fxi
               fy(j,m) = fy(j,m) - fyi
               fz(j,m) = fz(j,m) - fzi

               vir_bead(1,1,m) = vir_bead(1,1,m) + fxi*xij
               vir_bead(1,2,m) = vir_bead(1,2,m) + fxi*yij
               vir_bead(1,3,m) = vir_bead(1,3,m) + fxi*zij
               vir_bead(2,1,m) = vir_bead(2,1,m) + fyi*xij
               vir_bead(2,2,m) = vir_bead(2,2,m) + fyi*yij
               vir_bead(2,3,m) = vir_bead(2,3,m) + fyi*zij
               vir_bead(3,1,m) = vir_bead(3,1,m) + fzi*xij
               vir_bead(3,2,m) = vir_bead(3,2,m) + fzi*yij
               vir_bead(3,3,m) = vir_bead(3,3,m) + fzi*zij

!           /*   loop of atom pairs   */
            end do

!           /*   loop of atom pairs   */
            end do

!        /*   loop of beads   */
         end do

!-----------------------------------------------------------------------
!     /*   first loop                                                 */
!-----------------------------------------------------------------------

!     /*   periodic boundary   */
      else

!        /*   replicated boxes   */
         bigbox(:,1) = dble(nbox_eam(1))*box(:,1)
         bigbox(:,2) = dble(nbox_eam(2))*box(:,2)
         bigbox(:,3) = dble(nbox_eam(3))*box(:,3)

!        /*   inverse of box matrix   */
         call inv3 ( bigbox, bigboxinv )

!        /*   loop of beads   */
         do m = jstart_bead, jend_bead

!           /*   srho_eam = sum of electron density rho_eam   */
            srho_eam(:) = 0.d0

!           /*   loop of atom pairs   */
            do i = jstart_atom, jend_atom

!           /*   loop of atom pairs   */
            do j = 1, natom

!           /*   loop of replicated boxes   */
            do jx = 0, nbox_eam(1)-1
            do jy = 0, nbox_eam(2)-1
            do jz = 0, nbox_eam(3)-1

!              /*   square of box index   */
               j2 = jx*jx + jy*jy + jz*jz

!              /*   skip same atom   */
               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

!              /*   interatomic distance of i and j in same box   */
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              /*   distance of i and j in different box  */
               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

!              /*   vector in big box   */
               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

!              /*   apply periodic boundary in big box   */
               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

!              /*   distance of nearest i and j   */
               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

!              /*   interatomic distance squared   */
               rij2 = xij*xij + yij*yij + zij*zij

!              /*   interatomic distance   */
               if ( rij2 .gt. rcut_eam2 ) cycle

!              /*   interatomic distance   */
               rij = sqrt( rij2 )

!              /*   ikind_eam = species number   */
               k = ikind_eam(i)
               l = ikind_eam(j)

!              /*   sum of electron density   */
               srho_eam(i) = srho_eam(i) + rhor_eam(rij,l)

!           /*   loop of replicated boxes   */
            end do
            end do
            end do

!           /*   loop of atom pairs   */
            end do
            end do

!           /*   communication   */
            call my_mpi_allreduce_real_1_sub( srho_eam, natom )

!-----------------------------------------------------------------------
!           /*   second loop                                          */
!-----------------------------------------------------------------------

!           /*   initialize   */
            dfdrho_eam(:) = 0.d0

!           /*   loop of atoms   */
            do i = jstart_atom, jend_atom

!              /*   sum of electron density   */
               srho = srho_eam(i)

!              /*   species number   */
               k    = ikind_eam(i)

!              /*   embedding potential   */
               pot(m)  =  pot(m) + frho_eam(srho,k)

!              /*   gradient of embedding potential   */
               dfdrho_eam(i) =  frho_grad_eam(srho,k)

!           /*   loop of atoms   */
            end do

!           /*   communication   */
            call my_mpi_allreduce_real_1_sub( dfdrho_eam, natom )

!-----------------------------------------------------------------------
!           /*   third loop                                           */
!-----------------------------------------------------------------------

!           /*   loop of atom pairs   */
            do i = jstart_atom, jend_atom

!           /*   loop of atom pairs   */
            do j = 1, natom

!           /*   loop of replicated boxes   */
            do jx = 0, nbox_eam(1)-1
            do jy = 0, nbox_eam(2)-1
            do jz = 0, nbox_eam(3)-1

!              /*   square of box index   */
               j2 = jx*jx + jy*jy + jz*jz

!              /*   skip same atom   */
               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

!              /*   interatomic distance of i and j in same box   */
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              /*   distance of i and j in different box  */
               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

!              /*   vector in big box   */
               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

!              /*   apply periodic boundary in big box   */
               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

!              /*   distance of nearest i and j   */
               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

!              /*   interatomic distance squared   */
               rij2 = xij*xij + yij*yij + zij*zij

!              /*   interatomic distance   */
               if ( rij2 .gt. rcut_eam2 ) cycle

!              /*   interatomic distance   */
               rij = sqrt( rij2 )

!              /*   interatomic distance   */
               rinv = 1.d0/rij

!              /*   species number   */
               k = ikind_eam(i)
               l = ikind_eam(j)

!              /*   electron density   */
               drhoirdr =  rhor_grad_eam(rij,k)
               drhojrdr =  rhor_grad_eam(rij,l)

!              /*   gradient of pair potential   */
               dphirdr  =  phir_grad_eam(rij,k,l)

!              /*   gradient of embedding potential   */
               dfdrhoi  =  dfdrho_eam(i)
               dfdrhoj  =  dfdrho_eam(j)

!-----------------------------------------------------------------------
!              /*   pair potential   */
!-----------------------------------------------------------------------

               pot(m)   =  pot(m) + 0.5d0*phir_eam(rij,k,l)

!-----------------------------------------------------------------------
!              /*   forces   */
!-----------------------------------------------------------------------

               fxi = - dfdrhoi * drhojrdr * xij * rinv &
     &               - dfdrhoj * drhoirdr * xij * rinv &
     &               - dphirdr * xij * rinv

               fyi = - dfdrhoi * drhojrdr * yij * rinv &
     &               - dfdrhoj * drhoirdr * yij * rinv &
     &               - dphirdr * yij * rinv

               fzi = - dfdrhoi * drhojrdr * zij * rinv &
     &               - dfdrhoj * drhoirdr * zij * rinv &
     &               - dphirdr * zij * rinv

!-----------------------------------------------------------------------
!              /*   total force and virial   */
!-----------------------------------------------------------------------

               fx(i,m) = fx(i,m) + fxi
               fy(i,m) = fy(i,m) + fyi
               fz(i,m) = fz(i,m) + fzi

               vir_bead(1,1,m) = vir_bead(1,1,m) + 0.5d0*fxi*xij
               vir_bead(1,2,m) = vir_bead(1,2,m) + 0.5d0*fxi*yij
               vir_bead(1,3,m) = vir_bead(1,3,m) + 0.5d0*fxi*zij
               vir_bead(2,1,m) = vir_bead(2,1,m) + 0.5d0*fyi*xij
               vir_bead(2,2,m) = vir_bead(2,2,m) + 0.5d0*fyi*yij
               vir_bead(2,3,m) = vir_bead(2,3,m) + 0.5d0*fyi*zij
               vir_bead(3,1,m) = vir_bead(3,1,m) + 0.5d0*fzi*xij
               vir_bead(3,2,m) = vir_bead(3,2,m) + 0.5d0*fzi*yij
               vir_bead(3,3,m) = vir_bead(3,3,m) + 0.5d0*fzi*zij

!           /*   loop of replicated boxes   */
            end do
            end do
            end do

!           /*   loop of atom pairs   */
            end do

!           /*   loop of atom pairs   */
            end do

!        /*   loop of beads   */
         end do

      end if

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

      do m = jstart_bead, jend_bead

!        /*   sum potential energy   */
         call my_mpi_allreduce_real_0_sub( pot(m) )

!        /*   sum virial   */
         call my_mpi_allreduce_real_2_sub( vir_bead(:,:,m), 3, 3 )

!        /*   sum forces for local atoms   */
         call my_mpi_allreduce_real_1_sub( fx(:,m), natom )
         call my_mpi_allreduce_real_1_sub( fy(:,m), natom )
         call my_mpi_allreduce_real_1_sub( fz(:,m), natom )

      end do

      return
      end





!***********************************************************************
      subroutine force_eam_makelist_XMPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, au_length, iounit, natom, myrank

      use mm_variables, only : &
     &   x_list, y_list, z_list, rcut_list, rcut2_list, skin_eam, &
     &   dmax_list, rcut_eam, n_list, j_list, nmax_list

      use XMPI_variables, only : &
     &   jstart_bead, jend_bead, jstart_atom, jend_atom

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize variables
      implicit none

!     //   visit flag
      integer, save :: iset = 0

!     //   integers
      integer :: i, j, k, l, ierr

!     //   real variables
      real(8) :: dx, dy, dz, d2, dmax, d2max, rx, ry, rz, r2

!     //   real variables
      real(8) :: bohr2ang = au_length * 1.d+10

!     //   real variables
      real(8) :: skin_eam_default = 2.d0

!     //   integers
      integer :: jbead, jatom

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

!     //   constants
      jbead = jend_bead - jstart_bead + 1
      jatom = jend_atom - jstart_atom + 1

!     //   only first visit of this routine
      if ( iset .eq. 0 ) then

!        //   master rank only
         if ( myrank .eq. 0 ) then

!        //   open file
         open ( iounit, file = 'eam.dat' )

!        /*   tag   */
         call search_tag ( '<skin_eam>', 10, iounit, ierr )

!        /*   cut off distance   */
         read( iounit, *, iostat=ierr ) skin_eam

!        //   close file
         close( iounit )

!        //   master rank only
         end if

!        //   communicate
         call my_mpi_bcast_int_0( ierr )

!        //   default value
         if ( ierr .ne. 0 ) skin_eam = skin_eam_default

!        //   communicate
         call my_mpi_bcast_real_0( skin_eam )

!        //   angstrom to bohr
         skin_eam = skin_eam / bohr2ang

!        //   list cutoff radius
         rcut_list = rcut_eam + skin_eam

!        //   list cutoff radius squared
         rcut2_list = rcut_list * rcut_list

!        //   maximum deviation allowed without updating neighbor list
         dmax_list = 0.5d0 * skin_eam

!        //   memory allocation
         if ( .not. allocated(x_list) ) allocate( x_list(jatom,jbead) )
         if ( .not. allocated(y_list) ) allocate( y_list(jatom,jbead) )
         if ( .not. allocated(z_list) ) allocate( z_list(jatom,jbead) )

!        //   setup end
         iset = 1

!-----------------------------------------------------------------------
!     /*   deviation from last update of neighbor list                */
!-----------------------------------------------------------------------

!     //   from second visit to this routine
      else

!        //   initialize maximum deviation squared
         d2max = 0.d0

!        //   loop of beads and atoms
         do k = jstart_bead, jend_bead
         do i = jstart_atom, jend_atom

!           //   deviation
            dx = x(i,k) - x_list(i-jstart_atom+1,k-jstart_bead+1)
            dy = y(i,k) - y_list(i-jstart_atom+1,k-jstart_bead+1)
            dz = z(i,k) - z_list(i-jstart_atom+1,k-jstart_bead+1)

!           //   apply boundary condition
            call pbc_atom_MPI( dx, dy, dz )

!           //   deviation squared
            d2 = dx*dx + dy*dy + dz*dz

!           //   maximum deviation squared
            d2max = max( d2, d2max )

!        //   loop of beads and atoms
         end do
         end do

!        //   maximum deviation
         dmax = sqrt( d2max )

!        //   communication
         call my_mpi_allreduce_max_0( d2max )

!        //   if maximum deviation is small, skip neighbor list update
         if ( dmax .lt. dmax_list ) return

!     //   end of if statement
      end if

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( allocated( n_list ) ) deallocate( n_list )
      if ( .not. allocated(n_list) ) allocate( n_list(jatom,jbead) )

!-----------------------------------------------------------------------
!     /*   number of atoms in neighbor list                           */
!-----------------------------------------------------------------------

!     //   clear
      n_list(:,:) = 0

!     //   loop of beads
      do k = jstart_bead, jend_bead

!        //   loop of atom i
         do i = jstart_atom, jend_atom

!           //   counter
            l = 0

!           //  loop of atom j
            do j = 1, natom

!              //   same atom
               if ( i .eq. j ) cycle

!              //   separation of atoms i and j
               rx = x(j,k) - x(i,k)
               ry = y(j,k) - y(i,k)
               rz = z(j,k) - z(i,k)

!              //   apply boundary condition
               call pbc_atom_MPI( rx, ry, rz )

!              //   distance of atoms i and j squared
               r2 = rx*rx + ry*ry + rz*rz

!              //   if distance is smaller than list cutoff radius
               if ( r2 .lt. rcut2_list ) then

!                 //   update counter
                  l = l + 1

!              //   end of if statement
               end if

!           //   loop of atom j
            end do

!           //   number of j atoms in neighbor list for atom i, bead k
            n_list(i-jstart_atom+1,k-jstart_bead+1) = l

!        //   loop of atom i
         end do

!     //   loop of beads
      end do

!-----------------------------------------------------------------------
!     /*   maximum number of atoms                                    */
!-----------------------------------------------------------------------

!     //   counter
      nmax_list = 0

!     //   loop of beads
      do k = jstart_bead, jend_bead

!        //   loop of atom i
         do i = jstart_atom, jend_atom

!           //   maximum number of j atoms in neighbor list
            nmax_list = &
     &         max( n_list(i-jstart_atom+1,k-jstart_bead+1), nmax_list )

!        //   loop of atom i
         end do

!     //   loop of beads
      end do

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( allocated( j_list ) ) deallocate( j_list )
      allocate( j_list(nmax_list,jatom,jbead) )

!-----------------------------------------------------------------------
!     /*   atoms in neighbor list                                     */
!-----------------------------------------------------------------------

!     //   clear
      j_list(:,:,:) = 0

!     //   loop of beads
      do k = jstart_bead, jend_bead

!        //   loop of atom i
         do i = jstart_atom, jend_atom

!           //   counter
            l = 0

!           //  loop of atom j
            do j = 1, natom

!              //   same atom
               if ( i .eq. j ) cycle

!              //   separation of atoms i and j
               rx = x(j,k) - x(i,k)
               ry = y(j,k) - y(i,k)
               rz = z(j,k) - z(i,k)

!              //   apply boundary condition
               call pbc_atom_MPI( rx, ry, rz )

!              //   distance of atoms i and j squared
               r2 = rx*rx + ry*ry + rz*rz

!              //   if distance is smaller than list cutoff radius
               if ( r2 .lt. rcut2_list ) then

!                 //   update counter
                  l = l + 1

!                 //   list of j atoms for atom i, bead k
                  j_list(l,i-jstart_atom+1,k-jstart_bead+1) = j

               end if

!           //  loop of atom j
            end do

!        //   loop of atom i
         end do

!     //   loop of beads
      end do

!-----------------------------------------------------------------------
!     /*   update neighbor list                                       */
!-----------------------------------------------------------------------

      do k = jstart_bead, jend_bead
      do i = jstart_atom, jend_atom
         x_list(i-jstart_atom+1,k-jstart_bead+1) = x(i,k)
         y_list(i-jstart_atom+1,k-jstart_bead+1) = y(i,k)
         z_list(i-jstart_atom+1,k-jstart_bead+1) = z(i,k)
      end do
      end do

      return
      end
