!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, H. Kimizuka
!      Last updated:    May 21, 2020 by M. Shiga
!      Description:     energy and force from embedded atom method
!
!///////////////////////////////////////////////////////////////////////
!**********************************************************************
      subroutine force_eam_MPI
!**********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, vir, pot, au_length, au_charge, volume, &
     &   box, au_energy, natom, nbead, iounit, iboundary, &
     &   myrank_main, nprocs_main, myrank_sub, nprocs_sub

      use mm_variables, only : &
     &   srho_eam, dfdrho_eam, rcut_eam, bigbox, bigboxinv, rcut_eam2, &
     &   neam, nbox_eam, ikind_eam, n_list, j_list

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: m, i, j, k, l, n, jx, jy, jz, j2

      real(8) :: xij, yij, zij, rij, rinv, srho, phir_eam, ax, ay, az, &
     &           rhor_eam, rhor_grad_eam, frho_eam, frho_grad_eam, &
     &           phir_grad_eam, bx, by, bz, cx, cy, cz, &
     &           drhoirdr, drhojrdr, dphirdr, dfdrhoi, dfdrhoj, &
     &           absa, absb, absc, aij, bij, cij, rij2, fxi, fyi, fzi

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   for initial access   */
      if ( iset .eq. 0 ) then

!        /*   read eam files   */
         call force_eam_setup_MPI

!        /*   set complete   */
         iset = 1

!     /*   for initial access   */
      end if

!     /*   return if no eam   */
      if ( neam .eq. 0 ) return

!     /*   cut off distance squared   */
      rcut_eam2 = rcut_eam*rcut_eam

!-----------------------------------------------------------------------
!     /*   extention of simulation box in real space sum              */
!-----------------------------------------------------------------------

!     /*   free boundary   */
      if ( iboundary .eq. 0 ) then

!        /*   number of replicated boxes   */
         nbox_eam(1) = 1
         nbox_eam(2) = 1
         nbox_eam(3) = 1

!     /*   periodic boundary   */
      else

!        /*   vector product of lattice vectors b, c   */
         ax = box(2,2)*box(3,3) - box(2,3)*box(3,2)
         ay = box(3,2)*box(1,3) - box(3,3)*box(1,2)
         az = box(1,2)*box(2,3) - box(1,3)*box(2,2)

!        /*   vector product of lattice vectors c, a   */
         bx = box(2,3)*box(3,1) - box(2,1)*box(3,3)
         by = box(3,3)*box(1,1) - box(3,1)*box(1,3)
         bz = box(1,3)*box(2,1) - box(1,1)*box(2,3)

!        /*   vector product of lattice vectors a, b   */
         cx = box(2,1)*box(3,2) - box(2,2)*box(3,1)
         cy = box(3,1)*box(1,2) - box(3,2)*box(1,1)
         cz = box(1,1)*box(2,2) - box(1,2)*box(2,1)

!        /*   distance between parallel planes   */
         absa = volume / sqrt( ax*ax + ay*ay + az*az )
         absb = volume / sqrt( bx*bx + by*by + bz*bz )
         absc = volume / sqrt( cx*cx + cy*cy + cz*cz )

!        /*   number of replicated boxes   */
         nbox_eam(1) = int(2.d0*rcut_eam/absa) + 1
         nbox_eam(2) = int(2.d0*rcut_eam/absb) + 1
         nbox_eam(3) = int(2.d0*rcut_eam/absc) + 1

!     /*   boundary condition   */
      end if

!-----------------------------------------------------------------------
!     /*   first loop                                                 */
!-----------------------------------------------------------------------

!     /*   free boundary or minimum image convention   */
      if ( nbox_eam(1)*nbox_eam(2)*nbox_eam(3) .eq. 1 ) then

!        /*   make neighbor list   */
         call force_eam_makelist_MPI

!        /*   loop of beads   */
         do m = 1, nbead

!           /*   bead parallel   */
            if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

!           /*   srho_eam = sum of electron density rho_eam   */
            srho_eam(:) = 0.d0

!           /*   loop of atom pairs   */
            do i = 1, natom

!           /*   force parallel   */
            if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!           /*   loop of atom pairs   */
            do n = 1, n_list(i,m)

!              /*   atom in neighbor list   */
               j = j_list(n,i,m)

!              /*   interatomic distance   */
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              /*   apply free or periodic boundary   */
               call pbc_atom_MPI ( xij, yij, zij )

!              /*   interatomic distance squared   */
               rij2 = xij*xij + yij*yij + zij*zij

!              /*   neglect beyond cutoff distance   */
               if ( rij2 .gt. rcut_eam2 ) cycle

!              /*   interatomic distance   */
               rij = sqrt( rij2 )

!              /*   ikind_eam = species number   */
               k = ikind_eam(i)
               l = ikind_eam(j)

!              /*   sum of electron density   */
               srho_eam(i) = srho_eam(i) + rhor_eam(rij,l)
               srho_eam(j) = srho_eam(j) + rhor_eam(rij,k)

!           /*   loop of atom pairs   */
            end do
            end do

!           /*   communication   */
            call my_mpi_allreduce_real_1_sub( srho_eam, natom )

!-----------------------------------------------------------------------
!           /*   second loop                                          */
!-----------------------------------------------------------------------

!           /*   loop of atoms   */
            do i = 1, natom

!              /*   initialize   */
               dfdrho_eam(i) = 0.d0

!              /*   force parallel   */
               if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!              /*   sum of electron density   */
               srho = srho_eam(i)

!              /*   species number   */
               k    = ikind_eam(i)

!              /*   embedding potential   */
               pot(m)  =  pot(m) + frho_eam(srho,k)

!              /*   gradient of embedding potential   */
               dfdrho_eam(i) =  frho_grad_eam(srho,k)

!           /*   loop of atoms   */
            end do

!           /*   communication   */
            call my_mpi_allreduce_real_1_sub( dfdrho_eam, natom )

!-----------------------------------------------------------------------
!           /*   third loop                                           */
!-----------------------------------------------------------------------

!           /*   loop of atom pairs   */
            do i = 1, natom

!           /*   force parallel   */
            if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!           /*   loop of atom pairs   */
            do n = 1, n_list(i,m)

!              /*   atom in neighbor list   */
               j = j_list(n,i,m)

!              /*   interatomic distance   */
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              /*   apply free or periodic boundary   */
               call pbc_atom_MPI ( xij, yij, zij )

!              /*   interatomic distance squared   */
               rij2 = xij*xij + yij*yij + zij*zij

!              /*   neglect beyond cutoff distance   */
               if ( rij2 .gt. rcut_eam2 ) cycle

!              /*   interatomic distance   */
               rij = sqrt( rij2 )

!              /*   inverse of interatomic distance   */
               rinv = 1.d0/rij

!              /*   species number   */
               k = ikind_eam(i)
               l = ikind_eam(j)

!              /*   electron density   */
               drhoirdr =  rhor_grad_eam(rij,k)
               drhojrdr =  rhor_grad_eam(rij,l)

!              /*   gradient of pair potential   */
               dphirdr  =  phir_grad_eam(rij,k,l)

!              /*   gradient of embedding potential   */
               dfdrhoi  =  dfdrho_eam(i)
               dfdrhoj  =  dfdrho_eam(j)

!-----------------------------------------------------------------------
!              /*   pair potential   */
!-----------------------------------------------------------------------

               pot(m)   =  pot(m) + phir_eam(rij,k,l)

!-----------------------------------------------------------------------
!              /*   forces   */
!-----------------------------------------------------------------------

               fxi = - dfdrhoi * drhojrdr * xij * rinv &
     &               - dfdrhoj * drhoirdr * xij * rinv &
     &               - dphirdr * xij * rinv

               fyi = - dfdrhoi * drhojrdr * yij * rinv &
     &               - dfdrhoj * drhoirdr * yij * rinv &
     &               - dphirdr * yij * rinv

               fzi = - dfdrhoi * drhojrdr * zij * rinv &
     &               - dfdrhoj * drhoirdr * zij * rinv &
     &               - dphirdr * zij * rinv

!-----------------------------------------------------------------------
!              /*   total force and virial   */
!-----------------------------------------------------------------------

               fx(i,m) = fx(i,m) + fxi
               fy(i,m) = fy(i,m) + fyi
               fz(i,m) = fz(i,m) + fzi

               fx(j,m) = fx(j,m) - fxi
               fy(j,m) = fy(j,m) - fyi
               fz(j,m) = fz(j,m) - fzi

               vir(1,1) = vir(1,1) + fxi*xij
               vir(1,2) = vir(1,2) + fxi*yij
               vir(1,3) = vir(1,3) + fxi*zij
               vir(2,1) = vir(2,1) + fyi*xij
               vir(2,2) = vir(2,2) + fyi*yij
               vir(2,3) = vir(2,3) + fyi*zij
               vir(3,1) = vir(3,1) + fzi*xij
               vir(3,2) = vir(3,2) + fzi*yij
               vir(3,3) = vir(3,3) + fzi*zij

!           /*   loop of atom pairs   */
            end do

!           /*   loop of atom pairs   */
            end do

!        /*   loop of beads   */
         end do

!-----------------------------------------------------------------------
!     /*   first loop                                                 */
!-----------------------------------------------------------------------

!     /*   periodic boundary   */
      else

!        /*   replicated boxes   */
         bigbox(:,1) = dble(nbox_eam(1))*box(:,1)
         bigbox(:,2) = dble(nbox_eam(2))*box(:,2)
         bigbox(:,3) = dble(nbox_eam(3))*box(:,3)

!        /*   inverse of box matrix   */
         call inv3 ( bigbox, bigboxinv )

!        /*   loop of beads   */
         do m = 1, nbead

!           /*   bead parallel   */
            if ( mod( m-1, nprocs_main ) .ne. myrank_main ) cycle

!           /*   srho_eam = sum of electron density rho_eam   */
            srho_eam(:) = 0.d0

!           /*   loop of atom pairs   */
            do i = 1, natom

!           /*   force parallel   */
            if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!           /*   loop of atom pairs   */
            do j = 1, natom

!           /*   loop of replicated boxes   */
            do jx = 0, nbox_eam(1)-1
            do jy = 0, nbox_eam(2)-1
            do jz = 0, nbox_eam(3)-1

!              /*   square of box index   */
               j2 = jx*jx + jy*jy + jz*jz

!              /*   skip same atom   */
               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

!              /*   interatomic distance of i and j in same box   */
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              /*   distance of i and j in different box  */
               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

!              /*   vector in big box   */
               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

!              /*   apply periodic boundary in big box   */
               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

!              /*   distance of nearest i and j   */
               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

!              /*   interatomic distance squared   */
               rij2 = xij*xij + yij*yij + zij*zij

!              /*   interatomic distance   */
               if ( rij2 .gt. rcut_eam2 ) cycle

!              /*   interatomic distance   */
               rij = sqrt( rij2 )

!              /*   ikind_eam = species number   */
               k = ikind_eam(i)
               l = ikind_eam(j)

!              /*   sum of electron density   */
               srho_eam(i) = srho_eam(i) + rhor_eam(rij,l)

!           /*   loop of replicated boxes   */
            end do
            end do
            end do

!           /*   loop of atom pairs   */
            end do
            end do

!           /*   communication   */
            call my_mpi_allreduce_real_1_sub( srho_eam, natom )

!-----------------------------------------------------------------------
!           /*   second loop                                          */
!-----------------------------------------------------------------------

!           /*   loop of atoms   */
            do i = 1, natom

!              /*   initialize   */
               dfdrho_eam(i) = 0.d0

!              /*   force parallel   */
               if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!              /*   sum of electron density   */
               srho = srho_eam(i)

!              /*   species number   */
               k    = ikind_eam(i)

!              /*   embedding potential   */
               pot(m)  =  pot(m) + frho_eam(srho,k)

!              /*   gradient of embedding potential   */
               dfdrho_eam(i) =  frho_grad_eam(srho,k)

!           /*   loop of atoms   */
            end do

!           /*   communication   */
            call my_mpi_allreduce_real_1_sub( dfdrho_eam, natom )

!-----------------------------------------------------------------------
!           /*   third loop                                           */
!-----------------------------------------------------------------------

!           /*   loop of atom pairs   */
            do i = 1, natom

!           /*   force parallel   */
            if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!           /*   loop of atom pairs   */
            do j = 1, natom

!           /*   loop of replicated boxes   */
            do jx = 0, nbox_eam(1)-1
            do jy = 0, nbox_eam(2)-1
            do jz = 0, nbox_eam(3)-1

!              /*   square of box index   */
               j2 = jx*jx + jy*jy + jz*jz

!              /*   skip same atom   */
               if ( ( j2 .eq. 0 ) .and. ( i .eq. j ) ) cycle

!              /*   interatomic distance of i and j in same box   */
               xij = x(i,m) - x(j,m)
               yij = y(i,m) - y(j,m)
               zij = z(i,m) - z(j,m)

!              /*   distance of i and j in different box  */
               xij = xij - box(1,1)*jx - box(1,2)*jy - box(1,3)*jz
               yij = yij - box(2,1)*jx - box(2,2)*jy - box(2,3)*jz
               zij = zij - box(3,1)*jx - box(3,2)*jy - box(3,3)*jz

!              /*   vector in big box   */
               aij = bigboxinv(1,1)*xij + bigboxinv(1,2)*yij &
     &             + bigboxinv(1,3)*zij
               bij = bigboxinv(2,1)*xij + bigboxinv(2,2)*yij &
     &             + bigboxinv(2,3)*zij
               cij = bigboxinv(3,1)*xij + bigboxinv(3,2)*yij &
     &             + bigboxinv(3,3)*zij

!              /*   apply periodic boundary in big box   */
               aij = aij - nint(aij)
               bij = bij - nint(bij)
               cij = cij - nint(cij)

!              /*   distance of nearest i and j   */
               xij = bigbox(1,1)*aij + bigbox(1,2)*bij + bigbox(1,3)*cij
               yij = bigbox(2,1)*aij + bigbox(2,2)*bij + bigbox(2,3)*cij
               zij = bigbox(3,1)*aij + bigbox(3,2)*bij + bigbox(3,3)*cij

!              /*   interatomic distance squared   */
               rij2 = xij*xij + yij*yij + zij*zij

!              /*   interatomic distance   */
               if ( rij2 .gt. rcut_eam2 ) cycle

!              /*   interatomic distance   */
               rij = sqrt( rij2 )

!              /*   interatomic distance   */
               rinv = 1.d0/rij

!              /*   species number   */
               k = ikind_eam(i)
               l = ikind_eam(j)

!              /*   electron density   */
               drhoirdr =  rhor_grad_eam(rij,k)
               drhojrdr =  rhor_grad_eam(rij,l)

!              /*   gradient of pair potential   */
               dphirdr  =  phir_grad_eam(rij,k,l)

!              /*   gradient of embedding potential   */
               dfdrhoi  =  dfdrho_eam(i)
               dfdrhoj  =  dfdrho_eam(j)

!-----------------------------------------------------------------------
!              /*   pair potential   */
!-----------------------------------------------------------------------

               pot(m)   =  pot(m) + 0.5d0*phir_eam(rij,k,l)

!-----------------------------------------------------------------------
!              /*   forces   */
!-----------------------------------------------------------------------

               fxi = - dfdrhoi * drhojrdr * xij * rinv &
     &               - dfdrhoj * drhoirdr * xij * rinv &
     &               - dphirdr * xij * rinv

               fyi = - dfdrhoi * drhojrdr * yij * rinv &
     &               - dfdrhoj * drhoirdr * yij * rinv &
     &               - dphirdr * yij * rinv

               fzi = - dfdrhoi * drhojrdr * zij * rinv &
     &               - dfdrhoj * drhoirdr * zij * rinv &
     &               - dphirdr * zij * rinv

!-----------------------------------------------------------------------
!              /*   total force and virial   */
!-----------------------------------------------------------------------

               fx(i,m) = fx(i,m) + fxi
               fy(i,m) = fy(i,m) + fyi
               fz(i,m) = fz(i,m) + fzi

               vir(1,1) = vir(1,1) + 0.5d0*fxi*xij
               vir(1,2) = vir(1,2) + 0.5d0*fxi*yij
               vir(1,3) = vir(1,3) + 0.5d0*fxi*zij
               vir(2,1) = vir(2,1) + 0.5d0*fyi*xij
               vir(2,2) = vir(2,2) + 0.5d0*fyi*yij
               vir(2,3) = vir(2,3) + 0.5d0*fyi*zij
               vir(3,1) = vir(3,1) + 0.5d0*fzi*xij
               vir(3,2) = vir(3,2) + 0.5d0*fzi*yij
               vir(3,3) = vir(3,3) + 0.5d0*fzi*zij

!           /*   loop of replicated boxes   */
            end do
            end do
            end do

!           /*   loop of atom pairs   */
            end do

!           /*   loop of atom pairs   */
            end do

!        /*   loop of beads   */
         end do

      end if

!-----------------------------------------------------------------------
!     /*   all-reduce communication                                   */
!-----------------------------------------------------------------------

!     /*   potential   */
      call my_mpi_allreduce_real_1 ( pot, nbead )

!     /*   force   */
      call my_mpi_allreduce_real_2 ( fx, natom, nbead )
      call my_mpi_allreduce_real_2 ( fy, natom, nbead )
      call my_mpi_allreduce_real_2 ( fz, natom, nbead )

!     /*   virial   */
      call my_mpi_allreduce_real_2 ( vir, 3, 3 )

      return
      end





!***********************************************************************
      subroutine force_eam_makelist_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, au_length, natom, nbead, iounit, nprocs_main, &
     &   myrank_main, nprocs_sub, myrank_sub, myrank

      use mm_variables, only : &
     &   x_list, y_list, z_list, rcut_list, rcut2_list, skin_eam, &
     &   dmax_list, rcut_eam, n_list, j_list, nmax_list

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     //   initialize variables
      implicit none

!     //   visit flag
      integer, save :: iset = 0

!     //   integers
      integer :: i, j, k, l, ierr

!     //   real variables
      real(8) :: dx, dy, dz, d2, dmax, d2max, rx, ry, rz, r2

!     //   real variables
      real(8) :: bohr2ang = au_length * 1.d+10

!     //   real variables
      real(8) :: skin_eam_default = 2.d0

!-----------------------------------------------------------------------
!     /*   initial setup                                              */
!-----------------------------------------------------------------------

!     //   only first visit of this routine
      if ( iset .eq. 0 ) then

!        //   master rank only
         if ( myrank .eq. 0 ) then

!        //   open file
         open ( iounit, file = 'eam.dat' )

!        /*   tag   */
         call search_tag ( '<skin_eam>', 10, iounit, ierr )

!        /*   cut off distance   */
         read( iounit, *, iostat=ierr ) skin_eam

!        //   close file
         close( iounit )

!        //   master rank only
         end if

!        //   communicate
         call my_mpi_bcast_int_0( ierr )

!        //   default value
         if ( ierr .ne. 0 ) skin_eam = skin_eam_default

!        //   communicate
         call my_mpi_bcast_real_0( skin_eam )

!        //   angstrom to bohr
         skin_eam = skin_eam / bohr2ang

!        //   list cutoff radius
         rcut_list = rcut_eam + skin_eam

!        //   list cutoff radius squared
         rcut2_list = rcut_list * rcut_list

!        //   maximum deviation allowed without updating neighbor list
         dmax_list = 0.5d0 * skin_eam

!        //   memory allocation
         if ( .not. allocated(x_list) ) allocate( x_list(natom,nbead) )
         if ( .not. allocated(y_list) ) allocate( y_list(natom,nbead) )
         if ( .not. allocated(z_list) ) allocate( z_list(natom,nbead) )

!        //   setup end
         iset = 1

!-----------------------------------------------------------------------
!     /*   deviation from last update of neighbor list                */
!-----------------------------------------------------------------------

!     //   from second visit to this routine
      else

!        //   initialize maximum deviation squared
         d2max = 0.d0

!        //   loop of beads and atoms
         do k = 1, nbead
         do i = 1, natom

!           //   deviation
            dx = x(i,k) - x_list(i,k)
            dy = y(i,k) - y_list(i,k)
            dz = z(i,k) - z_list(i,k)

!           //   apply boundary condition
            call pbc_atom_MPI( dx, dy, dz )

!           //   deviation squared
            d2 = dx*dx + dy*dy + dz*dz

!           //   maximum deviation squared
            d2max = max( d2, d2max )

!        //   loop of beads and atoms
         end do
         end do

!        //   maximum deviation
         dmax = sqrt( d2max )

!        //   if maximum deviation is small, skip neighbor list update
         if ( dmax .lt. dmax_list ) return

!     //   end of if statement
      end if

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( allocated( n_list ) ) deallocate( n_list )
      if ( .not. allocated(n_list) ) allocate( n_list(natom,nbead) )

!-----------------------------------------------------------------------
!     /*   number of atoms in neighbor list                           */
!-----------------------------------------------------------------------

!     //   clear
      n_list(:,:) = 0

!     //   loop of beads
      do k = 1, nbead

!        /*   bead parallel   */
         if ( mod( k-1, nprocs_main ) .ne. myrank_main ) cycle

!        //   loop of atom i
         do i = 1, natom

!           /*   force parallel   */
            if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!           //   counter
            l = 0

!           //  loop of atom j
            do j = i+1, natom

!              //   separation of atoms i and j
               rx = x(j,k) - x(i,k)
               ry = y(j,k) - y(i,k)
               rz = z(j,k) - z(i,k)

!              //   apply boundary condition
               call pbc_atom_MPI( rx, ry, rz )

!              //   distance of atoms i and j squared
               r2 = rx*rx + ry*ry + rz*rz

!              //   if distance is smaller than list cutoff radius
               if ( r2 .lt. rcut2_list ) then

!                 //   update counter
                  l = l + 1

!              //   end of if statement
               end if

!           //   loop of atom j
            end do

!           //   number of j atoms in neighbor list for atom i, bead k
            n_list(i,k) = l

!        //   loop of atom i
         end do

!     //   loop of beads
      end do

!     //   communicate
      call my_mpi_allreduce_int_2( n_list, natom, nbead )

!-----------------------------------------------------------------------
!     /*   maximum number of atoms                                    */
!-----------------------------------------------------------------------

!     //   counter
      nmax_list = 0

!     //   loop of beads
      do k = 1, nbead

!        //   loop of atom i
         do i = 1, natom

!           //   maximum number of j atoms in neighbor list
            nmax_list = max( n_list(i,k), nmax_list )

!        //   loop of atom i
         end do

!     //   loop of beads
      end do

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

      if ( allocated( j_list ) ) deallocate( j_list )
      allocate( j_list(nmax_list,natom,nbead) )

!-----------------------------------------------------------------------
!     /*   atoms in neighbor list                                     */
!-----------------------------------------------------------------------

!     //   clear
      j_list(:,:,:) = 0

!     //   loop of beads
      do k = 1, nbead

!        /*   bead parallel   */
         if ( mod( k-1, nprocs_main ) .ne. myrank_main ) cycle

!        //   loop of atom i
         do i = 1, natom

!           /*   force parallel   */
            if ( mod( i-1, nprocs_sub ) .ne. myrank_sub ) cycle

!           //   counter
            l = 0

!           //  loop of atom j
            do j = i+1, natom

!              //   separation of atoms i and j
               rx = x(j,k) - x(i,k)
               ry = y(j,k) - y(i,k)
               rz = z(j,k) - z(i,k)

!              //   apply boundary condition
               call pbc_atom_MPI( rx, ry, rz )

!              //   distance of atoms i and j squared
               r2 = rx*rx + ry*ry + rz*rz

!              //   if distance is smaller than list cutoff radius
               if ( r2 .lt. rcut2_list ) then

!                 //   update counter
                  l = l + 1

!                 //   list of j atoms for atom i, bead k
                  j_list(l,i,k) = j

               end if

!           //  loop of atom j
            end do

!        //   loop of atom i
         end do

!     //   loop of beads
      end do

!     //   communicate
      call my_mpi_allreduce_int_3( j_list, nmax_list, natom, nbead )

!-----------------------------------------------------------------------
!     /*   update neighbor list                                       */
!-----------------------------------------------------------------------

      x_list(:,:) = x(:,:)
      y_list(:,:) = y(:,:)
      z_list(:,:) = z(:,:)

      return
      end





!***********************************************************************
      subroutine force_eam_setup_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   au_length, au_charge, au_energy, ikind, nkind, natom, iounit, &
     &   myrank

      use mm_variables, only : &
     &   xref_eam, yref_eam, y2ref_eam, srho_eam, dfdrho_eam, rcut_eam, &
     &   neam, nref_eam, ikind_eam, iphir_eam, irhor_eam, ifrho_eam

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, j, k, l, ierr, nrhor_eam, nfrho_eam, nphir_eam

!-----------------------------------------------------------------------
!     /*   number of tables                                           */
!-----------------------------------------------------------------------

!     /*   eam   */
      nrhor_eam = nkind
      nfrho_eam = nkind
      nphir_eam = nkind*(nkind+1)/2

!     /*   neam = total number of eam reference tables   */
      neam = nrhor_eam + nfrho_eam + nphir_eam

!-----------------------------------------------------------------------
!     /*   read eam data                                              */
!-----------------------------------------------------------------------

!     /*   master rank   */
      if ( myrank .eq. 0 ) then

!        /*   open file   */
         open (iounit,file = 'eam.dat')

!        /*   tag   */
         call search_tag ( '<nref_eam>', 5, iounit, ierr )

!        /*   number of reference data points in the table   */
         if ( neam .ne. 0 ) read( iounit, *, iostat=ierr ) nref_eam

!        /*   tag   */
         call search_tag ( '<rcut_eam>', 10, iounit, ierr )

!        /*   cut off distance   */
         if ( neam .ne. 0 ) read( iounit, *, iostat=ierr ) rcut_eam

!        /*   angstrom --> bohr   */
         rcut_eam = rcut_eam / au_length * 1.d-10

!        /*   close file   */
         close(iounit)

!        /*   on error, no eam   */
         if ( ierr .ne. 0 ) neam = 0

!     /*   master rank   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   error handling   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_adp_setup_MPI', 30 )

!     /*   communicate   */
      call my_mpi_bcast_int_0( neam )
      call my_mpi_bcast_int_0( nref_eam )
      call my_mpi_bcast_real_0( rcut_eam )

!-----------------------------------------------------------------------
!     /*   memory allocation                                          */
!-----------------------------------------------------------------------

!     /*   species number   */
      if ( .not. allocated( ikind_eam ) ) &
     &   allocate( ikind_eam(natom) )

!     /*   sum of electron density rho_eam   */
      if ( .not. allocated( srho_eam ) ) &
     &   allocate( srho_eam(natom) )

!     /*   gradient of embedding potential   */
      if ( .not. allocated( dfdrho_eam ) ) &
     &   allocate( dfdrho_eam(natom) )

!     /*   table number of rho-r, f-rho, phi-r   */
      if ( .not. allocated( irhor_eam ) ) &
     &   allocate( irhor_eam(nkind) )
      if ( .not. allocated( ifrho_eam ) ) &
     &   allocate( ifrho_eam(nkind) )
      if ( .not. allocated( iphir_eam ) ) &
     &   allocate( iphir_eam(nkind,nkind) )

!     /*   table rho-r, f-rho, phi-r   */
      if ( .not. allocated( xref_eam ) ) &
     &   allocate( xref_eam(nref_eam,neam) )
      if ( .not. allocated( yref_eam ) ) &
     &   allocate( yref_eam(nref_eam,neam) )

!     /*   spline dimensions for rho-r, f-rho, phi-r   */
      if ( .not. allocated( y2ref_eam ) ) &
     &   allocate( y2ref_eam(nref_eam,neam) )

!-----------------------------------------------------------------------
!     /*   make species number                                        */
!-----------------------------------------------------------------------

      ikind_eam(:) = ikind(:)

!-----------------------------------------------------------------------
!     /*   read eam table:  r[angstrom] - rho                         */
!-----------------------------------------------------------------------

!     /*   master rank   */
      if ( myrank .eq. 0 ) then

!        /*   open file   */
         open (iounit,file = 'eam.dat')

!        /*   tag   */
         call search_tag ( '<rhor_eam>', 10, iounit, ierr )

!        /*   loop of table   */
         do i = 1, nrhor_eam

!           /*   read species for the table   */
            read( iounit, *, iostat=ierr ) k

!           /*   read table for species k   */
            do j = 1, nref_eam
               read( iounit, *, iostat=ierr ) &
     &            xref_eam(j,i), yref_eam(j,i)
            end do

!           /*   check error   */
            if ( ierr .ne. 0 ) exit

!           /*   i-th table is for species k   */
            irhor_eam(k) = i

!           /*   angstrom --> bohr   */
            do j = 1, nref_eam
               xref_eam(j,i) = xref_eam(j,i) / au_length * 1.d-10
            end do

!        /*   loop of table   */
         end do

!        /*   close file   */
         close(iounit)

!     /*   master rank   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   error handling   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_eam_setup_MPI', 30 )

!-----------------------------------------------------------------------
!     /*   read eam table:  rho - f[electron volt]                    */
!-----------------------------------------------------------------------

!     /*   master rank   */
      if ( myrank .eq. 0 ) then

!        /*   open file   */
         open (iounit,file = 'eam.dat')

!        /*   tag   */
         call search_tag ( '<frho_eam>', 10, iounit, ierr )

!        /*   loop of table   */
         do i = nrhor_eam+1, nrhor_eam+nfrho_eam

!           /*   read species for the table   */
            read( iounit, *, iostat=ierr ) k

!           /*   read table for species k   */
            do j = 1, nref_eam
               read( iounit, *, iostat=ierr ) &
     &            xref_eam(j,i), yref_eam(j,i)
            end do

!           /*   check error   */
            if ( ierr .ne. 0 ) exit

!           /*   electron volt --> hartree   */
            do j = 1, nref_eam
               yref_eam(j,i) = yref_eam(j,i)
               yref_eam(j,i) = yref_eam(j,i) * au_charge / au_energy
            end do

!           /*   i-th table is for species k   */
            ifrho_eam(k) = i

!        /*   loop of table   */
         end do

!        /*   close file   */
         close(iounit)

!     /*   master rank   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   error handling   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_eam_setup_MPI', 30 )

!-----------------------------------------------------------------------
!     /*   read eam table:  r[angstrom] - phi[electron volt]          */
!-----------------------------------------------------------------------

!     /*   master rank   */
      if ( myrank .eq. 0 ) then

!        /*   open file   */
         open (iounit,file = 'eam.dat')

!        /*   tag   */
         call search_tag ( '<phir_eam>', 10, iounit, ierr )

!        /*   loop of table   */
         do i = nrhor_eam+nfrho_eam+1, nrhor_eam+nfrho_eam+nphir_eam

!           /*   read species for the table   */
            read( iounit, *, iostat=ierr ) k, l

!           /*   read table for species k-l   */
            do j = 1, nref_eam
               read( iounit, *, iostat=ierr ) &
     &            xref_eam(j,i), yref_eam(j,i)
            end do

!           /*   check error   */
            if ( ierr .ne. 0 ) exit

!           /*   angstrom --> bohr,  electron volt --> hartree   */
            do j = 1, nref_eam
               xref_eam(j,i) = xref_eam(j,i) / au_length * 1.d-10
               yref_eam(j,i) = yref_eam(j,i) * au_charge / au_energy
            end do

!           /*   i-th table is for species pair k-l   */
            iphir_eam(k,l) = i
            iphir_eam(l,k) = i

!        /*   loop of table   */
         end do

!        /*   close file   */
         close(iounit)

!     /*   master rank   */
      end if

!     /*   communicate   */
      call my_mpi_bcast_int_0( ierr )

!     /*   error handling   */
      call error_handling_MPI &
     &   ( ierr, 'subroutine force_eam_setup_MPI', 30 )

!-----------------------------------------------------------------------
!     /*   communication                                              */
!-----------------------------------------------------------------------

      call my_mpi_bcast_real_2( xref_eam, nref_eam, neam )
      call my_mpi_bcast_real_2( yref_eam, nref_eam, neam )
      call my_mpi_bcast_real_2( y2ref_eam, nref_eam, neam )

      call my_mpi_bcast_int_1( irhor_eam, nkind )
      call my_mpi_bcast_int_1( ifrho_eam, nkind )
      call my_mpi_bcast_int_2( iphir_eam, nkind, nkind )

!-----------------------------------------------------------------------
!     /*   initialization of spline                                   */
!-----------------------------------------------------------------------

      do i = 1, neam
         call spline_init_eam &
     &      ( xref_eam(:,i), yref_eam(:,i), y2ref_eam(:,i), nref_eam )
      end do

!-----------------------------------------------------------------------

      return
      end





!***********************************************************************
      real(8) function rhor_eam ( r, k )
!***********************************************************************

      use mm_variables, only : &
     &   xref_eam, yref_eam, y2ref_eam, irhor_eam, nref_eam

      implicit none

      integer :: i, k
      real(8) :: r

!-----------------------------------------------------------------------

!     /*   i-th table is for r-rho table for species k   */
      i = irhor_eam(k)

!-----------------------------------------------------------------------

      if ( ( r .lt. xref_eam(1,i) ) .or. &
     &     ( r .gt. xref_eam(nref_eam,i) ) ) then

         rhor_eam = 0.d0

      else

         rhor_eam = getspline_eam &
     &              ( xref_eam(:,i), yref_eam(:,i), y2ref_eam(:,i), &
     &                nref_eam, r )

      end if

!-----------------------------------------------------------------------

      return
      contains
      include 'getspline_eam_func.F90'
      end





!***********************************************************************
      real(8) function rhor_grad_eam ( r, k )
!***********************************************************************

      use mm_variables, only : &
     &   xref_eam, yref_eam, y2ref_eam, irhor_eam, nref_eam

      implicit none

      integer :: i, k
      real(8) :: r

!-----------------------------------------------------------------------

!     /*   i-th table is for r-rho table for species k   */
      i = irhor_eam(k)

!-----------------------------------------------------------------------

      if ( ( r .lt. xref_eam(1,i) ) .or. &
     &     ( r .gt. xref_eam(nref_eam,i) ) ) then

         rhor_grad_eam = 0.d0

      else

         rhor_grad_eam = getspline_grad_eam &
     &                   ( xref_eam(:,i), yref_eam(:,i), y2ref_eam(:,i), &
     &                     nref_eam, r )

      end if

!-----------------------------------------------------------------------

      return
      contains
      include 'getspline_grad_eam_func.F90'
      end





!***********************************************************************
      real(8) function frho_eam ( srho, k )
!***********************************************************************

      use mm_variables, only : &
     &   xref_eam, yref_eam, y2ref_eam, ifrho_eam, nref_eam

      implicit none

      integer :: i, k
      real(8) :: srho

!-----------------------------------------------------------------------

!     /*   i-th table is for rho-f table for species k   */
      i = ifrho_eam(k)

!-----------------------------------------------------------------------

      if ( ( srho .lt. xref_eam(1,i) ) .or. &
     &     ( srho .gt. xref_eam(nref_eam,i) ) ) then

         frho_eam = 0.d0

      else

         frho_eam = getspline_eam &
     &              ( xref_eam(:,i), yref_eam(:,i), y2ref_eam(:,i), &
     &                nref_eam, srho )

      end if

!-----------------------------------------------------------------------

      return
      contains
      include 'getspline_eam_func.F90'
      end





!***********************************************************************
      real(8) function frho_grad_eam ( srho, k )
!***********************************************************************

      use mm_variables, only : &
     &   xref_eam, yref_eam, y2ref_eam, ifrho_eam, nref_eam

      implicit none

      integer :: i, k
      real(8) :: srho

!-----------------------------------------------------------------------

!     /*   i-th table is for rho-f table for species k   */
      i = ifrho_eam(k)

!-----------------------------------------------------------------------

      if ( ( srho .lt. xref_eam(1,i) ) .or. &
     &     ( srho .gt. xref_eam(nref_eam,i) ) ) then

         frho_grad_eam = 0.d0

      else

         frho_grad_eam = getspline_grad_eam &
     &                   ( xref_eam(:,i), yref_eam(:,i), y2ref_eam(:,i), &
     &                     nref_eam, srho )

      end if

!-----------------------------------------------------------------------

      return
      contains
      include 'getspline_grad_eam_func.F90'
      end





!***********************************************************************
      real(8) function phir_eam ( r, k, l )
!***********************************************************************

      use mm_variables, only : &
     &   xref_eam, yref_eam, y2ref_eam, iphir_eam, nref_eam

      implicit none

      integer :: i, k, l
      real(8) :: r

!-----------------------------------------------------------------------

!     /*   i-th table is for r-phi table for species k   */
      i = iphir_eam(k,l)

!-----------------------------------------------------------------------

      if ( ( r .lt. xref_eam(1,i) ) .or. &
     &     ( r .gt. xref_eam(nref_eam,i) ) ) then

         phir_eam = 0.d0

      else

         phir_eam = getspline_eam &
     &              ( xref_eam(:,i), yref_eam(:,i), y2ref_eam(:,i), &
     &                nref_eam, r )

      end if

!-----------------------------------------------------------------------

      return
      contains
      include 'getspline_eam_func.F90'
      end





!***********************************************************************
      real(8) function phir_grad_eam ( r, k, l )
!***********************************************************************

      use mm_variables, only : &
     &   xref_eam, yref_eam, y2ref_eam, iphir_eam, nref_eam

      implicit none

      integer :: i, k, l
      real(8) :: r

!-----------------------------------------------------------------------

!     /*   i-th table is for r-phi table for species k   */
      i = iphir_eam(k,l)

!-----------------------------------------------------------------------

      if ( ( r .lt. xref_eam(1,i) ) .or. &
     &     ( r .gt. xref_eam(nref_eam,i) ) ) then

         phir_grad_eam = 0.d0

      else

         phir_grad_eam = getspline_grad_eam &
     &                   ( xref_eam(:,i), yref_eam(:,i), y2ref_eam(:,i), &
     &                     nref_eam, r )

      end if

!-----------------------------------------------------------------------

      return
      contains
      include 'getspline_grad_eam_func.F90'
      end





!***********************************************************************
      subroutine spline_init_eam ( x, y, y2, n )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      integer :: i, n, k

      real(8) :: yp1, ypn, sig, p, qn, un

      real(8) :: x(n), y(n), y2(n), u(n)

!-----------------------------------------------------------------------
!     /*   start                                                      */
!-----------------------------------------------------------------------

      yp1 = ( y(2)-y(1) ) / ( x(2)-x(1) )
      ypn = ( y(n)-y(n-1) ) / ( x(n)-x(n-1) )

      y2(1) = - 0.5d0
      u(1) = ( 3.d0 / ( x(2)-x(1) ) ) * ( (y(2)-y(1) ) &
     &           / ( x(2)-x(1) ) - yp1 )

      do i = 2, n-1
         sig = ( x(i)-x(i-1) ) / ( x(i+1)-x(i-1) )
         p = sig * y2(i-1) + 2.d0
         y2(i) = ( sig-1.d0 ) / p
         u(i) = ( 6.d0 * ( ( y(i+1) - y(i) ) &
     &              / ( x(i+1)-x(i) ) - ( y(i)-y(i-1) ) &
     &              / ( x(i)-x(i-1) ) ) / ( x(i+1)-x(i-1) ) &
     &              - sig * u(i-1) ) / p
      end do

      qn = 0.5d0
      un = ( 3.d0 / ( x(n) - x(n-1) ) ) * ( ypn - ( y(n)-y(n-1) ) &
     &    / ( x(n) - x(n-1) ) )

      y2(n) = ( un - qn * u(n-1) ) / ( qn * y2(n-1) + 1.d0 )

      do k = n-1, 1, -1
         y2(k) = y2(k) * y2(k+1) + u(k)
      end do

      return
      end

