!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga, B. Thomsen
!      Last updated:    Feb 21, 2020 by M. Shiga
!      Description:     energy and force from DFTB calculation
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      module dftb_variables
!***********************************************************************

!     /*   atomic charges   */
      real(8), dimension(:,:), allocatable :: q_dftb

!     /*   flag for restarting atomic charges   */
      integer :: istart_dftb

!     /*   DFTB+ version used        */
      character(len=80) :: dftbpver

!***********************************************************************
      end module dftb_variables
!***********************************************************************





!***********************************************************************
      subroutine force_dftb
!***********************************************************************
!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, fx, fy, fz, pot, vir, dipx, dipy, dipz, &
     &   au_length, dftb_exe_command, box, volume, iounit, mbox, &
     &   iounit_dftb, nbead, natom, iboundary

      use dftb_variables, only : &
     &   q_dftb, istart_dftb, dftbpver

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

      real(8), parameter :: bohr2ang = au_length/1.d-10

      character(len=80):: char_line, char_line_2, char_command
      character(len=80):: char_dummy
      character(len=3)::  char_num

      real(8) :: xi, yi, zi, sx, sy, sz

      real(8) :: stress_tensor(3,3)

      integer :: ierr, ibead, iblank, i, j, k, m1, m2, m3, itest

      integer, save :: iset = 0

!-----------------------------------------------------------------------
!     /*   initialize directory                                       */
!-----------------------------------------------------------------------

!     /*   error flag   */
      ierr = 0

!     /*   initial settings   */
      if ( iset .eq. 0 ) then

!        /*   memory allocation   */
         if ( .not. allocated( q_dftb ) ) &
     &      allocate( q_dftb(natom,nbead) )

!        /*   loop of beads   */
         do ibead = 1, nbead

!           /*   bead number   */
            call int3_to_char( ibead, char_num )

!           /*   remove old directory and create new directory   */
            call system ('rm -f -r ./' // char_num )
            call system ('mkdir -p ./' // char_num )

!c           /*   copy necessary files   */
!            call system ('cp *.skf ./' // char_num )

!        /*   loop of beads   */
         end do

!        /*   file open   */
         open ( iounit, file = 'input.dat' )

!        /*   search for tag    */
         call search_tag ( '<dftb_exe_command>', 18, iounit, ierr )

!        /*   read a line   */
         read ( iounit, *, iostat=ierr ) dftb_exe_command

!        /*   file close   */
         close ( iounit )

!        /*   default   */
         if ( ierr .ne. 0 ) then

!           /*   file open   */
            open ( iounit, file = 'input_default.dat' )

!           /*   search for tag    */
            call search_tag( '<dftb_exe_command>', 18, iounit, ierr )

!           /*   read a line
            read ( iounit, *, iostat=ierr ) dftb_exe_command

!           /*   file close   */
            close ( iounit )

!        /*   default   */
         end if

!-----------------------------------------------------------------------
!        /*   confirm dftb command                                    */
!-----------------------------------------------------------------------

!c        /*   dftb command   */
!         read( dftb_exe_command, *, iostat=ierr ) char_command

!        /*   dftb command   */
         char_command = ''
         do i = 1, 80
            if ( dftb_exe_command(i:i) .eq. ' ' ) exit
            char_command(i:i) = dftb_exe_command(i:i)
         end do

!        /*   check if dftb command exists   */
         call system &
     &      ("echo '0' > test.out")
         call system &
     &      ("sleep 0.1")
         call system &
     &      ("which " // trim(char_command) // &
     &       " > /dev/null 2>&1 && echo '1' > test.out")

!        /*   open file   */
         open ( iounit, file = 'test.out' )

!        /*   read itest   */
         read ( iounit, * ) itest

!        /*   close file   */
         close( iounit )

!        /*   itest: dftb command not found   */
         if ( itest .eq. 0 ) then

!           /*   error flag   */
            ierr = 1

!           /*   print   */
            write( 6, '(a)' ) 'Error - DFTB command not found: ' // &
     &                         trim(char_command)
            write( 6, '(a)' )

!        /*   itest: dftb command found   */
         else

!           /*   error flag   */
            ierr = 0

!           /*   print   */
            write( 6, '(a)' ) 'DFTB command found: ' // &
     &                         trim(char_command)
            write( 6, '(a)' )

!        /*   itest   */
         end if

!        /*   remove file   */
         call system('rm -f test.out')

!        /*   error termination   */
         call error_handling &
     &       ( ierr, 'subroutine force_dftb', 21 )

!        /*   dftb+ version   */
         call read_char &
     &      ( dftbpver, 80, '<dftb_version>', 14, iounit )

!        /*   default version   */
         if ( ( dftbpver(1:4) .ne. '18.2' ) .and. &
     &        ( dftbpver(1:4) .ne. '19.1' ) ) dftbpver = '19.1'

!        /*   set ended   */
         iset = 1

!     /*   initial settings   */
      end if

!-----------------------------------------------------------------------
!     /*   charges                                                    */
!-----------------------------------------------------------------------

!     /*   itest: file exists   */
      call testfile ( 'dftb.ini', 8, itest )

!     /*   initialize flag   */
      istart_dftb = 0

!     /*   file exists   */
      if ( itest .eq. 0 ) then

!        /*   flag   */
         istart_dftb = 1

!        /*   open the dftb restart file   */
         open ( iounit, file = 'dftb.ini' )

!        /*   read atomic charges   */
         do ibead = 1, nbead
         do i = 1, natom
            read( iounit, * ) j, k, q_dftb(i,ibead)
         end do
         end do

!        /*   close file   */
         close(iounit)

!     /*   file exists   */
      end if

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!     /*   make char_num according to bead                            */
!-----------------------------------------------------------------------

      call int3_to_char( ibead, char_num )

!-----------------------------------------------------------------------
!     /*   make dftb input:  dftb_in.hsd                              */
!-----------------------------------------------------------------------

!     /*   open the dftb prototype file   */
      open ( iounit,     file = 'dftb.dat'   )

!     /*   open the dftb input file   */
      open ( iounit_dftb, file = ('./' // char_num // '/dftb_in.hsd') )

!     /*   do loop end   */
      do

!        /*   read a line   */
         read ( iounit, '(a)', iostat=ierr ) char_line

!        /*   exit at the end of the line   */
         if ( ierr .ne. 0 ) exit

!        /*   adjust to the left   */
         char_line_2 = adjustl(char_line)

!        /*   line: geometry   */
         if ( char_line_2(1:8) .eq. 'Geometry' ) then

!           /*   write a copy of the line   */
            write( iounit_dftb, '(a)' ) char_line

!           /*   number of lines   */
            iblank = 0

!           /*   start loop of headlines   */
            do

!              /*   read a line   */
               read ( iounit, '(a)', iostat=ierr ) char_line

!              /*   write a copy of the line   */
               write( iounit_dftb, '(a)' ) char_line

!              /*   adjust to the left   */
               char_line_2 = adjustl(char_line)

!              /*   skip comment   */
               if ( char_line_2(1:1) .ne. '#' ) iblank = iblank + 1

!              /*   if two lines have been found   */
               if ( iblank .eq. 2 ) exit

!           /*   end loop of headlines   */
            end do

!           /*   start loop of atoms   */
            do i = 1, natom

!              /*   start test loop   */
               do

!                 /*   read a line   */
                  read ( iounit, '(a)', iostat=ierr ) char_line

!                 /*   adjust to the left   */
                  char_line_2 = adjustl(char_line)

!                 /*   comment found   */
                  if ( char_line_2(1:1) .eq. '#' ) then

!                    /*   write a copy of the line   */
                     write( iounit_dftb, '(a)' ) char_line

!                 /*   data found   */
                  else

!                    /*   go back one line   */
                     backspace( iounit )

!                    /*   exit test loop   */
                     exit

!                 /*   comment or data   */
                  end if

!              /*   end test loop   */
               end do

!              /*   coordinates   */
               xi = x(i,ibead) * bohr2ang
               yi = y(i,ibead) * bohr2ang
               zi = z(i,ibead) * bohr2ang

!              /*   read atomic number   */
               read ( iounit, *, iostat=ierr ) j, k

!              /*   write atomic number and coordinates   */
               write( iounit_dftb, '(2i4,3e24.16)' ) j, k, xi, yi, zi

!           /*   end loop of atoms   */
            end do

!           /*   periodic boundary condition   */
            if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then

!              /*   start test loop   */
               do

!                 /*   read a line   */
                  read ( iounit, '(a)', iostat=ierr ) char_line

!                 /*   adjust to the left   */
                  char_line_2 = adjustl(char_line)

!                 /*   comment found   */
                  if ( char_line_2(1:1) .eq. '#' ) then

!                    /*   write a copy of the line   */
                     write( iounit_dftb, '(a)' ) char_line

!                 /*   data found   */
                  else

!                    /*   go back one line   */
                     backspace( iounit )

!                    /*   exit test loop   */
                     exit

!                 /*   comment or data   */
                  end if

!              /*   end test loop   */
               end do

!              /*   read coordinates at origin   */
               read ( iounit, *, iostat=ierr ) xi, yi, zi

!              /*   write coordinate at origin   */
               write( iounit_dftb, '(3e24.16)' ) xi, yi, zi

!              /*   start loop of box   */
               do i = 1, 3

!                 /*   start test loop   */
                  do

!                    /*   read a line   */
                     read ( iounit, '(a)', iostat=ierr ) char_line

!                    /*   adjust to the left   */
                     char_line_2 = adjustl(char_line)

!                    /*   comment found   */
                     if ( char_line_2(1:1) .eq. '#' ) then

!                       /*   write a copy of the line   */
                        write( iounit_dftb, '(a)' ) char_line

!                    /*   data found   */
                     else

!                       /*   go back one line   */
                        backspace( iounit )

!                       /*   exit test loop   */
                        exit

!                    /*   comment or data   */
                     end if

!                 /*   end test loop   */
                  end do

!                 /*   read atomic coordinates   */
                  read ( iounit, *, iostat=ierr ) xi, yi, zi

                  xi = box(1,i) * bohr2ang
                  yi = box(2,i) * bohr2ang
                  zi = box(3,i) * bohr2ang

!                 /*   write atomic coordinates   */
                  write( iounit_dftb, '(3e24.16)' ) xi, yi, zi

!              /*   end loop of box   */
               end do

!           /*   periodic boundary condition   */
            end if

!        /*   line: charges   */
         else if ( char_line_2(1:14) .eq. 'AllAtomCharges' ) then

!           /*   write a copy of the line   */
            write( iounit_dftb, '(a)' ) char_line

!           /*   restart charges   */
            if ( istart_dftb .eq. 1 ) then

!              /*   read charges   */
               do i = 1, natom
                  read( iounit, *, iostat=ierr ) xi
               end do

!              /*   write charges   */
               do i = 1, natom
                  write( iounit_dftb, '(f12.8)' ) q_dftb(i,ibead)
               end do

!           /*   restart charges   */
            end if

!        /*   line: other   */
         else

!           /*   write a copy of the line   */
            write( iounit_dftb, '(a)' ) char_line

!        /*   line   */
         end if

!     /*   do loop end   */
      end do

!     /*   add a blank line at the end   */
      write( iounit_dftb, '(a)' )

!     /*   close files   */
      close( iounit )

!     /*   close files   */
      close( iounit_dftb )

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   stress   */
      stress_tensor(:,:) = 0.d0

!     /*   dipole moment   */
      dipx(:) = 0.d0
      dipy(:) = 0.d0
      dipz(:) = 0.d0

!     /*   charges   */
      q_dftb(:,:) = 0.d0

!-----------------------------------------------------------------------
!     /*   start loop of beads                                        */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!     /*   make char_num according to bead                            */
!-----------------------------------------------------------------------

      call int3_to_char( ibead, char_num )

!-----------------------------------------------------------------------
!     /*   run dftb                                                   */
!-----------------------------------------------------------------------

      call system ( 'cd ' // char_num // '; ' // &
     &              dftb_exe_command // ' > output.dftb' )

!-----------------------------------------------------------------------
!     /*   read dftb output:  potential                               */
!-----------------------------------------------------------------------

!     /*   open the dftb output file   */
      open ( iounit_dftb, file = ('./' // char_num // '/detailed.out') )

!     /*   do loop start   */
      do

!        /*   read a line   */
         read ( iounit_dftb, '(a)', iostat=ierr ) char_line

!        /*   exit at the end of the line   */
         if ( ierr .ne. 0 ) exit

!        /*   adjust to the left   */
         char_line = adjustl(char_line)

!        /*   see if the line matches DFTB+ version 18.2   */
         if ( ( dftbpver(1:4) .eq. '18.2') .and.  &
     &        ( char_line(1:12) .eq. 'Total Mermin' ) ) then

!           /*   step back one line   */
            backspace( iounit_dftb )

!           /*   read potential   */
            read ( iounit_dftb, * ) char_dummy, char_dummy, &
     &                              char_dummy, char_dummy, pot(ibead)

!           /*   finished   */
            exit

!        /*   see if the line matches DFTB+ version 19.1   */
         else if ( (dftbpver(1:4) .eq. '19.1' ) .and. &
     &            ( char_line(1:20) .eq. 'Force related energy' ) ) then

!           /*   step back one line   */
            backspace( iounit_dftb )

!           /*   read potential   */
            read ( iounit_dftb, * ) char_dummy, char_dummy, &
     &                              char_dummy, pot(ibead)
!           /*   finished   */
            exit

         end if

!     /*   do loop end   */
      end do

!     /*   close file   */
      close( iounit_dftb )

!-----------------------------------------------------------------------
!     /*   read dftb output:  potential gradient                      */
!-----------------------------------------------------------------------

!     /*   open the dftb output file   */
      open ( iounit_dftb, file = ('./' // char_num // '/detailed.out') )

!     /*   do loop start   */
      do

!        /*   read a line   */
         read ( iounit_dftb, '(a)', iostat=ierr ) char_line

!        /*   exit at the end of the line   */
         if ( ierr .ne. 0 ) exit

!        /*   adjust to the left   */
         char_line = adjustl(char_line)

!        /*   see if the line matches   */
         if ( char_line(1:12) .eq. 'Total Forces' ) then

!           /*   read forces   */
            if ( dftbpver(1:4) .eq. '18.2' ) then

               do i = 1, natom
                  read( iounit_dftb, * )  &
     &               fx(i,ibead), fy(i,ibead), fz(i,ibead)
               end do

!              /*   finished   */
               exit

            else if ( dftbpver(1:4) .eq. '19.1' ) then

               do i = 1, natom
                  read( iounit_dftb, * )  &
     &               char_dummy, fx(i,ibead), fy(i,ibead), fz(i,ibead)
               end do

!              /*   finished   */
               exit

            end if

!        /*   see if the line matches   */
         end if

!     /*   do loop end   */
      end do

!     /*   close file   */
      close(iounit_dftb)

!-----------------------------------------------------------------------
!     /*   read dftb output:  stress tensor                           */
!-----------------------------------------------------------------------

!     /*   periodic boundary  */
      if ( ( iboundary .eq. 1 ) .or. ( iboundary .eq. 2 ) ) then

!     /*   open the dftb output file   */
      open ( iounit_dftb, file = ('./' // char_num // '/detailed.out') )

!     /*   do loop start   */
      do

!        /*   read a line   */
         read ( iounit_dftb, '(a)', iostat=ierr ) char_line

!        /*   exit at the end of the line   */
         if ( ierr .ne. 0 ) exit

!        /*   adjust to the left   */
         char_line_2 = adjustl(char_line)

!        /*   see if the line matches   */
         if ( char_line_2(1:19) .eq. 'Total stress tensor' ) then

!           /*   read stress tensor   */

            do i = 1, 3
               read( iounit_dftb, * ) sx, sy, sz
               stress_tensor(i,1) = stress_tensor(i,1) + sx
               stress_tensor(i,2) = stress_tensor(i,2) + sy
               stress_tensor(i,3) = stress_tensor(i,3) + sz
            end do

!           /*   finished   */
            exit

!        /*   see if the line matches   */
         end if

!     /*   do loop end   */
      end do

!     /*   close file   */
      close(iounit_dftb)

!     /*   periodic boundary  */
      end if

!-----------------------------------------------------------------------
!     /*   read dftb output:  charges                                 */
!-----------------------------------------------------------------------

!     /*   open the dftb output file   */
      open ( iounit_dftb, file = ('./' // char_num // '/detailed.out') )

!     /*   do loop start   */
      do

!        /*   read a line   */
         read ( iounit_dftb, '(a)', iostat=ierr ) char_line

!        /*   exit at the end of the line   */
         if ( ierr .ne. 0 ) exit

!        /*   adjust to the left   */
         char_line_2 = adjustl(char_line)

!        /*   see if the line matches   */
         if ( ( char_line_2(1:20) .eq. 'Atomic gross charges' ) .or. &
     &        ( char_line_2(1:18) .eq. 'Net atomic charges' ) ) then

!           /*   read atomic charges   */
            read( iounit_dftb, * )
            do i = 1, natom
               read( iounit_dftb, * ) j, q_dftb(i,ibead)
            end do

!           /*   finished   */
            exit

!        /*   see if the line matches   */
         end if

!     /*   do loop end   */
      end do

!     /*   close file   */
      close(iounit_dftb)

!-----------------------------------------------------------------------
!     /*   read dftb output:  dipole moment not read                  */
!-----------------------------------------------------------------------

!     /*   loop of atoms   */
      do i = 1, natom

!        /*   coordinates   */
         xi = x(i,ibead)
         yi = y(i,ibead)
         zi = z(i,ibead)

!        /*   original box index   */
         m1 = mbox(1,i,ibead)
         m2 = mbox(2,i,ibead)
         m3 = mbox(3,i,ibead)

!        /*   apply periodic boundary   */
         call pbc_unfold( xi, yi, zi, m1, m2, m3 )

!        /*   dipole moment   */
         dipx(ibead) = dipx(ibead) + q_dftb(i,ibead) * xi
         dipy(ibead) = dipy(ibead) + q_dftb(i,ibead) * yi
         dipz(ibead) = dipz(ibead) + q_dftb(i,ibead) * zi

!     /*   loop of atoms   */
      end do

!-----------------------------------------------------------------------
!     /*   end loop of beads                                          */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   virial                                                     */
!-----------------------------------------------------------------------

!     /*   loop of beads   */
      do j = 1, nbead

!        /*   add stress contribution to virial   */
         vir(1,1) = vir(1,1) + stress_tensor(1,1) * volume / nbead
         vir(1,2) = vir(1,2) + stress_tensor(1,2) * volume / nbead
         vir(1,3) = vir(1,3) + stress_tensor(1,3) * volume / nbead
         vir(2,1) = vir(2,1) + stress_tensor(2,1) * volume / nbead
         vir(2,2) = vir(2,2) + stress_tensor(2,2) * volume / nbead
         vir(2,3) = vir(2,3) + stress_tensor(2,3) * volume / nbead
         vir(3,1) = vir(3,1) + stress_tensor(3,1) * volume / nbead
         vir(3,2) = vir(3,2) + stress_tensor(3,2) * volume / nbead
         vir(3,3) = vir(3,3) + stress_tensor(3,3) * volume / nbead

!     /*   loop of beads   */
      end do

!-----------------------------------------------------------------------
!     /*   charges                                                    */
!-----------------------------------------------------------------------

!     /*   open the dftb restart file   */
      open ( iounit, file = 'dftb.ini' )

!     /*   read atomic charges   */
      do ibead = 1, nbead
      do i = 1, natom
         write( iounit, '(i4,i8,f12.8)' ) ibead, i, q_dftb(i,ibead)
      end do
      end do

!     /*   close file   */
      close(iounit)

      return
      end

