!///////////////////////////////////////////////////////////////////////
!
!      Author:          M. Shiga
!      Last updated:    Nov 10, 2018 by M. Shiga
!      Description:     BEST method
!
!///////////////////////////////////////////////////////////////////////
!***********************************************************************
      subroutine force_best_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : method

!-----------------------------------------------------------------------
!     /*   for path integral methods                                  */
!-----------------------------------------------------------------------

      if ( ( method(1:6) .eq. 'PIMD  ' ) .or. &
     &     ( method(1:6) .eq. 'PIHMC ' ) .or. &
     &     ( method(1:6) .eq. 'CMD   ' ) .or. &
     &     ( method(1:6) .eq. 'RPMD  ' ) .or. &
     &     ( method(1:6) .eq. 'MTS   ' ) ) then

!        /*   centroid-based best is used   */
         call force_best_cent_MPI

!-----------------------------------------------------------------------
!     /*   for all other methods                                      */
!-----------------------------------------------------------------------

      else

!         /*   bead-based best is used   */
         call force_best_bead_MPI

      end if

      return
      end





!***********************************************************************
      subroutine force_best_bead_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   x, y, z, nbead, istep, iounit, myrank

      use best_variables, only : pot_best, v_best, dv_best, s_best, &
     &   sx_best, sy_best, sz_best, fx_best, fy_best, fz_best, &
     &   vir_best, s_sort, r_best, num_sort, ibest, nbest, ioption_best, &
     &   iobest, npbest, iprint_best, nbest_buf, npbest_buf, nsbest_buf

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j, k, ibead, ierr

!     /*   real values   */
      real(8) :: factor, fxi, fyi, fzi, xij, yij, zij

!     /*   small value   */
      real(8) :: small_value = 1.d-8

!-----------------------------------------------------------------------
!     /*   skip if best is off                                        */
!-----------------------------------------------------------------------

      if ( ioption_best .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   bias potential   */
      pot_best(:) = 0.d0

!     /*   bias force   */
      fx_best(:,:) = 0.d0
      fy_best(:,:) = 0.d0
      fz_best(:,:) = 0.d0

!     /*   bias virial   */
      vir_best(:,:) = 0.d0

!     /*   error flag   */
      ierr = 0

!-----------------------------------------------------------------------
!     /*   loop of beads                                              */
!-----------------------------------------------------------------------

      do ibead = 1, nbead

!-----------------------------------------------------------------------
!        /*   calculate index s: solute-solvent distances             */
!-----------------------------------------------------------------------

         call get_s_best_bead_MPI( ibead )

!-----------------------------------------------------------------------
!        /*   calculate bias function                                 */
!-----------------------------------------------------------------------

         call get_v_best( ierr )

!-----------------------------------------------------------------------
!        /*   error termination                                       */
!-----------------------------------------------------------------------

         if ( ierr .ne. 0 ) exit

!-----------------------------------------------------------------------
!        /*   bias potential                                          */
!-----------------------------------------------------------------------

         pot_best(ibead) = v_best

!-----------------------------------------------------------------------
!        /*   bias force and virial                                   */
!-----------------------------------------------------------------------

!        /*   loop of solvent atoms   */
         do k = 1, nbest

!           /*   solute atom   */
            i = iobest

!           /*   solvent atom   */
            j = ibest(k)

!           /*   solute-solvent distance   */
            xij = x(i,ibead) - x(j,ibead)
            yij = y(i,ibead) - y(j,ibead)
            zij = z(i,ibead) - z(j,ibead)

!           /*   apply periodic boundary condition   */
            call pbc_atom_MPI ( xij, yij, zij )

!           /*   small contribution is neglected   */
            if ( abs(s_best(k)) .lt. small_value ) cycle

!           /*   factor   */
            factor = dv_best(k) / s_best(k)

!           /*   solute-solute force   */
            fxi = - factor * sx_best(k)
            fyi = - factor * sy_best(k)
            fzi = - factor * sz_best(k)

!           /*   solvent force   */
            fx_best(j,ibead) = fx_best(j,ibead) - fxi
            fy_best(j,ibead) = fy_best(j,ibead) - fyi
            fz_best(j,ibead) = fz_best(j,ibead) - fzi

!           /*   solute force   */
            fx_best(i,ibead) = fx_best(i,ibead) + fxi
            fy_best(i,ibead) = fy_best(i,ibead) + fyi
            fz_best(i,ibead) = fz_best(i,ibead) + fzi

!           /*   virial   */
            vir_best(1,1) = vir_best(1,1) + fxi*xij
            vir_best(1,2) = vir_best(1,2) + fxi*yij
            vir_best(1,3) = vir_best(1,3) + fxi*zij
            vir_best(2,1) = vir_best(2,1) + fyi*xij
            vir_best(2,2) = vir_best(2,2) + fyi*yij
            vir_best(2,3) = vir_best(2,3) + fyi*zij
            vir_best(3,1) = vir_best(3,1) + fzi*xij
            vir_best(3,2) = vir_best(3,2) + fzi*yij
            vir_best(3,3) = vir_best(3,3) + fzi*zij

!        /*   loop of solvent atoms   */
         end do

!-----------------------------------------------------------------------
!     /*   loop of beads                                              */
!-----------------------------------------------------------------------

      end do

!-----------------------------------------------------------------------
!     /*   error termination                                          */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then
 
      if ( ierr .eq. 1 ) then

          write( 6, '(a)' ) &
     &       'Error - Bad geometry for BEST method.'

          write( 6, '(a)' )

          write( 6, '(a,f12.5)' ) 'cut off distance:', r_best

          write( 6, '(a)' )

          write( 6, '(a)' ) &
     &       '------------------------------------------------'
          write( 6, '(a)' ) &
     &       'old    atom       index  new    atom       index'
          write( 6, '(a)' ) &
     &       '------------------------------------------------'

          write( 6, '(a3,i8,f12.5,2x,a3,i8,f12.5)' ) &
     &          'A  ', iobest, 0.d0, &
     &          'A  ', iobest, 0.d0
          do i = 1, npbest
             write( 6, '(a3,i8,f12.5,2x,a3,i8,f12.5)' ) &
     &          'A  ', ibest(i), s_best(i), &
     &          'A  ', ibest(num_sort(i)), s_sort(i)
          end do
          do i = npbest+1, nbest
             write( 6, '(a3,i8,f12.5,2x,a3,i8,f12.5)' ) &
     &          'B  ', ibest(i), s_best(i), &
     &          'B  ', ibest(num_sort(i)), s_sort(i)
          end do

          write( 6, '(a)' ) 

      else if ( ierr .eq. 2 ) then

         write( 6, '(a)' ) &
     &      'Error - Buffer region for BEST method too large.'

      end if

      end if

      call error_handling_MPI &
     &   ( ierr, 'subroutine force_best_bead_MPI', 30 )

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then
      if ( iprint_best .gt. 0 ) then
      if ( mod(istep,iprint_best) .eq. 0 ) then

         open ( iounit, file = 'best.out', access = 'append' )

         do i = 1, nbead
            write( iounit, '(i8,3i4,f16.8)' ) &
     &      istep, nbest_buf, npbest_buf, nsbest_buf, pot_best(i)
         end do

         close( iounit )

      end if
      end if
      end if

      return
      end





!***********************************************************************
      subroutine force_best_cent_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   ux, uy, uz, nbead, istep, iounit, method, myrank

      use best_variables, only : pot_best, v_best, dv_best, s_best, &
     &   sx_best, sy_best, sz_best, fux_best, fuy_best, fuz_best, &
     &   vir_best, s_sort, r_best, num_sort, ibest, nbest, ioption_best, &
     &   iobest, npbest, iprint_best, nbest_buf, npbest_buf, nsbest_buf

      use qmmm_variables, only : &
     &    jref

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: i, j, k, ierr

!     /*   real values   */
      real(8) :: factor, fxi, fyi, fzi, xij, yij, zij

!     /*   small value   */
      real(8) :: small_value = 1.d-8

!-----------------------------------------------------------------------
!     /*   skip if best is off                                        */
!-----------------------------------------------------------------------

      if ( ioption_best .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   initialize                                                 */
!-----------------------------------------------------------------------

!     /*   bias potential   */
      pot_best(:) = 0.d0

!     /*   bias force   */
      fux_best(:,:) = 0.d0
      fuy_best(:,:) = 0.d0
      fuz_best(:,:) = 0.d0

!     /*   bias virial   */
      vir_best(:,:) = 0.d0

!-----------------------------------------------------------------------
!     /*   calculate index s: solute-solvent distances                */
!-----------------------------------------------------------------------

      call get_s_best_cent_MPI

!-----------------------------------------------------------------------
!     /*   calculate bias function                                    */
!-----------------------------------------------------------------------

      call get_v_best( ierr )

!-----------------------------------------------------------------------
!     /*   error termination                                          */
!-----------------------------------------------------------------------

      if ( ierr .ne. 0 ) go to 100

!-----------------------------------------------------------------------
!        /*   bias potential                                          */
!-----------------------------------------------------------------------

      do i = 1, nbead
         pot_best(i) = v_best / dble(nbead)
      end do

!-----------------------------------------------------------------------
!     /*   bias force and virial                                      */
!-----------------------------------------------------------------------

!     /*   loop of solvent atoms   */
      do k = 1, nbest

!        /*   solute atom   */
         i = iobest

!        /*   solvent atom   */
         j = ibest(k)

!        /*   solute-solvent distance   */
         xij = ux(i,1) - ux(j,1)
         yij = uy(i,1) - uy(j,1)
         zij = uz(i,1) - uz(j,1)

!        /*   apply periodic boundary condition   */
         call pbc_atom_MPI( xij, yij, zij )

!        /*   small contribution is neglected   */
         if ( abs(s_best(k)) .lt. small_value ) cycle

!        /*   factor   */
         factor = dv_best(k) / s_best(k)

!        /*   solute-solute force   */
         fxi = - factor * sx_best(k)
         fyi = - factor * sy_best(k)
         fzi = - factor * sz_best(k)

!        /*   solvent force   */
         fux_best(j,1) = fux_best(j,1) - fxi
         fuy_best(j,1) = fuy_best(j,1) - fyi
         fuz_best(j,1) = fuz_best(j,1) - fzi

!        /*   solute force   */
         fux_best(i,1) = fux_best(i,1) + fxi
         fuy_best(i,1) = fuy_best(i,1) + fyi
         fuz_best(i,1) = fuz_best(i,1) + fzi

!        /*   virial   */
         vir_best(1,1) = vir_best(1,1) + fxi*xij
         vir_best(1,2) = vir_best(1,2) + fxi*yij
         vir_best(1,3) = vir_best(1,3) + fxi*zij
         vir_best(2,1) = vir_best(2,1) + fyi*xij
         vir_best(2,2) = vir_best(2,2) + fyi*yij
         vir_best(2,3) = vir_best(2,3) + fyi*zij
         vir_best(3,1) = vir_best(3,1) + fzi*xij
         vir_best(3,2) = vir_best(3,2) + fzi*yij
         vir_best(3,3) = vir_best(3,3) + fzi*zij

      end do

!-----------------------------------------------------------------------
!     /*   normal mode to cartesian                                   */
!-----------------------------------------------------------------------

      call nm_trans_force_best_MPI( 0 )

!-----------------------------------------------------------------------
!     /*   error termination                                          */
!-----------------------------------------------------------------------

  100 continue

      if ( myrank .eq. 0 ) then

      if ( ierr .eq. 1 ) then

          write( 6, '(a)' ) &
     &       'Error - Bad geometry for BEST method.'

          write( 6, '(a)' )

          write( 6, '(a,f12.5)' ) 'cut off distance:', r_best

          write( 6, '(a)' )

          write( 6, '(a,f12.5)' ) 'cut off distance:', r_best

          write( 6, '(a)' )

          write( 6, '(a)' ) &
     &       '------------------------------------------------'
          write( 6, '(a)' ) &
     &       'old    atom       index  new    atom       index'
          write( 6, '(a)' ) &
     &       '------------------------------------------------'

          write( 6, '(a3,i8,f12.5,2x,a3,i8,f12.5)' ) &
     &          'A  ', iobest, 0.d0, &
     &          'A  ', iobest, 0.d0
          do i = 1, npbest
             write( 6, '(a3,i8,f12.5,2x,a3,i8,f12.5)' ) &
     &          'A  ', ibest(i), s_best(i), &
     &          'A  ', ibest(num_sort(i)), s_sort(i)
          end do
          do i = npbest+1, nbest
             write( 6, '(a3,i8,f12.5,2x,a3,i8,f12.5)' ) &
     &          'B  ', ibest(i), s_best(i), &
     &          'B  ', ibest(num_sort(i)), s_sort(i)
          end do

          write( 6, '(a)' ) 

      else if ( ierr .eq. 2 ) then

         write( 6, '(a)' ) &
     &      'Error - Buffer region for BEST method too large.'

      end if

      end if

      call error_handling_MPI &
     &   ( ierr, 'subroutine force_best_cent_MPI', 30 )

!-----------------------------------------------------------------------
!     /*   print                                                      */
!-----------------------------------------------------------------------

      if ( myrank .eq. 0 ) then
      if ( iprint_best .gt. 0 ) then
      if ( mod(istep,iprint_best) .eq. 0 ) then

      if ( method(1:4) .eq. 'MTS' ) then

         if ( jref .eq. 1 ) then

            open ( iounit, file = 'best.out', access = 'append' )

            do i = 1, nbead
               write( iounit, '(i8,3i4,f16.8)' ) &
     &            istep, nbest_buf, npbest_buf, nsbest_buf, pot_best(i)
            end do

            close( iounit )

         end if

      else

         open ( iounit, file = 'best.out', access = 'append' )

         do i = 1, nbead
            write( iounit, '(i8,3i4,f16.8)' ) &
     &         istep, nbest_buf, npbest_buf, nsbest_buf, pot_best(i)
         end do

         close( iounit )

      end if

      end if
      end if
      end if

      return
      end





!***********************************************************************
      subroutine force_best_add_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : &
     &   pot, fx, fy, fz, vir, natom, nbead

      use best_variables, only : &
     &   pot_best, fx_best, fy_best, fz_best, vir_best, ioption_best

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

      implicit none

!     /*   integers   */
      integer :: i, j

!-----------------------------------------------------------------------
!     /*   skip if best is off                                        */
!-----------------------------------------------------------------------

      if ( ioption_best .eq. 0 ) return

!-----------------------------------------------------------------------
!     /*   add contributions                                          */
!-----------------------------------------------------------------------

      do i = 1, nbead
         pot(i) = pot(i) + pot_best(i)
      end do

      do i = 1, nbead
      do j = 1, natom
         fx(j,i) = fx(j,i) + fx_best(j,i)
         fy(j,i) = fy(j,i) + fy_best(j,i)
         fz(j,i) = fz(j,i) + fz_best(j,i)
      end do
      end do

      do i = 1, 3
      do j = 1, 3
         vir(j,i) = vir(j,i) + vir_best(j,i)
      end do
      end do

      return
      end





!***********************************************************************
      subroutine get_s_best_bead_MPI( ibead )
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : x, y, z

      use best_variables, only : &
     &   s_best, sx_best, sy_best, sz_best, ibest, nbest, iobest

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: i, j, k, ibead

!     /*   real numbers   */
      real(8) :: xi, yi, zi, xj, yj, zj, xij, yij, zij, rij

!-----------------------------------------------------------------------
!     /*   calculate index s: solute-solvent distance                 */
!-----------------------------------------------------------------------

!     /*   loop of solvent atoms   */
      do k = 1, nbest

!        /*   solute atom   */
         i = iobest

!        /*   solvent atom   */
         j = ibest(k)

!        /*   position of solute atom   */
         xi = x(i,ibead)
         yi = y(i,ibead)
         zi = z(i,ibead)

!        /*   position of solvent atom   */
         xj = x(j,ibead)
         yj = y(j,ibead)
         zj = z(j,ibead)

!        /*   distance of solute-solvent atoms   */
         xij = xi - xj
         yij = yi - yj
         zij = zi - zj

!        /*   apply periodic boundary condition   */
         call pbc_atom_MPI( xij, yij, zij )

!        /*   distance of solute-solvent atoms   */
         rij = sqrt( xij*xij + yij*yij + zij*zij )

!        /*   x, y, z components of solute-solvent distance   */
         sx_best(k) = xij
         sy_best(k) = yij
         sz_best(k) = zij

!        /*   solute-solvent distance   */
         s_best(k) = rij

!     /*   loop of solvent atoms   */
      end do

      return
      end





!***********************************************************************
      subroutine get_s_best_cent_MPI
!***********************************************************************

!-----------------------------------------------------------------------
!     /*   shared variables                                           */
!-----------------------------------------------------------------------

      use common_variables, only : ux, uy, uz

      use best_variables, only : &
     &   s_best, sx_best, sy_best, sz_best, ibest, nbest, iobest

!-----------------------------------------------------------------------
!     /*   local variables                                            */
!-----------------------------------------------------------------------

!     /*   initialize   */
      implicit none

!     /*   integers   */
      integer :: i, j, k

!     /*   real numbers   */
      real(8) :: xi, yi, zi, xj, yj, zj, xij, yij, zij, rij

!-----------------------------------------------------------------------
!     /*   calculate index s: solute-solvent distance                 */
!-----------------------------------------------------------------------

!     /*   loop of solvent atoms   */
      do k = 1, nbest

!        /*   solute atom   */
         i = iobest

!        /*   solvent atom   */
         j = ibest(k)

!        /*   position of solute atom   */
         xi = ux(i,1)
         yi = uy(i,1)
         zi = uz(i,1)

!        /*   position of solvent atom   */
         xj = ux(j,1)
         yj = uy(j,1)
         zj = uz(j,1)

!        /*   distance of solute-solvent atoms   */
         xij = xi - xj
         yij = yi - yj
         zij = zi - zj

!        /*   apply periodic boundary condition   */
         call pbc_atom_MPI( xij, yij, zij )

!        /*   distance of solute-solvent atoms   */
         rij = sqrt( xij*xij + yij*yij + zij*zij )

!        /*   x, y, z components of solute-solvent distance   */
         sx_best(k) = xij
         sy_best(k) = yij
         sz_best(k) = zij

!        /*   solute-solvent distance   */
         s_best(k) = rij

!     /*   loop of solvent atoms   */
      end do

      return
      end

